/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
 * aumiks 0.1
 */

#include <cmath>

#include <ting/math.hpp>
#include <ting/types.hpp>

#include "PluckedString.hpp"
#include "../aumiks.hpp"
#include "synth.hpp"



using namespace ting;
using namespace aumiks;



bool SineWaveInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	bool stopPlaying = false;
	bool quit;
	ting::uint i = 0;//current sample
	do{//while(!quit)
		quit = true;
		switch(this->envPhase){
			case ATTACK:
				for(; i < mixBuf.Size();){
					real attack = this->curTime / this->attackTime;
					this->releaseFromLevel = attack;
					s32 smp = s32(real(0x7fff) * attack * ting::Sin(this->curPhase));
					this->curPhase += this->phaseStep;
					this->curTime += real(1000.0f / 44100.0f);

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					if(this->curTime >= this->attackTime){
						this->curTime -= this->attackTime;
						quit = false;
						this->envPhase = DECAY;
//						TRACE(<< "switch to DECAY" << std::endl)
						break;//~for
					}
				}
				break;
			case DECAY:
				for(; i < mixBuf.Size();){
					real decay = real(1) - (real(1) - this->sustainLevel) * (this->curTime / this->decayTime);
					this->releaseFromLevel = decay;
					s32 smp = s32(real(0x7fff) * decay * ting::Sin(this->curPhase));
					this->curPhase += this->phaseStep;
					this->curTime += real(1000.0f / 44100.0f);

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					if(this->curTime >= this->decayTime){
//						this->curTime = 0;
						quit = false;
						this->envPhase = SUSTAIN;
//						TRACE(<< "switch to SUSTAIN" << std::endl)
						break;//~for
					}
				}
				break;
			case SUSTAIN:
				if(this->stopFlag){
					this->stopFlag = false;
					this->curTime = 0;
					this->envPhase = RELEASE;
					quit = false;
			//		TRACE(<< "switch to RELEASE" << std::endl)
				}else{
					this->releaseFromLevel = this->sustainLevel;
					for(; i < mixBuf.Size();){
						s32 smp = s32(real(0x7fff) * this->sustainLevel * ting::Sin(this->curPhase));
						this->curPhase += this->phaseStep;
	//					this->curTime += 1.0f / float(44100);
				//		TRACE(<<"t =" << this->t)
				//		TRACE(<<"smp =" << smp)

						//interpolate volume
						ting::u8 vol = ting::u8(startVolume + (endVolume - startVolume) * i / mixBuf.Size());

						//apply volume
						smp = (smp * vol) / 0xff;

						mixBuf[i] += smp;
						++i;
						mixBuf[i] += smp;
						++i;
					}//~for
				}

				{
					real numPeriods = this->curPhase / D2Pi<real>();
					this->curPhase -= real(int(numPeriods)) * D2Pi<real>();
				}

				break;//~switch
			case RELEASE:
				for(; i < mixBuf.Size();){
					real release = this->releaseFromLevel * (real(1) - this->curTime / this->releaseTime);
					s32 smp = s32(real(0x7fff) * release * ting::Sin(this->curPhase));
					this->curPhase += this->phaseStep;
					this->curTime += real(1000.0f / 44100.0f);

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					if(this->curTime >= this->releaseTime){
						this->curTime = 0;
						stopPlaying = true;
						this->envPhase = ATTACK;
//						TRACE(<< "switch to STOP playing" << std::endl)
						break;//~for
					}
				}
				break;
			default:
				ASSERT(false)
				break;
		}//~switch
	}while(!quit);
	
	return stopPlaying;
}



Ref<Instrument::Channel> SineWaveInstrument::CreateChannel(){
	Ref<SineWaveInstrument::Channel> ch(new SineWaveInstrument::Channel());
	return ch;
}



void SineWaveInstrument::Channel::SetFrequency(float freq){
	this->phaseStep = D2Pi<real>() * real(freq) / real(44100);
	this->attackTime = 60;//1000.0f * 30.0f / freq;
	this->decayTime = 50;//1000.0f * 50.0f / freq;
	this->sustainLevel = 0.8f;
	this->releaseTime = 50;//1000.0f * 30.0f / freq;
}



//override
bool PluckedStringInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	bool ret = false;

	switch(this->state){
		case PLAYING:
			if(this->stopFlag){
				this->state = RELEASING;
				this->curTime = 0.2;
				this->pluckedString.Mute();
			}
			break;
		case RELEASING:
			this->curTime -= float(mixBuf.Size()) / (2 * 44100.0f);
			if(curTime <= 0){
				this->state = STOPPED;
				ret = true;
			}
			break;
		default:
			break;
	}

	for(ting::uint i=0; i < mixBuf.Size();){
		float s = this->pluckedString.Tick();
		s32 smp = s32(0x7fff * s);

		//with distortion
//		s32 smp = s32( 0x7fff * (s > 0 ? tanh(4 * s) : s ) );

		//interpolate volume
		ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

		//apply volume
		smp = (smp * vol) / 0xff;


		mixBuf[i] += smp;
		++i;
		mixBuf[i] += smp;
		++i;
	}//~for

	return ret;
}



//override
bool BowedInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	bool ret = false;

	if(this->stopFlag){
		ret = true;
	}

	for(ting::uint i=0; i < mixBuf.Size();){
		float s = this->bowed.Tick();
		s32 smp = s32(0x7fff * s);

		//interpolate volume
		ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

		//apply volume
		smp = (smp * vol) / 0xff;


		mixBuf[i] += smp;
		++i;
		mixBuf[i] += smp;
		++i;
	}//~for

	return ret;
}



//override
bool SampledInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	ASSERT(this->instr)
	const Array<real> &sample = this->instr->sample->data;
	ASSERT(sample.Size() > 0)

	real sustainLevel = this->instr->adsrParams.sustainLevel;

	bool stopPlaying = false;
	bool quit;
	ting::uint i = 0;//current sample index
	
	do{//while(!quit)
		quit = true;
		
		if(this->stopFlag){
			this->stopFlag = false;
			this->curTime = 0;
			this->envPhase = RELEASE;
		}

		switch(this->envPhase){
			case ATTACK:
				for(; i < mixBuf.Size();){
					real attack = this->curTime / this->attackTime;
					this->releaseFromLevel = attack;

					ASSERT(ting::uint(this->curPhase) < sample.Size())
					s32 smp = s32(real(0x7fff) * attack * sample[ting::uint(this->curPhase)]);
					this->curPhase += this->phaseStep;
					if(ting::uint(this->curPhase) >= sample.Size()){
						this->curPhase -= real(sample.Size());
					}

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					this->curTime += real(1000.0f / 44100.0f);
					if(this->curTime >= this->attackTime){
						this->curTime -= this->attackTime;
						quit = false;
						this->envPhase = DECAY;
//						TRACE(<< "switch to DECAY" << std::endl)
						break;//~for
					}
				}
				break;
				
			case DECAY:
				for(; i < mixBuf.Size();){
					real decay = real(1) - (real(1) - sustainLevel) * (this->curTime / this->decayTime);
					this->releaseFromLevel = decay;

					ASSERT(ting::uint(this->curPhase) < sample.Size())
					s32 smp = s32(real(0x7fff) * decay * sample[ting::uint(this->curPhase)]);
					this->curPhase += this->phaseStep;
					if(ting::uint(this->curPhase) >= sample.Size()){
						this->curPhase -= real(sample.Size());
					}
					
					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					this->curTime += real(1000.0f / 44100.0f);
					if(this->curTime >= this->decayTime){
//						this->curTime = 0;
						quit = false;
						this->envPhase = SUSTAIN;
//						TRACE(<< "switch to SUSTAIN" << std::endl)
						break;//~for
					}
				}
				break;
				
			case SUSTAIN:
				this->releaseFromLevel = sustainLevel;
				for(; i < mixBuf.Size();){

					ASSERT(ting::uint(this->curPhase) < sample.Size())
					s32 smp = s32(real(0x7fff) * sustainLevel * sample[ting::uint(this->curPhase)]);
					this->curPhase += this->phaseStep;
					if(ting::uint(this->curPhase) >= sample.Size()){
						this->curPhase -= real(sample.Size());
					}

//						TRACE(<<"t =" << this->t)
//						TRACE(<<"s =" << sample[ting::uint(this->curPhase)] << std::endl)
//						TRACE(<<"smp =" << smp << std::endl)

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + (endVolume - startVolume) * i / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;
				}//~for
				break;//~switch
				
			case RELEASE:
				for(; i < mixBuf.Size();){
					real release = this->releaseFromLevel * (real(1) - this->curTime / this->releaseTime);

					ASSERT(ting::uint(this->curPhase) < sample.Size())
					s32 smp = s32(real(0x7fff) * release * sample[ting::uint(this->curPhase)]);
					this->curPhase += this->phaseStep;
					if(ting::uint(this->curPhase) >= sample.Size()){
						this->curPhase -= real(sample.Size());
					}

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					this->curTime += real(1000.0f / 44100.0f);
					if(this->curTime >= this->releaseTime){
						this->curTime = 0;
						stopPlaying = true;
						this->envPhase = ATTACK;
//						TRACE(<< "switch to STOP playing" << std::endl)
						break;//~for
					}
				}
				break;
				
			default:
				ASSERT(false)
				break;
		}//~switch
	}while(!quit);

	return stopPlaying;
}



//override
Ref<Instrument::Channel> SampledInstrument::CreateChannel(){
	Ref<SampledInstrument::Channel> ch(new SampledInstrument::Channel(
			Ref<SampledInstrument>(this)
		));
	return ch;
};



//static
Ref<SampledInstrument::Sample> TromboneInstrument::GenerateSample(){
	Ref<Sample> ret(new Sample());

	//sampling rate 44100Hz, sound base freq is 25Hz
	ret->freq = real(25);
	ret->data.Init(44100 / ting::uint(ret->freq));//hold one period

	Array<real> &d = ret->data;

	typedef float T_Real;

	float f = D2Pi<float>() * float(ret->freq);

	for(ting::uint i = 0; i < ret->data.Size(); ++i){
		ASSERT(i < d.Size())
		float t = float(i) * float(1.0f / 44100.0f);
		d[i] = real(
				T_Real(0.002f)  * Cos(f * t)             + T_Real(-0.181f) * Sin(f * t) +
				T_Real(-0.256f) * Cos(T_Real(2) * f * t) + T_Real(-0.128f) * Sin(T_Real(2) * f * t) +
				T_Real(-0.177f) * Cos(T_Real(3) * f * t) + T_Real(0.122f)  * Sin(T_Real(3) * f * t) +
				T_Real(-0.268f) * Cos(T_Real(4) * f * t) + T_Real(0.021f)  * Sin(T_Real(4) * f * t) +
				T_Real(-0.056f) * Cos(T_Real(5) * f * t) + T_Real(0.034f)  * Sin(T_Real(5) * f * t) +
				T_Real(0.010f)  * Cos(T_Real(6) * f * t) + T_Real(-0.024f) * Sin(T_Real(6) * f * t) +
				T_Real(0.013f)  * Cos(T_Real(7) * f * t) + T_Real(0.000f)  * Sin(T_Real(7) * f * t) +
				T_Real(0.001f)  * Cos(T_Real(8) * f * t) + T_Real(-0.004f) * Sin(T_Real(8) * f * t)
			);
	}

	return ret;
}



SampledInstrument::ADSRParams TromboneInstrument::GetADSRParams(){
	ADSRParams adsr = {
		15, //attack
		30, //decay
		0.8, //sustainLevel
		5 //release
	};
	return adsr;
}



//static
Ref<SampledInstrument::Sample> ClarinetInstrument::GenerateSample(){
	Ref<Sample> ret(new Sample());

	//sampling rate 44100Hz, sound base freq is 25Hz
	ret->freq = real(25);
	ret->data.Init(44100 / ting::uint(ret->freq));//hold one period

	Array<real> &d = ret->data;

	typedef float T_Real;

	float f = D2Pi<float>() * float(ret->freq);

	for(ting::uint i = 0; i < ret->data.Size(); ++i){
		ASSERT(i < d.Size())
		float t = float(i) * float(1.0f / 44100.0f);
		d[i] = real(
				T_Real(-0.010f) * Cos(f * t)             + T_Real(0.077f)  * Sin(f * t) +
				T_Real(-0.130f) * Cos(T_Real(3) * f * t) + T_Real(-0.005f) * Sin(T_Real(3) * f * t) +
				T_Real(-0.210f) * Cos(T_Real(5) * f * t) + T_Real(0.134f)  * Sin(T_Real(5) * f * t) +
				T_Real(-0.040f) * Cos(T_Real(6) * f * t) + T_Real(0.030f)  * Sin(T_Real(6) * f * t) +
				T_Real(-0.050f) * Cos(T_Real(7) * f * t) + T_Real(-0.030f) * Sin(T_Real(7) * f * t) +
				T_Real(-0.030f) * Cos(T_Real(9) * f * t) + T_Real(0.016f)  * Sin(T_Real(9) * f * t)
			);
	}

	return ret;
}



SampledInstrument::ADSRParams ClarinetInstrument::GetADSRParams(){
	ADSRParams adsr = {
		10, //attack
		10, //decay
		0.8, //sustainLevel
		20 //release
	};
	return adsr;
}



//static
Ref<SampledInstrument::Sample> ViolinInstrument::GenerateSample(){
	Ref<Sample> ret(new Sample());

	//sampling rate 44100Hz, sound base freq is 25Hz
	ret->freq = real(25);
	ret->data.Init(44100 / ting::uint(ret->freq));//hold one period

	typedef float T_Real;

	T_Real y[] = {
		0,
		0.4,
		0.28,
		0.36,
		0.2,
		0.76,
		0.04,
		- 0.2,
		- 0.2,
		0.52,
		0.16,
		1,
		- 0.44,
		0.92,
		- 0.68,
		- 0.24,
		- 0.48,
		- 0.48,
		- 0.04,
		- 0.88,
		- 0.88,
		- 0.72,
		- 0.88,
		- 0.76,
		0
	};

	T_Real x[] = {
		0,
		4,
		8,
		10,
		13,
		17,
		28,
		34,
		40,
		51,
		56,
		64,
		77,
		90,
		97,
		105,
		109,
		112,
		124,
		134,
		140,
		143,
		146,
		149,
		152
	};

	ting::uint numPoints = sizeof(x) / sizeof(x[0]);
	ASSERT(numPoints == sizeof(y) / sizeof(y[0]))
	ASSERT(numPoints > 0)

	T_Real t = T_Real(ret->data.Size());//period in samples
	T_Real d = t / x[numPoints - 1];// time scale

	ting::uint j = 0;

	for(ting::uint i = 0; i < numPoints - 1 && j < ret->data.Size(); ++i){
		T_Real segLenX = (x[i + 1] - x[i]) * d;
		T_Real segDelta = y[i + 1] - y[i];
		T_Real k = segDelta/segLenX;

		T_Real b = - k * x[i] * d + y[i];

		while(j < x[i + 1] * d && j < ret->data.Size()){
			ASSERT_INFO(j < ret->data.Size(), "j = " << j << " ret->data.Size() = " << ret->data.Size())
			ret->data[j] = k * j + b;
//			TRACE(<< (ret->data[j]) << std::endl)
			++j;
		}
	}


	return ret;
}



SampledInstrument::ADSRParams ViolinInstrument::GetADSRParams(){
	ADSRParams adsr = {
		150, //attack
		100, //decay
		0.9, //sustainLevel
		20 //release
	};
	return adsr;
}

