// (c) Ivan Gagis
// e-mail: igagis@gmail.com
// Version: 1

// Description:
//	Theremin server

#pragma once

#include <list>

#include <ting/Timer.hpp>

#include <cliser/srv/Server.hpp>

#include "Client.hpp"
#include "Cast.hpp"
#include "DatabaseThread.hpp"



class Therver : public cliser::Server{
	friend class NetworkDataFromClientReceivedMessage;
	friend class ClientDisconnectedMessage;
	friend class KillIdleCastsMessage;

	T_ClientsList waitingForListeningClients;

	T_CastsList casts;

	static inline ting::u32 DIdleCastDuration(){
		return 30 * 1000; //30 seconds
	}

	class KillIdleCastsTimer : public ting::Timer{
		Therver *t;
		
	public:
		//override
		ting::u32 OnExpire();
		
		KillIdleCastsTimer(Therver* thread) :
				t(thread)
		{}

		inline void StartTimer(){
			this->Start(Therver::DIdleCastDuration());
		}
	} killIdleCastsTimer;

	DatabaseThread dbThread;

public:

	Therver(ting::u16 port, const std::string& pathToDataDir) :
			cliser::Server(port, 2),
			killIdleCastsTimer(this),
			dbThread(this, pathToDataDir)
	{
		this->dbThread.Start();

		//Call OnExpire() to post the first 'kill-idle-casts' message
		//to the message queue, thus, when the main loop is started
		//it will handle the message and the handler will schedule
		//this->killIdleCastsTimer. After that the timer will be
		//rescheduled periodically by 'kill-idle-casts' message handler.
		this->killIdleCastsTimer.OnExpire();
	}

	~Therver(){
		this->dbThread.PushQuitMessage();
		this->dbThread.Join();
	}

	//override
	ting::Ref<cliser::Client> CreateClientObject();

	//override
	void OnClientConnected(ting::Ref<cliser::Client>& client);

	//override
	void OnClientDisconnected(ting::Ref<cliser::Client>& client);

	//override
	void OnDataReceivedFromClient(ting::Ref<cliser::Client>& client, ting::Array<ting::byte> d);

private:
	void HandleRegisterFromClient(ting::Ref<Client>& c, ting::Array<ting::byte> d);
	void HandleLoginFromClient(ting::Ref<Client>& c, ting::Array<ting::byte> p);
	void HandlePlayingDataFromClient(ting::Ref<Client>& c, ting::Array<ting::byte> d);
	void HandleListenNextCommandFromClient(ting::Ref<Client>& c);
	void HandleStopListeningCommandFromClient(ting::Ref<Client>& c);

	T_CastsIter RemoveCast(T_CastsIter castIter);

	void KillIdleCasts();

	void MoveClientToIdleState(ting::Ref<Client>& c);

	void MoveWaitingListenersToCast();
	void RemoveClientFromWaitingListenersList(ting::Ref<Client>& c);

public:
	void PushRegisterResultMessage(Client::ERegisterRes result, ting::Ref<Client> c);
};



class ClientDisconnectedMessage : public ting::Message{
	Therver *t;//this mesage should hold reference to the thread this message is sent to

	ting::Ref<Client> c;

public:
	ClientDisconnectedMessage(Therver* thread, ting::Ref<Client>& client) :
			t(thread),
			c(client)
	{
		ASSERT(this->t)
		ASSERT(this->c.IsValid())
	}

	//override
	void Handle();
};



class NetworkDataFromClientReceivedMessage : public ting::Message{
	Therver *thr;//this mesage should hold reference to the thread this message is sent to

	ting::Ref<Client> client;

	ting::Array<ting::byte> data;

public:
	NetworkDataFromClientReceivedMessage(
			Therver* thread,
			ting::Ref<Client>& clt,
			ting::Array<ting::byte> d
		) :
			thr(thread),
			client(clt),
			data(d)
	{
		ASSERT(this->thr)
		ASSERT(this->client)
		ASSERT(this->data.Size()!=0)
	}

	//override
	void Handle();
};



class NewUserRegisteredMessage : public ting::Message{
	Therver *t;

	ting::Ref<Client> c;

	Client::ERegisterRes res;
	
public:
	NewUserRegisteredMessage(
			Therver *thread,
			ting::Ref<Client> client,
			Client::ERegisterRes result
		) :
			t(thread),
			c(client),
			res(result)
	{
		ASSERT(this->t)
		ASSERT(this->c)
	}

	//override
	void Handle();
};



class UserPasswordQueriedMessage : public ting::Message{
	Therver *t;

	ting::Ref<Client> c;

	std::string pw;//queried password

public:
	enum EResult{
		SUCCESS,
		USERNAME_NOT_FOUND,
		SOME_ERROR
	};

private:
	EResult res;

public:
	UserPasswordQueriedMessage(
			Therver *thread,
			ting::Ref<Client> client,
			const std::string& password,
			EResult result
		) :
			t(thread),
			c(client),
			pw(password.c_str()), //copy string
			res(result)
	{
		ASSERT(this->t)
		ASSERT(this->c)
	}

	//override
	void Handle();
};



class KillIdleCastsMessage : public ting::Message{
	Therver *t;

public:
	KillIdleCastsMessage(Therver *thread) :
			t(thread)
	{
		ASSERT(this->t)
	}

	//override
	void Handle(){
		this->t->KillIdleCasts();
	}
};


