// (c) Ivan Gagis
// e-mail: igagis@gmail.com
// Version: 1

// Description:
//          Main Thread class

#include <vector>
#include <string>
#include <ting/Timer.hpp>
#include <ting/Thread.hpp>
#include <ting/Singleton.hpp>
#include <ting/utils.hpp>

#include <cliser/clt/NetworkThread.hpp>

#include "stdafx.hpp"
#include "MainThread.hpp"
#include "Preferences.hpp"
#include "ting/Buffer.hpp"



using namespace ting;



MainThread::MainThread() :
		networkThread(this),
		networkState(OFFLINE),
		reconnectTime(MainThread::DMinReconnectTime()),
		isListening(false)
{
	this->musicActionsBuffer.reserve(8196);//8kb
}



MainThread::~MainThread(){
	//Stop timer if it is not stopped. Just to be on the safe side.
	this->sendMusicActionsTimer.Stop();
}



void MainThread::Run(){
	this->networkThread.Start();//launch network thread
	this->playbackThread.Start();//launch music playback thread

	//Check if autologin is on and try to connect if it is.
	if(Preferences::Inst().GetAutologin()){
		this->Connect_ts();
	}

	while(!this->quitFlag){
		this->queue.GetMsg()->Handle();
	}

	this->playbackThread.PushQuitMessage();
	this->networkThread.PushQuitMessage();

	this->playbackThread.Join();
	this->networkThread.Join();
}



void MainThread::AddNetworkStateListener_ts(
		MainThread::NetworkStateListener* listener
	)
{
	ASSERT(listener)
	this->networkStateListeners.Add_ts(listener);

	//notify listener about current state
	{
		ting::Mutex::Guard mutexGuard(this->networkStateMutex);
		listener->OnStateChange(this->networkState);
	}
}



void MainThread::RemoveNetworkStateListener_ts(
		MainThread::NetworkStateListener* listener
	)
{
	ASSERT(listener)
	this->networkStateListeners.Remove_ts(listener);
}



void ThNetworkThread::SendLoginToServer_ts(
		const std::string& username,
		const std::string& password
	)
{
	ASSERT(username.size() < 1000)
	ASSERT(password.size() < 1000)

	ting::Array<ting::byte> p(1 + 2 + username.size() + 2 + password.size());//packet

	ting::uint i = 0;

	ASSERT(i < p.Size())
	p[i] = 0;//login message ID
	++i;

	ASSERT(i + 2 <= p.Size())
	ting::ToNetworkFormat16(u16(username.size()), &p[i]);
	TRACE(<< "unLen = " << u32(p[i]) << " " << u32(p[i + 1]) << std::endl)
	i += 2;

	if(username.size() > 0){
		ASSERT(i < p.Size() && i + username.size() <= p.Size())
		memcpy(&p[i], username.c_str(), username.size());
		i += username.size();
	}

	ASSERT(i + 2 <= p.Size())
	ting::ToNetworkFormat16(u16(password.size()), &p[i]);
	TRACE(<< "pwLen = " << u32(p[i]) << " " << u32(p[i + 1]) << std::endl)
	i += 2;

	if(password.size() > 0){
		ASSERT(i < p.Size() && i + password.size() <= p.Size())
		memcpy(&p[i], password.c_str(), password.size());
		ASSERT(i + password.size() == p.Size())
	}

	this->SendData_ts(p);
}



void ThNetworkThread::SendStopListeningToServer_ts(){
	Array<byte> p(1);
	p[0] = 0x02;

	this->SendData_ts(p);
}



void MainThread::SetUpReconnectTimerIfAutologinIsOn(){
	//set up reconnect timer if autologin is on
	if(!Preferences::Inst().GetAutologin()){
		return;
	}

	u32 time = this->reconnectTime;
	this->reconnectTime *= 2;//update reconnect time BEFORE staring the timer!!!
	ting::ClampTop(this->reconnectTime, MainThread::DMaxReconnectTime());
	this->reconnectTimer.Start(time);
}



void ConnectedMessage::Handle(){
	TRACE(<< "ConnectedMessage::Handle(): invoked" << std::endl)

	if(this->reason == cliser::NetworkThread::SUCCESS){
		this->mt->ChangeNetworkState(MainThread::CONNECTED);

		//init reconnectTime
		this->mt->reconnectTime = MainThread::DMinReconnectTime();

		//initiate login
		this->mt->Login_ts();
	}else{
		this->mt->ChangeNetworkState(MainThread::OFFLINE);

		this->mt->SetUpReconnectTimerIfAutologinIsOn();
	}
}



void DisconnectedMessage::Handle(){
	TRACE(<<"DisconnectedMessage::Handle(): invoked"<<std::endl)

	//if state is already OFFLINE_LOGIN_FAILED then do not change it to simple OFFLINE
	if(this->mt->networkState != MainThread::OFFLINE_LOGIN_FAILED){
		this->mt->ChangeNetworkState(MainThread::OFFLINE);

		//Do not reconnect if disconnected because of login failed
		//(incorrect login/passwordd supplied).
		this->mt->SetUpReconnectTimerIfAutologinIsOn();
	}
}



void NetworkDataFromServerMessage::Handle(){
//	TRACE(<<"C_NetworkDataFromServerMessage::Handle(): invoked"<<std::endl)
	ASSERT(this->data.Size() > 0)
	switch(this->data[0]){
		case 0x00: //login result
			this->mt->HandleLoginResultFromServer(this->data);
			break;
		case 0x01:
			this->mt->HandlePlayerSwitchMessageFromServer(this->data);
			break;
		case 0x02:
			this->mt->HandleStopListeningMessageFromServer(this->data);
			break;
		case 0x03:
			this->mt->HandleListeningMusicDataFromServer(this->data);
			break;
		default:
			ASSERT_INFO(false, "unknown message id = " << ((u32)this->data[0]))
			break;
	}
}



//override
void ThNetworkThread::OnConnect(EConnectFailReason failReason){
	this->mt->PushMessage(
			ting::Ptr<ting::Message>(
					new ConnectedMessage(this->mt, failReason)
				)
		);
}



//override
void ThNetworkThread::OnDisconnect(){
	this->mt->PushMessage(
			ting::Ptr<ting::Message>(
					new DisconnectedMessage(this->mt)
				)
		);
}



//override
void ThNetworkThread::OnDataReceived(ting::Array<ting::byte> data){
	TRACE(<<"ThNetworkThread::OnDataReceived(): " << data.SizeInBytes() << " bytes" << std::endl)
	this->mt->PushMessage(
			ting::Ptr<ting::Message>(
					new NetworkDataFromServerMessage(this->mt, data)
				)
		);
}



//override
void MusicActionMessage::Handle(){
//	TRACE(<< "MusicActionMessage::Handle(): invoked" << std::endl)

	//flush bffer if it is full (send to server)
	if(this->mt->musicActionsBuffer.size() >= this->mt->musicActionsBuffer.capacity() / 2){
		//stop the timer
		this->mt->sendMusicActionsTimer.Stop();

		TRACE(<< "MusicActionMessage::Handle(): flushing actions" << std::endl)
		this->mt->SendMusicActionsBufferToServer();
	}

	//if buffer is empty
	if(this->mt->musicActionsBuffer.size() == 0){
		TRACE(<< "MusicActionMessage::Handle(): buffer is empty" << std::endl)

		this->mt->numActionsInBuffer = 0;//clear actions counter

		this->mt->musicActionsBuffer.push_back(0x03);//message type

		switch(Preferences::Inst().GetTimbre()){
			default:
			case Preferences::SINE_PLUS_ADSR:
				this->mt->musicActionsBuffer.push_back(0x00);
				break;
			case Preferences::PLUCKED_STRING:
				this->mt->musicActionsBuffer.push_back(0x01);
				break;
		}

		//push 2 bytes for music actions counter
		this->mt->musicActionsBuffer.push_back(0);
		this->mt->musicActionsBuffer.push_back(0);

		//start 3 seconds timer to send music actions at least every 3 seconds
		//if there are any actions
		this->mt->sendMusicActionsTimer.Start(3000);
	}

	//append command to buffer
	++this->mt->numActionsInBuffer;

	switch(this->type){
		case NOTE_ON:
			this->mt->musicActionsBuffer.push_back(0x00);

			//push timestamp
			for(ting::uint i = 0; i < sizeof(u32); ++i)
				this->mt->musicActionsBuffer.push_back(0);
			ting::ToNetworkFormat32(
					this->timeStamp,
					&this->mt->musicActionsBuffer[this->mt->musicActionsBuffer.size() - 4]
				);

			//push volume
			this->mt->musicActionsBuffer.push_back(this->volume);

			//push frequency
			ASSERT(sizeof(float) == sizeof(u32))
			for(ting::uint i = 0; i < sizeof(u32); ++i)
				this->mt->musicActionsBuffer.push_back(0);
			ting::ToNetworkFormat32(
					*reinterpret_cast<u32*>(&this->freq),
					&this->mt->musicActionsBuffer[this->mt->musicActionsBuffer.size() - 4]
				);

			break;

		default:
		case NOTE_OFF:
			this->mt->musicActionsBuffer.push_back(0x01);

			//push timestamp
			for(ting::uint i = 0; i < sizeof(u32); ++i)
				this->mt->musicActionsBuffer.push_back(0);
			ting::ToNetworkFormat32(
					this->timeStamp,
					&this->mt->musicActionsBuffer[this->mt->musicActionsBuffer.size() - 4]
				);
			break;
	}//~switch
}



void MainThread::SendMusicActionsBufferToServer(){
	if(this->networkState != LOGGED_IN){
		return;
	}

	if(this->musicActionsBuffer.size() == 0){
		return;
	}

	Array<byte> buf(this->musicActionsBuffer.size());

	//set number of action commands in buffer
	ting::ToNetworkFormat16(
			u16(this->numActionsInBuffer),
			&this->musicActionsBuffer[2]
		);

	memcpy(&buf[0], &this->musicActionsBuffer[0], buf.SizeInBytes());

	this->musicActionsBuffer.clear();

	this->networkThread.SendData_ts(buf);
}



void SendMusicActionsToServerMessage::Handle(){
	ASSERT(this->mt->musicActionsBuffer.size() > 0)
	this->mt->SendMusicActionsBufferToServer();
}



//override
u32 MainThread::SendMusicActionsTimer::OnExpire(){
	TRACE(<< "MainThread::SendMusicActionsTimer::OnExpire(): 3 sec expired" << std::endl)
	MainThread::Inst().PushMessage(
			Ptr<Message>(
					new SendMusicActionsToServerMessage(&MainThread::Inst())
				)
		);

	return 0;//no need to reschedule the timer
}



void MainThread::ListenNext_ts(){
	this->isListening = true;
	this->networkThread.SendListenNextToServer_ts();
}



void MainThread::StopListening_ts(){
	this->isListening = false;

	//tell server to remove listener from cast
	this->networkThread.SendStopListeningToServer_ts();

	//reset playback to prevent any notes remaining on
	this->playbackThread.ResetPlayback_ts();
}



void ThNetworkThread::SendListenNextToServer_ts(){
	Array<byte> p(1);

	p[0] = 0x01;

	this->SendData_ts(p);
}



//override
void ConnectRequestMessage::Handle(){
	TRACE(<< "ConnectRequestMessage::HANDLE(): invoked" << std::endl)

	if(this->manuallyRequestedConnect){
		this->t->reconnectTime = MainThread::DMinReconnectTime();

		//If previous login attempt failed at login then
		//change the state to simple OFFLINE, because
		//OFFLINE_LOGIN_FAILED is to prevent further automatic logins,
		//while we have manually requested login here, so we do not want
		//the login to be stopped.
		if(this->t->networkState == MainThread::OFFLINE_LOGIN_FAILED){
			this->t->networkState = MainThread::OFFLINE;
		}
	}

	//Check if already connecting or connected
	if(this->t->networkState != MainThread::OFFLINE){
		//Notify listeners by changing state to current state (no state change in fact).
		this->t->ChangeNetworkState(this->t->networkState);
		return;
	}

	//send connect request
	this->t->networkThread.Connect_ts(DHost(), DPort());

	//change state to CONNECTING
	this->t->ChangeNetworkState(MainThread::CONNECTING);
}



//override
void LoginRequestMessage::Handle(){
	TRACE(<< "LoginRequestMessage::Handle(): invoked" << std::endl)

	//Check if already loggin in or logged in
	if(this->t->networkState == MainThread::LOGGING_IN ||
			this->t->networkState == MainThread::LOGGED_IN
		)
	{
		//Notify listeners by changing state to current state (no state change in fact).
		this->t->ChangeNetworkState(this->t->networkState);
		TRACE(<< "LoginRequestMessage::Handle(): already logged in, returning" << std::endl)
		return;
	}

	if(this->t->networkState != MainThread::CONNECTED){
		//ignore request, do not alter network state
		TRACE(<< "LoginRequestMessage::Handle(): state is not CONNECTED, returning" << std::endl)
		return;
	}

//	TRACE(<< "LoginRequestMessage::Handle(): sending login request" << std::endl)
	
	//Send login request to server and change current network state to LOGGING_IN.
	this->t->networkThread.SendLoginToServer_ts(
			Preferences::Inst().GetUsername(),
			Preferences::Inst().GetPassword()
		);

//	TRACE(<< "LoginRequestMessage::Handle(): login request sent" << std::endl)

	this->t->ChangeNetworkState(MainThread::LOGGING_IN);
//	TRACE(<< "LoginRequestMessage::Handle(): exit" << std::endl)
}



void MainThread::ConnectInternal_ts(bool manuallyRequested){
	this->PushMessage(
			Ptr<Message>(
					new ConnectRequestMessage(this, manuallyRequested)
				)
		);
}



void MainThread::Disconnect_ts(){
	this->networkThread.Disconnect_ts();
}



void MainThread::Login_ts(){
	this->PushMessage(
			Ptr<Message>(
					new LoginRequestMessage(this)
				)
		);
}



void MainThread::ChangeNetworkState(ENetworkState newState){
	ting::Mutex::Guard mutexGuard(this->networkStateMutex);

	this->networkState = newState;//change state

	//notify listeners
	this->networkStateListeners.stateToNotifiy = this->networkState;
	this->networkStateListeners.Notify_ts();
}



//override
ting::u32 MainThread::ReconnectTimer::OnExpire(){
	MainThread::Inst().ConnectInternal_ts(false);
	return 0; //do not reschedule timer
}



void MainThread::HandleLoginResultFromServer(ting::Array<ting::byte> p){
	TRACE(<<"MainThread::HandleLoginResultFromServer(): invoked"<<std::endl)

	ASSERT(p[0] == 0)// 0 = login result

	if(p.Size() != 2){
		ASSERT_INFO(false, "bad formed login result packet received")
		return;//ignore
	}

	if(this->networkState == LOGGED_IN){
		//already logged in, ignore this message
		ASSERT_INFO(false, "already logged in")//normally, should not get this inLOGGED_IN state???
		return;
	}

	if(p[1] == 0){//success
		this->ChangeNetworkState(MainThread::LOGGED_IN);
	}else{//failure
		this->ChangeNetworkState(MainThread::OFFLINE_LOGIN_FAILED);
		this->networkThread.Disconnect_ts();
	}
}



void MainThread::HandleListeningMusicDataFromServer(ting::Array<ting::byte> p){
	if(!this->isListening){
		//not in listening state, ignore packet
		return;
	}

	this->playbackThread.PushNewPlaybackData_ts(p);
}



void MainThread::HandlePlayerSwitchMessageFromServer(ting::Array<ting::byte> p){
	ting::uint i = 0;

	if(i + 1 > p.Size()){
		//bad formed packet, ignore
		ASSERT_INFO(false, "bad formed packet")
		return;
	}
	if(p[0] != 0x01){
		ASSERT_INFO(false, "bad formed packet")
		return;
	}
	++i;

	//parse player username
	std::string playerUsername;
	{
		if(i + 2 > p.Size()){
			//bad formed packet, ignore
			ASSERT_INFO(false, "bad formed packet")
			return;
		}
		ting::uint unLen = ting::FromNetworkFormat16(&p[i]);
		i += 2;

		if(unLen == 0){
			ASSERT_INFO(false, "empty username not allowed")
			return;//ignore
		}

		ASSERT(unLen != 0)
		if(i + unLen > p.Size()){
			//bad formed packet, ignore
			ASSERT_INFO(false, "bad formed packet")
			return;
		}
		playerUsername = std::string(reinterpret_cast<char*>(&p[i]), unLen);
		ASSERT(i + unLen == p.Size())
	}

	this->playbackThread.HandlePlayerSwitch_ts();

	this->currentPlayerListeners.SetCurrentPlayer(playerUsername);
	this->currentPlayerListeners.Notify_ts();
}



void MainThread::HandleStopListeningMessageFromServer(ting::Array<ting::byte> p){
	if(p.Size() < 1){
		//bad formed packet, ignore
		ASSERT_INFO(false, "bad formed packet")
		return;
	}

	//reset playback to prevent any notes remaining on
	this->playbackThread.ResetPlayback_ts();

	this->currentPlayerListeners.SetCurrentPlayer(std::string());//empty string
	this->currentPlayerListeners.Notify_ts();
}

