/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

#include <string>

#include <gtkmm/box.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/button.h>
#include <gtkmm/label.h>
#include <glibmm/dispatcher.h>

#include <ting/Array.hpp>
#include <ting/types.hpp>
#include <ting/utils.hpp>

#include <cliser/clt/NetworkThread.hpp>

#include "stdafx.hpp"
#include "RegisterDialog.hpp"



using namespace ting;



RegisterDialog::RegisterDialog(Gtk::Window &parent) :
		Gtk::Dialog("Register in system", parent, true)
{
	Gtk::VBox *vbox = this->get_vbox();
	vbox->set_spacing(5);

	vbox->add(*Gtk::manage(new Gtk::Label(
			"Choose username:",
			Gtk::ALIGN_LEFT
		)));
	vbox->add(this->usernameEntryField);

	vbox->add(*Gtk::manage(new Gtk::Label(
			"Choose password:",
			Gtk::ALIGN_LEFT
		)));

#ifdef GLIBMM_PROPERTIES_ENABLED
	this->passwordEntryField.property_visibility().set_value(false);//set password mode
#else
	this->passwordEntryField.set_property("visibility", false);//set password mode
#endif
	
	vbox->add(this->passwordEntryField);

	vbox->add(*Gtk::manage(new Gtk::Label(
			"Confirm password:",
			Gtk::ALIGN_LEFT
		)));

#ifdef GLIBMM_PROPERTIES_ENABLED
	this->passwordConfirmEntryField.property_visibility().set_value(false);//set password mode
#else
	this->passwordConfirmEntryField.set_property("visibility", false);//set password mode
#endif


	vbox->add(this->passwordConfirmEntryField);

	{
		Gtk::Button *b = new Gtk::Button("Register");
		b->signal_clicked().connect(
				sigc::mem_fun(*this, &RegisterDialog::OnRegisterButtonPressed)
			);
		this->get_action_area()->add(*Gtk::manage(b));
	}

//	this->add_button("Ok", M_DIALOG_RESULT_OK);
	this->add_button("Cancel", Gtk::RESPONSE_CANCEL);
	
	this->show_all_children();//Show all window child widgets
}



enum ERegisterResult{
	REGISTERRES_UNKNOWN,
	REGISTERRES_OK,
	REGISTERRES_USERNAME_TAKEN,
	REGISTERRES_COULD_NOT_CONNECT,
	REGISTERRES_SOME_ERROR
};



class RegisterNetThread : public cliser::NetworkThread{
	Gtk::Dialog* d;

	//NOTE: dispatchers are created by UI thread, since the whole
	//RegisterNetThread object is created by UI thread in the
	//RegisterDialog::OnRegisterButtonPressed() method.
	Glib::Dispatcher resOk;
	Glib::Dispatcher resUsernameAlreadTaken;
	Glib::Dispatcher resCouldNotConnectToServer;
	Glib::Dispatcher resSomeError;

	ERegisterResult res;

	std::string un;
	std::string pw;

	void OkDispatched(){
		d->response(REGISTERRES_OK);
	}

	void UsernameAlreadyTakenDispatched(){
		d->response(REGISTERRES_USERNAME_TAKEN);
	}

	void CouldNotConnectToServerDispatched(){
		d->response(REGISTERRES_COULD_NOT_CONNECT);
	}

	void SomeErrorDispatched(){
		d->response(REGISTERRES_SOME_ERROR);
	}

	//override
	void OnConnect(cliser::NetworkThread::EConnectFailReason failReason){
		if(failReason != SUCCESS){
			//notify about result
			this->resCouldNotConnectToServer.emit();
			return;
		}

		//send register request
		ASSERT(this->un.size() <= 1000)
		ASSERT(this->pw.size() <= 1000)

		Array<byte> p(1 + 2 + this->un.size() + 2 + this->pw.size());

		ting::uint i = 0;

		p[i] = 0x04;//register request
		++i;

		ting::ToNetworkFormat16(this->un.size(), &p[i]);
		i += 2;

		memcpy(&p[i], this->un.c_str(), this->un.size());
		i += this->un.size();

		ting::ToNetworkFormat16(this->pw.size(), &p[i]);
		i += 2;

		memcpy(&p[i], this->pw.c_str(), this->pw.size());
		ASSERT(i + this->pw.size() == p.SizeInBytes())

		this->SendData_ts(p);
	}

	//override
	void OnDisconnect(){
		TRACE(<< "RegisterNetThread::OnDisconnect(): invoked" << std::endl)
		switch(this->res){
			default:
			case REGISTERRES_UNKNOWN:
				//spontaneous disconnect (res = UNKNOWN) or some other error
				this->resSomeError.emit();
				break;
			case REGISTERRES_OK:
				this->resOk.emit();
				break;
			case REGISTERRES_USERNAME_TAKEN:
				this->resUsernameAlreadTaken.emit();
				break;
		}
	}

	//override
	void OnDataReceived(ting::Array<ting::byte> data){
		ASSERT(data.Size() > 0)

		if(data[0] != 0x04){
			//ignore
			return;
		}

		if(data.Size() != 2){
			//bad formed packet received form server, ignore
			ASSERT_INFO(false, "bad formed packet received form server")
			return;
		}

		switch(data[1]){
			case 0: //success
				this->res = REGISTERRES_OK;
				break;
			case 1: //username already exists
				this->res = REGISTERRES_USERNAME_TAKEN;
				break;
			case 2: //username contains forbidden characters
				//TODO:
//				break;//fallthrough
			default: //some error
				this->res = REGISTERRES_SOME_ERROR;
				break;
		}

		//initiate disconnect, result will be returned in OnDisconnect() via dispatchers.
		this->Disconnect_ts();
	}

public:
	RegisterNetThread(
			Gtk::Dialog *respDialog,
			const std::string& username,
			const std::string& password
		) :
			d(respDialog),
			res(REGISTERRES_UNKNOWN),
			un(username),
			pw(password)
	{
		ASSERT(this->d)
		ASSERT(this->un.size() > 0)
		ASSERT(this->pw.size() > 0)

		//connect dipatchers
		this->resOk.connect(sigc::mem_fun(
				*this,
				&RegisterNetThread::OkDispatched
			));
		this->resUsernameAlreadTaken.connect(sigc::mem_fun(
				*this,
				&RegisterNetThread::UsernameAlreadyTakenDispatched
			));
		this->resCouldNotConnectToServer.connect(sigc::mem_fun(
				*this,
				&RegisterNetThread::CouldNotConnectToServerDispatched
			));
		this->resSomeError.connect(sigc::mem_fun(
				*this,
				&RegisterNetThread::SomeErrorDispatched
			));
	}
};//~class RegisterNetThread



class ErrorDialog : public Gtk::MessageDialog{
public:
	ErrorDialog(Gtk::Window& parent, const std::string& errorMessage) :
			Gtk::MessageDialog(
					parent,
					"Error",
					false, //do not use markup
					Gtk::MESSAGE_ERROR,
					Gtk::BUTTONS_OK,
					true //true = modal
				)
	{
		this->set_deletable(false);
		this->set_secondary_text(errorMessage);
	}

	inline void Run(){
		this->run();
		this->hide();
	}
};



class PleaseWaitDialog : public Gtk::MessageDialog{
public:
	PleaseWaitDialog(Gtk::Window& parent) :
			Gtk::MessageDialog(
					parent,
					"Registering...",
					false, //do not use markup
					Gtk::MESSAGE_INFO,
					Gtk::BUTTONS_NONE,
					true //true = modal
				)
	{
		this->set_secondary_text("Please wait.");
		this->set_deletable(false);

		//connect to signal "close" in order to ignore Escape key press, because normal
		//Gtk::Dialog is closing by Escape key press.
		g_signal_connect(G_OBJECT(this->gobj()), "close", G_CALLBACK(&PleaseWaitDialog::OnCloseEvent), NULL);
	}
	
private:
	static gboolean OnCloseEvent(
			GtkWidget *widget,
			GdkEvent  *event,
			gpointer   data
		)
	{
//		TRACE(<< "CLOSE_EVENT()" << std::endl)

		//The "close" of a dialog is emitted when Escape key is pressed.
		//Prevent default "close" handler to be called which closes dialog window.
		//Simply returning TRUE does not work for some reason.
		g_signal_stop_emission_by_name(G_OBJECT(widget), "close");

		return TRUE;
	}

};



void RegisterDialog::OnRegisterButtonPressed(){
	//verify if entered passwords are identical
	if(this->usernameEntryField.get_text().size() == 0){
		ErrorDialog d(*this, "Username cannot be empty.");
		d.Run();
		return;
	}

	if(this->passwordEntryField.get_text().compare(
			this->passwordConfirmEntryField.get_text()
		) != 0)
	{
		ErrorDialog d(*this, "Password confirmation is not the same as password.");
		d.Run();
		return;
	}

	if(this->passwordEntryField.get_text().size() == 0){
		ErrorDialog d(*this, "Password cannot be empty.");
		d.Run();
		return;
	}
	
	PleaseWaitDialog pleaseWaitDialog(*this);

	std::string un = this->usernameEntryField.get_text();
	std::string pw = this->passwordEntryField.get_text();

	RegisterNetThread netThr(&pleaseWaitDialog, un, pw);
	netThr.Start();

	//initiate registering process by connecting to server
	netThr.Connect_ts(DHost(), DPort());

	//NOTE: theoretically, netThr can finish its work (creating an account) before
	//the pleaseWaitDialog.run() is called, but, since result returning is done through
	//dispatchers, we can expect that at least one cycle of main loop inside
	//the pleaseWaitDialog.run() will be completed and only then the
	//pleaseWaitDialog.response() will be called by handler connected to the dispatcher.

	//Show "Please wait..." dialog
	ERegisterResult res = ERegisterResult(pleaseWaitDialog.run());
	switch(res){
		case REGISTERRES_OK:
			{
				Gtk::MessageDialog d(
						*this,
						"Successful",
						false, //do not use markup
						Gtk::MESSAGE_INFO,
						Gtk::BUTTONS_OK,
						true //true = modal
					);
				d.set_secondary_text("You have successfuly registered in the system.");
				d.set_deletable(false);
				d.run();
				d.hide();
			}
			//set new username and password
			this->username = un;
			this->password = pw;
			this->response(Gtk::RESPONSE_OK);//close register dialog
			break;
		case REGISTERRES_USERNAME_TAKEN:
			{
				ErrorDialog d(*this, "That username is already taken, please choose a different one.");
				d.Run();
			}
			break;
		case REGISTERRES_COULD_NOT_CONNECT:
			{
				ErrorDialog d(*this, "Could not connect to server, check your internet connection.");
				d.Run();
			}
			break;
		default:
			//some error
			{
				ErrorDialog d(*this, "Registration failed, try again later.");
				d.Run();
			}
			break;
	}

	pleaseWaitDialog.hide();

	//make sure the thread has finished before destroying thread object
	netThr.PushQuitMessage();
	netThr.Join();
}

