#include <pugixml/pugixml.hpp>


#include <ting/debug.hpp>


#include "Preferences.hpp"
#include "FSFile.hpp"


const char* DOptionsTag = "Options";
const char* DNumOctavesAttribute = "numOctaves";
const char* DTimbreAttribute = "timbre";
const char* DLockCurrentOctaveAttribute = "lockCurrentOctave";

const char* DTuningTag = "Tuning";
const char* DBasePitchAttribute = "basePitch";

const char* DNetworkOptionsTag = "NetworkOptions";
const char* DUsernameAttribute = "username";
const char* DPasswordAttribute = "password";
const char* DAutologinAttribute = "autologin";
const char* DBroadcastAttribute = "broadcast";



//static
std::string Preferences::TimbreTypeToName(Preferences::E_Timbre timbre){
	switch(timbre){
		default:
		case SINE_PLUS_ADSR:
			return "SineWave";
			break;
		case PLUCKED_STRING:
			return "PluckedString";
			break;
	}
}



//static
Preferences::E_Timbre Preferences::TimbreNameToType(std::string name){
	for(ting::uint t = 0; t < ting::uint(NUM_TIMBRES); ++t){
		if(Preferences::TimbreTypeToName(E_Timbre(t)).compare(name) == 0)
			return E_Timbre(t);
	}
	return SINE_PLUS_ADSR;
}



void Preferences::ClampBasePitch(){
	if(this->basePitch + 12 * this->numOctaves > (DHighestOctave() + 1) * 12){
		this->basePitch = (DHighestOctave() - (this->numOctaves)) * 12;
	}
}



void Preferences::LoadFromFile(){
//	TRACE(<< "MainWindow::LoadPreferencesFromFile(): enter" << std::endl)

	file::FSFile f(file::File::GetHomeDir() + ".theremin/" + this->prefsFileName);

//	TRACE(<< "MainWindow::LoadPreferencesFromFile(): reading file" << std::endl)

	pugi::xml_document doc;
	{
		ting::Array<ting::byte> rawFile;
		try{
			rawFile = f.LoadWholeFileIntoMemory();
		}catch(file::File::Exc &e){
			LOG(
					<< "Preferences::LoadFromFile(): could not load preferences file." << std::endl
					<< "\t" << " e = " << e.What() << std::endl
				)
			return;
		}

		doc.parse(reinterpret_cast<char*>(rawFile.Buf()));
	}

	{
		pugi::xml_node opts = doc.child(DOptionsTag);
		if(!opts.empty()){
			pugi::xml_attribute numOctavesAttr = opts.attribute(DNumOctavesAttribute);
			if(!numOctavesAttr.empty()){
				ting::uint num = numOctavesAttr.as_uint();
				if(num != 0 && num <= 3)//TODO: check for that inside of this->SetNumOctaves()
					this->SetNumOctaves(num);
			}

			pugi::xml_attribute timbreAttr = opts.attribute(DTimbreAttribute);
			if(!timbreAttr.empty()){
				this->SetTimbre(Preferences::TimbreNameToType(timbreAttr.value()));
			}

			pugi::xml_attribute lockCurOctaveAttr = opts.attribute(DLockCurrentOctaveAttribute);
			if(!timbreAttr.empty()){
				this->SetLockCurrentOctave(lockCurOctaveAttr.as_bool());
			}
		}
	}

	{
		pugi::xml_node tun = doc.child(DTuningTag);
		if(!tun.empty()){
			pugi::xml_attribute basePitchAttr = tun.attribute(DBasePitchAttribute);
			if(!basePitchAttr.empty()){
				this->SetBasePitch(basePitchAttr.as_uint());
			}
		}
		TRACE(<< "Load prefs, new base pitch = " << this->GetBasePitch() << std::endl)
	}

	//Read network settings
	{
		pugi::xml_node opts = doc.child(DNetworkOptionsTag);
		if(!opts.empty()){
			pugi::xml_attribute usernameAttr = opts.attribute(DUsernameAttribute);
			if(!usernameAttr.empty()){
				this->SetUsername(usernameAttr.value());
			}

			pugi::xml_attribute passwordAttr = opts.attribute(DPasswordAttribute);
			if(!passwordAttr.empty()){
				this->SetPassword(passwordAttr.value());
			}

			pugi::xml_attribute autologinAttr = opts.attribute(DAutologinAttribute);
			if(!autologinAttr.empty()){
				this->SetAutologin(autologinAttr.as_bool());
			}

			pugi::xml_attribute broadcastAttr = opts.attribute(DBroadcastAttribute);
			if(!broadcastAttr.empty()){
				this->SetBroadcast(broadcastAttr.as_bool());
			}
		}
	}//~read network settings
}



void Preferences::SaveToFile(){
	pugi::xml_document doc;

	{
		pugi::xml_node n = doc.append_child(pugi::node_declaration);
		n.set_name("xml");
		n.append_attribute("version").set_value("1.0");
	}

	{
		pugi::xml_node opts = doc.append_child();
		opts.set_name(DOptionsTag);
		opts.append_attribute(DNumOctavesAttribute) = this->GetNumOctaves();
		opts.append_attribute(DTimbreAttribute).set_value(
				Preferences::TimbreTypeToName(this->GetTimbre()).c_str()
			);
		opts.append_attribute(DLockCurrentOctaveAttribute) = this->GetLockCurrentOctave();
	}

	{
		pugi::xml_node tun = doc.append_child();
		tun.set_name(DTuningTag);
		tun.append_attribute(DBasePitchAttribute) = this->GetBasePitch();
	}


	//Save network settings
	{
		pugi::xml_node ns = doc.append_child();//create NetworkSettings node
		ns.set_name(DNetworkOptionsTag);
		
		ns.append_attribute(DUsernameAttribute) = ASS(this->GetUsername().c_str());
		ns.append_attribute(DPasswordAttribute) = ASS(this->GetPassword().c_str());
		ns.append_attribute(DAutologinAttribute) = this->GetAutologin();
		ns.append_attribute(DBroadcastAttribute) = this->GetBroadcast();
	}



	class PrefsWriter : public pugi::xml_writer{
		file::FSFile f;
	public:
		PrefsWriter(const std::string& fileName) :
				f(fileName)
		{
			f.Open(file::File::CREATE);
		}

		//override
		void write(const void *data, size_t size){
			ting::Array<ting::byte> buf(size);
			memcpy(buf.Buf(), data, size);
			f.Write(buf);
		}
	} writer(file::File::GetHomeDir() + ".theremin/" + this->prefsFileName);
	
    doc.print(writer);
}
