/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

#include "Preferences.hpp"



#include "stdafx.hpp"
#include "OptionsDialog.hpp"
#include "MainWindow.hpp"
#include "File.hpp"
#include "FSFile.hpp"

#include <ting/Exc.hpp>
#include <ting/math.hpp>
#include <ting/Array.hpp>
#include <ting/Ref.hpp>
#include <pugixml/pugixml.hpp>

#include <cmath>
#include <gtkmm/dialog.h>
#include <gtkmm/main.h>
#include <cstdlib>


MainWindow::MainWindow() :
		isFullscreen(false),
		optionsMenuItem("Options"),
		saveTuningMenuItem("Save tuning"),
		exitMenuItem("Exit"),
		aboutMenuItem("About"),
		optionsDialog(*this),
		aboutDialog(*this),
		prefs(MainWindow::DPreferencesFileName()),
//		numOctaves(1),
//		basePitch(12 * 4),//A4
		tappedX(0)
{
	this->add_events(
			Gdk::BUTTON_PRESS_MASK |
			Gdk::BUTTON_RELEASE_MASK |
			Gdk::POINTER_MOTION_MASK |
			Gdk::BUTTON_MOTION_MASK
		);

	this->signal_button_press_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnButtonPressEvent)
		);
	this->signal_button_release_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnButtonReleaseEvent)
		);
	this->signal_motion_notify_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnMotionEvent)
		);

	//NOTE: if use simple connect() then Arrow Keys press events does not come,
	//this is the bug in Gtkmm. Thus, process arrow keys in OnKeyPress_Notify()
	//while process all other keys in OnKeyPress().
	this->signal_key_press_event().connect_notify(
			sigc::mem_fun(*this, &MainWindow::OnKeyPress_Notify)
		);
	this->signal_key_press_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnKeyPress)
		);
	this->signal_expose_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnExposeEvent)
		);

	//menu
	this->optionsMenuItem.set_size_request(-1, DFingerFriendlyMinSize());
	this->mainMenu.append(this->optionsMenuItem);
	this->optionsMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &MainWindow::ShowOptionsDialog)
		);

	this->saveTuningMenuItem.set_size_request(-1, DFingerFriendlyMinSize());
	this->mainMenu.append(this->saveTuningMenuItem);
	this->saveTuningMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &MainWindow::SavePreferencesToFile)
		);

	this->mainMenu.append(this->menuSeparator2);

	this->exitMenuItem.set_size_request(-1, DFingerFriendlyMinSize());
	this->mainMenu.append(this->exitMenuItem);
	this->exitMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &MainWindow::Exit)
		);

	this->mainMenu.append(this->menuSeparator1);

	this->aboutMenuItem.set_size_request(-1, DFingerFriendlyMinSize());
	this->mainMenu.append(this->aboutMenuItem);
	this->aboutMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &MainWindow::ShowAboutDialog)
		);

	this->mainMenu.show_all();//Show the main menu and all its child widgets.
	this->set_main_menu(this->mainMenu);

	this->show_all_children();//Show all window child widgets

	//set default timbre
	this->ApplyTimbre(Preferences::SINE_PLUS_ADSR);

//	TRACE(<< "MainWindow::ShowOptionsDialog(): loading preferences" << std::endl)
	this->LoadPreferencesFromFile();
};



void MainWindow::Exit(){
	Gtk::Main::quit();
}



void MainWindow::ShowAboutDialog(){
	this->aboutDialog.run();
	this->aboutDialog.hide();
}



void MainWindow::ShowOptionsDialog(){
//	TRACE(<< "MainWindow::ShowOptionsDialog(): enter" << std::endl)
	this->optionsDialog.SetNumOctaves(this->prefs.GetNumOctaves());
	this->optionsDialog.SetTimbre(this->prefs.GetTimbre());
	this->optionsDialog.SetLockToTappedOctave(this->prefs.GetLockCurrentOctave());

	switch(this->optionsDialog.run()){
		case 0: //OK
			this->prefs.SetNumOctaves(this->optionsDialog.GetNumOctaves());
			this->prefs.SetTimbre(this->optionsDialog.GetTimbre());
			this->prefs.SetLockCurrentOctave(this->optionsDialog.GetLockToTappedOctave());
			this->ApplyTimbre(this->prefs.GetTimbre());
			
			this->SavePreferencesToFile();
			break;
		default: //Cancel and all other cases
			this->optionsDialog.SetNumOctaves(this->prefs.GetNumOctaves());
			this->optionsDialog.SetTimbre(this->prefs.GetTimbre());
			this->optionsDialog.SetLockToTappedOctave(this->prefs.GetLockCurrentOctave());
			break;
	}
	
	this->optionsDialog.hide();
	this->queue_draw();//request redrawing entire window
//	TRACE(<< "MainWindow::ShowOptionsDialog(): exit" << std::endl)
};



void MainWindow::ApplyTimbre(Preferences::E_Timbre timbre){
	switch(timbre){
		default:
		case Preferences::SINE_PLUS_ADSR:
			this->instr.Seize(new aumiks::SineWaveInstrument());
			break;
		case Preferences::PLUCKED_STRING:
			this->instr.Seize(new aumiks::PluckedStringInstrument());
			break;
	}
}



Preferences::E_Timbre MainWindow::GetCurrentTimbreType()const{
	if(this->instr.DynamicCast<aumiks::PluckedStringInstrument>().IsValid()){
		return Preferences::PLUCKED_STRING;
	}else{
		return Preferences::SINE_PLUS_ADSR;
	}
}



bool MainWindow::OnExposeEvent(GdkEventExpose* event){
//	TRACE(<< "MainWindow::OnExposeEvent(): enter" << std::endl)

	Glib::RefPtr<Gdk::Window> window = this->get_window();
	if(!window)
		return true;

	Gtk::Allocation allocation = get_allocation();
	const int width = allocation.get_width();
	const int height = allocation.get_height();

	Cairo::RefPtr<Cairo::Context> c = window->create_cairo_context();

	if(event){
		// clip to the area indicated by the expose event so that we only
		// redraw the portion of the window that needs to be redrawn
		c->rectangle(event->area.x, event->area.y, event->area.width, event->area.height);
		c->clip();
	}


	c->set_line_width(1.0);
	c->set_source_rgb(0, 0, 0);//black
	c->select_font_face("Sans", Cairo::FONT_SLANT_NORMAL, Cairo::FONT_WEIGHT_NORMAL);
	c->set_font_size(50.0);

	for(ting::uint i = 0; i < this->prefs.GetNumOctaves(); ++i){
		c->save();
		c->translate(0, (i * height) / this->prefs.GetNumOctaves());
		MainWindow::DrawOctave(
				c,
				width,
				height / this->prefs.GetNumOctaves(),
				this->prefs.GetBasePitch() + (this->prefs.GetNumOctaves() - 1 - i) * 12
			);
		c->restore();
	}

	//draw tapped indicator
//	c->set_line_width(2.0);
//	c->set_source_rgb(0.7f, 0, 0);//darker red
//	c->move_to(this->tappedX, 0);
//	c->line_to(this->tappedX, width);
//	c->stroke();


//	TRACE(<< "MainWindow::OnExposeEvent(): exit" << std::endl)

	return true;
};

static const char* notes[] = {
	"A", 0, "B", "C", 0, "D", 0, "E", "F", 0, "G", 0
};

//static
void MainWindow::DrawOctave(
		Cairo::RefPtr<Cairo::Context> c,
		ting::uint w,
		ting::uint h,
		ting::uint basePitch)
{
	float step = float(w) / 12.0f;
	float curX = 0;
	float height = float(h);
	c->set_source_rgb(0, 0, 0);//black

	for(ting::uint i = 0, n = basePitch % 12; i < 12; ++i, ++n){
		if(n >= 12)
			n -= 12;

		c->move_to(curX, 0);
		c->line_to(curX, height);
		c->stroke();

		ASSERT(n < 12)
		//draw notes
		if(notes[n]){
			Cairo::TextExtents extents;
			c->get_text_extents(notes[n], extents);
			c->move_to(curX + step / 2 - (extents.width / 2) - extents.x_bearing, height / 2);
			c->show_text(notes[n]);
		}

		curX += step;
	}
	c->move_to(0, h);
	c->line_to(w, h);
	c->stroke();

	//draw octave number
	c->set_source_rgb(0.5f, 0, 0);//dark red
	c->move_to(0, h);
	char buf[33];
	snprintf(buf, sizeof(buf)-1, "%u", basePitch / 12);
	buf[32] = 0;
	c->show_text(buf);
};



struct MenuPlacer{
	int menuX, menuY;
	void onPopupMenuPosition(int& x, int& y, bool& pushIn){
		x = this->menuX;
		y = this->menuY;
		pushIn = true;
	}
};



bool MainWindow::OnKeyPress(GdkEventKey* event){
	TRACE(<<"OnKeyPress(): keyval = " << event->keyval << std::endl)
	switch(event->keyval){
		case GDK_F6:
//			TRACE(<<"OnKeyPress(): f6 pressed" << std::endl)
			if(this->isFullscreen)
				this->unfullscreen();
			else
				this->fullscreen();
			this->isFullscreen = !this->isFullscreen;
			break;

		case GDK_Escape:
			{
				Glib::RefPtr<Gdk::Window> window = this->get_window();
				if(!window)
					return false;

				MenuPlacer placer;
				window->get_origin(placer.menuX, placer.menuY);

				this->mainMenu.popup(sigc::mem_fun(placer, &MenuPlacer::onPopupMenuPosition), 0, 0);
			}
			break;
		default:
			return false;
			break;
	}
	return true;
};



void MainWindow::OnKeyPress_Notify(GdkEventKey* event){
	TRACE(<<"OnKeyPress_Notify(): keyval = " << event->keyval << std::endl)
	switch(event->keyval){
		case GDK_Up:
			//DHighestOctave() + 1 is the number of octaves
			if(this->prefs.GetBasePitch() + this->PitchRange() >= DHighestOctave() * 12){
				this->prefs.SetBasePitch(
						(DHighestOctave() + 1 - this->prefs.GetNumOctaves()) * 12
					);
			}else{
				this->prefs.SetBasePitch(
						this->prefs.GetBasePitch() + 12
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Down:
			if(this->prefs.GetBasePitch() <= 12){
				this->prefs.SetBasePitch(0);
			}else{
				this->prefs.SetBasePitch(
						this->prefs.GetBasePitch() - 12
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Left:
			if(this->prefs.GetBasePitch() != 0){
				this->prefs.SetBasePitch(
						this->prefs.GetBasePitch() - 1
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Right:
			//octaves index starts from 0, here need the number of octaves,
			//i.e. DHighestOctave() + 1
			if(this->prefs.GetBasePitch() + this->PitchRange() < 12 * (DHighestOctave() + 1)){
				this->prefs.SetBasePitch(
						this->prefs.GetBasePitch() + 1
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		default:
			break;
	}
};



//static
float MainWindow::CalcFreq(float baseFreq, float numOctavesToShift){
	//formula: freq = baseFreq * 2 ^ numOctavesToShift;
	return baseFreq * ::exp(numOctavesToShift * ting::DLnOf2<float>());
};



//static
ting::u8 MainWindow::CalcVol(float y, float height){
	float factor = (1 - y / height);
	//quadric dependency on factor allows more precise volume control by finger
	return ting::u8(0xff * factor * factor);
};



MainWindow::FreqVol MainWindow::CalcFreqAndVol(
		float width,
		float height,
		float x,
		float y,
		bool isPressEvent)
{
	FreqVol ret;
	x -= (width / 12) / 2;

	float octHeight = height / float(this->prefs.GetNumOctaves());

	//find the screen octave number of the touched octave
	ting::uint touchedOctave = 0;
	for(ting::uint i = 1; i <= this->prefs.GetNumOctaves(); ++i){
		if(y < i * octHeight){
			touchedOctave = i;
			break;
		}
	}

	if(this->prefs.GetLockCurrentOctave()){
		//if press event then update current playing octave, this is to prevent volume leaps
		if(isPressEvent)
			this->curOctave = touchedOctave;
	}else{
		this->curOctave = touchedOctave;
	}

	float pow = 1;
	for(ting::uint j = 0; j < (this->prefs.GetNumOctaves() - this->curOctave); ++j)
		pow *= 2;

	ret.freq = MainWindow::CalcFreq(
			MainWindow::PitchToFreq(this->prefs.GetBasePitch()) * pow,
			x / width
		);

	if(touchedOctave == this->curOctave)
		ret.vol = MainWindow::CalcVol(y - octHeight * (this->curOctave - 1), octHeight);
	else if(touchedOctave < this->curOctave)
		ret.vol = 0xff;//max volume
	else //touchedOctave > this->curOctave
		ret.vol = 0;

	return ret;
}



//static
float MainWindow::PitchToFreq(ting::uint pitch){
	return MainWindow::CalcFreq(27.5f, float(pitch) / 12);
};



bool MainWindow::OnButtonPressEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< this->get_width() << " height = "<< this->get_height() << std::endl)

	this->tappedX = float(event->x);
//	this->queue_draw_area (this->tappedX - 2, 0, this->tappedX + 2, this->get_height());

	FreqVol fv = this->CalcFreqAndVol(
			float(this->get_width()),
			float(this->get_height()),
			float(event->x),
			float(event->y),
			true //press event
		);

	if(this->ch.IsValid())
		this->ch->Stop();

#ifndef M_NO_AUDIO
	this->ch = instr->CreateChannel();
	this->ch->SetFrequency(fv.freq);
	this->ch->SetVolume(fv.vol);
	this->ch->Play();
#endif

	return true;
};



bool MainWindow::OnButtonReleaseEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< window->get_width() << " height = "<< window->get_height() << std::endl)

//	this->queue_draw_area(this->tappedX - 2, 0, this->tappedX + 2, this->get_height());
	this->tappedX = -1;

	if(this->ch.IsValid())
		this->ch->Stop();

	return true;
}



bool MainWindow::OnMotionEvent(GdkEventMotion* event){
	if(event->state & GDK_BUTTON1_MASK){
		this->tappedX = float(event->x);
//		this->queue_draw_area (this->tappedX - 2, 0, this->tappedX + 2, this->get_height());

		if(this->ch.IsValid()){
			FreqVol fv = this->CalcFreqAndVol(
					float(this->get_width()),
					float(this->get_height()),
					float(event->x),
					float(event->y),
					false //not a press event
				);

			this->ch->SetFrequency(fv.freq);
			this->ch->SetVolume(fv.vol);
		}
	}else{
		this->tappedX = -1;
	}
	return true;
}



void MainWindow::LoadPreferencesFromFile(){
	this->prefs.LoadFromFile();

	this->ApplyTimbre(this->prefs.GetTimbre());
}



void MainWindow::SavePreferencesToFile(){
	this->prefs.SaveToFile();
}
