/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
 * aumiks 0.1
 */

#include <list>

#include "aumiks.hpp"

#include <ting/debug.hpp>
#include <ting/types.hpp>
#include <ting/Thread.hpp>
#include <ting/Array.hpp>
#include <ting/Buffer.hpp>

#include "backend/AudioBackend.hpp"
#include "backend/ALSABackend.hpp"
//#include "backend/ESoundBackend.hpp"

using namespace aumiks;

Lib::Lib(ting::uint requestedBufferSizeInFrames) :
		thread(requestedBufferSizeInFrames)
{
	this->thread.Start();
}



Lib::~Lib(){
	//need to stop thread before closing the device
	//because it can attempt write data after the device is closed
	this->thread.Join();
};



void Lib::PlayChannel(ting::Ref<Channel> ch){
	ASSERT(ch.IsValid())
	
	ting::Mutex::LockerUnlocker mut(this->thread.chPoolMutex);
	if(ch->IsPlaying())
		return;//already playing
	this->thread.chPoolToAdd.push_back(ch);//queue channel to be added to playing pool
	ch->isPlaying = true;//mark channel as playing
};



Lib::SoundThread::SoundThread(ting::uint requestedBufferSizeInFrames) :
		desiredBufferSizeInFrames(requestedBufferSizeInFrames)
{
	TRACE(<< "SoundThread(): invoked" << std::endl)
}



static void CopyFromMixBufToPlayBuf(const Array<s32>& mixBuf, Array<ting::byte>& playBuf){
	ASSERT(mixBuf.Size() == playBuf.Size() / 2) //2 bytes per sample

	//TODO: make it work for other bytes per sample values?

	const ting::s32 *src = mixBuf.Buf();
	ting::s16* dst = reinterpret_cast<ting::s16*>(playBuf.Buf());
	for(ting::uint i = 0; i < mixBuf.Size(); ++i){
		ting::s32 tmp = *src;
		if(tmp > 0x7fff)
			tmp = 0x7fff;
		if(tmp < -0x7fff)
			tmp = -0x7fff;

		*dst = ting::s16(tmp);
		++dst;
		++src;
	}
}



void Lib::SoundThread::Run(){
	ting::Ptr<aumiks::AudioBackend> backend(new ALSABackend(this->desiredBufferSizeInFrames));
//	ting::Ptr<aumiks::AudioBackend> backend(new ESoundBackend(this->desiredBufferSizeInFrames));

	ting::Array<s32> mixBuf(backend->BufferSizeInSamples());
	ting::Array<ting::byte> playBuf(backend->BufferSizeInBytes());

	TRACE(<< "mixBuf.Size() = " << mixBuf.Size() << std::endl)

	while(!this->quitFlag){
		//clean mixBuf
		memset(mixBuf.Buf(), 0, mixBuf.SizeInBytes());

		{//add queued channels to playing pool
			ting::Mutex::LockerUnlocker mut(this->chPoolMutex);//lock mutex
//			TRACE(<< "chPoolToAdd.size() = " << this->chPoolToAdd.size() << std::endl)
			while(this->chPoolToAdd.size() != 0){
				this->chPool.push_back(this->chPoolToAdd.front());
				this->chPoolToAdd.pop_front();
			}
		}

		//mix channels to mixbuf
		for(TChIter i = this->chPool.begin(); i != this->chPool.end();){
			if((*i)->MixToMixBuf(mixBuf)){
				(*i)->isPlaying = false;//clear playing flag
				i = this->chPool.erase(i);
			}else
				++i;
		}

//		TRACE(<< "chPool.size() = " << this->chPool.size() << std::endl)
//		TRACE(<< "mixed, mixbuf[0] = "<< this->mixBuf[0] << std::endl)

		CopyFromMixBufToPlayBuf(mixBuf, playBuf);

//		TRACE(<< "SoundThread::Run(): writing data" << std::endl)

		//write data
		backend->Write(playBuf);
	}//~while

	LOG(<< "Sound thread exiting" << std::endl)
};


