/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

#include "File.hpp"


#include "ting/Array.hpp"


#include "stdafx.hpp"
#include "OptionsDialog.hpp"
#include "MainWindow.hpp"
#include "FSFile.hpp"
#include "ting/Exc.hpp"

#include <ting/math.hpp>
#include <pugixml/pugixml.hpp>

#include <cmath>
#include <gtkmm/dialog.h>
#include <gtkmm/main.h>
#include <cstdlib>


MainWindow::MainWindow() :
		sineInstr(new aumiks::SineWaveInstrument),
		isFullscreen(false),
		optionsMenuItem("Options"),
		exitMenuItem("Exit"),
		aboutMenuItem("About"),
		optionsDialog(*this),
		aboutDialog(*this),
		numOctaves(1),
		basePitch(12 * 4)//A4
{
	this->add_events(
			Gdk::BUTTON_PRESS_MASK |
			Gdk::BUTTON_RELEASE_MASK |
			Gdk::POINTER_MOTION_MASK |
			Gdk::BUTTON_MOTION_MASK
		);

	this->signal_button_press_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnButtonPressEvent)
		);
	this->signal_button_release_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnButtonReleaseEvent)
		);
	this->signal_motion_notify_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnMotionEvent)
		);

	//NOTE: if use simple connect() then Arrow Keys press events does not come,
	//this is the bug in Gtkmm. Thus, process arrow keys in OnKeyPress_Notify()
	//while process all other keys in OnKeyPress().
	this->signal_key_press_event().connect_notify(
			sigc::mem_fun(*this, &MainWindow::OnKeyPress_Notify)
		);
	this->signal_key_press_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnKeyPress)
		);
	this->signal_expose_event().connect(
			sigc::mem_fun(*this, &MainWindow::OnExposeEvent)
		);

	//menu
	this->optionsMenuItem.set_size_request(-1, DFingerFriendlyMinSize());
	this->mainMenu.append(this->optionsMenuItem);
	this->optionsMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &MainWindow::ShowOptionsDialog)
		);

	this->exitMenuItem.set_size_request(-1, DFingerFriendlyMinSize());
	this->mainMenu.append(this->exitMenuItem);
	this->exitMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &MainWindow::Exit)
		);

	this->mainMenu.append(this->menuSeparator1);

	this->aboutMenuItem.set_size_request(-1, DFingerFriendlyMinSize());
	this->mainMenu.append(this->aboutMenuItem);
	this->aboutMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &MainWindow::ShowAboutDialog)
		);

	this->mainMenu.show_all();//Show the main menu and all its child widgets.
	this->set_main_menu(this->mainMenu);

	this->show_all_children();//Show all window child widgets

//	TRACE(<< "MainWindow::ShowOptionsDialog(): loading preferences" << std::endl)
	this->LoadPreferencesFromFile();
};



void MainWindow::Exit(){
	Gtk::Main::quit();
}



void MainWindow::ClampBasePitch(){
	if(this->basePitch + 12 * this->numOctaves > (DHighestOctave() + 1) * 12){
		this->basePitch = (DHighestOctave() - (this->numOctaves)) * 12;
	}
}



void MainWindow::ShowAboutDialog(){
	this->aboutDialog.run();
	this->aboutDialog.hide();
}



void MainWindow::ShowOptionsDialog(){
//	TRACE(<< "MainWindow::ShowOptionsDialog(): enter" << std::endl)
	this->optionsDialog.SetNumOctaves(this->numOctaves);

	switch(this->optionsDialog.run()){
		case 0: //OK
			this->numOctaves = this->optionsDialog.GetNumOctaves();
			break;
		default: //Cancel and all other cases
			this->optionsDialog.SetNumOctaves(this->numOctaves);
			break;
	}

	//adjust base pitch
	this->ClampBasePitch();

	this->SavePreferencesToFile();
	
	this->optionsDialog.hide();
	this->queue_draw();//request redrawing entire window
//	TRACE(<< "MainWindow::ShowOptionsDialog(): exit" << std::endl)
};



bool MainWindow::OnExposeEvent(GdkEventExpose* event){
//	TRACE(<< "MainWindow::OnExposeEvent(): enter" << std::endl)

	Glib::RefPtr<Gdk::Window> window = this->get_window();
	if(!window)
		return true;

	Gtk::Allocation allocation = get_allocation();
	const int width = allocation.get_width();
	const int height = allocation.get_height();

	Cairo::RefPtr<Cairo::Context> c = window->create_cairo_context();

	c->set_line_width(1.0);
	c->set_source_rgb(0, 0, 0);//black
	c->select_font_face("Sans", Cairo::FONT_SLANT_NORMAL, Cairo::FONT_WEIGHT_NORMAL);
	c->set_font_size(50.0);

	for(ting::uint i = 0; i < this->numOctaves; ++i){
		c->save();
		c->translate(0, (i * height) / this->numOctaves);
		MainWindow::DrawOctave(
				c,
				width,
				height / this->numOctaves,
				this->basePitch + (this->numOctaves - 1 - i) * 12
			);
		c->restore();
	}

//	TRACE(<< "MainWindow::OnExposeEvent(): exit" << std::endl)

	return true;
};

static const char* notes[] = {
	"A", 0, "B", "C", 0, "D", 0, "E", "F", 0, "G", 0
};

//static
void MainWindow::DrawOctave(
		Cairo::RefPtr<Cairo::Context> c,
		ting::uint w,
		ting::uint h,
		ting::uint basePitch)
{
	float step = float(w) / 12.0f;
	float curX = 0;
	float height = float(h);
	c->set_source_rgb(0, 0, 0);//black

	for(ting::uint i = 0, n = basePitch % 12; i < 12; ++i, ++n){
		if(n >= 12)
			n -= 12;

		c->move_to(curX, 0);
		c->line_to(curX, height);
		c->stroke();

		ASSERT(n < 12)
		//draw notes
		if(notes[n]){
			Cairo::TextExtents extents;
			c->get_text_extents(notes[n], extents);
			c->move_to(curX + step / 2 - (extents.width / 2) - extents.x_bearing, height / 2);
			c->show_text(notes[n]);
		}

		curX += step;
	}
	c->move_to(0, h);
	c->line_to(w, h);
	c->stroke();

	//draw octave number
	c->set_source_rgb(0.5f, 0, 0);//dark red
	c->move_to(0, h);
	char buf[33];
	snprintf(buf, sizeof(buf)-1, "%u", basePitch / 12);
	buf[32] = 0;
	c->show_text(buf);
};



struct MenuPlacer{
	int menuX, menuY;
	void onPopupMenuPosition(int& x, int& y, bool& pushIn){
		x = this->menuX;
		y = this->menuY;
		pushIn = true;
	}
};



bool MainWindow::OnKeyPress(GdkEventKey* event){
	TRACE(<<"OnKeyPress(): keyval = " << event->keyval << std::endl)
	switch(event->keyval){
		case GDK_F6:
//			TRACE(<<"OnKeyPress(): f6 pressed" << std::endl)
			if(this->isFullscreen)
				this->unfullscreen();
			else
				this->fullscreen();
			this->isFullscreen = !this->isFullscreen;
			break;

		case GDK_Escape:
			{
				Glib::RefPtr<Gdk::Window> window = this->get_window();
				if(!window)
					return false;

				MenuPlacer placer;
				window->get_origin(placer.menuX, placer.menuY);

				this->mainMenu.popup(sigc::mem_fun(placer, &MenuPlacer::onPopupMenuPosition), 0, 0);
			}
			break;
		default:
			return false;
			break;
	}
	return true;
};



void MainWindow::OnKeyPress_Notify(GdkEventKey* event){
	TRACE(<<"OnKeyPress_Notify(): keyval = " << event->keyval << std::endl)
	switch(event->keyval){
		case GDK_Up:
			//DHighestOctave() + 1 is the number of octaves
			if(this->basePitch + this->PitchRange() >= DHighestOctave() * 12)
				this->basePitch = (DHighestOctave() + 1 - this->numOctaves) * 12;
			else
				this->basePitch += 12;

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Down:
			if(this->basePitch <= 12)
				this->basePitch = 0;
			else
				this->basePitch -= 12;

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Left:
			if(this->basePitch != 0)
				--this->basePitch;

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Right:
			//octaves index starts from 0, here need the number of octaves,
			//i.e. DHighestOctave() + 1
			if(this->basePitch + this->PitchRange() < 12 * (DHighestOctave() + 1))
				++this->basePitch;

			this->queue_draw();//request redrawing etire window
			break;

		default:
			break;
	}
};



//static
float MainWindow::CalcFreq(float baseFreq, float numOctavesToShift){
	//formula: freq = baseFreq * 2 ^ numOctavesToShift;
	return baseFreq * ::exp(numOctavesToShift * ting::DLnOf2<float>());
};



//static
ting::u8 MainWindow::CalcVol(float y, float height){
	float factor = (1 - y / height);
	//quadric dependency on factor allows more precise volume control by finger
	return ting::u8(0xff * factor * factor);
};



MainWindow::FreqVol MainWindow::CalcFreqAndVol(
		float width,
		float height,
		float x,
		float y,
		bool isPressEvent)
{
	FreqVol ret;
	x -= (width / 12) / 2;

	float octHeight = height / float(this->numOctaves);

	//find the screen octave number of the touched octave
	ting::uint touchedOctave = 0;
	for(ting::uint i = 1; i <= this->numOctaves; ++i){
		if(y < i * octHeight){
			touchedOctave = i;
			break;
		}
	}

	//if press event then update current playing octave, this is to prevent volume leaps
	if(isPressEvent)
		this->curOctave = touchedOctave;

	float pow = 1;
	for(ting::uint j = 0; j < (this->numOctaves - this->curOctave); ++j)
		pow *= 2;

	ret.freq = MainWindow::CalcFreq(
			MainWindow::PitchToFreq(this->basePitch) * pow,
			x / width
		);

	if(touchedOctave == this->curOctave)
		ret.vol = MainWindow::CalcVol(y - octHeight * (this->curOctave - 1), octHeight);
	else if(touchedOctave < this->curOctave)
		ret.vol = 0xff;//max volume
	else //touchedOctave > this->curOctave
		ret.vol = 0;

	return ret;
}



//static
float MainWindow::PitchToFreq(ting::uint pitch){
	return MainWindow::CalcFreq(27.5f, float(pitch) / 12);
};



bool MainWindow::OnButtonPressEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< this->get_width() << " height = "<< this->get_height() << std::endl)

	FreqVol fv = this->CalcFreqAndVol(
			float(this->get_width()),
			float(this->get_height()),
			float(event->x),
			float(event->y),
			true //press event
		);

	if(this->ch.IsValid())
		this->ch->Stop();

#ifndef M_NO_AUDIO
	this->ch = sineInstr->CreateChannel();
	this->ch->SetFrequency(fv.freq);
	this->ch->SetVolume(fv.vol);
	this->ch->Play();
#endif

	return true;
};



bool MainWindow::OnButtonReleaseEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< window->get_width() << " height = "<< window->get_height() << std::endl)

	if(this->ch.IsValid())
		this->ch->Stop();

	return true;
}



bool MainWindow::OnMotionEvent(GdkEventMotion* event){
	if(this->ch.IsValid()){
		FreqVol fv = this->CalcFreqAndVol(
				float(this->get_width()),
				float(this->get_height()),
				float(event->x),
				float(event->y),
				false //not a press event
			);

		this->ch->SetFrequency(fv.freq);
		this->ch->SetVolume(fv.vol);
	}
	return true;
}



void MainWindow::LoadPreferencesFromFile(){
//	TRACE(<< "MainWindow::LoadPreferencesFromFile(): enter" << std::endl)

	file::FSFile f(file::File::GetHomeDir() + ".theremin/" + MainWindow::DPreferencesFileName());

//	TRACE(<< "MainWindow::LoadPreferencesFromFile(): reading file" << std::endl)

	pugi::xml_document doc;
	{
		ting::Array<ting::byte> rawFile;
		try{
			rawFile = f.LoadWholeFileIntoMemory();
		}catch(file::File::Exc &e){
			LOG(
					<< "MainWindow::LoadPreferencesFromFile(): could not load preferences file." << std::endl
					<< "\t" << " e = " << e.What() << std::endl
				)
			return;
		}

		doc.parse(reinterpret_cast<char*>(rawFile.Buf()));
	}

	{
		pugi::xml_node opts = doc.child("Options");
		if(!opts.empty()){
			pugi::xml_attribute numOctavesAttr = opts.attribute("numOctaves");
			if(!numOctavesAttr.empty()){
				ting::uint num = numOctavesAttr.as_uint();
				if(num != 0 && num <= 3)
					this->numOctaves = num;
			}
		}
	}

	{
		pugi::xml_node tun = doc.child("Tuning");
		if(!tun.empty()){
			pugi::xml_attribute basePitchAttr = tun.attribute("basePitch");
			if(!basePitchAttr.empty()){
				this->basePitch = basePitchAttr.as_uint();
				this->ClampBasePitch();
			}
		}
		TRACE(<< "Load prefs, new basePitch = " << this->basePitch << std::endl)
	}
}



void MainWindow::SavePreferencesToFile(){
	pugi::xml_document doc;
	
	{
		pugi::xml_node n = doc.append_child(pugi::node_declaration);
		n.set_name("xml");
		n.append_attribute("version").set_value("1.0");
	}

	{
		pugi::xml_node opts = doc.append_child();
		opts.set_name("Options");
		opts.append_attribute("numOctaves") = this->numOctaves;
	}

	{
		pugi::xml_node tun = doc.append_child();
		tun.set_name("Tuning");
		tun.append_attribute("basePitch") = this->basePitch;
	}

	class PrefsWriter : public pugi::xml_writer{
		file::FSFile f;
	public:
		PrefsWriter() :
				f(file::File::GetHomeDir() + ".theremin/" + MainWindow::DPreferencesFileName())
		{
			f.Open(file::File::CREATE);
		}

		//override
		void write(const void *data, size_t size){
			ting::Array<ting::byte> buf(size);
			memcpy(buf.Buf(), data, size);
			f.Write(buf);
		}
	} writer;
    doc.print(writer);
}
