/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

#ifndef M_File_hpp
#define M_File_hpp

#include <string>

#include <ting/types.hpp>
#include <ting/Buffer.hpp>
#include <ting/Array.hpp>
#include <ting/utils.hpp>
#include <ting/Exc.hpp>

namespace file{

class File{
	std::string path;

	//TODO:
	//add file permissions

protected:
	bool isOpened;

public:
	//define exception class
	class Exc : public ting::Exc{
	public:
		Exc() : ting::Exc("File::Exc: unknown")
		{}

		Exc(const std::string& descr):
			ting::Exc((std::string("File::Exc: ") + descr).c_str())
		{}
	};

	//modes of file opening
	enum EMode{
		UNKNOWN,
		READ,  //Open existing file for read only
		WRITE, //Open existing file for read and write
		CREATE //Create new file and open it for read and write. If file exists it will be replaced by empty file.
			   //This mode is used for C_File::Open() method only. Later the mode is stored as WRITE in the this->mode variable
	};

protected:
	EMode ioMode;
public:

	inline File(const std::string& pathName = std::string()) :
			path(pathName),
			isOpened(false),
			ioMode(UNKNOWN)
	{}

private:
	inline File(const File& f){
		ASSERT(false)//forbidden
	}
public:

	virtual ~File(){}

	inline void SetPath(const std::string& pathName){
		if(this->isOpened)
			throw File::Exc("cannot set path while file is opened");

		this->path = pathName;
	}

	inline const std::string& Path()const{
		return this->path;
	}

	std::string ExtractExtension()const{
		//TODO:
		ASSERT(false)
		return std::string();
	}
	
	virtual void Open(EMode mode) = 0;

	virtual void Close() = 0;
	
	inline bool IsOpened()const{
		return this->isOpened;
	}

	virtual ting::Array<std::string> ListDirFiles(){
		return ting::Array<std::string>();
	}

	//returns number of bytes actually read
	virtual ting::uint Read(
			ting::Buffer<ting::byte>& buf,
			ting::uint numBytesToRead = 0,
			ting::uint offset = 0
		) = 0;

	//returns number of bytes actually written
	virtual ting::uint Write(
			const ting::Buffer<ting::byte>& buf,
			ting::uint numBytesToWrite = 0,
			ting::uint offset = 0
		) = 0;

	virtual void SeekFwd(ting::uint numBytesToSeek){
		throw File::Exc("Seek forward is not supported");
	}

	virtual void MkDir(){
		throw File::Exc("Make directory is not supported");
	}
private:
	static inline ting::uint DReadBlockSize(){
		return 4096;
	}
public:
	ting::Array<ting::byte> LoadWholeFileIntoMemory(){
		if(this->IsOpened())
			throw File::Exc("file should not be opened");

		File::Closer fileCloser(this);//make sure we close the file upon exit from the function

		TRACE(<< "Opening file..." <<std::endl)

		//try to open the file
		this->Open(File::READ);

		TRACE(<< "File opened" <<std::endl)

		//two arrays
		ting::Array<ting::byte> a(DReadBlockSize()); //start with 4kb
		ting::Array<ting::byte> b;

		//two pointers
		ting::Array<ting::byte> *currArr = &a;
		ting::Array<ting::byte> *freeArr = &b;
 
		ting::uint numBytesRead = 0;
		ting::uint numBytesReadLastOp = 0;
		for(;;){
			if( currArr->Size() < (numBytesRead + DReadBlockSize()) ){
				freeArr->Init( currArr->Size() + DReadBlockSize() );
				ASSERT(freeArr->Size() > numBytesRead);
				memcpy(freeArr->Buf(), currArr->Buf(), numBytesRead);
				currArr->Reset();//free memory
				ting::Exchange(currArr, freeArr);
			}

			numBytesReadLastOp = this->Read(*currArr, DReadBlockSize(), numBytesRead);

			numBytesRead += numBytesReadLastOp;//update number of bytes read

			if(numBytesReadLastOp != DReadBlockSize())
				break;
		}//~for
		freeArr->Init(numBytesRead);
		memcpy(freeArr->Buf(), currArr->Buf(), numBytesRead);

		return *freeArr;
	}

	virtual bool Exists()const{
		if(this->Path().size() == 0)
			return false;

		if(this->Path()[this->Path().size() - 1] == '/')
			throw File::Exc("Checking for directory existance is not supported");
		
		if(this->IsOpened())
			return true;

		//try opening and closing the file to find out if it exists or not
		ASSERT(!this->IsOpened())
		try{
			File::Closer fileCloser(const_cast<file::File* const>(this));//make sure we close the file upon exit from try/catch block
			const_cast<file::File* const>(this)->Open(READ);
		}catch(File::Exc &e){
			return false;//file opening failed, assume the file does not exist
		}
		return true;//file open succeeded => file exists
	}
private:

public:
	class Closer{
		File* f;
	  public:
		Closer(File *file) : f(file)
		{
			ASSERT(this->f)
		}
		
		~Closer(){
			f->Close();
		}
	};

	//retuns string of format: "/home/someuser/"
	static std::string GetHomeDir(){
		char * home = getenv("HOME");
		if(!home)
			throw File::Exc("HOME environment variable does not exist");
		std::string ret(home);

		if(ret.size() != 0 && ret[ret.size() - 1] != '/'){
			ret += '/';
		}

		return ret;
	}
};

}//~namespace
#endif//~once
