/*
 *  Copyright (c) 2010 Andry Gunawan <angun33@gmail.com>
 *
 *  Parts of this file are based on Telescope which is
 *  Copyright (c) 2010 Ilya Skriblovsky <Ilya.Skriblovsky@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <libintl.h>
#include <locale.h>

#include "Section.h"
#include "constant.h"
#include "XTools.h"

XftFont *Section::_xftFont = 0;

Section::Section(const gchar *name)
{
    setName(name);
    _partNo = 1;
    _applications = g_ptr_array_new();
}

Section::~Section()
{
    g_log(G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "Section dtor");

    for (guint i = 0; i < _applications->len; i++)
    {
        delete getApplication(i);
    }
    g_ptr_array_free(_applications, TRUE);
    g_free(_name);

    if ( _xftFont != 0 )
    {
        XftFontClose( _dpy, _xftFont );
        _xftFont = 0;
    }
}

void Section::setPart(guint partNo)
{
    _partNo = partNo;
}

guint Section::getPart()
{
    return _partNo;
}

void Section::setName(const gchar *name)
{
    _name = g_strdup(name);
}

const gchar * Section::getName()
{
    return _name;
}

gchar * Section::getNameWithPart()
{
    if (_partNo != 1)
        return g_strdup_printf("%s - part %u", gettext( _name ), _partNo);
    else
        return g_strdup( gettext( _name ) );
}

Application * Section::getApplication(guint index)
{
    return (Application *) g_ptr_array_index(_applications, index);
}

void Section::addApplication(Application *app)
{
    g_ptr_array_add(_applications, (gpointer) app);
}

guint Section::getApplicationsSize()
{
    return _applications->len;
}

void Section::draw(Display *dpy, Pixmap pixmap, Picture picture, int width, int height )
{
    _dpy = dpy;

    // position the applications and draw them
    uint col, row;
    col = row = 0;
    Application * app;

    int xborder = 68;
    int yborder = 70;

    int xmingap = 12;

    uint cols = (width - 2 * xborder + xmingap) / (Application::width() + xmingap);
    int xgap = (width - 2 * xborder - Application::width() * cols) / (cols - 1);

    int ygap = 36;

    for ( uint i = 0; i < _applications->len; i++ )
    {
        if ( col >= cols )
        {
            col = 0;
            row++;
        }

//        if ( row >= NUM_ROWS )
//            break;

        // draw the app icon
        app = getApplication ( i );
        app->setPosition(
            xborder + (col * (Application::width() + xgap)),
            yborder + (row * (Application::height() + ygap))
        );

        // blit the icon into the section pixmap
        XRenderComposite ( _dpy, PictOpOver,
                           app->picture(), None, picture,
                           0, 0,
                           0, 0,
                           app->x(), app->y(),
                           app->width(), app->height());

        col++;
    }

    // draw the section name
    char * sectionName = ( char * ) getNameWithPart();

    if ( _xftFont == 0 )
    {
        _xftFont = XftFontOpen(_dpy, DefaultScreen(_dpy),
            XFT_FAMILY, XftTypeString, "sans",
            XFT_PIXEL_SIZE, XftTypeInteger, 24,
            XFT_WEIGHT, XftTypeInteger, XFT_WEIGHT_BOLD,
            0);
    }

    XGlyphInfo textInfo;
    XftTextExtentsUtf8 ( _dpy, _xftFont, ( const FcChar8* ) sectionName, strlen ( sectionName ), &textInfo );

    XftDraw * xftDraw = XftDrawCreate ( _dpy, pixmap, //XTools::rgbaVisual()->visual,
                                        DefaultVisual(_dpy, DefaultScreen(_dpy)),
                                        DefaultColormap ( _dpy, DefaultScreen ( _dpy ) ) ) ;

    XftColor fontColor;
    fontColor.pixel = 0;

    int titleXPos, titleYPos;
    titleXPos = ( width - textInfo.width ) / 2;
    titleYPos = ( 70 - textInfo.height ) / 2;

    fontColor.color.red = fontColor.color.green = fontColor.color.blue = 0;
    fontColor.color.alpha = 0x3FFF; // 0xFFFF * 25%
    XftDrawStringUtf8 ( xftDraw, &fontColor, _xftFont,
                        titleXPos + 1, titleYPos + textInfo.y + 2,
                        ( const FcChar8* ) sectionName,
                        strlen ( sectionName )
                      );
    fontColor.color.red = fontColor.color.green = fontColor.color.blue = 0;
    fontColor.color.alpha = 0x4CCC; // 0xFFFF * 30%
    XftDrawStringUtf8 ( xftDraw, &fontColor, _xftFont,
                        titleXPos + 2, titleYPos + textInfo.y + 2,
                        ( const FcChar8* ) sectionName,
                        strlen ( sectionName )
                      );
    fontColor.color.red = fontColor.color.green = fontColor.color.blue = 0;
    fontColor.color.alpha = 0x9999; // 0xFFFF * 60%;
    XftDrawStringUtf8 ( xftDraw, &fontColor, _xftFont,
                        titleXPos + 1, titleYPos + textInfo.y + 1,
                        ( const FcChar8* ) sectionName,
                        strlen ( sectionName )
                      );
    fontColor.color.red = fontColor.color.green = fontColor.color.blue = 0xFFFF;
    fontColor.color.alpha = 0xFFFF;
    XftDrawStringUtf8 ( xftDraw, &fontColor, _xftFont,
                        titleXPos, titleYPos + textInfo.y,
                        ( const FcChar8* ) sectionName,
                        strlen ( sectionName )
                      );

    // cleanup
    XftDrawDestroy ( xftDraw );
    g_free ( sectionName );
}

