//
// Telescope - graphical task switcher
//
// (c) Ilya Skriblovsky, 2010
// <Ilya.Skriblovsky@gmail.com>
//

// $Id: Resources.cpp 145 2010-08-09 10:07:51Z mitrandir $

#include "Resources.h"

#include <stdio.h>
#include <string.h>

#include <Imlib2.h>

#include <sys/stat.h>

#include "XTools.h"
#include "constant.h"
#include "Settings.h"

Resources* Resources::_instance = 0;


Resources::Resources(Display *dpy): _dpy(dpy)
{
    if (_instance != 0)
        printf("Resources singleton created twice!\n");

    _instance = this;

    _bgPixmap = 0;

    Window rootWindow = XTools::rootWindow();

// Loading header images

    _indicatorWidth = _indicatorHeight = 12;
    _currentIndicatorWidth = _currentIndicatorHeight = 18;

    _navigationWidth  = 55;
    _navigationHeight = 90;

    bool ok = true;
    ok = ok && loadARGBImage(
        Settings::instance()->headerLeftFilename(),
        rootWindow,
        &_headerLeftPix,
        &_headerLeftPict,
        &_headerLeftWidth,
        &_headerHeight);
    ok = ok && loadARGBImage(
        Settings::instance()->headerRightFilename(),
        rootWindow,
        &_headerRightPix,
        &_headerRightPict,
        &_headerRightWidth,
        0);
    ok = ok && loadARGBImage(
        Settings::instance()->headerMiddleFilename(),
        rootWindow,
        &_headerMiddlePix,
        &_headerMiddlePict,
        0, 0);

    ok = ok && loadARGBImage(
        Settings::instance()->headerLeftSelectedFilename(),
        rootWindow,
        &_headerLeftSelectedPix,
        &_headerLeftSelectedPict,
        0, 0);
    ok = ok && loadARGBImage(
        Settings::instance()->headerRightSelectedFilename(),
        rootWindow,
        &_headerRightSelectedPix,
        &_headerRightSelectedPict,
        0, 0);
    ok = ok && loadARGBImage(
        Settings::instance()->headerMiddleSelectedFilename(),
        rootWindow,
        &_headerMiddleSelectedPix,
        &_headerMiddleSelectedPict,
        0, 0);

    ok = ok && loadARGBImage(
        Settings::instance()->brokenPatternFilename(),
        rootWindow,
        &_brokenPatternPix,
        &_brokenPatternPict,
        0, 0);

    ok = ok && loadARGBImage (
             SHARE_PATH"navigation.png",
             XTools::rootWindow(),
             &_navigatorPixmap,
             &_navigatorPicture,
             0, 0);
    ok = ok && loadARGBImage (
             SHARE_PATH"indicator.png",
             XTools::rootWindow(),
             &_indicatorPixmap,
             &_indicatorPicture,
             0, 0);
    ok = ok && loadARGBImage (
             SHARE_PATH"text-background.png",
             XTools::rootWindow(),
             &_textBgPixmap,
             &_textBgPicture,
             &_textBgWidth,
             &_textBgHeight );

    if (! ok)
        printf("Cannot load all pixmaps\n");


    XRenderPictureAttributes pictureAttrs;
    pictureAttrs.repeat = RepeatNormal;
    XRenderChangePicture(_dpy, _headerMiddlePict, CPRepeat, &pictureAttrs);
    XRenderChangePicture(_dpy, _headerMiddleSelectedPict, CPRepeat, &pictureAttrs);
    XRenderChangePicture(_dpy, _brokenPatternPict, CPRepeat, &pictureAttrs);


    XRenderParseColor(_dpy, Settings::instance()->borderColor(), &_borderColor);
    XRenderParseColor(_dpy, Settings::instance()->borderActiveColor(), &_borderActiveColor);

    reloadBackground();
}


Resources::~Resources()
{
    XRenderFreePicture(_dpy, _headerLeftPict);
    XRenderFreePicture(_dpy, _headerRightPict);
    XRenderFreePicture(_dpy, _headerMiddlePict);
    XRenderFreePicture(_dpy, _headerLeftSelectedPict);
    XRenderFreePicture(_dpy, _headerRightSelectedPict);
    XRenderFreePicture(_dpy, _headerMiddleSelectedPict);
    XFreePixmap(_dpy, _headerLeftPix);
    XFreePixmap(_dpy, _headerRightPix);
    XFreePixmap(_dpy, _headerMiddlePix);
    XFreePixmap(_dpy, _headerLeftSelectedPix);
    XFreePixmap(_dpy, _headerRightSelectedPix);
    XFreePixmap(_dpy, _headerMiddleSelectedPix);


    XRenderFreePicture ( _dpy, _indicatorPicture );
    XRenderFreePicture ( _dpy, _navigatorPicture );
    XRenderFreePicture ( _dpy, _textBgPicture );

    XFreePixmap ( _dpy, _indicatorPixmap );
    XFreePixmap ( _dpy, _navigatorPixmap );
    XFreePixmap ( _dpy, _textBgPixmap );

    if (_bgPixmap)
        XFreePixmap(_dpy, _bgPixmap);
}


bool Resources::loadARGBImage(
    const char *filename,
    Drawable parent,
    Pixmap *pixmap,
    Picture *picture,
    int *width,
    int *height)
{
    static Visual *defaultVisual = 0;
    static XVisualInfo rgbaVisual;
    static XRenderPictFormat *defaultFormat;
    static XRenderPictFormat *rgbaFormat;
    static Colormap colormap;

    *pixmap = 0;
    *picture = 0;

    if (defaultVisual == 0)
    {
        // Initializing

        int scr = DefaultScreen(_dpy);
        defaultVisual = DefaultVisual(_dpy, scr);
        colormap = DefaultColormap(_dpy, scr);
        defaultFormat = XRenderFindVisualFormat(_dpy, defaultVisual);

        if (XMatchVisualInfo(_dpy, scr, 32, TrueColor, &rgbaVisual) == 0)
        {
            fprintf(stderr, "Cannot find rgba visual\n");
            return false;
        }
        rgbaFormat = XRenderFindVisualFormat(_dpy, rgbaVisual.visual);
    }


    Imlib_Image image = imlib_load_image(filename);
    if (image == 0)
        return false;

    imlib_context_set_image(image);
    int w = imlib_image_get_width();
    int h = imlib_image_get_height();

    if (width) *width = w;
    if (height) *height = h;

    // Making image with premultiplied alpha
    Imlib_Image premul = imlib_create_image(w, h);
    imlib_context_set_image(premul);
    imlib_context_set_color(0, 0, 0, 255);
    imlib_image_fill_rectangle(0, 0, w, h);
    imlib_context_set_blend(1);
    imlib_blend_image_onto_image(image, 0, 0, 0, w, h, 0, 0, w, h);
    imlib_image_copy_alpha_to_image(image, 0, 0);

    *pixmap = XCreatePixmap(_dpy, parent, w, h, 32);
    imlib_context_set_display(_dpy);
    imlib_context_set_colormap(colormap);
    imlib_context_set_visual(rgbaVisual.visual);
    imlib_context_set_drawable(*pixmap);
    imlib_context_set_blend(0);
    imlib_render_image_on_drawable(0, 0);

    imlib_free_image();
    imlib_context_set_image(image);
    imlib_free_image();

    *picture = XRenderCreatePicture(_dpy, *pixmap, rgbaFormat, 0, 0);

    return true;
}


void Resources::drawIndicator ( Picture canvas, int x, int y )
{
    XRenderComposite ( _dpy, PictOpOver,
                       _indicatorPicture, None, canvas,
                       _currentIndicatorWidth, 0,
                       0, 0,
                       x, y,
                       _indicatorWidth, // width
                       _indicatorHeight // height
                     );
}

void Resources::drawCurrentIndicator ( Picture canvas, int x, int y )
{
    XRenderComposite ( _dpy, PictOpOver,
                       _indicatorPicture, None, canvas,
                       0, 0,
                       0, 0,
                       x, y,
                       _currentIndicatorWidth, // width
                       _currentIndicatorHeight // height
                     );
}

void Resources::drawNextNavigation ( Picture canvas, int x, int y )
{
    XRenderComposite ( _dpy, PictOpOver,
                       _navigatorPicture, None, canvas,
                       _navigationWidth, 0,
                       0, 0,
                       x, y,
                       _navigationWidth, // width
                       _navigationHeight // height
                     );
}

void Resources::drawPrevNavigation ( Picture canvas, int x, int y )
{
    XRenderComposite ( _dpy, PictOpOver,
                       _navigatorPicture, None, canvas,
                       0, 0,
                       0, 0,
                       x, y,
                       _navigationWidth, // width
                       _navigationHeight // height
                     );
}

void Resources::drawNextPressedNavigation ( Picture canvas, int x, int y )
{
    XRenderComposite ( _dpy, PictOpOver,
                       _navigatorPicture, None, canvas,
                       _navigationWidth, _navigationHeight,
                       0, 0,
                       x, y,
                       _navigationWidth, // width
                       _navigationHeight // height
                     );
}

void Resources::drawPrevPressedNavigation ( Picture canvas, int x, int y )
{
    XRenderComposite ( _dpy, PictOpOver,
                       _navigatorPicture, None, canvas,
                       0, _navigationHeight,
                       0, 0,
                       x, y,
                       _navigationWidth, // width
                       _navigationHeight // height
                     );
}



void Resources::reloadBackground()
{
    if (_bgPixmap != 0)
        XFreePixmap(_dpy, _bgPixmap);

    Window rootWindow = XTools::rootWindow();
    XWindowAttributes attrs;
    XGetWindowAttributes(_dpy, rootWindow, &attrs);
    int width = attrs.width;
    int height = attrs.height;

    _bgPixmap = XCreatePixmap(_dpy, rootWindow,
        width, height,
        attrs.depth);

    printf("Loading background from '%s'\n", Settings::instance()->backgroundFilename());
    printf("Background mode: %d\n", Settings::instance()->backgroundMode());

    int bgXpos = 0;
    int bgYpos = 0;
    Imlib_Image background = imlib_load_image(Settings::instance()->backgroundFilename());
    if (background == 0)
    {
        printf("Cannot load background\n");
        printf("rgb = (%d, %d, %d)\n",
            Settings::instance()->backgroundR(),
            Settings::instance()->backgroundG(),
            Settings::instance()->backgroundB());
        background = imlib_create_image(width, height);
        imlib_context_set_image(background);
        imlib_context_set_color(
            Settings::instance()->backgroundR(),
            Settings::instance()->backgroundG(),
            Settings::instance()->backgroundB(),
            255);
        imlib_image_fill_rectangle(0, 0, width, height);
    }
    else
    {
        Imlib_Image bgToDraw = 0;

        imlib_context_set_image(background);
        switch (Settings::instance()->backgroundMode())
        {
            case Settings::Stretched:
            {
                bgToDraw = imlib_create_cropped_scaled_image(
                    0, 0,
                    imlib_image_get_width(), imlib_image_get_height(),
                    width, height
                );
                break;
            }

            case Settings::Centered:
            {
                // Hildon's logic is not so easy in this case

//                bgToDraw = imlib_create_cropped_image(
//                    (imlib_image_get_width() - width) / 2,
//                    (imlib_image_get_height()- height) / 2,
//                    width, height
//                );

                double scalex = 1.0;
                double scaley = 1.0;

                if (imlib_image_get_width() > width * 2)
                    scalex = (double)imlib_image_get_width() / 2.0 / width;
                if (imlib_image_get_height() > height * 2)
                    scaley = (double)imlib_image_get_height() / 2.0 / height;

                double scale = scalex;
                if (scaley > scale)
                    scale = scaley;

                int newwidth = (int)(scale * width);
                int newheight= (int)(scale * height);

                bgToDraw = imlib_create_cropped_scaled_image(
                    (imlib_image_get_width() - newwidth) / 2,
                    (imlib_image_get_height()- newheight)/ 2,
                    newwidth, newheight,
                    width, height
                );

                break;
            }

            case Settings::Scaled: case Settings::Cropped:
            {
                double scalex = (double)width / imlib_image_get_width();
                double scaley = (double)height / imlib_image_get_height();
                double scale = scalex;

                if (Settings::instance()->backgroundMode() == Settings::Scaled)
                {
                    if (scaley < scale) scale = scaley;
                }
                else
                {
                    if (scaley > scale) scale = scaley;
                }

                int newwidth = (int)(imlib_image_get_width() * scale);
                int newheight= (int)(imlib_image_get_height() * scale);

                bgToDraw = imlib_create_cropped_scaled_image(
                    0, 0,
                    imlib_image_get_width(), imlib_image_get_height(),
                    newwidth, newheight
                );

                bgXpos = (width - newwidth) / 2;
                bgYpos = (height - newheight) / 2;
                break;
            }

            default:
                printf("Unknown background mode: %d\n", Settings::instance()->backgroundMode());
        }

        imlib_free_image();

        imlib_context_set_image(bgToDraw);
    }

    imlib_context_set_drawable(_bgPixmap);
    imlib_context_set_visual(DefaultVisual(_dpy, DefaultScreen(_dpy)));

    imlib_render_image_on_drawable(bgXpos, bgYpos);

    imlib_free_image();
}
