/*
 *  Copyright (c) 2010 Andry Gunawan <angun33@gmail.com>
 *
 *  Parts of this file are based on Telescope which is
 *  Copyright (c) 2010 Ilya Skriblovsky <Ilya.Skriblovsky@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <sys/time.h>
#include <X11/Xatom.h>
#include <X11/keysymdef.h>

#include "LauncherWindow.h"
#include "DBus.h"
#include "Resources.h"
#include "MenuReader.h"
#include "XTools.h"
#include "constant.h"

#include "XEventLoop.h"


LauncherWindow* LauncherWindow::_instance = 0;


LauncherWindow::LauncherWindow ( Display *dpy, SectionList *list )
        :_dpy ( dpy ), _sections ( list )
{
    LauncherWindow::_instance = this;


    int scr = DefaultScreen ( _dpy );
    _rootWindow = RootWindow ( _dpy, scr );
    int depth = DefaultDepth ( _dpy, scr );
    Visual *visual = DefaultVisual ( _dpy, scr );

    _width = DisplayWidth(_dpy, scr);
    _height = DisplayHeight(_dpy, scr);

    _prevButtonPressed = false;
    _nextButtonPressed = false;

    // Creating main window
    XSetWindowAttributes createAttrs;
    createAttrs.save_under = True; // This seems to not work, but let's keep it

    _win = XCreateWindow ( _dpy, _rootWindow, 0, 0, _width, _height,
                           0, depth, InputOutput, visual,
                           CWSaveUnder,
                           &createAttrs );

    _gc = XCreateGC ( _dpy, _win, 0, 0 );
    XSetGraphicsExposures ( _dpy, _gc, false );

    XSelectInput ( _dpy, _win,
                   ExposureMask           |
                   ButtonPressMask        |
                   ButtonReleaseMask      |
                   StructureNotifyMask    |
                   KeyPressMask           |
                   KeyReleaseMask         |
                   ButtonMotionMask
                 );


    // Setting Window type as Splash. This tells window manager
    // to not try to decorate our window.
    Atom windowType = XTools::_NET_WM_WINDOW_TYPE_SPLASH;
    XChangeProperty ( _dpy, _win, XTools::_NET_WM_WINDOW_TYPE,
                      XA_ATOM, 32, PropModeReplace,
                      ( const unsigned char* ) &windowType, 1 );

    XSelectInput ( _dpy, _rootWindow, StructureNotifyMask | SubstructureNotifyMask | PropertyChangeMask );

    // Double buffering pixmap
    _bufferPix = 0;
    recreateBufferPixmap();
    redrawSections();

    XRenderPictureAttributes pa;
    pa.subwindow_mode = IncludeInferiors;

    _canvas = XRenderCreatePicture ( _dpy, _win, XTools::xrenderFormat(), CPSubwindowMode, &pa );


    _shown = false;

    _currentSection = 0;

    _repaintOnIdle = false;

	_sliderStart = -1;
    XEventLoop::instance()->addHandler(this);
    XEventLoop::instance()->addIdleTask(this);
}

LauncherWindow::~LauncherWindow()
{
    delete _sections;

    XRenderFreePicture ( _dpy, _canvas );
    XRenderFreePicture ( _dpy, _bufferCanvas );

    XFreePixmap ( _dpy, _bufferPix );

    XFreeGC ( _dpy, _gc );
    XDestroyWindow ( _dpy, _win );
}

Display* LauncherWindow::display()
{
    return _dpy;
}


Window LauncherWindow::window()
{
    return _win;
}


void LauncherWindow::show()
{
    _shown = true;

    XMapWindow ( _dpy, _win );

    XTools::switchToWindow ( _win );
}

void LauncherWindow::hide()
{
    XUnmapWindow ( _dpy, _win );

    _shown = false;
}

bool LauncherWindow::shown()
{
    return _shown;
}

void LauncherWindow::quit()
{
    hide();
    _repaintOnIdle = false;
    _breakEventLoop = true;
}

void LauncherWindow::_goPrevious()
{
    _prevButtonPressed = false;

    // go to previous section
    if ( _currentSection == 0 )
        _currentSection = _sections->getSize() - 1;
    else
        _currentSection--;

    if ( _shown )
        _repaintOnIdle = true;
}

void LauncherWindow::_goNext()
{
    _nextButtonPressed = false;
    // go to next section
    if ( _currentSection == _sections->getSize() - 1 )
        _currentSection = 0;
    else
        _currentSection++;

    if ( _shown )
        _repaintOnIdle = true;
}

void LauncherWindow::redrawSections()
{
//    for ( uint i = 0; i < _sections->getSize(); i++ )
//    {
//        printf("drawing section %d\n", i);
//        _sections->get(i)->redraw(_dpy, _width, _height);
//    }
}


void LauncherWindow::onEvent(XEvent *event)
{
    if ( event->xany.window == _rootWindow )
    {
        _onRootWinEvent(event);
    }
    else if ( event->xany.window == _win )
    {
        _onWinEvent(event);
    }
}

void LauncherWindow::_onRootWinEvent(XEvent *event)
{
    if ( event->type == ConfigureNotify )
    {
        if (
            event->xconfigure.window == _rootWindow &&
            (event->xconfigure.width != _width || event->xconfigure.height != _height)
           )
        {
            setNewSize(event->xconfigure.width, event->xconfigure.height);
        }
        else if ( event->xconfigure.above == _win  )
        {
            hide();
        }
    }
}

void LauncherWindow::setNewSize(int width, int height)
{
    _width = width;
    _height = height;
    XMoveResizeWindow(_dpy, _win, 0, 0, _width, _height);

    recreateBufferPixmap();
}

void LauncherWindow::_onKeyPress(XKeyEvent *event)
{
    // Redraw the left or right image as being pressed
    if ( event->keycode == XKeysymToKeycode ( _dpy, XK_Left ) )
    {
        _prevButtonPressed = true;
        if ( _shown )
            _repaintOnIdle = true;
    }
    else if ( event->keycode == XKeysymToKeycode ( _dpy, XK_Right ) )
    {
        _nextButtonPressed = true;
        if ( _shown )
            _repaintOnIdle = true;
    }
}

void LauncherWindow::_onKeyRelease(XKeyEvent *event)
{
    if ( event->keycode == XKeysymToKeycode ( _dpy, XK_Left ) )
        _goPrevious();
    else if ( event->keycode == XKeysymToKeycode ( _dpy, XK_Right ) )
        _goNext();
    else if ( _shown )
        hide();
}

void LauncherWindow::_onMotion(XMotionEvent *event)
{
    if ( _sliderStart >= 0 )
    {
        int distance = event->x - _sliderStart;
        int goTo = ( ( int ) _sliderStartSection + ( int ) floor ( distance / 50 ) ) % ( int ) _sections->getSize();

        if ( goTo < 0 )
            _currentSection = _sections->getSize() + goTo;
        else
            _currentSection = goTo;

        if ( _currentSection != _sliderStartSection )
        {
            // printf("distance: %d | start section: %d | current: %d | goto: %d\n", distance, _sliderStartSection, _currentSection, goTo);
            _repaintOnIdle = true;
        }
    }
}

void LauncherWindow::_onButtonEvent(XEvent *event)
{
    int navHeight = Resources::instance()->getNavigationHeight();
    if ( event->xbutton.x >= 0 && event->xbutton.x <= 68 && event->xbutton.y >= ( _height - navHeight ) / 2 && event->xbutton.y <= ( ( _height - navHeight ) / 2 ) + navHeight )
    {
        if ( event->type == ButtonPress )
        {
            _prevButtonPressed = true;
            if ( _shown )
                _repaintOnIdle = true;
        }
        else
            _goPrevious();
    }
    else if ( event->xbutton.x >= _width - 68 && event->xbutton.x <= _width && event->xbutton.y >= ( _height - navHeight ) / 2 && event->xbutton.y <= ( ( _height - navHeight ) / 2 ) + navHeight )
    {
        if ( event->type == ButtonPress )
        {
            _nextButtonPressed = true;
            if ( _shown )
                _repaintOnIdle = true;
        }
        else
            _goNext();
    }
    else if ( event->xbutton.y >= _height - 50 )
    {
        if ( event->type == ButtonPress)
        {
            _sliderStart = event->xbutton.x;
            _sliderStartSection = _currentSection;
        }
        else
            _sliderStart = -1;
    }
    else if (event->type == ButtonRelease)
    {
        if (_sliderStart != -1)
        {
            _sliderStart = -1;
            return;
        }

        // loop thru apps
        Section *currentSection = _sections->get ( _currentSection );
        bool aHit = FALSE;
        bool success;
        Application *app;
        for ( uint i = 0; i < currentSection->getApplicationsSize(); i++ )
        {
            app = currentSection->getApplication ( i );
            aHit = app->isAHit ( event->xbutton.x, event->xbutton.y ) ;
            if ( aHit )
            {
                success = app->execute();

                if (success)
                    showNotification( g_strconcat ("Starting ", app->getApplicationName(), NULL) );
                else
                    showNotification( "Execution failed" );
                break;
            }
        }

        if ( _shown )
            hide();
    }
}

void LauncherWindow::_onWinEvent(XEvent *event)
{
    if ( event->type == KeyPress )
    {
        _onKeyPress( & event->xkey );
    }
    else if ( event->type == KeyRelease )
    {
        _onKeyRelease( & event->xkey );
    }
    else if ( event->type == MotionNotify )
    {
        _onMotion( & event->xmotion );
    }
    else if ( event->type == ButtonPress || event->type == ButtonRelease )
    {
        _onButtonEvent(event);
    }
    else if ( event->type == Expose && event->xexpose.count < 1 )
        _repaintOnIdle = true;
}

void LauncherWindow::showNotification(const char *message)
{
    DBusMessage *call = dbus_message_new_method_call(
        "org.freedesktop.Notifications",
        "/org/freedesktop/Notifications",
        "org.freedesktop.Notifications",
        "SystemNoteInfoprint"
    );
    
    dbus_message_append_args (call,
                              DBUS_TYPE_STRING, &message,
                              DBUS_TYPE_INVALID);

    dbus_connection_send(DBus::instance()->getConnection(), call, 0);
    
    dbus_message_unref(call);
}

void LauncherWindow::paint()
{
    if ( ! _shown )
        return;

//    timeval before;
//    gettimeofday(&before, 0);

    // draw background
    XCopyArea ( _dpy, Resources::instance()->getBackgroundPixmap(), _bufferPix, _gc,
                0, 0, _width, _height, 0, 0
              );

    Resources * resources = Resources::instance();
	// draw the current section
    Section *currentSection = _sections->get ( _currentSection );
//    XRenderComposite ( _dpy, PictOpOver,
//                       currentSection->getPicture(), None, _bufferCanvas,
//                       0, 0, 0, 0, 0, 0,
//                       _width, _height);
    currentSection->draw(_dpy, _bufferPix, _bufferCanvas, _width, _height);

    // draw previous and next button
    if ( _prevButtonPressed )
        resources->drawPrevPressedNavigation ( _bufferCanvas, 0, ( _height - resources->getNavigationHeight() ) / 2 );
    else
        resources->drawPrevNavigation ( _bufferCanvas, 0, ( _height - resources->getNavigationHeight() ) / 2 );

    if ( _nextButtonPressed )
        resources->drawNextPressedNavigation ( _bufferCanvas,
                                              _width - resources->getNavigationWidth(),
                                              ( _height - resources->getNavigationHeight() ) / 2 );
    else
        resources->drawNextNavigation ( _bufferCanvas,
                                       _width - resources->getNavigationWidth(),
                                       ( _height - resources->getNavigationHeight() ) / 2 );

    // draw page indicator
    int indicatorTtlWidth = ( _sections->getSize() * resources->getIndicatorWidth() ) + ( ( _sections->getSize() - 1 ) * 10 );
    for ( uint i = 0; i < _sections->getSize(); i++ )
    {
        if ( i == _currentSection )
        {
            resources->drawCurrentIndicator ( _bufferCanvas,
                                             ( ( _width - indicatorTtlWidth ) / 2 ) + ( i * ( 10 + resources->getIndicatorWidth() ) ) - ( ( resources->getCurrentIndicatorWidth() - resources->getIndicatorWidth() ) / 2 ),
                                             _height - 10 - resources->getIndicatorHeight() - ( ( resources->getCurrentIndicatorHeight() - resources->getIndicatorHeight() ) / 2 )
                                           );
        }
        else
        {
            resources->drawIndicator ( _bufferCanvas,
                                      ( ( _width - indicatorTtlWidth ) / 2 ) + ( i * ( 10 + resources->getIndicatorWidth() ) ),
                                      _height - 10 - resources->getIndicatorHeight()
                                    );
        }
    }

    blitBuffer();


//    timeval after;
//    gettimeofday(&after, 0);
//    printf("---------\n");
//    printf("%ld:%ld\n", before.tv_sec, before.tv_usec);
//    printf("%ld:%ld\n", after.tv_sec, after.tv_usec);
}

void LauncherWindow::blitBuffer()
{
    XCopyArea ( _dpy, _bufferPix, _win, _gc,
                0, 0,
                _width, _height,
                0, 0
              );
}


void LauncherWindow::onIdle()
{
    if ( _repaintOnIdle )
    {
        paint();
        _repaintOnIdle = false;
    }

    // check if the application menu file has been changed
    if ( ! _shown && MenuReader::getInstance()->hasChange())
    {
        printf("menu file has changed\n");
        delete _sections;

        _sections = MenuReader::getInstance()->processMenu();
        _currentSection = 0;

        redrawSections();
    }
}

void LauncherWindow::recreateBufferPixmap()
{
    if ( _bufferPix )
    {
        XRenderFreePicture ( _dpy, _bufferCanvas );
        XFreePixmap ( _dpy, _bufferPix );
    }

    int scr = DefaultScreen ( _dpy );
    int depth = DefaultDepth ( _dpy, scr );


    // Double-buffering pixmap
    _bufferPix = XCreatePixmap ( _dpy, _rootWindow, _width, _height, depth );

    // XRender picture
    _bufferCanvas = XRenderCreatePicture ( _dpy, _bufferPix, XTools::xrenderFormat(), 0, 0 );
}
