#include "Thumbnail.h"

#include <math.h>
#include <stdlib.h>

#include "TeleWindow.h"
#include "XTools.h"


Thumbnail::Thumbnail(TeleWindow *teleWindow, Window clientWindow)
{
    _teleWindow = teleWindow;
    _dpy = teleWindow->display();
    _clientWindow = clientWindow;

    _depth = DefaultDepth(_dpy, DefaultScreen(_dpy));

    _title = XTools::windowTitle_alloc(_clientWindow);

    _clientDestroyed = false;

    XSelectInput(_dpy, _clientWindow, StructureNotifyMask | PropertyChangeMask);


    _damage = XDamageCreate(_dpy, _clientWindow, XDamageReportNonEmpty);


    XWindowAttributes attrs;
    XGetWindowAttributes(_dpy, _clientWindow, &attrs);

#ifdef KDE
    Window root;
    Window parent;
    Window *children;
    unsigned int nchildren;
    XQueryTree(_dpy, _clientWindow, &root, &parent, &children, &nchildren);
    XFree(children);

    XWindowAttributes decoAttrs;
    XGetWindowAttributes(_dpy, parent, &decoAttrs);

    _clientDecoX = decoAttrs.x;
    _clientDecoY = decoAttrs.y;
#else
    #ifdef MAEMO
        _clientDecoX = attrs.x;
        _clientDecoY = attrs.y;
    #else
        #error Unknown window manager
    #endif
#endif


    _cachePixmap = 0;
    _cachePicture = 0;
    _cacheValid = false;


    // First setGeometry call will compare this with new dimensions
    _width = -1;
    _height = -1;


    _minimized = false;


    XRenderPictFormat *format = XRenderFindVisualFormat(_dpy, attrs.visual);

    XRenderPictureAttributes pa;
    pa.subwindow_mode = IncludeInferiors;

    _clientPict = XRenderCreatePicture(_dpy, _clientWindow, format, CPSubwindowMode, &pa);
}

Thumbnail::~Thumbnail()
{
    XFreePixmap(_dpy, _cachePixmap);
    XRenderFreePicture(_dpy, _cachePicture);

    if (! _clientDestroyed)
    {
        XSelectInput(_dpy, _clientWindow, 0);
        XDamageDestroy(_dpy, _damage);
        XRenderFreePicture(_dpy, _clientPict);
    }

    free(_title);
}


Window Thumbnail::clientWindow()
{
    return _clientWindow;
}


char* Thumbnail::title()
{
    return _title;
}


void Thumbnail::setGeometry(int x, int y, int w, int h)
{
    _x = x;
    _y = y;

    if (_width != w || _height != h)
    {
        _width = w;
        _height = h;
        onResize();
    }
    else
    {
        _width = w;
        _height = h;
    }
}

bool Thumbnail::inside(int x, int y)
{
    return x >= _x && y >= _y && x < _x + _width && y < _y + _height;
}

void Thumbnail::onClientEvent(XEvent *event)
{
    if (event->type == XTools::damageEventBase() + XDamageNotify)
    {
        _cacheValid = false;
        if (_teleWindow->shown())
        {
            paint();
            _teleWindow->onThumbRedrawed(this);
        }

        XDamageSubtract(_dpy, ((XDamageNotifyEvent*)event)->damage, None, None);
    }
    else if (event->type == ConfigureNotify)
    {
        onClientResize(event);
    }
//    else if (event->type == UnmapNotify)
//    {
//        _teleWindow->updateThumbnailsList();
//    }
    else if (event->type == PropertyNotify)
    {
        if (event->xproperty.atom == XTools::_NET_WM_NAME ||
            event->xproperty.atom == XTools::WM_NAME)
        {
            free(_title);
            _title = XTools::windowTitle_alloc(_clientWindow);

            if (_teleWindow->shown())
                _teleWindow->paint();
        }
        else if (event->xproperty.atom == XTools::WM_STATE);
        {
            _minimized = XTools::checkIfWindowMinimized(_clientWindow);
        }
    }
//    else
//        printf("Thumbnail: unknown client event (%d)\n", event->type);
}


void Thumbnail::onResize()
{
    XWindowAttributes attrs;
    XGetWindowAttributes(_dpy, _clientWindow, &attrs);

    _clientWidth = attrs.width;
    _clientHeight= attrs.height;

#ifdef MAEMO
    _clientDecoX = attrs.x;
    _clientDecoY = attrs.y;
#endif

    double xscale = ((double)_width) / _clientWidth;
    double yscale = ((double)_height) / _clientHeight;

    double scale;

    if (xscale < yscale)
        scale = xscale;
    else
        scale = yscale;

    int oldClientScaledWidth = _clientScaledWidth;
    int oldClientScaledHeight= _clientScaledHeight;

    _clientScaledWidth = (int)round(scale * _clientWidth);
    _clientScaledHeight= (int)round(scale * _clientHeight);

    _clientDecoXScaled = (int)round(scale * _clientDecoX);
    _clientDecoYScaled = (int)round(scale * _clientDecoY);

    _clientOffsetX = (_width - _clientScaledWidth ) >> 1;
    _clientOffsetY = (_height- _clientScaledHeight) >> 1;

    XTransform xform = {{
        { XDoubleToFixed(1.0/scale), XDoubleToFixed(0), XDoubleToFixed(0) },
        { XDoubleToFixed(0), XDoubleToFixed(1.0/scale), XDoubleToFixed(0) },
        { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(1) },
    }};

//    double s2 = sqrt(2.0) / 2;
//
//    XTransform xform = {{
//        { XDoubleToFixed(s2), XDoubleToFixed(s2), XDoubleToFixed(-10) },
//        { XDoubleToFixed(-s2), XDoubleToFixed(s2), XDoubleToFixed(0) },
//        { XDoubleToFixed( 0), XDoubleToFixed( 0), XDoubleToFixed(1) },
//    }};

    XRenderSetPictureTransform(_dpy, _clientPict, &xform);


    Pixmap oldCachePixmap = _cachePixmap;
    Picture oldCachePicture = _cachePicture;

    XGetWindowAttributes(_dpy, _teleWindow->window(), &attrs);
    XRenderPictFormat *format = XRenderFindVisualFormat(_dpy, attrs.visual);
    _cachePixmap = XCreatePixmap(_dpy, _teleWindow->window(), _clientScaledWidth, _clientScaledHeight, _depth);
    _cachePicture = XRenderCreatePicture(_dpy, _cachePixmap, format, 0, 0);

    if (_minimized && oldCachePixmap != 0)
    {
        // Workaround for corner case: if this function is called when
        // client is minimized, we need to rescale cached pixmap, because
        // can't grab new picture from window
        // Yes, this function CAN be called when client is minimised - for
        // example when thumbnail needs to be resized because number of
        // windows changed

        XTransform xform = {{
            { XDoubleToFixed(((double)oldClientScaledWidth)/_clientScaledWidth), XDoubleToFixed(0), XDoubleToFixed(0) },
            { XDoubleToFixed(0), XDoubleToFixed(((double)oldClientScaledHeight)/_clientScaledHeight), XDoubleToFixed(0) },
            { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(1) }
        }};

        XRenderSetPictureTransform(_dpy, oldCachePicture, &xform);
        XRenderComposite(_dpy, PictOpSrc,
            oldCachePicture, 0, _cachePicture,
            0, 0, 0, 0, 0, 0,
            _clientScaledWidth, _clientScaledHeight
        );
    }

    if (oldCachePixmap != 0)
    {
        XRenderFreePicture(_dpy, oldCachePicture);
        XFreePixmap(_dpy, oldCachePixmap);
    }

    _cacheValid = false;


    paint();
}


void Thumbnail::onClientResize(XEvent *event)
{

    onResize();
}


void Thumbnail::paint()
{
//    XClearArea(_dpy, _teleWindow->window(), _x, _y, _width, _height, False);

    if (! _cacheValid)
    {
        if (! _minimized)
            XRenderComposite(_dpy, PictOpSrc,
                    _clientPict, None, _cachePicture,
                    _clientDecoXScaled-_clientDecoX, _clientDecoYScaled - _clientDecoY,
                    0, 0,
                    0, 0,
                    _clientScaledWidth, _clientScaledHeight);

        _cacheValid = true;
    }

    // FIXME: duplicating calculation from TeleWindow::paint()
    int xoffset = (_width - _clientScaledWidth) / 2;
    int yoffset = (_height - _clientScaledHeight) / 2;

    XCopyArea(_dpy, _cachePixmap, _teleWindow->_bufferPix, _teleWindow->_gc,
        0, 0,
        _clientScaledWidth, _clientScaledHeight,
        _teleWindow->scrollX() + _x + xoffset,// + _clientOffsetX,
        _teleWindow->scrollY() + _y + yoffset// + _clientOffsetY
    );
}


void Thumbnail::switchToClient()
{
    XTools::switchToWindow(_clientWindow);
}


void Thumbnail::closeClient()
{
    XTools::closeWindow(_clientWindow);
}
