#include "TeleWindow.h"

#include <string.h>
#include <stdlib.h>

#include <sys/time.h>

#include "XTools.h"
#include "Thumbnail.h"
#include "Settings.h"


const int TeleWindow::XMargin = 10;
const int TeleWindow::YMargin = 15;


TeleWindow::TeleWindow(Display *dpy)
    :_dpy(dpy), _mappings(dpy)
{
    int scr = DefaultScreen(_dpy);
    _rootWindow = RootWindow(_dpy, scr);
    int depth = DefaultDepth(_dpy, scr);
    Visual *visual = DefaultVisual(_dpy, scr);
    Colormap colormap = DefaultColormap(_dpy, scr);

    _width = DisplayWidth(_dpy, scr);
    _height = DisplayHeight(_dpy, scr);


    // Initializing imlib2

    imlib_context_set_display(_dpy);
    imlib_context_set_visual(visual);
    imlib_context_set_colormap(colormap);


    // Loading background
    _bgPixmap = 0;
    reloadBackground();



// Loading header images

    bool ok = true;
    ok = ok && loadARGBImage(
        Settings::instance()->headerLeftFilename(),
        _rootWindow,
        &_headerLeftPix,
        &_headerLeftPict,
        &_headerLeftWidth,
        &_headerHeight);
    ok = ok && loadARGBImage(
        Settings::instance()->headerRightFilename(),
        _rootWindow,
        &_headerRightPix,
        &_headerRightPict,
        &_headerRightWidth,
        0);
    ok = ok && loadARGBImage(
        Settings::instance()->headerMiddleFilename(),
        _rootWindow,
        &_headerMiddlePix,
        &_headerMiddlePict,
        0, 0);

    ok = ok && loadARGBImage(
        Settings::instance()->headerLeftSelectedFilename(),
        _rootWindow,
        &_headerLeftSelectedPix,
        &_headerLeftSelectedPict,
        0, 0);
    ok = ok && loadARGBImage(
        Settings::instance()->headerRightSelectedFilename(),
        _rootWindow,
        &_headerRightSelectedPix,
        &_headerRightSelectedPict,
        0, 0);
    ok = ok && loadARGBImage(
        Settings::instance()->headerMiddleSelectedFilename(),
        _rootWindow,
        &_headerMiddleSelectedPix,
        &_headerMiddleSelectedPict,
        0, 0);

    if (! ok)
        printf("Cannot load all pixmaps\n");

    XRenderPictureAttributes pictureAttrs;
    pictureAttrs.repeat = RepeatNormal;
    XRenderChangePicture(_dpy, _headerMiddlePict, CPRepeat, &pictureAttrs);
    XRenderChangePicture(_dpy, _headerMiddleSelectedPict, CPRepeat, &pictureAttrs);


    XRenderParseColor(_dpy, Settings::instance()->borderColor(), &_borderColor);
    XRenderParseColor(_dpy, Settings::instance()->borderActiveColor(), &_borderActiveColor);


    _borderWidth = Settings::instance()->borderWidth();
    _textLeftMargin = Settings::instance()->textLeftMargin();
    _textRightMargin = Settings::instance()->textRightMargin();

    _closeButtonXSpan = Settings::instance()->closeButtonXSpan();
    _closeButtonYSpan = Settings::instance()->closeButtonYSpan();



    // Initialization of scrolling

    _scrollBaseX = 0;
    _scrollBaseY = 0;
    _scrollX = 0;
    _scrollY = 0;
    _buttonPressed = false;




    // Creating main window

    XSetWindowAttributes createAttrs;
    createAttrs.save_under = True; // This seems to not work, but let's keep it

    _win = XCreateWindow(_dpy, _rootWindow, 0, 0, _width, _height,
        0, depth, InputOutput, visual,
        CWSaveUnder,
        &createAttrs);

    _gc = XCreateGC(_dpy, _win, 0, 0);

    XSelectInput(_dpy, _win,
        ExposureMask        |
        ButtonPressMask     |
        ButtonReleaseMask   |
        ButtonMotionMask    |
        StructureNotifyMask |
        KeyPressMask        |
        KeyReleaseMask
    );


    // Initializing Xft

    char fontname[50];
    snprintf(fontname, 50, "sans:pixelsize=%d", Settings::instance()->fontSize());
    XftPattern *pattern = XftNameParse(fontname);
    XftResult result;
    XftPattern *pattern2= XftFontMatch(_dpy, scr, pattern, &result);
    _xftFont = XftFontOpenPattern(_dpy, pattern2);
    XftPatternDestroy(pattern);
    XftPatternDestroy(pattern2);


    // Double buffering pixmap
    _bufferPix = 0;
    _xrenderFormat = 0;
    recreateBufferPixmap();


    XRenderPictureAttributes pa;
    pa.subwindow_mode = IncludeInferiors;

    _picture = XRenderCreatePicture(_dpy, _win, _xrenderFormat, CPSubwindowMode, &pa);

    // Xft Draw object
    _xftDraw = XftDrawCreate(_dpy, _bufferPix, visual, colormap);


    _shown = false;




    XSelectInput(_dpy, _rootWindow, StructureNotifyMask | PropertyChangeMask);


    // Hotkey keycode
    KeySym keysym = XStringToKeysym("F5");
    _hotKeyCode = XKeysymToKeycode(_dpy, keysym);
    _hotKeyPressed = false;

    _activeThumbnail = 0;


    markThumbnailsListDirty();
}

TeleWindow::~TeleWindow()
{
    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        delete *i;


    XftDrawDestroy(_xftDraw);

    XRenderFreePicture(_dpy, _picture);
    XRenderFreePicture(_dpy, _bufferPicture);

    XFreePixmap(_dpy, _bgPixmap);
    XFreePixmap(_dpy, _bufferPix);


    XRenderFreePicture(_dpy, _headerLeftPict);
    XRenderFreePicture(_dpy, _headerRightPict);
    XRenderFreePicture(_dpy, _headerMiddlePict);
    XFreePixmap(_dpy, _headerLeftPix);
    XFreePixmap(_dpy, _headerRightPix);
    XFreePixmap(_dpy, _headerMiddlePix);


    XFreeGC(_dpy, _gc);
    XDestroyWindow(_dpy, _win);
}


Display* TeleWindow::display()
{
    return _dpy;
}


Window TeleWindow::window()
{
    return _win;
}


void TeleWindow::show()
{
    _shown = true;

    Window activeWindow = XTools::activeWindow();

    _activeThumbnail = 0;
    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        if ((*i)->clientWindow() == activeWindow)
            _activeThumbnail = *i;


    XMapWindow(_dpy, _win);

    XEvent event;

    memset(&event, 0, sizeof(event));
    event.type = ClientMessage;
    event.xclient.window = _win;
    event.xclient.message_type = XTools::_NET_WM_STATE;
    event.xclient.format = 32;
    event.xclient.data.l[0] = 1;
    event.xclient.data.l[1] = XTools::_NET_WM_STATE_FULLSCREEN;
    event.xclient.data.l[2] = 0;


    XSendEvent(_dpy, _rootWindow, False, SubstructureNotifyMask, &event);

    XTools::switchToWindow(_win);


//    if (_activeThumbnail)
//        animate(_activeThumbnail, true);
}

void TeleWindow::hide()
{
    XUnmapWindow(_dpy, _win);

    _shown = false;
}

bool TeleWindow::shown()
{
    return _shown;
}



void TeleWindow::markThumbnailsListDirty()
{
    _thumbnailsListDirty = true;

    if (_shown)
        updateThumbnailsList();
}



void TeleWindow::updateThumbnailsList()
{
    if (_thumbnailsListDirty == false)
        return;

    bool wasChanged = false;


    LinkedList<Window> windows = XTools::windowList(_rootWindow);
    for (LinkedList<Window>::Iter i = windows.head(); i; ++i)
        if (*i != _win)
        {
            bool found = false;
            for (LinkedList<Thumbnail*>::Iter j = _thumbnails.head(); j; ++j)
                if ((*j)->clientWindow() == *i)
                {
                    found = true;
                    break;
                }

            if (! found)
            {
                Thumbnail *th = new Thumbnail(this, *i);
                _thumbnails.append(th);
                wasChanged = true;
            }
        }

    LinkedList<Thumbnail*> thumbsToDelete;
    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        if (! windows.contains((*i)->clientWindow()))
            thumbsToDelete.append(*i);

    for (LinkedList<Thumbnail*>::Iter i = thumbsToDelete.head(); i; ++i)
    {
        removeThumbnail(*i);
        delete *i;
        wasChanged = true;
    }


    if (wasChanged)
        layoutThumbnails();

    _thumbnailsListDirty = false;
}


inline int min(int a, int b)
{
    return a < b ? a : b;
}


void TeleWindow::layoutThumbnails()
{
    int n = _thumbnails.size();

    if (n == 0)
    {
        paint();
        return;
    }

    int maxSize = 0;
    int maxColumns = 0;

    for (int columns = 1; columns <= n; ++columns)
    {
        int rows = (n + columns - 1) / columns;

        int widthWithBorder = _width / columns - 2 * XMargin;
        int heightWithBorder = _height / rows - 2 * YMargin;

        int width = widthWithBorder - 2 * _borderWidth;
        int height = heightWithBorder - _headerHeight - _borderWidth;

        int resultSize = height * _width / _height;
        if (width < resultSize)
            resultSize = width;

        if (resultSize > maxSize)
        {
            maxSize = resultSize;
            maxColumns = columns;
        }
    }

    int columns = maxColumns;
    int rows = (n + columns - 1) / columns;

    int tileWidth = _width / columns;
    int tileHeight = _height / rows;

    int thumbWidth, thumbHeight;

    if (n > 1)
    {
        thumbWidth = maxSize;
        thumbHeight= maxSize * _height / _width;
    }
    else
    {
        // If there is only one window we don't want it become to big
        thumbWidth = _width * 2 / 3;
        thumbHeight = _height * 2 / 3;
    }

    int widthWithBorder = thumbWidth + 2*_borderWidth;
    int heightWithBorder= thumbHeight+ _borderWidth + _headerHeight;

    int tileXOffset = (tileWidth - widthWithBorder) / 2;
    int tileYOffset = (tileHeight-heightWithBorder) / 2;

    int lastRowX = 0;

    int index = 0;
    LinkedList<Thumbnail*>::Iter thumb = _thumbnails.head();
    for (int row = 0; row < rows; ++row)
    {
        if (row == rows - 1)
        {
            lastRowX = (_width - (n - index) * tileWidth) / 2;
        }

        int tileY = row * tileHeight;

        for (int column = 0; column < columns; ++column, ++thumb, ++index)
        {
            if (thumb == 0)
                break;

            int tileX = lastRowX + column * tileWidth;

            int x = tileX + tileXOffset + _borderWidth;
            int y = tileY + tileYOffset + _headerHeight;

            (*thumb)->setGeometry(x, y, thumbWidth, thumbHeight);
        }
    }

    if (_shown)
        paint();
}


void TeleWindow::removeThumbnail(Thumbnail *thumb)
{
    _thumbnails.removeByValue(thumb);
}


void TeleWindow::eventLoop()
{
    XEvent event;


    XGrabKey(_dpy, _hotKeyCode, AnyModifier, _rootWindow, False, GrabModeAsync, GrabModeAsync);

    _breakEventLoop = false;
    while (! _breakEventLoop)
    {
        do
        {
            XNextEvent(_dpy, &event);

            if (event.xany.window == _rootWindow)
            {
                onRootEvent(&event);
                continue;
            }

            if (event.xany.window == _win)
            {
                onEvent(&event);
                continue;
            }
            else
            {
                bool found = false;
                for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
                {
                    if ((*i)->clientWindow() == event.xany.window)
                    {
                        (*i)->onClientEvent(&event);
                        found = true;
                        break;
                    }
                }
                if (found)
                    continue;
            }
        } while (XPending(_dpy));

        onIdle();
    }

    XUngrabKey(_dpy, _hotKeyCode, AnyModifier, _rootWindow);
}



void TeleWindow::onRootEvent(XEvent *event)
{
    if (event->type == KeyPress && event->xkey.keycode == _hotKeyCode)
        onHotKeyPress();
    if (event->type == KeyRelease && event->xkey.keycode == _hotKeyCode)
        onHotKeyRelease();
//    else if (event->type == MapNotify)
//        updateThumbnailsList();
    else if (event->type == PropertyNotify)
    {
//        char *name = XGetAtomName(_dpy, event->xproperty.atom);
//        printf("Root PropertyNotify: %s\n", name);
//        XFree(name);

        if (_shown && event->xproperty.atom == XTools::_NET_ACTIVE_WINDOW)
            if (XTools::activeWindow() != _win)
                hide();

        if (event->xproperty.atom == XTools::_NET_CLIENT_LIST)
            markThumbnailsListDirty();
    }
    else if (event->type == ConfigureNotify)
    {
        if (event->xconfigure.width != _width || event->xconfigure.height != _height)
        {
            setNewSize(event->xconfigure.width, event->xconfigure.height);
        }
    }
//    else
//        printf("Unknown root event: %d\n", event->type);
}



void TeleWindow::onEvent(XEvent *event)
{
    if (event->type == ButtonPress)
        onButtonPress(&event->xbutton);
    else if (event->type == ButtonRelease)
        onButtonRelease(&event->xbutton);
    else if (event->type == MotionNotify)
        onButtonMotion(&event->xmotion);
    else if (event->type == Expose && event->xexpose.count < 1)
        //paint();
        _repaintOnIdle = true;
    else if (event->type == ConfigureNotify)
    {
        bool relayout = _width != event->xconfigure.width || _height != event->xconfigure.height;
        _width = event->xconfigure.width;
        _height= event->xconfigure.height;
        if (relayout)
            layoutThumbnails();
    }
    else if (event->type == KeyPress)
    {
        _mappings.handleEvent(this, Mapping::Press, event->xkey.keycode);
    }
}


void TeleWindow::paint()
{
    if (! _shown)
        return;


//    timeval before;
//    gettimeofday(&before, 0);

    XCopyArea(_dpy, _bgPixmap, _bufferPix, _gc,
        0, 0, _width, _height, 0, 0
    );


    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
    {
        int x = (*i)->x();
        int y = (*i)->y();
        int w = (*i)->realWidth();
        int h = (*i)->realHeight();
        int xoffset = ((*i)->width() - w) / 2;
        int yoffset = ((*i)->height() - h) / 2;

        bool selected = (*i) == _activeThumbnail;
        XRenderColor borderColor = selected ? _borderActiveColor : _borderColor;

        Picture left = selected ? _headerLeftSelectedPict : _headerLeftPict;
        Picture right = selected ? _headerRightSelectedPict : _headerRightPict;
        Picture middle = selected ? _headerMiddleSelectedPict : _headerMiddlePict;

        XRenderComposite(_dpy, PictOpOver,
            left, None, _bufferPicture,
            0, 0,
            0, 0,
            _scrollX + x + xoffset - _borderWidth,
            _scrollY + y + yoffset - _headerHeight,
            _headerLeftWidth, _headerHeight
        );

        XRenderComposite(_dpy, PictOpOver,
            right, None, _bufferPicture,
            0, 0,
            0, 0,
            _scrollX + x + xoffset + w + _borderWidth - _headerRightWidth,
            _scrollY + y + yoffset - _headerHeight,
            _headerRightWidth, _headerHeight
        );

        XRenderComposite(_dpy, PictOpSrc,
            middle, None, _bufferPicture,
            0, 0,
            0, 0,
            _scrollX + x + xoffset - _borderWidth + _headerLeftWidth,
            _scrollY + y + yoffset - _headerHeight,
            w + 2 * _borderWidth - _headerLeftWidth - _headerRightWidth,
            _headerHeight
        );

        // Left border
        XRenderFillRectangle(_dpy, PictOpSrc, _bufferPicture, &borderColor,
            _scrollX + x + xoffset - _borderWidth,
            _scrollY + y + yoffset,
            _borderWidth, h
        );
        // Right border
        XRenderFillRectangle(_dpy, PictOpSrc, _bufferPicture, &borderColor,
            _scrollX + x + xoffset + w,
            _scrollY + y + yoffset,
            _borderWidth, h
        );
        // Bottom border
        XRenderFillRectangle(_dpy, PictOpSrc, _bufferPicture, &borderColor,
            _scrollX + x + xoffset - _borderWidth,
            _scrollY + y + yoffset + h,
            w + 2*_borderWidth, _borderWidth
        );
    }


    XftColor fontColor;
    fontColor.pixel = 0;
    fontColor.color.red     = 0xffff;
    fontColor.color.green   = 0xffff;
    fontColor.color.blue    = 0xffff;
    fontColor.color.alpha   = 0xffff;


    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
    {
        int xoffset = ((*i)->width() - (*i)->realWidth()) / 2;
        int yoffset = ((*i)->height() - (*i)->realHeight()) / 2;

        XRectangle rect = {
            _scrollX + (*i)->x() + xoffset - _borderWidth + _textLeftMargin,
            _scrollY + (*i)->y() + yoffset - _headerHeight,
            (*i)->realWidth() + 2*_borderWidth - _textLeftMargin - _textRightMargin,
            _headerHeight
        };
        Region clip = XCreateRegion();
        XUnionRectWithRegion(&rect, clip, clip);

        XftDrawSetClip(_xftDraw, clip);

        XftDrawStringUtf8(_xftDraw, &fontColor, _xftFont, 
            _scrollX + (*i)->x() + xoffset - _borderWidth + _textLeftMargin,
            _scrollY + (*i)->y() + yoffset + Settings::instance()->textYOffset(),
            (const FcChar8*)(*i)->title(),
            strlen((*i)->title())
        );
        
        XDestroyRegion(clip);
    }

    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        (*i)->paint();






    XCopyArea(_dpy, _bufferPix, _win, _gc,
        0, 0, _width, _height, 0, 0
    );

//    timeval after;
//    gettimeofday(&after, 0);
//    printf("---------\n");
//    printf("%ld:%ld\n", before.tv_sec, before.tv_usec);
//    printf("%ld:%ld\n", after.tv_sec, after.tv_usec);
}


void TeleWindow::onThumbRedrawed(Thumbnail *thumb)
{
    XCopyArea(_dpy, _bufferPix, _win, _gc,
        _scrollX + thumb->x(), _scrollY + thumb->y(),
        thumb->width(), thumb->height(),
        _scrollX + thumb->x(), _scrollY + thumb->y()
    );
}



void TeleWindow::onHotKeyPress()
{
    if (_hotKeyPressed)
    {
        if (_shown)
            hide();
        XTools::showDesktop();
        return;
    }

    _hotKeyPressed = true;

    if (! _shown)
        show();
    else
        _mappings.handleEvent(this, Mapping::Press, _hotKeyCode);
}

void TeleWindow::onHotKeyRelease()
{
    char keysPressed[32];
    XQueryKeymap(_dpy, keysPressed);
    if ( (keysPressed[_hotKeyCode >> 3] >> (_hotKeyCode & 0x07)) & 0x01)
        return;

    _hotKeyPressed = false;
}



void TeleWindow::onButtonPress(XButtonEvent *event)
{
    _wasScrolling = false;
    _buttonPressed = true;
    _buttonPressX = event->x;
    _buttonPressY = event->y;
    _scrollBaseX = _scrollX;
    _scrollBaseY = _scrollY;
}

void TeleWindow::onButtonRelease(XButtonEvent *event)
{
    _buttonPressed = false;

    if (! _wasScrolling)
    {
        bool missed = true;

        for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        {
            // Checking close button
            {
                int closeX = (*i)->x() + (*i)->width() + _borderWidth - _closeButtonXSpan;
                int closeY = (*i)->y() - _headerHeight;


                if (- _scrollX + event->x >= closeX &&
                    - _scrollX + event->x <= closeX + _closeButtonXSpan &&
                    - _scrollY + event->y >= closeY &&
                    - _scrollY + event->y <= closeY + _closeButtonYSpan
                    )
                {
                    (*i)->closeClient();
                    missed = false;
                }
            }

            if (missed && (*i)->inside(- _scrollX + event->x, - _scrollY + event->y))
            {
//                animate(*i, false);
                (*i)->switchToClient();
                hide();
                missed = false;
            }
        }

        if (missed)
        {
            hide();
            XTools::showDesktop();
        }
    }
}


void TeleWindow::onButtonMotion(XMotionEvent *event)
{
    if (_buttonPressed && Settings::instance()->scrollingEnabled())
    {
        _scrollX = _scrollBaseX + event->x - _buttonPressX;
        _scrollY = _scrollBaseY + event->y - _buttonPressY;

        if (! _wasScrolling)
        {
            if (abs(event->x - _buttonPressX) > 20 ||
                abs(event->y - _buttonPressY) > 20)
                _wasScrolling = true;
        }


//        paint();
        _repaintOnIdle = true;
    }
}


void TeleWindow::onIdle()
{
    if (_repaintOnIdle)
    {
        paint();
        _repaintOnIdle = false;
    }
}


void TeleWindow::internalCommand(const char *action)
{
    if (strcmp(action, "switchToSelected") == 0)
    {
        if (_activeThumbnail != 0)
            _activeThumbnail->switchToClient();
        else
            XTools::showDesktop();
        hide();
    }
    else if (strcmp(action, "selectNext") == 0)
    {
        if (_activeThumbnail == 0)
        {
            if (_thumbnails.size() > 0)
                _activeThumbnail = *_thumbnails.head();
        }
        else
        {
            int index = 0;
            bool changed = false;
            for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i, ++index)
                if ((*i) == _activeThumbnail)
                {
                    if (index < _thumbnails.size() - 1)
                        _activeThumbnail = *(++i);
                    else
                        _activeThumbnail = *_thumbnails.head();
                    changed = true;
                    break;
                }

            if (! changed)
                _activeThumbnail = *_thumbnails.head();
        }

        paint();
    }
    else if (strcmp(action, "selectPrev") == 0)
    {
        if (_activeThumbnail == 0)
        {
            if (_thumbnails.size() > 0)
                _activeThumbnail = *_thumbnails.tail();
        }
        else
        {
            int index = 0;
            bool changed = false;
            for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i, ++index)
                if ((*i) == _activeThumbnail)
                {
                    if (index > 0)
                        _activeThumbnail = *(--i);
                    else
                        _activeThumbnail = *_thumbnails.tail();
                    changed = true;
                    break;
                }

            if (! changed)
                _activeThumbnail = *_thumbnails.tail();
        }

        paint();
    }
    else
    {
        fprintf(stderr, "Unknown internal command: %s\n", action);
    }
}



void TeleWindow::setNewSize(int width, int height)
{
    printf("setNewSize(%d, %d)\n", width, height);

    _width = width;
    _height = height;

    XResizeWindow(_dpy, _win, _width, _height);

    recreateBufferPixmap();

    reloadBackground();

    layoutThumbnails();
}



void TeleWindow::reloadBackground()
{
    if (_bgPixmap != 0)
        XFreePixmap(_dpy, _bgPixmap);

    _bgPixmap = XCreatePixmap(_dpy, _rootWindow,
        _width, _height,
        DefaultDepth(_dpy, DefaultScreen(_dpy)));

    printf("Loading background from '%s'\n", Settings::instance()->backgroundFilename());
    printf("Background mode: %d\n", Settings::instance()->backgroundMode());
    Imlib_Image background = imlib_load_image(Settings::instance()->backgroundFilename());
    if (background == 0)
        printf("Cannot load background\n");
    else
    {
        Imlib_Image bgToDraw = 0;

        int bgXpos = 0;
        int bgYpos = 0;

        imlib_context_set_image(background);
        switch (Settings::instance()->backgroundMode())
        {
            case Settings::Stretched:
            {
                bgToDraw = imlib_create_cropped_scaled_image(
                    0, 0,
                    imlib_image_get_width(), imlib_image_get_height(),
                    _width, _height
                );
                break;
            }

            case Settings::Centered:
            {
                // Hildon's logic is not so easy in this case

//                bgToDraw = imlib_create_cropped_image(
//                    (imlib_image_get_width() - _width) / 2,
//                    (imlib_image_get_height()- _height) / 2,
//                    _width, _height
//                );

                double scalex = 1.0;
                double scaley = 1.0;

                if (imlib_image_get_width() > _width * 2)
                    scalex = (double)imlib_image_get_width() / 2.0 / _width;
                if (imlib_image_get_height() > _height * 2)
                    scaley = (double)imlib_image_get_height() / 2.0 / _height;

                double scale = scalex;
                if (scaley > scale)
                    scale = scaley;

                int newwidth = (int)(scale * _width);
                int newheight= (int)(scale * _height);

                bgToDraw = imlib_create_cropped_scaled_image(
                    (imlib_image_get_width() - newwidth) / 2,
                    (imlib_image_get_height()- newheight)/ 2,
                    newwidth, newheight,
                    _width, _height
                );

                break;
            }

            case Settings::Scaled: case Settings::Cropped:
            {
                double scalex = (double)_width / imlib_image_get_width();
                double scaley = (double)_height / imlib_image_get_height();
                double scale = scalex;

                if (Settings::instance()->backgroundMode() == Settings::Scaled)
                {
                    if (scaley < scale) scale = scaley;
                }
                else
                {
                    if (scaley > scale) scale = scaley;
                }

                int newwidth = (int)(imlib_image_get_width() * scale);
                int newheight= (int)(imlib_image_get_height() * scale);

                bgToDraw = imlib_create_cropped_scaled_image(
                    0, 0,
                    imlib_image_get_width(), imlib_image_get_height(),
                    newwidth, newheight
                );

                bgXpos = (_width - newwidth) / 2;
                bgYpos = (_height - newheight) / 2;
                break;
            }

            default:
                printf("Unknown background mode: %d\n", Settings::instance()->backgroundMode());
        }

        imlib_free_image();

        imlib_context_set_image(bgToDraw);
        imlib_context_set_drawable(_bgPixmap);
        imlib_context_set_visual(DefaultVisual(_dpy, DefaultScreen(_dpy)));


        imlib_render_image_on_drawable(bgXpos, bgYpos);

        imlib_free_image();
    }
}


void TeleWindow::recreateBufferPixmap()
{
    if (_bufferPix)
    {
        XRenderFreePicture(_dpy, _bufferPicture);
        XFreePixmap(_dpy, _bufferPix);
    }

    int scr = DefaultScreen(_dpy);
    int depth = DefaultDepth(_dpy, scr);
    Visual *visual = DefaultVisual(_dpy, scr);

    if (_xrenderFormat == 0)
    {
        _xrenderFormat = XRenderFindVisualFormat(_dpy, visual);
        XVisualInfo rgbaVisual;
        if (XMatchVisualInfo(_dpy, scr, 32, TrueColor, &rgbaVisual) == 0)
            fprintf(stderr, "Cannot find rgba visual\n");
        _xrenderRGBAFormat = XRenderFindVisualFormat(_dpy, rgbaVisual.visual);
    }

    // Double-buffering pixmap
    _bufferPix = XCreatePixmap(_dpy, _rootWindow, _width, _height, depth);

    // XRender picture
    _bufferPicture = XRenderCreatePicture(_dpy, _bufferPix, _xrenderFormat, 0, 0);
}


bool TeleWindow::loadARGBImage(
    const char *filename,
    Drawable parent,
    Pixmap *pixmap,
    Picture *picture,
    int *width,
    int *height)
{
    static Visual *defaultVisual = 0;
    static XVisualInfo rgbaVisual;
    static XRenderPictFormat *defaultFormat;
    static XRenderPictFormat *rgbaFormat;
    static Colormap colormap;

    *pixmap = 0;
    *picture = 0;

    if (defaultVisual == 0)
    {
        // Initializing

        int scr = DefaultScreen(_dpy);
        defaultVisual = DefaultVisual(_dpy, scr);
        colormap = DefaultColormap(_dpy, scr);
        defaultFormat = XRenderFindVisualFormat(_dpy, defaultVisual);

        if (XMatchVisualInfo(_dpy, scr, 32, TrueColor, &rgbaVisual) == 0)
        {
            fprintf(stderr, "Cannot find rgba visual\n");
            return false;
        }
        rgbaFormat = XRenderFindVisualFormat(_dpy, rgbaVisual.visual);
    }


    Imlib_Image image = imlib_load_image(filename);
    if (image == 0)
        return false;

    imlib_context_set_image(image);
    int w = imlib_image_get_width();
    int h = imlib_image_get_height();

    if (width) *width = w;
    if (height) *height = h;

    // Making image with premultiplied alpha
    Imlib_Image premul = imlib_create_image(w, h);
    imlib_context_set_image(premul);
    imlib_context_set_color(0, 0, 0, 255);
    imlib_image_fill_rectangle(0, 0, w, h);
    imlib_context_set_blend(1);
    imlib_blend_image_onto_image(image, 0, 0, 0, w, h, 0, 0, w, h);
    imlib_image_copy_alpha_to_image(image, 0, 0);

    *pixmap = XCreatePixmap(_dpy, parent, w, h, 32);
    imlib_context_set_display(_dpy);
    imlib_context_set_colormap(colormap);
    imlib_context_set_visual(rgbaVisual.visual);
    imlib_context_set_drawable(*pixmap);
    imlib_context_set_blend(0);
    imlib_render_image_on_drawable(0, 0);

    imlib_free_image();
    imlib_context_set_image(image);
    imlib_free_image();

    *picture = XRenderCreatePicture(_dpy, *pixmap, rgbaFormat, 0, 0);

    return true;
}


void TeleWindow::animate(Thumbnail *thumb, bool toSmall)
{
        XFlush(_dpy);

        
        XWindowAttributes attrs;
        XGetWindowAttributes(_dpy, thumb->clientWindow(), &attrs);

        int bigx, bigy;
        int bigw = attrs.width;
        int bigh = attrs.height;
        Window child;
        XTranslateCoordinates(_dpy, thumb->clientWindow(), _rootWindow,
            0, 0, &bigx, &bigy, &child);

        int xoffset = (thumb->width() - thumb->realWidth()) / 2;
        int yoffset = (thumb->height() - thumb->realHeight()) / 2;
        int smallx = thumb->x() + xoffset;
        int smally = thumb->y() + yoffset;
        int smallw = thumb->realWidth();
        int smallh = thumb->realHeight();


        if (! toSmall)
        {
            int t;
            t = smallx; smallx = bigx; bigx = t;
            t = smally; smally = bigy; bigy = t;
            t = smallw; smallw = bigw; bigw = t;
            t = smallh; smallh = bigh; bigh = t;
        }

        int prevx = 0, prevy = 0, prevw = _width, prevh = _height;


        double step = toSmall ? 0.33 : 0.50;
        for (double a = 0.0; a <= 1.0; a += step)
        {
            int x = (int)(((double)bigx) * (1.0 - a) + ((double)smallx) * a);
            int y = (int)(((double)bigy) * (1.0 - a) + ((double)smally) * a);
            int w = (int)(((double)bigw) * (1.0 - a) + ((double)smallw) * a);
            int h = (int)(((double)bigh) * (1.0 - a) + ((double)smallh) * a);

            double scale = w;
            scale /= attrs.width;

            XTransform xform = {{
                { XDoubleToFixed(1), XDoubleToFixed(0), XDoubleToFixed(0) },
                { XDoubleToFixed(0), XDoubleToFixed(1), XDoubleToFixed(0) },
                { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(scale) },
            }};

            XRenderSetPictureTransform(_dpy, thumb->clientPicture(), &xform);

            if (toSmall)
                XCopyArea(_dpy, _bufferPix, _win, _gc,
                    prevx, prevy, prevw, prevh, prevx, prevy);

            XRenderComposite(_dpy, PictOpSrc,
                thumb->clientPicture(), None, _picture,
                (int)(- (1.0 - scale) * thumb->clientDecoX()),
                (int)(- (1.0 - scale) * thumb->clientDecoY()),
                0, 0,
                x, y,
                w, h
            );

            prevx = x;
            prevy = y;
            prevw = w;
            prevh = h;

            XSync(_dpy, 0);

            //struct timeval ts;
            //ts.tv_sec = 0;
            //ts.tv_usec = 1000;
            //select(0, 0, 0, 0, &ts);
        }
}
