#include "TeleWindow.h"

#include <string.h>
#include <stdlib.h>

#include <sys/time.h>

#include "XTools.h"
#include "Thumbnail.h"
#include "Settings.h"


const int TeleWindow::XMargin = 10;
const int TeleWindow::YMargin = 15;


TeleWindow::TeleWindow(Display *dpy)
    :_dpy(dpy)
{
    int scr = DefaultScreen(_dpy);
    _rootWindow = RootWindow(_dpy, scr);
    int depth = DefaultDepth(_dpy, scr);
    Visual *visual = DefaultVisual(_dpy, scr);
    Colormap colormap = DefaultColormap(_dpy, scr);


    // Initializing imlib2

    imlib_context_set_display(_dpy);
    imlib_context_set_visual(visual);
    imlib_context_set_colormap(colormap);


    // Loading background

    _bgPixmap = XCreatePixmap(_dpy, _rootWindow, 800, 480, depth);

    Imlib_Image background = imlib_load_image(Settings::instance()->backgroundFilename());
    if (background == 0)
        printf("Cannot load background\n");
    else
    {
        imlib_context_set_image(background);
        imlib_context_set_drawable(_bgPixmap);
        imlib_render_image_on_drawable(0, 0);
        imlib_free_image();
    }


// Loading header images

    // We need RGBA visual for transparency
    XVisualInfo rgbaVisual;
    if (XMatchVisualInfo(_dpy, scr, 32, TrueColor, &rgbaVisual) == 0)
        fprintf(stderr, "Cannot find rgba visual\n");
    XRenderPictFormat *rgbaFormat = XRenderFindVisualFormat(_dpy, rgbaVisual.visual);

    // Blend == 0 makes imlib to copy alpha channel instead of using it for blending
    imlib_context_set_visual(rgbaVisual.visual);
    imlib_context_set_blend(0);

    Imlib_Image headerLeft = imlib_load_image(Settings::instance()->headerLeftFilename());
    Imlib_Image headerRight = imlib_load_image(Settings::instance()->headerRightFilename());
    Imlib_Image headerMiddle = imlib_load_image(Settings::instance()->headerMiddleFilename());
    if (headerLeft == 0 || headerRight == 0 || headerMiddle == 0)
    {
        fprintf(stderr, "Cannot load header pixmaps\n");
        exit(1); // Ugly, i know
    }

    imlib_context_set_image(headerLeft);
    _headerLeftWidth = imlib_image_get_width();
    _headerLeftPix = XCreatePixmap(_dpy, _rootWindow, imlib_image_get_width(), imlib_image_get_height(), 32);
    imlib_context_set_drawable(_headerLeftPix);
    imlib_render_image_on_drawable(0, 0);
    imlib_free_image();
    _headerLeftPict = XRenderCreatePicture(_dpy, _headerLeftPix, rgbaFormat, 0, 0);

    imlib_context_set_image(headerRight);
    _headerRightWidth = imlib_image_get_width();
    _headerRightPix = XCreatePixmap(_dpy, _rootWindow, imlib_image_get_width(), imlib_image_get_height(), 32);
    imlib_context_set_drawable(_headerRightPix);
    imlib_render_image_on_drawable(0, 0);
    imlib_free_image();
    _headerRightPict = XRenderCreatePicture(_dpy, _headerRightPix, rgbaFormat, 0, 0);

    imlib_context_set_image(headerMiddle);
    _headerHeight = imlib_image_get_height();
    _headerMiddlePix = XCreatePixmap(_dpy, _rootWindow, imlib_image_get_width(), imlib_image_get_height(), 32);
    imlib_context_set_drawable(_headerMiddlePix);
    imlib_render_image_on_drawable(0, 0);
    imlib_free_image();

    XRenderPictureAttributes pictAttrs;
    pictAttrs.repeat = RepeatNormal;
    _headerMiddlePict = XRenderCreatePicture(_dpy, _headerMiddlePix, rgbaFormat, CPRepeat, &pictAttrs);

    XRenderParseColor(_dpy, Settings::instance()->borderColor(), &_borderColor);

    _borderWidth = Settings::instance()->borderWidth();
    _textLeftMargin = Settings::instance()->textLeftMargin();
    _textRightMargin = Settings::instance()->textRightMargin();

    _closeButtonXSpan = Settings::instance()->closeButtonXSpan();
    _closeButtonYSpan = Settings::instance()->closeButtonYSpan();



    // Initialization of scrolling

    _scrollBaseX = 0;
    _scrollBaseY = 0;
    _scrollX = 0;
    _scrollY = 0;
    _buttonPressed = false;




    // Creating main window

    XSetWindowAttributes createAttrs;
    createAttrs.background_pixmap = _bgPixmap;
    createAttrs.save_under = True; // Thus seems to not work, but let's keep it

    _win = XCreateWindow(_dpy, _rootWindow, 0, 0, 800, 480,
        0, depth, InputOutput, visual,
        CWSaveUnder,// 0,//CWBackPixmap,
        &createAttrs);

    _gc = XCreateGC(_dpy, _win, 0, 0);

    XSelectInput(_dpy, _win,
        ExposureMask        |
        ButtonPressMask     |
        ButtonReleaseMask   |
        ButtonMotionMask    |
        StructureNotifyMask |
        KeyPressMask        |
        KeyReleaseMask
    );

    _width = 800;
    _height = 480;


    // Initializing Xft

    char fontname[50];
    snprintf(fontname, 50, "sans:pixelsize=%d", Settings::instance()->fontSize());
    XftPattern *pattern = XftNameParse(fontname);
    XftResult result;
    XftPattern *pattern2= XftFontMatch(_dpy, scr, pattern, &result);
    _xftFont = XftFontOpenPattern(_dpy, pattern2);
    XftPatternDestroy(pattern);
    XftPatternDestroy(pattern2);


    // Double buffering pixmap
    _bufferPix = XCreatePixmap(_dpy, _rootWindow, _width, _height, depth);

    // Xft Draw object
    _xftDraw = XftDrawCreate(_dpy, _bufferPix, visual, colormap);

    _shown = false;



    // XRender picture for main window and double buffer
    XWindowAttributes attrs;
    XGetWindowAttributes(_dpy, _win, &attrs);
    XRenderPictFormat *format = XRenderFindVisualFormat(_dpy, attrs.visual);

    XRenderPictureAttributes pa;
    pa.subwindow_mode = IncludeInferiors;

    _picture = XRenderCreatePicture(_dpy, _win, format, CPSubwindowMode, &pa);
    _bufferPicture = XRenderCreatePicture(_dpy, _bufferPix, format, 0, 0);


    // For intercepting MapNotify, UnmapNotify and DestroyNotify
    XSelectInput(_dpy, _rootWindow, SubstructureNotifyMask);


    // Hotkey keycode
    KeySym keysym = XStringToKeysym("F5");
    _hotKeyCode = XKeysymToKeycode(_dpy, keysym);
    _hotKeyPressed = false;


    updateThumbnailsList();
}

TeleWindow::~TeleWindow()
{
    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        delete *i;


    XRenderFreePicture(_dpy, _picture);
    XRenderFreePicture(_dpy, _bufferPicture);

    XFreePixmap(_dpy, _bgPixmap);
    XFreePixmap(_dpy, _bufferPix);


    XRenderFreePicture(_dpy, _headerLeftPict);
    XRenderFreePicture(_dpy, _headerRightPict);
    XRenderFreePicture(_dpy, _headerMiddlePict);
    XFreePixmap(_dpy, _headerLeftPix);
    XFreePixmap(_dpy, _headerRightPix);
    XFreePixmap(_dpy, _headerMiddlePix);


    XFreeGC(_dpy, _gc);
    XDestroyWindow(_dpy, _win);
}


Display* TeleWindow::display()
{
    return _dpy;
}


Window TeleWindow::window()
{
    return _win;
}

Picture TeleWindow::picture()
{
    return _picture;
}


void TeleWindow::show()
{
    _shown = true;

    XMapWindow(_dpy, _win);

    XEvent event;

    memset(&event, 0, sizeof(event));
    event.type = ClientMessage;
    event.xclient.window = _win;
    event.xclient.message_type = XTools::_NET_WM_STATE;
    event.xclient.format = 32;
    event.xclient.data.l[0] = 1;
    event.xclient.data.l[1] = XTools::_NET_WM_STATE_FULLSCREEN;
    event.xclient.data.l[2] = 0;


    XSendEvent(_dpy, _rootWindow, False, SubstructureNotifyMask, &event);

    XTools::switchToWindow(_win);
}

void TeleWindow::hide()
{
    XUnmapWindow(_dpy, _win);

    _shown = false;
}

bool TeleWindow::shown()
{
    return _shown;
}



void TeleWindow::updateThumbnailsList()
{
    bool wasChanged = false;


    LinkedList<Window> windows = XTools::windowList(_rootWindow);
    for (LinkedList<Window>::Iter i = windows.head(); i; ++i)
        if (*i != _win)
        {
            bool found = false;
            for (LinkedList<Thumbnail*>::Iter j = _thumbnails.head(); j; ++j)
                if ((*j)->clientWindow() == *i)
                {
                    found = true;
                    break;
                }

            if (! found)
            {
                Thumbnail *th = new Thumbnail(this, *i);
                _thumbnails.append(th);
                wasChanged = true;
            }
        }

    LinkedList<Thumbnail*> thumbsToDelete;
    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        if (! windows.contains((*i)->clientWindow()))
            thumbsToDelete.append(*i);

    for (LinkedList<Thumbnail*>::Iter i = thumbsToDelete.head(); i; ++i)
    {
        removeThumbnail(*i);
        delete *i;
        wasChanged = true;
    }


    if (wasChanged)
        layoutThumbnails();
}


inline int min(int a, int b)
{
    return a < b ? a : b;
}


void TeleWindow::layoutThumbnails()
{
    int n = _thumbnails.size();

    if (n == 0)
    {
        paint();
        return;
    }

    int maxSize = 0;
    int maxColumns = 0;

    for (int columns = 1; columns <= n; ++columns)
    {
        int rows = (n + columns - 1) / columns;

        int widthWithBorder = _width / columns - 2 * XMargin;
        int heightWithBorder = _height / rows - 2 * YMargin;

        int width = widthWithBorder - 2 * _borderWidth;
        int height = heightWithBorder - _headerHeight - _borderWidth;

        int resultSize = height * _width / _height;
        if (width < resultSize)
            resultSize = width;

        if (resultSize > maxSize)
        {
            maxSize = resultSize;
            maxColumns = columns;
        }
    }

    int columns = maxColumns;
    int rows = (n + columns - 1) / columns;

    int tileWidth = _width / columns;
    int tileHeight = _height / rows;

    int thumbWidth = maxSize;
    int thumbHeight= maxSize * _height / _width;

    int widthWithBorder = thumbWidth + 2*_borderWidth;
    int heightWithBorder= thumbHeight+ _borderWidth + _headerHeight;

    int tileXOffset = (tileWidth - widthWithBorder) / 2;
    int tileYOffset = (tileHeight-heightWithBorder) / 2;

    int lastRowX = 0;

    int index = 0;
    LinkedList<Thumbnail*>::Iter thumb = _thumbnails.head();
    for (int row = 0; row < rows; ++row)
    {
        if (row == rows - 1)
        {
            lastRowX = (_width - (n - index) * tileWidth) / 2;
        }

        int tileY = row * tileHeight;

        for (int column = 0; column < columns; ++column, ++thumb, ++index)
        {
            if (thumb == 0)
                break;

            int tileX = lastRowX + column * tileWidth;

            int x = tileX + tileXOffset + _borderWidth;
            int y = tileY + tileYOffset + _headerHeight;

            (*thumb)->setGeometry(x, y, thumbWidth, thumbHeight);
        }
    }

    paint();
}


void TeleWindow::removeThumbnail(Thumbnail *thumb)
{
    _thumbnails.removeByValue(thumb);
}


void TeleWindow::eventLoop()
{
    XEvent event;


    XGrabKey(_dpy, _hotKeyCode, AnyModifier, _rootWindow, False, GrabModeAsync, GrabModeAsync);

    _breakEventLoop = false;
    while (! _breakEventLoop)
    {
        do
        {
            XNextEvent(_dpy, &event);

            if (event.xany.window == _rootWindow)
            {
                onRootEvent(&event);
                continue;
            }

            if (event.xany.window == _win)
            {
                onEvent(&event);
                continue;
            }
            else
            {
                bool found = false;
                for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
                {
                    if ((*i)->clientWindow() == event.xany.window)
                    {
                        (*i)->onClientEvent(&event);
                        found = true;
                        break;
                    }
                }
                if (found)
                    continue;
            }
        } while (XPending(_dpy));

        onIdle();
    }

    XUngrabKey(_dpy, _hotKeyCode, AnyModifier, _rootWindow);
}



void TeleWindow::onRootEvent(XEvent *event)
{
    if (event->type == KeyPress && event->xkey.keycode == _hotKeyCode)
        onHotKeyPress();
    if (event->type == KeyRelease && event->xkey.keycode == _hotKeyCode)
        onHotKeyRelease();
    else if (event->type == MapNotify)
        updateThumbnailsList();
//    else
//        printf("Unknown root event: %d\n", event->type);
}



void TeleWindow::onEvent(XEvent *event)
{
    if (event->type == ButtonPress)
        onButtonPress(&event->xbutton);
    else if (event->type == ButtonRelease)
        onButtonRelease(&event->xbutton);
    else if (event->type == MotionNotify)
        onButtonMotion(&event->xmotion);
    else if (event->type == Expose && event->xexpose.count < 1)
        //paint();
        _repaintOnIdle = true;
    else if (event->type == ConfigureNotify)
    {
        bool relayout = _width != event->xconfigure.width || _height != event->xconfigure.height;
        _width = event->xconfigure.width;
        _height= event->xconfigure.height;
        if (relayout)
            layoutThumbnails();
    }
}


void TeleWindow::paint()
{
    if (! _shown)
        return;


//    timeval before;
//    gettimeofday(&before, 0);

    XCopyArea(_dpy, _bgPixmap, _bufferPix, _gc,
        0, 0, _width, _height, 0, 0
    );

    XRenderColor borderColor;
    borderColor.red     = 0x8000;
    borderColor.green   = 0x8000;
    borderColor.blue    = 0x8000;
    borderColor.alpha   = 0x8000;

    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
    {
//        XRenderFillRectangle(_dpy, PictOpOver, _bufferPicture, &borderColor,
//            _scrollX + (*i)->x() - BorderSize, _scrollY + (*i)->y() - TitleSize,
//            (*i)->width() + 2*BorderSize, (*i)->height() + BorderSize + TitleSize
//        );

        int x = (*i)->x();
        int y = (*i)->y();
        int w = (*i)->realWidth();
        int h = (*i)->realHeight();

        XRenderComposite(_dpy, PictOpOver,
            _headerLeftPict, None, _bufferPicture,
            0, 0,
            0, 0,
            _scrollX + x - _borderWidth,
            _scrollY + y - _headerHeight,
            _headerLeftWidth, _headerHeight
        );

        XRenderComposite(_dpy, PictOpOver,
            _headerRightPict, None, _bufferPicture,
            0, 0,
            0, 0,
            _scrollX + x + w + _borderWidth - _headerRightWidth,
            _scrollY + y - _headerHeight,
            _headerRightWidth, _headerHeight
        );

        XRenderComposite(_dpy, PictOpSrc,
            _headerMiddlePict, None, _bufferPicture,
            0, 0,
            0, 0,
            _scrollX + x - _borderWidth + _headerLeftWidth,
            _scrollY + y - _headerHeight,
            w + 2 * _borderWidth - _headerLeftWidth - _headerRightWidth,
            _headerHeight
        );

        // Left border
        XRenderFillRectangle(_dpy, PictOpSrc, _bufferPicture, &_borderColor,
            _scrollX + x - _borderWidth, _scrollY + y,
            _borderWidth, h
        );
        // Right border
        XRenderFillRectangle(_dpy, PictOpSrc, _bufferPicture, &_borderColor,
            _scrollX + x + w, _scrollY + y,
            _borderWidth, h
        );
        // Bottom border
        XRenderFillRectangle(_dpy, PictOpSrc, _bufferPicture, &_borderColor,
            _scrollX + x - _borderWidth, _scrollY + y + h,
            w + 2*_borderWidth, _borderWidth
        );
    }


    XftColor fontColor;
    fontColor.pixel = 0;
    fontColor.color.red     = 0xffff;
    fontColor.color.green   = 0xffff;
    fontColor.color.blue    = 0xffff;
    fontColor.color.alpha   = 0xffff;


    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
    {
        XRectangle rect = {
            _scrollX + (*i)->x() - _borderWidth + _textLeftMargin,
            _scrollY + (*i)->y() - _headerHeight,
            (*i)->width() + 2*_borderWidth - _textLeftMargin - _textRightMargin,
            _headerHeight
        };
        Region clip = XCreateRegion();
        XUnionRectWithRegion(&rect, clip, clip);

        XftDrawSetClip(_xftDraw, clip);

        XftDrawStringUtf8(_xftDraw, &fontColor, _xftFont, 
            _scrollX + (*i)->x() - _borderWidth + _textLeftMargin,
            _scrollY + (*i)->y() + Settings::instance()->textYOffset(),
            (const FcChar8*)(*i)->title(),
            strlen((*i)->title())
        );
        
        XDestroyRegion(clip);



//        XRenderComposite(_dpy, PictOpOver,
//            _closePicture, None, _bufferPicture,
//            0, 0,
//            0, 0,
//            _scrollX + (*i)->x() + (*i)->width() - _closeImageWidth,
//            _scrollY + (*i)->y() - TitleSize + _closeImageYOffset,
//            _closeImageWidth, _closeImageHeight
//        );

    }

    for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        (*i)->paint(_bufferPicture);






    XCopyArea(_dpy, _bufferPix, _win, _gc,
        0, 0, _width, _height, 0, 0
    );

//    timeval after;
//    gettimeofday(&after, 0);
//    printf("---------\n");
//    printf("%ld:%ld\n", before.tv_sec, before.tv_usec);
//    printf("%ld:%ld\n", after.tv_sec, after.tv_usec);
}


void TeleWindow::onThumbRedrawed(Thumbnail *thumb)
{
    XCopyArea(_dpy, _bufferPix, _win, _gc,
        _scrollX + thumb->x(), _scrollY + thumb->y(),
        thumb->width(), thumb->height(),
        _scrollX + thumb->x(), _scrollY + thumb->y()
    );
}



void TeleWindow::onHotKeyPress()
{
    if (_hotKeyPressed)
    {
        if (_shown)
            hide();
        XTools::showDesktop();
        return;
    }

    _hotKeyPressed = true;

    show();
}

void TeleWindow::onHotKeyRelease()
{
    char keysPressed[32];
    XQueryKeymap(_dpy, keysPressed);
    if ( (keysPressed[_hotKeyCode >> 3] >> (_hotKeyCode & 0x07)) & 0x01)
        return;

    _hotKeyPressed = false;
}



void TeleWindow::onButtonPress(XButtonEvent *event)
{
    _wasScrolling = false;
    _buttonPressed = true;
    _buttonPressX = event->x;
    _buttonPressY = event->y;
    _scrollBaseX = _scrollX;
    _scrollBaseY = _scrollY;
}

void TeleWindow::onButtonRelease(XButtonEvent *event)
{
    _buttonPressed = false;

    if (! _wasScrolling)
    {
        bool missed = true;

        for (LinkedList<Thumbnail*>::Iter i = _thumbnails.head(); i; ++i)
        {
            // Checking close button
            {
                int closeX = (*i)->x() + (*i)->width() + _borderWidth - _closeButtonXSpan;
                int closeY = (*i)->y() - _headerHeight;


                if (- _scrollX + event->x >= closeX &&
                    - _scrollX + event->x <= closeX + _closeButtonXSpan &&
                    - _scrollY + event->y >= closeY &&
                    - _scrollY + event->y <= closeY + _closeButtonYSpan
                    )
                {
                    (*i)->closeClient();
                    missed = false;
                }
            }

            if (missed && (*i)->inside(- _scrollX + event->x, - _scrollY + event->y))
            {
                (*i)->switchToClient();
                hide();
                missed = false;
            }
        }

        if (missed)
            XTools::showDesktop();
    }
}


void TeleWindow::onButtonMotion(XMotionEvent *event)
{
    if (_buttonPressed && Settings::instance()->scrollingEnabled())
    {
        _scrollX = _scrollBaseX + event->x - _buttonPressX;
        _scrollY = _scrollBaseY + event->y - _buttonPressY;

        if (! _wasScrolling)
        {
            if (abs(event->x - _buttonPressX) > 20 ||
                abs(event->y - _buttonPressY) > 20)
                _wasScrolling = true;
        }


//        paint();
        _repaintOnIdle = true;
    }
}


void TeleWindow::onIdle()
{
    if (_repaintOnIdle)
    {
        paint();
        _repaintOnIdle = false;
    }
}
