#include "Settings.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>


#define CONFIG_FILE     "/etc/telescope.conf"


Settings* Settings::_instance = 0;

Settings* Settings::instance() { return _instance; }


Settings::Settings()
{
    _instance = this;

    // Default values
    _scrollingEnabled = false;

    _backgroundFilename = 0; // Will be set later if not in config file

    _headerLeftFilename = strdup("/usr/share/telescope/header-left.png");
    _headerRightFilename = strdup("/usr/share/telescope/header-right.png");
    _headerMiddleFilename = strdup("/usr/share/telescope/header-middle.png");
    _borderColor = strdup("#303030");

    _borderWidth = 3;
    _textLeftMargin = 7;
    _textRightMargin = 25;
    _closeButtonXSpan = 40;
    _closeButtonYSpan = 40;

    _fontSize = 18;    
    _textYOffset = -5;


    // Reading
    FILE *f = fopen(CONFIG_FILE, "r");
    if (f == 0)
        fprintf(stderr, "Config file not found: " CONFIG_FILE "\n");
    else
    {
        while (! feof(f))
        {
            char line[256];
            if (fgets(line, sizeof(line), f) != 0)
                parseLine(line);
        }


        fclose(f);
    }


    // Use hildon background if no background specified in config file
    if (_backgroundFilename == 0)
    {
        FILE *homeBgFile = fopen("/home/user/.osso/hildon-desktop/home-background.conf", "r");
        if (homeBgFile)
        {
            char line[1024];
            static const char *prefix1 = "BackgroundImage=";
            static const char *prefix3 = "CachedAs=";
            while (fgets(line, sizeof(line), homeBgFile) != 0)
            {
                if (strstr(line, prefix1) == line)
                {
                    char tmp[1024];
                    strncpy(
                        tmp,
                        &line[strlen(prefix1)],
                        strcspn(
                            &line[strlen(prefix1)],
                            "\n"
                        )
                    );
                    _backgroundFilename = urldecode(tmp);
                }
                else if (strstr(line, prefix3) == line)
                {
                    int valuelen = strcspn(&line[strlen(prefix3)], "\n");
                    if (valuelen > 0)
                    {
                        free(_backgroundFilename);
                        _backgroundFilename = strndup(&line[strlen(prefix3)], valuelen);
                    }
                }
            }

            fclose(homeBgFile);
        }

        if (_backgroundFilename == 0)
            _backgroundFilename = (char*)calloc(1, 1); // Emtpy string fallback
    }
}

Settings::~Settings()
{
    free(_backgroundFilename);
    free(_headerLeftFilename);
    free(_headerRightFilename);
    free(_headerMiddleFilename);
    free(_borderColor);
}



int hexdigit(char hex)
{
    if (hex >= '0' && hex <= '9')
        return hex - '0';

    if (hex >= 'A' && hex <= 'F')
        return hex - 'A' + 0xA;

    if (hex >= 'a' && hex <= 'f')
        return hex - 'a' + 0xA;

    return 0;
}



char* Settings::urldecode(const char *url)
{
    if (strstr(url, "file://") == url)
        url += 7;

    char *result = (char*)malloc(strlen(url));
    char *r = result;

    while (*url)
    {
        if (*url == '%')
        {
            ++url;
            int val = hexdigit(*url++);
            val <<= 4;
            val |= hexdigit(*url++);
            *r++ = val;
        }
        else
            *r++ = *url++;
    }

    return result;
}



#define SPACECHARS  " \t"

void Settings::parseLine(const char *line)
{
    const char *key = &line[strspn(line, SPACECHARS)];

    if (*key == '\n')
        return; // Empty line

    if (*key == '\0')
        return; // Empty line

    if (*key == '#')
        return; // Comment

    int keylen = strcspn(key, " =:");


    const char *value = &key[keylen + strspn(&key[keylen], " =:")];
    int valuelen = strcspn(value, "\n");

    if (keylen >= 200) keylen = 199;
    if (valuelen >= 200) valuelen = 199;

    char keybuf[200];
    strncpy(keybuf, key, keylen);
    keybuf[keylen] = '\0';
    char valuebuf[200];
    strncpy(valuebuf, value, valuelen);
    valuebuf[valuelen] = '\0';


    parseOpt(keybuf, valuebuf);
}


void Settings::parseOpt(const char *key, const char *value)
{
    printf("[Opt]\t'%s' = '%s'\n", key, value);

    if (strcmp(key, "scrolling.enabled") == 0)
    {
        _scrollingEnabled = parseBool(value);

        if (_scrollingEnabled)
            printf("Scrolling enabled\n");
        else
            printf("Scrolling disabled\n");
    }
    else if (strcmp(key, "background.filename") == 0)
    {
        free(_backgroundFilename);
        _backgroundFilename = strdup(value);

        printf("Background: '%s'\n", _backgroundFilename);
    }
    else if (strcmp(key, "header.left.filename") == 0)
    {
        free(_headerLeftFilename);
        _headerLeftFilename = strdup(value);
    }
    else if (strcmp(key, "header.right.filename") == 0)
    {
        free(_headerRightFilename);
        _headerRightFilename = strdup(value);
    }
    else if (strcmp(key, "header.middle.filename") == 0)
    {
        free(_headerMiddleFilename);
        _headerMiddleFilename = strdup(value);
    }
    else if (strcmp(key, "border.color") == 0)
    {
        free(_borderColor);
        _borderColor = strdup(value);
    }
    else if (strcmp(key, "border.width") == 0)
        _borderWidth = atoi(value);
    else if (strcmp(key, "text.leftMargin") == 0)
        _textLeftMargin = atoi(value);
    else if (strcmp(key, "text.rightMargin") == 0)
        _textRightMargin = atoi(value);
    else if (strcmp(key, "closeButton.xSpan") == 0)
        _closeButtonXSpan = atoi(value);
    else if (strcmp(key, "closeButton.ySpan") == 0)
        _closeButtonYSpan = atoi(value);
    else if (strcmp(key, "text.font.size") == 0)
        _fontSize = atoi(value);
    else if (strcmp(key, "text.yOffset") == 0)
        _textYOffset = atoi(value);
}


bool Settings::parseBool(const char *value)
{
    if (*value == '1')
        return true;
    if (strcmp(value, "on") == 0)
        return true;
    if (strcmp(value, "yes") == 0)
        return true;

    return false;
}
