#
# This is the main module for SplitTheTab.
# Copyright (C) 2009, Henning Spruth
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA  02110-1301, USA.
#



import gtk
import os

try:
    import hildon
    isHildon=True
except:
    isHildon=False

from Preferences import Preferences
from Database import Database
from CurrencyModel import CurrencyModel
from AccountModel import AccountModel
from PaymentModel import PaymentModel
from MainWindow import MainWindow
from PaymentEditor import PaymentEditor
from Report import Report
from Exporter import Exporter


version="1.0"

if not isHildon:
    class  hildon:
        def __init__(self):
	    pass
	    
	class Program:
	    def __init__(self):
		pass


def errorPopup(message, toplevel=None):
    dialog=gtk.MessageDialog(toplevel,
                             gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                             gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE,
                             message)
    dialog.set_title('Error')
    dialog.connect('response', lambda dialog, response: dialog.destroy())
    dialog.run()


class Main(hildon.Program):

    
    def __init__(self):
        hildon.Program.__init__(self)
        self.prefs=Preferences()
        self.currencyModel=CurrencyModel()
        self.accountModel=AccountModel()
        self.paymentModel=PaymentModel(self.currencyModel, self.accountModel)
        self.db=Database(self.prefs)

        
    def new_cb(self, data):
        # Callback from MainWindow for File->New
        fn=self.queryFileName("Specify the new database's file name",
                              gtk.FILE_CHOOSER_ACTION_SAVE)
        if fn:
            if os.path.exists(fn):
                errorPopup("The specified file already exists",\
                           self.mainWindow.window())
            else:
                self.db.createNewDatabase(fn)
                self.updateData(fn)

    def open_cb(self, data):
        # Callback from MainWindow for File->New
        fn=self.queryFileName("Specify the database file name",
                              gtk.FILE_CHOOSER_ACTION_OPEN)
        if fn:
            if not os.path.isfile(fn):
                errorPopup("The specified file does not exist or is not a file",
                           self.mainWindow.window())
            else:
                self.db.openDatabase(fn)
                self.updateData(fn)
        
    def vacuum_cb(self, data):
        self.sql.execute("vacuum")

    def report_cb(self, data):
        r=Report(self.sql,
                 self.currencyModel,
                 self.accountModel,
                 self.paymentModel)
        dialog=r.createGui(self.mainWindow.window())
        dialog.run()
        dialog.destroy()

    def summary_cb(self, data):
        r=Report(self.sql,
                 self.currencyModel,
                 self.accountModel,
                 self.paymentModel)
        dialog=r.createSummaryGui(self.mainWindow.window())
        dialog.run()
        dialog.destroy()

    def export_cb(self,data):
        fn=self.queryFileName("Specify an ODS file to create",
                              gtk.FILE_CHOOSER_ACTION_SAVE)
        if fn:
            exporter=Exporter(self.sql)
            exporter.writeOdf(fn)
            if isHildon:
                hildon.hildon_banner_show_information(self.mainWindow.window(),
                                                      None,
                                                      "Spreadsheet saved")

    def about_cb(self, data):
        message="SplitTheTab - keeping track of common expenses\nbetween friends\n\nVersion "+str(version)+", Copyright (c) 2009 Henning Spruth\n\nThis program is free software and licensed under the\nGPL, see /usr/share/common-licenses/GPL-2"
        dialog=gtk.MessageDialog(self.mainWindow.window(),
                             gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                             gtk.MESSAGE_INFO, gtk.BUTTONS_CLOSE,
                             message)
        dialog.set_title('About')
        dialog.connect('response', lambda dialog, response: dialog.destroy())
        dialog.run()


    def purge_cb(self, dummy):
        #
        # Permanently pruge all elements that are marked as deleted from the
        # database
        #

        # determine how many deleted items exist
        cursor=self.sql.cursor()
        cursor.execute("SELECT COUNT(id) FROM payment WHERE isDeleted=1;")
        count=cursor.fetchone()[0]

        # Note: might show a an error popup if there are no entries to purge
        # instead of saying "do you want to delete 0 entries" ...

        # get confirmation from the user
        dialog=gtk.MessageDialog(parent=self.mainWindow.window(),
                                 flags=gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                                 type=gtk.MESSAGE_WARNING,
                                 buttons=gtk.BUTTONS_OK_CANCEL,
                                 message_format="Do you really want to purge "+str(count)+" deleted entries from the database?"
                                 )
        if dialog.run()==gtk.RESPONSE_OK:
            # User said yes - remove items
            self.sql.execute('DELETE FROM payment WHERE isDeleted=1;')
        # we're done with this dialog
        dialog.destroy()


    def createGui(self):


        self.mainWindow=MainWindow(self.currencyModel,
                                   self.accountModel,
                                   self.paymentModel,
                                   self)
        
        self.mainWindow.window().show()


    def queryFileName(self, title, action):
        if isHildon:
            dialog=hildon.FileChooserDialog(self.mainWindow.window(), action)
        else:
            dialog=gtk.FileChooserDialog(
                title=title,
                action=action,
                buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK)
                )
        response=dialog.run()
        fn=dialog.get_filename()
        dialog.destroy()
        if response==gtk.RESPONSE_CANCEL:
            fn=None
        return fn


    def updateData(self, fn):
        self.prefs.setDbFn(fn)
        self.sql=self.db.getSql()
        self.currencyModel.initFromDb(self.sql)
        self.accountModel.initFromDb(self.sql)
        self.paymentModel.initFromDb(self.sql)
        #self.paymentModel.setSelection(self.paymentModel.store.get_iter(0))
        # Update the editor's pulldown menus and other widget
        self.mainWindow.editor.updateFromCurrencyModel()
        self.mainWindow.editor.updateFromAccountModel()

        # Note: won't save updated preferences if database initialization
        # throws an exceptino
        self.prefs.saveIfNeeded()
        self.mainWindow.setTitle(fn)


    def run(self):
        fn=self.prefs.getDbFn()
        if not fn:
            fn=self.queryFileName("Specify the name of a new database file",
                                  gtk.FILE_CHOOSER_ACTION_SAVE)
            if not fn:
                gtk.main_quit()
        if fn and os.path.isfile(fn):
            #print "Loading ",fn
            self.db.openDatabase(fn)
        else:
            self.db.createNewDatabase(fn)

        self.updateData(fn)
        gtk.main()



    def load(self):
        self.db=SplitTheTab.Database.Database(self.prefs)


