import time
from gtkeasyapp.log import *
from gtkeasyapp.easyprocess import *
from const import *
from structs import *
from foxyprocesses import *


##SpeedcamAlerterInternetConnection process
##Manage internet connection
class SpeedcamAlerterProcess(EasyProcess):
	##Constants
	FOXYTAGS_TIME_VALIDITY = 240.0 #seconds
	FOXYTAGS_DIST_VALIDITY = 5000 #meters
	TIMEOUT_NORMAL = 1.0 #seconds
	TIMEOUT_ALERT = 0.25 #seconds
	MIN_SPEED = 2 #km/h
	
	MIN_POST_SPEED = 20 #km/h
	EPH_PRECISION = 35 #meters
	EPD_PRECISION = 7 #degrees
	EPS_PRECISION = 7 #km/h
	

	##Constructor
	##@param ui : EasyWindow instance
	##@connectionName : name of internet connection
	##@param onConnected : callback when connected
	def __init__(self, ui):
		#super constructor
		EasyProcess.__init__(self, PROCESS_SCA, ui)
		
		#store usefull pref values
		self.__alertFixedSpeedcam = ui._prefFile.getItem('alarms', 'fixed')
		self.__alertMobileSpeedcam = ui._prefFile.getItem('alarms', 'mobile')
		self.__alertGhostSpeedcam = ui._prefFile.getItem('alarms', 'ghost')
		
		#configure process
		self._timeout = self.TIMEOUT_NORMAL #execute process every 1 seconds
		
		#process
		self.__gps = ui.getProcess(PROCESS_GPS)
		self.__foxy = ui.getProcess(PROCESS_FOX)
		
		#internal flags
		self.__gpsOk = False
		self.__internetOk = False
		self.__foxyOk = False
		
		#set members
		self.__previousAlert = False
		self.__currentPosition = None
		self.__speedcams = []
		self.__closestSpeedcam = None
		self.__lastFoxyTagsUpdateTime = 0
		self.__lastFoxyTagsUpdateUtm = None
		self.__alerting = False
		self.__mutexSpeedcams = threading.Lock()
		self.__positionToPost = GpsInfos()
		
		
	##Process : check connection
	def _process(self):
	
		#check if speedcamalerter is ready to work
		if self.__gpsOk: #and self.__internetOk and self.__foxyOk:
			#get current position
			currentPosition = GpsInfos( self.__gps.getData() )
			
			#check if speed is sufficient to update radar
			#TODO check angle ?
			if currentPosition.speed>=self.MIN_SPEED:
				#save current position
				self.__currentPosition = currentPosition
			
				#compute closest speedcam
				self.__computeClosestSpeedcam()
			
				#alert
				self.__checkAlert()
					
				#update UI
				self.__updateUi()
			
				#check if foxytags are still valid
				self.__checkFoxyTags()
			
			
	##Stopping process
	def _stopping(self):
		#nothing to stop
		pass
				
		
	##Check process (if running or existing)
	def __checkProcess(self, process):
		if process==None:
			return True
		elif process!=None and not process.isAlive():
			return True
		else:
			return False
	
	
	##Compute closest speedcam
	def __computeClosestSpeedcam(self):
		#init
		closestSpeedcam = None
		
		#lock speedcams
		self.__mutexSpeedcams.acquire()
		
		#first speedcam is the closest
		if len(self.__speedcams)>0:
			#compute distance between car and speedcam
			self.__speedcams[0].computeDistanceFromCar(self.__currentPosition.utm)
		
			#compute if speedcam and car are in same direction
			self.__speedcams[0].computeCarDirection(self.__currentPosition.angle)
		
			#save as closest one
			closestSpeedcam = self.__speedcams[0]
			
		#search closest one
		for i in range(1, len(self.__speedcams)):
			#compute distance between car and speedcam
			self.__speedcams[i].computeDistanceFromCar(self.__currentPosition.utm)
		
			#compute if speedcam and car are in same direction
			self.__speedcams[i].computeCarDirection(self.__currentPosition.angle)
			
			#check if current speedcam is the closest
			if self.__speedcams[i].distanceFromCar<closestSpeedcam.distanceFromCar:
				#new one is closest
				closestSpeedcam = self.__speedcams[i]
		
		#release lock
		self.__mutexSpeedcams.release()
		
		#save closest speedcam
		self.__closestSpeedcam = closestSpeedcam

	
	##Check if alert is necessary
	def __checkAlert(self):
		#init
		alert = False
		
		#search if alert is necessary
		if self.__closestSpeedcam!=None:
			if self.__closestSpeedcam.carDirection and self.__closestSpeedcam.foxytag.kind==FoxyTag.FOXYTAG_FIXED and self.__closestSpeedcam.distanceFromCar<=self.__alertFixedSpeedcam.getValue():
				alert = True
			elif self.__closestSpeedcam.carDirection and self.__closestSpeedcam.foxytag.kind==FoxyTag.FOXYTAG_MOBILE and self.__closestSpeedcam.distanceFromCar<=self.__alertMobileSpeedcam.getValue():
				alert = True
			elif self.__closestSpeedcam.carDirection and self.__closestSpeedcam.foxytag.kind==FoxyTag.FOXYTAG_GHOST and self.__closestSpeedcam.distanceFromCar<=self.__alertGhostSpeedcam.getValue():
				alert = True
		
		#confirm alert?
		if self.__previousAlert and alert:
			#alarm confirmed
			pass
		else:
			#alarm not confirmed, save current state
			self.__previousAlert = alert
			alert = False
				
		#alert?
		if self.__currentPosition.angle>=0.0 and self.__currentPosition.angle<=360.0:
			if alert and not self.__alerting:
				#turn on alert
				self.__alerting = True
				self._ui.updateUi(UI_RADAR_ALERT, True)
				
				#update thread timeout
				self._timeout = self.TIMEOUT_ALERT
				
			elif not alert and self.__alerting:
				#turn off alert
				self.__alerting = False
				self._ui.updateUi(UI_RADAR_ALERT, False)
				
				#update thread timeout
				self._timeout = self.TIMEOUT_NORMAL

	

	##Update User Interface
	def __updateUi(self):
		#init
		distanceFromSpeedcam = NAN
		
		#update closest speedcam
		if self.__closestSpeedcam!=None:
			distanceFromSpeedcam = self.__closestSpeedcam.distanceFromCar
			self._ui.updateUi(UI_CLOSEST_SPEEDCAM, self.__closestSpeedcam)
	
		#check if current position is valid
		if self.__currentPosition.speed<=350.0 and self.__currentPosition.angle>=0.0 and self.__currentPosition.angle<=360.0:
			#update current position
			self._ui.updateUi(UI_CURRENT_POS, self.__currentPosition, distanceFromSpeedcam)
		
			#update speed
			self._ui.updateUi(UI_SPEED, self.__currentPosition.speed)
	

	##Check if foxy tags are still valid according to distance and last update time
	def __checkFoxyTags(self):
		#update by time
		diffTime = time.time() - self.__lastFoxyTagsUpdateTime
		if self.__lastFoxyTagsUpdateTime==0 or diffTime>=self.FOXYTAGS_TIME_VALIDITY or GISCompute.distanceUtm(self.__lastFoxyTagsUpdateUtm, self.__currentPosition.utm)>=self.FOXYTAGS_DIST_VALIDITY:
			#update position and time infos
			self.__lastFoxyTagsUpdateTime = time.time()
			self.__lastFoxyTagsUpdateUtm = self.__currentPosition.utm
		
			#foxytags update necessary
			self.__foxy.addNewAction(FoxyTagsProcess.ACTION_GET, self.__onFoxyTagsDownloaded, False, self.__currentPosition)
			
	
	##Foxytags downloaded
	def __onFoxyTagsDownloaded(self, result):
		if result.result:
			#download completed
			
			#lock self.__speedcams
			self.__mutexSpeedcams.acquire()
			
			#remove existing speedcams
			del self.__speedcams[0:len(self.__speedcams)]
			
			#add new speedcams
			for speedcam in result.results[0]:
				self.__speedcams.append(speedcam)
				
			#release lock
			self.__mutexSpeedcams.release()
				
			#update radar speedcams
			self._ui.updateUi(UI_UPDATE_SPEEDCAMS, self.__speedcams)
			
			#update ui
			self._ui.updateUi(UI_STATUS, STATUS_INFOS, '%d FoxyTags getted' % len(result.results[0]) )
			
		else:
			#download failed
			Log.log('SpeedcamAlerter::__onFoxyTagsDownloaded : Unable to download foxytags')
			
			#update ui
			self._ui.updateUi(UI_MESSAGE, 'Unable to get FoxyTags, drive carrefully!')
			self._ui.updateUi(UI_STATUS, STATUS_INFOS, 'unable to get tags')

				
	##Post new foxytag
	def __postFoxyTag(self, kind, carDirection):
		#check if data is valid
		Log.log('eph=%s epd=%s eps=%s' % (self.__positionToPost.eph, self.__positionToPost.epd, self.__positionToPost.eps))
		if self.__positionToPost.eph>=self.EPH_PRECISION:
			#precision is not valid
			self._ui.updateUi(UI_MESSAGE, 'GPS data not enough accurate, unable to post FoxyTags')
			self._ui.updateUi(UI_STATUS, STATUS_INFOS, 'unable to post tag')
			
		elif self.__positionToPost.speed<=self.MIN_POST_SPEED:
			#speed is not valid
			self._ui.updateUi(UI_MESSAGE, 'You running to slow (<20km/h), unable to post FoxyTags')
			self._ui.updateUi(UI_STATUS, STATUS_INFOS, 'unable to post tag')
		
		else:
			#all is valid
		
			#prepare FoxyTag to post
			foxytag = FoxyTag()
			foxytag.lat = self.__positionToPost.wgs84.latitude
			foxytag.long = self.__positionToPost.wgs84.longitude
			foxytag.kind = kind
			foxytag.angle = self.__positionToPost.angle
			foxytag.postedOpposite = not carDirection
			Log.log('FoxyTag::__postFoxyTag : %s' % (str(foxytag)))
		
			#add post action
			self.__foxy.addNewAction(FoxyTagsProcess.ACTION_POST, self.__onFoxyTagsUploaded, False, foxytag, self.__positionToPost.speed)
		
	
	##Foxytag uploaded
	def __onFoxyTagsUploaded(self, result):
		if result.result:
			#upload completed
			Log.log('SpeedcamAlerter::__onFoxyTagsUploaded : New foxytag posted')
			
			#update ui
			self._ui.updateUi(UI_STATUS, STATUS_INFOS, 'new FoxyTag posted')
			
		else:
			#upload failed
			Log.log('SpeedcamAlerter::__onFoxyTagsUploaded : Unable to upload new foxytag')
			
			#update ui
			self._ui.updateUi(UI_MESSAGE, 'Unable to post new FoxyTag')
			self._ui.updateUi(UI_STATUS, STATUS_INFOS, 'unable to post tag')
		
	
	##Callback when connected to internet
	def connectionEstablished(self, connected):
		self.__internetOk = connected
			
	
	##Callback when GPS fixed
	def gpsFixed(self):
		self.__gpsOk = True
		
		
	##Callback when foxytag connected
	def foxytagConnected(self, connected):
		self.__foxyOk = connected
		
		
	##Store position when user want to tag speedcam
	def savePositionToPost(self):
		if self.__currentPosition!=None:
			self.__positionToPost.copy(self.__currentPosition)
		
	
	##Post fixed speedcam
	def postFixedSpeedcam(self):
		self.__postFoxyTag(FoxyTag.FOXYTAG_FIXED, True)
		
		
	##Post opposite fixed speedcam
	def postOppositeFixedSpeedcam(self):
		self.__postFoxyTag(FoxyTag.FOXYTAG_FIXED, False)
		

	##Post fixed speedcam
	def postMobileSpeedcam(self):
		self.__postFoxyTag(FoxyTag.FOXYTAG_MOBILE, True)
		
		
	##Post opposite fixed speedcam
	def postOppositeMobileSpeedcam(self):
		self.__postFoxyTag(FoxyTag.FOXYTAG_MOBILE, False)
	
	
	##Remove speedcam
	def removeSpeedcam(self):
		self.__postFoxyTag(FoxyTag.FOXYTAG_CANCEL, False)
	
	