import math
import cairo
import sys
import os
import traceback
import gobject
import pygtk
pygtk.require('2.0')
import gtk
from gtkeasyapp.log import Log
from gtkeasyapp.gis import *
from const import *
from foxytag import FoxyTag

path = os.path.dirname(__file__)
RESOURCES_DIR = os.path.join(path, 'res')

##FoxyRadarPoint class
##Contains coordinates of Foxy tag point
class FoxyRadarPoint():
	##members
	x = 0
	y = 0
	
	##To string
	def __str__(self):
		return 'FoxyRadarPoint(%d, %d)' % (self.x, self.y)

		
##Store all image infos needed by foxyradar (avoid to compute all the time same thing)
class FoxyRadarImage():
	##constructor
	def __init__(self, image):
		#set members
		self.image = image
		self.pixbuf = cairo.ImageSurface.create_from_png(os.path.join(RESOURCES_DIR,image))
		self.width = cairo.ImageSurface.get_width(self.pixbuf)
		self.height = cairo.ImageSurface.get_height(self.pixbuf)


##FoxyRadar class
##GTK widget to display speedcam radar
class FoxyRadar(gtk.DrawingArea):
	##Constants
	BORDER = 10 #in pixel
	SPEEDCAM_CANCEL_DISTANCE = 75 #in meter

	##Constructor
	def __init__(self):
		#super constructor
		gtk.DrawingArea.__init__(self)
		
		#set members
		self.__alert = False #alert flag
		self.__speedcams = None #speedcams to display
		self.__currentPosition = None #GpsInfos instance of current position
		self.__circlesCount = 5
		self.__buttonsDisplay = False
		self.__distanceFromSpeedcam = 10000
		self.__offsets = None
		self.__modeChooseSpeedcam = True
		self.__chooseFixedSpeedcam = True
		self.__testMode = False
		self.__cacheRadarNormal = None
		self.__cacheRadarAlert = None
		self.__displayNorthDirection = False
		
		#load images
		self.__car = FoxyRadarImage('car.png')
		self.__waiting_gps = FoxyRadarImage('waiting_gps.png')
		
		self.__speedcam_fixed = FoxyRadarImage('speedcam_fixed_40x40.png')
		self.__speedcam_mobile = FoxyRadarImage('speedcam_mobile_40x40.png')
		self.__speedcam_ghost = FoxyRadarImage('speedcam_ghost_40x40.png')
		self.__speedcam_opposite = FoxyRadarImage('speedcam_opposite_40x40.png')
		
		self.__button_cancel = FoxyRadarImage('button_cancel.png')
		self.__button_add_fixed = FoxyRadarImage('button_add_fixed.png')
		self.__button_add_mobile = FoxyRadarImage('button_add_mobile.png')
		self.__button_delete = FoxyRadarImage('button_delete.png')
		self.__button_direction_car = FoxyRadarImage('button_direction_car.png')
		self.__button_direction_opposite = FoxyRadarImage('button_direction_opposite.png')
		self.__button_direction_both = FoxyRadarImage('button_direction_both.png')
		
		#create custom signals
		gobject.signal_new('remove-speedcam', FoxyRadar, gobject.SIGNAL_ACTION, gobject.TYPE_NONE, ())
		gobject.signal_new('add-fixed-speedcam', FoxyRadar, gobject.SIGNAL_ACTION, gobject.TYPE_NONE, ())
		gobject.signal_new('add-opposite-fixed-speedcam', FoxyRadar, gobject.SIGNAL_ACTION, gobject.TYPE_NONE, ())
		gobject.signal_new('add-mobile-speedcam', FoxyRadar, gobject.SIGNAL_ACTION, gobject.TYPE_NONE, ())
		gobject.signal_new('add-opposite-mobile-speedcam', FoxyRadar, gobject.SIGNAL_ACTION, gobject.TYPE_NONE, ())
		gobject.signal_new('action-canceled', FoxyRadar, gobject.SIGNAL_ACTION, gobject.TYPE_NONE, ())
		gobject.signal_new('action-started', FoxyRadar, gobject.SIGNAL_ACTION, gobject.TYPE_NONE, ())
		
		#enable mouse click on drawing area
		self.add_events(gtk.gdk.BUTTON_PRESS_MASK)
		
		#events
		self.connect("expose_event", self.expose)
		self.connect('button-press-event', self.__click)
		self.connect('size_allocate', self.resize)
		
		
	
	
	#############################################################################################
	## EVENTS
	#############################################################################################
	
	##click event
	def __click(self, widget, event):
		if not self.__buttonsDisplay:
			#emit signal
			self.emit('action-started')
		
			#buttons are not displayed, display them
			self.__buttonsDisplay = True
			
			#choose speedcam action first
			self.__modeChooseSpeedcam = True
			
			#redraw canvas to display buttons
			self.redraw_canvas()
			
		else:
			#buttons are already displayed, detect which button are clicked
			
			#get mouse coordinates
			mouse = FoxyRadarPoint()
			mouse.x = event.get_coords()[0]
			mouse.y = event.get_coords()[1]
			
			#compute center coordinates
			boundaries = self.__getBoundaries()
			if boundaries!=None:
				center = self.__utmPointToFoxyRadarPoint(self.__currentPosition.utm, boundaries)
				unit = FoxyRadarPoint()
				unit.x = center.x / 2
				unit.y = center.y / 2
				
				#detect clicked zone
				topLeftButton = False
				topRightButton = False
				bottomLeftButton = False
				bottomRightButton = False
				if mouse.y<=center.y and mouse.x<=center.x:
					#top left
					topLeftButton = True
				elif mouse.y<=center.y and mouse.x>=center.x:
					#top right
					topRightButton = True
				elif mouse.y>=center.y and mouse.x<=center.x:
					#bottom left
					bottomLeftButton = True
				elif mouse.y>=center.y and mouse.x>=center.x:
					#bottom right
					bottomRightButton = True
				else:
					#bad position
					pass
				
				#execute clicked action
				signals = None
				if self.__modeChooseSpeedcam:
					#speedcam mode
					if topLeftButton:
						#add fixed speedcam
						self.__chooseFixedSpeedcam = True
						#direction mode
						self.__modeChooseSpeedcam = False
					elif topRightButton:
						#add mobile speedcam
						self.__chooseFixedSpeedcam = False
						#direction mode
						self.__modeChooseSpeedcam = False
					elif bottomLeftButton and self.__alert and self.__distanceFromSpeedcam<=self.SPEEDCAM_CANCEL_DISTANCE:
						#delete speedcam
						signals = ('remove-speedcam',)
						#speedcam mode next time
						self.__modeChooseSpeedcam = True
					elif bottomRightButton:
						#cancel
						signals = ('action-canceled',)
					else:
						#stay in choose speedcam mode
						self.__modeChooseSpeedcam = True
						self.__chooseFixedSpeedcam = True
				else:
					#direction mode
					if topLeftButton:
						#car direction
						if self.__chooseFixedSpeedcam:
							signals = ('add-fixed-speedcam',)
						else:
							signals = ('add-mobile-speedcam',)
					elif topRightButton:
						#opposite direction
						if self.__chooseFixedSpeedcam:
							signals = ('add-opposite-fixed-speedcam',)
						else:
							signals = ('add-opposite-mobile-speedcam',)
					elif bottomLeftButton:
						#both directions
						if self.__chooseFixedSpeedcam:
							signals = ('add-fixed-speedcam', 'add-opposite-fixed-speedcam')
						else:
							signals = ('add-mobile-speedcam', 'add-opposite-mobile-speedcam')
					elif bottomRightButton:
						#action canceled
						signals = ('action-canceled',)
					else:
						#nothing to do
						pass
				
				if signals!=None:
					#emit signals
					for signal in signals:
						self.emit(signal)
					
					#there were signals, action done, remove buttons
					self.__buttonsDisplay = False
				
				#redraw canvas
				self.redraw_canvas()
				
				
				
				
				
	#############################################################################################
	## WIDGET DRAWING
	#############################################################################################
	
	##Create radar pixmap for caching
	##@param rect : widget size
	##@param boundaries : radar boundaries
	##@param center : center position
	##@param alertMode : True to draw radar in alertMode
	##@param testMode : True to draw radar in test mode
	def __createRadarPixmap(self, rect, boundaries, center, alertMode, testMode):
		#init
		pixmap = gtk.gdk.Pixmap(self.window, rect.width, rect.height, -1)
		#gc = self.get_style().base_gc[gtk.STATE_NORMAL]
		#textGc = self.get_style().text_gc[gtk.STATE_NORMAL]
		context = pixmap.cairo_create()
	
		#draw background
		#context.rectangle(rect.x, rect.y, rect.width, rect.height)
		context.rectangle(0, 0, rect.width, rect.height)
		context.set_source_rgb(1.0, 1.0, 1.0)
		context.fill()
			
		#draw main circle
		context.arc(center.x, center.y, self.__minPixelRadius*self.__circlesCount, 0, 2*math.pi)
		if testMode:
			context.set_source_rgb(1.0, 0.75, 0.75)
		else:
			context.set_source_rgba(0.9, 0.9, 0.9)
		context.fill()
			
		#draw circles
		for i in range(self.__circlesCount):
			context.arc(center.x, center.y, self.__minPixelRadius * (i+1), 0, 2*math.pi)
			context.set_source_rgb(0, 0, 0)
			context.set_line_width(0.5)
			context.stroke()
			
		#draw car vision
		context.arc(center.x, center.y, self.__maxPixelRadius, GISConvert.degToRad(225), GISConvert.degToRad(315))
		context.line_to(center.x, center.y)
		if alertMode:
			#speedcam is close, vision is red
			context.set_source_rgba(1.0, 0.0, 0.0, 0.40)
		else:
			#speedcam is too far away, vision is green
			context.set_source_rgba(0.0, 1.0, 0.0, 0.30)
		context.fill()
		context.arc(center.x, center.y, self.__maxPixelRadius, GISConvert.degToRad(225), GISConvert.degToRad(315))
		context.line_to(center.x, center.y)
		context.close_path()
		if alertMode:
			#speedcam is close, vision is red
			context.set_source_rgb(0.8, 0.0, 0.0)
		else:
			#speedcam is too far away, vision is green
			context.set_source_rgb(0.0, 0.8, 0.0)
		context.set_line_width(2)
		context.stroke()
		
		#draw car
		context.set_source_surface(self.__car.pixbuf, math.ceil(center.x-self.__car.width/2), math.ceil(center.y-self.__car.height/2))
		context.paint()
		
		return pixmap
		
	
	##Expose widget
	def expose(self, widget, event):
		context = widget.window.cairo_create()

		# set a clip region for the expose event
		context.rectangle(event.area.x, event.area.y, event.area.width, event.area.height)
		context.clip()
			
		#draw
		try:
			self.draw(context)
		except Exception, e:
			type,value,tb = sys.exc_info()
			Log.log(traceback.format_exception(type,value,tb))

		return False
		
		
	##Redraw canvas
	def redraw_canvas(self):
		if self.window:
			alloc = self.get_allocation()
			self.queue_draw_area(alloc.x, alloc.y, alloc.width, alloc.height)
			self.window.process_updates(True)
			
			
	##Widget is resized
	def resize(self, *args):
		if self.window!=None:
			self.__cacheRadarNormal = None
			self.__cacheRadarAlert = None
			
	
	##Draw widget content
	def draw(self, context):
		#get widget rect
		rect = self.get_allocation()
		
		#fix rect to have real widget size
		#rect.width = rect.width - rect.x
		#rect.height = rect.height - rect.y
		#rect.x = 0
		#rect.y = 0
		#print rect.x, rect.y, rect.width, rect.height
	
		#get radar boundaries
		boundaries = self.__getBoundaries()
		#print 'boundaries %s' % boundaries

		if boundaries!=None:
			#compute dimensions, scales and useful positions
			self.__computeDimensionsAndScale(rect)
			center = self.__utmPointToFoxyRadarPoint(self.__currentPosition.utm, boundaries)
			
			#cache all pixmap if necessary
			if self.__cacheRadarNormal==None or self.__cacheRadarAlert==None:
				self.__cacheRadarNormal = self.__createRadarPixmap(rect, boundaries, center, False, self.__testMode)
				self.__cacheRadarAlert = self.__createRadarPixmap(rect, boundaries, center, True, self.__testMode)
			
			#draw background
			if not self.__alert:
				self.window.draw_drawable(self.get_style().fg_gc[gtk.STATE_NORMAL], self.__cacheRadarNormal, 0, 0, 0, 0, rect.width, rect.height)
			else:
				self.window.draw_drawable(self.get_style().fg_gc[gtk.STATE_NORMAL], self.__cacheRadarAlert, 0, 0, 0, 0, rect.width, rect.height)
				
			#draw north direction
			if self.__displayNorthDirection:
				north = FoxyRadarPoint()
				north.x = center.x
				north.y = self.BORDER/2
				north = self.__rotateFoxyRadarPoint(center, north, 360-self.__currentPosition.angle)
				context.arc(north.x, north.y, 4, 0, 2 * math.pi)
				context.set_source_rgb(0,0,0)
				context.fill()
			
			#first draw ghosts
			self.__drawSpeedcams(FoxyTag.FOXYTAG_GHOST, center, boundaries, context)
			#then draw mobile speedcams
			self.__drawSpeedcams(FoxyTag.FOXYTAG_MOBILE, center, boundaries, context)
			#and finally fixed speedcams
			self.__drawSpeedcams(FoxyTag.FOXYTAG_FIXED, center, boundaries, context)
	
			#draw buttons
			if self.__buttonsDisplay:
				self.__drawButtons(context, rect)
				
		else:
			#draw waiting gps data
			context.set_source_surface(self.__waiting_gps.pixbuf, math.ceil(rect.width/2-self.__waiting_gps.width/2), math.ceil(rect.height/2-self.__waiting_gps.height/2))
			context.paint()
			
			#disable buttons
			self.__buttonsDisplay = False
	
	
	##Draw speedcams according to specified kind
	##@param kind : speedcam kind to draw (see FoxyTag class to known different kinds)
	def __drawSpeedcams(self, kind, center, boundaries, context):
		#draw speedcams
		if self.__speedcams!=None and len(self.__speedcams)>0:
		
			#load image according to speedcam kind
			if kind==FoxyTag.FOXYTAG_FIXED:
				#fixed speedcam
				image = self.__speedcam_fixed
			elif kind==FoxyTag.FOXYTAG_MOBILE:
				#mobile speedcam
				image = self.__speedcam_mobile
			else:
				#ghost or unknown speedcam
				image = self.__speedcam_ghost
		
			#display speedcams in opposite car direction
			for speedcam in self.__speedcams:
				if speedcam.foxytag.kind==kind and not speedcam.carDirection:
					#get pixel coordinates
					speedcamPoint = self.__utmPointToFoxyRadarPoint(speedcam.utm, boundaries)
					
					if speedcamPoint!=None:
						#in bounds, display it
						#rotate
						speedcamPoint = self.__rotateFoxyRadarPoint(center, speedcamPoint, 360-self.__currentPosition.angle)
						
						#draw speedcam
						context.set_source_surface(self.__speedcam_opposite.pixbuf, math.ceil(speedcamPoint.x-self.__speedcam_opposite.width/2), math.ceil(speedcamPoint.y-self.__speedcam_opposite.height/2))
						context.paint()
			
			#display speedcams in car direction
			for speedcam in self.__speedcams:
				if speedcam.foxytag.kind==kind and speedcam.carDirection:
					#get pixel coordinates
					speedcamPoint = self.__utmPointToFoxyRadarPoint(speedcam.utm, boundaries)
					
					if speedcamPoint!=None:
						#in bounds, display it
						#rotate
						speedcamPoint = self.__rotateFoxyRadarPoint(center, speedcamPoint, 360-self.__currentPosition.angle)
							
						#draw speedcam
						context.set_source_surface(image.pixbuf, math.ceil(speedcamPoint.x-image.width/2), math.ceil(speedcamPoint.y-image.height/2))
						context.paint()

	##Draw radar buttons
	##@param context : context to draw in
	##@rect : rect
	def __drawButtons(self, context, rect):
		#draw main rectangle
		context.rectangle(0, 0, rect.width, rect.height)
		context.set_source_rgba(1, 1, 1, 0.75)
		context.fill()
		
		#compute placement coordinates
		coords = FoxyRadarPoint()
		coords.x = rect.width / 4
		coords.y = rect.height / 4
		
		if self.__modeChooseSpeedcam:
			#display buttons to choose speedcam actions
			
			#add fixed speedcam button
			context.set_source_surface(self.__button_add_fixed.pixbuf, math.ceil(coords.x-self.__button_add_fixed.width/2), math.ceil(coords.y-self.__button_add_fixed.height/2))
			context.paint()
			
			#add mobile speedcam button
			context.set_source_surface(self.__button_add_mobile.pixbuf, math.ceil(coords.x*3-self.__button_add_mobile.width/2), math.ceil(coords.y-self.__button_add_mobile.height/2))
			context.paint()
			
			#add delete speedcam button
			if self.__alert and self.__distanceFromSpeedcam<=self.SPEEDCAM_CANCEL_DISTANCE:
				context.set_source_surface(self.__button_delete.pixbuf, math.ceil(coords.x-self.__button_delete.width/2), math.ceil(coords.y*3-self.__button_delete.height/2))
				context.paint()
				
			#add cancel button
			context.set_source_surface(self.__button_cancel.pixbuf, math.ceil(coords.x*3-self.__button_cancel.width/2), math.ceil(coords.y*3-self.__button_cancel.height/2))
			context.paint()
			
		else:
			#display buttons to choose direction
			
			#add car direction button
			context.set_source_surface(self.__button_direction_car.pixbuf, math.ceil(coords.x-self.__button_direction_car.width/2), math.ceil(coords.y-self.__button_direction_car.height/2))
			context.paint()
			
			#add opposite direction button
			context.set_source_surface(self.__button_direction_opposite.pixbuf, math.ceil(coords.x*3-self.__button_direction_opposite.width/2), math.ceil(coords.y-self.__button_direction_opposite.height/2))
			context.paint()
			
			#add both directions button
			context.set_source_surface(self.__button_direction_both.pixbuf, math.ceil(coords.x-self.__button_direction_both.width/2), math.ceil(coords.y*3-self.__button_direction_both.height/2))
			context.paint()
			
			#add cancel button
			context.set_source_surface(self.__button_cancel.pixbuf, math.ceil(coords.x*3-self.__button_cancel.width/2), math.ceil(coords.y*3-self.__button_cancel.height/2))
			context.paint()
			
			
			
				
				
	#############################################################################################
	## WIDGET INTERNAL COMPUTATIONS
	#############################################################################################
	
	##Return radar boundaries
	def __getBoundaries(self):
		#init
		result = None
	
		if self.__currentPosition!=None:
			#get radius in meters
			maxRadius = self.__circlesCount * 1000
			
			#we can compute boundaries
			result = UTMRectangle()
			result.topLeft.x = self.__currentPosition.utm.x - maxRadius
			result.topLeft.y = self.__currentPosition.utm.y + maxRadius
			
			result.topRight.x = self.__currentPosition.utm.x + maxRadius
			result.topRight.y = self.__currentPosition.utm.y + maxRadius
			
			result.bottomLeft.x = self.__currentPosition.utm.x - maxRadius
			result.bottomLeft.y = self.__currentPosition.utm.y - maxRadius
			
			result.bottomRight.x = self.__currentPosition.utm.x + maxRadius
			result.bottomRight.y = self.__currentPosition.utm.y - maxRadius

		return result
		
		
	##Convert UTM point to foxy radar point
	##@param utm : UTM point to convert
	##@param boundaries : radar boundaries to use to compute foxy radar point coordinates
	def __utmPointToFoxyRadarPoint(self, utm, boundaries):
		#init
		foxyRadarPoint = None
	
		#check if utm point in boundaries
		if boundaries!=None and boundaries.isInRectangle(utm):
			#point in boundaries
			
			#compute pixel coordinates
			foxyRadarPoint = FoxyRadarPoint()
			foxyRadarPoint.x = ((utm.x - boundaries.topLeft.x) * self.__scale) + (self.BORDER / 2)
			foxyRadarPoint.y = ((boundaries.topLeft.y - utm.y) * self.__scale) + (self.BORDER / 2)
			
			#add offsets
			foxyRadarPoint.x += self.__offsets.x
			foxyRadarPoint.y += self.__offsets.y
		#else:
		#	Log.log('ND : %s in %s' % (str(utm), str(boundaries)))

		return foxyRadarPoint
		
	
	##Rotate speedcam point according to current angle
	##@param carPoint : position on the widget of the car
	##@param speedcamPoint : position on the widget of the speedcam
	##@return (x,y) : widget coordinates of the speedcam after rotating
	def __rotateFoxyRadarPoint(self, carPoint, speedcamPoint, angle):
		#translate speedcam axes
		x1 = speedcamPoint.x - carPoint.x
		y1 = carPoint.y - speedcamPoint.y
		
		#rotate
		#x' = x*cos(angle) + y*sin(angle)
		x2 = (x1 * math.cos(GISConvert.degToRad(angle))) + (y1 * math.sin(GISConvert.degToRad(angle)))
		#y' = -x*sin(angle) + y*cos(angle)
		y2 = -(x1 * math.sin(GISConvert.degToRad(angle))) + (y1 * math.cos(GISConvert.degToRad(angle)))
		
		#translate again
		rotatedPoint = FoxyRadarPoint()
		rotatedPoint.x = x2 + carPoint.x
		rotatedPoint.y = carPoint.y - y2

		return rotatedPoint
		
	
	##Compute widget dimensions and scales
	def __computeDimensionsAndScale(self, rect):
		#compute max radius (to have real circle)
		width = (rect.width - self.BORDER) / 2
		height = (rect.height - self.BORDER) / 2
		if width>height:
			self.__maxPixelRadius = height
		else:
			self.__maxPixelRadius = width
		
		#compute left (x) and top (y) offset
		self.__offsets = FoxyRadarPoint()
		if rect.width>rect.height:
			self.__offsets.x = (rect.width - rect.height) / 2
			self.__offsets.y = 0
		else:
			self.__offsets.x = 0
			self.__offsets.y = (rect.height - rect.width) / 2
		
		#compute min radius according to number of circles to draw
		self.__minPixelRadius = self.__maxPixelRadius / self.__circlesCount
			
		#compute scale to tranform meters to pixel
		#each circle measures 1 km
		self.__scale = float(self.__maxPixelRadius * 2) / float(self.__circlesCount * 1000 * 2)
		
	
				
	
	#############################################################################################
	## WIDGET CONGIGURATION
	#############################################################################################
	
	##Set radius
	##@param radius : new radar radius
	def setRadius(self, radius):
		if radius>=2 and radius<=6:
			self.__circlesCount = radius
			
	
	##Set if in test mode
	##@param testMode : True if in test mode
	def setTestMode(self, testMode):
		self.__cacheRadarNormal = None
		self.__cacheRadarAlert = None
		self.__testMode = testMode
		
		
	##Set if widget must display north direction
	def setDisplayNorthDirection(self, displayNorthDirection):
		self.__displayNorthDirection = displayNorthDirection
				
			
	##Set current position of the car
	##@param currentPosition : GpsInfos instance containing current position
	##@param distanceFromSpeedcam : distance between car and closest speedcam
	def setPosition(self, currentPosition, distanceFromSpeedcam):
		#save current position
		self.__currentPosition = currentPosition
		self.__distanceFromSpeedcam = distanceFromSpeedcam
		
		#redraw canvas
		self.redraw_canvas()
		
	
	##Set alert if necessary
	##@param alert : True if it's necessary to display alert
	def setAlert(self, alert):
		#save alert
		self.__alert = alert
	
		#redraw canvas
		self.redraw_canvas()
		
	
	##Set speedcams collection (shared with speedcam alerter)
	def setSpeedcams(self, speedcams):
		self.__speedcams = speedcams

	
	