from gtkeasyapp.log import *
from gtkeasyapp.preffile import *
from gtkeasyapp.connectionprocess import *
from gtkeasyapp.gpsprocess import *
from devicemanager.screenmanager import *
from const import *
from structs import *
from speedcamalerter import *
from gtkeasyapp.prefdialog import *
from gtkeasyapp.easywindow import *
from foxyradar import *
from foxyprocesses import *


##version
VERSION = "1.2.0"

##Constants
path = os.path.dirname(__file__)
RESOURCES_DIR = os.path.join(path, 'res')

DOT_XPM = [
"10 10 2 1",
"  c black",
". c None",
"..........",
"...     ..",
"..       .",
".        .",
".         ",
".         ",
".        .",
".        .",
"..      ..",
"....  ...."
]

##SpeedcamAlerter menu
class SpeedcamAlerterMenu(gtk.Menu):
	##Constructor
	def __init__(self, pref):
		#super constructor
		gtk.Menu.__init__(self)

		#add preferences item
		self.item_preferences = gtk.MenuItem('Preferences')
		self.append(self.item_preferences)
		
		#add separator
		self.append( gtk.SeparatorMenuItem() )
		
		#add help item
		self.item_help = gtk.MenuItem('Help')
		self.append(self.item_help)
		
		#add about item
		self.item_about = gtk.MenuItem('About')
		self.append(self.item_about)
		
		#add separator
		self.append( gtk.SeparatorMenuItem() )
		
		#add exit item
		self.item_exit = gtk.MenuItem('Exit')
		self.append(self.item_exit)
		
		
##SpeedcamAlerter Preferences dialog
class SpeedcamAlerterPreferences(PrefDialog):
	##Constructor
	def __init__(self, parent, prefFile):
		#super constructor
		PrefDialog.__init__(self, parent, 'Preferences')
		self.set_position(gtk.WIN_POS_CENTER)
		
		#load items from pref file
		prefItemRadius = prefFile.getItem('general', 'radius')
		prefItemEco = prefFile.getItem('general', 'eco')
		prefItemDebug = prefFile.getItem('general', 'debug')
		prefItemGps = prefFile.getItem('general', 'gps')
		prefItemNorth = prefFile.getItem('general', 'north')
		
		prefItemFront = prefFile.getItem('alarms', 'front')
		prefItemFixed = prefFile.getItem('alarms', 'fixed')
		prefItemMobile = prefFile.getItem('alarms', 'mobile')
		prefItemGhost = prefFile.getItem('alarms', 'ghost')
		prefItemSound = prefFile.getItem('alarms', 'sound')
				
		prefItemConnection = prefFile.getItem('connections', 'connection')
		prefItemFoxyLogin = prefFile.getItem('connections', 'login')
		prefItemFoxyPassword = prefFile.getItem('connections', 'password')
		prefItemDownload = prefFile.getItem('connections', 'download')
		prefItemUpload = prefFile.getItem('connections', 'upload')
		
		prefItemDummyDownload = PrefFileItemDummy()
		prefItemDummyUpload = PrefFileItemDummy()
		
		#add sections
		self.addSection('general', 'General', os.path.join(RESOURCES_DIR, 'section_general.png'))
		self.addSection('alarms', 'Alarms', os.path.join(RESOURCES_DIR, 'section_alarms.png'))
		self.addSection('connections', 'Connections', os.path.join(RESOURCES_DIR, 'section_connections.png'))
		
		#create pref dialog items
		#section general
		itemRadius = PrefDialogItemPlusMinus(prefItemRadius, 'Radar radius (km)', 3, 6)
		itemRadius.setHelp('Area covered by the radar')
		self.addItem('general', itemRadius)
		itemEco = PrefDialogItemTrueFalse(prefItemEco, 'Eco mode')
		itemEco.setHelp('Turns on screen only when necessary to save power')
		self.addItem('general', itemEco)
		itemNorth = PrefDialogItemTrueFalse(prefItemNorth, 'Display north')
		itemNorth.setHelp('Display north direction on radar')
		self.addItem('general', itemNorth)
		
		#section alarms
		itemFront = PrefDialogItemTrueFalse(prefItemFront, 'Bring to front')
		itemFront.setHelp('Bring to front SpeedcamAlerter application when necessary')
		self.addItem('alarms', itemFront)
		itemSound = PrefDialogItemTrueFalse(prefItemSound, 'Play sound')
		itemSound.setHelp('Play sound when speedcam approaching?')
		self.addItem('alarms', itemSound)
		itemFixed = PrefDialogItemInput(prefItemFixed, 'Fixed alert at (m)')
		itemFixed.setHelp('Distance at which the alarm is triggered when fixed speedcam is close to the car')
		self.addItem('alarms', itemFixed)
		itemMobile = PrefDialogItemInput(prefItemMobile, 'Mobile alert at (m)')
		itemMobile.setHelp('Distance at which the alarm is triggered when mobile speedcam is close to the car')
		self.addItem('alarms', itemMobile)
		itemGhost = PrefDialogItemInput(prefItemGhost, 'Ghost alert at (m)')
		itemGhost.setHelp('Distance at which the alarm is triggered when ghost speedcam is close to the car')
		self.addItem('alarms', itemGhost)
		
		#section connections
		itemConnection = PrefDialogItemList(prefItemConnection, 'Data connection')
		itemConnection.setHelp('Data connection to use to connect to internet and retrieve foxy tags')
		self.addItem('connections', itemConnection)
		itemSeparator1 = PrefDialogItemSeparator('<b>FoxyTag params</b>')
		self.addItem('connections', itemSeparator1)
		itemLogin = PrefDialogItemInput(prefItemFoxyLogin, 'Login')
		self.addItem('connections', itemLogin)
		itemPassword = PrefDialogItemPassword(prefItemFoxyPassword, 'Password')
		self.addItem('connections', itemPassword)
		itemFoxyTagInscription = PrefDialogItemLink('FoxyTag webpage', 'http://www.foxytag.com/fr/download.html', 'http://www.foxytag.com')
		itemFoxyTagInscription.setHelp('All information about FoxyTag registration is specified at bottom of webpage')
		self.addItem('connections', itemFoxyTagInscription)
		itemSeparator1 = PrefDialogItemSeparator('<b>Data counters</b>')
		self.addItem('connections', itemSeparator1)
		download = float(prefItemDownload.getValue() * 0.001)
		prefItemDummyDownload.setValue('%.2f' % download)
		itemDownload = PrefDialogItemInput(prefItemDummyDownload, 'Download (Kb)')
		itemDownload.setHelp('Amount of data downloaded from foxytag server')
		itemDownload.setReadOnly(True)
		self.addItem('connections', itemDownload)
		upload = float(prefItemUpload.getValue() * 0.001)
		prefItemDummyUpload.setValue('%.2f' % upload)
		itemUpload = PrefDialogItemInput(prefItemDummyUpload, 'Upload (Kb)')
		itemUpload.setHelp('Amount of data uploaded to foxytag server')
		itemUpload.setReadOnly(True)
		self.addItem('connections', itemUpload)
		
		#add data connections
		connections = ConnectionManager.getConnections(ConnectionItem.CNX_TYPE_DATA)
		options = ['']
		for connection in connections:
			options.append(connection.name)
		itemConnection.setOptions(options)
		
		
##SpeedcamAlerter about dialog
class SpeedcamAlerterAboutDialog(gtk.Dialog):
	##Constructor
	def __init__(self, parent):
		#super constructor
		gtk.Dialog.__init__(self, 'About', parent, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT, (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
		self.set_default_size(400, 250)
		self.set_position(gtk.WIN_POS_CENTER)
		self.set_has_separator(False)
		
		#add foxy tag image
		image = gtk.Image()
		image.set_from_file(os.path.join(RESOURCES_DIR, 'foxytag.png'))
		self.vbox.pack_start(image)
		
		#add speedcamalerter infos (text is extracted from eyekiwi website)
		label = gtk.Label('SpeedcamAlerter uses FoxyTag, the world largest speed camera database. The collaborative structure of FoxyTag offers you continuously the most accurate and always up to date speed camera location data.')
		label.set_line_wrap(True)
		self.vbox.pack_start(label)
		
		self.show_all()
		

##HelpDialogLabel build pre-defined label widget
class HelpDialogLabel(gtk.HBox):
	##Constructor
	def __init__(self, window, label, color=None, addDot=True):
		#super constructor
		gtk.HBox.__init__(self)
		
		#add dot
		if addDot:
			xpm,mask = gtk.gdk.pixmap_create_from_xpm_d(window.window, None, DOT_XPM)
			dot = gtk.Image()
			dot.set_from_pixmap(xpm, mask)
			#dot.set_padding(5, 5)
			dot.set_size_request(10, 10)
			self.pack_start(dot, False, False, 5)
		
		#set attributes
		lbl = gtk.Label()
		lbl.set_line_wrap(True)
		lbl.set_selectable(False)
		#lbl.set_padding(2, 2)
		lbl.set_alignment(0.0, 0.5)
		if color!=None:
			lbl.set_markup('<span foreground="%s">%s</span>' % (color,label) )
		else:
			lbl.set_markup('%s' % (label) )
		self.pack_start(lbl, True, True, 5)
			
			
			
##HelpDialogImageLabel build image-label widget
class HelpDialogImageLabel(gtk.HBox):
	##Constructor
	def __init__(self, window, label, image, addDot=True):
		#super constructor
		gtk.HBox.__init__(self)
		
		#add dot
		if addDot:
			xpm,mask = gtk.gdk.pixmap_create_from_xpm_d(window.window, None, DOT_XPM)
			dot = gtk.Image()
			dot.set_from_pixmap(xpm, mask)
			#dot.set_padding(5, 5)
			dot.set_size_request(10, 10)
			self.pack_start(dot, False, False, 5)
		
		#add image
		pixbuf = gtk.gdk.pixbuf_new_from_file(image)
		image = gtk.Image()
		image.set_from_pixbuf(pixbuf)
		#image.set_padding(5, 5)
		image.set_size_request(pixbuf.get_width(), pixbuf.get_height())
		self.pack_start(image, False, False, 5)

		#add label
		label = gtk.Label(label)
		label.set_line_wrap(True)
		#label.set_padding(2, 2)
		label.set_selectable(False)
		label.set_alignment(0.0, 0.5)
		self.pack_start(label, True, True)
		
		
		
		
##SpeedcamAlerter help dialog
class SpeedcamAlerterHelpDialog(gtk.Dialog):
	##Constructor
	def __init__(self, parent):
		#Super constructor
		gtk.Dialog.__init__(self, 'Help', parent, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT, (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
		self.set_default_size(650, 325)
		self.set_has_separator(False)
		self.set_position(gtk.WIN_POS_CENTER)
		scrolled = gtk.ScrolledWindow()
		scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
		self.vbox.pack_start(scrolled)
		vbox = gtk.VBox(spacing=5)
		scrolled.add_with_viewport(vbox)
		
		#add instructions label
		instructions = HelpDialogLabel(parent, '<b><u>FoxyTag instructions:</u></b>', None, False)
		vbox.pack_start(instructions)

		#add instructions
		instruction1 = HelpDialogLabel(parent, 'User must tag or confirm when he is as close as possible to the speed camera and not already as soon as he sees the camera. Otherwise there will be a second tag for the same camera and other users will decrease his trust links.', 'red')
		vbox.pack_start(instruction1)
		instruction2 = HelpDialogLabel(parent, 'A camera inside a tunnel or close to a tunnel exit (typically  less than 10 seconds after the exit) must be tagged at the tunnel entry.', 'red')
		vbox.pack_start(instruction2)
		instruction3 = HelpDialogLabel(parent, 'When the tagging needs several user-interactions (like first choosing the kind of camera and then choosing its direction), it is when he presses the first button that the current position is recorded. He can therefore take as much time as he needs to finish the tagging process.', 'red')
		vbox.pack_start(instruction3)
		instruction4 = HelpDialogLabel(parent, 'It is useless to confirm several times the same speed camera. In some particular situations this can even be bad for his trust links.', 'orange')
		vbox.pack_start(instruction4)
		instruction5 = HelpDialogLabel(parent, 'In case of a doubt it is better not to tag instead of tagging wrong. Many users think that if they forget to tag or to confirm a speed camera their trust links will decrease, which is wrong.', 'orange')
		vbox.pack_start(instruction5)
		instruction6 = HelpDialogLabel(parent, 'If he misuses the application (like confirming a speed camera when he is not close enough) his trust links will decrease and he risks therefore to be excluded from the system.')
		vbox.pack_start(instruction6)
		instruction7 = HelpDialogLabel(parent, 'If there can be a doubt whether a camera can flash in both directions or not, it is better to tag for both directions.')
		vbox.pack_start(instruction7)
		instruction8 = HelpDialogLabel(parent, 'A traffic light camera should be tagged and treated like a speed camera.')
		vbox.pack_start(instruction8)
		
		#add separator
		separator = gtk.Label('')
		vbox.pack_start(separator)

		#add help label
		helps = HelpDialogLabel(parent, '<b><u>SpeedcamAlerter help:</u></b>', None, False)
		vbox.pack_start(helps)
		
		#add helps
		help1 = HelpDialogLabel(parent, 'Tap on the radar to add, confirm or cancel speedcam. Then follow screen instructions.')
		vbox.pack_start(help1)
		help2 = HelpDialogLabel(parent, 'When radar approaching, screen turns on automatically.')
		vbox.pack_start(help2)
		help3 = HelpDialogImageLabel(parent, 'Red logo represents fixed speedcam in car direction.', os.path.join(RESOURCES_DIR, 'speedcam_fixed_40x40.png'))
		vbox.pack_start(help3)
		help4 = HelpDialogImageLabel(parent, 'Blue logo represents mobile speedcam in car direction.', os.path.join(RESOURCES_DIR, 'speedcam_mobile_40x40.png'))
		vbox.pack_start(help4)
		help5 = HelpDialogImageLabel(parent, 'Grey logo represents ghost speedcam in car direction.', os.path.join(RESOURCES_DIR, 'speedcam_ghost_40x40.png'))
		vbox.pack_start(help5)
		help6 = HelpDialogImageLabel(parent, 'Green logo represents speedcam in opposite car direction.', os.path.join(RESOURCES_DIR, 'speedcam_opposite_40x40.png'))
		vbox.pack_start(help6)
		
		self.show_all()
		
		
		
		
##SpeedcamAlerter User Interface
class SpeedcamAlerterUI(EasyWindow):
	##Constants
	STATUS_GPS_MSG = 'GPS : '
	STATUS_INTERNET_MSG = 'Network : '
	STATUS_FOXY_MSG = 'FoxyTag : '
	

	##Constructor
	def __init__(self, pref, log):
		#get preferences filepath
		self.__clearMessageTimer = None
		self.__clearMessageCounter = 0
		self.__display = ScreenManager()
		self.__gpsOk = 'KO'
		self.__internetOk = 'KO'
		self.__foxyOk = 'KO'
		self.__statusProcessesMessage = 'GPS=%s  Netwk=%s  Foxy=%s'
	
		#super constructor
		EasyWindow.__init__(self, 'SpeecamAlerter', pref, log, False)
		
	
	##init log
	##@param prefFile : preferences file
	##@param logFile : log file
	def _getLogMode(self, prefFile, logFile):
		#get debug item
		debug = prefFile.getItem('general', 'debug')
		
		if debug==None:
			#no log
			return Log.MODE_NONE
			
		elif debug.getValue().lower()=='stdout':
			#stdout
			return Log.MODE_STDOUT

		elif debug.getValue().lower()=='file' and logFile:
			#file
			return Log.MODE_FILE
			
		else:
			#no log
			return Log.MODE_NONE
	
	
	##Get old version (version during last launch, usualy read from preferences file)
	##@return None if no version or error occured
	def _getOldVersion(self, prefFile=None):
		if prefFile!=None:
			version = prefFile.getItem('general', 'version')
			if version!=None:
				return version.getValue()
		return None
		
		
	##Get new version (usualy read from application global variable)
	##@return None if no version or error occured
	def _getNewVersion(self, prefFile=None):
		return VERSION
		
		
	##Update version number in preferences file
	##@param prefFile : PrefFile instance
	##@param version : version to save into file
	def _updateVersion(self, prefFile, version):
		prefFile.getItem('general', 'version').setValue(version)
		prefFile.save()
		
	
	##Create new preferences file
	def _createDefaultPreferencesFile(self, filePath):
		#prepare xml content
		xml =  '<config>\n'
		xml += '  <section id="general">\n'
		xml += '    <key id="radius" type="int">4</key>\n'
		xml += '    <key id="eco" type="bool">1</key>\n'
		xml += '    <key id="debug" type="string"></key>\n'
		xml += '    <key id="version" type="string"></key>\n'
		xml += '    <key id="north" type="bool">0</key>\n'
		xml += '  </section>\n'
		xml += '  <section id="alarms">\n'
		xml += '    <key id="sound" type="bool">1</key>\n'
		xml += '    <key id="front" type="bool">1</key>\n'
		xml += '    <key id="fixed" type="int">1000</key>\n'
		xml += '    <key id="mobile" type="int">1000</key>\n'
		xml += '    <key id="ghost" type="int">800</key>\n'
		xml += '  </section>\n'
		xml += '  <section id="connections">\n'
		xml += '    <key id="connection" type="string"></key>\n'
		xml += '    <key id="download" type="int">0</key>\n'
		xml += '    <key id="upload" type="int">0</key>\n'
		xml += '    <key id="login" type="string"></key>\n'
		xml += '    <key id="password" type="string"></key>\n'
		xml += '  </section>\n'
		xml += '</config>'
		
		#write xml
		conf = open(filePath, 'wb')
		conf.write(xml)
		conf.close()
		
	
	##Check preferences file after new version detected (useful to add or remove item)
	##@param prefFile : PrefFile instance
	##@param filePath : preferences file path
	##@return True if pref file needs to be reloaded
	def _checkPreferencesFile(self, prefFile, filePath):
		#prepare xml content
		xml =  '<config>\n'
		xml += '  <section id="general">\n'
		xml += '    <key id="radius" type="int">' + self.__getPrefItemValue(prefFile, 'display', 'radius', 4) + '</key>\n'
		xml += '    <key id="eco" type="bool">' + self.__getPrefItemValue(prefFile, 'display', 'eco', True) + '</key>\n'
		xml += '    <key id="version" type="string">' + self.__getPrefItemValue(prefFile, 'general', 'version', VERSION) + '</key>\n'
		xml += '    <key id="debug" type="string">' + self.__getPrefItemValue(prefFile, 'general', 'debug', '') + '</key>\n'
		xml += '    <key id="north" type="bool">' + self.__getPrefItemValue(prefFile, 'general', 'north', False) + '</key>\n'
		xml += '  </section>\n'
		xml += '  <section id="alarms">\n'
		xml += '    <key id="sound" type="bool">True</key>\n'
		xml += '    <key id="front" type="bool">' + self.__getPrefItemValue(prefFile, 'general', 'front', True) + '</key>\n'
		xml += '    <key id="fixed" type="int">' + self.__getPrefItemValue(prefFile, 'general', 'fixed', 800) + '</key>\n'
		xml += '    <key id="mobile" type="int">' + self.__getPrefItemValue(prefFile, 'general', 'mobile', 800) + '</key>\n'
		xml += '    <key id="ghost" type="int">' + self.__getPrefItemValue(prefFile, 'general', 'ghost', 500) + '</key>\n'
		xml += '  </section>\n'
		xml += '  <section id="connections">\n'
		xml += '    <key id="connection" type="string">' + self.__getPrefItemValue(prefFile, 'connections', 'connection', '') + '</key>\n'
		xml += '    <key id="download" type="int">' + self.__getPrefItemValue(prefFile, 'connections', 'download', 0) + '</key>\n'
		xml += '    <key id="upload" type="int">' + self.__getPrefItemValue(prefFile, 'connections', 'upload', 0) + '</key>\n'
		xml += '    <key id="login" type="string">' + self.__getPrefItemValue(prefFile, 'connections', 'login', '') + '</key>\n'
		xml += '    <key id="password" type="string">' + self.__getPrefItemValue(prefFile, 'connections', 'password', '') + '</key>\n'
		xml += '  </section>\n'
		xml += '</config>'
		
		#write xml
		conf = open(filePath, 'wb')
		conf.write(xml)
		conf.close()
		
		#reload file
		return True
		
		
	##Return pref item value and if it's not possible return specified one
	##@param prefFile : preferences file instance
	##@param section : pref item section
	##@param key : pref item key
	##@param defaultValue : default value in case item not found
	def __getPrefItemValue(self, prefFile, section, key, defaultValue):
		#load item
		item = prefFile.getItem(section, key)
		
		if item!=None:
			return str(item.getValue())
		else:
			return str(defaultValue)
	
	
	##Create application menu
	##@param window : main gtk window
	##@param prefFile : PrefFile object
	##@return menu : gtk.Menu object
	def _createMenu(self, window, prefFile):
		menu = SpeedcamAlerterMenu(self._prefFile)
		#debug = prefFile.getItem('general', 'debug')
		#if debug!=None and debug.getValue():
		#	menu.item_speedcamalerter_start.connect('activate', self.__startSpeedcamAlerter)
		#	menu.item_speedcamalerter_stop.connect('activate', self.__stopSpeedcamAlerter)
		menu.item_preferences.connect('activate', self.__preferences)
		menu.item_help.connect('activate', self.__help)
		menu.item_about.connect('activate', self.__about)
		menu.item_exit.connect('activate', self._quit)
		return menu
	
	
	##Add main container
	##@param window : main gtk window
	##@param prefFile : PrefFile object
	##@return main GTK container
	def _createMainContainer(self, window, prefFile):
		#create main grid
		container = gtk.Table(5, 2, False)
		return container
	
	
	##Create widgets
	##@param window : main gtk window
	##@param prefFile : PrefFile object
	##@return nothing, just create all body widgets
	def _createWidgets(self, window, prefFile):
		#add foxytag logo
		self.__image_logo = gtk.Image()
		self.__image_logo.set_from_file(os.path.join(RESOURCES_DIR, 'foxytag.png'))
		self._mainContainer.attach(self.__image_logo, 1, 2, 0, 1, gtk.FILL, gtk.FILL,5 ,5)
		
		#add speed widget
		self.__label_speed = gtk.Label('')
		self.__label_speed.set_alignment(0.5, 0.5)
		self.__label_speed.modify_font(pango.FontDescription("sans 40"))
		label_speed_event = gtk.EventBox()
		label_speed_event.add(self.__label_speed)
		#label_speed_event.modify_bg(gtk.STATE_NORMAL, label_speed_event.get_colormap().alloc_color("green"))
		self._mainContainer.attach(label_speed_event, 1, 2, 1, 2, gtk.FILL, gtk.FILL, 0, 0)
		
		#add distance hbox 
		hbox = gtk.HBox()
		self._mainContainer.attach(hbox, 1, 2, 2, 3, gtk.FILL, gtk.FILL, 0, 0)
		
		#add distance image
		self.__image_speedcam_fixed = gtk.gdk.pixbuf_new_from_file(os.path.join(RESOURCES_DIR, 'speedcam_fixed_50x50.png'))
		self.__image_speedcam_mobile = gtk.gdk.pixbuf_new_from_file(os.path.join(RESOURCES_DIR, 'speedcam_mobile_50x50.png'))
		self.__image_speedcam_ghost = gtk.gdk.pixbuf_new_from_file(os.path.join(RESOURCES_DIR, 'speedcam_ghost_50x50.png'))
		self.__image_speedcam_opposite = gtk.gdk.pixbuf_new_from_file(os.path.join(RESOURCES_DIR, 'speedcam_opposite_50x50.png'))
		self.__image_speedcam = gtk.Image()
		self.__image_speedcam.set_padding(5,5)
		hbox.pack_start(self.__image_speedcam, False, True, 8)
		
		#add distance widget
		self.__label_distance = gtk.Label('')
		self.__label_distance.set_alignment(0, 0.5)
		self.__label_distance.modify_font(pango.FontDescription("sans 40"))
		label_distance_event = gtk.EventBox();
		label_distance_event.add(self.__label_distance)
		hbox.pack_start(label_distance_event, False, False, 0)
				
		#add message widget
		self.__label_message = gtk.Label('')
		self.__label_message.set_alignment(0.5, 0.5)
		self.__label_message.set_justify(gtk.JUSTIFY_CENTER)
		self.__label_message.set_ellipsize(pango.ELLIPSIZE_END)
		self.__label_message.set_selectable(False)
		self.__label_message.set_single_line_mode(False) #enable multiline
		self.__label_message.modify_font(pango.FontDescription("sans 18"))
		self._mainContainer.attach(self.__label_message, 0, 2, 3, 4, gtk.EXPAND|gtk.FILL, gtk.EXPAND|gtk.FILL, 2, 2)
		#self.__label_message.set_line_wrap(True)
		#self.__label_message.set_width_chars(60)
		#label_message_event = gtk.EventBox()
		#label_message_event.add(self.__label_message)
		#self._mainContainer.attach(label_message_event, 0, 2, 3, 4, 0, 0, 10, 10)
		
		#add radar widget
		self.__radar = FoxyRadar()
		radius = prefFile.getItem('general', 'radius')
		if radius!=None:
			self.__radar.setRadius(radius.getValue())
		north = prefFile.getItem('general', 'north')
		if north!=None:
			self.__radar.setDisplayNorthDirection(north.getValue())
		self._mainContainer.attach(self.__radar, 0,1,0,3, gtk.EXPAND|gtk.FILL, gtk.EXPAND|gtk.FILL, 2, 2)
		self.__radar.connect('add-fixed-speedcam', self.__addFixedSpeedcam)
		self.__radar.connect('add-opposite-fixed-speedcam', self.__addOppositeFixedSpeedcam)
		self.__radar.connect('add-mobile-speedcam', self.__addMobileSpeedcam)
		self.__radar.connect('add-opposite-mobile-speedcam', self.__addOppositeMobileSpeedcam)
		self.__radar.connect('remove-speedcam', self.__removeSpeedcam)
		self.__radar.connect('action-started', self.__radarActionStarted)
		
		#add statusbar
		hbox = gtk.HBox()
		self._mainContainer.attach(hbox, 0,2,4,5, gtk.FILL, gtk.FILL, 0, 0)
		self.__statusbarInfos = gtk.Statusbar()
		self.__statusbarInfos.set_has_resize_grip(False)
		self.__statusbarProcesses = gtk.Statusbar()
		self.__statusbarProcesses.set_has_resize_grip(False)
		hbox.pack_start(self.__statusbarInfos)
		hbox.pack_start(self.__statusbarProcesses)
		self.__statusbarInfosId = self.__statusbarInfos.get_context_id('main')
		self.__statusbarProcessesId = self.__statusbarProcesses.get_context_id('main')
	
	
	##Create processes
	##@param prefFile : PrefFile object
	def _createProcesses(self, prefFile):
		#internet process
		prefConnectionName = prefFile.getItem('connections', 'connection')
		ConnectionProcess(PROCESS_CNX, self, prefConnectionName.getValue(), self.__onConnectedToInternet, self.__onDisconnectedFromInternet)
	
		#gps process
		GpsProcess(PROCESS_GPS, self, self.__onGpsFixed)
		
		#foxytag process
		FoxyTagsProcess(self)
		
		#SpeedcamAlerter process
		SpeedcamAlerterProcess(self)
		
		
	##Startup treatment (like open splash screen...)
	##@param window : main gtk window
	##@param prefFile : PrefFile object
	##@param newVersion : True if it's a new version
	##@return nothing
	def _startupTreatment(self, window, prefFile, newVersion):
		#init
		firstLaunch = False
		prefLogin = prefFile.getItem('connections', 'login')
		prefVersion = prefFile.getItem('general', 'version')
		self.__radius = prefFile.getItem('general', 'radius')
		self.__bringToFront = prefFile.getItem('alarms', 'front')
		password = prefFile.getItem('connections', 'password')
		alertSound = prefFile.getItem('alarms', 'sound')
		self.__alertSoundFile = ''
		
		#set default status message
		self.updateUi(UI_STATUS, STATUS_PROCESSES, self.__statusProcessesMessage % (self.__gpsOk, self.__internetOk, self.__foxyOk))
		
		#save sound file path
		if alertSound!=None and alertSound.getValue():
			self.__alertSoundFile = os.path.join(RESOURCES_DIR, 'alert.mp3')
		
		#check test mode
		if password!=None:
			if len(password.getValue())==0:
				self.__radar.setTestMode(True)
				self.__updateMessage(['MSG','Test mode activated.'])
			else:
				self.__radar.setTestMode(False)
		
		#check version
		if newVersion:
			#new version
			firstLaunch = True
			self.__firstLaunch()
			
		#check login
		if not firstLaunch and prefLogin!=None and len(prefLogin.getValue())==0:
			self.__noAccount()
			
			
	##Before quit callback
	def _beforeQuit(self):			
		#stop message cleaner timer
		if self.__clearMessageTimer!=None:
			self.__clearMessageTimer.cancel()
		
	
	
	
	#############################################################################################
	## MENU AND POPUPS
	#############################################################################################
		
	##first launch info
	def __firstLaunch(self, *args):
		dialog = gtk.MessageDialog(self._window, gtk.DIALOG_MODAL, gtk.MESSAGE_INFO, gtk.BUTTONS_OK, 'It\'s the first launch (fresh install or update).\nPlease take a look in preferences dialog.')
		dialog.run()
		dialog.destroy()
			
	
	##Message to display if no foxytag account specified
	def __noAccount(self, *args):
		dialog = gtk.MessageDialog(self._window, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT, gtk.MESSAGE_INFO, gtk.BUTTONS_OK, 'SpeedcamAlerter needs FoxyTag account to start (it\'s free),\nplease configure it in preferences dialog.')
		dialog.run()
		dialog.destroy()
		
		
	##Open about dialog
	def __about(self, *args):
		dialog = SpeedcamAlerterAboutDialog(self._window)
		dialog.run()
		dialog.destroy()
		
		
	##Open help dialog
	def __help(self, *args):
		dialog = SpeedcamAlerterHelpDialog(self._window)
		dialog.run()
		dialog.destroy()
	
	
	##Open preferences dialog
	def __preferences(self, *args):
		#init
		passwordChanged = False
		loginChanged = False
		connectionChanged = False
		password = ""
		login = ""
		connection = ""
		
		#save foxytag login before opening preferences
		loginItem = self._prefFile.getItem('connections', 'login')
		if loginItem!=None:
			login += str(loginItem.getValue()) #make a copy
	
		#save foxytag password before opening preferences
		passwordItem = self._prefFile.getItem('connections', 'password')
		if passwordItem!=None:
			password += str(passwordItem.getValue()) #make a copy
			
		#save connection before opening preferences
		connectionItem = self._prefFile.getItem('connections', 'connection')
		if connectionItem!=None:
			connection += str(connectionItem.getValue()) #make a copy
	
		#open preferences
		dialog = SpeedcamAlerterPreferences(self._window, self._prefFile)
		response = dialog.run()
		if response==gtk.RESPONSE_OK:
			#save changed config
			self._prefFile.save()
			
			#update radius
			radius = self._prefFile.getItem('general', 'radius')
			if radius!=None:
				self.__radar.setRadius(radius.getValue())
				
			#show north direction
			northItem = self._prefFile.getItem('general', 'north')
			if northItem!=None:
				if northItem.getValue():
					self.__radar.setDisplayNorthDirection(True)
				else:
					self.__radar.setDisplayNorthDirection(False)
				
			#test mode
			passwordItem = self._prefFile.getItem('connections', 'password')
			if passwordItem!=None:
				if len(passwordItem.getValue())==0:
					self.__radar.setTestMode(True)
					self.__updateMessage(['MSG','Test mode activated.'])
				else:
					self.__radar.setTestMode(False)

			#new password ?
			if passwordItem!=None and passwordItem.getValue()!=password:
				#user changed password, display help
				passwordChanged = True
				
			#new login ?
			loginItem = self._prefFile.getItem('connections', 'login')
			if loginItem!=None and loginItem.getValue()!=login:
				#login changed, need to restart speedcamalerter
				loginChanged = True
				
			#new connection ?
			connectionItem = self._prefFile.getItem('connections', 'connection')
			if connectionItem!=None and connectionItem.getValue()!=connection:
				#connection changed, need to restart speedcamalerter
				connectionChanged = True
		
		#destroy preferences dialog
		dialog.destroy()
		
		#display help if password modified
		if passwordChanged:
			self.__help()
			
		#restart speedcamalerter if login, password or connection changed
		if loginChanged or passwordChanged or connectionChanged:
			#restart processes
			self._restartProcesses()

	
	#############################################################################################
	## UI UPDATES
	#############################################################################################
	
	
	##Update UI
	##This method is used to centralize UI updates
	##This method can call "protected" _updateUi() to securely update UI
	def updateUi(self, *args):
		if args[0]==UI_SPEED:
			self._updateUi(self.__updateSpeed, args)
		elif args[0]==UI_CLOSEST_SPEEDCAM:
			self._updateUi(self.__updateClosestSpeedcam, args)
		elif args[0]==UI_CURRENT_POS:
			self._updateUi(self.__updateCurrentPosition, args)
		elif args[0]==UI_RADAR_ALERT:
			self._updateUi(self.__updateRadarAlert, args)
		elif args[0]==UI_STATUS:
			self._updateUi(self.__updateStatusMessage, args)
		elif args[0]==UI_MESSAGE:
			self._updateUi(self.__updateMessage, args)
		elif args[0]==UI_UPDATE_SPEEDCAMS:
			self._updateUi(self.__updateRadarSpeedcams, args)
	
	
	##Update speed
	def __updateSpeed(self, args):
		speed = '%.0fkm/h' % args[1]
		self.__label_speed.set_label(str(speed))
		
		
	##Update distance
	def __updateClosestSpeedcam(self, args):
		speedcam = args[1]
		
		if speedcam!=None and speedcam.distanceFromCar<=(self.__radius.getValue()*1000):
			#update distance
			distance = ''
			if speedcam.distanceFromCar>1000:
				distance = '%.2fkm' % (speedcam.distanceFromCar / 1000.0)
			else:
				distance = '%dm' % (speedcam.distanceFromCar)
			self.__label_distance.set_label(distance)
			
			#update speedcam logo
			if not speedcam.carDirection:
				self.__image_speedcam.set_from_pixbuf(self.__image_speedcam_opposite)
			elif speedcam.foxytag.kind==FoxyTag.FOXYTAG_FIXED:
				self.__image_speedcam.set_from_pixbuf(self.__image_speedcam_fixed)
			elif speedcam.foxytag.kind==FoxyTag.FOXYTAG_MOBILE:
				self.__image_speedcam.set_from_pixbuf(self.__image_speedcam_mobile)
			elif speedcam.foxytag.kind==FoxyTag.FOXYTAG_GHOST:
				self.__image_speedcam.set_from_pixbuf(self.__image_speedcam_ghost)
			
		else:
			#no speedcam specified, clear all
			#update distance
			self.__label_distance.set_label('')
			
			#update speedcam logo
			self.__image_speedcam.clear()
	
		
	##Update radar position
	def __updateCurrentPosition(self, args):
		gpsInfos = args[1]
		distanceFromSpeedcam = args[2]
		self.__radar.setPosition(gpsInfos, distanceFromSpeedcam)
			
	
	##Alert radar
	def __updateRadarAlert(self, args):
		alert = args[1]
	
		#update radar
		self.__radar.setAlert(alert)
		
		#UI actions
		if alert:
			#bring speedcamalerter to front
			if self.__bringToFront:
				self.bringToFront()
				self.__display.turnScreenOnDuring60Seconds()
				
			#play alert sound
			if len(self.__alertSoundFile)>0:
				self._playSound(self.__alertSoundFile)
				
			
	##Update message
	def __updateMessage(self, args):
		message = args[1]

		#set message
		self.__label_message.set_label(message)
		
		#launch timer to remove message after 2 minutes
		if self.__clearMessageTimer!=None:
			#destroy existing timer
			self.__clearMessageTimer.cancel()
			del self.__clearMessageTimer
		self.__clearMessageTimer = threading.Timer(120.0, self.__clearMessage)
		self.__clearMessageTimer.start()
		
		#bring UI to front and switch on screen
		self.__display.turnScreenOnDuring60Seconds()
		self.bringToFront()
		
	
	##Clear message
	def __clearMessage(self, args=[]):
		#clear message
		self.__label_message.set_label('')
			
		
	##Update radar speedcams
	def __updateRadarSpeedcams(self, args):
		self.__radar.setSpeedcams(args[1])
		
	
	##Update status message
	def __updateStatusMessage(self, args):
		status = args[1]
		message = args[2]
			
		#get correct statusbar
		statusbar = None
		statusId = None
		if status==STATUS_INFOS:
			statusbar = self.__statusbarInfos
			statusId = self.__statusbarInfosId
			message = '%s : %s' % (time.strftime('%H:%M'), message)
		elif status==STATUS_PROCESSES:
			statusbar = self.__statusbarProcesses
			statusId = self.__statusbarProcessesId

		#remove existing message
		statusbar.pop(statusId)
		
		#add new message
		statusbar.push(statusId, message)
		
		
		
	
	#############################################################################################
	## UI EVENTS
	#############################################################################################
	
	##Radar view is clicked
	def __radarActionStarted(self, widget, data=None):
		self.getProcess(PROCESS_SCA).savePositionToPost()
	
	
	##Remove fixed speedcam callback
	def __addFixedSpeedcam(self, widget, data=None):
		self.getProcess(PROCESS_SCA).postFixedSpeedcam()
	
	
	##Remove opposite fixed speedcam callback
	def __addOppositeFixedSpeedcam(self, widget, data=None):
		self.getProcess(PROCESS_SCA).postOppositeFixedSpeedcam()
	
	
	##Remove mobile speedcam callback
	def __addMobileSpeedcam(self, widget, data=None):
		self.getProcess(PROCESS_SCA).postMobileSpeedcam()
	
	
	##Add opposite mobile speedcam callback
	def __addOppositeMobileSpeedcam(self, widget, data=None):
		self.getProcess(PROCESS_SCA).postOppositeMobileSpeedcam()
	
	
	##Remove speedcam callback
	def __removeSpeedcam(self, widget, data=None):
		self.getProcess(PROCESS_SCA).removeSpeedcam()
	
	
	##On GPS fix
	def __onGpsFixed(self):
		Log.log('SpeedcamAlerterUi::__onGpsFixed : GPS device is fixed')
		
		#update status message
		self.__gpsOk = 'ok'
		self.updateUi(UI_STATUS, STATUS_PROCESSES, self.__statusProcessesMessage % (self.__gpsOk, self.__internetOk, self.__foxyOk))
		
		#say speedcamalerter gps is now fixed
		self.getProcess(PROCESS_SCA).gpsFixed()
		
		
	##On internet connection established
	def __onConnectedToInternet(self, *args):
		Log.log('SpeedcamAlerterUi::__onInternetConnectionEstablished : Connected to internet')
		
		#update status message 
		self.__internetOk = 'ok'
		self.updateUi(UI_STATUS, STATUS_PROCESSES, self.__statusProcessesMessage % (self.__gpsOk, self.__internetOk, self.__foxyOk))
		#clear message to remove previous 'network unreachable message'. It confuses user when network is up again.
		self.__clearMessage()
		
		#say to speedcamalerter internet is ok
		self.getProcess(PROCESS_SCA).connectionEstablished(True)
		
		#try to connect to foxytag
		self.getProcess(PROCESS_FOX).connectionEstablished(True)
		self.getProcess(PROCESS_FOX).addNewAction(FoxyTagsProcess.ACTION_CONNECT, self.__onFoxyTagConnected, True)
		
		
	##On disconnected from internet
	def __onDisconnectedFromInternet(self, *args):
		Log.log('SpeedcamAlerterUi::__onDisconnectedFromInternet : Disconnected from internet')
		
		#update status message 
		self.__internetOk = 'KO'
		self.__foxyOk = 'KO'
		self.updateUi(UI_STATUS, STATUS_PROCESSES, self.__statusProcessesMessage % (self.__gpsOk, self.__internetOk, self.__foxyOk))
		
		#say foxy process internet is ko
		self.getProcess(PROCESS_FOX).connectionEstablished(False)
		
		#say to speedcamalerter internet is ko (and foxytag connection disconnected)
		self.getProcess(PROCESS_SCA).connectionEstablished(False)
		self.getProcess(PROCESS_SCA).foxytagConnected(False)
		
		
	##On Foxytag connected
	def __onFoxyTagConnected(self, result):
		if isinstance(result, FoxyTagsProcessResult):
			if result.result:
				Log.log('SpeedcamAlerterUi::__onFoxyTagConnected : Connected to foxytag')
				
				#update status message 
				self.__foxyOk = 'ok'
				self.updateUi(UI_STATUS, STATUS_PROCESSES, self.__statusProcessesMessage % (self.__gpsOk, self.__internetOk, self.__foxyOk))
				self.updateUi(UI_STATUS, STATUS_INFOS, 'connected to FoxyTag server')
				
				#say to speedcamalerter foxytag is ok
				self.getProcess(PROCESS_SCA).foxytagConnected(True)
				
			else:
				#unable to connect to foxytag
				self.updateUi(UI_STATUS, STATUS_INFOS, 'unable to connect to FoxyTag server')
				Log.log('SpeedcamAlerterUi::__onFoxyTagConnected : No way to connect to foxytag, check internet connection and restart application')
				
		else:
			Log.log('SpeedcamAlerterUi::__onFoxyTagConnected : Not awaited result!')