from Queue import Queue
from const import *
from structs import *
from gtkeasyapp.log import *
from gtkeasyapp.easyprocess import *
from foxytag import *


class FoxyTagsProcessResult():
	def __init__(self, result, *results):
		self.result = result #bool that contain action result
		self.results = results #result values
		

		
class FoxyTagsProcessAction():
	def __init__(self, action, callback, params):
		self.action = action
		self.callback = callback
		self.params = params
		self.attemps = 0
	
	def __str__(self):
		result = ''
		if self.action==FoxyTagsProcess.ACTION_CONNECT:
			result += 'CONNECT'
		elif self.action==FoxyTagsProcess.ACTION_GET:
			result += 'GET'
		elif self.action==FoxyTagsProcess.ACTION_POST:
			result += 'POST'
		result += str(self.params)
		
		return result

##FoxyTagProcess
##Base EasyProcess using FoxyTag instance
class FoxyTagsProcess(EasyProcess):
	##Consts
	ACTION_CONNECT = 0
	ACTION_GET = 1
	ACTION_POST = 2
	
	RESULT_OK = 10
	RESULT_KO = 9
	RESULT_RETRY = 8
	RESULT_DISCONNECT = 7
	RESULT_FATAL = 6
	
	DEFAULT_RADIUS = 7250 #in meters
	
	RETRY_CONNECT = 30.0 #in seconds
	RETRY_POST = 15.0 #in seconds
	RETRY_GET = 15.0 #in seconds
	MAX_ATTEMPS = 3

	
	##Constructor
	##@param ui : EasyWindow instance
	##@onDownloaded : callback when foxy tags downloaded
	def __init__(self, ui):
		#super constructor
		EasyProcess.__init__(self, PROCESS_FOX, ui)
		
		#set members
		self._timeout = 1.0 #check action to execute every second
		self.__actions = Queue()
		self.__priorityActions = Queue()
		self.__foxy = Foxy()
		self.__internetOk = False
		self.__timers = []
		
		
	##Process : execute pending actions
	def _process(self):
		#purge ended timers
		timersToRemove = []
		for timer in self.__timers:
			if not timer.isAlive():
				timersToRemove.append(timer)
		for timerToRemove in timersToRemove:
			self.__timers.remove(timerToRemove)
		del timersToRemove[0:len(timersToRemove)]
		
		#execute action
		if self.__internetOk:	
			action = None
		
			#priority actions first
			if not self.__priorityActions.empty():
				action = self.__priorityActions.get()
			elif not self.__actions.empty():
				action = self.__actions.get()
			
			#try:
			if action!=None:
				Log.log('FoxyTagsProcess::_process : %s' % str(action))
			
				if action.action==self.ACTION_GET:
					self.__get(action)
			
				elif action.action==self.ACTION_POST:		
					self.__post(action)
			
				elif action.action==self.ACTION_CONNECT:
					self.__connect(action)
					
			#except Exception,e:
			#	print 'FoxyTagsProcess::_process : problem executing action [%s]' % (e)
				
				
	##Stopping thread treatment
	def _stopping(self):
		#cancel all timers
		for timer in self.__timers:
			timer.cancel()
		del self.__timers[0:len(self.__timers)]

	
	##Internet connection established or not
	def connectionEstablished(self, connected):
		Log.log('FoxyProcess::connectionEstablished : network ok ? %s' % (connected))
		#set internet connection flag
		self.__internetOk = connected
		
		
	##Update data counters
	def __updateCounters(self, upload, download):
		#update upload counter
		uploadItem = self._ui._prefFile.getItem('connections', 'upload')
		uploadItem.setValue( uploadItem.getValue() + upload )
		
		#update download counter
		downloadItem = self._ui._prefFile.getItem('connections', 'download')
		downloadItem.setValue( downloadItem.getValue() + download )
		
		#save
		self._ui._prefFile.save()
		
	
	##Check tag and display UI message if necessary
	##@param messages : FoxyResult messages
	##@return 1 if result was ok, 0 if ko but retry is permitted or -1 if it was ko
	def __checkFoxyResultMessages(self, messages):
		#init
		result = self.RESULT_OK
		tempResult = 1
	
		for message in messages:
			if message.type==FoxyResultMessage.TYPE_NEED_RESTART:
				#Request failed, disconnect then restart action : ko, retry
				
				#show message
				#self._ui.updateUi(UI_MESSAGE, 'Not valid FoxyTag response!\nRestarting connection...')
				tempResult = self.RESULT_DISCONNECT
			
			elif message.type==FoxyResultMessage.TYPE_UNKNOWN:
				#unknow result (internal error) : ko, no retry
				#self._ui.updateUi(UI_MESSAGE, 'FoxyTag request failed (internal error).\nRequest canceled.')
				tempResult = self.RESULT_KO
				
			elif message.type==FoxyResultMessage.TYPE_OK_INIT:
				#connection ok : ok, no retry
				tempResult = self.RESULT_OK
				
			elif message.type==FoxyResultMessage.TYPE_KO:
				#failed for an unknown reason : ko, retry
				#self._ui.updateUi(UI_MESSAGE, 'FoxyTag request failed for an unknown reason.\nRetrying...')
				tempResult = self.RESULT_RETRY
				
			elif message.type==FoxyResultMessage.TYPE_KO_CMD:
				#Unknown command : ko, retry
				#self._ui.updateUi(UI_MESSAGE, 'Unknow command sent to FoxyTag.\nRetrying...')
				tempResult = self.RESULT_RETRY
			
			elif message.type==FoxyResultMessage.TYPE_KO_VERSION:
				#Wrong client version : ko, no retry
				self._ui.updateUi(UI_MESSAGE, 'Wrong client version, unable to connect to FoxyTag server.')
				tempResult = self.RESULT_FATAL
			
			elif message.type==FoxyResultMessage.TYPE_KO_LOGIN:
				#Wrong username or password : ko, no retry
				self._ui.updateUi(UI_MESSAGE, 'Wrong login or password, check FoxyTag account.')
				tempResult = self.RESULT_FATAL
		
			elif message.type==FoxyResultMessage.TYPE_ERR:
				#foxytag error message : ok, no retry
				self._ui.updateUi(UI_MESSAGE, 'FoxyTag error : %s' % (message.message))
				tempResult = self.RESULT_OK
				
			elif message.type==FoxyResultMessage.TYPE_INF:
				#foxytag info message : ok, no retry
				self._ui.updateUi(UI_MESSAGE, 'FoxyTag info : %s' % (message.message))
				tempResult = self.RESULT_OK
				
			elif message.type==FoxyResultMessage.TYPE_PUB:
				#foxytag adverting : ok, no retry
				self._ui.updateUi(UI_MESSAGE, 'FoxyTag ad : %s' % (message.message))
				tempResult = self.RESULT_OK
			
			elif message.type==FoxyResultMessage.TYPE_OK:
				#request ok : ok, no retry
				tempResult = self.RESULT_OK
				
			#compute result value
			if tempResult<result:
				result = tempResult
				
		return result
		

	##Add new foxytag action
	##@param action : foxytag action id (constant)
	##@param callback : callback function
	##@param priority : priority action (will be executed before not priority action)
	##@param *params : useful params for specified action
	def addNewAction(self, action, callback, priority, *params):
		if priority:
			self.__priorityActions.put( FoxyTagsProcessAction(action, callback, params) )
		else:
			self.__actions.put( FoxyTagsProcessAction(action, callback, params) )
		
	
	##Add existing FoxyTagsProcessAction
	##@param action : FoxyTagsProcessAction instance
	##@param priority : priority action (will be executed before not priority action)
	def addAction(self, action, priority):
		if isinstance(action, FoxyTagsProcessAction):
			Log.log('FoxyTagsProcess::addAction : Re-add existing action')
			if priority:
				self.__priorityActions.put( action )
			else:
				self.__actions.put( action )
		else:
			Log.log('FoxyTagsProcess::addAction : action "%s" is not FoxyTagsProcessAction' % (type(action)))
	
	
	##Add delayed action (always priority)
	##@param delay : timeout to add action
	##@param action : existing FoxyTagsProcessAction
	def __addDelayedAction(self, delay, action):
		t = threading.Timer(delay, self.addAction, [action, True])
		t.start()
		self.__timers.append(t)
		
	
	##Connect to foxytag
	##@param action : action to execute
	def __connect(self, action):
		#get foxytag connection infos
		login = self._ui._prefFile.getItem('connections', 'login')
		password = self._ui._prefFile.getItem('connections', 'password')
	
		#try to connect to foxy
		foxy = self.__foxy.connect(login.getValue(), password.getValue())
		Log.log('FoxyProcess::__connect : %s' % foxy)
			
		#check result
		result = self.__checkFoxyResultMessages(foxy.messages)
		Log.log('FoxyProcesses::__connect: result=%s' % (result))
		if result==self.RESULT_OK:
			Log.log('FoxyProcess::__connect : RESULT_OK')
			#set connection is successful
			action.callback( FoxyTagsProcessResult(True) )

		elif result==self.RESULT_DISCONNECT:
			Log.log('FoxyProcess::__connect : RESULT_DISCONNECT')
			#disconnect
			self._ui.getProcess(PROCESS_CNX).disconnect()
			
			#and re-insert action
			self.addAction(action, True)
			
		elif result==self.RESULT_RETRY and action.attemps<self.MAX_ATTEMPS:
			Log.log('FoxyProcess::__connect : RESULT_RETRY')
			#retry connecting
			action.attemps += 1
			self.__addDelayedAction(self.RETRY_CONNECT, action)
		
		elif result==self.RESULT_KO:
			Log.log('FoxyProcess::__connect : RESULT_KO')
			#error, reconnect until connection was successfull
			#self._ui.updateUi(UI_STATUS, STATUS_INFOS, 'Unable to connect to FoxyTag.')
			#self._ui.updateUi(UI_MESSAGE, 'Unable to connect to server. Retrying...')
			
			#disconnect
			self._ui.getProcess(PROCESS_CNX).disconnect()
			
			#and re-insert action
			self.addAction(action, True)
			
		else:
			Log.log('FoxyProcess::__connect : RESULT_FATAL')
			#fatal error
			action.callback( FoxyTagsProcessResult(False) )
					
		#update data counters
		self.__updateCounters(foxy.upload, foxy.download)
				
	
	##Get foxytags
	##@param action : action to execute
	def __get(self, action):
		#init
		currentPosition = action.params[0]
	
		#request
		foxy = self.__foxy.get(currentPosition.wgs84.latitude, currentPosition.wgs84.longitude, self.DEFAULT_RADIUS)
		Log.log('FoxyProcess::__get : %s' % foxy)
		
		#check result
		result = self.__checkFoxyResultMessages(foxy.messages)
		if result==self.RESULT_OK:
			Log.log('FoxyProcess::__get : RESULT_OK')
			#prepare speedcam object
			speedcams = []
			for foxytag in foxy.tags:
				speedcams.append(Speedcam(foxytag))
		
			#set download is successful
			Log.log('FoxyTagsProcess::__get : %d speedcams' % len(speedcams))
			action.callback( FoxyTagsProcessResult(True, speedcams) )

		elif result==self.RESULT_DISCONNECT:
			Log.log('FoxyProcess::__get : RESULT_DISCONNECT')
			#disconnect
			self._ui.getProcess(PROCESS_CNX).disconnect()
			
			#message
			self._ui.updateUi(UI_MESSAGE, 'No network, drive carrefully!')
			
			#and re-insert action
			self.addAction(action, True)
			
		elif result==self.RESULT_RETRY and action.attemps<self.MAX_ATTEMPS:
			Log.log('FoxyProcess::__get : RESULT_RETRY')
			#retry
			action.attemps += 1
			self.__addDelayedAction(self.RETRY_GET, action)
			
		else :
			Log.log('FoxyProcess::__get : RESULT_FATAL')
			#fatal error
			#self._ui.updateUi(UI_MESSAGE, 'Unable to get tags, be careful!')
			action.callback( FoxyTagsProcessResult(False) )
		
		#update data counters
		self.__updateCounters(foxy.upload, foxy.download)
	
			
	##Post tag
	##@param action : action to execute
	def __post(self, action):
		#init
		foxytag = action.params[0]
		speed = action.params[1]
	
		#request
		foxy = self.__foxy.post(foxytag, speed)
		Log.log('FoxyProcess::__post : %s' % foxy)
		
		#check result
		result = self.__checkFoxyResultMessages(foxy.messages)
		if result==self.RESULT_OK:
			Log.log('FoxyProcess::__post : RESULT_OK')
			#set upload was sucessful
			action.callback( FoxyTagsProcessResult(True) )
			
		elif result==self.RESULT_DISCONNECT:
			Log.log('FoxyProcess::__post : RESULT_DISCONNECT')
			#disconnect
			self._ui.getProcess(PROCESS_CNX).disconnect()
			
			#and re-insert action
			self.addAction(action, True)

		elif result==self.RESULT_RETRY and action.attemps<self.MAX_ATTEMPS:
			Log.log('FoxyProcess::__post : RESULT_RETRY')
			#retry
			action.attemps += 1
			self.__addDelayedAction(self.RETRY_POST, action)
			
		else :
			Log.log('FoxyProcess::__post : RESULT_FATAL')
			#fatal error
			#self._ui.updateUi(UI_STATUS, STATUS_INFOS, 'Unable to post FoxyTag.')
			action.callback( FoxyTagsProcessResult(False) )
		
		#update data counters
		self.__updateCounters(foxy.upload, foxy.download)
	
