#ifndef __LX__LIST_H
#define __LX__LIST_H

#include <core/String.h>
#include <lx/Widget.h>
#include <lx/Image.h>

#include "ListDrawer.h"

namespace lx
{

class Image;

template <typename ItemType>
class List: public Widget
{
    public:
        List(Widget* parent, ListDrawer<ItemType>* drawer)
            : Widget(parent), _drawer(drawer), _buffer(0), _scrolling(false)
        {
            recreateBuffer();
        }

        ~List()
        {
            delete _drawer;
            delete _buffer;
        }


        void addItem(const ItemType& item)
        {
            _items.append(item);
        }


        Point scroll() const { return _scroll; }
        void setScroll(const Point& scroll)
        {
            int oldscroll = _scroll.y;
            _scroll = scroll;

            int height = _drawer->itemHeight();

            if (_scroll.y + size().h > _items.size() * height)
                _scroll.y = _items.size() * height - size().h;
            if (_scroll.y < 0) _scroll.y = 0;

            if (oldscroll == _scroll.y)
                return;

            if (_scroll.y > oldscroll)
            {
                _buffer->copyCanvas(
                    _buffer,
                    Rect(
                        0, _scroll.y - oldscroll,
                        size().w, size().h - (_scroll.y - oldscroll)
                    ),
                    Point(0, 0),
                    true
                );
                realPaint(Rect(
                    0, size().h - (_scroll.y - oldscroll),
                    size().w, _scroll.y - oldscroll
                ));
            }
            else
            {
                _buffer->copyCanvas(
                    _buffer,
                    Rect(
                        0, 0,
                        size().w, size().h - (oldscroll - _scroll.y)
                    ),
                    Point(0, oldscroll - _scroll.y),
                    true
                );
                realPaint(Rect(
                    0, 0,
                    size().w, oldscroll - _scroll.y
                ));
            }

            repaint();
        }


        virtual void setRect(const Rect& rect)
        {
            Widget::setRect(rect);

            setScroll(_scroll);

            recreateBuffer();
        }

        virtual void paint(const Rect& dirty)
        {
            copyCanvas(_buffer, dirty, dirty.origin);
        }

        virtual void mousePress(const Point& point)
        {
            _scrolling = true;
            _dragStart = point;
            _scrollStart = _scroll;
        }

        virtual void mouseRelease(const Point& point)
        {
            _scrolling = false;
        }

        virtual void mouseMove(const Point& point)
        {
            if (_scrolling)
                setScroll(_scrollStart + _dragStart - point);
        }


        void realPaint()
        {
            realPaint(Rect(Point(), size()));
        }

        void realPaint(const Rect& dirty)
        {
            _buffer->fillRectangle(dirty,  Color::transparent);

            int height = _drawer->itemHeight();
            int y = - _scroll.y;

            for (LinkedList<String>::Iter i = _items.head(); i; i++)
            {
                Rect itemRect(Rect(0, y, size().w, height));

                if (dirty.intersects(itemRect))
                {
                    _drawer->paintItem(this, &(*i), _buffer, itemRect, dirty.intersect(itemRect));
                }

                y += height;
            }
        }


    private:
        ListDrawer<ItemType>* _drawer;

        LinkedList<String> _items;

        Image* _buffer;

        Point _scroll;

        bool _scrolling;
        Point _dragStart;
        Point _scrollStart;


        void recreateBuffer()
        {
            delete _buffer;
            _buffer = new Image(display(), size(), rgba());

            realPaint();
        }
};


}

#endif
