#include <lx/Display.h>
#include <lx/Window.h>
#include <lx/Image.h>
#include <lx/BorderImage.h>
#include <lx/Color.h>
#include <lx/MainLoop.h>
#include <lx/HBox.h>
#include <lx/Font.h>
#include <widgets/ScalableButtonStyle.h>
#include <widgets/ScalableButton.h>
#include <widgets/SliderStyle.h>
#include <widgets/Slider.h>
#include <widgets/SimpleButton.h>
#include <widgets/List.h>

#include <cstdio>


class ColorWidget: public lx::Widget
{
    public:
        ColorWidget(Widget* parent, lx::Color color): Widget(parent), _color(color) { }

        virtual void paint(const lx::Rect& dirty)
        {
            fillRectangle(dirty.intersect(rect() - position()), _color);

            Widget::paint(dirty);
        }

    private:
        lx::Color _color;
};


class MainWindow: public lx::Window
{
    private:
        class Drawer: public lx::ListDrawer<lx::String>
        {
            private:
                lx::Font* _font;
                int _fontHeight;

            public:
                Drawer(lx::Display* display)
                {
                    _font = new lx::Font(display, "sans", 26);
                    _fontHeight = _font->extents("W").h;
                }

                ~Drawer()
                {
                    delete _font;
                }

                virtual void paintItem(const lx::List<lx::String>* list, const lx::String* item, lx::Canvas* canvas, const lx::Rect& rect, const lx::Rect& dirty)
                {
                    canvas->drawText(
                        _font, lx::Color(0x20, 0x20, 0x20),
                        *item,
                        rect.origin + lx::Point(10 + 1, rect.size.h / 2 + _fontHeight / 2 + 1),
                        dirty
                    );
                    canvas->drawText(
                        _font, lx::Color::white,
                        *item,
                        rect.origin + lx::Point(10, rect.size.h / 2 + _fontHeight / 2),
                        dirty
                    );

                    canvas->drawLine(
                        rect.origin + lx::Point(0, 1),
                        rect.origin + lx::Point(rect.size.w, 1),
                        lx::Color(0x80, 0x80, 0x80)
                    );
                    canvas->drawLine(
                        rect.origin + lx::Point(0, 0),
                        rect.origin + lx::Point(rect.size.w, 0),
                        lx::Color(0x20, 0x20, 0x20)
                    );
                }

                virtual int itemHeight()
                {
                    return 50;
                }
        };

    public:
        MainWindow(lx::Display* display, bool rgba): lx::Window(display, rgba)
        {
            setBackgroundColor(lx::Color(0x40, 0x3f, 0x40));

            _buttonBox = new ColorWidget(this, lx::Color::black);
            _buttonBox->setLayout(new lx::HBox);


            new lx::SimpleButton(_buttonBox, new lx::Image(display, "playlists.png", true), new lx::Image(display, "playlists-pressed.png", true));

            lx::SimpleButton* prev = new lx::SimpleButton(_buttonBox, new lx::Image(display, "prev.png", true), new lx::Image(display, "prev-pressed.png", true));
            prev->onClick = lx::Delegate0<>(this, &MainWindow::onPrev);

            new lx::SimpleButton(_buttonBox, new lx::Image(display, "play.png", true), new lx::Image(display, "play-pressed.png", true));

            lx::SimpleButton* next = new lx::SimpleButton(_buttonBox, new lx::Image(display, "next.png", true), new lx::Image(display, "next-pressed.png", true));
            next->onClick = lx::Delegate0<>(this, &MainWindow::onNext);

            lx::SimpleButton* rotate = new lx::SimpleButton(_buttonBox, new lx::Image(display, "rotate.png", true), new lx::Image(display, "rotate-pressed.png", true));
            rotate->onClick = lx::Delegate(this, &MainWindow::onRotate);


            _list = new lx::List<lx::String>(this, new Drawer(display));

            for (int i = 0; i < 3; i++)
            {
                _list->addItem("Item one");
                _list->addItem("Item two");
                _list->addItem("Item three");
                _list->addItem("Item four");
                _list->addItem("Item five");
                _list->addItem("Item six");
                _list->addItem("Item seven");
                _list->addItem("Item eight");
                _list->addItem("Item nine");
                _list->addItem("Item ten");
                _list->addItem("Item eleven");
                _list->addItem("Item twelve");
                _list->addItem("Item thirteen");
                _list->addItem("Item fourteen");
                _list->addItem("Item fifteen");
                _list->addItem("Item sixteen");
                _list->addItem("Item seventeen");
                _list->addItem("Item eighteen");
                _list->addItem("Item nineteen");
                _list->addItem("Item twenty");
            }


            _sliderBackground = new lx::Image(display, "slider-background.png", true);
            _sliderButton = new lx::Image(display, "slider-button.png", true);
            _sliderStyle = new lx::SliderStyle(_sliderBackground, _sliderButton, 12, 26);

            _slider = new lx::Slider(this, _sliderStyle);


            setSize(800, 480);
        }

        ~MainWindow()
        {
            delete _sliderStyle;
            delete _sliderBackground;
            delete _sliderButton;
        }


        void onPrev()
        {
            _slider->setValue(_slider->value() - 0.1);
        }

        void onNext()
        {
            _slider->setValue(_slider->value() + 0.1);
        }

        void onRotate()
        {
            switch (orientation())
            {
                case lx::CW0  : setOrientation(lx::CW90); break;
                case lx::CW90 : setOrientation(lx::CW180); break;
                case lx::CW180: setOrientation(lx::CW270); break;
                case lx::CW270: setOrientation(lx::CW0); break;
            }
        }

        void onTimeout()
        {
            printf("TIMEOUT!\n");
            lx::MainLoop::instance()->addTimeout(2, (lx::TimeoutCallback)Delegate(this, &MainWindow::onTimeout));
        }


        virtual void layout()
        {
            _buttonBox->setRect(lx::Rect(
                0, size().h - _buttonBox->size().h,
                size().w, _buttonBox->size().h
            ));

            _slider->setPosition(- _sliderStyle->sideGap(), _buttonBox->position().y - 44);
            _slider->setSize(size().w + 2 * _sliderStyle->sideGap(), _slider->size().h);

            _list->setRect(lx::Rect(
                0, 0,
                size().w, _slider->position().y + 30
            ));
        }


        virtual void setRect(const lx::Rect& rect)
        {
            lx::Window::setRect(rect);
        }



    private:
        ColorWidget* _buttonBox;
        lx::Slider* _slider;

        lx::Image* _sliderBackground;
        lx::Image* _sliderButton;
        lx::SliderStyle* _sliderStyle;

        lx::Font* _font;
        lx::List<lx::String>* _list;
};



int main(int argc, char *argv[])
{
#ifndef DEBUG
    const char* sharedir = "/usr/share/simp-test";
    if (chdir(sharedir) != 0)
    {
        fprintf(stderr, "Cannot chdir to %s\n", sharedir);
        return 1;
    }
#else
    printf("DEBUG BUILD\n");
#endif

    lx::Display display;

    MainWindow window(&display, true);
    window.show();

    lx::MainLoop mainLoop;
    mainLoop.addSource(&display);

    window.onClose = lx::Delegate(&mainLoop, &lx::MainLoop::stop);

    mainLoop.run();
}
