#include "Window.h"


#include "Exception.h"
#include "Image.h"


namespace lx
{


LinkedList<Window*> Window::_windows;




Window::RealCanvas::RealCanvas(): _window(0) { }
Window::RealCanvas::RealCanvas(Window* window): _window(window) { }

GC       Window::RealCanvas::xgc() const { return _window->_xgc; }
Drawable Window::RealCanvas::xdrawable() const { return _window->_xwindow; }
Picture  Window::RealCanvas::xpicture() const { return _window->_xpicture; }
Display* Window::RealCanvas::display() const { return _window->_display; }
bool     Window::RealCanvas::rgba() const { return _window->_rgba; }
Point    Window::RealCanvas::absolutePosition() const { return Point(0, 0); }




Window::Window(Display *display, bool rgba)
    : Widget(0), _display(display), _rgba(rgba), _mouseGrabber(0), _backgroundColor(Color(0, 0, 0, 0)),
      _orientation(CW0)
{
    _windows.append(this);

    XSetWindowAttributes attrs;
    attrs.colormap = rgba ? _display->rgbaColormap() : _display->defaultColormap();
    attrs.background_pixel = 0;
    attrs.border_pixmap = 0;

    _xwindow = XCreateWindow(
        _display->xdisplay(), _display->root(),
        position().x, position().y, size().w, size().h,
        0,
        rgba ? 32 : 24,
        InputOutput,
        rgba ? _display->rgbaVisual() : _display->defaultVisual(),
        CWBackPixel | CWColormap | CWBorderPixel, &attrs
    );


//    XSizeHints* xsizeHints = XAllocSizeHints();
//    XWMHints* xwmHints = XAllocWMHints();
//    XClassHint* xclassHint = XAllocClassHint();
//
//    char* name = "WindowName";
//    char* icon = "apple-red";
//
//    XTextProperty winNameProp, winIconProp;
//
//    xsizeHints->flags = PPosition | PSize;
//    XStringListToTextProperty(&name, 1, &winNameProp);
//    XStringListToTextProperty(&icon, 1, &winIconProp);
//    xwmHints->initial_state = IconicState;
//    xwmHints->input = True;
//    xwmHints->flags = StateHint | InputHint;
//    xclassHint->res_name = "resname";
//    xclassHint->res_class = "resclass";
//
//    XSetWMProperties(
//        _display->xdisplay(), _xwindow,
//        &winNameProp, &winIconProp,
//        0, NULL,
//        xsizeHints, xwmHints, xclassHint
//    );


    _xgc = XCreateGC(_display->xdisplay(), _xwindow, 0, 0);


    _xpicture = XRenderCreatePicture(
        _display->xdisplay(),
        _xwindow,
        rgba ? _display->rgbaPictFormat() : _display->rgbPictFormat(),
        0, 0
    );


    _realCanvas = RealCanvas(this);

    _buffer = 0;
    recreateBuffer();


    Atom wmDelete = _display->WM_DELETE_WINDOW();
    XSetWMProtocols(_display->xdisplay(), _xwindow, &wmDelete, 1);


    XSelectInput(
        _display->xdisplay(), _xwindow,
        ExposureMask | ButtonPressMask | ButtonReleaseMask
        | ButtonMotionMask | PointerMotionHintMask
        | StructureNotifyMask
    );
}


Window::~Window()
{
    delete _buffer;

    XRenderFreePicture(_display->xdisplay(), _xpicture);

    XFreeGC(_display->xdisplay(), _xgc);

    XDestroyWindow(_display->xdisplay(), _xwindow);

    _windows.removeByValue(this);
}



::GC Window::xgc() const { return _buffer->xgc(); }
::Drawable Window::xdrawable() const { return _buffer->xdrawable(); }
::Picture Window::xpicture() const { return _buffer->xpicture(); }
bool Window::rgba() const { return _rgba; }
XftDraw* Window::xftDraw() const { return _buffer->xftDraw(); }

Point Window::absolutePosition() const
{
    return Point(0, 0);
}



void Window::setVisible(bool visible)
{
    if (visible)
        XMapWindow(_display->xdisplay(), _xwindow);
    else
        XUnmapWindow(_display->xdisplay(), _xwindow);

    Widget::setVisible(visible);
}


void Window::recreateBuffer()
{
    delete _buffer;
    _buffer = new Image(_display, size(), _rgba);

    repaint();
}


void Window::setRect(const Rect& rect)
{
    if (rect != this->rect())
    {
        if (rect.size.rotateFrom(_orientation) != _realSize)
        {
            _realSize = rect.size.rotateFrom(_orientation);

            XResizeWindow(
                _display->xdisplay(), _xwindow,
                _realSize.w, _realSize.h
            );
        }

        Widget::setRect(Rect(Point(), rect.size));

        recreateBuffer();
    }
}




void Window::setBackgroundColor(const Color& color)
{
    _backgroundColor = color;
    repaint();
}




void Window::processXEvent(XEvent *event)
{
    switch (event->type)
    {
        case Expose:
        {
            Rect r = realToOriented(Rect(
                event->xexpose.x, event->xexpose.y,
                event->xexpose.width, event->xexpose.height
            ));
            paint(r);
            break;
        }

        case ButtonPress:
        {
            Point p = realToOriented(Point(event->xbutton.x, event->xbutton.y));
            Widget* child = childAt(p);
            child->mousePress(p - child->absolutePosition());

            _mouseGrabber = child;

            break;
        }

        case ButtonRelease:
        {
            Point p = realToOriented(Point(event->xbutton.x, event->xbutton.y));
            Widget* child = _mouseGrabber ? _mouseGrabber : childAt(p);
            child->mouseRelease(p - child->absolutePosition());
            break;
        }

        case MotionNotify:
        {
            Point p;
            ::Window rootXWin, childXWin;
            int rootx, rooty;
            unsigned int mask;
            XQueryPointer(
                _display->xdisplay(), _xwindow,
                &rootXWin, &childXWin,
                &rootx, &rooty,
                &p.x, &p.y,
                &mask
            );

            p = realToOriented(p);

            Widget* child = _mouseGrabber ? _mouseGrabber : childAt(p);
            child->mouseMove(p - child->absolutePosition());
            break;
        }

        case ConfigureNotify:
        {
            if (Size(event->xconfigure.width, event->xconfigure.height) != _realSize)
                setRect(Rect(
                    Point(0, 0),
                    Size(event->xconfigure.width, event->xconfigure.height).rotateTo(_orientation)
                ));

            break;
        }

        case ClientMessage:
        {
            if ((Atom)event->xclient.data.l[0] == _display->WM_DELETE_WINDOW())
            {
                if (onCloseQuery == 0 || onCloseQuery() == true)
                {
                    hide();
                    if (onClose)
                        onClose();
                }
            }

            break;
        }
    }
}



void Window::paint(const Rect& rect)
{
    fillRectangle(rect, _backgroundColor);

    Widget::paint(rect);

    blit(rect);
}

void Window::blit(const Rect& rect)
{
    _realCanvas.copyRotatedCanvas(_buffer, _orientation, rect, orientedToReal(rect).origin, true);
}



void Window::setOrientation(Orientation orientation)
{
    _orientation = orientation;
    setSize(_realSize.rotateTo(_orientation));
}



Point Window::realToOriented(const Point& point)
{
    switch (_orientation)
    {
        case CW0: return point;

        case CW90: return Point(point.y, _realSize.w - point.x);

        case CW180: return Point(_realSize.w - point.x, _realSize.h - point.y);

        case CW270: return Point(_realSize.h - point.y, point.x);
    }
}


Point Window::orientedToReal(const Point& point)
{
    switch (_orientation)
    {
        case CW0: return point;

        case CW90: return Point(_realSize.h - point.y, point.x);

        case CW180: return Point(_realSize.w - point.x, _realSize.h - point.y);

        case CW270: return Point(point.y, _realSize.w - point.x);
    }
}


Rect Window::realToOriented(const Rect& rect)
{
    switch (_orientation)
    {
        case CW0: return rect;

        case CW90: return Rect(
            rect.origin.y, _realSize.w - rect.origin.x - rect.size.w,
            rect.size.h, rect.size.w
        );

        case CW180: return Rect(
            _realSize.w - rect.origin.x - rect.size.w, _realSize.h - rect.origin.y - rect.size.h,
            rect.size.w, rect.size.h
        );

        case CW270: return Rect(
            _realSize.h - rect.origin.y - rect.size.h, rect.origin.x,
            rect.size.h, rect.size.w
        );
    }
}


Rect Window::orientedToReal(const Rect& rect)
{
    switch (_orientation)
    {
        case CW0: return rect;

        case CW90: return Rect(
            _realSize.w - rect.origin.y - rect.size.h, rect.origin.x,
            rect.size.h, rect.size.w
        );

        case CW180: return Rect(
            _realSize.w - rect.origin.x - rect.size.w, _realSize.h - rect.origin.y - rect.size.h,
            rect.size.w, rect.size.h
        );

        case CW270: return Rect(
            rect.origin.y, _realSize.h - rect.origin.x - rect.size.w,
            rect.size.h, rect.size.w
        );
    }
}


}
