#include "MainLoop.h"

#include <cstdio>
#include <math.h>

#include "EventSource.h"
#include "Exception.h"


namespace lx
{

Timeout::Timeout()
{
    _absTime.tv_sec = 0;
    _absTime.tv_usec = 0;
    _userData = 0;
}

Timeout::Timeout(struct timeval absTime, TimeoutCallback callback, void* userData)
    : _absTime(absTime), _callback(callback), _userData(userData)
{
}

void Timeout::fire()
{
    _callback(this, _userData);
}



MainLoop* MainLoop::_instance = 0;



MainLoop::MainLoop()
{
    if (_instance != 0)
        throw Exception("Multiple MainLoops not allowed");

    _instance = this;

    FD_ZERO(&_fdset_orig);
    _maxfd = 0;
}

MainLoop::~MainLoop()
{
    for (LinkedList<Timeout*>::Iter i = _timeouts.head(); i; i++)
        delete *i;
}


void MainLoop::addSource(EventSource* source)
{
    _sources.append(source);

    int fd = source->fd();
    FD_SET(fd, &_fdset_orig);
    if (fd > _maxfd) _maxfd = fd;
}


void MainLoop::run()
{
    _stopped = false;

    for (LinkedList<EventSource*>::Iter i = _sources.head(); i; i++)
        (*i)->fire();


    while (! _stopped)
    {
        Timeout* nearestTimeout = 0;
        struct timeval remaining;

        if (_timeouts.size() > 0)
        {
            while (_timeouts.size() > 0 && nearestTimeout == 0)
            {
                nearestTimeout = *_timeouts.head();

                remaining = *nearestTimeout->absTime();

                struct timeval cur;
                gettimeofday(&cur, 0);

                if (timercmp(&cur, &remaining, >))
                {
                    // already should be fired
                    nearestTimeout->fire();
                    _timeouts.remove(0);
                    delete nearestTimeout;
                    nearestTimeout = 0;
                    continue;
                }

                timersub(&remaining, &cur, &remaining);
            }
        }


        fd_set fdset = _fdset_orig;

        if (select(_maxfd + 1, &fdset, 0, 0, nearestTimeout ? &remaining : 0))
        {
            for (LinkedList<EventSource*>::Iter i = _sources.head(); i; i++)
                if (FD_ISSET((*i)->fd(), &fdset))
                    (*i)->fire();
        }
        else
            if (nearestTimeout)
            {
                _timeouts.remove(0);
                nearestTimeout->fire();
                delete nearestTimeout;
            }
    }
}


void MainLoop::stop()
{
    _stopped = true;
}


Timeout* MainLoop::addTimeout(float sec, TimeoutCallback callback, void* userData)
{
    struct timeval tv;
    gettimeofday(&tv, 0);
    addToTimeval(sec, &tv);

    Timeout* timeout = new Timeout(tv, callback, userData);

    LinkedList<Timeout*>::Iter i = _timeouts.head();
    int n = 0;
    while (i)
    {
        if (timercmp((*i)->absTime(), &tv, >))
            break;

        i++; n++;
    }

    _timeouts.insert(n, timeout);

    return timeout;
}



void MainLoop::addToTimeval(float sec, struct timeval *tv)
{
    double intpart;
    tv->tv_usec += int(modf(sec, &intpart) * 1000000);
    tv->tv_sec += (time_t)intpart;
    if (tv->tv_usec > 1000000)
        tv->tv_sec++;
}

}
