#include "Canvas.h"

#include <cstdio>

#include "Display.h"
#include "Image.h"
#include "Font.h"

namespace lx
{


void Canvas::drawLine(const Point& a, const Point& b, const Color& color)
{
    XSetForeground(display()->xdisplay(), xgc(), color.uint());

    Point offset = absolutePosition();

    XDrawLine(
        display()->xdisplay(), xdrawable(), xgc(),
        a.x + offset.x, a.y + offset.y,
        b.x + offset.x, b.y + offset.y
    );
}


void Canvas::drawImage(const Image* image, const Point& origin, bool copyArea)
{
    copyCanvas(image, Rect(Point(0, 0), image->size()), origin, copyArea);
}


void Canvas::copyCanvas(const Canvas* canvas, const Rect& src, const Point& origin, bool copyAlpha)
{
    Point offset = absolutePosition();

    if (
        (rgba() == false && canvas->rgba() == false) ||
        (rgba() == true && canvas->rgba() == true && copyAlpha)
       )
    {
        XCopyArea(
            display()->xdisplay(),
            canvas->xdrawable(), xdrawable(),
            xgc(),
            src.origin.x, src.origin.y,
            src.size.w, src.size.h,
            origin.x + offset.x, origin.y + offset.y
        );
    }
    else
    {
        XRenderComposite(
            display()->xdisplay(), copyAlpha ? PictOpSrc : PictOpOver,
            canvas->xpicture(), None, xpicture(),
            src.origin.x, src.origin.y,
            0, 0,
            origin.x + offset.x, origin.y + offset.y,
            src.size.w, src.size.h
        );
    }
}


void Canvas::copyScaleCanvas(const Canvas* canvas, const Rect& src, const Rect& dst)
{
    if (src.size == dst.size)
    {
        copyCanvas(canvas, src, dst.origin);
        return;
    }

    if (dst.size.w <= 0 || dst.size.h <= 0)
        return;

    XTransform xform = {{
        { XDoubleToFixed((double)src.size.w / dst.size.w), XDoubleToFixed(0), XDoubleToFixed(src.origin.x) },
        { XDoubleToFixed(0), XDoubleToFixed((double)src.size.h / dst.size.h), XDoubleToFixed(src.origin.y) },
        { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(1) },
    }};
    XRenderSetPictureTransform(display()->xdisplay(), canvas->xpicture(), &xform);

    Point offset = absolutePosition();

    XRenderComposite(
        display()->xdisplay(), PictOpOver,
        canvas->xpicture(), None, xpicture(),
        0, 0,
        0, 0,
        dst.origin.x + offset.x, dst.origin.y + offset.y,
        dst.size.w, dst.size.h
    );

    XTransform identity = {{
        { XDoubleToFixed(1), XDoubleToFixed(0), XDoubleToFixed(0) },
        { XDoubleToFixed(0), XDoubleToFixed(1), XDoubleToFixed(0) },
        { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(1) },
    }};
    XRenderSetPictureTransform(display()->xdisplay(), canvas->xpicture(), &identity);
}


void Canvas::copyRotatedCanvas(const Canvas* canvas, Orientation orn, const Rect& src, const Point& origin, bool copyAlpha)
{
    if (orn == CW0)
        return copyCanvas(canvas, src, origin, copyAlpha);

    Point offset = absolutePosition();

    Size rotatedSize = src.size.rotateTo(orn);

    int op = copyAlpha ? PictOpSrc : PictOpOver;

    switch (orn)
    {
        case CW90:
        {
            XTransform xform = {{
                { XDoubleToFixed(0), XDoubleToFixed(1), XDoubleToFixed(src.origin.x) },
                { XDoubleToFixed(-1), XDoubleToFixed(0), XDoubleToFixed(src.origin.y + src.size.h) },
                { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(1) },
            }};
            XRenderSetPictureTransform(display()->xdisplay(), canvas->xpicture(), &xform);

            break;
        }


        case CW180:
        {
            XTransform xform = {{
                { XDoubleToFixed(-1), XDoubleToFixed(0), XDoubleToFixed(src.origin.x + src.size.w) },
                { XDoubleToFixed(0), XDoubleToFixed(-1), XDoubleToFixed(src.origin.y + src.size.h) },
                { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(1) },
            }};
            XRenderSetPictureTransform(display()->xdisplay(), canvas->xpicture(), &xform);

            break;
        }


        case CW270:
        {
            XTransform xform = {{
                { XDoubleToFixed(0), XDoubleToFixed(-1), XDoubleToFixed(src.size.w + src.origin.x) },
                { XDoubleToFixed(1), XDoubleToFixed(0), XDoubleToFixed(src.origin.y) },
                { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(1) },
            }};
            XRenderSetPictureTransform(display()->xdisplay(), canvas->xpicture(), &xform);

            break;
        }
    }

    XRenderComposite(
        display()->xdisplay(), op,
        canvas->xpicture(), None, xpicture(),
        0, 0,
        0, 0,
        origin.x + offset.x, origin.y + offset.y,
        rotatedSize.w, rotatedSize.h
    );

    XTransform identity = {{
        { XDoubleToFixed(1), XDoubleToFixed(0), XDoubleToFixed(0) },
        { XDoubleToFixed(0), XDoubleToFixed(1), XDoubleToFixed(0) },
        { XDoubleToFixed(0), XDoubleToFixed(0), XDoubleToFixed(1) },
    }};
    XRenderSetPictureTransform(display()->xdisplay(), canvas->xpicture(), &identity);
}


void Canvas::drawRectangle(const Rect& rect, const Color& color)
{
    Point offset = absolutePosition();

    XSetForeground(display()->xdisplay(), xgc(), color.uint());

    XDrawRectangle(
        display()->xdisplay(), xdrawable(), xgc(),
        rect.origin.x + offset.x, rect.origin.y + offset.y,
        rect.size.w, rect.size.h
    );
}


void Canvas::fillRectangle(const Rect& rect, const Color& color)
{
    Point offset = absolutePosition();

    XSetForeground(display()->xdisplay(), xgc(), color.uint());

    XFillRectangle(
        display()->xdisplay(), xdrawable(), xgc(),
        rect.origin.x + offset.x, rect.origin.y + offset.y,
        rect.size.w, rect.size.h
    );
}



void Canvas::drawText(const Font* font, const Color& color, const String& text, const Point& origin)
{
    drawText(font, color, text.utf8(), origin);
}

void Canvas::drawText(const Font* font, const Color& color, const char* utf8, const Point& origin)
{
    if (xftDraw() == 0)
        throw Exception("XftDraw == 0 in drawText!");

    Point offset = absolutePosition();

    XftDrawStringUtf8(
        xftDraw(), color.xftColor(), font->xftFont(),
        origin.x + offset.x, origin.y + offset.y,
        (const FcChar8*)utf8,
        strlen(utf8)
    );
}


void Canvas::drawText(const Font* font, const Color& color, const String& text, const Point& origin, const Rect& clip)
{
    drawText(font, color, text.utf8(), origin, clip);
}

void Canvas::drawText(const Font* font, const Color& color, const char* utf8, const Point& origin, const Rect& clip)
{
    if (xftDraw() == 0)
        throw Exception("XftDraw == 0 in drawText!");

    XRectangle xrect = (clip + absolutePosition()).xrectangle();
    Region region = XCreateRegion();
    XUnionRectWithRegion(&xrect, region, region);

    XftDrawSetClip(xftDraw(), region);

    drawText(font, color, utf8, origin);

    XDestroyRegion(region);
}


}
