/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

// Disable symbol overrides so that we can use system headers.
#define FORBIDDEN_SYMBOL_ALLOW_ALL

#include "tables.h"
#include "create_kyradat.h"
#include "common/language.h"
#include "common/platform.h"

#define EXTRACT_END_ENTRY { UNK_LANG, kPlatformUnknown, { 0, 0, { { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 } } } }

namespace {

// Id provider tables

const ExtractEntrySearchData k1ForestSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000195, 0x00001455, { { 0x86, 0xC2, 0x45, 0xEB, 0x16, 0x8A, 0x44, 0x73, 0xB4, 0x59, 0x13, 0x9B, 0x85, 0x40, 0x82, 0x81 } } } }, // floppy
	{ UNK_LANG, kPlatformUnknown, { 0x000002DE, 0x00003946, { { 0x85, 0x79, 0xDC, 0xB8, 0x62, 0xAD, 0xD6, 0x01, 0xA5, 0x66, 0x6B, 0xC4, 0x61, 0xCA, 0x82, 0xF5 } } } }, // CD + Amiga

	{ UNK_LANG, kPlatformUnknown, { 0x00000273, 0x00001D80, { { 0x18, 0x32, 0xB7, 0xFB, 0xD8, 0x1A, 0x6D, 0x83, 0x75, 0xF5, 0x2B, 0xF7, 0xC0, 0xFD, 0x85, 0xEF } } } }, // Amiga + CD demo

	{ UNK_LANG, kPlatformFMTowns, { 0x00000196, 0x00001468, { { 0x06, 0x8D, 0xC1, 0x9B, 0xBC, 0x28, 0xE8, 0x36, 0x43, 0x67, 0x34, 0xAA, 0x78, 0xB9, 0xCB, 0x72 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x0000013B, 0x0000128A, { { 0xD8, 0xB1, 0xFF, 0x57, 0xC9, 0x5D, 0xE1, 0x23, 0xA5, 0xA0, 0x4B, 0x0F, 0x28, 0xAC, 0x56, 0x83 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1KallakWritingSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000007F5, 0x00006DCD, { { 0x0E, 0x0A, 0x75, 0xA5, 0x71, 0x5A, 0xC6, 0x32, 0x52, 0x7F, 0xEC, 0x2B, 0x0B, 0xF2, 0x22, 0xE6 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000805, 0x0000719E, { { 0xA2, 0xFE, 0x0A, 0xAA, 0xD9, 0x43, 0x5A, 0xBE, 0x56, 0x38, 0x73, 0x42, 0xAC, 0xA9, 0x1B, 0x01 } } } }, // CD

	{ UNK_LANG, kPlatformAmiga, { 0x000007F4, 0x00006DAD, { { 0x84, 0x9D, 0x82, 0xD2, 0x4E, 0x8B, 0xE2, 0x86, 0x82, 0x5E, 0xE7, 0x2C, 0x7A, 0xD6, 0xFE, 0x19 } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x000007F7, 0x00006E23, { { 0x97, 0x8C, 0x63, 0x8D, 0x91, 0x68, 0xA5, 0x63, 0x55, 0x43, 0xE7, 0xE0, 0x12, 0xF9, 0x80, 0xA4 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x000007F3, 0x00006D9E, { { 0x3E, 0x83, 0xA4, 0xBF, 0x43, 0xAC, 0xDD, 0x41, 0x33, 0xC3, 0x33, 0x4E, 0xAE, 0x08, 0xB6, 0xDA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1KyrandiaLogoSeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000005C, 0x000003D3, { { 0x75, 0xC5, 0x55, 0x54, 0x00, 0xD7, 0xE9, 0x26, 0x0F, 0x01, 0xB1, 0xB7, 0xA9, 0x29, 0xF7, 0x69 } } } },
	{ UNK_LANG, kPlatformPC, { 0x000000A6, 0x00000A09, { { 0x70, 0x6E, 0x74, 0xBB, 0xD7, 0x09, 0x05, 0xDD, 0xA3, 0x09, 0x96, 0x40, 0xBD, 0x01, 0x19, 0x05 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000054, 0x00000300, { { 0x4E, 0xAD, 0xD0, 0x73, 0xD2, 0x3C, 0xF6, 0x6B, 0x65, 0x23, 0xA2, 0x30, 0xD6, 0xEE, 0x1C, 0x13 } } } }, // demo

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1KallakMalcolmSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000026B, 0x00002132, { { 0x51, 0x07, 0x32, 0xA2, 0x09, 0x47, 0x97, 0x02, 0x85, 0x31, 0x39, 0x93, 0x3A, 0x53, 0x47, 0xA5 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000267, 0x00002100, { { 0xD9, 0x5E, 0x59, 0xF0, 0x7B, 0xC8, 0xF1, 0x40, 0x4F, 0x68, 0x6F, 0xEC, 0xB5, 0xE8, 0x88, 0xE2 } } } }, // floppy
	{ UNK_LANG, kPlatformUnknown, { 0x0000027B, 0x0000220A, { { 0xB7, 0xC1, 0x57, 0x04, 0x9B, 0x67, 0x82, 0x7B, 0x6E, 0xFD, 0x59, 0xF2, 0x10, 0x93, 0x89, 0x12 } } } }, // CD + Amiga

	{ UNK_LANG, kPlatformUnknown, { 0x000002B8, 0x0000280B, { { 0x98, 0xC8, 0x36, 0x8C, 0xF8, 0x92, 0xC2, 0xB9, 0x1B, 0x71, 0x6B, 0x4C, 0xA4, 0x6C, 0xF6, 0x30 } } } }, // Amiga + CD demo

	{ UNK_LANG, kPlatformFMTowns, { 0x0000026D, 0x00002188, { { 0x7E, 0x3B, 0x06, 0x69, 0x11, 0x59, 0xBE, 0x42, 0xCB, 0x33, 0x9C, 0x15, 0xCF, 0x61, 0xF6, 0xB0 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x0000026C, 0x00002145, { { 0x95, 0xB7, 0x9E, 0x63, 0x99, 0xFA, 0x22, 0xBF, 0x62, 0x37, 0xC2, 0x2B, 0x40, 0x73, 0x53, 0x83 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1MalcolmTreeSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000216, 0x000014FB, { { 0xBA, 0x6F, 0x63, 0xD5, 0x44, 0xCE, 0xE8, 0x20, 0xD4, 0x4C, 0x06, 0x58, 0x21, 0x8E, 0x07, 0x8B } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000230, 0x0000192A, { { 0x28, 0xCC, 0x89, 0x40, 0x8A, 0xF7, 0xCB, 0xC0, 0x11, 0x8F, 0x0F, 0xE5, 0x5F, 0x24, 0x2D, 0x32 } } } }, // CD

	{ UNK_LANG, kPlatformAmiga, { 0x00000214, 0x000014C4, { { 0xA2, 0x60, 0x16, 0x14, 0x20, 0xE4, 0x8D, 0x89, 0xC4, 0x30, 0x07, 0x21, 0xC9, 0x26, 0xA6, 0x84 } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x00000218, 0x00001551, { { 0x30, 0x42, 0xD3, 0x7D, 0x6B, 0x43, 0x3B, 0xC7, 0x63, 0x56, 0x7D, 0x12, 0xE0, 0x53, 0x41, 0x48 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x00000217, 0x0000150E, { { 0x41, 0xFC, 0xD3, 0xAC, 0x8D, 0x1C, 0x11, 0x4E, 0xE0, 0x0C, 0xF6, 0x93, 0xA9, 0x22, 0x53, 0x7F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1WestwoodLogoSeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000004B, 0x000002FE, { { 0x6E, 0xDA, 0x5D, 0x1E, 0xF4, 0x38, 0xAE, 0x7A, 0x9F, 0xBC, 0x00, 0x7C, 0x77, 0x9D, 0x03, 0xAF } } } },
	{ UNK_LANG, kPlatformPC, { 0x0000004B, 0x0000032A, { { 0x9A, 0x54, 0x6F, 0x6C, 0x70, 0xD1, 0x02, 0x94, 0xD1, 0xA1, 0xA7, 0xBE, 0x1B, 0x10, 0xBD, 0x2F } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000044, 0x00000243, { { 0x00, 0x11, 0x10, 0x64, 0x60, 0xE8, 0xB5, 0x59, 0x00, 0x60, 0xF5, 0x10, 0xB5, 0x63, 0x9D, 0x55 } } } }, // demo

	{ UNK_LANG, kPlatformAmiga, { 0x0000004D, 0x00000319, { { 0xE9, 0x5D, 0xD4, 0x60, 0x4E, 0xA6, 0x92, 0x30, 0x9C, 0x77, 0x99, 0xBC, 0xB1, 0x97, 0xFB, 0xBD } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Demo1SeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000026, 0x0000016E, { { 0x61, 0xF2, 0x84, 0x42, 0x89, 0x6E, 0xB8, 0x0D, 0x01, 0xB7, 0xE4, 0x49, 0xA1, 0x72, 0xDC, 0x66 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Demo2SeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000023, 0x00000151, { { 0xE7, 0x9F, 0xA7, 0x80, 0xE3, 0xC1, 0x63, 0xF2, 0x3F, 0x7B, 0x64, 0xD9, 0x70, 0x3A, 0xF1, 0x55 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Demo3SeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000059, 0x000001A8, { { 0x9E, 0x90, 0x4F, 0x9E, 0xBC, 0x39, 0x37, 0xD1, 0x10, 0x67, 0x8C, 0xE5, 0x84, 0xFD, 0xD9, 0x7C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Demo4SeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x0000013E, { { 0xB7, 0x67, 0x1D, 0x50, 0x11, 0xD0, 0x16, 0x91, 0x40, 0x40, 0xD1, 0xC0, 0xA0, 0x79, 0x3D, 0xCE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AmuleteAnimSeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002A, 0x0000025D, { { 0xB1, 0x59, 0x3C, 0x9A, 0x61, 0xC0, 0x1D, 0x80, 0xB5, 0x0C, 0x2E, 0xF9, 0xF6, 0x36, 0xED, 0x7D } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1OutroReunionSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000547, 0x0000781C, { { 0xCF, 0xD6, 0x1D, 0x3D, 0x14, 0x40, 0x88, 0x35, 0x36, 0x4F, 0x0B, 0x1F, 0x9A, 0x1C, 0x3D, 0xAC } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000547, 0x000077E0, { { 0x80, 0xC4, 0xFC, 0xD5, 0xEB, 0xAA, 0xA5, 0x87, 0x58, 0x5E, 0xAA, 0xE7, 0x01, 0x8F, 0x59, 0x3F } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x000005E5, 0x00008918, { { 0x6A, 0x33, 0x8C, 0xB0, 0x16, 0x57, 0x2D, 0xEB, 0xB2, 0xE1, 0x64, 0x80, 0x98, 0x99, 0x98, 0x19 } } } }, // CD

	{ UNK_LANG, kPlatformAmiga, { 0x0000054A, 0x0000785F, { { 0x55, 0xEA, 0xB8, 0x7F, 0x3A, 0x86, 0xCD, 0xA6, 0xBC, 0xA7, 0x9A, 0x39, 0xED, 0xF5, 0x30, 0x0A } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x00000547, 0x00007876, { { 0x7A, 0xC7, 0x80, 0x34, 0x7A, 0x1B, 0xAB, 0xF8, 0xA7, 0x2F, 0x63, 0x3C, 0xDA, 0x89, 0x3F, 0x82 } } } }, // some floppy DOS + FM-TOWNS

	{ UNK_LANG, kPlatformPC98, { 0x00000548, 0x00007889, { { 0x17, 0xE9, 0x40, 0xBA, 0x4A, 0xBD, 0xC6, 0x62, 0x56, 0x96, 0xD6, 0x1B, 0x11, 0xF3, 0xD8, 0x14 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1IntroCPSStringsProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000015, 0x0000071D, { { 0x59, 0x65, 0x08, 0xF9, 0x4C, 0x81, 0xA1, 0xE3, 0x68, 0xF7, 0xE3, 0xF6, 0x33, 0x5F, 0xF5, 0x36 } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x00000014, 0x0000071D, { { 0xBA, 0xB6, 0x58, 0xB3, 0x28, 0x5E, 0x9F, 0x77, 0x69, 0x9D, 0x77, 0x53, 0x9D, 0x0D, 0xB0, 0x29 } } } }, // floppy + PC98
	{ UNK_LANG, kPlatformPC, { 0x00000015, 0x00000786, { { 0xCF, 0x09, 0xE1, 0xD9, 0x8E, 0x34, 0x5D, 0xEA, 0xBC, 0xAC, 0xC4, 0xF4, 0x4A, 0xEC, 0xFF, 0xC1 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000019, 0x000008DB, { { 0x3A, 0xDC, 0x1D, 0xAD, 0xF4, 0x5E, 0xC9, 0x19, 0xE9, 0x84, 0xD1, 0x31, 0x89, 0x6B, 0x6C, 0xF7 } } } }, // Old floppy
	{ UNK_LANG, kPlatformPC, { 0x0000000C, 0x00000413, { { 0xA1, 0xE3, 0x06, 0x53, 0x23, 0x9A, 0xE0, 0xF1, 0xE4, 0xFD, 0xD9, 0x05, 0x22, 0xA6, 0x28, 0x46 } } } }, // demo

	{ UNK_LANG, kPlatformAmiga, { 0x00000016, 0x0000070A, { { 0xD9, 0xDB, 0x91, 0xCD, 0x93, 0x81, 0xC4, 0x3F, 0x14, 0xF1, 0xC5, 0x02, 0xE7, 0x3F, 0x3A, 0x6C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1IntroCOLStringsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000F09, { { 0x05, 0x7B, 0x69, 0xB7, 0x6A, 0xC3, 0x7F, 0xD9, 0x7E, 0x51, 0x87, 0xA5, 0x31, 0xD8, 0x80, 0xB3 } } } },
	{ UNK_LANG, kPlatformUnknown, { 0x0000002B, 0x00000F09, { { 0x01, 0x40, 0x65, 0xC1, 0x33, 0xAA, 0x64, 0xA7, 0x44, 0xD5, 0x89, 0x68, 0x17, 0x47, 0xC7, 0x1D } } } }, // floppy(?) + PC98
	{ UNK_LANG, kPlatformPC, { 0x00000046, 0x0000174F, { { 0xAA, 0x59, 0x01, 0x20, 0x06, 0xB4, 0x80, 0xC0, 0x84, 0x38, 0xFD, 0x97, 0xBF, 0x78, 0xDD, 0x78 } } } }, // demo

	{ UNK_LANG, kPlatformAmiga, { 0x0000002E, 0x00000F09, { { 0x2A, 0x42, 0xB8, 0x4E, 0xC6, 0xF0, 0x5F, 0x4A, 0x08, 0xB4, 0xE3, 0xE5, 0xBD, 0x32, 0x11, 0x06 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1IntroWSAStringsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000093, 0x00002E5D, { { 0xAB, 0xD1, 0x32, 0x89, 0x2A, 0x8D, 0xC0, 0x2C, 0x87, 0x87, 0xA3, 0x14, 0x2A, 0x2A, 0x22, 0x7F } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x00000086, 0x00002E5D, { { 0xD2, 0x8D, 0x62, 0xEF, 0xEB, 0x75, 0x04, 0x13, 0x64, 0x85, 0x6B, 0x6C, 0xE8, 0x6C, 0xCF, 0x52 } } } },
	{ UNK_LANG, kPlatformPC, { 0x0000004D, 0x00001AEC, { { 0x15, 0xBC, 0x8C, 0xD9, 0x33, 0x0C, 0xC5, 0x66, 0x87, 0x3B, 0x76, 0xEE, 0x0C, 0x41, 0x4D, 0x1F } } } }, // demo

	{ UNK_LANG, kPlatformUnknown, { 0x00000080, 0x00002E5D, { { 0x67, 0x03, 0x20, 0xDF, 0xAA, 0x3C, 0x6A, 0xB5, 0x45, 0xD0, 0x00, 0x2B, 0x30, 0xCC, 0x36, 0xB6 } } } }, // Amiga + CD Demo

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1IntroStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000004F5, 0x0001A7B1, { { 0xC7, 0x65, 0x5B, 0x5A, 0x56, 0x43, 0x94, 0x55, 0x5B, 0x00, 0xFD, 0x5D, 0xF4, 0xB5, 0x04, 0x15 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x000004F6, 0x0001A7B1, { { 0xDE, 0x41, 0xB1, 0x98, 0xD5, 0xAD, 0x6B, 0xBA, 0x33, 0x04, 0x19, 0x5D, 0xCC, 0x07, 0xB5, 0x56 } } } }, // CD
	{ EN_ANY, kPlatformPC, { 0x000002FA, 0x00010181, { { 0x48, 0xD3, 0x1B, 0x8E, 0x8D, 0xBC, 0x1F, 0x51, 0x35, 0x71, 0xE2, 0xAB, 0xC8, 0xBE, 0xFC, 0x88 } } } }, // demo

	{ FR_FRA, kPlatformPC, { 0x000005EF, 0x000207E6, { { 0xDC, 0x6C, 0x9A, 0x7C, 0x19, 0x45, 0x2B, 0x25, 0x84, 0xCC, 0xE1, 0x40, 0xFE, 0x8E, 0xF4, 0x51 } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x000005F0, 0x000207E6, { { 0xB8, 0x51, 0xEA, 0x44, 0x64, 0x04, 0xE4, 0x6A, 0x7B, 0xBD, 0x31, 0x62, 0xC0, 0xDD, 0xEA, 0xEF } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x00000605, 0x000212AB, { { 0x8F, 0x3A, 0x9D, 0x9A, 0x81, 0xF2, 0x4A, 0x90, 0xBC, 0xBD, 0x62, 0x8B, 0xD8, 0xD6, 0xF4, 0x5D } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000606, 0x000212AB, { { 0xB0, 0x78, 0x86, 0xFB, 0xCB, 0x3D, 0x1E, 0x1E, 0x86, 0xE7, 0x2C, 0x2B, 0x3A, 0x3C, 0x6F, 0x1C } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x0000057C, 0x0001DF02, { { 0x4B, 0x09, 0xBC, 0xBC, 0xB3, 0x3C, 0x69, 0xEB, 0xA6, 0xB5, 0x76, 0xED, 0xC7, 0x57, 0x71, 0xB0 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x000005F0, 0x0001CF13, { { 0x4F, 0xA0, 0x47, 0x57, 0x6D, 0x2D, 0xA6, 0x62, 0x06, 0xBE, 0x86, 0xD2, 0xD7, 0xEA, 0x2E, 0xD6 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x000005CF, 0x00020415, { { 0xCC, 0xE5, 0x9F, 0xB8, 0xCA, 0xFA, 0x2D, 0x05, 0xB8, 0xAF, 0x9F, 0x1F, 0x8A, 0xA8, 0x56, 0xDE } } } },

	{ RU_RUS, kPlatformPC, { 0x000004F6, 0x000131C6, { { 0x77, 0x76, 0x12, 0xB1, 0xDA, 0x9C, 0xA9, 0xB5, 0x21, 0x1E, 0x49, 0x08, 0x46, 0xB3, 0xE4, 0x61 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x0000050A, 0x0001A7B1, { { 0x1B, 0x74, 0x71, 0x4C, 0xAB, 0x81, 0x10, 0x59, 0x8A, 0x21, 0x50, 0xBB, 0xFE, 0x6F, 0xD0, 0xE8 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000626, 0x00021319, { { 0x80, 0x55, 0x54, 0x14, 0x5D, 0x6F, 0x49, 0x04, 0x4A, 0xEF, 0x92, 0xB8, 0x5B, 0x01, 0x0F, 0x97 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x00000545, 0x0001A84C, { { 0xEF, 0xE2, 0xE4, 0x92, 0xE1, 0x18, 0xF8, 0x3C, 0x91, 0x29, 0x4D, 0x82, 0xA4, 0x3E, 0xE6, 0x7D } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x000005FB, 0x0003163E, { { 0x16, 0x28, 0x83, 0xCE, 0x29, 0x94, 0x9F, 0x82, 0xDC, 0xD4, 0x06, 0xE8, 0x67, 0xC9, 0x56, 0x29 } } } },

	{ JA_JPN, kPlatformPC98, { 0x000005D0, 0x00032AD7, { { 0xEE, 0x36, 0x93, 0xDD, 0x71, 0x02, 0xE9, 0x0D, 0x54, 0xDE, 0xB6, 0xE1, 0x09, 0x59, 0x60, 0x64 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1OutroHomeStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000005, 0x00000189, { { 0xA5, 0xF3, 0x03, 0x3F, 0x43, 0x2B, 0x77, 0xCC, 0x5C, 0x96, 0x64, 0xB1, 0x75, 0xB5, 0x3E, 0xF6 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x00000007, 0x000003E8, { { 0x68, 0x35, 0x87, 0x53, 0xD9, 0x53, 0x1F, 0x13, 0x24, 0x61, 0x0D, 0x8D, 0x33, 0x91, 0xF3, 0x47 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000007, 0x00000267, { { 0xD0, 0xC1, 0x11, 0x49, 0x3D, 0x3F, 0xCE, 0x84, 0x8F, 0xEF, 0xE3, 0xFD, 0x43, 0xCA, 0x92, 0xB0 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000000E, 0x00000473, { { 0xB6, 0xEB, 0xE8, 0x22, 0x67, 0x24, 0xA9, 0xA3, 0x94, 0x55, 0xC6, 0x57, 0x17, 0x15, 0x5B, 0x04 } } } },

	{ ES_ESP, kPlatformPC, { 0x00000005, 0x00000178, { { 0x2E, 0x9C, 0x94, 0x0F, 0x29, 0x77, 0x27, 0x1D, 0x77, 0x1E, 0x5A, 0xF8, 0x0E, 0x8D, 0x09, 0x6B } } } },
	{ IT_ITA, kPlatformPC, { 0x00000005, 0x00000178, { { 0x2E, 0x9C, 0x94, 0x0F, 0x29, 0x77, 0x27, 0x1D, 0x77, 0x1E, 0x5A, 0xF8, 0x0E, 0x8D, 0x09, 0x6B } } } },

	{ IT_ITA, kPlatformPC, { 0x00000007, 0x000001B8, { { 0x17, 0x95, 0x5B, 0x4F, 0xE2, 0x07, 0x5A, 0x49, 0xFA, 0xCE, 0x53, 0x8B, 0xE7, 0x46, 0x69, 0xC7 } } } }, // (fan) CD

	{ RU_RUS, kPlatformPC, { 0x00000005, 0x000000EF, { { 0xA0, 0xB4, 0xF2, 0x11, 0x16, 0x92, 0xC8, 0xEB, 0xF2, 0x0C, 0xFE, 0x43, 0xFE, 0x18, 0xF6, 0xBB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1RoomFilenamesProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x0000034C, 0x0000B7AA, { { 0x7C, 0x1B, 0x89, 0x09, 0xF0, 0x87, 0x3C, 0x64, 0x3D, 0x0E, 0x61, 0xCC, 0x09, 0x8D, 0xC9, 0x95 } } } },
	{ UNK_LANG, kPlatformUnknown, { 0x000002DC, 0x0000B7AA, { { 0x31, 0x56, 0x41, 0x69, 0x1E, 0x84, 0x54, 0x2C, 0xF7, 0x7B, 0x98, 0x25, 0xF3, 0x91, 0x56, 0xFD } } } }, // DOS + PC98

	{ UNK_LANG, kPlatformAmiga, { 0x00000308, 0x0000B7AA, { { 0x6D, 0xA4, 0xE8, 0x27, 0x68, 0x24, 0xCF, 0xAA, 0x30, 0x32, 0x71, 0x74, 0x36, 0x68, 0x82, 0x20 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1RoomListProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x000064E8, 0x0010312B, { { 0x94, 0x5C, 0x87, 0x35, 0x35, 0x6B, 0x3E, 0xBF, 0x55, 0x3D, 0xDB, 0xD9, 0xFB, 0x97, 0x27, 0x5D } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x00004DD6, 0x0010312B, { { 0xC6, 0xF0, 0xC4, 0x2C, 0x5A, 0xD7, 0x48, 0xE4, 0x41, 0x23, 0x65, 0x6D, 0xC8, 0xC7, 0xCE, 0xF5 } } } }, // DOS + PC98
	{ UNK_LANG, kPlatformUnknown, { 0x00004DD6, 0x0010315D, { { 0x4A, 0x1B, 0xA2, 0x35, 0xE1, 0x22, 0xD3, 0x7A, 0xE9, 0x69, 0x12, 0x3A, 0x9C, 0x92, 0x6F, 0x5C } } } }, // Old DOS floppy

	{ UNK_LANG, kPlatformAmiga, { 0x00004ECC, 0x0010312B, { { 0x9A, 0x91, 0xF1, 0x9C, 0x8A, 0x96, 0x1C, 0x7B, 0xB7, 0xE4, 0xF1, 0xE9, 0x7D, 0xEF, 0x40, 0xBF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1CharacterImageFilenamesProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x000000F7, 0x00003A8F, { { 0x1F, 0xAB, 0x8E, 0x5F, 0xAD, 0x76, 0x57, 0x37, 0xD7, 0x9B, 0x3F, 0x35, 0x4B, 0xA0, 0x98, 0x37 } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x000000E4, 0x00003A8F, { { 0x7A, 0x77, 0x6D, 0x26, 0xC8, 0x43, 0xCB, 0x74, 0x46, 0x2E, 0x04, 0x22, 0x89, 0x27, 0x33, 0x06 } } } }, // DOS + PC98

	{ UNK_LANG, kPlatformAmiga, { 0x000000F0, 0x00003A8F, { { 0xC0, 0xD9, 0x8C, 0x2B, 0x65, 0x51, 0x7C, 0x13, 0x52, 0x7E, 0xF7, 0x63, 0xA6, 0xA9, 0xCE, 0x40 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AudioTracksProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000038, 0x00000D5C, { { 0x65, 0x35, 0x2F, 0xA3, 0x93, 0x22, 0x15, 0xA0, 0xC6, 0x2B, 0x73, 0x7C, 0x3E, 0xB8, 0x7A, 0xB5 } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x0000005D, 0x0000154E, { { 0xA7, 0x7E, 0x03, 0x0A, 0x81, 0x54, 0xD2, 0x5D, 0x7B, 0x33, 0x07, 0xBF, 0x70, 0x01, 0x4B, 0x79 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AudioTracks2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000009, 0x00000363, { { 0x16, 0xA2, 0x68, 0x21, 0x04, 0xA8, 0x39, 0x7E, 0xA1, 0x7D, 0x70, 0xFD, 0x86, 0xC7, 0x69, 0x28 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AudioTracksIntroProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000006, 0x0000022C, { { 0x5F, 0xC9, 0xE1, 0x4B, 0x34, 0x52, 0xB9, 0xF8, 0xFF, 0x37, 0x8B, 0xF4, 0xEF, 0x5E, 0xC5, 0xDA } } } }, // floppy + demo
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x00000458, { { 0xEB, 0xB3, 0x96, 0xA5, 0x07, 0xE6, 0x11, 0x58, 0xDB, 0x3F, 0x34, 0x30, 0xFB, 0x7B, 0x92, 0xC8 } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1ItemNamesProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000468, 0x00018B13, { { 0xCE, 0xFE, 0x19, 0x5D, 0x3F, 0xD7, 0x2A, 0xF6, 0xD0, 0xC5, 0xBC, 0x17, 0x6A, 0x52, 0xDB, 0xFE } } } },

	{ FR_FRA, kPlatformPC, { 0x00000506, 0x0001C3CC, { { 0x46, 0xD4, 0x19, 0x04, 0x2A, 0x46, 0xCD, 0x63, 0x41, 0x5A, 0x8E, 0xA8, 0x00, 0xDD, 0x02, 0x77 } } } },

	{ DE_DEU, kPlatformPC, { 0x00000512, 0x0001DD2C, { { 0x5D, 0xB0, 0x6B, 0x8E, 0x9C, 0x70, 0x09, 0x90, 0x3C, 0xC6, 0x58, 0x1F, 0x71, 0x24, 0x76, 0x56 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000591, 0x0001F869, { { 0x4D, 0xD7, 0x66, 0xBB, 0x9A, 0xD8, 0x79, 0x1E, 0x92, 0x2B, 0xE5, 0xC2, 0x0F, 0x02, 0x0B, 0x35 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000506, 0x0001C1D9, { { 0xF2, 0x2D, 0x18, 0x13, 0x61, 0xFB, 0x57, 0x93, 0x5C, 0xDC, 0x79, 0x44, 0x96, 0x01, 0xA6, 0x90 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000530, 0x0001D90A, { { 0x52, 0xFB, 0xA8, 0x3F, 0xA3, 0x6F, 0xC2, 0x67, 0x55, 0x9F, 0x66, 0x9F, 0xFD, 0x79, 0x44, 0xDF } } } },

	{ RU_RUS, kPlatformPC, { 0x000004AE, 0x00011888, { { 0x6F, 0x4D, 0xBE, 0xC8, 0xAE, 0x7C, 0x12, 0x3E, 0x69, 0x0B, 0x39, 0xCB, 0x4D, 0x4B, 0xA8, 0x3A } } } }, // floppy

	{ EN_ANY, kPlatformAmiga, { 0x00000380, 0x00012960, { { 0x2D, 0x81, 0xCF, 0x7A, 0x9D, 0x71, 0x83, 0xB7, 0xE5, 0x00, 0xB0, 0x6E, 0x25, 0x94, 0xCB, 0xA4 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x000003E5, 0x0001607D, { { 0x6D, 0xBE, 0xAD, 0xE5, 0xD1, 0x41, 0x6C, 0x42, 0x71, 0x79, 0x9C, 0x78, 0x93, 0x84, 0xC8, 0x11 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x00000505, 0x00018B13, { { 0x62, 0x76, 0x84, 0x5D, 0x8D, 0xBA, 0x23, 0xC5, 0x67, 0x26, 0x22, 0xBD, 0xBE, 0x81, 0x2D, 0xFF } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000587, 0x000247A1, { { 0xE8, 0xCA, 0x1E, 0x34, 0x05, 0xE5, 0x08, 0xCA, 0xA7, 0xFE, 0x55, 0xCB, 0x75, 0xD0, 0xA7, 0x19 } } } },

	{ JA_JPN, kPlatformPC98, { 0x000004CF, 0x000247A1, { { 0xBE, 0x80, 0x3F, 0x10, 0xF7, 0x3A, 0x85, 0xD1, 0xE8, 0x24, 0xE6, 0x4E, 0x8F, 0x17, 0xC9, 0x2E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TakenStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000010, 0x000004C2, { { 0x08, 0xC0, 0x66, 0xB1, 0x45, 0x04, 0x78, 0xF9, 0xAF, 0x0F, 0x52, 0xC8, 0x0E, 0x0C, 0x69, 0x86 } } } },

	{ FR_FRA, kPlatformPC, { 0x0000001E, 0x000008FA, { { 0xC9, 0xCC, 0x2E, 0x0B, 0xE4, 0xE0, 0x44, 0xBD, 0xC2, 0x10, 0x09, 0xA3, 0x36, 0xB1, 0x5D, 0x1C } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x0000000F, 0x0000047D, { { 0xE8, 0xB5, 0x15, 0xD5, 0x96, 0x81, 0xD2, 0x84, 0xAB, 0x33, 0xC1, 0x1C, 0x47, 0x1B, 0xFE, 0xDB } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x0000001C, 0x000009C0, { { 0xA8, 0x64, 0xDD, 0x9D, 0x91, 0x42, 0x04, 0x26, 0xD0, 0x80, 0x40, 0xED, 0x08, 0x51, 0xC9, 0xBB } } } },

	{ IT_ITA, kPlatformPC, { 0x00000014, 0x00000588, { { 0x5B, 0x75, 0xAE, 0xD7, 0x50, 0x6C, 0x3E, 0x65, 0x5C, 0xF8, 0x84, 0x34, 0x48, 0x58, 0x0E, 0x6E } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000000F, 0x000002F0, { { 0x37, 0x22, 0x62, 0xE4, 0x1B, 0x1F, 0xD6, 0x9E, 0x94, 0xA3, 0x41, 0x00, 0xD5, 0x29, 0x28, 0x09 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000014, 0x000005D8, { { 0xD6, 0x00, 0x90, 0x6A, 0x75, 0x3B, 0xF1, 0xFE, 0xF4, 0x3E, 0x0E, 0x1D, 0x39, 0xEB, 0x2D, 0xC8 } } } },

	{ RU_RUS, kPlatformPC, { 0x00000010, 0x00000262, { { 0x1E, 0x90, 0x20, 0xC8, 0xD3, 0x08, 0x53, 0x4F, 0x28, 0x95, 0x6A, 0xA4, 0x14, 0x37, 0x05, 0xF0 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000008, 0x00000261, { { 0x93, 0x5B, 0x79, 0xE8, 0x89, 0x8E, 0xB5, 0x37, 0x39, 0x2A, 0xB0, 0x04, 0x98, 0x80, 0x5A, 0x4E } } } },
	{ DE_DEU, kPlatformAmiga, { 0x0000000E, 0x000004E0, { { 0x52, 0x4D, 0x74, 0x91, 0x70, 0x0D, 0x4C, 0x40, 0x5C, 0x7E, 0xBA, 0xDA, 0x24, 0x49, 0xF3, 0x1A } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000001C, 0x00000AED, { { 0x32, 0xD2, 0x87, 0x67, 0x3D, 0x01, 0x50, 0xF3, 0xFA, 0x19, 0x7A, 0xB0, 0xF8, 0x6F, 0x8A, 0x07 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000001F, 0x00000EB7, { { 0x7F, 0x8C, 0xFF, 0x35, 0x30, 0x71, 0x9A, 0x78, 0xCC, 0x07, 0x11, 0x86, 0xE3, 0x0D, 0x16, 0xDF } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000016, 0x00000C56, { { 0x97, 0xF0, 0x57, 0xCC, 0x03, 0x06, 0x41, 0xDC, 0xFC, 0xD5, 0xA2, 0x35, 0x0C, 0x71, 0x37, 0xCC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PlacedStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000009, 0x000002B7, { { 0xE7, 0xCF, 0x8A, 0x19, 0xEF, 0x98, 0xE5, 0x74, 0x46, 0x46, 0x47, 0x65, 0xB6, 0x81, 0x70, 0x0A } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000000D, 0x0000074C, { { 0x0E, 0x1C, 0x1F, 0xD2, 0xCF, 0xBF, 0x40, 0xE1, 0x59, 0x0F, 0x1B, 0x46, 0xED, 0x8B, 0x96, 0x8E } } } },

	{ FR_FRA, kPlatformPC, { 0x00000011, 0x00000545, { { 0x05, 0xEF, 0x74, 0x54, 0x06, 0xB7, 0x64, 0x47, 0x21, 0x56, 0x55, 0x74, 0x70, 0x2D, 0xA8, 0x23 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000000B, 0x00000389, { { 0xB9, 0x74, 0xBE, 0x63, 0xB4, 0xA4, 0x8A, 0x04, 0xD9, 0x50, 0x73, 0xB7, 0x01, 0x9E, 0x77, 0xD3 } } } },

	{ IT_ITA, kPlatformPC, { 0x0000000D, 0x0000040D, { { 0x9C, 0x71, 0x53, 0x35, 0xC3, 0xE8, 0x46, 0xB9, 0xD2, 0xFA, 0x1C, 0x8C, 0xC3, 0xFF, 0xBC, 0x1F } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000011, 0x000003B8, { { 0xC8, 0xA6, 0xE4, 0x8A, 0xF7, 0x4C, 0x3F, 0xA6, 0x24, 0x7F, 0xEF, 0xE4, 0x63, 0x8B, 0x72, 0xF3 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000000D, 0x00000439, { { 0x57, 0xAE, 0x1C, 0xC1, 0xF5, 0xE8, 0x5B, 0x9E, 0x90, 0x02, 0xB9, 0x8D, 0x86, 0x38, 0xFB, 0xA8 } } } },

	{ RU_RUS, kPlatformPC, { 0x00000009, 0x00000203, { { 0x7D, 0xAE, 0x67, 0x94, 0x8E, 0x73, 0x35, 0xC1, 0x11, 0xB4, 0x55, 0x6E, 0x92, 0x25, 0x39, 0xE4 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1DroppedStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000A, 0x0000033C, { { 0x96, 0x63, 0xFD, 0x73, 0x71, 0xCC, 0x62, 0x59, 0x86, 0x02, 0x47, 0x55, 0x82, 0xF8, 0x5E, 0x1F } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000000B, 0x0000059F, { { 0xDD, 0x5E, 0x51, 0x7E, 0xD9, 0xFC, 0xCD, 0xAD, 0x6B, 0x93, 0x71, 0xBE, 0x83, 0x63, 0x3F, 0x88 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000011, 0x00000579, { { 0x3D, 0x0C, 0x3C, 0xAD, 0x2E, 0xE3, 0x7A, 0x5A, 0x4A, 0x21, 0x1C, 0x96, 0xC5, 0x5A, 0xA7, 0x9E } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000011, 0x00000612, { { 0xEC, 0xCD, 0x99, 0x58, 0xF7, 0x08, 0x92, 0x43, 0x95, 0x10, 0x6B, 0xDD, 0x61, 0xF7, 0x5C, 0xA4 } } } },

	{ IT_ITA, kPlatformPC, { 0x0000000B, 0x0000031B, { { 0x32, 0x0C, 0xCD, 0x83, 0xF6, 0x97, 0xC8, 0x76, 0x57, 0x6B, 0x7C, 0xAD, 0x51, 0xEB, 0x91, 0x1F } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000011, 0x000003B8, { { 0xC8, 0xA6, 0xE4, 0x8A, 0xF7, 0x4C, 0x3F, 0xA6, 0x24, 0x7F, 0xEF, 0xE4, 0x63, 0x8B, 0x72, 0xF3 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000008, 0x00000261, { { 0x1D, 0xB5, 0xFB, 0x23, 0x94, 0xA7, 0x86, 0x7A, 0xAC, 0x53, 0xDA, 0x6F, 0xCC, 0x41, 0x0F, 0xD7 } } } },

	{ RU_RUS, kPlatformPC, { 0x0000000A, 0x000001F5, { { 0xAA, 0x21, 0x88, 0x6D, 0xD0, 0xAB, 0x5C, 0x15, 0x7F, 0xAD, 0x0E, 0x3B, 0x2F, 0x17, 0xBF, 0xAD } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1NoDropStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000047, 0x00001881, { { 0xEF, 0x81, 0x2B, 0x8F, 0x06, 0x0C, 0xA6, 0xBA, 0x50, 0x9C, 0x99, 0x40, 0x53, 0xFD, 0xC1, 0x55 } } } },

	{ FR_FRA, kPlatformPC, { 0x0000004C, 0x00001B39, { { 0xFF, 0xD5, 0x5D, 0x1C, 0xF1, 0x8A, 0xA4, 0x21, 0xB1, 0x56, 0x2B, 0xAC, 0x9F, 0x40, 0x30, 0x53 } } } },

	{ DE_DEU, kPlatformPC, { 0x0000004C, 0x00001AFF, { { 0xE5, 0x05, 0x79, 0x10, 0x91, 0x95, 0x64, 0x40, 0xB2, 0x73, 0x39, 0xFA, 0x50, 0x8D, 0xCE, 0x8A } } } },

	{ IT_ITA, kPlatformPC, { 0x00000045, 0x00001850, { { 0x50, 0x34, 0x82, 0xA9, 0x28, 0xDE, 0x44, 0xB3, 0x2D, 0x59, 0x0B, 0x7F, 0xCE, 0x42, 0x1F, 0x73 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000004C, 0x00001650, { { 0x42, 0x7A, 0x04, 0xCD, 0x69, 0xB1, 0x06, 0x9A, 0xD8, 0x7E, 0x7A, 0x14, 0x5B, 0x1C, 0x70, 0xB8 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000003D, 0x000015FA, { { 0x87, 0x2E, 0xE6, 0x8A, 0xF1, 0xC9, 0xC9, 0xEC, 0xD3, 0x2C, 0x25, 0x7E, 0x36, 0x02, 0xF7, 0xAC } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000048, 0x00001881, { { 0x69, 0xA8, 0x0B, 0x47, 0xFD, 0xA0, 0x94, 0x12, 0x82, 0x1D, 0xE0, 0x9C, 0xB1, 0x10, 0x6D, 0x11 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x0000004D, 0x00001AFF, { { 0xF8, 0x83, 0x2E, 0x08, 0x88, 0x7B, 0x72, 0x7F, 0x71, 0xD4, 0x05, 0xF3, 0x1A, 0x78, 0xF1, 0x9D } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000007A, 0x0000323C, { { 0x00, 0x7F, 0x8E, 0x2E, 0x7B, 0xFC, 0x06, 0x0B, 0x7A, 0x7F, 0x17, 0xAF, 0x47, 0x82, 0xC1, 0x16 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000061, 0x00002E9C, { { 0x8A, 0xFA, 0x43, 0x26, 0x68, 0x6D, 0x5A, 0x59, 0x0C, 0x9E, 0xCB, 0x28, 0x43, 0x33, 0x54, 0x49 } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000042, 0x000025C2, { { 0x6F, 0x34, 0x59, 0x17, 0x22, 0xCE, 0x1A, 0x82, 0x18, 0xE3, 0x9F, 0x86, 0x84, 0xF1, 0x8E, 0x0C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PutDownStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000026, 0x00000D27, { { 0x83, 0x09, 0x51, 0xD3, 0x94, 0x2B, 0xB1, 0x8B, 0x8B, 0x16, 0xA9, 0xEB, 0xE8, 0xA2, 0x4F, 0x0C } } } },

	{ JA_JPN, kPlatformUnknown, { 0x00000029, 0x0000187D, { { 0xDE, 0xEE, 0x66, 0x88, 0x57, 0xF5, 0xF9, 0x2E, 0xD2, 0x14, 0xF5, 0x83, 0xA0, 0x0D, 0x96, 0x86 } } } },

	{ FR_FRA, kPlatformPC, { 0x0000002E, 0x0000101D, { { 0xBC, 0xEA, 0x6C, 0x1F, 0x31, 0xCF, 0x30, 0x26, 0x2D, 0x24, 0xC5, 0xEA, 0x97, 0x49, 0xDC, 0x4C } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000028, 0x00000E7E, { { 0x26, 0x93, 0x76, 0x37, 0x41, 0x2E, 0xF3, 0xED, 0xF5, 0x7B, 0xA7, 0xEB, 0x80, 0x61, 0x3B, 0x84 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000034, 0x0000131D, { { 0x36, 0x87, 0x4D, 0x73, 0x2A, 0x67, 0xBA, 0xCE, 0xCB, 0x33, 0x3C, 0x10, 0xCC, 0x3E, 0x7E, 0xAD } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000002E, 0x00000EB2, { { 0x14, 0x08, 0xE6, 0xD1, 0x54, 0x76, 0x2A, 0x9E, 0xBE, 0x5A, 0x15, 0xEB, 0x52, 0x01, 0x52, 0x97 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000002D, 0x00001052, { { 0x12, 0x0A, 0x23, 0x11, 0xDF, 0x8A, 0x59, 0xD4, 0xF2, 0xCA, 0xA5, 0xA7, 0x76, 0x1B, 0x54, 0xB6 } } } },

	{ RU_RUS, kPlatformPC, { 0x00000024, 0x0000099F, { { 0x05, 0xD7, 0xB8, 0x32, 0x95, 0x93, 0x29, 0x5F, 0xF3, 0x1A, 0xF0, 0x2E, 0xBA, 0x3A, 0x0D, 0x27 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1WaitAmuletStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000003D, 0x0000150D, { { 0x11, 0xC8, 0x7E, 0x83, 0x7A, 0xC3, 0x5D, 0xD9, 0x72, 0xFC, 0x75, 0xCB, 0x57, 0xE9, 0xF1, 0x36 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x00000029, 0x000017A7, { { 0xFD, 0xA3, 0xAD, 0x24, 0x16, 0x49, 0x22, 0x6F, 0x52, 0xE9, 0x50, 0x11, 0x51, 0x38, 0xCA, 0xE0 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000049, 0x00001ADD, { { 0xFD, 0xCF, 0x9C, 0xAD, 0xA2, 0xAA, 0x74, 0x11, 0x8E, 0x89, 0x02, 0xCB, 0x2B, 0x5B, 0x2B, 0x1C } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000048, 0x000019CF, { { 0x82, 0xDD, 0xBD, 0xA3, 0xD0, 0xE0, 0x19, 0xC0, 0x6D, 0xC5, 0xEE, 0x7A, 0x0B, 0x49, 0x06, 0xCE } } } },

	{ IT_ITA, kPlatformPC, { 0x0000004A, 0x00001B4C, { { 0x38, 0x26, 0x54, 0x10, 0x3D, 0x32, 0xCD, 0xAE, 0x3F, 0x96, 0xFE, 0x6F, 0x4A, 0x94, 0x72, 0xB0 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000049, 0x0000184F, { { 0x11, 0x42, 0xB2, 0xD6, 0x76, 0xEA, 0xF9, 0xCA, 0x46, 0xE4, 0xA6, 0x35, 0xE8, 0xB9, 0x8B, 0xE9 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000042, 0x000017FD, { { 0x0A, 0x8A, 0x7E, 0x9A, 0x5F, 0x4A, 0x35, 0x06, 0x4D, 0x6B, 0xBF, 0x29, 0x1B, 0xAD, 0xD8, 0x37 } } } },

	{ RU_RUS, kPlatformPC, { 0x0000003C, 0x00000EF1, { { 0xC1, 0x0A, 0xFA, 0xBB, 0x65, 0xC3, 0x31, 0xC9, 0x80, 0x9B, 0x0C, 0x16, 0xED, 0xBF, 0x47, 0xFA } } } },

	{ EN_ANY, kPlatformUnknown, { 0x0000003E, 0x0000150D, { { 0xA8, 0xBF, 0x99, 0x9B, 0xC1, 0x36, 0x21, 0x47, 0x6D, 0x99, 0x4F, 0x34, 0xE6, 0x61, 0x47, 0xFD } } } }, // Amiga + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1BlackJewelStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000023, 0x00000B73, { { 0xBE, 0xE9, 0xBD, 0x65, 0xB9, 0xB1, 0xCD, 0xF0, 0x45, 0x03, 0x01, 0x70, 0xBE, 0xD0, 0x0F, 0x80 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x00000035, 0x0000205F, { { 0x33, 0xD6, 0x19, 0xEE, 0xEA, 0xDD, 0x9F, 0xB5, 0x77, 0x0C, 0x8E, 0x84, 0x5E, 0x61, 0x22, 0x3E } } } },

	{ FR_FRA, kPlatformPC, { 0x0000002B, 0x00000F11, { { 0x68, 0xA0, 0xA3, 0xA0, 0xB6, 0xD1, 0x40, 0x22, 0xB4, 0x10, 0x3D, 0x5A, 0x7F, 0x1F, 0x4B, 0xE7 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000031, 0x0000113B, { { 0xA5, 0x00, 0xD1, 0xDC, 0x25, 0x99, 0x3E, 0x6B, 0xFF, 0xC8, 0x6C, 0xDA, 0xA4, 0xCF, 0x51, 0xD7 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000022, 0x00000B6E, { { 0xAE, 0xA0, 0xDF, 0x07, 0xDD, 0x88, 0x98, 0x6B, 0xDD, 0x9E, 0x2E, 0xC0, 0xBD, 0x45, 0x2E, 0x2B } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000002B, 0x00000D6D, { { 0x08, 0x5E, 0x1F, 0xA8, 0x8F, 0x1F, 0x6A, 0xE0, 0xF3, 0x49, 0xEF, 0x44, 0xA8, 0xCD, 0xDE, 0x45 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000025, 0x00000CF6, { { 0x4B, 0x13, 0x39, 0xCB, 0x3F, 0x44, 0x18, 0x46, 0x43, 0xDB, 0x94, 0xC5, 0x3E, 0x6B, 0xC4, 0x74 } } } },

	{ RU_RUS, kPlatformPC, { 0x00000021, 0x000007FF, { { 0x3F, 0x26, 0xB4, 0xB4, 0x11, 0x0C, 0xEF, 0xC0, 0x6A, 0xD1, 0xCC, 0x0E, 0x68, 0x7D, 0xA5, 0x1A } } } },

	{ EN_ANY, kPlatformUnknown, { 0x00000024, 0x00000B73, { { 0x8D, 0x57, 0x5F, 0x93, 0x85, 0x75, 0xF2, 0xD8, 0x36, 0xC2, 0x7C, 0x0E, 0x3B, 0xEA, 0xE0, 0x0A } } } }, // Amiga + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PoisonGoneStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000002D, 0x00000F59, { { 0x9A, 0x88, 0x07, 0x5A, 0x9C, 0xC3, 0x7B, 0xB0, 0x79, 0x69, 0xE0, 0x84, 0x11, 0x9B, 0x82, 0xCD } } } },

	{ FR_FRA, kPlatformPC, { 0x00000059, 0x00001E91, { { 0xDA, 0x9C, 0xC4, 0x7A, 0x79, 0x73, 0x60, 0xBF, 0x31, 0xF8, 0x8F, 0xAF, 0xC1, 0x4D, 0x25, 0xEC } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x0000002A, 0x00000EAA, { { 0x0B, 0x4E, 0xDF, 0x03, 0x9E, 0x64, 0x3B, 0x13, 0x10, 0x6A, 0xF9, 0x4C, 0x66, 0x0D, 0x09, 0xC5 } } } }, // (fan) CD

	{ DE_DEU, kPlatformPC, { 0x00000036, 0x00001324, { { 0x30, 0x67, 0xEE, 0x7C, 0x4D, 0x36, 0xFE, 0x46, 0x05, 0x44, 0x23, 0xA1, 0xB4, 0x07, 0x16, 0xB5 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000034, 0x0000128D, { { 0x4E, 0x54, 0xFE, 0x58, 0x7F, 0x15, 0xFA, 0xC1, 0x50, 0xF8, 0x47, 0x29, 0x15, 0xE8, 0x83, 0xC7 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000002A, 0x00000DFD, { { 0x57, 0xDC, 0xE0, 0x45, 0x2E, 0x89, 0x77, 0x7F, 0x4B, 0x41, 0xCF, 0x4A, 0xFE, 0x0B, 0xAF, 0xC1 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000033, 0x0000127E, { { 0x67, 0xEB, 0xD3, 0x00, 0xF8, 0x4F, 0xF1, 0x79, 0x48, 0xE6, 0x9C, 0xB2, 0xA7, 0xCF, 0x76, 0x07 } } } },

	{ RU_RUS, kPlatformPC, { 0x00000027, 0x00000952, { { 0x36, 0x64, 0x30, 0x1C, 0x5A, 0xC0, 0x0D, 0x73, 0xE5, 0xA6, 0x2F, 0xD8, 0x64, 0x98, 0x81, 0x56 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x0000002E, 0x00000F59, { { 0xAD, 0x95, 0xF3, 0xA7, 0xBB, 0x04, 0x08, 0x77, 0xD0, 0x71, 0xFC, 0x8B, 0x33, 0x2A, 0x6D, 0xD3 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000037, 0x00001324, { { 0xB3, 0xE6, 0x0A, 0x49, 0x37, 0x73, 0x3C, 0xAF, 0x78, 0x9E, 0x7D, 0x13, 0x75, 0xAE, 0xA8, 0x89 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000003C, 0x000013B4, { { 0xE9, 0x54, 0xF1, 0xEC, 0x64, 0x5A, 0xEB, 0xC7, 0xAA, 0x7E, 0xA2, 0x9F, 0xE8, 0xFB, 0x3A, 0x82 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000003D, 0x000017D1, { { 0x7C, 0x93, 0x62, 0x32, 0x47, 0xB9, 0x78, 0x03, 0x86, 0x8D, 0x99, 0xB8, 0x86, 0x68, 0xEF, 0x3A } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000016, 0x00000B60, { { 0xFF, 0x8C, 0xC0, 0x1B, 0x2C, 0x0C, 0x23, 0x89, 0x93, 0x79, 0xC5, 0xCD, 0xAC, 0xF9, 0xEA, 0xA7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1HealingTipStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000002D, 0x00000F04, { { 0xC0, 0xC7, 0x0C, 0x3D, 0xA7, 0x62, 0x14, 0xFB, 0xE8, 0x52, 0x05, 0x0D, 0xFE, 0xF6, 0xC7, 0x28 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000002B, 0x00001949, { { 0xC7, 0xE3, 0x0A, 0x6B, 0x8F, 0xCA, 0xBC, 0x3A, 0xDC, 0x76, 0x48, 0xD3, 0x8B, 0xD9, 0x44, 0x2E } } } },

	{ FR_FRA, kPlatformPC, { 0x00000036, 0x0000132C, { { 0x86, 0x65, 0x80, 0x10, 0x40, 0x32, 0x6C, 0x2F, 0x5E, 0x76, 0xFB, 0xDC, 0x19, 0x33, 0x02, 0x3C } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000003C, 0x00001537, { { 0x78, 0xC2, 0x76, 0x24, 0x0D, 0x1F, 0xC0, 0x13, 0x4E, 0x34, 0x0A, 0x50, 0x4C, 0x35, 0x7E, 0xB1 } } } },

	{ IT_ITA, kPlatformPC, { 0x0000003F, 0x0000170E, { { 0x0B, 0x37, 0xA4, 0x5E, 0x05, 0x4A, 0x96, 0x1F, 0x2E, 0x02, 0x43, 0xBE, 0xCC, 0xF9, 0x21, 0x5E } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000036, 0x00001364, { { 0x59, 0x70, 0x13, 0x8D, 0x93, 0x9C, 0x1B, 0x69, 0x2F, 0x13, 0x9A, 0xB2, 0x4C, 0x97, 0x7E, 0x95 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000028, 0x00000E0F, { { 0x3E, 0x40, 0xCA, 0x2A, 0x5F, 0xFE, 0x74, 0x30, 0x8C, 0x31, 0x41, 0x09, 0xBD, 0xFD, 0xA3, 0x7E } } } },

	{ RU_RUS, kPlatformPC, { 0x00000026, 0x000008EE, { { 0x7C, 0xC0, 0x62, 0x39, 0x66, 0x9E, 0x63, 0xCD, 0x21, 0x3D, 0x72, 0x91, 0xB8, 0xB9, 0xB6, 0x92 } } } },

	{ EN_ANY, kPlatformUnknown, { 0x0000002E, 0x00000F04, { { 0x95, 0x39, 0x36, 0x89, 0xC4, 0x60, 0x7C, 0x0C, 0xDC, 0x06, 0xF7, 0x86, 0x1A, 0xF7, 0x93, 0x2B } } } }, // Amiga + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1WispJewelStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000053, 0x00001C62, { { 0x9A, 0xF1, 0xCC, 0xB3, 0x09, 0xF0, 0x9C, 0x33, 0x0E, 0xF0, 0xE3, 0xF8, 0x24, 0x63, 0x40, 0x95 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000052, 0x00001CE8, { { 0x6D, 0x69, 0x6D, 0x15, 0x4F, 0xF1, 0xD8, 0xA6, 0x53, 0x9F, 0xE3, 0x77, 0x75, 0x25, 0xE7, 0x93 } } } }, // CD

	{ FR_FRA, kPlatformPC, { 0x0000005B, 0x00001E9A, { { 0xC1, 0x26, 0xF3, 0x64, 0x18, 0x7D, 0xF7, 0x5A, 0xB6, 0x2B, 0x44, 0x57, 0xDC, 0x60, 0x20, 0x3C } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x00000050, 0x00001B8D, { { 0x9A, 0xCE, 0x1D, 0x62, 0xFA, 0xC6, 0x73, 0x80, 0x02, 0x0A, 0x61, 0x24, 0xDC, 0x2A, 0xEF, 0xE6 } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x00000061, 0x00002184, { { 0x4B, 0x20, 0xD7, 0xE6, 0x00, 0xBF, 0x32, 0x40, 0xB7, 0xD0, 0xD6, 0xEC, 0x9C, 0x0F, 0x0B, 0xF0 } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000058, 0x00001E88, { { 0x0E, 0x14, 0xD1, 0xF3, 0x03, 0x29, 0x48, 0x1D, 0x32, 0x2A, 0xBB, 0x0B, 0x48, 0xB3, 0x2E, 0x47 } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x00000074, 0x00002945, { { 0x17, 0x8B, 0x7E, 0xD9, 0x04, 0x7C, 0xD4, 0x9D, 0xCE, 0xF0, 0x79, 0x00, 0xAC, 0x82, 0x7C, 0x82 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000050, 0x00001A10, { { 0xFB, 0x79, 0x74, 0x14, 0xB9, 0x75, 0x4F, 0x44, 0x83, 0xB1, 0xD4, 0xBE, 0x17, 0x0F, 0xC4, 0xD6 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000005F, 0x0000211E, { { 0xE7, 0x0A, 0x85, 0x25, 0x44, 0x41, 0x47, 0x3B, 0x7A, 0xA6, 0x62, 0xAE, 0xAE, 0xD5, 0x92, 0x45 } } } },

	// only one of two strings translated in the fan translation
	{ RU_RUS, kPlatformPC, { 0x00000053, 0x0000191F, { { 0x14, 0xEB, 0x38, 0x54, 0x40, 0x40, 0x04, 0xA6, 0xA0, 0xFE, 0xDB, 0xD0, 0x8C, 0xA6, 0x1F, 0x55 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000056, 0x00001C62, { { 0x43, 0x28, 0x3C, 0x0F, 0x78, 0x52, 0xE7, 0x2A, 0x77, 0xF3, 0x21, 0x5A, 0xF0, 0xFC, 0x9E, 0xF8 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000063, 0x00002184, { { 0x6B, 0xDC, 0x6B, 0xCF, 0xD4, 0xC7, 0x2A, 0x9A, 0x2E, 0x34, 0x71, 0x4E, 0xB7, 0xF6, 0xAF, 0xDA } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x00000075, 0x00002B94, { { 0xA5, 0x4C, 0xBF, 0x5F, 0xD3, 0xF8, 0x6D, 0xE4, 0xB4, 0x8E, 0x64, 0x84, 0xD9, 0xBA, 0x2B, 0x33 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000067, 0x00002C6B, { { 0x55, 0x56, 0x6B, 0xEA, 0xEF, 0x34, 0xDA, 0x23, 0x23, 0xF4, 0x9D, 0x0F, 0xBF, 0xC3, 0x1F, 0xDA } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000037, 0x00001EDF, { { 0xA8, 0x42, 0x79, 0xEB, 0x82, 0xAF, 0x03, 0xD7, 0x03, 0xF9, 0xE0, 0x60, 0xD4, 0x87, 0xAD, 0x2E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1MagicJewelStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000013, 0x0000069E, { { 0x7C, 0xF2, 0xEE, 0x7C, 0x8A, 0xCE, 0x5B, 0x09, 0x32, 0xDC, 0x31, 0x1F, 0x45, 0x21, 0x84, 0xFF } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000000F, 0x0000087E, { { 0xD8, 0xF1, 0x40, 0x9D, 0x9C, 0x15, 0x9E, 0xBD, 0x69, 0xE5, 0xE1, 0x51, 0x34, 0x22, 0xF3, 0x75 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000011, 0x000005DB, { { 0x44, 0x99, 0x35, 0x77, 0x9B, 0x3D, 0xF8, 0xAF, 0x7A, 0xA6, 0x14, 0xD0, 0x99, 0xC6, 0x80, 0x7B } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000014, 0x000006EF, { { 0xDE, 0x77, 0xB6, 0x8C, 0x39, 0x5C, 0x6F, 0xDF, 0x7D, 0x04, 0x0F, 0x65, 0x24, 0x75, 0xED, 0x95 } } } },

	{ IT_ITA, kPlatformPC, { 0x0000001D, 0x00000A83, { { 0x60, 0x69, 0x18, 0x6C, 0x34, 0xB0, 0x56, 0xE3, 0x06, 0x7C, 0xB1, 0xDD, 0x26, 0x80, 0xAE, 0x30 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000011, 0x000005E4, { { 0xD5, 0x11, 0xDA, 0x40, 0x09, 0x6A, 0x5D, 0x76, 0x5C, 0x20, 0xC4, 0x43, 0x4D, 0xF3, 0x67, 0xF0 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000011, 0x000005CD, { { 0x32, 0x2A, 0xFF, 0x9F, 0x10, 0x75, 0x6B, 0xD6, 0x46, 0xAE, 0x55, 0xD3, 0x68, 0x4F, 0xBB, 0x5A } } } },

	{ RU_RUS, kPlatformPC, { 0x00000012, 0x0000047D, { { 0xB1, 0xC3, 0x66, 0xBC, 0x42, 0xAD, 0x5B, 0xD8, 0xF5, 0x3D, 0xB9, 0x50, 0x77, 0x32, 0xA7, 0x15 } } } },

	{ EN_ANY, kPlatformUnknown, { 0x00000014, 0x0000069E, { { 0x6A, 0x1C, 0x9B, 0x85, 0x61, 0xC7, 0x28, 0xA9, 0xA3, 0xF4, 0xFA, 0x47, 0x90, 0x8F, 0x06, 0xB4 } } } }, // Amiga + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1ThePoisonStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000057, 0x00001C24, { { 0xBC, 0x31, 0x5C, 0x25, 0x50, 0x36, 0x58, 0x20, 0x55, 0xFC, 0x75, 0x47, 0x2D, 0x43, 0x73, 0x78 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000068, 0x000022D8, { { 0x39, 0xDA, 0xB3, 0xD2, 0xDA, 0x3F, 0xAB, 0x40, 0x1A, 0x4F, 0x9D, 0x02, 0xBA, 0x37, 0xEC, 0x4D } } } },

	{ DE_DEU, kPlatformPC, { 0x00000072, 0x00002690, { { 0xEF, 0x62, 0x85, 0xA3, 0x8B, 0x20, 0x4B, 0x65, 0xF8, 0xE7, 0x2C, 0x02, 0x3F, 0x3F, 0x25, 0x65 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000063, 0x000021F8, { { 0x3C, 0x2E, 0xF8, 0xAD, 0xCF, 0xB5, 0xE7, 0x2C, 0x98, 0x7A, 0xB9, 0x6F, 0x7A, 0x42, 0xE1, 0x92 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000068, 0x00002077, { { 0x5D, 0xF4, 0xF9, 0xD9, 0x87, 0x34, 0x14, 0xAB, 0xE4, 0x9A, 0x67, 0xAF, 0x18, 0x8B, 0x40, 0x98 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000059, 0x00001DF7, { { 0x16, 0x7B, 0x5F, 0x91, 0x06, 0x5B, 0xFC, 0x9C, 0x88, 0x61, 0xCC, 0x1B, 0x52, 0x4F, 0x91, 0xC5 } } } },

	{ RU_RUS, kPlatformPC, { 0x00000052, 0x0000136F, { { 0xEF, 0xD2, 0xA0, 0x5F, 0xD5, 0xE6, 0x77, 0x96, 0xFA, 0xC5, 0x60, 0x7C, 0xB7, 0xA8, 0x7C, 0x7A } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000058, 0x00001C24, { { 0xBA, 0x1F, 0xBD, 0x5C, 0x85, 0x3D, 0x3C, 0x92, 0xD1, 0x13, 0xF3, 0x40, 0x2E, 0xBB, 0x1C, 0xE2 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000073, 0x00002690, { { 0x44, 0xAE, 0xC9, 0xFD, 0x9F, 0x8E, 0x1B, 0xDD, 0x3F, 0xE4, 0x4D, 0x4B, 0x5A, 0x13, 0xE5, 0x99 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x00000092, 0x0000334E, { { 0xC9, 0x02, 0xB5, 0xEF, 0x90, 0xA8, 0x9C, 0x6E, 0xAD, 0xD6, 0x1F, 0xCC, 0x71, 0x9F, 0x19, 0x2C } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x000000A0, 0x00003EEB, { { 0x4B, 0x0C, 0xD5, 0xF0, 0xFE, 0x05, 0xF5, 0x09, 0x96, 0xA9, 0xE4, 0xDD, 0x86, 0x8D, 0x79, 0x3C } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000045, 0x00002350, { { 0xAD, 0x17, 0xA0, 0x40, 0x86, 0x0E, 0x1D, 0x2F, 0xA7, 0x14, 0x5B, 0x81, 0x38, 0x09, 0x80, 0x21 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1FluteStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000003C, 0x00001599, { { 0x96, 0x72, 0x5A, 0x8A, 0xA0, 0xEE, 0xA2, 0xCE, 0x4D, 0x21, 0x01, 0x6C, 0xC5, 0x1A, 0xEB, 0x21 } } } }, // floppy + AMIGA
	{ EN_ANY, kPlatformPC, { 0x0000009C, 0x00002334, { { 0xA8, 0xA3, 0x1F, 0x0D, 0x36, 0x25, 0x19, 0x98, 0x50, 0xA0, 0x13, 0x65, 0xF4, 0xE8, 0x27, 0x19 } } } },      // CD

	{ FR_FRA, kPlatformPC, { 0x00000045, 0x000018EE, { { 0x3E, 0x3B, 0x20, 0x46, 0xEB, 0xB0, 0x88, 0x26, 0xB2, 0x0D, 0xD6, 0x8F, 0xF7, 0x53, 0x74, 0xAE } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000040, 0x000016F2, { { 0x70, 0xFF, 0x86, 0x73, 0x9E, 0x75, 0xDD, 0x17, 0x27, 0x86, 0x44, 0xE9, 0x78, 0x3D, 0x48, 0xDB } } } },

	{ IT_ITA, kPlatformPC, { 0x0000004C, 0x00001BFB, { { 0x06, 0x4F, 0x6A, 0x37, 0x8B, 0x44, 0x27, 0x26, 0xCE, 0x28, 0xB2, 0x47, 0x71, 0xE2, 0x27, 0x73 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000045, 0x00001726, { { 0x5A, 0xCF, 0x64, 0x4B, 0x61, 0x57, 0xEC, 0xB3, 0xBF, 0x32, 0x45, 0x51, 0x5D, 0x02, 0xC5, 0xB6 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000052, 0x00001D8E, { { 0x9D, 0xA5, 0xF1, 0x42, 0xD1, 0x48, 0xEB, 0x8F, 0x4B, 0xDC, 0xD9, 0x10, 0x55, 0xBD, 0x12, 0xBB } } } },

	// not translated in the fan translation
	{ RU_RUS, kPlatformPC, { 0x0000003C, 0x00001599, { { 0x96, 0x72, 0x5A, 0x8A, 0xA0, 0xEE, 0xA2, 0xCE, 0x4D, 0x21, 0x01, 0x6C, 0xC5, 0x1A, 0xEB, 0x21 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000005A, 0x000024F9, { { 0xCA, 0x1F, 0x62, 0x23, 0x22, 0x25, 0x4A, 0x94, 0x8A, 0x50, 0x59, 0xD5, 0xB4, 0x4E, 0xF1, 0xA6 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000053, 0x00002745, { { 0x7A, 0xBB, 0xFC, 0x30, 0xB6, 0xCE, 0x61, 0xD4, 0xDB, 0xB0, 0xE6, 0xB2, 0xF4, 0x4D, 0x81, 0x35 } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000032, 0x00001CDF, { { 0xB7, 0x08, 0x8D, 0x67, 0xB2, 0x0F, 0x59, 0x2D, 0xA6, 0x36, 0x21, 0x7B, 0x8B, 0x89, 0xE3, 0xED } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1FlaskFullStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000001C, 0x000009C8, { { 0xA5, 0xC7, 0x5E, 0x6E, 0x02, 0xEE, 0x8C, 0xF8, 0xF0, 0xBF, 0x69, 0xB8, 0x05, 0x1E, 0xF5, 0x28 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000001F, 0x00001135, { { 0x90, 0x52, 0x4A, 0x95, 0xE4, 0x89, 0xD6, 0x0A, 0xE1, 0x2E, 0x98, 0x11, 0x02, 0xF5, 0x79, 0x37 } } } },

	{ FR_FRA, kPlatformPC, { 0x0000001E, 0x00000AC7, { { 0x4E, 0x7A, 0xC8, 0xC5, 0xD4, 0xB8, 0xD0, 0x73, 0x66, 0x2B, 0x2A, 0x13, 0xDE, 0xE1, 0x32, 0xAA } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000001E, 0x00000A71, { { 0x11, 0x88, 0x37, 0x8E, 0x17, 0xF2, 0x82, 0x1E, 0x72, 0xF0, 0xA3, 0x2B, 0x4F, 0x76, 0x5F, 0xBA } } } },

	{ IT_ITA, kPlatformPC, { 0x00000021, 0x00000C19, { { 0xC0, 0xFA, 0xD2, 0xB8, 0xCA, 0x94, 0x67, 0x10, 0x65, 0x01, 0x7F, 0xB0, 0x95, 0xC6, 0x0E, 0xFF } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000001E, 0x00000973, { { 0xD3, 0x93, 0x3C, 0xA6, 0xF7, 0x79, 0xE6, 0x05, 0x49, 0x88, 0x89, 0xDE, 0xCA, 0x77, 0x2F, 0x1B } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000001B, 0x0000099D, { { 0x13, 0x23, 0x5D, 0x38, 0x9B, 0xFB, 0x00, 0x5C, 0xA1, 0x3A, 0x22, 0xD6, 0xCD, 0x5C, 0x09, 0xAE } } } },

	{ RU_RUS, kPlatformPC, { 0x0000001A, 0x0000066E, { { 0x36, 0x43, 0xB6, 0xB2, 0xED, 0xBA, 0x21, 0x0C, 0x16, 0x54, 0x99, 0xF9, 0x2E, 0x6E, 0x0A, 0x28 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1FullFlaskStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000097, 0x00003521, { { 0x9C, 0x30, 0xC7, 0xC9, 0x97, 0xF3, 0x0E, 0xC4, 0x6E, 0xCA, 0x63, 0xC6, 0xD8, 0x06, 0xFF, 0x30 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000086, 0x00002E16, { { 0x94, 0x6C, 0x6D, 0xB6, 0x4C, 0x84, 0xBB, 0xF6, 0xC8, 0x50, 0x4C, 0x00, 0x38, 0xE3, 0x09, 0x5B } } } },

	{ DE_DEU, kPlatformPC, { 0x000000AF, 0x00003E38, { { 0x66, 0x86, 0xAF, 0x1A, 0x2D, 0x33, 0x40, 0xE7, 0x96, 0x9B, 0x32, 0x26, 0xFF, 0x9F, 0x79, 0x8B } } } },

	{ IT_ITA, kPlatformPC, { 0x00000086, 0x00002CDB, { { 0xB1, 0xAE, 0x2A, 0x03, 0x28, 0xEC, 0x6D, 0x2F, 0xED, 0x71, 0xBE, 0x60, 0xCF, 0xAB, 0x8C, 0x00 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x000000B5, 0x00004111, { { 0xA3, 0x4D, 0xA0, 0xF5, 0xE5, 0xD7, 0xFE, 0x11, 0xBB, 0x2C, 0xB5, 0xF2, 0xE0, 0x52, 0xF1, 0xF0 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000009A, 0x0000363B, { { 0x38, 0x25, 0xE6, 0xB5, 0xCB, 0x78, 0x5E, 0xAD, 0x2D, 0xD4, 0x2E, 0x8B, 0x89, 0x20, 0xB1, 0x95 } } } },

	{ RU_RUS, kPlatformPC, { 0x00000094, 0x0000232B, { { 0xBF, 0x68, 0xF9, 0x8F, 0x82, 0xE9, 0xE7, 0x69, 0x33, 0xD6, 0x41, 0x15, 0x2C, 0xFE, 0x72, 0xAB } } } },

	{ EN_ANY, kPlatformAmiga, { 0x0000009A, 0x00003521, { { 0x26, 0xE5, 0xC8, 0x6D, 0x14, 0x81, 0x9F, 0x90, 0x38, 0x3C, 0x00, 0x9D, 0x8E, 0x72, 0xB1, 0x83 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x000000B0, 0x00003E38, { { 0x8A, 0x6D, 0x42, 0x36, 0x29, 0x06, 0xB2, 0xCE, 0xA3, 0x41, 0x14, 0xE8, 0xB1, 0xEF, 0x6E, 0x3B } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000009D, 0x00003521, { { 0xAB, 0x74, 0x71, 0x5C, 0x6E, 0x65, 0xF5, 0x5E, 0xB0, 0x6E, 0x0E, 0xC7, 0x56, 0xCB, 0x55, 0x98 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000007D, 0x000039CE, { { 0x9E, 0xBE, 0x13, 0x7F, 0x4D, 0x4B, 0xCB, 0x08, 0xC3, 0xE5, 0xF1, 0xC2, 0xD3, 0x69, 0x9E, 0xDF } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000079, 0x000039CE, { { 0xEF, 0x3F, 0x36, 0xD4, 0x6C, 0x64, 0x74, 0x9A, 0xFE, 0x13, 0xB5, 0x77, 0x9A, 0xCF, 0x15, 0x5C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1VeryCleverStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000032, 0x0000118D, { { 0x4B, 0x6D, 0xD4, 0xDC, 0x3E, 0xA2, 0x2D, 0x6D, 0x2C, 0x5A, 0xF7, 0x67, 0x4B, 0x6D, 0x40, 0xEF } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000002F, 0x00001BDF, { { 0x32, 0x2C, 0x60, 0x1E, 0xB7, 0xE0, 0xFE, 0x5F, 0xEA, 0xEF, 0xA4, 0x73, 0xAC, 0xCB, 0xBA, 0xFE } } } },

	{ FR_FRA, kPlatformPC, { 0x00000027, 0x00000DE1, { { 0x43, 0xC2, 0xC7, 0xD2, 0xA3, 0x70, 0x36, 0x38, 0xB5, 0x5B, 0x22, 0xEF, 0xBA, 0x0D, 0x37, 0x27 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000043, 0x00001871, { { 0x33, 0x3F, 0xEA, 0x7F, 0x5F, 0x56, 0x98, 0xF3, 0x03, 0x8D, 0x3E, 0xAC, 0xA1, 0x62, 0x2A, 0xD7 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000038, 0x000013CF, { { 0x58, 0x7B, 0x59, 0x50, 0x84, 0xBB, 0x5B, 0x70, 0xDF, 0x76, 0x2F, 0xC3, 0x7E, 0xEC, 0x6A, 0x8B } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000027, 0x00000D01, { { 0x72, 0x7E, 0x88, 0xFB, 0xDA, 0xC0, 0x9E, 0x31, 0xAC, 0x32, 0xFD, 0x90, 0x7D, 0x01, 0x86, 0xD0 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000036, 0x000013F8, { { 0x2D, 0x9B, 0x7D, 0x58, 0xD1, 0x94, 0x04, 0x45, 0x6E, 0x81, 0xCC, 0x1E, 0x2F, 0xC5, 0xC9, 0xEA } } } },

	// not translated in the fan translation
	{ RU_RUS, kPlatformPC, { 0x00000032, 0x0000118D, { { 0x4B, 0x6D, 0xD4, 0xDC, 0x3E, 0xA2, 0x2D, 0x6D, 0x2C, 0x5A, 0xF7, 0x67, 0x4B, 0x6D, 0x40, 0xEF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1NewGameStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000015, 0x0000052B, { { 0xAB, 0xD2, 0x16, 0x26, 0xC2, 0x86, 0xFA, 0xC8, 0x42, 0xCD, 0x16, 0xCD, 0x25, 0xB7, 0x44, 0xDC } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000001B, 0x00000EC8, { { 0x13, 0x9A, 0xBC, 0x8F, 0xE2, 0x4B, 0xD7, 0x0B, 0xC0, 0x81, 0x60, 0x10, 0xC2, 0xA6, 0x9C, 0xFA } } } },

	{ FR_FRA, kPlatformPC, { 0x0000001D, 0x0000079B, { { 0x2B, 0x5B, 0x54, 0x29, 0x76, 0x31, 0x5E, 0x89, 0xC2, 0xD1, 0xDB, 0x3B, 0xF0, 0xBC, 0xCD, 0x66 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000001D, 0x0000076F, { { 0x5C, 0x84, 0x2A, 0xB1, 0x61, 0xDD, 0x7B, 0xB3, 0xD4, 0x6A, 0xD2, 0xEF, 0xA4, 0x5F, 0x81, 0xFE } } } },

	{ IT_ITA, kPlatformPC, { 0x0000001C, 0x0000075E, { { 0xC8, 0xCB, 0x35, 0x5E, 0x73, 0x09, 0xDB, 0xA1, 0xF1, 0x8D, 0x14, 0x54, 0x8D, 0xF7, 0xB1, 0xD3 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000001D, 0x000007BD, { { 0xF4, 0x5C, 0x50, 0x82, 0x91, 0x7E, 0x09, 0x52, 0x44, 0x95, 0xC2, 0x2E, 0xC9, 0x52, 0x12, 0x2B } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000001B, 0x00000701, { { 0x2B, 0x87, 0xC3, 0x82, 0x68, 0xA5, 0xFC, 0xC5, 0x64, 0x9E, 0xAB, 0xD2, 0x8A, 0x07, 0x9C, 0x1E } } } },

	{ RU_RUS, kPlatformPC, { 0x00000015, 0x0000035F, { { 0x7E, 0x49, 0xC1, 0xCB, 0x2D, 0x61, 0xA7, 0x4C, 0x20, 0xAC, 0xEC, 0x54, 0x80, 0x14, 0x6A, 0xCA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1DefaultShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000048A, 0x00012C0A, { { 0x90, 0x9E, 0x21, 0x9F, 0x90, 0x16, 0x1D, 0xA5, 0x57, 0xF8, 0x71, 0xE1, 0x15, 0x44, 0x29, 0x73 } } } },

	// TODO: Where did this come from?
	/*{ UNK_LANG, kPlatformUnknown, { 0x000004F3, 0x00014AE4, { { 0x6E, 0x37, 0x81, 0x15, 0x4D, 0xDD, 0xBA, 0xA6, 0xE3, 0x5F, 0xB6, 0xC8, 0x20, 0xE9, 0xE0, 0x63 } } } },*/

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Healing1ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000009A, 0x00002F17, { { 0x92, 0x2C, 0xF1, 0x5C, 0x8B, 0xCE, 0xEB, 0x0B, 0xE4, 0x07, 0x16, 0xAA, 0x25, 0x4F, 0x83, 0xEB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Healing2ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000D2, 0x00004257, { { 0xAC, 0x35, 0xAA, 0x72, 0x50, 0x67, 0x58, 0xE4, 0x70, 0xB0, 0xBB, 0xCA, 0xA7, 0x4C, 0x91, 0x82 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PoisonDeathShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000008C, 0x00002E90, { { 0xBC, 0x44, 0xFB, 0x98, 0xE7, 0x42, 0xF6, 0xC8, 0x87, 0xDD, 0x00, 0x42, 0x85, 0xD8, 0x1E, 0x81 } } } },
	{ UNK_LANG, kPlatformUnknown, { 0x0000008C, 0x00002E7C, { { 0xA5, 0xD7, 0x13, 0xFC, 0x43, 0x22, 0x13, 0xBC, 0x5F, 0x3F, 0xC8, 0x28, 0xDA, 0x04, 0xB0, 0xDD } } } }, // Old Dos Floppy
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1FluteShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000FC, 0x00004F0F, { { 0xB1, 0xF0, 0x56, 0x6E, 0xAB, 0x3C, 0x88, 0x13, 0x97, 0x7C, 0xCC, 0x61, 0x85, 0x8A, 0x28, 0x12 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Winter1ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000031, 0x00000E70, { { 0x3A, 0x15, 0x36, 0x51, 0x2F, 0xF8, 0xAF, 0xE6, 0x41, 0x71, 0x8D, 0x44, 0x99, 0x48, 0xB3, 0x4F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Winter2ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000F5, 0x00005192, { { 0xD3, 0x64, 0xAC, 0x64, 0x42, 0x9F, 0xB9, 0xCA, 0x0C, 0xC8, 0x97, 0x67, 0x8C, 0x92, 0xB4, 0x79 } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x000000F5, 0x000029E4, { { 0xF0, 0x7B, 0x05, 0xD8, 0xC1, 0x97, 0x27, 0x17, 0xFA, 0x32, 0xE9, 0x1A, 0x46, 0xA5, 0xEB, 0x84 } } } }, // CD + Amiga

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Winter3ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001C, 0x00000A5E, { { 0x84, 0xDD, 0xC0, 0xEA, 0xE3, 0xBE, 0x98, 0x66, 0xE0, 0xA3, 0x4D, 0x00, 0x88, 0x31, 0x61, 0x1F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1DrinkShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000069, 0x00001EDA, { { 0x21, 0xFA, 0x2F, 0x41, 0x8F, 0x70, 0x3B, 0xBB, 0x93, 0x35, 0x49, 0xEC, 0x1C, 0xEE, 0xDD, 0xAC } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x0000006A, 0x00001EDA, { { 0xE2, 0xDA, 0xEC, 0x87, 0x4A, 0x27, 0xD8, 0xC1, 0x57, 0xE7, 0xD8, 0xD7, 0x87, 0x5F, 0x3A, 0xEB } } } },

	{ UNK_LANG, kPlatformPC98, { 0x0000006A, 0x00001EDA, { { 0xE2, 0xDA, 0xEC, 0x87, 0x4A, 0x27, 0xD8, 0xC1, 0x57, 0xE7, 0xD8, 0xD7, 0x87, 0x5F, 0x3A, 0xEB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1WispShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B6, 0x00003A53, { { 0x34, 0x52, 0x9D, 0x06, 0x4F, 0x1B, 0xCD, 0xB7, 0x30, 0xF7, 0xB4, 0x9F, 0x0F, 0x34, 0xDD, 0xF7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1MagicAnimShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000023, 0x00000A2D, { { 0x48, 0x8E, 0x2A, 0x61, 0x93, 0xCD, 0x58, 0x3B, 0x45, 0x62, 0xD9, 0x01, 0x8F, 0xBA, 0x2A, 0x1A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1BranStoneShapesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000006E, 0x00001E41, { { 0x77, 0x99, 0x54, 0xED, 0x4C, 0x31, 0x08, 0x50, 0x41, 0x22, 0x6B, 0xED, 0x34, 0xF4, 0x17, 0xA1 } } } },

	{ UNK_LANG, kPlatformAmiga, { 0x0000006E, 0x00002026, { { 0x5A, 0xAF, 0x28, 0xF3, 0x0F, 0x82, 0x05, 0xB4, 0x33, 0x83, 0xCE, 0x56, 0xDB, 0xE7, 0x5C, 0xB4 } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x00000063, 0x00001E41, { { 0xDA, 0xA1, 0x93, 0x68, 0xBF, 0xB3, 0xBB, 0xD1, 0x2E, 0xF9, 0xB2, 0x25, 0x72, 0xD9, 0xA6, 0x00 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x00000063, 0x00001E41, { { 0xDA, 0xA1, 0x93, 0x68, 0xBF, 0xB3, 0xBB, 0xD1, 0x2E, 0xF9, 0xB2, 0x25, 0x72, 0xD9, 0xA6, 0x00 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x0000024B, { { 0x6C, 0x6C, 0xE9, 0xB6, 0xC4, 0xC2, 0x29, 0xA5, 0xB6, 0xB3, 0x82, 0xB6, 0x03, 0x5B, 0x66, 0x7A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x00000470, { { 0x63, 0x02, 0x97, 0x7D, 0xBE, 0x97, 0x5E, 0x88, 0x2E, 0x50, 0x18, 0x6A, 0x4A, 0x61, 0x09, 0xA6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette3Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x0000042A, { { 0xF8, 0x32, 0x65, 0x55, 0x1C, 0xC5, 0x6E, 0x71, 0xC6, 0x6A, 0x8E, 0xC5, 0x3C, 0xBC, 0x1F, 0xA8 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette4Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x000003A4, { { 0xA1, 0xA6, 0x6F, 0x53, 0xA2, 0x1E, 0x27, 0x6F, 0x0D, 0xD0, 0x4F, 0xDC, 0x4C, 0xDC, 0xAE, 0x17 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette5Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x0000033D, { { 0x22, 0x02, 0xB3, 0xCD, 0x4E, 0xD5, 0x6E, 0x65, 0x2B, 0x38, 0x7D, 0x3E, 0x27, 0x03, 0x1A, 0x59 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette6Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x000001EA, { { 0xC7, 0xA7, 0x0E, 0xBA, 0x3F, 0x5E, 0xD3, 0x02, 0x99, 0x3B, 0x32, 0x00, 0x39, 0x3C, 0xD2, 0xAC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette7Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000036, 0x0000040C, { { 0x1A, 0x8E, 0x3E, 0x5F, 0x09, 0xFE, 0x4A, 0xF1, 0x5D, 0x2E, 0x2C, 0xD3, 0xFF, 0x75, 0x7D, 0x1C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette8Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000036, 0x000008E4, { { 0x55, 0xB4, 0xD3, 0xD1, 0x89, 0x49, 0xA5, 0x7E, 0x40, 0xF9, 0xFE, 0x6E, 0x16, 0xE8, 0xBD, 0xBB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette9Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x000001DB, { { 0x7A, 0x04, 0x16, 0xDD, 0x6A, 0xBA, 0x51, 0x9E, 0x24, 0x9E, 0xEA, 0x76, 0xD0, 0xFA, 0x92, 0xF4 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette10Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x0000017A, { { 0xD2, 0x20, 0x1F, 0x12, 0xE1, 0x65, 0x68, 0xA7, 0x9C, 0x69, 0x8E, 0x99, 0xDF, 0xC5, 0xD2, 0x8E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette11Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x00000136, { { 0xBF, 0x38, 0xB1, 0xFE, 0xBF, 0x25, 0xBC, 0xD3, 0xCC, 0x60, 0x40, 0x97, 0xB6, 0x4A, 0xC7, 0xB2 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette12Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x00000102, { { 0x1C, 0x1E, 0x6A, 0x30, 0xE1, 0xD1, 0xF9, 0x3D, 0x9A, 0xF1, 0x84, 0xBD, 0x44, 0x07, 0x51, 0xEA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette13Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x00000000, { { 0x76, 0x22, 0x21, 0x4B, 0x85, 0x36, 0xAF, 0xE7, 0xB8, 0x9B, 0x1C, 0x66, 0x06, 0x06, 0x9B, 0x0D } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette14Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x000003BF, { { 0x78, 0xE8, 0xA8, 0xF2, 0x4D, 0x54, 0x48, 0x47, 0xC9, 0xC1, 0x5B, 0x6C, 0x69, 0xCB, 0x65, 0xB6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette15Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x000002DD, { { 0x9F, 0x66, 0x50, 0xAC, 0xED, 0x0C, 0x63, 0x20, 0xE9, 0x4E, 0x4D, 0x3D, 0xC4, 0xBC, 0x94, 0x8A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette16Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x0000036E, { { 0xA9, 0xDB, 0x1C, 0x35, 0xED, 0xED, 0xE4, 0x40, 0x68, 0x29, 0xB1, 0x6E, 0x60, 0xE4, 0xB0, 0xDB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette17Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x000004D7, { { 0xC0, 0xB5, 0x7B, 0x94, 0xED, 0xD7, 0x48, 0x66, 0x37, 0x54, 0xE8, 0xFC, 0x0F, 0xA5, 0x54, 0x8A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette18Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x000002FD, { { 0x0A, 0xDD, 0x81, 0xA8, 0x89, 0xD2, 0x53, 0x05, 0x25, 0xCC, 0x2B, 0x8B, 0x48, 0x1D, 0xAB, 0x0A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette19Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x00000134, { { 0xCB, 0x0F, 0xA9, 0xD2, 0xD8, 0xEA, 0x2A, 0x1F, 0x16, 0x7B, 0x90, 0xE5, 0x72, 0xB3, 0x52, 0x50 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette20Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x00000396, { { 0x3B, 0xD8, 0xA5, 0xC5, 0x66, 0x3E, 0xC7, 0x88, 0xD0, 0x3B, 0x42, 0x29, 0x84, 0x58, 0xA7, 0x00 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette21Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x00000134, { { 0x96, 0x70, 0x7D, 0x5B, 0xC4, 0xBA, 0x89, 0x89, 0xC9, 0x42, 0xF4, 0x70, 0x33, 0xF5, 0xE8, 0x02 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette22Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x00000376, { { 0xA2, 0x92, 0x28, 0x65, 0xAB, 0xCC, 0x16, 0xE8, 0xA5, 0x72, 0x2F, 0x00, 0x66, 0x8F, 0x8D, 0x96 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette23Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x000002C2, { { 0x5C, 0x78, 0x74, 0x81, 0x76, 0xB5, 0x26, 0x8E, 0x20, 0xAE, 0x50, 0xCD, 0x2B, 0x9E, 0x81, 0x8B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette24Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000472, { { 0xC6, 0xF0, 0xA6, 0x33, 0x7E, 0xAE, 0xBC, 0xF9, 0xAE, 0xE2, 0xF4, 0x7B, 0x71, 0x44, 0x50, 0xEF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette25Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x000004CD, { { 0x57, 0x03, 0xB5, 0x27, 0x64, 0x48, 0xBF, 0x6A, 0x0F, 0xC2, 0xC0, 0x4A, 0x73, 0x9A, 0xFE, 0xB3 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette26Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x000004CA, { { 0xF8, 0x39, 0x8D, 0x6F, 0x8D, 0x9C, 0x6E, 0x0A, 0x33, 0xE8, 0xF2, 0xA6, 0xF6, 0x40, 0x70, 0xE5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette27Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x000004D9, { { 0xF6, 0xD4, 0x88, 0x57, 0x6F, 0x1A, 0xA3, 0x1F, 0x6B, 0x17, 0x4D, 0xA0, 0x9E, 0xD8, 0x09, 0x66 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette28Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x0000051D, { { 0x59, 0x02, 0xE3, 0xC6, 0x28, 0xF6, 0xBC, 0xD4, 0xC8, 0x7E, 0x58, 0x6F, 0xE4, 0xA2, 0x28, 0x4F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette29Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000534, { { 0xC2, 0x26, 0x94, 0x47, 0x28, 0x11, 0xAA, 0xE6, 0x10, 0xA2, 0xD4, 0xD9, 0x61, 0x57, 0x62, 0x22 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette30Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000453, { { 0xE1, 0x30, 0xC2, 0xCD, 0x79, 0xD8, 0x36, 0x8D, 0xAC, 0xF7, 0x87, 0x05, 0x49, 0xD7, 0x7A, 0x5F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette31Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000027, 0x000002FB, { { 0xA6, 0x65, 0xE1, 0xE1, 0xCC, 0x71, 0x83, 0x63, 0x9A, 0x04, 0xE2, 0x00, 0xBF, 0x90, 0xDF, 0x95 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette32Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000027, 0x0000016C, { { 0xAA, 0x88, 0x97, 0xE3, 0xCA, 0x06, 0x11, 0x07, 0x8F, 0xFA, 0x75, 0x26, 0xA8, 0x7A, 0x49, 0x28 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette33Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x0000094C, { { 0x2E, 0x5E, 0x34, 0x3E, 0x99, 0x61, 0x6F, 0x88, 0x91, 0xA7, 0x75, 0xAA, 0x42, 0x4B, 0x1F, 0x5D } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1GUIStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000001B0, 0x00009562, { { 0x78, 0xB0, 0x81, 0x15, 0x96, 0x09, 0x1E, 0x6C, 0x20, 0x7B, 0xF7, 0xB1, 0x29, 0x2A, 0x3D, 0xE4 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000209, 0x00009788, { { 0xDD, 0xA0, 0x5C, 0x30, 0x60, 0x68, 0xDC, 0x24, 0xBE, 0x3F, 0xA2, 0x31, 0xE0, 0x81, 0x6C, 0xCA } } } }, // CD

	{ FR_FRA, kPlatformPC, { 0x00000234, 0x0000C262, { { 0x08, 0x83, 0x32, 0x4E, 0x48, 0xEC, 0x83, 0x4B, 0x1B, 0x82, 0x58, 0x48, 0xA3, 0x30, 0x05, 0x33 } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x0000025D, 0x0000CBDC, { { 0x36, 0xF0, 0x92, 0x9C, 0x5F, 0xFF, 0x5D, 0x65, 0x5F, 0x0B, 0xC0, 0x72, 0x9F, 0x30, 0x47, 0xAB } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x00000236, 0x0000BA9C, { { 0xEB, 0xA6, 0x90, 0x9D, 0x99, 0x4B, 0x4F, 0xD9, 0xCE, 0xFF, 0x44, 0x3F, 0x77, 0x6F, 0xE1, 0x71 } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000261, 0x0000C538, { { 0xF3, 0xED, 0x5E, 0x73, 0x6F, 0x6E, 0x06, 0xF8, 0x48, 0xF8, 0x30, 0x69, 0x34, 0x5D, 0x99, 0x0C } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x00000225, 0x0000B0E1, { { 0x57, 0xE2, 0x39, 0xA8, 0xF5, 0x70, 0x48, 0xB4, 0x3D, 0xD4, 0x2A, 0x8C, 0x7C, 0x76, 0xB7, 0x8D } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000025D, 0x0000AFD0, { { 0x8E, 0x97, 0xA9, 0x55, 0x16, 0xF7, 0x42, 0x83, 0xA3, 0x68, 0x2F, 0xD6, 0x37, 0x1C, 0x9A, 0xD1 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000023A, 0x0000C3BD, { { 0xED, 0x0D, 0xE7, 0x5B, 0xDC, 0x21, 0x41, 0x54, 0x68, 0x7D, 0x8E, 0x97, 0x1A, 0xB1, 0xA1, 0x4A } } } }, // floppy

	{ RU_RUS, kPlatformPC, { 0x000001B1, 0x000065E8, { { 0x91, 0x22, 0x61, 0x8B, 0xCD, 0x7C, 0x0E, 0xD4, 0x32, 0x00, 0xC3, 0x6E, 0x50, 0x7F, 0x3C, 0x82 } } } }, // floppy

	{ EN_ANY, kPlatformAmiga, { 0x000001DF, 0x00009042, { { 0x0D, 0xD3, 0x1A, 0x92, 0x8D, 0x9C, 0x72, 0x55, 0xEF, 0xFB, 0x81, 0x21, 0x3B, 0x43, 0xA7, 0xE8 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000237, 0x0000BAF7, { { 0xD7, 0x1A, 0x8E, 0xCC, 0x6D, 0x3E, 0xA9, 0xDD, 0x9A, 0x6B, 0x71, 0xFE, 0xD4, 0x50, 0x30, 0x6E } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x000001FC, 0x00009D9A, { { 0xE8, 0x39, 0x46, 0x0A, 0x41, 0xC3, 0xDC, 0x33, 0x6E, 0xC5, 0x5D, 0x99, 0xCC, 0xF6, 0xDD, 0xCC } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000025F, 0x0000E56A, { { 0x77, 0x9A, 0x02, 0x52, 0xED, 0x62, 0x23, 0x5A, 0x58, 0x78, 0x19, 0x52, 0x03, 0xD7, 0x88, 0x40 } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000213, 0x0000EC5A, { { 0x3B, 0xBA, 0xFB, 0x61, 0x34, 0x60, 0x25, 0xDA, 0x48, 0xBF, 0x95, 0x7F, 0x7F, 0xB6, 0xE4, 0x55 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1ConfigStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000003F, 0x000016E9, { { 0x4F, 0x19, 0x60, 0x67, 0xA8, 0x31, 0x0B, 0xD5, 0x3D, 0x06, 0x39, 0xF1, 0x42, 0xB0, 0xFD, 0x5C } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000061, 0x00002249, { { 0x5D, 0xE2, 0x05, 0xA7, 0xEC, 0x0A, 0x73, 0xF5, 0x01, 0x5B, 0x64, 0x71, 0x83, 0xEC, 0x56, 0x24 } } } }, // CD

	{ FR_FRA, kPlatformPC, { 0x0000004B, 0x00001AE8, { { 0xE3, 0xC9, 0x0F, 0x54, 0x48, 0x31, 0x6E, 0x5F, 0x51, 0x8E, 0xF8, 0xE6, 0xAC, 0x16, 0xAC, 0x1C } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x0000007A, 0x00002C37, { { 0x9E, 0xD8, 0x85, 0xB4, 0x9D, 0x32, 0x9D, 0x80, 0x58, 0xE4, 0x08, 0xC7, 0x75, 0xD2, 0x9C, 0x1E } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x0000005E, 0x00002280, { { 0xC7, 0xD2, 0x22, 0xB5, 0xD3, 0xBC, 0x7A, 0x97, 0x26, 0x04, 0x07, 0x78, 0xAA, 0xAB, 0x1D, 0x5C } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000082, 0x00002E94, { { 0xC5, 0xDC, 0x44, 0x6C, 0x4B, 0x3E, 0x4E, 0x27, 0x0C, 0xCC, 0x65, 0x6C, 0x20, 0x8D, 0x71, 0x07 } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x00000054, 0x00001E98, { { 0x10, 0x6B, 0x6B, 0x9B, 0x91, 0x05, 0xE9, 0x30, 0xE2, 0x75, 0xE2, 0x45, 0x79, 0x17, 0x73, 0xC9 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000007A, 0x00002904, { { 0xD3, 0xD2, 0x96, 0x6E, 0xE7, 0xE8, 0x64, 0x77, 0x73, 0xCD, 0xC2, 0x9E, 0x57, 0xB5, 0xD3, 0xCD } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000004A, 0x00001B7B, { { 0x6B, 0x69, 0x50, 0x92, 0x9B, 0x35, 0x58, 0xE1, 0xEA, 0xBF, 0x42, 0x0B, 0xEB, 0x88, 0x41, 0x8D } } } }, // floppy

	// not translated in the fan translation
	{ RU_RUS, kPlatformPC, { 0x0000003F, 0x00000B0D, { { 0x0E, 0x60, 0x0F, 0x4A, 0xA9, 0xF0, 0x1B, 0x76, 0xBB, 0x33, 0xB2, 0x4B, 0x5C, 0xB5, 0x4A, 0x97 } } } }, // floppy

	{ EN_ANY, kPlatformAmiga, { 0x0000002E, 0x00000FA1, { { 0x5E, 0xFF, 0xFF, 0x3D, 0xF8, 0x11, 0x6F, 0x3B, 0xC5, 0x39, 0x8F, 0x25, 0x8F, 0x0F, 0xE9, 0x2B } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000043, 0x00001783, { { 0xB2, 0x2B, 0xAB, 0x27, 0x06, 0x9A, 0x1E, 0x4B, 0xA7, 0xD3, 0xFF, 0xEB, 0xFD, 0x12, 0xDC, 0x94 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000004A, 0x000016E9, { { 0x37, 0x12, 0xAD, 0x4C, 0xB8, 0x2C, 0xF5, 0x75, 0x20, 0x74, 0xA0, 0xFD, 0x58, 0xB1, 0x71, 0x77 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000057, 0x00002262, { { 0xC5, 0x9A, 0x4A, 0xE3, 0xDF, 0xD3, 0x0E, 0x8E, 0xBB, 0xAF, 0x75, 0xD3, 0x67, 0xB1, 0xF6, 0x08 } } } },

	{ JA_JPN, kPlatformPC98, { 0x0000003F, 0x00001E9D, { { 0x8A, 0x3C, 0x78, 0x6B, 0x3F, 0x0A, 0x7D, 0x92, 0x16, 0x39, 0xEB, 0x1C, 0x84, 0x9E, 0x4F, 0x84 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TownsMusicFadeTableProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000B10, 0x000076DE, { { 0x9F, 0x08, 0x5B, 0xD6, 0x25, 0x7F, 0x11, 0x08, 0x87, 0x45, 0x92, 0xD3, 0xE5, 0xA8, 0x7C, 0x2F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TownsSFXwdTableProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00012608, 0x006717A1, { { 0x34, 0xDD, 0x2D, 0xA5, 0x14, 0x05, 0xEE, 0x2F, 0x93, 0x7C, 0x78, 0x4D, 0xCA, 0x13, 0xED, 0x93 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TownsSFXbtTableProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000100, 0x00007FFE, { { 0xEF, 0x3D, 0x9F, 0x31, 0xE7, 0x19, 0x1E, 0x4F, 0xF3, 0x66, 0x42, 0xC8, 0x0D, 0x89, 0x96, 0x20 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TownsCDATableProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x000002C4, 0x0000C5ED, { { 0x76, 0x86, 0x77, 0xF9, 0x85, 0x23, 0x32, 0x56, 0x3C, 0x06, 0x03, 0xCF, 0x0C, 0x49, 0x94, 0xB5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PC98StoryStringsProvider[] = {
	{ JA_JPN, kPlatformPC98, { 0x00000054, 0x00002FAD, { { 0xCB, 0x25, 0xD1, 0xFA, 0x65, 0xCE, 0xD5, 0xE6, 0xAD, 0xE3, 0x48, 0x25, 0xC8, 0x51, 0xC6, 0x12 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PC98IntroSfxProvider[] = {
	{ UNK_LANG, kPlatformPC98, { 0x00000300, 0x00017445, { { 0xC3, 0x54, 0x4A, 0x58, 0x00, 0x33, 0xEA, 0xF1, 0x20, 0xA8, 0x0C, 0xDE, 0x7A, 0x94, 0xAB, 0x5E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1CreditsStringsProvider[] = {
	{ EN_ANY, kPlatformAmiga, { 0x000004CC, 0x000182F7, { { 0x7D, 0x14, 0xA6, 0xEC, 0x12, 0x67, 0xAC, 0xA7, 0x7B, 0xF8, 0x44, 0x19, 0x52, 0xCC, 0x3A, 0x65 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x000004FB, 0x00019987, { { 0x8B, 0xF1, 0x61, 0xB9, 0x3E, 0xA4, 0x28, 0x9B, 0x45, 0x86, 0xD1, 0xF0, 0xC3, 0x35, 0x44, 0xA0 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000058C, 0x0001AD16, { { 0xFC, 0x1B, 0xE6, 0xEB, 0x03, 0x1C, 0x54, 0x93, 0x64, 0x53, 0xB1, 0x17, 0x9D, 0x95, 0x1B, 0x4A } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000058C, 0x0001AD16, { { 0xFC, 0x1B, 0xE6, 0xEB, 0x03, 0x1C, 0x54, 0x93, 0x64, 0x53, 0xB1, 0x17, 0x9D, 0x95, 0x1B, 0x4A } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000530, 0x00019571, { { 0x38, 0x4F, 0xE0, 0x74, 0x27, 0x21, 0xEC, 0x63, 0x41, 0x32, 0xB1, 0xD9, 0xA5, 0x65, 0x62, 0x71 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AmigaIntroSFXTableProvider[] = {
	{ UNK_LANG, kPlatformAmiga, { 0x00000140, 0x00000D82, { { 0x61, 0xF6, 0x8F, 0xE5, 0xCD, 0x15, 0x58, 0x9F, 0xDF, 0x0E, 0x22, 0xAA, 0x9A, 0xCF, 0xC4, 0xA6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AmigaGameSFXTableProvider[] = {
	{ UNK_LANG, kPlatformAmiga, { 0x000003C0, 0x00005505, { { 0x8D, 0x6D, 0x2C, 0x58, 0x50, 0xF7, 0xBF, 0x60, 0xC3, 0x15, 0x20, 0x23, 0x1F, 0x90, 0x3E, 0xD6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayPakFilesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000022, 0x000008E0, { { 0xB7, 0x3B, 0x51, 0x46, 0x78, 0x2D, 0x4D, 0x82, 0xD4, 0x30, 0xFC, 0x6E, 0xC4, 0x5B, 0x27, 0x3E } } } }, // DOS floppy + PC98

	{ UNK_LANG, kPlatformPC, { 0x00000046, 0x0000121A, { { 0x42, 0xC2, 0x5A, 0xDC, 0x27, 0x2D, 0xB4, 0x44, 0x85, 0x58, 0x0F, 0xB6, 0x6D, 0x76, 0x04, 0x4F } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000014, 0x000004FA, { { 0xBB, 0x4C, 0x16, 0xEB, 0x56, 0xEF, 0xAC, 0x68, 0x6B, 0x49, 0x96, 0x3A, 0x9F, 0x00, 0x75, 0xF6 } } } }, // demo
	{ UNK_LANG, kPlatformPC, { 0x0000000C, 0x00000308, { { 0xC0, 0xAC, 0x10, 0xF1, 0x12, 0xD1, 0x21, 0x92, 0xA1, 0x62, 0x5B, 0x6E, 0xCF, 0x88, 0x32, 0x7C } } } }, // Lands of Lore demo

	{ UNK_LANG, kPlatformFMTowns, { 0x00000033, 0x00000BBD, { { 0x92, 0x5B, 0xDA, 0xE3, 0x7C, 0x39, 0xC4, 0x75, 0x24, 0xBA, 0x67, 0x27, 0x6E, 0x21, 0xA3, 0xF7 } } } },

	// TODO: Where did this come from?
	//{ UNK_LANG, kPlatformUnknown, { 0x0000002E, 0x00000C09, { { 0x6A, 0x25, 0xF6, 0x03, 0xA6, 0x88, 0xA0, 0x53, 0xE1, 0xF4, 0xB9, 0x0C, 0x71, 0x17, 0x36, 0x6B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000008C8, 0x0002FDE3, { { 0x62, 0xD1, 0x6F, 0xBC, 0xEC, 0xE6, 0xCF, 0xE8, 0xD8, 0xE9, 0xDE, 0xFB, 0x09, 0xAF, 0x34, 0x92 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000916, 0x00031417, { { 0x3E, 0x15, 0xDA, 0xF4, 0x77, 0x44, 0x80, 0x47, 0xDB, 0x32, 0x0E, 0x6D, 0xCA, 0x32, 0x65, 0x1B } } } }, // CD
	{ EN_ANY, kPlatformPC, { 0x00000102, 0x00005BD6, { { 0xB3, 0x00, 0xE5, 0x39, 0x02, 0x3D, 0xBF, 0xDD, 0x54, 0x70, 0xEA, 0xC1, 0xCB, 0xAC, 0xAA, 0xF7 } } } }, // Lands of Lore demo

	{ DE_DEU, kPlatformPC, { 0x000009EC, 0x000370F9, { { 0x56, 0x94, 0xCC, 0xE5, 0x38, 0x19, 0xAD, 0xD1, 0xBB, 0x6A, 0x6E, 0xEC, 0xC5, 0x1B, 0x57, 0x7B } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000A3F, 0x000389F4, { { 0x51, 0xC1, 0x87, 0x16, 0x53, 0x8C, 0x68, 0x8E, 0x9B, 0x81, 0xA1, 0xD2, 0xCE, 0x5F, 0x83, 0x31 } } } }, // CD

	{ FR_FRA, kPlatformPC, { 0x000009C9, 0x00036324, { { 0x2C, 0xC2, 0xD9, 0xCF, 0x90, 0x2A, 0xDF, 0xE9, 0x85, 0x9E, 0x5D, 0xBB, 0x1D, 0x9A, 0x14, 0x69 } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x00000995, 0x000352D7, { { 0xED, 0x7C, 0x49, 0x7B, 0x1A, 0x2C, 0x73, 0x61, 0x73, 0xAF, 0x16, 0x89, 0x1E, 0x01, 0xE2, 0xAE } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x00000916, 0x0003188F, { { 0xDC, 0x46, 0x06, 0xE1, 0xB0, 0x66, 0xBC, 0x18, 0x2E, 0x6E, 0x9E, 0xC9, 0xA4, 0x14, 0x8D, 0x08 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x000008C8, 0x00030947, { { 0x7F, 0x75, 0x5F, 0x99, 0x94, 0xFE, 0xA1, 0xE6, 0xEF, 0xB8, 0x93, 0x71, 0x83, 0x1B, 0xAC, 0x4A } } } }, // (fan) CD

	{ RU_RUS, kPlatformPC, { 0x00000916, 0x00032C49, { { 0xEA, 0x5C, 0xE5, 0x06, 0x05, 0x5F, 0x36, 0xE8, 0x31, 0x3E, 0xBF, 0x74, 0x73, 0xFB, 0xAB, 0xFF } } } }, // (fan) CD - intro and outro strings haven't been translated in this fan translation
	{ RU_RUS, kPlatformPC, { 0x000008C8, 0x00028639, { { 0xF9, 0x1D, 0x6A, 0x85, 0x23, 0x5E, 0x2A, 0x64, 0xBC, 0x45, 0xB2, 0x48, 0x13, 0x49, 0xD4, 0xF7 } } } }, // (fan) floppy

	{ EN_ANY, kPlatformFMTowns, { 0x00000990, 0x00030C61, { { 0x60, 0x51, 0x11, 0x83, 0x3F, 0x06, 0xC3, 0xA3, 0xE0, 0xC0, 0x2F, 0x41, 0x29, 0xDE, 0x65, 0xB1 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x000008A8, 0x00036831, { { 0x56, 0x5B, 0x23, 0x61, 0xE8, 0x3B, 0xE1, 0x36, 0xD6, 0x62, 0xD0, 0x84, 0x00, 0x04, 0x05, 0xAD } } } },

	{ EN_ANY, kPlatformPC98, { 0x000008F4, 0x00030C61, { { 0xA4, 0x4C, 0xD9, 0x05, 0x65, 0x24, 0xE5, 0x3E, 0x6C, 0x8B, 0xEE, 0x8D, 0x4F, 0x09, 0x6F, 0x00 } } } },
	{ JA_JPN, kPlatformPC98, { 0x000007F5, 0x00036831, { { 0xF7, 0xCF, 0xF1, 0x2B, 0x26, 0x1E, 0xEC, 0xC8, 0x13, 0xDE, 0xC7, 0x71, 0x72, 0x05, 0x0F, 0xA7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplaySfxFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000173, 0x00008198, { { 0x2D, 0xF5, 0x5C, 0xAE, 0xDB, 0x71, 0x60, 0xFE, 0x78, 0x04, 0xFC, 0xAD, 0x4B, 0x43, 0xB1, 0x4B } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000234, 0x0000B0B2, { { 0xE4, 0x19, 0x4F, 0x8C, 0x53, 0xFD, 0x5F, 0x2A, 0xFB, 0x77, 0x1C, 0x32, 0xB3, 0x39, 0x25, 0x85 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x000000DE, 0x00004EA4, { { 0x97, 0xA8, 0x4A, 0xED, 0x12, 0x87, 0xEB, 0x82, 0x42, 0xFA, 0x45, 0x0D, 0x45, 0x8F, 0xFE, 0x76 } } } }, // demo
	{ UNK_LANG, kPlatformPC, { 0x000000C5, 0x00004508, { { 0xBF, 0xA9, 0x1A, 0x37, 0x0B, 0xF8, 0x13, 0x9D, 0x2D, 0x67, 0x3D, 0x9B, 0xE9, 0x16, 0x07, 0x0C } } } }, // Lands of Lore demo

	{ UNK_LANG, kPlatformFMTowns, { 0x000001A3, 0x00008098, { { 0xA9, 0xD2, 0xE2, 0x32, 0x59, 0xC4, 0x97, 0x74, 0x11, 0x49, 0x8F, 0x2B, 0xBF, 0x00, 0xF9, 0xF1 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x00000173, 0x00008098, { { 0x29, 0xA6, 0xF1, 0x2D, 0x35, 0x38, 0xFD, 0x3B, 0xF0, 0xB1, 0xE1, 0xFE, 0x45, 0xB1, 0x43, 0xAE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayTlkFilesProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000009D, 0x0000286B, { { 0x58, 0x30, 0x72, 0x62, 0xC8, 0x77, 0x2A, 0x06, 0xD6, 0x24, 0x1A, 0x7A, 0xAF, 0x79, 0xFF, 0xAE } } } },
	{ FR_FRA, kPlatformPC, { 0x0000009D, 0x00002878, { { 0x28, 0x5D, 0x7F, 0x5B, 0x57, 0xC2, 0xFF, 0x73, 0xC1, 0x8E, 0xD6, 0xE0, 0x4D, 0x03, 0x99, 0x2C } } } },
	{ DE_DEU, kPlatformPC, { 0x0000009D, 0x00002885, { { 0x87, 0x24, 0xB6, 0xE9, 0xD6, 0xAA, 0x68, 0x2D, 0x6B, 0x05, 0xDF, 0xE1, 0x2B, 0xA4, 0x79, 0xE5 } } } },
	{ IT_ITA, kPlatformPC, { 0x0000009D, 0x0000286B, { { 0x58, 0x30, 0x72, 0x62, 0xC8, 0x77, 0x2A, 0x06, 0xD6, 0x24, 0x1A, 0x7A, 0xAF, 0x79, 0xFF, 0xAE } } } },
	{ RU_RUS, kPlatformPC, { 0x0000009D, 0x0000286B, { { 0x58, 0x30, 0x72, 0x62, 0xC8, 0x77, 0x2A, 0x06, 0xD6, 0x24, 0x1A, 0x7A, 0xAF, 0x79, 0xFF, 0xAE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplaySeqDataProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000006F4, 0x0000F5AD, { { 0x23, 0x1E, 0x30, 0xF0, 0xF1, 0x5B, 0xFF, 0x4B, 0xDF, 0xB1, 0x78, 0xD8, 0x48, 0x7A, 0xA3, 0xDD } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x000006F4, 0x0000FA52, { { 0xC8, 0x33, 0xCC, 0x0A, 0x40, 0x8A, 0x96, 0x4F, 0x42, 0x7E, 0x8B, 0x62, 0x9C, 0x2F, 0x33, 0x0B } } } }, // floppy (French)
	{ UNK_LANG, kPlatformPC, { 0x000006F4, 0x0000F8A6, { { 0x8B, 0xF3, 0x80, 0xE1, 0xB8, 0xCA, 0xE1, 0x18, 0x72, 0xA3, 0xF0, 0x4C, 0xBF, 0x81, 0x97, 0x71 } } } }, // floppy (German)
	{ UNK_LANG, kPlatformPC, { 0x000006F4, 0x0000FA35, { { 0x4C, 0x26, 0x1B, 0xF6, 0x8E, 0x73, 0xB7, 0xD2, 0x58, 0x59, 0xB8, 0x97, 0x58, 0xFB, 0x0E, 0x2A } } } }, // floppy (Italian)
	{ UNK_LANG, kPlatformPC, { 0x000006BE, 0x0000E9FF, { { 0x9A, 0xCD, 0x40, 0x44, 0xEF, 0x9F, 0x05, 0x86, 0xDA, 0x49, 0x8A, 0x85, 0x68, 0xDF, 0xA7, 0x55 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x0000027C, 0x00007742, { { 0x42, 0x37, 0xF8, 0xDD, 0xA2, 0x8C, 0xA6, 0x26, 0x89, 0x5F, 0xCF, 0x61, 0xDD, 0x8F, 0xC7, 0x1E } } } }, // demo
	{ UNK_LANG, kPlatformPC, { 0x0000024C, 0x00004C10, { { 0x21, 0x3A, 0x86, 0x60, 0xA8, 0xFF, 0x42, 0x19, 0x35, 0x32, 0xA5, 0xB9, 0x3A, 0xDD, 0xA9, 0xC7 } } } }, // Lands of Lore demo

	{ UNK_LANG, kPlatformFMTowns, { 0x000006BC, 0x0000DCE3, { { 0x27, 0x09, 0x22, 0xDC, 0xED, 0xDE, 0x88, 0xAE, 0xDB, 0x97, 0x52, 0x12, 0x96, 0x8D, 0x8D, 0x37 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x000006BE, 0x0000E043, { { 0x35, 0xA0, 0xF1, 0x14, 0xC7, 0x60, 0x80, 0x35, 0xA2, 0xDB, 0xF2, 0x73, 0x96, 0xC0, 0xAE, 0x84 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayCreditsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000AE8, 0x0002E520, { { 0x19, 0x59, 0xEA, 0x70, 0x23, 0xE2, 0x41, 0x78, 0x2D, 0xF5, 0xE6, 0x28, 0xEA, 0x0C, 0xCF, 0x79 } } } },
	{ IT_ITA, kPlatformPC, { 0x00000AE8, 0x0002EE4A, { { 0x3E, 0x5C, 0x6F, 0x9E, 0x72, 0x2D, 0xAB, 0x17, 0x1D, 0x27, 0xFF, 0x32, 0xD3, 0x48, 0x46, 0xBA } } } }, // (fan) CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayCreditsSpecialProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001A, 0x00000829, { { 0xDF, 0x31, 0x0F, 0xB9, 0xC9, 0xC4, 0xF2, 0xB5, 0xF7, 0x98, 0x6F, 0x44, 0x4E, 0x9C, 0xFE, 0xA7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayIntroTracksProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x00000592, { { 0xDE, 0xD9, 0x5E, 0xCA, 0x5A, 0x9F, 0x8B, 0xB1, 0xE2, 0xDB, 0xA4, 0xA1, 0xAF, 0xD4, 0x75, 0x46 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000008, 0x000002C9, { { 0xFA, 0x80, 0x3C, 0x78, 0x66, 0x2C, 0xD5, 0x91, 0x5D, 0xF1, 0xE9, 0xC1, 0x6A, 0x65, 0xEB, 0xD6 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x00000542, { { 0xE5, 0xAA, 0x6A, 0xB9, 0x19, 0xAB, 0x35, 0x26, 0x43, 0x70, 0x32, 0xEF, 0xB9, 0x7F, 0x0F, 0x19 } } } }, // demo
	{ UNK_LANG, kPlatformPC, { 0x0000001A, 0x0000096C, { { 0xC2, 0xBD, 0x2C, 0x65, 0xD3, 0xFE, 0xF1, 0x6A, 0xE7, 0x34, 0x18, 0x0C, 0x86, 0x95, 0x66, 0x35 } } } }, // Lands of Lore demo

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayFinaleTracksProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000012, 0x00000618, { { 0x22, 0x61, 0x4D, 0x6F, 0xCE, 0x38, 0x64, 0xAB, 0x20, 0xD8, 0x34, 0x33, 0x44, 0x21, 0xE0, 0x23 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000009, 0x0000030C, { { 0x24, 0x3F, 0x57, 0xFA, 0x7E, 0x6D, 0x61, 0x6C, 0x53, 0xBC, 0x44, 0x38, 0x61, 0xE9, 0x3E, 0xE6 } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayIntroCDAProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000018, 0x00000036, { { 0xCE, 0x10, 0x68, 0x7E, 0x84, 0x68, 0x3A, 0x23, 0x36, 0x41, 0x07, 0xE8, 0x59, 0xDB, 0xA3, 0x31 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayFinaleCDAProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000008, 0x000000AE, { { 0x38, 0x47, 0x35, 0xAA, 0xEC, 0xBA, 0xD3, 0xD0, 0x3E, 0x17, 0xFD, 0xB8, 0x9F, 0xB9, 0xAD, 0x61 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayShapeAnimDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B8, 0x00000270, { { 0xBD, 0x3B, 0xFA, 0xE6, 0xA0, 0x40, 0x4E, 0x6E, 0x3E, 0x43, 0x81, 0xF3, 0x35, 0x57, 0x62, 0x27 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngamePakFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000049F, 0x00012F40, { { 0x4B, 0x2A, 0x84, 0xE3, 0xA5, 0x34, 0xE9, 0xB3, 0xB7, 0x39, 0xF1, 0x1B, 0x4C, 0xE6, 0x43, 0x79 } } } }, // floppy

	{ UNK_LANG, kPlatformFMTowns, { 0x0000011C, 0x00003FB8, { { 0x66, 0x34, 0xE8, 0x1C, 0xF9, 0xFF, 0x84, 0x90, 0x20, 0x71, 0x42, 0xA3, 0x2C, 0x4A, 0xE9, 0x46 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x000000F4, 0x00003F57, { { 0x8A, 0xA7, 0x2C, 0xB9, 0x6E, 0x8D, 0x95, 0xBF, 0xF8, 0xB9, 0x98, 0x63, 0x20, 0xF8, 0xB0, 0x7B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameSfxFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000006F1, 0x0001545E, { { 0xD3, 0x8A, 0xA1, 0xD4, 0x83, 0x77, 0x96, 0x6D, 0x87, 0xB1, 0x71, 0x8F, 0x38, 0x6A, 0x34, 0xDC } } } },
	{ UNK_LANG, kPlatformFMTowns, { 0x00000967, 0x0002101A, { { 0x09, 0xC7, 0xB7, 0x2A, 0x76, 0xF1, 0x4B, 0x87, 0xC5, 0x83, 0xFF, 0xF3, 0xDB, 0x3C, 0x66, 0x60 } } } },
	{ UNK_LANG, kPlatformPC98,	{ 0x000006F1, 0x0001545E, { { 0xD3, 0x8A, 0xA1, 0xD4, 0x83, 0x77, 0x96, 0x6D, 0x87, 0xB1, 0x71, 0x8F, 0x38, 0x6A, 0x34, 0xDC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameSfxIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000358, 0x000074F4, { { 0xC8, 0x30, 0x1D, 0x9C, 0xFC, 0xF8, 0xD5, 0xBA, 0x7E, 0xAB, 0x13, 0x3F, 0x43, 0x50, 0xFA, 0xC4 } } } }, // CD

	{ UNK_LANG, kPlatformUnknown, { 0x00000358, 0x0000747E, { { 0x20, 0x99, 0x4D, 0xB8, 0xF2, 0x05, 0xF2, 0xA7, 0x07, 0x28, 0x97, 0xFB, 0x70, 0x1F, 0x2A, 0x73 } } } }, // floppy + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameTracksProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000007E, 0x00001E45, { { 0xC6, 0x52, 0xCC, 0x1A, 0x23, 0x3B, 0xDA, 0x03, 0xF3, 0xC2, 0x46, 0xDF, 0x01, 0xA3, 0x9E, 0x50 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameCDAProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000130, 0x00001B9D, { { 0x9E, 0xFB, 0x01, 0xE7, 0xBA, 0xE1, 0x4F, 0x3C, 0xC5, 0xB9, 0x75, 0x80, 0xEA, 0xAC, 0xB0, 0x0F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameTalkObjIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000078, 0x00000A9C, { { 0x27, 0x65, 0xFA, 0xB1, 0xF8, 0x04, 0x7F, 0x93, 0x2B, 0x2D, 0x44, 0x3C, 0x95, 0x13, 0x50, 0xD2 } } } }, // floppy (English)
	{ UNK_LANG, kPlatformPC, { 0x00000078, 0x00000A53, { { 0x18, 0x5B, 0x79, 0x92, 0x54, 0xDB, 0x4B, 0x0C, 0x78, 0x89, 0x7C, 0x59, 0xBF, 0x53, 0x37, 0x14 } } } }, // floppy (German + French)
	{ UNK_LANG, kPlatformPC, { 0x00000078, 0x00000A7C, { { 0x33, 0x89, 0x69, 0xEF, 0x62, 0x94, 0x3A, 0xC0, 0xB2, 0xBA, 0xFB, 0x0C, 0x86, 0x2C, 0xAC, 0x5C } } } }, // floppy (Italian)
	{ UNK_LANG, kPlatformPC, { 0x00000084, 0x00000CD0, { { 0x82, 0x2E, 0xF4, 0xDD, 0x28, 0x75, 0xDD, 0x79, 0x78, 0x62, 0x55, 0x48, 0xDD, 0xF7, 0x16, 0x81 } } } }, // CD

	{ UNK_LANG, kPlatformFMTowns, { 0x0000007A, 0x00000744, { { 0x74, 0x84, 0xB4, 0xA6, 0x9D, 0x91, 0xBF, 0x24, 0x1E, 0xD5, 0xDE, 0xD4, 0x73, 0x93, 0x1F, 0xE6 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x00000078, 0x0000099D, { { 0x2F, 0x06, 0x8D, 0x4E, 0x76, 0x9E, 0x92, 0xFD, 0x43, 0xA4, 0x12, 0xED, 0xBA, 0x30, 0xFE, 0x94 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameTimJpStringsProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x0000002D, 0x00001302, { { 0x15, 0x29, 0xFD, 0xB1, 0xEB, 0xC0, 0x00, 0xA4, 0x19, 0xDA, 0xC7, 0x2E, 0x17, 0x0F, 0x3F, 0xB7 } } } },
	{ UNK_LANG, kPlatformPC98, { 0x0000002C, 0x00001302, { { 0xAB, 0xBA, 0xC8, 0x2A, 0xD0, 0xD1, 0x1C, 0xEA, 0x6F, 0x35, 0x12, 0x5D, 0x85, 0xFB, 0x92, 0x84 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameShapeAnimDataProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000348, 0x0001AAEB, { { 0x67, 0xD1, 0x03, 0xBF, 0x4B, 0xEC, 0x80, 0x26, 0x5D, 0x1D, 0x0E, 0x5B, 0xF1, 0xE8, 0x4C, 0x64 } } } }, // CD

	{ UNK_LANG, kPlatformUnknown, { 0x00000348, 0x00002473, { { 0xBC, 0x18, 0x51, 0xFA, 0x51, 0x1D, 0x24, 0x7D, 0x44, 0xD0, 0xE2, 0x38, 0x31, 0x33, 0x3B, 0x56 } } } }, // floppy + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameTlkDemoStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000000CB, 0x00004741, { { 0xF3, 0x54, 0x37, 0xEE, 0x61, 0x9E, 0xE0, 0x82, 0x64, 0x8B, 0xF3, 0x53, 0xC7, 0x2F, 0x49, 0x1F } } } },
	{ FR_FRA, kPlatformPC, { 0x000000BE, 0x00004327, { { 0x8C, 0xF2, 0x5E, 0x85, 0xDA, 0xA1, 0x91, 0x90, 0xE5, 0xC7, 0x2A, 0xBA, 0x48, 0xC3, 0x9A, 0xCA } } } },
	{ DE_DEU, kPlatformPC, { 0x000000DC, 0x00004C7B, { { 0xC9, 0x75, 0x3A, 0x4A, 0xF4, 0xB0, 0xE8, 0x61, 0x90, 0x74, 0x34, 0x84, 0x53, 0x54, 0xA2, 0x4F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3MainMenuStringsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B0, 0x00003E75, { { 0xA7, 0x91, 0x9B, 0x61, 0x31, 0x12, 0x74, 0xB5, 0x4B, 0x4F, 0x8A, 0x44, 0xA9, 0xB1, 0x03, 0xC0 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3MusicFilesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000001D5, 0x00007436, { { 0x27, 0xC2, 0x96, 0xFF, 0x22, 0xEC, 0xB5, 0x26, 0xF9, 0x4F, 0x04, 0xDD, 0x58, 0x17, 0xFE, 0xC6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3ScoreTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000C8, 0x0000038F, { { 0x4E, 0x2E, 0xC3, 0xD9, 0x13, 0xA0, 0x74, 0x7E, 0xAB, 0xA1, 0x23, 0x2F, 0x11, 0x35, 0x4D, 0xA5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3SfxFilesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000008BF, 0x0001A766, { { 0xBD, 0x5D, 0x65, 0x5D, 0xC4, 0x34, 0xDF, 0x36, 0xF5, 0xC0, 0xE8, 0x7B, 0x51, 0x9F, 0xE5, 0xC7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3SfxMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000004C0, 0x0000C4F9, { { 0x06, 0x61, 0x45, 0x38, 0xC2, 0x6A, 0x12, 0x23, 0x55, 0x2C, 0x7F, 0xA5, 0xE2, 0x79, 0x81, 0x9E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3ItemAnimDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000001C8, 0x00003777, { { 0x8F, 0xE8, 0x80, 0x4A, 0x88, 0x83, 0xC0, 0x48, 0x55, 0x47, 0x81, 0xD3, 0xB7, 0xD5, 0x09, 0x8F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3ItemMagicTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00000F9E, { { 0x87, 0xA7, 0xC0, 0xA2, 0x0F, 0xA3, 0x5B, 0x08, 0x76, 0x6F, 0x02, 0x68, 0x13, 0x17, 0x3B, 0x76 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3ItemStringMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x0000003C, { { 0x4C, 0xA7, 0xAA, 0xB0, 0x3C, 0xCC, 0x78, 0x91, 0x0A, 0xF2, 0x7F, 0x25, 0x80, 0xB3, 0x74, 0x74 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenStrings1Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x000000CA, 0x00003BC8, { { 0x27, 0xEA, 0xE3, 0x0D, 0x55, 0xB3, 0x69, 0x3E, 0xC2, 0x66, 0x58, 0x64, 0xAA, 0xC2, 0x80, 0x58 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x000000C3, 0x000038F6, { { 0x20, 0x68, 0xAB, 0xD4, 0xBF, 0x49, 0x04, 0xC0, 0x91, 0xB4, 0x71, 0xB0, 0xB6, 0xC9, 0x11, 0xF0 } } } }, // EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x000000C7, 0x00003ADB, { { 0x0D, 0x03, 0x7A, 0xE6, 0x7D, 0x41, 0x89, 0x49, 0x0C, 0xB6, 0xD0, 0x4F, 0xEA, 0x1E, 0x54, 0xFF } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenStrings2Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000004B, 0x000011AE, { { 0x28, 0x98, 0x4C, 0xA3, 0x98, 0xB0, 0xA2, 0x17, 0x9C, 0x80, 0x4F, 0x3F, 0x58, 0x3B, 0x2C, 0xFB } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000004E, 0x0000129D, { { 0xED, 0xF3, 0x36, 0x16, 0xE2, 0x1B, 0x32, 0x95, 0xFE, 0xE8, 0x3E, 0x7D, 0xE6, 0x32, 0x34, 0xD4 } } } }, // EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000004A, 0x00001267, { { 0xD6, 0xE2, 0x27, 0x6A, 0x6F, 0x7E, 0xB4, 0xCE, 0xA8, 0xE9, 0x79, 0x31, 0x5C, 0x13, 0xA1, 0x8F } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenStartLevelsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000054, { { 0xAB, 0x68, 0x74, 0x3E, 0x0D, 0x45, 0xA3, 0x50, 0xA7, 0x72, 0x6A, 0xDF, 0x9C, 0x23, 0x98, 0x74 } } } },	// EOB 1
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x000000B1, { { 0xFD, 0x3F, 0x6B, 0xB5, 0xE4, 0xEE, 0x32, 0x3B, 0xBD, 0x72, 0x37, 0x88, 0x52, 0x84, 0xBD, 0xC6 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenStatStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000055, 0x000015D6, { { 0xB8, 0x29, 0x4B, 0xA4, 0x4F, 0x45, 0x16, 0x1A, 0x07, 0x28, 0x14, 0x86, 0x1B, 0xDF, 0xAC, 0xDF } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000056, 0x000015F9, { { 0xBB, 0x5A, 0x7D, 0xCF, 0xC3, 0x90, 0x9A, 0xB3, 0x73, 0xB2, 0x4D, 0x46, 0xB8, 0x89, 0x7D, 0xAE } } } }, // EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000053, 0x0000159A, { { 0x1D, 0xA6, 0x84, 0xDB, 0xC5, 0x81, 0xC7, 0xF0, 0x1C, 0xA4, 0xE3, 0x10, 0x4F, 0x8A, 0xF3, 0xCE } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenRaceSexStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000098, 0x00002572, { { 0x8D, 0xF9, 0xDE, 0x92, 0xFC, 0xA8, 0xFC, 0xE9, 0x0A, 0x98, 0x6D, 0xA4, 0x6F, 0x21, 0xCD, 0xF4 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x000000AA, 0x00002A1E, { { 0x8E, 0xAF, 0x4B, 0x20, 0xEA, 0xFE, 0x71, 0x8E, 0x8B, 0x4B, 0x46, 0x62, 0x91, 0x48, 0x08, 0xAF } } } }, // EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000098, 0x00002502, { { 0xA4, 0x8B, 0x20, 0xF5, 0x97, 0xFE, 0x34, 0x6D, 0x9F, 0xF0, 0xA8, 0xE9, 0xE8, 0xFA, 0x23, 0x9B } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenClassStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x000000D5, 0x000035A7, { { 0xAF, 0x89, 0x9A, 0x11, 0x9A, 0x8D, 0x39, 0x6F, 0x26, 0x41, 0x4E, 0x20, 0xAD, 0x91, 0xC5, 0xDA } } } },
	{ DE_DEU, kPlatformUnknown, { 0x000000FA, 0x00003FD8, { { 0xBD, 0x78, 0xF7, 0xEC, 0x9D, 0x9A, 0x3A, 0x22, 0xAB, 0xD9, 0x10, 0xAD, 0x8E, 0x1D, 0x4D, 0xDE } } } }, // EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x000000E4, 0x00003BE5, { { 0xDE, 0x1B, 0x25, 0x4F, 0xE6, 0xD0, 0xB5, 0x95, 0xD0, 0xA6, 0x69, 0xE6, 0x53, 0xB8, 0x20, 0x1E } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenAlignmentStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000078, 0x00001F44, { { 0xBB, 0x52, 0x3C, 0xA6, 0x79, 0x87, 0xDC, 0xB8, 0x21, 0x7A, 0xA0, 0x17, 0x45, 0xEA, 0xF2, 0x50 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000008A, 0x00002423, { { 0xA3, 0x36, 0x0D, 0x64, 0x33, 0xFD, 0x54, 0xA5, 0xA9, 0xD7, 0xFA, 0x34, 0x39, 0xAD, 0x6A, 0x98 } } } }, // EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000007F, 0x000021F8, { { 0xBD, 0xB2, 0x06, 0xF9, 0xC9, 0x36, 0x5D, 0x91, 0x43, 0x08, 0x3A, 0x2C, 0x5F, 0x1C, 0xF3, 0x9C } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenEnterGameStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000020, 0x00000A0E, { { 0x98, 0x7F, 0x2C, 0x2E, 0xBB, 0x5E, 0xAA, 0xD0, 0x72, 0xF5, 0xBC, 0x4A, 0x34, 0x5B, 0xB4, 0xF5 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000021, 0x00000AB6, { { 0x02, 0x7F, 0x19, 0x5A, 0xA9, 0xB7, 0x8C, 0xE2, 0xF7, 0x35, 0xB0, 0xD8, 0xA8, 0x0C, 0x24, 0x44 } } } }, // EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000001E, 0x00000925, { { 0xDA, 0x83, 0x00, 0xD2, 0x94, 0xF0, 0xD8, 0xFC, 0x3D, 0xA8, 0xD2, 0x4E, 0xF2, 0xD7, 0x2B, 0x7E } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenClassMinStatsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B4, 0x00000165, { { 0x83, 0x5E, 0x91, 0x10, 0x4D, 0x75, 0x6B, 0xF9, 0x45, 0x1B, 0x65, 0x13, 0x37, 0x3E, 0xC0, 0xAE } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenRaceMinStatsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x000000B2, { { 0x08, 0xF0, 0x8F, 0x22, 0x9D, 0xD8, 0xBE, 0x52, 0x70, 0x7C, 0xCA, 0x8D, 0xB2, 0xF5, 0xC6, 0xB8 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseChargenRaceMaxStatsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00000479, { { 0xBD, 0xA0, 0x8C, 0xC2, 0x05, 0xCA, 0x0D, 0x4B, 0x82, 0x9B, 0x3D, 0xB5, 0x4B, 0xDB, 0xD2, 0xC1 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSaveThrowTable1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000032, 0x00000214, { { 0x3D, 0x89, 0x30, 0x0A, 0x5C, 0x4A, 0x0F, 0xC3, 0xC7, 0x6B, 0x72, 0x7C, 0x12, 0x51, 0x8D, 0x8E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSaveThrowTable2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000019, 0x000000E7, { { 0xF4, 0x0D, 0xDF, 0xA3, 0x23, 0x71, 0x76, 0x2A, 0xC5, 0x6F, 0xF1, 0x59, 0x5F, 0x45, 0x73, 0x05 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSaveThrowTable3Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000023, 0x00000155, { { 0x42, 0x98, 0x84, 0x00, 0x70, 0x8A, 0x7B, 0x26, 0xC0, 0x96, 0xA3, 0x28, 0x41, 0x36, 0x4B, 0x21 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSaveThrowTable4Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001E, 0x0000013B, { { 0xAB, 0x84, 0x2B, 0x0A, 0xC2, 0x46, 0xFF, 0x83, 0x03, 0xF8, 0x3F, 0x32, 0x53, 0xA2, 0x95, 0x65 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSaveThrwLvlIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000006, 0x00000070, { { 0x57, 0x48, 0x5F, 0x75, 0x79, 0xD4, 0xAA, 0x7D, 0xB7, 0xEB, 0x19, 0x9F, 0xCF, 0x99, 0x29, 0x29 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSaveThrwModDivProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000006, 0x00000012, { { 0x50, 0x29, 0x51, 0x65, 0x0B, 0xF1, 0xCC, 0xDA, 0x2C, 0xA4, 0x7E, 0xEE, 0x20, 0xB0, 0x29, 0xB1 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSaveThrwModExtProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000006, 0x00000030, { { 0x07, 0x7D, 0x61, 0x1C, 0x95, 0xEC, 0x9A, 0xCE, 0xA1, 0x29, 0x83, 0x2F, 0xCA, 0x95, 0x95, 0xF5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBasePryDoorStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x000000E8, 0x00004D9F, { { 0xDE, 0x01, 0x69, 0x00, 0x0B, 0x32, 0xFA, 0x20, 0xB8, 0x11, 0xD6, 0xD9, 0xE2, 0xEA, 0xF5, 0xE8 } } } },	// EOB 1
	{ EN_ANY, kPlatformUnknown, { 0x000000D2, 0x000043D2, { { 0x82, 0x3C, 0xF4, 0x4A, 0x87, 0x84, 0xFE, 0xF9, 0xBA, 0xC6, 0x67, 0x3A, 0x0D, 0x0F, 0x76, 0x78 } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x00000128, 0x0000657C, { { 0xA3, 0xC8, 0x48, 0xA7, 0x1F, 0x75, 0xDF, 0xB0, 0x37, 0xDA, 0x75, 0x2E, 0x9F, 0x4E, 0x45, 0xB0 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x000000D9, 0x00004769, { { 0x24, 0x59, 0x00, 0x8F, 0x9A, 0x3E, 0x95, 0xAB, 0x14, 0x9A, 0x3B, 0x19, 0x34, 0xDB, 0x9B, 0x18 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseWarningStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000052, 0x00001A7B, { { 0x81, 0x7A, 0xDF, 0xD2, 0x4F, 0xA7, 0x92, 0xA7, 0x44, 0xE5, 0x22, 0x73, 0xF6, 0xB3, 0x29, 0x5C } } } },	// EOB 1
	{ EN_ANY, kPlatformUnknown, { 0x00000085, 0x00002B5C, { { 0xF1, 0xCE, 0x7C, 0x53, 0xEF, 0x5B, 0x59, 0x71, 0xA9, 0xEB, 0x00, 0xBA, 0xB7, 0x59, 0xC5, 0x2E } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x0000005F, 0x00001FD2, { { 0xBA, 0x85, 0x97, 0x63, 0x84, 0x80, 0x79, 0x44, 0x50, 0x99, 0x1A, 0x01, 0x37, 0x37, 0x0E, 0xD1 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000096, 0x000032BF, { { 0x07, 0x95, 0x91, 0x1A, 0x1B, 0xC8, 0xA3, 0xEE, 0x76, 0x0A, 0x48, 0x11, 0x37, 0x6F, 0xBA, 0x05 } } } },	// EOB 1
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseItemSuffixStringsRingsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000002B, 0x00000F7B, { { 0x8A, 0x27, 0x87, 0x81, 0x5F, 0x74, 0x27, 0xA9, 0x5E, 0x1B, 0xEE, 0xC0, 0xF7, 0x22, 0x8F, 0x57 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000022, 0x00000C02, { { 0x7D, 0x5F, 0x40, 0xEA, 0xAD, 0xDD, 0x1B, 0xA0, 0xA6, 0xE0, 0x57, 0x7D, 0x0D, 0x60, 0xF4, 0x2C } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000002E, 0x00000FF2, { { 0xE1, 0x50, 0xB7, 0xE2, 0xEF, 0xAD, 0x5B, 0x6D, 0x27, 0x35, 0x9C, 0xE7, 0x2D, 0xB2, 0x2E, 0xD0 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseItemSuffixStringsPotionsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000054, 0x00001DDB, { { 0xB6, 0x78, 0xD9, 0x09, 0x1D, 0x09, 0x63, 0xF8, 0x96, 0x74, 0xF0, 0x75, 0x23, 0xF5, 0xD4, 0xC4 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000064, 0x000024ED, { { 0x10, 0x5A, 0xB8, 0xCA, 0x0F, 0x0D, 0x44, 0x19, 0x9D, 0x3D, 0x76, 0x54, 0xA1, 0x69, 0x97, 0x8B } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000006F, 0x00002792, { { 0x1A, 0x71, 0x2B, 0xCC, 0xCA, 0xDA, 0xF6, 0xED, 0x5E, 0xF0, 0x24, 0x20, 0xD7, 0x2D, 0x18, 0x49 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseItemSuffixStringsWandsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000003C, 0x000014EB, { { 0xB5, 0x38, 0x35, 0x57, 0xF2, 0xF8, 0x0E, 0xBA, 0x75, 0x03, 0x1C, 0xCD, 0x46, 0x7D, 0x03, 0x83 } } } },	// EOB 1
	{ EN_ANY, kPlatformUnknown, { 0x0000004A, 0x000019B2, { { 0x44, 0x10, 0xE4, 0xAF, 0xAB, 0x19, 0x25, 0x87, 0x2B, 0x15, 0x1C, 0x4C, 0x03, 0x50, 0x41, 0xC4 } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x00000040, 0x000016B5, { { 0xEC, 0xF4, 0x71, 0xC1, 0x69, 0x5C, 0xF9, 0xC1, 0xED, 0xC1, 0xED, 0x0C, 0x25, 0x3E, 0x13, 0xB1 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000069, 0x0000252B, { { 0x12, 0x06, 0xEA, 0x2F, 0xAF, 0x47, 0x55, 0x52, 0xB6, 0xD9, 0x11, 0xA4, 0x4F, 0x30, 0xCE, 0x9D } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseRipItemStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000025, 0x00000AEA, { { 0x7A, 0x2D, 0x03, 0xA5, 0x94, 0xD1, 0xA2, 0x2C, 0x41, 0x1F, 0xEB, 0x5C, 0xFB, 0xB2, 0xC6, 0x9E } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000024, 0x00000B1B, { { 0xD0, 0x26, 0x19, 0x0B, 0xA5, 0x8A, 0x38, 0x73, 0x14, 0x25, 0x40, 0x5D, 0x24, 0xB8, 0x4E, 0xC5 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000002E, 0x00000D38, { { 0xCE, 0xC5, 0x00, 0x63, 0xBB, 0xF0, 0xC4, 0x0D, 0x50, 0x2B, 0x82, 0x1C, 0xC0, 0xCD, 0xF1, 0xAF } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCursedStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000D, 0x000003C7, { { 0x7F, 0x6B, 0x6A, 0xFE, 0x63, 0xF4, 0x17, 0xAF, 0xFD, 0x00, 0x31, 0x4A, 0x20, 0x9E, 0x8C, 0xEB } } } },	// EOB 1
	{ EN_ANY, kPlatformUnknown, { 0x0000000D, 0x000003C7, { { 0x59, 0xD8, 0x84, 0x25, 0xE0, 0x06, 0x51, 0xA4, 0x70, 0xC5, 0x78, 0x22, 0xF0, 0x2D, 0xA0, 0x43 } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x00000010, 0x00000514, { { 0x97, 0x41, 0xA6, 0xAE, 0xF8, 0xA8, 0x3E, 0x85, 0xA8, 0x16, 0x01, 0x15, 0x0E, 0x46, 0x13, 0x45 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000013, 0x000005A5, { { 0xEC, 0xD3, 0xA5, 0xD2, 0xAD, 0x7C, 0x5E, 0x0F, 0x42, 0xBC, 0x6E, 0xDE, 0x7E, 0x36, 0x0B, 0x43 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseEnchantedStringProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000007, 0x0000016C, { { 0x98, 0x62, 0xD3, 0xA3, 0x11, 0xAE, 0x0A, 0xBA, 0x8F, 0xE8, 0x30, 0x0B, 0xDC, 0x12, 0x90, 0x3B } } } },	// EOB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000007, 0x0000016C, { { 0x01, 0x91, 0xBD, 0x89, 0xAE, 0x0E, 0x71, 0xEE, 0xBE, 0x31, 0xD9, 0x55, 0x21, 0x61, 0x19, 0xAE } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicObjectStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000002B, 0x00000E7D, { { 0x7E, 0x8F, 0x17, 0xEB, 0xE5, 0x5D, 0xEB, 0x9A, 0x84, 0xFF, 0x86, 0x6A, 0x01, 0x3E, 0x04, 0x84 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000003A, 0x000014E4, { { 0x3D, 0x34, 0x3C, 0xCA, 0xDC, 0xD1, 0xCF, 0x15, 0x69, 0x57, 0xC3, 0xB1, 0x58, 0xDF, 0xE9, 0x9D } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000002A, 0x00000EE4, { { 0x9C, 0x38, 0x4B, 0x9B, 0x67, 0x30, 0x4E, 0x88, 0xA9, 0xA2, 0xF8, 0x78, 0x8E, 0xC7, 0xC3, 0x86 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicObject5StringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000006, 0x000001FE, { { 0x74, 0x8D, 0xB9, 0x76, 0xD2, 0x2F, 0x60, 0xD2, 0x36, 0x45, 0x98, 0x4C, 0x0A, 0xE5, 0xE5, 0x0D } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000006, 0x00000204, { { 0xE4, 0xC1, 0xAD, 0x71, 0x87, 0x80, 0x9D, 0x97, 0x91, 0x80, 0x3F, 0x71, 0xD3, 0x62, 0x06, 0xD5 } } } }, // EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000007, 0x0000027A, { { 0x44, 0x47, 0x79, 0x46, 0x9B, 0xE5, 0xBD, 0x3C, 0xE8, 0x8D, 0xC6, 0xC5, 0x4E, 0x88, 0x13, 0xC0 } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBasePatternSuffixProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000009, 0x00000245, { { 0x67, 0x3F, 0x33, 0xA5, 0x3B, 0x5D, 0x2C, 0x9E, 0x15, 0x82, 0x04, 0xE2, 0xD7, 0x34, 0x42, 0x24 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000006, 0x0000015D, { { 0x33, 0xD6, 0x91, 0x2D, 0xED, 0xE1, 0x43, 0x42, 0x23, 0xB9, 0xE9, 0x3D, 0x48, 0x82, 0x92, 0x1E } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000008, 0x00000219, { { 0xCD, 0xDC, 0x7F, 0x8B, 0xBE, 0xD6, 0x05, 0x37, 0xDA, 0xDC, 0x11, 0xC3, 0x1E, 0x7A, 0xE7, 0x13 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBasePatternGrFix1Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000009, 0x00000245, { { 0x67, 0x3F, 0x33, 0xA5, 0x3B, 0x5D, 0x2C, 0x9E, 0x15, 0x82, 0x04, 0xE2, 0xD7, 0x34, 0x42, 0x24 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000006, 0x0000015D, { { 0x33, 0xD6, 0x91, 0x2D, 0xED, 0xE1, 0x43, 0x42, 0x23, 0xB9, 0xE9, 0x3D, 0x48, 0x82, 0x92, 0x1E } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000007, 0x0000018A, { { 0x02, 0x5C, 0x86, 0xD9, 0x62, 0x0C, 0x71, 0xB3, 0x77, 0x9C, 0x7B, 0xBC, 0x4D, 0x5B, 0xDB, 0xE7 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBasePatternGrFix2Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000009, 0x00000245, { { 0x67, 0x3F, 0x33, 0xA5, 0x3B, 0x5D, 0x2C, 0x9E, 0x15, 0x82, 0x04, 0xE2, 0xD7, 0x34, 0x42, 0x24 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000006, 0x0000015D, { { 0x33, 0xD6, 0x91, 0x2D, 0xED, 0xE1, 0x43, 0x42, 0x23, 0xB9, 0xE9, 0x3D, 0x48, 0x82, 0x92, 0x1E } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000006, 0x00000150, { { 0x48, 0xBE, 0xED, 0xD3, 0xA5, 0x2E, 0xCD, 0xE0, 0x34, 0xBA, 0xA6, 0x8D, 0x7D, 0x00, 0xA2, 0xFF } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseValidateArmorStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000023, 0x00000B78, { { 0xC2, 0x33, 0x6B, 0xB9, 0xE1, 0x5E, 0x88, 0x5E, 0x22, 0xF2, 0x97, 0x83, 0xF8, 0xC8, 0x8C, 0xAB } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000026, 0x00000D19, { { 0xAD, 0x19, 0xE2, 0xDE, 0x04, 0xF9, 0x8F, 0x92, 0xAC, 0x1A, 0x05, 0xEA, 0x7B, 0xB5, 0x9F, 0x09 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000029, 0x00000E7A, { { 0xEC, 0xA8, 0x2E, 0x8D, 0xB1, 0xC8, 0x0F, 0xCD, 0x24, 0xBD, 0x4B, 0x39, 0x16, 0xC9, 0x53, 0x08 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseValidateCursedStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000002E, 0x00000F35, { { 0xE7, 0x0E, 0xA1, 0xCE, 0xCC, 0x13, 0xBC, 0x4B, 0x2B, 0x19, 0xEB, 0xA4, 0x05, 0xCF, 0xCF, 0x65 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000037, 0x000012D8, { { 0x3C, 0x7F, 0x16, 0xCE, 0x40, 0x58, 0xF1, 0x3A, 0xAB, 0x4C, 0x37, 0x82, 0x32, 0x88, 0xA4, 0x2D } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseValidateNoDropStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000001F, 0x00000A8F, { { 0x61, 0x99, 0x3E, 0x36, 0x49, 0x19, 0xB4, 0xE4, 0xBC, 0xFA, 0xB5, 0x71, 0x0E, 0xD6, 0x15, 0x3C } } } },	// EOB 1
	{ EN_ANY, kPlatformUnknown, { 0x00000020, 0x00000AB6, { { 0xAA, 0x0E, 0x64, 0xD1, 0xA2, 0xA6, 0x62, 0x76, 0x51, 0xDF, 0x9E, 0x76, 0x85, 0x42, 0xE1, 0x4A } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x00000024, 0x00000C31, { { 0x10, 0xD9, 0x55, 0x69, 0xFE, 0x0A, 0x8C, 0xE5, 0xF7, 0x05, 0x5F, 0x09, 0x3B, 0xC9, 0x93, 0x38 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000027, 0x00000D9F, { { 0xA5, 0xF0, 0x8E, 0x78, 0x0A, 0x37, 0x31, 0xDC, 0xE0, 0xDF, 0xE5, 0xCB, 0x86, 0xDC, 0x21, 0x73 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBasePotionStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000017, 0x0000070E, { { 0xD9, 0xCB, 0x26, 0xB6, 0x6F, 0x17, 0x12, 0xB7, 0xB0, 0x95, 0x1B, 0x2A, 0xD8, 0x83, 0x0D, 0x2B } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000001E, 0x000009BD, { { 0xCA, 0xD0, 0x29, 0xB0, 0x7A, 0x2B, 0x0B, 0x69, 0xCA, 0xA4, 0xCA, 0x97, 0xCF, 0x8B, 0x03, 0xAD } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000001D, 0x00000964, { { 0x5D, 0xE2, 0xA5, 0x0D, 0x72, 0xE9, 0x8F, 0xC9, 0xFA, 0xF3, 0x41, 0x5A, 0x3F, 0x33, 0xAA, 0x15 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseWandStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000035, 0x000011EC, { { 0x7C, 0x3D, 0xF1, 0x28, 0x0C, 0x23, 0xD3, 0x18, 0xEE, 0xAD, 0xA7, 0xF4, 0x58, 0xD7, 0x1C, 0x8E } } } },	// EOB 1
	{ EN_ANY, kPlatformUnknown, { 0x00000029, 0x00000E47, { { 0xED, 0x2E, 0xD4, 0x4D, 0xDB, 0x90, 0x3F, 0xD0, 0xFB, 0x95, 0xB8, 0xF2, 0xCF, 0x06, 0x08, 0xAF } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x00000046, 0x0000186E, { { 0x54, 0x8F, 0x53, 0x34, 0xE8, 0x81, 0x76, 0x71, 0x53, 0x3F, 0x99, 0xE7, 0xCF, 0xB7, 0xC9, 0xD9 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000002F, 0x00001070, { { 0x86, 0x18, 0x00, 0x54, 0x05, 0x3D, 0xC2, 0x26, 0xA7, 0xD9, 0x68, 0xE6, 0xC2, 0x0D, 0x26, 0x99 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseItemMisuseStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000066, 0x000022F6, { { 0xE8, 0xB9, 0x07, 0x61, 0x29, 0x90, 0xB0, 0x22, 0x30, 0xC5, 0x0F, 0xAD, 0xCA, 0x6C, 0x83, 0xC6 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000068, 0x00002472, { { 0xCA, 0xD7, 0xFD, 0x5B, 0x65, 0x72, 0xC7, 0x15, 0xB3, 0xFE, 0xFC, 0xEF, 0x53, 0xFB, 0x57, 0x6C } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000007E, 0x00002C87, { { 0x5E, 0x11, 0xC9, 0x93, 0xF4, 0xAB, 0x1A, 0x9D, 0xA7, 0x62, 0x71, 0x94, 0x37, 0xCA, 0xE2, 0x25 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseTakenStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000009, 0x0000026E, { { 0x3B, 0x73, 0x70, 0x2E, 0x22, 0x90, 0x0D, 0xC1, 0xDE, 0x32, 0x11, 0xCC, 0x97, 0xBA, 0xA3, 0x58 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000000F, 0x000004ED, { { 0x8D, 0x12, 0x1E, 0x91, 0xD3, 0xF4, 0x34, 0x15, 0xC7, 0x4F, 0xE7, 0x23, 0x5B, 0xE8, 0x66, 0xB7 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBasePotionEffectStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000065, 0x0000248F, { { 0x4F, 0x60, 0x7F, 0xA7, 0x6F, 0x81, 0xD4, 0xAA, 0x68, 0xD5, 0xAA, 0xBE, 0xBC, 0xD4, 0x92, 0x3A } } } },	// EOB 1
	{ EN_ANY, kPlatformUnknown, { 0x0000005D, 0x0000219D, { { 0x87, 0x60, 0x9F, 0xF3, 0x1B, 0x30, 0x4B, 0x2B, 0xE4, 0x94, 0xEF, 0x22, 0xEA, 0x36, 0xE4, 0x7F } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x0000006E, 0x00002840, { { 0x04, 0xF8, 0x53, 0x38, 0x09, 0xD8, 0x58, 0xFC, 0x5F, 0xE9, 0x69, 0xFB, 0x9C, 0x0D, 0x92, 0x2E } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000059, 0x000020D2, { { 0xB6, 0xA1, 0x57, 0xD6, 0x46, 0xE3, 0xCF, 0x04, 0x5A, 0xC8, 0xBB, 0x59, 0x8D, 0xE3, 0x6F, 0xBF } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseYesNoStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000007, 0x0000022E, { { 0xF1, 0x30, 0x61, 0xA7, 0x20, 0x71, 0x3B, 0x75, 0xBE, 0xA7, 0xD6, 0x78, 0x34, 0xF7, 0x19, 0x06 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000008, 0x00000275, { { 0xAF, 0x3E, 0xC5, 0x5A, 0x60, 0x34, 0x9B, 0x39, 0x37, 0x9E, 0xE2, 0x17, 0x23, 0x8E, 0x23, 0x23 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonMoreStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000005, 0x00000133, { { 0xA6, 0x1A, 0x3A, 0xB8, 0xCC, 0x92, 0xB8, 0xBE, 0x28, 0xD6, 0x64, 0x8F, 0x0A, 0x2A, 0x39, 0xCD } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000005, 0x0000012C, { { 0x82, 0x30, 0x00, 0xD6, 0xFA, 0x53, 0x17, 0x69, 0x64, 0xCA, 0xFE, 0x0F, 0x92, 0xEF, 0x87, 0x7A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseNpcMaxStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000052, 0x00001D23, { { 0x95, 0xB0, 0xBF, 0xF9, 0xE6, 0x8C, 0xCF, 0x9B, 0x36, 0xE3, 0x81, 0x22, 0x1E, 0x68, 0x9E, 0xBE } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000064, 0x00002341, { { 0xC0, 0xA6, 0xCD, 0x5E, 0x8E, 0xFA, 0x89, 0xE4, 0x98, 0xE5, 0x3D, 0x13, 0x6B, 0xE7, 0x8F, 0x6E } } } },	// EoB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000003E, 0x00001613, { { 0x4E, 0x31, 0x7F, 0xC4, 0xC7, 0x9C, 0xB5, 0x7D, 0x36, 0x85, 0xD8, 0x81, 0xE2, 0x06, 0xF9, 0xAE } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseOkStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000003, 0x0000009A, { { 0x88, 0xD2, 0x76, 0x1C, 0x80, 0x02, 0xC5, 0x5B, 0x35, 0x57, 0x0E, 0xEB, 0xCA, 0xD6, 0xC9, 0x2E } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000003, 0x0000009A, { { 0x88, 0xD2, 0x76, 0x1C, 0x80, 0x02, 0xC5, 0x5B, 0x35, 0x57, 0x0E, 0xEB, 0xCA, 0xD6, 0xC9, 0x2E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseNpcJoinStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000015, 0x000006C7, { { 0x5A, 0xBF, 0xA2, 0x3E, 0x36, 0xC4, 0x23, 0xC8, 0xA8, 0x86, 0x06, 0x80, 0xAF, 0xB1, 0xDD, 0xAB } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000020, 0x00000A93, { { 0x4A, 0xFD, 0x70, 0xB7, 0x7A, 0x0B, 0x7C, 0x32, 0x07, 0x5A, 0x4A, 0xC7, 0x84, 0x9D, 0x2D, 0xF3 } } } },	// EoB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000018, 0x00000848, { { 0xC9, 0xEE, 0x71, 0x04, 0xA2, 0xA5, 0x52, 0x87, 0x7C, 0x6D, 0x3C, 0x15, 0x7D, 0x41, 0xE0, 0xE7 } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCancelStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000007, 0x000001A6, { { 0x21, 0xED, 0xF9, 0x71, 0xEF, 0x74, 0xD7, 0x9E, 0xF3, 0x02, 0xE5, 0x03, 0x06, 0xDE, 0xD5, 0x09 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000000A, 0x0000027A, { { 0xBD, 0x88, 0xB2, 0xA0, 0xAF, 0x8D, 0xFE, 0x5B, 0xAC, 0xDF, 0xB5, 0x9F, 0xA0, 0x23, 0x6E, 0xAE } } } },	// EoB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000006, 0x00000145, { { 0x22, 0x45, 0x35, 0xC6, 0x28, 0x00, 0x22, 0xAA, 0xD1, 0x15, 0x48, 0xE6, 0xE5, 0x62, 0x73, 0x37 } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseAbortStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000006, 0x00000178, { { 0xDD, 0xEC, 0x68, 0x6D, 0x2E, 0x10, 0x34, 0x94, 0x46, 0x25, 0xF9, 0xAB, 0x29, 0x27, 0x32, 0xA8 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000006, 0x00000145, { { 0x22, 0x45, 0x35, 0xC6, 0x28, 0x00, 0x22, 0xAA, 0xD1, 0x15, 0x48, 0xE6, 0xE5, 0x62, 0x73, 0x37 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsMainProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000067, 0x0000245E, { { 0xD9, 0xE0, 0x74, 0x9D, 0x43, 0x96, 0xDC, 0x3B, 0xDF, 0x90, 0x03, 0xDE, 0x91, 0xE6, 0xA0, 0x1E } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000068, 0x000025D9, { { 0x17, 0xEB, 0xAB, 0x4F, 0x95, 0xD1, 0x7F, 0xEB, 0xF4, 0x92, 0x42, 0xD1, 0xD2, 0xA8, 0xC4, 0xA8 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000069, 0x0000265B, { { 0x4C, 0xA9, 0x38, 0x28, 0xE1, 0xD0, 0xE3, 0x35, 0xBB, 0xDC, 0xFB, 0x6B, 0xAB, 0xB1, 0x62, 0x88 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsSaveLoadProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000077, 0x00002513, { { 0x70, 0xD9, 0x48, 0xC2, 0x3A, 0x38, 0x1D, 0xD0, 0x8B, 0x90, 0x08, 0x8D, 0x80, 0xF5, 0x24, 0x59 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000086, 0x00002D2F, { { 0x2B, 0x48, 0x5C, 0x78, 0xF9, 0xB9, 0xD6, 0xA8, 0x1D, 0xF4, 0x97, 0xAC, 0xF1, 0x09, 0x26, 0xA7 } } } }, // EOB1
	{ EN_ANY, kPlatformUnknown, { 0x000000A9, 0x00003850, { { 0xC3, 0x09, 0x7B, 0x18, 0xD6, 0x08, 0x0E, 0x2A, 0xB6, 0x66, 0x43, 0x14, 0xD7, 0x59, 0x34, 0xF7 } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x000000A2, 0x00003942, { { 0x6E, 0x10, 0x87, 0x4B, 0x80, 0xE8, 0x89, 0xC4, 0x31, 0xDC, 0xAC, 0xA9, 0xA3, 0x8D, 0x79, 0x41 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsOnOffProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000007, 0x00000178, { { 0x4D, 0xA7, 0x13, 0x00, 0x05, 0xF2, 0x44, 0xCB, 0xF7, 0x12, 0x72, 0x54, 0xDE, 0x35, 0x04, 0xEC } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000007, 0x00000178, { { 0xC7, 0x6F, 0x60, 0x72, 0x47, 0x89, 0x47, 0xF0, 0x29, 0x57, 0x45, 0x41, 0xD5, 0x80, 0x40, 0x7B } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsSpellsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x000001EF, 0x0000A0D0, { { 0xBA, 0x80, 0x5C, 0xAB, 0x93, 0x19, 0x53, 0x45, 0x17, 0xBC, 0x86, 0x5B, 0x1B, 0x01, 0x3E, 0x98 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x000001EA, 0x00009DE0, { { 0x00, 0xB0, 0x1F, 0xE7, 0x16, 0x48, 0x51, 0x25, 0xE5, 0xD8, 0xA1, 0x31, 0x13, 0x81, 0x8D, 0xB6 } } } }, // EOB1
	{ EN_ANY, kPlatformUnknown, { 0x000001FB, 0x0000A658, { { 0xAD, 0x6A, 0xFE, 0x13, 0xE5, 0xEA, 0x6A, 0xD1, 0xC9, 0x80, 0x1C, 0xEE, 0xD7, 0x2A, 0xF8, 0xB2 } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x00000222, 0x0000B1C9, { { 0x24, 0xC8, 0x9B, 0x51, 0xEE, 0x45, 0x14, 0xFC, 0x1B, 0xE4, 0x37, 0x8B, 0xEC, 0x94, 0xD9, 0x0B } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsRestProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x000000B3, 0x00003CED, { { 0x82, 0xF9, 0xA1, 0x74, 0xE6, 0x95, 0xA4, 0xFC, 0xE6, 0x5E, 0xB4, 0x43, 0x7D, 0x10, 0xFD, 0x12 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x000000B3, 0x00003BE4, { { 0x7F, 0xE8, 0xFE, 0xA4, 0xD9, 0x5C, 0x49, 0x66, 0x38, 0x8F, 0x84, 0xB8, 0xF5, 0x03, 0xCD, 0x70 } } } }, // EOB
	{ DE_DEU, kPlatformUnknown, { 0x000000C0, 0x000040A6, { { 0x05, 0x97, 0x45, 0x72, 0xE2, 0x33, 0xBE, 0xDE, 0x56, 0x26, 0x26, 0x15, 0x3A, 0x56, 0x93, 0xFD } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsDropProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000002E, 0x00000FCA, { { 0x88, 0xCB, 0xD2, 0xB3, 0xDA, 0x36, 0x97, 0x3D, 0xB8, 0x75, 0xFF, 0x36, 0xE1, 0x4E, 0xF4, 0x6D } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000039, 0x0000131E, { { 0x74, 0x0B, 0xE9, 0x04, 0x76, 0x26, 0xD2, 0xE8, 0x03, 0x48, 0x38, 0x18, 0xAC, 0x19, 0xBD, 0x7E } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000033, 0x0000119C, { { 0x8F, 0x2B, 0xC3, 0x01, 0xB2, 0xDE, 0x1F, 0xC6, 0x82, 0xC3, 0x58, 0x7C, 0x50, 0x23, 0x37, 0x65 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsExitProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000002B, 0x00000E3D, { { 0x1C, 0xD6, 0x39, 0xA9, 0xC7, 0x3D, 0x32, 0x4A, 0xF2, 0xAE, 0xEC, 0x08, 0x6F, 0xC7, 0xA6, 0x7B } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x0000002D, 0x00000E68, { { 0x4B, 0x2F, 0x65, 0x39, 0x69, 0xE7, 0x3D, 0x7B, 0x10, 0x15, 0x6F, 0x1F, 0xD8, 0x8E, 0xEA, 0x55 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000026, 0x00000CBD, { { 0x0C, 0x5D, 0xE4, 0xD2, 0x6F, 0xA3, 0x91, 0xDA, 0x5F, 0xE2, 0x57, 0x77, 0x74, 0x22, 0xE7, 0x85 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsStarveProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000003D, 0x0000150C, { { 0x40, 0xEB, 0x79, 0xC3, 0x99, 0x4C, 0xEA, 0xCD, 0x8A, 0xB4, 0x54, 0xB8, 0xAA, 0xEC, 0xAD, 0x4F } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000037, 0x00001296, { { 0x51, 0x3C, 0x90, 0x91, 0x4E, 0x1C, 0x73, 0x2F, 0x0C, 0x7A, 0x6D, 0x03, 0x1E, 0x54, 0x65, 0xF1 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000030, 0x00001057, { { 0xF3, 0x5E, 0xFC, 0xC3, 0x9D, 0xB5, 0xFE, 0x4E, 0x66, 0x9D, 0x6A, 0xC6, 0x61, 0xC8, 0x0A, 0x17 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsScribeProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000103, 0x000055E1, { { 0x1B, 0x56, 0xD2, 0x78, 0x3F, 0x67, 0x7A, 0x5B, 0xB6, 0x2B, 0x70, 0x3D, 0x6A, 0xBB, 0x08, 0x0A } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x0000010C, 0x00005B1C, { { 0xD7, 0xBF, 0x37, 0x21, 0xA2, 0x63, 0x8C, 0x6A, 0x02, 0x92, 0x13, 0x32, 0xD6, 0xA6, 0x1C, 0xDC } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000100, 0x0000560F, { { 0x69, 0x15, 0x2C, 0x2D, 0xE7, 0x40, 0x4A, 0xE0, 0x86, 0x0D, 0xC8, 0x66, 0x87, 0x1C, 0x27, 0x0B } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsDrop2Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000084, 0x00002ACE, { { 0xAB, 0x78, 0x42, 0x29, 0xFB, 0xC5, 0x34, 0x96, 0x9D, 0x8A, 0x21, 0x46, 0xE7, 0x6B, 0x06, 0xBA } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x0000008C, 0x00002D02, { { 0x11, 0x3F, 0x0C, 0xB2, 0xBF, 0xA7, 0x39, 0x23, 0xDC, 0x00, 0xB4, 0xEA, 0x5E, 0xFE, 0x40, 0xB7 } } } }, // EOB1
	{ EN_ANY, kPlatformUnknown, { 0x0000008E, 0x00002FFB, { { 0xCE, 0x7A, 0xCC, 0xA4, 0x02, 0x54, 0x1A, 0x78, 0xF1, 0xFC, 0xE6, 0x6C, 0x76, 0xCD, 0xFD, 0x9E } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x00000090, 0x000031CE, { { 0x01, 0x72, 0x59, 0xBE, 0x62, 0x72, 0xD4, 0x99, 0x76, 0xC9, 0x92, 0x0E, 0xE9, 0x1A, 0xCD, 0x65 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsHeadProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000021, 0x00000B27, { { 0x04, 0x06, 0x01, 0xF8, 0x50, 0x54, 0x11, 0x61, 0xFF, 0xB4, 0xE1, 0x97, 0xFA, 0x08, 0xAA, 0x1B } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000024, 0x00000CF5, { { 0x96, 0xD6, 0xB5, 0xB0, 0x2E, 0x71, 0xA4, 0x0A, 0x34, 0x41, 0x94, 0x02, 0x2F, 0xB0, 0x4C, 0x36 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000025, 0x00000D92, { { 0xE4, 0x73, 0x2D, 0x29, 0xAD, 0x30, 0xE5, 0x8D, 0xAE, 0xC6, 0xD7, 0xF5, 0x35, 0xD8, 0xA4, 0x98 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsPoisonProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000002E, 0x00001077, { { 0x14, 0x7E, 0xFC, 0xE0, 0x88, 0xFE, 0x86, 0xA8, 0x96, 0x94, 0xB1, 0x71, 0x90, 0x47, 0x2D, 0x78 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000036, 0x000013A2, { { 0x18, 0xD9, 0x1D, 0xE5, 0x3D, 0xFD, 0x52, 0xB6, 0x18, 0x17, 0x61, 0xE8, 0xA5, 0x32, 0x9F, 0xA8 } } } }, // EOB1
	{ EN_ANY, kPlatformUnknown, { 0x0000002D, 0x00001006, { { 0xD6, 0x0B, 0x11, 0x79, 0xAD, 0x61, 0x5B, 0x3A, 0x72, 0x7D, 0x53, 0x6F, 0xA9, 0x08, 0x73, 0xDC } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x00000035, 0x000013BE, { { 0x73, 0x38, 0x76, 0x2C, 0x42, 0x87, 0x43, 0x7E, 0x8E, 0x4C, 0x41, 0x57, 0x3F, 0x04, 0xBA, 0x11 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsMgcProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000020, 0x00000857, { { 0xD1, 0x9E, 0xBF, 0xF7, 0xCF, 0xF7, 0xD0, 0x94, 0x14, 0x56, 0xD2, 0x4F, 0x59, 0x91, 0x57, 0x52 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000020, 0x0000086C, { { 0x12, 0x36, 0x84, 0x2F, 0x00, 0xAD, 0x12, 0x42, 0x3A, 0xA2, 0xC5, 0xC9, 0x59, 0x90, 0x64, 0x5F } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000021, 0x0000090B, { { 0x26, 0xA7, 0x58, 0x7C, 0x0C, 0x9E, 0x67, 0xB9, 0x05, 0xE6, 0x91, 0x59, 0xE3, 0xDF, 0x9C, 0x52 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsPrefsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000041, 0x00001392, { { 0xB1, 0x7E, 0xE3, 0x73, 0xB2, 0xA2, 0x63, 0x39, 0x20, 0xE8, 0xF3, 0x38, 0x45, 0xB6, 0xAC, 0xC8 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000045, 0x000015F6, { { 0x53, 0xBA, 0x7E, 0x6D, 0x24, 0x88, 0x2C, 0x19, 0x10, 0x71, 0x6F, 0xAB, 0x85, 0x8E, 0x97, 0xF6 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x0000003D, 0x00001246, { { 0x03, 0xFB, 0x7C, 0x80, 0x33, 0x45, 0x6C, 0x27, 0x89, 0x7B, 0x7C, 0xAC, 0x7A, 0xE1, 0xDE, 0x49 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsRest2Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000004D, 0x00001744, { { 0x63, 0xA5, 0x6F, 0x09, 0x6F, 0x5E, 0x4B, 0x89, 0xFF, 0x33, 0x63, 0xCB, 0xAA, 0x04, 0x59, 0x63 } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x0000004D, 0x00001769, { { 0x2C, 0xA9, 0x7D, 0x4C, 0xC5, 0x13, 0xE2, 0xEB, 0x89, 0x6C, 0xAE, 0x25, 0xC3, 0x3E, 0x56, 0x7E } } } }, // EOB1
	{ EN_ANY, kPlatformUnknown, { 0x00000052, 0x000017F6, { { 0x7C, 0x49, 0xFC, 0x89, 0x90, 0x5D, 0xFF, 0x86, 0x86, 0xE9, 0xB2, 0x29, 0x60, 0xB2, 0x22, 0x7F } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x0000004C, 0x000014FF, { { 0x0C, 0x94, 0x6D, 0x5A, 0x42, 0x68, 0xE0, 0xDC, 0xCD, 0xB9, 0x1A, 0x4A, 0xC1, 0xCC, 0xE6, 0x91 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsRest3Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000002B, 0x00000DF4, { { 0x42, 0x90, 0x49, 0xA7, 0x2E, 0x61, 0x77, 0x7F, 0x9F, 0x53, 0xAD, 0x3C, 0x87, 0xE2, 0x0E, 0x36 } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x00000027, 0x00000D45, { { 0x8D, 0xAB, 0xBF, 0x57, 0xF3, 0x2C, 0x3F, 0x93, 0xBF, 0x33, 0x58, 0x2D, 0x97, 0x78, 0x71, 0x7F } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsRest4Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000029, 0x00000DEC, { { 0x1C, 0x86, 0x3D, 0x40, 0x2C, 0x5E, 0xCA, 0xA0, 0xA1, 0xB8, 0x23, 0x42, 0x9C, 0x6B, 0xFA, 0xBB } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x00000034, 0x00001238, { { 0xE9, 0x95, 0x27, 0x79, 0x1C, 0x0D, 0xF5, 0x94, 0x92, 0xFC, 0xCA, 0x22, 0x17, 0xA8, 0x36, 0x96 } } } }, // EOB1
	{ EN_ANY, kPlatformUnknown, { 0x0000002A, 0x00000DEB, { { 0x0E, 0xD3, 0xC5, 0xA9, 0x8B, 0x06, 0x57, 0xB0, 0x20, 0x1A, 0xEE, 0x42, 0x49, 0x2E, 0xA1, 0x50 } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x00000033, 0x00001189, { { 0x56, 0x1B, 0x6B, 0x00, 0x47, 0xFD, 0x56, 0xD3, 0x12, 0x03, 0x79, 0x7D, 0xFF, 0x83, 0xCF, 0xAA } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsDefeatProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000006D, 0x00002643, { { 0x94, 0xBA, 0xAC, 0xA4, 0x87, 0x6A, 0xEA, 0x7D, 0x98, 0x6E, 0x09, 0x82, 0xE0, 0x16, 0x65, 0x4F } } } }, // EOB1
	{ DE_DEU, kPlatformUnknown, { 0x0000006A, 0x00002456, { { 0xE0, 0x9A, 0x10, 0xE2, 0x73, 0x42, 0xF6, 0x79, 0xCB, 0x65, 0xA2, 0x50, 0xF0, 0x2B, 0xFD, 0x9B } } } }, // EOB1
	{ EN_ANY, kPlatformUnknown, { 0x00000056, 0x00001E4F, { { 0x97, 0x07, 0x5F, 0xA2, 0x0D, 0x58, 0xD2, 0xDF, 0xD6, 0x04, 0xA2, 0x16, 0x0B, 0x1F, 0x7E, 0x23 } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x00000042, 0x000016B1, { { 0xCA, 0x57, 0xDC, 0x2B, 0xC6, 0xC7, 0x78, 0x1E, 0x84, 0x0A, 0x10, 0x88, 0xCA, 0xCD, 0xFF, 0x89 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsTransferProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000114, 0x00005E09, { { 0xBF, 0xCE, 0x7F, 0xE4, 0x17, 0x15, 0xC6, 0x10, 0xDF, 0x16, 0xF9, 0x3C, 0xDA, 0x29, 0xA0, 0xA6 } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x000000D1, 0x00004811, { { 0x2E, 0x00, 0xD1, 0xA6, 0x9F, 0x53, 0xC5, 0x4B, 0x25, 0x4A, 0xAC, 0x9E, 0x11, 0x6C, 0x58, 0x5E } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsSpecProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000006F, 0x00002785, { { 0xAE, 0xC7, 0x88, 0x89, 0x39, 0xB8, 0xF7, 0xB4, 0xD5, 0x82, 0xBC, 0x46, 0xA1, 0xCB, 0x04, 0x1F } } } }, // EOB2
	{ DE_DEU, kPlatformUnknown, { 0x00000075, 0x00002871, { { 0xB4, 0x38, 0x0F, 0x94, 0x8B, 0xB1, 0x8D, 0xA3, 0xF8, 0xDA, 0x37, 0x75, 0x6F, 0x39, 0x3E, 0xB5 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuStringsSpellNoProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000006, 0x000000A5, { { 0x0D, 0x4A, 0x8B, 0x40, 0x70, 0x79, 0xCD, 0xB3, 0x0F, 0x5A, 0x5A, 0x3F, 0x6E, 0xE8, 0xF9, 0x74 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMenuYesNoStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000007, 0x000001EE, { { 0x8C, 0xF1, 0x35, 0x1F, 0xD6, 0x1F, 0xA4, 0xA1, 0xD6, 0xD6, 0x0A, 0x27, 0xB9, 0xFC, 0x9E, 0x62 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000008, 0x00000235, { { 0xC7, 0x06, 0xCF, 0xA8, 0xC0, 0xDE, 0xD4, 0x8C, 0x7F, 0xA2, 0x3A, 0xD3, 0x48, 0x51, 0x36, 0x89 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSpellLevelsMageProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001A, 0x00000042, { { 0x4F, 0xA3, 0x70, 0x0F, 0x6D, 0xB4, 0xC2, 0xAF, 0x12, 0xB4, 0x2E, 0x26, 0xEF, 0x0B, 0x37, 0x92 } } } }, // EOB1
	{ UNK_LANG, kPlatformUnknown, { 0x00000023, 0x00000074, { { 0xBE, 0x10, 0xFA, 0xD9, 0xB3, 0xB0, 0x4E, 0x73, 0xC9, 0xA1, 0xE2, 0xCE, 0xE8, 0xEC, 0x85, 0x0F } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSpellLevelsClericProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000019, 0x00000045, { { 0x9E, 0xDA, 0xF2, 0x94, 0x3E, 0x0B, 0xA0, 0x23, 0x08, 0x41, 0xD5, 0x3C, 0x61, 0x77, 0xFD, 0x78 } } } }, // EOB1
	{ UNK_LANG, kPlatformUnknown, { 0x0000001D, 0x00000066, { { 0xDB, 0x7F, 0x93, 0xE2, 0x2E, 0xCF, 0x69, 0xCC, 0x2A, 0xEF, 0x7C, 0x1E, 0x92, 0x6B, 0x51, 0x4E } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseNumSpellsClericProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000032, 0x0000004C, { { 0x87, 0xDD, 0xD0, 0xF8, 0x52, 0x84, 0x26, 0xC4, 0x9C, 0x5D, 0x0E, 0x46, 0x1A, 0xE8, 0x19, 0xD6 } } } }, // EOB1
	{ UNK_LANG, kPlatformUnknown, { 0x00000088, 0x00000114, { { 0xA0, 0xB7, 0x2F, 0xED, 0x50, 0xE7, 0xC6, 0x11, 0xC9, 0x25, 0xB2, 0xB9, 0x81, 0xFB, 0xD8, 0x59 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseNumSpellsWisAdjProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000028, 0x0000001D, { { 0xA5, 0xCA, 0x1D, 0x96, 0xAE, 0x89, 0xBC, 0x7A, 0x32, 0x50, 0xCE, 0x44, 0x5D, 0x93, 0x25, 0x4B } } } }, // EOB1
	{ UNK_LANG, kPlatformUnknown, { 0x00000040, 0x0000001D, { { 0x07, 0x31, 0x0D, 0x12, 0x55, 0x11, 0x11, 0xB6, 0x68, 0xC7, 0xEE, 0xDE, 0xC6, 0xED, 0x82, 0x5A } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseNumSpellsPalProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000012, { { 0x96, 0x7E, 0x17, 0x9E, 0xFD, 0x39, 0xC9, 0x3A, 0xB7, 0x3E, 0x8D, 0xA8, 0xED, 0xA3, 0x07, 0xEB } } } }, // EOB1
	{ UNK_LANG, kPlatformUnknown, { 0x00000088, 0x0000002F, { { 0x19, 0x1A, 0x9B, 0x42, 0xA0, 0x67, 0x10, 0x1A, 0xAC, 0x00, 0x0F, 0xF7, 0xBE, 0x04, 0x61, 0x36 } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseNumSpellsMageProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x0000005E, { { 0x61, 0x30, 0x1A, 0x74, 0x9B, 0x4C, 0x8C, 0x83, 0xD5, 0xE6, 0x39, 0x6E, 0xCA, 0x18, 0x16, 0x63 } } } }, // EOB1
	{ UNK_LANG, kPlatformUnknown, { 0x00000114, 0x00000102, { { 0x33, 0xEE, 0x32, 0x9C, 0xB2, 0xB3, 0x60, 0x66, 0x91, 0xE0, 0x90, 0x0E, 0x8F, 0xE1, 0xA5, 0x4A } } } }, // EOB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharGuiStringsHpProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000F, 0x00000352, { { 0x9C, 0x13, 0x3D, 0x2A, 0x68, 0x11, 0x81, 0xA4, 0x77, 0x54, 0x47, 0x43, 0xA1, 0xDA, 0x55, 0x50 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000000E, 0x000002AC, { { 0xBB, 0xD5, 0x36, 0xB2, 0x8A, 0x60, 0x78, 0x04, 0x46, 0x2D, 0x35, 0x59, 0x3E, 0x42, 0xB9, 0x83 } } } },	// EoB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000000E, 0x000002B8, { { 0x77, 0x76, 0xA0, 0x38, 0xE9, 0xB6, 0x0C, 0x43, 0xFE, 0x5A, 0x51, 0xC7, 0x1B, 0x59, 0xD3, 0x63 } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharGuiStringsWp1Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000A, 0x00000253, { { 0x1D, 0xF4, 0xB9, 0x56, 0xE6, 0x16, 0x7D, 0x08, 0xA4, 0x00, 0x1E, 0x1A, 0x60, 0x49, 0xE9, 0x29 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000000A, 0x00000236, { { 0x17, 0xEC, 0x54, 0xA0, 0x43, 0xFB, 0x52, 0x66, 0xC5, 0x44, 0x1B, 0xDF, 0x95, 0x47, 0x62, 0xB3 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharGuiStringsWp2Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000F, 0x00000371, { { 0x42, 0xF4, 0x52, 0x60, 0x20, 0xFC, 0x34, 0x94, 0x49, 0x1E, 0x67, 0x54, 0xB5, 0x6A, 0x97, 0x2A } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000010, 0x000003D6, { { 0x10, 0xD1, 0x77, 0x6C, 0xCD, 0x00, 0x94, 0xC7, 0xD0, 0x53, 0x47, 0x9F, 0x70, 0x77, 0x50, 0xD1 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharGuiStringsWrProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000014, 0x00000477, { { 0xAA, 0x26, 0xD5, 0xFD, 0xE6, 0x16, 0x53, 0x19, 0x39, 0x46, 0xEB, 0xCD, 0x88, 0xEC, 0x5E, 0xCB } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000015, 0x000004A4, { { 0x53, 0x54, 0x37, 0x35, 0x27, 0x1F, 0xB9, 0x09, 0x9C, 0xE9, 0x5E, 0x11, 0xBD, 0x8F, 0x15, 0xAE } } } },	// EoB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000011, 0x00000402, { { 0xE0, 0x92, 0xF4, 0x49, 0xB7, 0x74, 0xBB, 0xEB, 0x90, 0x0D, 0x75, 0x65, 0xBB, 0xF6, 0xB6, 0xE9 } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharGuiStringsSt1Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000003B, 0x0000104B, { { 0xC0, 0xD9, 0x0F, 0x7B, 0x6D, 0x17, 0x02, 0xBD, 0x7B, 0xB1, 0x76, 0x72, 0xA1, 0xEE, 0x5E, 0xAD } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000039, 0x00000F69, { { 0x09, 0x42, 0x35, 0x47, 0x48, 0x50, 0x05, 0x09, 0x3B, 0x81, 0x14, 0x01, 0xF9, 0xB5, 0x04, 0xB2 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharGuiStringsSt2Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000045, 0x000012E7, { { 0x49, 0x48, 0x30, 0x73, 0xDA, 0x42, 0xDB, 0xB9, 0xF4, 0x07, 0x00, 0x26, 0x96, 0x1F, 0x02, 0x4E } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000042, 0x0000114D, { { 0x88, 0x6E, 0x45, 0xF9, 0xAE, 0xEF, 0xE8, 0x54, 0x9C, 0xEF, 0xD2, 0x73, 0x78, 0x41, 0xD9, 0xAF } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharGuiStringsInProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000023, 0x000008CB, { { 0xF0, 0xE9, 0xF1, 0x05, 0x47, 0x3A, 0x5D, 0xCA, 0x9F, 0x75, 0x9D, 0x51, 0x9E, 0xEC, 0x9B, 0x67 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000020, 0x00000810, { { 0xF5, 0x39, 0x1E, 0x0E, 0x65, 0xEF, 0x09, 0xF2, 0x8D, 0x90, 0xC4, 0xF6, 0x8A, 0xED, 0xAD, 0xDF } } } },	// EoB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000023, 0x00000940, { { 0xAB, 0xF6, 0xE4, 0xD4, 0x07, 0x07, 0xDA, 0x92, 0x71, 0xE2, 0x73, 0x1F, 0x06, 0xE3, 0x12, 0xEB } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharStatusStrings7Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000022, 0x00000B95, { { 0xCB, 0xB7, 0x16, 0x77, 0x9C, 0xEB, 0x70, 0x83, 0xB2, 0xBE, 0xF7, 0x67, 0xB1, 0xE9, 0xD0, 0x5E } } } },	// EOB 1 + 2
	{ DE_DEU, kPlatformUnknown, { 0x0000002B, 0x00000EE3, { { 0xC8, 0x81, 0x23, 0xC3, 0x03, 0xBD, 0x4C, 0xF5, 0x41, 0x47, 0xFA, 0x32, 0xA0, 0x98, 0x0A, 0x8E } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000023, 0x00000C68, { { 0xF5, 0x55, 0x09, 0xD8, 0x73, 0xF8, 0xF0, 0xE3, 0x14, 0xCD, 0x78, 0x84, 0x58, 0xB0, 0x64, 0x5B } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharStatusStrings81Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000021, 0x00000B03, { { 0x44, 0xFC, 0xC2, 0x23, 0x4B, 0x78, 0xA8, 0xF8, 0xA5, 0x46, 0x5B, 0x89, 0x1F, 0x9D, 0x4E, 0xFA } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000020, 0x00000A52, { { 0x81, 0xDA, 0x22, 0x8A, 0xD3, 0x7D, 0xE7, 0xF5, 0x39, 0x6A, 0x62, 0x41, 0xE5, 0x8D, 0x45, 0x20 } } } },	// EOB 1
};

const ExtractEntrySearchData kEoBBaseCharStatusStrings82Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000023, 0x00000B0F, { { 0xBD, 0xED, 0xFE, 0xFD, 0x40, 0x95, 0x42, 0x21, 0x1F, 0x55, 0x67, 0x65, 0xA8, 0xC3, 0x99, 0xA1 } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x0000001A, 0x00000789, { { 0x8C, 0xF3, 0xB8, 0x3C, 0x6E, 0x85, 0xED, 0xD6, 0x2B, 0xD7, 0xAE, 0x8A, 0xFC, 0x25, 0x5E, 0x8F } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharStatusStrings9Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000001C, 0x00000952, { { 0x2F, 0x41, 0x17, 0x95, 0xF8, 0xC8, 0x4E, 0x88, 0xC6, 0x83, 0x38, 0x9B, 0xAB, 0x23, 0x48, 0xB9 } } } },	// EOB 1 + 2
	{ DE_DEU, kPlatformUnknown, { 0x0000001D, 0x0000099F, { { 0x5E, 0xB4, 0xBE, 0xA9, 0x0C, 0xB2, 0xF2, 0x4E, 0x63, 0xF8, 0x7B, 0x98, 0x67, 0x2D, 0xC9, 0xBF } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x0000001E, 0x00000A52, { { 0xD4, 0x65, 0x3F, 0x35, 0xDD, 0x8A, 0x33, 0x44, 0x2F, 0x8C, 0xAC, 0x2F, 0xEC, 0x96, 0x5B, 0x02 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharStatusStrings12Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000010, 0x00000503, { { 0x81, 0x22, 0xE9, 0x0F, 0xA5, 0xEA, 0xFE, 0xB5, 0xB6, 0x43, 0x36, 0x22, 0x87, 0x24, 0x2C, 0x40 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000014, 0x00000683, { { 0x5A, 0xF8, 0xAA, 0x16, 0x97, 0xBE, 0xD5, 0x22, 0xCE, 0x3F, 0xBC, 0x00, 0x44, 0xC1, 0x27, 0xD3 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharStatusStrings131Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000B, 0x0000027A, { { 0x70, 0x1A, 0x83, 0x35, 0x0A, 0x51, 0xEA, 0x27, 0x6E, 0xCD, 0xEB, 0xAD, 0x20, 0x74, 0x28, 0x7D } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000000C, 0x000002EE, { { 0xFE, 0xF9, 0x45, 0xE7, 0x89, 0x7B, 0xA4, 0x82, 0x90, 0x63, 0x91, 0x5B, 0x9E, 0x83, 0x80, 0x10 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseCharStatusStrings132Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000D, 0x00000286, { { 0x00, 0x1E, 0x11, 0xCC, 0x57, 0xFA, 0xEF, 0x2A, 0x0A, 0xFF, 0xFF, 0xE9, 0x3E, 0xA3, 0x62, 0x21 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000000A, 0x0000018A, { { 0x10, 0x54, 0x6F, 0xC3, 0x08, 0xC4, 0xD2, 0xBB, 0x34, 0x0A, 0x04, 0x65, 0x49, 0xFC, 0x5E, 0x15 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseLevelGainStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000002A, 0x00000CC9, { { 0xF2, 0x1F, 0xDF, 0xE0, 0xA5, 0x86, 0x46, 0xF4, 0xFC, 0x71, 0xB0, 0x22, 0x32, 0x46, 0x71, 0xB6 } } } },	// EOB 1
	{ DE_DEU, kPlatformUnknown, { 0x00000029, 0x00000D38, { { 0x18, 0xA3, 0xFD, 0xCC, 0xF2, 0x68, 0x18, 0x9E, 0x80, 0x5A, 0xC0, 0x22, 0xFD, 0x15, 0x83, 0x84 } } } },	// EOB 1
	{ EN_ANY, kPlatformUnknown, { 0x0000001C, 0x0000078C, { { 0x15, 0x70, 0x37, 0xE4, 0x0B, 0x50, 0x32, 0xCA, 0xAE, 0xF6, 0x81, 0xD0, 0x98, 0x9B, 0x36, 0x8A } } } },	// EOB 2
	{ DE_DEU, kPlatformUnknown, { 0x0000001F, 0x000008E3, { { 0x07, 0x2C, 0x51, 0x5E, 0x47, 0xAA, 0xCC, 0x27, 0x77, 0xD8, 0x17, 0x59, 0x6B, 0xBE, 0xF5, 0x87 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseExperienceTable0Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000C9E, { { 0xDB, 0x47, 0xD9, 0x0D, 0x6E, 0x68, 0x04, 0xE4, 0x17, 0xCB, 0x60, 0x89, 0x35, 0x3E, 0xA9, 0xEE } } } },	// EoB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000038, 0x00000E24, { { 0xBC, 0xF3, 0x96, 0x8A, 0xD5, 0x0C, 0xAA, 0x94, 0xBB, 0xB5, 0x08, 0x73, 0xF8, 0x5C, 0xF0, 0xA9 } } } }, // EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseExperienceTable1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000C80, { { 0x35, 0x45, 0x0D, 0x4F, 0xE0, 0x84, 0xA2, 0x1B, 0xB0, 0x0D, 0x60, 0x4D, 0x1D, 0xD5, 0x6C, 0x72 } } } },	// EoB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000038, 0x00000E6F, { { 0xBD, 0x3F, 0x42, 0x54, 0x75, 0x41, 0xAA, 0x58, 0x0D, 0xD8, 0xC0, 0x07, 0x63, 0x35, 0x83, 0x6B } } } }, // EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseExperienceTable2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x00000E10, { { 0xA5, 0x4D, 0xCB, 0xF3, 0x5F, 0x89, 0x71, 0x24, 0x6F, 0x70, 0xCA, 0x51, 0xAA, 0x1C, 0x0A, 0x97 } } } },	// EoB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000038, 0x00001149, { { 0xF9, 0xF1, 0x7E, 0x6B, 0xB2, 0xFE, 0x04, 0xC4, 0x29, 0x3D, 0xE3, 0x42, 0x5D, 0x92, 0x77, 0x53 } } } }, // EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseExperienceTable3Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000ADC, { { 0x42, 0x2E, 0x2E, 0xF5, 0xF8, 0x65, 0x69, 0x28, 0x50, 0x67, 0x43, 0xDF, 0x91, 0x67, 0x9B, 0x09 } } } },	// EoB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000038, 0x00000C94, { { 0x67, 0x09, 0x98, 0x19, 0x1F, 0x6B, 0x4D, 0xEB, 0x1D, 0x4D, 0x55, 0xA8, 0xFF, 0xD1, 0xAB, 0xE1 } } } }, // EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseExperienceTable4Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000DA7, { { 0x45, 0x58, 0x34, 0xC9, 0x09, 0x61, 0xD7, 0xE1, 0xF8, 0x68, 0x80, 0xBC, 0xEF, 0x7A, 0x24, 0x03 } } } },	// EoB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000038, 0x00000FE1, { { 0x26, 0x7F, 0x83, 0x53, 0x4A, 0xC6, 0xA2, 0x7B, 0xD2, 0xFB, 0x73, 0xB2, 0x08, 0x0A, 0xF7, 0xFD } } } }, // EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseWllFlagPresetProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000019, 0x00000BD9, { { 0xBE, 0x5A, 0xA6, 0xC8, 0xE4, 0x04, 0x4C, 0x32, 0x35, 0x61, 0x48, 0x73, 0x29, 0xA9, 0x99, 0x54 } } } },	// EoB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000019, 0x00000BC9, { { 0x56, 0xC0, 0x66, 0x4D, 0xE1, 0x3A, 0x27, 0x89, 0x9D, 0x73, 0x63, 0x93, 0x08, 0x2B, 0x13, 0xBC } } } }, // EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscShapeCoordsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000168, 0x0000F206, { { 0xB8, 0xDF, 0x10, 0xBB, 0x06, 0xA1, 0x46, 0xC6, 0xD2, 0xE3, 0xD7, 0x64, 0x4A, 0xC6, 0x40, 0xC0 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorScaleOffsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x0000010F, { { 0x7B, 0x7D, 0x03, 0xDE, 0x33, 0x95, 0xB8, 0xFD, 0xAD, 0x72, 0x44, 0x7D, 0x47, 0xFE, 0x04, 0x3D } } } },	// EoB1
	{ UNK_LANG, kPlatformPC, { 0x00000035, 0x00000139, { { 0x74, 0x63, 0x18, 0xE7, 0xAB, 0xA4, 0x22, 0xCF, 0x32, 0x19, 0x28, 0x9E, 0x7F, 0x97, 0xA7, 0x37 } } } },	// EoB2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorScaleMult1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000026, { { 0x5D, 0x17, 0xFB, 0x6A, 0x7F, 0x51, 0x55, 0xFB, 0x55, 0xB9, 0x50, 0xB0, 0x7F, 0xE4, 0xDF, 0x67 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorScaleMult2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000006, { { 0x98, 0xD8, 0xF8, 0x0C, 0x98, 0xC4, 0xF1, 0x87, 0x59, 0x32, 0x78, 0x31, 0xFA, 0x98, 0x8A, 0x43 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorScaleMult3Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000013, { { 0xEE, 0xB6, 0xA5, 0x6E, 0x0C, 0x8E, 0xAB, 0x38, 0xD9, 0x23, 0xC6, 0x21, 0xB3, 0x7E, 0x97, 0x78 } } } },	// EOB 1
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000019, { { 0x86, 0xD8, 0x04, 0xD2, 0x66, 0x6F, 0x43, 0x24, 0x2E, 0x93, 0xB9, 0xAE, 0xEB, 0x44, 0xCA, 0x48 } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorScaleMult4Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000006, { { 0x98, 0xD8, 0xF8, 0x0C, 0x98, 0xC4, 0xF1, 0x87, 0x59, 0x32, 0x78, 0x31, 0xFA, 0x98, 0x8A, 0x43 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorScaleMult5Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000020, { { 0x37, 0xA1, 0x0D, 0x64, 0xD6, 0x1E, 0xBA, 0xA3, 0xD9, 0x0A, 0x6C, 0xAB, 0x6B, 0xA3, 0x59, 0x24 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorScaleMult6Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000006, { { 0x98, 0xD8, 0xF8, 0x0C, 0x98, 0xC4, 0xF1, 0x87, 0x59, 0x32, 0x78, 0x31, 0xFA, 0x98, 0x8A, 0x43 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorType5OffsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x00000012, { { 0x73, 0xBB, 0x61, 0xD6, 0xA7, 0x75, 0xC8, 0x7B, 0xD6, 0xA4, 0x53, 0x1B, 0x54, 0xE9, 0x30, 0x3F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorXEProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x0000010F, { { 0x7B, 0x7D, 0x03, 0xDE, 0x33, 0x95, 0xB8, 0xFD, 0xAD, 0x72, 0x44, 0x7D, 0x47, 0xFE, 0x04, 0x3D } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorY1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x000000D7, { { 0x25, 0xAE, 0xF4, 0x99, 0xE8, 0x97, 0x47, 0xAE, 0x75, 0xF3, 0xA9, 0x70, 0x4C, 0x70, 0xF3, 0x11 } } } },	// EOB 1
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x000000D8, { { 0xB4, 0xAA, 0x0D, 0x91, 0x58, 0x22, 0x16, 0xCF, 0xC5, 0x9D, 0x8D, 0xA1, 0xB4, 0x40, 0x83, 0x0E } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorY3Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000058, { { 0xF0, 0x3C, 0x3B, 0x97, 0x10, 0x95, 0x89, 0x18, 0x3B, 0xA9, 0xE8, 0x77, 0x9B, 0x10, 0xDC, 0xF1 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorY4Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000076, { { 0x84, 0xB6, 0x8F, 0x7E, 0x9A, 0x17, 0xAC, 0x59, 0xB1, 0x4C, 0xDE, 0x11, 0xA6, 0x95, 0xE3, 0x76 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorY5Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x000000D9, { { 0x5D, 0x27, 0x1D, 0xD6, 0x5F, 0x98, 0xF9, 0x7D, 0x65, 0x7B, 0xE0, 0x67, 0x34, 0xA0, 0xE8, 0x30 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorY6Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x000000D9, { { 0x4D, 0x15, 0x4A, 0xF1, 0x17, 0x09, 0xC1, 0xA6, 0x08, 0x4A, 0xCD, 0xB2, 0x68, 0xC2, 0x59, 0x52 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorY7Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x000000DA, { { 0xA9, 0x24, 0x71, 0x8A, 0x18, 0x24, 0x6D, 0x0A, 0x65, 0x12, 0xBB, 0x1F, 0xE7, 0x95, 0xC5, 0xA4 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscDoorCoordsExtProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000048, 0x00000C8E, { { 0x2E, 0x0E, 0xB2, 0xAC, 0xE7, 0x0F, 0xDF, 0x38, 0xDF, 0x92, 0xB7, 0xB5, 0xA2, 0xFD, 0x40, 0x2D } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscItemPosIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x00000018, { { 0x74, 0x90, 0x47, 0xE6, 0xFB, 0xC0, 0x34, 0xDF, 0x92, 0x5B, 0xA1, 0xCB, 0x06, 0x33, 0xCA, 0x6B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscItemShpXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000024, 0x00000F2C, { { 0x9E, 0x22, 0x3F, 0x8F, 0x31, 0x83, 0xF7, 0x7C, 0x59, 0x60, 0x7C, 0x0A, 0xEB, 0xD2, 0x18, 0x85 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscItemPosUnkProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000012, 0x00000433, { { 0xA4, 0x7B, 0x08, 0x07, 0x81, 0xEA, 0x4F, 0x99, 0x77, 0x74, 0x93, 0x65, 0xBF, 0x0C, 0x3B, 0x94 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscItemTileIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000012, 0x00000D23, { { 0x0E, 0x17, 0xE1, 0x1F, 0x34, 0x7D, 0x30, 0xF6, 0xAE, 0x0B, 0xAC, 0x9D, 0x21, 0xB6, 0x97, 0xCC } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscItemShapeMapProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000005A, 0x00000B23, { { 0x41, 0x4A, 0x95, 0x7F, 0x82, 0x85, 0x28, 0x55, 0xD4, 0xD5, 0xD6, 0xD8, 0xA9, 0xAE, 0xF4, 0xC0 } } } },	// EoB 1
	{ UNK_LANG, kPlatformPC, { 0x00000071, 0x00000860, { { 0xEA, 0x5D, 0x33, 0xB6, 0x38, 0x30, 0x65, 0x29, 0x7F, 0x08, 0x89, 0x04, 0xC5, 0x97, 0x76, 0xCB } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscTelptrShpCoordsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000009C, 0x00000EBE, { { 0x2D, 0x1D, 0x74, 0x39, 0x29, 0xC3, 0x6F, 0x53, 0xD9, 0xA5, 0x4B, 0x9F, 0xD6, 0xDD, 0x73, 0xE9 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBasePortalSeqDataProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000007E, 0x000002D0, { { 0x18, 0x7E, 0x65, 0x17, 0x4C, 0xD2, 0xB5, 0x2E, 0x81, 0xF8, 0x1C, 0xAC, 0x37, 0x21, 0x62, 0x2A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseManDefProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000078, 0x000002CD, { { 0x33, 0x9B, 0x0C, 0x6A, 0x2E, 0x4F, 0xE9, 0x02, 0x7B, 0xEE, 0xF1, 0x04, 0xA3, 0xBA, 0xD4, 0xF3 } } } },	// EoB 1
	{ DE_DEU, kPlatformPC, { 0x00000078, 0x000002C4, { { 0x92, 0x20, 0x58, 0x5F, 0x44, 0x09, 0x0B, 0xF0, 0xDA, 0x09, 0xE2, 0x44, 0x0B, 0xB7, 0x95, 0x96 } } } },	// EoB 1
	{ EN_ANY, kPlatformPC, { 0x000000C8, 0x00000834, { { 0x18, 0xEA, 0x33, 0xB7, 0x4B, 0x72, 0x23, 0x8D, 0x0E, 0x9F, 0x4E, 0xF5, 0x09, 0xA3, 0x9C, 0xEA } } } },	// EoB 2
	{ DE_DEU, kPlatformPC, { 0x000000C8, 0x00000622, { { 0xFE, 0x1D, 0x94, 0x3A, 0x0B, 0x17, 0x89, 0xEF, 0x60, 0x18, 0xB2, 0x43, 0x7A, 0x02, 0xDB, 0x61 } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseManWordProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000000E0, 0x00005134, { { 0x68, 0x9C, 0x19, 0x2B, 0x5F, 0x38, 0x36, 0x41, 0xA7, 0x7E, 0xB7, 0x51, 0x41, 0x60, 0x1D, 0x67 } } } },	// EoB 1
	{ DE_DEU, kPlatformPC, { 0x000000EA, 0x00005458, { { 0xEC, 0x14, 0x11, 0xE9, 0x19, 0xFD, 0xF8, 0xFC, 0xA8, 0x46, 0x3D, 0xCD, 0x56, 0x08, 0xC3, 0x4A } } } },	// EoB 1
	{ EN_ANY, kPlatformPC, { 0x0000017E, 0x00008B64, { { 0x66, 0x38, 0x09, 0x5B, 0x2E, 0x50, 0x54, 0x43, 0x1C, 0xEC, 0x56, 0x3B, 0x72, 0x39, 0xF9, 0xC3 } } } },	// EoB 2
	{ DE_DEU, kPlatformPC, { 0x0000015B, 0x00007C37, { { 0x44, 0xA3, 0x32, 0x88, 0x9F, 0x63, 0x28, 0xA0, 0xBD, 0x00, 0xF1, 0x08, 0xCA, 0xE5, 0xFE, 0x5F } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseManPromptProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000041, 0x000013AC, { { 0x40, 0x2B, 0xB5, 0x99, 0xEF, 0x8F, 0x3C, 0x9F, 0xB1, 0x5A, 0xBE, 0xE4, 0x80, 0x8E, 0xBB, 0x96 } } } },	// EoB 1
	{ DE_DEU, kPlatformPC, { 0x00000048, 0x000015A5, { { 0x0B, 0xB4, 0x9E, 0xAD, 0xB3, 0x56, 0x75, 0xC1, 0xAE, 0x29, 0xF7, 0xB5, 0x82, 0x14, 0xD1, 0x27 } } } },	// EoB 1
	{ EN_ANY, kPlatformPC, { 0x00000041, 0x000013AC, { { 0x40, 0x2B, 0xB5, 0x99, 0xEF, 0x8F, 0x3C, 0x9F, 0xB1, 0x5A, 0xBE, 0xE4, 0x80, 0x8E, 0xBB, 0x96 } } } },	// EoB 2
	{ DE_DEU, kPlatformPC, { 0x0000005C, 0x00001D08, { { 0x10, 0xCE, 0x2D, 0xED, 0xA9, 0xA0, 0x7C, 0xA1, 0x91, 0x3F, 0xD8, 0x43, 0x03, 0x53, 0x97, 0xCA } } } },	// EoB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscMonsterFrmOffsTbl1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00001000, { { 0x98, 0x27, 0x57, 0x25, 0x3B, 0x04, 0x7D, 0x14, 0x3A, 0xD4, 0xA2, 0x5D, 0xBA, 0x04, 0x45, 0xAC } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDscMonsterFrmOffsTbl2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000828, { { 0x7E, 0x8A, 0x0C, 0xEB, 0x5C, 0xBC, 0x6C, 0xBD, 0xD2, 0x48, 0x08, 0xCC, 0xF7, 0x7B, 0x81, 0x03 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseInvSlotXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000036, 0x000010BF, { { 0x50, 0x6E, 0x67, 0x2B, 0x7D, 0x6C, 0xF2, 0x21, 0x73, 0xA2, 0xD5, 0xBB, 0xCE, 0x3B, 0x71, 0xAA } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseInvSlotYProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000001B, 0x00000A5B, { { 0x47, 0x55, 0x7D, 0x84, 0x45, 0x91, 0xC4, 0x44, 0x10, 0xD5, 0x39, 0xC4, 0xC8, 0x4F, 0x01, 0xA4 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSlotValidationFlagsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000036, 0x00001F6B, { { 0x87, 0x4F, 0x9A, 0x97, 0x20, 0x20, 0xB2, 0xA6, 0xF7, 0xC2, 0x5F, 0xAA, 0x17, 0xEA, 0xB4, 0x50 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseProjectileWeaponTypesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000008, 0x0000061C, { { 0x05, 0x55, 0xA6, 0xD1, 0x3C, 0x12, 0x84, 0xDA, 0xA9, 0x33, 0xCF, 0x07, 0x05, 0x2A, 0xB2, 0x29 } } } }, // EOB 1
	{ UNK_LANG, kPlatformPC, { 0x0000000F, 0x00000829, { { 0x9F, 0x6A, 0x13, 0x8A, 0xA7, 0x40, 0xE8, 0x40, 0x2E, 0x87, 0x49, 0x6B, 0x67, 0xED, 0xE8, 0xCE } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseWandTypesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000007, 0x00000162, { { 0xDB, 0x5D, 0x34, 0x70, 0x41, 0xAB, 0x8F, 0x75, 0xC8, 0x61, 0x8E, 0x44, 0x82, 0xCF, 0x28, 0x03 } } } }, // EOB 1
	{ UNK_LANG, kPlatformPC, { 0x00000008, 0x00000175, { { 0x01, 0xC2, 0xF0, 0xC6, 0x1C, 0xD0, 0x14, 0xD9, 0xB8, 0xF5, 0x9C, 0xFC, 0x22, 0xE4, 0xA0, 0xA7 } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseDrawObjPosIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000014, 0x00000028, { { 0x44, 0x46, 0x8C, 0x94, 0x76, 0x24, 0x08, 0xC7, 0x1F, 0x1B, 0x10, 0xD7, 0xDF, 0x18, 0x6C, 0x0D } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseFlightObjFlipIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x00000008, { { 0xEB, 0xF0, 0x27, 0x7E, 0xA8, 0x09, 0x3A, 0x95, 0x3B, 0x71, 0x2A, 0x43, 0x2E, 0xCF, 0x22, 0x0B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseFlightObjShpMapProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000058, 0x000051BD, { { 0xC3, 0xD2, 0xD1, 0xE5, 0x78, 0xEE, 0xA7, 0xAA, 0x71, 0xD1, 0xDD, 0xDF, 0x40, 0xBB, 0xAF, 0x66 } } } },	// EOB 1
	{ UNK_LANG, kPlatformPC, { 0x0000002D, 0x000025E6, { { 0x64, 0x26, 0x3D, 0xDC, 0x6C, 0x1A, 0xFC, 0x36, 0x9E, 0x5A, 0xBF, 0x64, 0xAD, 0xF4, 0xA3, 0x5D } } } },	// EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseFlightObjSclIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000048, 0x00002A0E, { { 0xAC, 0xBB, 0x7D, 0x73, 0x98, 0xF4, 0x1E, 0x4A, 0x77, 0xF0, 0x98, 0x75, 0x11, 0xBF, 0xF7, 0xD5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseBookNumbersProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000020, 0x00000AC8, { { 0x35, 0x05, 0x37, 0x4C, 0x05, 0x74, 0x04, 0x08, 0xAD, 0xA3, 0x64, 0xBF, 0xC0, 0x67, 0xF2, 0xF7 } } } },
	{ DE_DEU, kPlatformPC, { 0x00000028, 0x00000E5D, { { 0x80, 0x98, 0x05, 0x54, 0x84, 0x90, 0xD3, 0xB3, 0x9B, 0xFB, 0x8F, 0xB9, 0xA0, 0x43, 0xAA, 0xFD } } } },
	{ EN_ANY, kPlatformPC, { 0x00000020, 0x00000AC8, { { 0x35, 0x05, 0x37, 0x4C, 0x05, 0x74, 0x04, 0x08, 0xAD, 0xA3, 0x64, 0xBF, 0xC0, 0x67, 0xF2, 0xF7 } } } },
	{ DE_DEU, kPlatformPC, { 0x00000022, 0x00000BCA, { { 0x93, 0x0E, 0xE0, 0x6D, 0xDD, 0x40, 0xBC, 0x89, 0x67, 0xBD, 0x8A, 0xCB, 0xD2, 0xCF, 0x78, 0x8D } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMageSpellsListProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000122, 0x00006304, { { 0xD7, 0x14, 0x28, 0x83, 0x04, 0xC3, 0x42, 0x5A, 0x15, 0x49, 0x91, 0x12, 0x1D, 0x49, 0x17, 0x5B } } } },
	{ DE_DEU, kPlatformPC, { 0x0000013A, 0x00007155, { { 0x94, 0x45, 0xB9, 0x15, 0x57, 0x6E, 0xC6, 0x70, 0x66, 0x5F, 0xA7, 0x90, 0xA0, 0xC7, 0xC9, 0xE9 } } } },
	{ EN_ANY, kPlatformPC, { 0x00000195, 0x00008AC0, { { 0x55, 0xB8, 0x75, 0x35, 0x09, 0x23, 0x83, 0x11, 0x22, 0xF8, 0x23, 0x1E, 0x8F, 0x08, 0x57, 0x66 } } } },
	{ DE_DEU, kPlatformPC, { 0x0000019A, 0x0000929F, { { 0xB3, 0xA0, 0x2D, 0x3B, 0xF3, 0x72, 0x9B, 0x75, 0xA3, 0xC4, 0xD8, 0x72, 0x4B, 0xDE, 0x69, 0x82 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseClericSpellsListProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000013B, 0x00006BE6, { { 0x34, 0x63, 0x0B, 0xBA, 0xED, 0xC2, 0x9B, 0x31, 0xC3, 0x65, 0x51, 0xFF, 0xEF, 0xD8, 0x25, 0x92 } } } },
	{ DE_DEU, kPlatformPC, { 0x0000016D, 0x00007E74, { { 0x6E, 0xDE, 0x28, 0xE6, 0x13, 0x3D, 0xA6, 0x42, 0x80, 0xAB, 0xE7, 0xED, 0xAD, 0xC8, 0x62, 0x48 } } } },
	{ EN_ANY, kPlatformPC, { 0x00000164, 0x000079B3, { { 0x93, 0x16, 0x25, 0xFB, 0x76, 0xFF, 0xBC, 0x70, 0x9A, 0xB7, 0x93, 0xFC, 0x2E, 0xC3, 0x61, 0x7F } } } },
	{ DE_DEU, kPlatformPC, { 0x0000018B, 0x00008BB1, { { 0x8C, 0x21, 0xED, 0xE0, 0x1F, 0xF1, 0xDB, 0x72, 0xC4, 0x46, 0x36, 0x50, 0x16, 0xD5, 0xA8, 0x68 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSpellNamesProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000029A, 0x0000F1C8, { { 0xCA, 0xE1, 0x30, 0xDC, 0xAB, 0xD1, 0x87, 0xE8, 0x51, 0xA2, 0xA2, 0x1C, 0x23, 0x4A, 0x34, 0x58 } } } },
	{ DE_DEU, kPlatformPC, { 0x000002D3, 0x0001080D, { { 0x5F, 0xDB, 0x9E, 0x48, 0x30, 0x03, 0xE1, 0x8E, 0xC7, 0xDC, 0x98, 0x10, 0xCE, 0xA1, 0x28, 0x31 } } } },
	{ EN_ANY, kPlatformPC, { 0x00000366, 0x00013B1A, { { 0x15, 0xCB, 0x0E, 0xA9, 0x4E, 0x78, 0x30, 0x99, 0xA1, 0xCF, 0xF7, 0x05, 0xAB, 0x00, 0x66, 0x82 } } } },
	{ DE_DEU, kPlatformPC, { 0x000003BA, 0x0001626B, { { 0x0E, 0x4F, 0xF6, 0xFB, 0x78, 0x5E, 0x03, 0xE7, 0x82, 0xC4, 0xE2, 0x7B, 0xD9, 0xB2, 0xD7, 0xB2 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicStrings1Provider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000084, 0x000029B0, { { 0xC6, 0x90, 0x19, 0x61, 0xA1, 0x66, 0xF6, 0x03, 0x7A, 0x1F, 0x10, 0x00, 0xCA, 0x8F, 0x69, 0x3B } } } },
	{ DE_DEU, kPlatformPC, { 0x0000009D, 0x000033E4, { { 0x4B, 0xCF, 0x40, 0xCE, 0x0F, 0x86, 0x98, 0x36, 0x03, 0x59, 0xFE, 0x32, 0xFA, 0x4C, 0x14, 0x75 } } } },
	{ EN_ANY, kPlatformPC, { 0x00000085, 0x000029BD, { { 0xAB, 0x22, 0x4A, 0x70, 0xBB, 0x29, 0xB8, 0xBD, 0xAF, 0xC5, 0x0D, 0x1F, 0x23, 0x38, 0xBD, 0x06 } } } },
	{ DE_DEU, kPlatformPC, { 0x0000008C, 0x00002D68, { { 0x4B, 0x0A, 0x09, 0x22, 0xF7, 0x77, 0x82, 0x4B, 0xFE, 0x0B, 0xF1, 0x8F, 0x1C, 0xEA, 0x1A, 0x0C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicStrings2Provider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000051, 0x00001AD6, { { 0x28, 0x18, 0x2B, 0xF0, 0x0E, 0xC6, 0xEB, 0x01, 0xB0, 0x9A, 0x0A, 0x65, 0x05, 0xCB, 0x8F, 0x41 } } } },
	{ DE_DEU, kPlatformPC, { 0x0000004F, 0x00001A82, { { 0x77, 0x85, 0x17, 0x25, 0x07, 0x72, 0x4A, 0x7F, 0x4F, 0x39, 0x6C, 0xDD, 0xB6, 0x70, 0x11, 0x02 } } } },
	{ EN_ANY, kPlatformPC, { 0x00000090, 0x00002E35, { { 0x39, 0xD7, 0xA3, 0x21, 0xF0, 0xB7, 0x93, 0x9D, 0xDD, 0xEE, 0x33, 0xC2, 0x05, 0xE6, 0xE3, 0x63 } } } },
	{ DE_DEU, kPlatformPC, { 0x000000A1, 0x0000365C, { { 0x9A, 0x2D, 0xDB, 0x38, 0xB3, 0xF4, 0x0E, 0xF4, 0x36, 0x87, 0x60, 0xAE, 0xF8, 0x7E, 0xCA, 0x8A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicStrings3Provider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000008D, 0x00002DC8, { { 0x35, 0x5E, 0xDD, 0x32, 0x2D, 0x55, 0x1E, 0xBC, 0x93, 0x49, 0x55, 0x48, 0x8F, 0xCD, 0x87, 0xEB } } } },
	{ DE_DEU, kPlatformPC, { 0x000000A8, 0x0000381C, { { 0x12, 0x95, 0x55, 0x57, 0x2B, 0xA0, 0x1A, 0x75, 0xD3, 0x43, 0xFF, 0x3E, 0x00, 0xB6, 0xEC, 0x35 } } } },
	{ EN_ANY, kPlatformPC, { 0x00000088, 0x00002CD4, { { 0xD8, 0xBA, 0x5D, 0x14, 0x92, 0x84, 0x5A, 0x07, 0xC6, 0x76, 0xDF, 0x11, 0x1D, 0x84, 0x7A, 0x98 } } } },
	{ DE_DEU, kPlatformPC, { 0x00000081, 0x00002B14, { { 0xC8, 0xB7, 0x77, 0xBC, 0x3A, 0xB6, 0xDC, 0xB7, 0x00, 0xF3, 0x06, 0xEB, 0x77, 0x10, 0x7C, 0x7E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicStrings4Provider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000017, 0x0000071C, { { 0x96, 0x50, 0xA8, 0x08, 0x1B, 0x2D, 0x0C, 0xF6, 0x90, 0x6A, 0xE7, 0x9F, 0x65, 0xCC, 0x71, 0xA0 } } } },
	{ DE_DEU, kPlatformPC, { 0x0000001B, 0x00000840, { { 0xA2, 0xCF, 0x81, 0x3E, 0x87, 0xA8, 0x10, 0x1B, 0x44, 0x8D, 0x5B, 0x8B, 0xAE, 0x23, 0x30, 0xD3 } } } },
	{ EN_ANY, kPlatformPC, { 0x0000000C, 0x000003A5, { { 0x72, 0x64, 0xBD, 0x1C, 0xED, 0x05, 0x28, 0xFC, 0x94, 0x4B, 0x8F, 0x3C, 0x38, 0x08, 0x77, 0xED } } } },
	{ DE_DEU, kPlatformPC, { 0x00000010, 0x0000054E, { { 0xD9, 0x97, 0xA8, 0x24, 0x27, 0x7B, 0x01, 0x3F, 0x03, 0xBA, 0x2A, 0x43, 0x81, 0x8F, 0x97, 0x03 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicStrings6Provider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000029, 0x00000DA4, { { 0x5C, 0x6F, 0xA1, 0xC2, 0x56, 0xDE, 0xFE, 0xD5, 0x01, 0xFB, 0x65, 0x00, 0x24, 0xD1, 0x49, 0x7B } } } },
	{ DE_DEU, kPlatformPC, { 0x00000032, 0x00001211, { { 0x13, 0xBC, 0xF1, 0x03, 0x49, 0xDB, 0x16, 0xA5, 0xC3, 0x7C, 0xBF, 0x14, 0x8F, 0x40, 0x07, 0x8E } } } },
	{ EN_ANY, kPlatformPC, { 0x00000030, 0x00000FF5, { { 0xE4, 0x2B, 0xB9, 0xF0, 0x26, 0x3D, 0x30, 0xCD, 0xEF, 0xCD, 0xF5, 0xC0, 0x4E, 0xA4, 0xC4, 0x92 } } } },
	{ DE_DEU, kPlatformPC, { 0x00000029, 0x00000E6D, { { 0xE1, 0xBD, 0x4B, 0x42, 0x17, 0xA2, 0xB6, 0x6C, 0xF2, 0x7F, 0xEB, 0x41, 0x2C, 0x82, 0x8C, 0x76 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicStrings7Provider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000014, 0x00000406, { { 0xBD, 0xE1, 0x0A, 0x75, 0xD1, 0x18, 0xF7, 0x08, 0x2D, 0x2B, 0x65, 0x36, 0xA7, 0x59, 0x2E, 0x13 } } } },
	{ DE_DEU, kPlatformPC, { 0x0000000F, 0x000001E5, { { 0x1F, 0xC9, 0x46, 0x8B, 0x41, 0xAD, 0xAD, 0x2B, 0x5A, 0xA9, 0xAB, 0x94, 0x9A, 0x1E, 0x36, 0xAC } } } },
	{ EN_ANY, kPlatformPC, { 0x00000065, 0x000021AF, { { 0x76, 0x35, 0xAE, 0x1D, 0xC2, 0x54, 0x36, 0x11, 0x4D, 0x3E, 0x96, 0x11, 0xB2, 0xDC, 0x15, 0x20 } } } },
	{ DE_DEU, kPlatformPC, { 0x0000006F, 0x000026BA, { { 0xC9, 0x46, 0xD7, 0xF3, 0xF2, 0x5F, 0xF4, 0xB1, 0x22, 0xC8, 0x30, 0x16, 0x8E, 0x75, 0x4D, 0xA8 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicStrings8Provider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000056, 0x00001C95, { { 0x7E, 0x43, 0x73, 0xEC, 0x94, 0x0D, 0xF8, 0x1B, 0xF3, 0x1A, 0x62, 0x19, 0x96, 0x6A, 0x2C, 0xB5 } } } },
	{ DE_DEU, kPlatformPC, { 0x00000061, 0x0000213B, { { 0xE2, 0x3B, 0xA7, 0xB7, 0xE6, 0xA5, 0x0D, 0x0F, 0xE0, 0x94, 0x9B, 0xAE, 0xE1, 0x11, 0x97, 0x93 } } } },
	{ EN_ANY, kPlatformPC, { 0x00000085, 0x00002C0E, { { 0x6A, 0xEC, 0xF2, 0x5F, 0xA6, 0x3F, 0xB1, 0x1A, 0x74, 0x49, 0x5A, 0x47, 0xB0, 0x7A, 0xE6, 0x99 } } } },
	{ DE_DEU, kPlatformPC, { 0x00000096, 0x0000342E, { { 0x83, 0x48, 0x3B, 0xED, 0x73, 0x02, 0x03, 0xCA, 0xA9, 0x4D, 0x40, 0x0F, 0xDE, 0x17, 0x7D, 0x40 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseExpObjectTlModeProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000012, 0x0000000C, { { 0x98, 0x29, 0x54, 0xCD, 0xED, 0xAC, 0x7B, 0x61, 0x8D, 0x4F, 0x19, 0xE8, 0xA6, 0xB1, 0x51, 0x80 } } } },
	EXTRACT_END_ENTRY
};
const ExtractEntrySearchData kEoBBaseExpObjectTblIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000009, 0x00000005, { { 0xFE, 0xEA, 0xC4, 0x54, 0x62, 0x7E, 0x43, 0x6E, 0x89, 0x48, 0x03, 0xE7, 0x47, 0xBF, 0x7D, 0x9D } } } }, // EOB 1
	{ UNK_LANG, kPlatformPC, { 0x0000000E, 0x00000004, { { 0x63, 0x27, 0x19, 0x17, 0xBD, 0xC3, 0x8A, 0xA7, 0x1E, 0xF7, 0xD1, 0x78, 0x39, 0x3B, 0xD4, 0x4F } } } }, // EOB 2
	EXTRACT_END_ENTRY
};
const ExtractEntrySearchData kEoBBaseExpObjectShpStartProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000034, { { 0x27, 0xC5, 0x09, 0x97, 0x8E, 0xD4, 0xF1, 0x8D, 0x77, 0xEB, 0x1D, 0x34, 0x55, 0xB2, 0x48, 0x38 } } } },
	EXTRACT_END_ENTRY
};
const ExtractEntrySearchData kEoBBaseExpObjectTbl1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000D, 0x0000005D, { { 0x49, 0xC4, 0x47, 0x55, 0xDC, 0x25, 0x08, 0x03, 0x3D, 0x23, 0xAD, 0x09, 0x5F, 0x9C, 0x34, 0x06 } } } },
	EXTRACT_END_ENTRY
};
const ExtractEntrySearchData kEoBBaseExpObjectTbl2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000A, 0x0000005C, { { 0xAB, 0x6A, 0x97, 0x35, 0xCC, 0x13, 0xC4, 0x17, 0x0B, 0xF2, 0xD3, 0xFD, 0xA2, 0x1C, 0x6C, 0xA8 } } } },
	EXTRACT_END_ENTRY
};
const ExtractEntrySearchData kEoBBaseExpObjectTbl3Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000B, 0x00000032, { { 0x59, 0x23, 0xB9, 0xBE, 0x0E, 0xFA, 0xEB, 0xDD, 0x82, 0x68, 0x5B, 0xB0, 0xBE, 0x9B, 0x1D, 0x8E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseExpObjectYProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000008, 0x0000016C, { { 0xCF, 0x5B, 0x04, 0xAB, 0x1A, 0xAF, 0xDD, 0x56, 0xAC, 0xF6, 0x23, 0x86, 0x33, 0x06, 0x5A, 0xC6 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkDefStepsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000008, 0x000002FD, { { 0xB5, 0x6F, 0x31, 0x5F, 0xC6, 0x47, 0xE9, 0x23, 0x0E, 0x73, 0xBF, 0x77, 0xC7, 0xEE, 0xDB, 0x27 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkDefSubStepsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x000000FF, { { 0x18, 0x27, 0x73, 0x45, 0x26, 0x58, 0x81, 0x82, 0x70, 0x86, 0x7A, 0x0D, 0xDE, 0xC1, 0x08, 0x52 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkDefShiftProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x0000000C, { { 0xCC, 0xDC, 0x78, 0xF9, 0xFE, 0x88, 0xF3, 0x87, 0xFD, 0x08, 0xE8, 0x8A, 0x38, 0xD5, 0x4C, 0x53 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkDefAddProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000008, 0x0000007F, { { 0x7F, 0x86, 0x2E, 0x14, 0xDB, 0x36, 0xED, 0x99, 0xD9, 0xCE, 0xAF, 0x11, 0xC2, 0x89, 0x21, 0x6B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkDefXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000C, 0x000000A5, { { 0x77, 0xD7, 0xE0, 0x2D, 0xD4, 0x25, 0x94, 0x6E, 0x59, 0x3B, 0xAF, 0x9B, 0x16, 0x4F, 0x6D, 0x4C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkDefYProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x00000138, { { 0xB9, 0xA2, 0x72, 0x01, 0x2A, 0xD7, 0x61, 0xAB, 0x02, 0x57, 0x87, 0xC8, 0x86, 0x83, 0xDF, 0xB3 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkOfFlags1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000002C, 0x00000BF4, { { 0x94, 0x8C, 0x1B, 0x77, 0xBF, 0x3A, 0x51, 0x17, 0x89, 0x16, 0xD0, 0x74, 0x95, 0xBD, 0x85, 0x98 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkOfFlags2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000040, 0x000003FC, { { 0x40, 0x13, 0x5A, 0x9D, 0xBD, 0x29, 0x2E, 0x9C, 0xC1, 0xE7, 0xD4, 0xC9, 0x26, 0xFA, 0xF2, 0x70 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkOfShiftProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x000000F0, { { 0xC5, 0xC8, 0x91, 0x7E, 0x78, 0x2F, 0xF1, 0xE5, 0xE0, 0x06, 0xB2, 0x39, 0xDC, 0x0D, 0x7A, 0x5F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkOfXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x00000528, { { 0x58, 0xE6, 0x24, 0x6A, 0xD3, 0xA4, 0xEF, 0x58, 0x4A, 0x9C, 0x32, 0x31, 0x4C, 0x61, 0xBC, 0x1C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSparkOfYProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x000002D4, { { 0x74, 0x31, 0xFE, 0x7C, 0x38, 0x16, 0x0C, 0x05, 0x64, 0xAB, 0x8A, 0x69, 0xEA, 0x66, 0x29, 0x2F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseSpellPropertiesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000003EF, 0x0000BE7A, { { 0x10, 0xEA, 0x14, 0x26, 0xE2, 0xFC, 0xA1, 0xCB, 0xD9, 0x80, 0xFE, 0x9F, 0x69, 0x58, 0x4A, 0xCA } } } },
	{ UNK_LANG, kPlatformPC, { 0x000003EF, 0x00008FCE, { { 0xC9, 0x36, 0xDD, 0x7B, 0x05, 0x6E, 0x92, 0xBA, 0x2B, 0x39, 0x87, 0xA7, 0x3A, 0x7E, 0xB0, 0xAD } } } },
	{ UNK_LANG, kPlatformPC, { 0x000006D6, 0x0000CA78, { { 0xEB, 0x3B, 0x9F, 0xFD, 0x4E, 0x3F, 0x5C, 0xDE, 0xC6, 0xBA, 0xFE, 0x83, 0xB4, 0x10, 0x6D, 0x95 } } } },
	{ UNK_LANG, kPlatformPC, { 0x000006D6, 0x0000EC32, { { 0x52, 0xAE, 0x4D, 0xC2, 0x24, 0xC8, 0xD3, 0xBE, 0x09, 0x45, 0x98, 0x38, 0x17, 0x7D, 0xFF, 0xE4 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMagicFlightPropsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000060, 0x0000166F, { { 0x38, 0x30, 0xCA, 0x07, 0x64, 0xBA, 0xC4, 0xA4, 0x4F, 0x75, 0xB4, 0x84, 0x3A, 0x92, 0xFD, 0xE3 } } } },
	{ UNK_LANG, kPlatformPC, { 0x00000038, 0x00000DDC, { { 0x23, 0x32, 0x8D, 0x34, 0x4F, 0x72, 0x37, 0xE1, 0x0C, 0x1B, 0x47, 0x17, 0x5D, 0xDF, 0xDB, 0xF5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseTurnUndeadEffectProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000008C, 0x00002E8B, { { 0x96, 0x15, 0x61, 0x12, 0x43, 0xCF, 0x3A, 0x84, 0x1A, 0x89, 0xB5, 0x32, 0x0D, 0xB3, 0x20, 0x67 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseBurningHandsDestProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000008, 0x0000000C, { { 0x61, 0xD7, 0xAB, 0xE1, 0x56, 0x54, 0x51, 0x5B, 0xD9, 0x59, 0x2D, 0x3D, 0xAE, 0xA4, 0x49, 0x31 } } } },	// EOB1
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x0000003E, { { 0xA5, 0x8C, 0xCA, 0x13, 0xED, 0x0F, 0xB7, 0xA2, 0xD7, 0x9C, 0xCD, 0x11, 0x65, 0x11, 0x4B, 0xD8 } } } },	// EOB2

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseConeOfColdDest1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000007, 0x00000500, { { 0x48, 0xF1, 0xFE, 0x48, 0xEC, 0x64, 0x17, 0x51, 0x5C, 0x9A, 0x91, 0x35, 0x95, 0xC3, 0x73, 0x8E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseConeOfColdDest2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000007, 0x00000210, { { 0xBA, 0x62, 0xA0, 0x4F, 0x50, 0x0C, 0x02, 0xC3, 0xAD, 0x7C, 0x39, 0x63, 0x5F, 0x41, 0xB4, 0xFB } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseConeOfColdDest3Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000007, 0x00000200, { { 0xA0, 0x1F, 0xAC, 0x3A, 0x2D, 0x25, 0x1F, 0x5C, 0xD2, 0x04, 0xAC, 0xAB, 0x97, 0x8B, 0x61, 0xD7 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseConeOfColdDest4Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000007, 0x000004F0, { { 0xB3, 0x9A, 0x2B, 0x3A, 0x51, 0x24, 0x95, 0xBE, 0xDE, 0x0F, 0xD5, 0xE9, 0xE9, 0x21, 0x96, 0x04 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseConeOfColdGfxTblProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x0000003E, { { 0x0A, 0xBA, 0xFD, 0x3F, 0xD8, 0x49, 0x3F, 0xD2, 0x26, 0x1B, 0x19, 0x53, 0x4F, 0x84, 0xB9, 0x4F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1MainMenuStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000037, 0x00000D79, { { 0x1D, 0x72, 0x7F, 0x8F, 0xEB, 0x4A, 0xBF, 0x82, 0xB7, 0xB5, 0x9D, 0xB0, 0x7B, 0xDA, 0xEC, 0xEE } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000034, 0x00000C6F, { { 0xF2, 0x5F, 0xBE, 0xFB, 0x27, 0x1C, 0x91, 0x33, 0x25, 0x09, 0xC1, 0xA0, 0x27, 0x89, 0xD7, 0xD5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1BonusStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000093, 0x000031B6, { { 0xC1, 0x54, 0x1D, 0x02, 0x4A, 0x35, 0x7F, 0x5D, 0x84, 0x2D, 0x2C, 0x9C, 0x06, 0x97, 0x29, 0x8D } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000093, 0x000031CD, { { 0x3E, 0x0F, 0x52, 0x02, 0xC7, 0x9E, 0x83, 0xB3, 0xB1, 0xAB, 0x03, 0x3A, 0x18, 0xE2, 0x87, 0x2E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroFilesOpeningProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000003F, 0x00001044, { { 0xF5, 0x8C, 0xC8, 0x39, 0x38, 0xBB, 0x0B, 0xCA, 0x34, 0x38, 0x1D, 0x11, 0x46, 0x91, 0xEF, 0x7E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroFilesTowerProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000001A, 0x000006E6, { { 0xBD, 0x06, 0x3B, 0x7D, 0x24, 0x79, 0xD6, 0xC2, 0xFA, 0xDA, 0x31, 0x15, 0x3E, 0xE2, 0x75, 0xF8 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroFilesOrbProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000015, 0x00000565, { { 0xA7, 0x91, 0x97, 0x5B, 0x29, 0xE8, 0x27, 0x90, 0xB3, 0x8F, 0xD5, 0x13, 0x77, 0x4A, 0x93, 0x37 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroFilesWdEntryProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000002C, 0x00000B42, { { 0x5C, 0xDF, 0xB1, 0x2A, 0x83, 0x03, 0x73, 0x47, 0x1E, 0x29, 0x7C, 0x16, 0x2E, 0x5D, 0x0F, 0xA4 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroFilesKingProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000016, 0x000005AE, { { 0xB5, 0xB5, 0x80, 0xD3, 0xC0, 0xF4, 0x9F, 0xE1, 0x12, 0x3C, 0xCB, 0xD6, 0xF2, 0x7F, 0x15, 0x5B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroFilesHandsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000A, 0x0000027C, { { 0x90, 0xC7, 0x36, 0xE6, 0x7D, 0x6D, 0xCB, 0x77, 0xA0, 0x03, 0x45, 0x48, 0x46, 0xF3, 0x80, 0xC8 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroFilesWdExitProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000033, 0x00000D2A, { { 0xA8, 0xF0, 0x36, 0x0E, 0x37, 0xC6, 0xCC, 0xDB, 0x9B, 0xB8, 0x52, 0x64, 0x02, 0x1E, 0x9D, 0x1C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroFilesTunnelProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000001A, 0x000006E2, { { 0xA1, 0xDD, 0x20, 0x50, 0x7A, 0xB6, 0x89, 0x67, 0x13, 0xAA, 0x47, 0x6B, 0xC0, 0xA0, 0x8A, 0xFD } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroOpeningFrmDelayProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000A, 0x000001E0, { { 0xDA, 0xE3, 0x06, 0xA2, 0x41, 0xF6, 0x5A, 0x6A, 0xBD, 0x0B, 0xA6, 0x09, 0x69, 0x03, 0x1D, 0x2C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroWdEncodeXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000001F, 0x000001BB, { { 0x00, 0x50, 0x8E, 0xF5, 0x51, 0xA6, 0xF5, 0x57, 0x0D, 0x55, 0x6C, 0x14, 0x62, 0xCD, 0xD0, 0x7E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroWdEncodeYProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000001F, 0x0000000B, { { 0x39, 0x38, 0x02, 0xCE, 0x9D, 0x89, 0x1E, 0xBF, 0x32, 0x86, 0xA0, 0x79, 0xA4, 0xBE, 0xC5, 0x81 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroWdEncodeWHProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000001F, 0x00000027, { { 0xA8, 0x6C, 0x13, 0x2B, 0x4C, 0x26, 0x38, 0x3D, 0xDA, 0xC2, 0x90, 0xB3, 0x97, 0xA9, 0x45, 0x84 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroWdDsXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000003E, 0x0000104A, { { 0xAC, 0x1F, 0xA6, 0x20, 0xD0, 0x02, 0xF0, 0x9D, 0x75, 0x93, 0x6C, 0x12, 0x0A, 0x76, 0x1B, 0x3F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroWdDsYProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000001F, 0x00000655, { { 0xF3, 0xF7, 0x65, 0xEC, 0xEA, 0x5C, 0x08, 0xCF, 0xAD, 0x48, 0x35, 0xA2, 0x5B, 0x82, 0xB0, 0xC5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroTvlX1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x00000027, { { 0x7F, 0x14, 0x7D, 0x8C, 0x20, 0x49, 0xDB, 0xC3, 0x31, 0x1A, 0xC3, 0x95, 0xA4, 0x8C, 0x96, 0xDC } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroTvlY1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x000000EC, { { 0x29, 0xB4, 0x8D, 0xE1, 0xDF, 0x36, 0x39, 0x27, 0xC8, 0xF6, 0x32, 0x1A, 0x3B, 0x74, 0xA1, 0x4F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroTvlX2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x00000051, { { 0x51, 0x33, 0x0A, 0x55, 0x76, 0xA2, 0x91, 0xDA, 0x59, 0xD6, 0x09, 0xD9, 0x3D, 0xD4, 0xB8, 0xFE } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroTvlY2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x0000016A, { { 0xD5, 0xA3, 0xF6, 0x12, 0x90, 0x87, 0xF2, 0xC7, 0x6A, 0x22, 0x77, 0xB5, 0x48, 0xB2, 0xCB, 0xCA } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroTvlWProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x0000004E, { { 0xCF, 0xC7, 0xA8, 0x59, 0x6A, 0x5B, 0x35, 0x7F, 0xC9, 0xEC, 0x59, 0x7E, 0x88, 0x31, 0x32, 0xA6 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1IntroTvlHProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x0000013D, { { 0x26, 0x7B, 0x3D, 0x5F, 0x64, 0x97, 0xF9, 0x1B, 0xB6, 0x65, 0x99, 0x95, 0x0A, 0x98, 0x38, 0x92 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1DoorShapeDefsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000060, 0x00000F8A, { { 0x95, 0x53, 0x1B, 0x07, 0x64, 0x81, 0x0E, 0x04, 0xC0, 0xDA, 0xB5, 0x74, 0x57, 0x04, 0x10, 0xE2 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1DoorSwitchShapeDefsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000060, 0x0000119E, { { 0xA4, 0xE6, 0x96, 0x36, 0x59, 0x05, 0xB8, 0x57, 0xF4, 0x6D, 0x79, 0x1D, 0x29, 0x52, 0xA0, 0xEE } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1DoorSwitchCoordsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000030, 0x000007F7, { { 0x85, 0x20, 0x98, 0x20, 0xE1, 0xD6, 0xA5, 0xBD, 0x9E, 0x59, 0x63, 0x6A, 0xEF, 0xEF, 0x80, 0x19 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1MonsterPropertiesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000252, 0x000038E5, { { 0x5E, 0xD7, 0xEF, 0x3B, 0xD5, 0xDA, 0x2A, 0x09, 0x78, 0xF6, 0xD8, 0x57, 0x68, 0xB4, 0x90, 0xCA } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1EnemyMageSpellListProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000A, 0x0000000F, { { 0x01, 0x1B, 0x9C, 0x51, 0xC9, 0xA2, 0x10, 0xBB, 0xA7, 0x82, 0xD4, 0x91, 0x7E, 0x84, 0x54, 0x93 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1EnemyMageSfxProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000A, 0x0000029B, { { 0xA2, 0x9F, 0x2E, 0xDE, 0x15, 0x23, 0x78, 0xDD, 0x26, 0x98, 0x6E, 0xA3, 0x77, 0xEA, 0xB5, 0x80 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1BeholderSpellListProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000079, { { 0x8E, 0x13, 0x54, 0x9D, 0x54, 0xF6, 0xC9, 0x6E, 0x10, 0xF1, 0xC0, 0xE9, 0x66, 0xDD, 0x95, 0xED } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1BeholderSfxProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x000000F5, { { 0xA9, 0x90, 0x41, 0x0D, 0xB5, 0xE0, 0x28, 0xFD, 0x0A, 0xC3, 0xF9, 0xEC, 0xC8, 0x47, 0xC1, 0x57 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1TurnUndeadStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000027, 0x00000BF2, { { 0x43, 0x0A, 0x1E, 0xEE, 0x84, 0xD6, 0xD6, 0x87, 0x20, 0x9F, 0x15, 0x22, 0x9B, 0x65, 0x24, 0xDB } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000030, 0x00000F48, { { 0xDA, 0x59, 0xEC, 0xC1, 0x9B, 0xCF, 0x90, 0x4A, 0x93, 0x3E, 0xE5, 0x26, 0x20, 0x8B, 0x74, 0x92 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingDefaultProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x0000002C, { { 0x7E, 0x1C, 0x75, 0xC3, 0x8E, 0xF7, 0x56, 0x62, 0x9B, 0xB6, 0xF4, 0x3A, 0x21, 0x03, 0xFA, 0xF5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingAltProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000030, { { 0x2A, 0x8C, 0xF6, 0xD7, 0x87, 0xFA, 0x7B, 0x22, 0x28, 0x2A, 0x50, 0xE2, 0x26, 0x7B, 0xC7, 0x44 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingInvProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x0000002E, { { 0x3A, 0x06, 0xBF, 0x0C, 0xD4, 0xD0, 0x15, 0x1F, 0xB5, 0xC5, 0x49, 0xFD, 0x21, 0xE1, 0xE1, 0x66 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingItemsLProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x0000002A, { { 0xE0, 0x85, 0xA1, 0x3A, 0x3D, 0xC9, 0xF8, 0x56, 0x17, 0x0A, 0xD8, 0x44, 0x56, 0xDF, 0x3C, 0x57 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingItemsSProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000036, { { 0x2E, 0x6F, 0xD4, 0x2E, 0xB2, 0x84, 0xB2, 0xC3, 0x36, 0x88, 0x80, 0xC1, 0x67, 0x5A, 0xEB, 0x60 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingThrownProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000030, { { 0x0C, 0x3D, 0x1E, 0xAB, 0x0B, 0x25, 0x9F, 0x78, 0xE6, 0xB1, 0x52, 0x79, 0x0F, 0x96, 0x33, 0x97 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingIconsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000039, { { 0x99, 0x50, 0x1A, 0xE1, 0xF3, 0x52, 0xC3, 0x5A, 0x4E, 0xBD, 0x03, 0x74, 0x2C, 0x39, 0xCA, 0x71 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingDecoProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000035, { { 0xA5, 0x17, 0xED, 0xEE, 0x02, 0x87, 0x8C, 0x9D, 0xAC, 0x96, 0xC6, 0x07, 0xB0, 0x8E, 0x5D, 0xE3 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaLevelMappingIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000000C, 0x00000013, { { 0x48, 0x5D, 0xDF, 0x8F, 0xFD, 0x5D, 0xA0, 0xB0, 0x00, 0xD8, 0xB3, 0x09, 0x90, 0x5D, 0x13, 0x3F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingLevel0Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000035, { { 0xC2, 0x4D, 0x2F, 0x0A, 0xB0, 0x3E, 0x46, 0x80, 0xD1, 0xEE, 0x32, 0x5F, 0xBA, 0x5C, 0xCC, 0x7A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingLevel1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000030, { { 0x94, 0x8E, 0xAE, 0x12, 0xB5, 0x68, 0xCD, 0x43, 0x95, 0xD2, 0x01, 0x21, 0x0C, 0xA1, 0x34, 0xF5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingLevel2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000030, { { 0x20, 0x6F, 0x9F, 0x57, 0x0C, 0xFD, 0xDA, 0x5C, 0xA0, 0x1D, 0x28, 0xB4, 0x88, 0x24, 0x68, 0x68 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingLevel3Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000030, { { 0x44, 0x95, 0x9A, 0x69, 0x70, 0xB2, 0x63, 0xB6, 0xFB, 0xD0, 0xFF, 0xD9, 0xF0, 0xCD, 0xD4, 0x75 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1CgaMappingLevel4Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000031, { { 0xEA, 0xC4, 0x01, 0xC0, 0x21, 0xFE, 0x66, 0xDD, 0xD4, 0x83, 0xC1, 0x2C, 0x09, 0xD3, 0xD0, 0x97 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1NpcShpDataProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000004C, 0x00000A42, { { 0x70, 0x21, 0x85, 0x8C, 0xD4, 0x04, 0xAA, 0x20, 0x1D, 0x0E, 0x9D, 0xB7, 0x74, 0x58, 0xCC, 0x0C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1NpcSubShpIndex1Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x00000035, { { 0x9A, 0x83, 0xF9, 0xA4, 0x27, 0xBA, 0xFC, 0xD2, 0xDE, 0x03, 0x65, 0xF2, 0xFA, 0x37, 0xDA, 0xF1 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1NpcSubShpIndex2Provider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x00000051, { { 0x7E, 0xAC, 0x0E, 0x54, 0x59, 0x5D, 0xF6, 0x53, 0x03, 0x22, 0x1D, 0xC7, 0xFC, 0x16, 0xC8, 0x88 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1NpcSubShpYProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000006, 0x00000143, { { 0xC1, 0xED, 0x93, 0x5E, 0x84, 0xCE, 0x48, 0xCF, 0x4C, 0xF3, 0x9C, 0x93, 0xBF, 0xFE, 0xB8, 0x6F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc0StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000044, 0x000016E2, { { 0x7C, 0x28, 0x72, 0xC9, 0x57, 0xF5, 0xAB, 0x02, 0xD1, 0x42, 0xE8, 0xA3, 0xF9, 0x33, 0x70, 0xEE } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000050, 0x00001B13, { { 0x69, 0x05, 0xEB, 0xB6, 0x86, 0x81, 0xAC, 0x09, 0x53, 0x35, 0x4D, 0x55, 0xF3, 0x13, 0x6F, 0xC0 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc11StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000001B, 0x00000928, { { 0x86, 0x08, 0x95, 0x6B, 0xBF, 0x12, 0x2D, 0xF9, 0x62, 0x25, 0xD9, 0xAE, 0x25, 0x10, 0xDF, 0xDC } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000001A, 0x000008DB, { { 0xBD, 0xBB, 0x48, 0x8E, 0x04, 0x7D, 0xE4, 0x78, 0xBB, 0x59, 0x6E, 0x86, 0xE1, 0x06, 0x27, 0x50 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc12StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000016, 0x0000079C, { { 0x22, 0x57, 0x3A, 0x9C, 0x7C, 0xDB, 0x55, 0xD0, 0x9C, 0x84, 0x28, 0xA6, 0x9D, 0x40, 0x38, 0x6E } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000014, 0x000006ED, { { 0x88, 0x1C, 0x09, 0x61, 0x5D, 0x9D, 0xDE, 0x8A, 0x54, 0x1C, 0x40, 0xCF, 0x28, 0x2B, 0x52, 0x9D } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc21StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000014, 0x000006FD, { { 0x55, 0x77, 0x2F, 0xB0, 0xB3, 0x2D, 0x81, 0x29, 0xDE, 0x71, 0x83, 0x41, 0x06, 0x5B, 0x72, 0x21 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000015, 0x00000748, { { 0x3E, 0x15, 0x27, 0xFD, 0x76, 0xFB, 0x14, 0x8C, 0xF6, 0x14, 0x3E, 0x20, 0x0A, 0x04, 0xF5, 0x32 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc22StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000F, 0x000004D4, { { 0xE5, 0x97, 0x06, 0x45, 0x6A, 0xAC, 0x96, 0x6D, 0x0A, 0xC9, 0xDF, 0x8F, 0x96, 0x2D, 0x01, 0x5D } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000000D, 0x00000439, { { 0x87, 0xCB, 0x17, 0xD2, 0xC8, 0x7F, 0x34, 0xDA, 0x82, 0x30, 0xB2, 0x68, 0xB0, 0x10, 0xD9, 0x52 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc31StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000011, 0x00000597, { { 0x5C, 0xEB, 0x0A, 0xE6, 0xB1, 0x37, 0x0E, 0x8F, 0x14, 0xB4, 0x68, 0x86, 0xE5, 0xD2, 0xDE, 0xC7 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000012, 0x00000603, { { 0x8E, 0x68, 0x55, 0xCD, 0x29, 0x1E, 0x3C, 0x06, 0x7B, 0x97, 0xE1, 0x07, 0x49, 0x09, 0xF0, 0x57 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc32StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000020, 0x00000AE4, { { 0xED, 0x09, 0x04, 0xEC, 0xE3, 0x43, 0xDA, 0xEE, 0x5D, 0x78, 0x32, 0x63, 0x68, 0xFC, 0x4F, 0x9E } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000020, 0x00000B13, { { 0x87, 0x40, 0x88, 0xA5, 0xE2, 0x6F, 0x83, 0xBC, 0x99, 0x2B, 0xD3, 0xF5, 0x8D, 0x6B, 0x6E, 0x7D } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc4StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000D, 0x0000043C, { { 0x2C, 0xE7, 0xE5, 0xAA, 0xF3, 0x50, 0xA8, 0x6D, 0xC2, 0xC6, 0x88, 0xFE, 0x12, 0x96, 0xFE, 0x54 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000014, 0x00000720, { { 0xF8, 0x58, 0x9A, 0xDB, 0xE5, 0x3F, 0x67, 0x53, 0x1F, 0x27, 0x2E, 0x8D, 0x6E, 0xAD, 0x45, 0xF5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc5StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000021, 0x00000ABC, { { 0xF1, 0xB5, 0x9E, 0x51, 0x9E, 0xF8, 0x84, 0x95, 0x55, 0x55, 0xE7, 0xDF, 0x36, 0xE1, 0x78, 0x9A } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000001D, 0x00000A8C, { { 0x4A, 0xAE, 0x5B, 0x3B, 0xAD, 0x18, 0x91, 0x3F, 0xC9, 0x5A, 0x82, 0x5D, 0xA7, 0x06, 0x1A, 0xAE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc6StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000011, 0x00000612, { { 0x1B, 0xE2, 0x23, 0xD9, 0x00, 0x5C, 0xB9, 0x54, 0xCE, 0xA7, 0x6A, 0x51, 0xF6, 0xBB, 0x8A, 0xC9 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000012, 0x00000647, { { 0x6C, 0x3F, 0xE2, 0xD0, 0xB0, 0x75, 0x2D, 0x73, 0xEE, 0x6F, 0x17, 0x74, 0xAA, 0x7D, 0xA2, 0x21 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB1Npc7StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000018, 0x00000777, { { 0x60, 0xB4, 0x17, 0x72, 0x89, 0x87, 0x47, 0xE3, 0xD9, 0xC3, 0x59, 0x16, 0xFD, 0x03, 0x31, 0xD4 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000016, 0x000007B6, { { 0xAE, 0xB6, 0x3C, 0x14, 0x2B, 0x34, 0xB8, 0x7C, 0xCF, 0x87, 0xDA, 0x70, 0xBF, 0x0D, 0xAB, 0xE2 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2MainMenuStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000005F, 0x000017BE, { { 0x77, 0x8A, 0x50, 0x9F, 0x42, 0xD8, 0x00, 0x05, 0x60, 0x2A, 0x80, 0x25, 0x00, 0xDC, 0x7C, 0x92 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000005E, 0x000017F3, { { 0xD0, 0x93, 0x2E, 0x5F, 0x9D, 0xDB, 0xC4, 0xFB, 0x9E, 0x9F, 0x14, 0xD6, 0xB4, 0xBE, 0x3D, 0x0C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2TransferPortraitFramesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000040, 0x00000B25, { { 0x13, 0x25, 0x69, 0xC6, 0xE4, 0x9D, 0x35, 0x11, 0xAB, 0xE2, 0xC1, 0xEF, 0x21, 0x8B, 0xB8, 0x28 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2TransferConvertTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000031, 0x000004BC, { { 0x96, 0x53, 0xA2, 0xF1, 0x26, 0xFE, 0x1B, 0x5E, 0xDF, 0x62, 0x2C, 0x8C, 0xBD, 0x62, 0x5A, 0xF9 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2TransferItemTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000025, { { 0xD0, 0xA4, 0xB3, 0x7D, 0x74, 0x4D, 0x16, 0x43, 0x56, 0x07, 0x84, 0xAA, 0x96, 0xBD, 0x82, 0x25 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2TransferExpTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x0000076B, { { 0x91, 0x65, 0x5B, 0x8D, 0xE8, 0x5B, 0x28, 0x32, 0x4D, 0x7A, 0x57, 0x8E, 0x18, 0x5B, 0x1A, 0xE9 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2TransferStrings1Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000010, 0x000003D3, { { 0x31, 0xE4, 0x65, 0x69, 0x0A, 0xA1, 0x1D, 0xD1, 0xFE, 0xF8, 0x5C, 0x29, 0xB1, 0x46, 0xBD, 0xBE } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000010, 0x000003E2, { { 0xF0, 0x10, 0xF8, 0x9F, 0x05, 0x1E, 0x31, 0x33, 0x4E, 0xC8, 0x49, 0xBC, 0x9E, 0xAD, 0xD4, 0x99 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2TransferStrings2Provider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000006A, 0x00002681, { { 0x12, 0x4D, 0x29, 0x9D, 0xD3, 0xFC, 0x39, 0x22, 0x73, 0x1E, 0x5C, 0xAF, 0x1F, 0xD1, 0xAA, 0x87 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000057, 0x00001F8E, { { 0x85, 0xD8, 0x39, 0x1E, 0x6D, 0x97, 0xBD, 0x0E, 0xDD, 0xCF, 0x19, 0x47, 0x31, 0xDC, 0x7C, 0x1A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2TransferLabelsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000A, 0x00000240, { { 0x2A, 0x8B, 0x54, 0x99, 0x94, 0x35, 0x2B, 0xAB, 0x7F, 0x7F, 0x98, 0xA3, 0xFD, 0x57, 0x20, 0xDE } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000009, 0x000001DF, { { 0x47, 0x6B, 0xBA, 0xCD, 0x99, 0x74, 0xCA, 0x3C, 0xAA, 0xC6, 0xB4, 0x50, 0x38, 0x90, 0x25, 0xB8 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2IntroStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000321, 0x0000DBC3, { { 0x11, 0x9B, 0x54, 0xB3, 0x34, 0xF0, 0xB5, 0xE1, 0xFA, 0x6A, 0x31, 0x02, 0x59, 0x29, 0xCA, 0x94 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000038E, 0x0001119C, { { 0x92, 0x63, 0x18, 0xDD, 0x9F, 0x62, 0xF5, 0xBC, 0x3D, 0x93, 0xDC, 0x6E, 0xE5, 0xBE, 0x8C, 0x0B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2IntroCPSFilesProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x000000A2, 0x0000296A, { { 0xE9, 0x28, 0x4A, 0x6E, 0xAA, 0x44, 0xF4, 0xD7, 0xD1, 0x29, 0xBF, 0x90, 0x6B, 0x82, 0xD1, 0x77 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x000000A2, 0x0000296B, { { 0x03, 0xA2, 0x0A, 0xAB, 0x76, 0x78, 0x04, 0x88, 0x6A, 0xE0, 0x36, 0x8B, 0x3A, 0x87, 0x44, 0xC8 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData00Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x000003E1, { { 0x38, 0xC2, 0x0F, 0xE1, 0x43, 0x6A, 0xE8, 0x7C, 0x82, 0x65, 0x9B, 0x4A, 0x9F, 0x83, 0xCD, 0xC8 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData01Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x000003A3, { { 0x84, 0x44, 0xF4, 0x46, 0x4E, 0x2B, 0xD7, 0xC6, 0xAD, 0x14, 0xF1, 0x9E, 0x8A, 0xBE, 0x7B, 0x42 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData02Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x00000446, { { 0x0C, 0xCA, 0x41, 0x0C, 0x89, 0x59, 0xD5, 0x28, 0x9A, 0xDC, 0x51, 0x1C, 0x0B, 0x8C, 0xD2, 0xDB } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData03Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x0000010E, { { 0xAB, 0x48, 0x64, 0x02, 0xB3, 0xF3, 0x6C, 0x82, 0x9D, 0x37, 0x5F, 0x52, 0x0F, 0x5B, 0xDF, 0x96 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData04Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x00000149, { { 0x3B, 0xAC, 0x14, 0x51, 0xDF, 0x5D, 0x22, 0x15, 0x46, 0x4E, 0xCD, 0xF3, 0xD4, 0x61, 0x29, 0x4A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData05Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x0000010E, { { 0x28, 0x29, 0x5F, 0x31, 0x23, 0x53, 0xBA, 0xD7, 0x24, 0xB9, 0x21, 0x70, 0x84, 0x8A, 0x1C, 0x2E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData06Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B0, 0x00001365, { { 0x91, 0x28, 0x2F, 0x10, 0x45, 0x4D, 0xCF, 0x3E, 0x70, 0x1E, 0xD4, 0xBA, 0x0E, 0x70, 0xDE, 0xD0 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData07Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x000003C4, { { 0x8C, 0x72, 0xDE, 0x4F, 0x92, 0x52, 0x0A, 0xED, 0xF4, 0x79, 0xD6, 0x3D, 0x8F, 0x59, 0x9D, 0x69 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData08Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000042, 0x00000442, { { 0xD2, 0x91, 0x51, 0xEB, 0x91, 0x13, 0x43, 0xCE, 0x7E, 0x60, 0xB8, 0xFF, 0xA7, 0xE2, 0x4C, 0x11 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData09Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x000004BC, { { 0xD6, 0xC7, 0x44, 0x2E, 0xE7, 0x2A, 0x44, 0x09, 0x39, 0xC3, 0xD3, 0xA8, 0x02, 0xC8, 0xA0, 0x19 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData10Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000006E, 0x00000C12, { { 0x91, 0xDB, 0x41, 0x7A, 0x4F, 0x7C, 0x7B, 0x83, 0x32, 0x13, 0x68, 0xF6, 0x58, 0x79, 0xDA, 0x99 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData11Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B0, 0x0000073C, { { 0x17, 0x1F, 0x4D, 0x05, 0x3F, 0x14, 0x2E, 0x77, 0xD3, 0xDB, 0x78, 0x67, 0xBB, 0x18, 0xDC, 0x85 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData12Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x00000228, { { 0xC9, 0x50, 0x68, 0x51, 0xD0, 0xC1, 0x5D, 0xD4, 0xFF, 0x08, 0x28, 0xDE, 0xC4, 0x41, 0x8C, 0xDB } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData13Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x00000340, { { 0x03, 0xCA, 0x5D, 0xD1, 0x15, 0xFA, 0x60, 0xD7, 0x70, 0x64, 0x3D, 0x44, 0x08, 0xB8, 0xDB, 0xAD } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData14Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000042, 0x000007C0, { { 0x82, 0xA9, 0x0B, 0x90, 0x9D, 0x65, 0x1E, 0xC7, 0x03, 0x5E, 0xB7, 0xDF, 0x6E, 0x1F, 0xED, 0xD6 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData15Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x00000504, { { 0xA7, 0x91, 0x4F, 0xAD, 0xB1, 0x77, 0x80, 0x3A, 0xC7, 0xDE, 0x35, 0x7A, 0x96, 0x16, 0xD2, 0x13 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData16Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000079, 0x00000B3D, { { 0xCC, 0x63, 0x5A, 0x11, 0xEE, 0x8A, 0xAE, 0x3A, 0x14, 0xC3, 0xBC, 0xDA, 0xAF, 0x1D, 0xD4, 0x2C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData17Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000084, 0x00000911, { { 0x09, 0x1C, 0x4B, 0xD9, 0x0B, 0x2A, 0xD6, 0xC1, 0xE3, 0x8D, 0xFE, 0x43, 0x8F, 0x2E, 0x21, 0x51 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData18Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000058, 0x000008FA, { { 0xFE, 0x58, 0xD9, 0x67, 0x78, 0x97, 0xE2, 0xCD, 0x82, 0xB8, 0xC9, 0xC0, 0x1F, 0xCA, 0x7C, 0xF5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData19Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000009A, 0x00000D6B, { { 0xA1, 0xDD, 0x7B, 0x8B, 0x25, 0xA5, 0x96, 0x5A, 0x33, 0x5E, 0x80, 0x5F, 0xA5, 0xBB, 0xAC, 0x11 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData20Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000009A, 0x00000D6C, { { 0x19, 0xF9, 0x93, 0x1D, 0x01, 0xEE, 0x7C, 0x8B, 0x6C, 0x3E, 0x35, 0x2C, 0x5C, 0x88, 0xCD, 0xB6 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData21Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000009A, 0x00000D83, { { 0xCB, 0x4F, 0x21, 0x29, 0x63, 0x5B, 0x8C, 0xF2, 0xBA, 0x03, 0x49, 0xD1, 0xAF, 0x22, 0xB0, 0xD5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData22Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x00000200, { { 0x66, 0xEE, 0x45, 0xB1, 0x87, 0x66, 0xC4, 0x55, 0xCE, 0x60, 0x0C, 0x5B, 0xBB, 0x3C, 0x7D, 0x33 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData23Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x0000020D, { { 0xC4, 0x49, 0xE2, 0x5B, 0x2E, 0x17, 0x68, 0xC4, 0xBA, 0x20, 0xEC, 0xB1, 0xEB, 0x1A, 0xFB, 0xE0 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData24Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x00000214, { { 0xF1, 0x46, 0x82, 0xEF, 0x6D, 0xCA, 0x68, 0xA2, 0xF3, 0x55, 0x63, 0xD2, 0x13, 0x25, 0x19, 0xF7 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData25Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x00000256, { { 0x8F, 0xB9, 0xCD, 0xB8, 0x58, 0xCB, 0x90, 0x03, 0xFC, 0xB6, 0x95, 0x6F, 0x52, 0xF8, 0x7D, 0x19 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData26Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x00000263, { { 0x7A, 0x37, 0x07, 0xC4, 0x67, 0x72, 0x1F, 0xCB, 0xAC, 0x98, 0x46, 0x9A, 0xF3, 0x5F, 0xBA, 0x78 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData27Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x0000026A, { { 0x80, 0x11, 0xEE, 0x44, 0xDA, 0xE1, 0x26, 0x1F, 0x14, 0x7E, 0x93, 0x99, 0x44, 0x44, 0x9F, 0x85 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData28Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x000001F6, { { 0x45, 0xA1, 0xA5, 0xEC, 0x85, 0x06, 0xE2, 0x91, 0x28, 0xE0, 0xBB, 0x53, 0x74, 0x44, 0xD9, 0xA6 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData29Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x000001F9, { { 0x3F, 0x03, 0x2F, 0x8B, 0xFB, 0x6A, 0x97, 0x05, 0xED, 0xBB, 0xD6, 0xA0, 0xF5, 0x7A, 0x6D, 0x08 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData30Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x00000204, { { 0xA1, 0x37, 0x57, 0xC3, 0x72, 0x08, 0x98, 0xA6, 0xF4, 0x5E, 0x58, 0x9E, 0xF3, 0x11, 0x88, 0x1E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData31Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x00000212, { { 0x19, 0xCC, 0x6F, 0xA8, 0x29, 0xB5, 0x3B, 0x15, 0x2F, 0x2C, 0x43, 0xED, 0x7A, 0xF5, 0xC5, 0x69 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData32Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x000006C9, { { 0xBF, 0x65, 0xBA, 0x3F, 0x44, 0xEE, 0xB0, 0x5C, 0x8B, 0xBD, 0x15, 0xAB, 0x03, 0xD1, 0x55, 0x21 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData33Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000009A, 0x00001585, { { 0xB5, 0x44, 0x06, 0xC9, 0xE8, 0x27, 0x75, 0x6E, 0x63, 0x77, 0xE9, 0xA9, 0x68, 0x73, 0xF5, 0x78 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData34Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000058, 0x00000B43, { { 0x52, 0xB4, 0x1E, 0x14, 0x88, 0xBD, 0x8A, 0xD7, 0x38, 0xDF, 0x25, 0xB0, 0xAF, 0xAE, 0x76, 0xE1 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData35Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x000005A4, { { 0xFB, 0x82, 0xE7, 0xB2, 0x54, 0xDB, 0xB5, 0xE1, 0xCE, 0xFB, 0xD1, 0x23, 0x39, 0x8F, 0xA1, 0x0D } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData36Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000042, 0x00000572, { { 0x2C, 0x16, 0xD9, 0xBE, 0xDB, 0xBA, 0x04, 0xCA, 0x97, 0xB5, 0x88, 0x43, 0xA8, 0x62, 0xE2, 0x04 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData37Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x0000024E, { { 0xFF, 0x28, 0xD8, 0x62, 0xC6, 0xAD, 0x48, 0xC7, 0x31, 0x84, 0x6C, 0xBA, 0x9F, 0x4D, 0x15, 0xDA } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData38Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000001D9, 0x00001FB1, { { 0x16, 0xB0, 0xDF, 0x86, 0x8C, 0xB3, 0x52, 0xEF, 0x21, 0x04, 0x22, 0x6D, 0xC0, 0x03, 0xB8, 0xC6 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData39Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000004D, 0x00000582, { { 0x11, 0x6C, 0xBB, 0xF6, 0x1B, 0x3C, 0xAE, 0xAA, 0x40, 0x27, 0x3F, 0x86, 0x33, 0x92, 0xCB, 0xA9 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData40Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000160, 0x000010A2, { { 0xD9, 0x9D, 0xF1, 0x7D, 0xE1, 0x7C, 0x61, 0xC0, 0xD4, 0xD3, 0x05, 0x0C, 0x79, 0xDD, 0xDB, 0xD1 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData41Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x00000355, { { 0x92, 0x85, 0xBE, 0x5A, 0x38, 0x08, 0xF3, 0xDF, 0xC6, 0x56, 0x74, 0xC3, 0x0B, 0x3F, 0x72, 0x4D } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData42Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x0000010B, { { 0x68, 0xF8, 0x1D, 0x74, 0x6D, 0x32, 0x1E, 0x3A, 0x1C, 0xD1, 0x1D, 0x4B, 0x89, 0x3D, 0x5F, 0x2B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2IntroAnimData43Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000037, 0x00000116, { { 0xD5, 0x46, 0xCB, 0x3F, 0x27, 0xBD, 0x2B, 0xD6, 0x35, 0x69, 0x9E, 0x0A, 0x28, 0xDA, 0xC9, 0x84 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2IntroShapes00Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000005A, 0x00000A86, { { 0xF9, 0xD5, 0xD2, 0x93, 0xBC, 0xC4, 0x86, 0x3F, 0x83, 0x0D, 0xDB, 0x38, 0x60, 0x6E, 0xA7, 0xDA } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2IntroShapes01Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x0000009B, { { 0xAA, 0xDD, 0x25, 0x21, 0x57, 0x6A, 0xB7, 0xEB, 0xDA, 0xFD, 0x72, 0x3B, 0xCA, 0x68, 0xDB, 0x34 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2IntroShapes04Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000417, { { 0x13, 0x95, 0x81, 0x46, 0x84, 0x36, 0xF2, 0xFC, 0xDE, 0x15, 0x85, 0x81, 0xB3, 0x9A, 0x9D, 0x20 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2IntroShapes07Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000006C, 0x000021F1, { { 0x6F, 0x7C, 0x28, 0xBB, 0xC3, 0x52, 0xE4, 0x13, 0xB4, 0xE9, 0xA4, 0x47, 0x9A, 0xBE, 0x19, 0xDA } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2FinaleStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000031C, 0x0000E287, { { 0x1E, 0x73, 0x93, 0x79, 0xB7, 0xF8, 0x17, 0xEE, 0xE4, 0xFC, 0xF0, 0x34, 0x9D, 0x06, 0x4F, 0x55 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000039F, 0x00011660, { { 0xBC, 0x1D, 0x95, 0x20, 0x32, 0xF5, 0x83, 0xCF, 0xF7, 0x11, 0xE4, 0x1D, 0x89, 0x47, 0xF0, 0x65 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2CreditsDataProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000090C, 0x00023464, { { 0x55, 0x31, 0x9D, 0x60, 0x2C, 0xA1, 0x0B, 0xF9, 0xED, 0x46, 0xDF, 0x44, 0x1A, 0x9F, 0xB1, 0xB0 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000B11, 0x0002BBD7, { { 0x46, 0x24, 0x78, 0xE9, 0xCE, 0x75, 0x45, 0x7B, 0x3B, 0xAA, 0x15, 0xD8, 0x5B, 0xCB, 0x06, 0x3A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2FinaleCPSFilesProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000009C, 0x00002853, { { 0x1F, 0xB9, 0x3D, 0x48, 0x47, 0x31, 0x0D, 0xA8, 0x92, 0x52, 0xD1, 0x54, 0x48, 0x42, 0x47, 0xBD } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000009C, 0x00002878, { { 0x48, 0x3B, 0x7A, 0xC2, 0x9C, 0xEC, 0x10, 0x07, 0xD1, 0xB6, 0x9E, 0x89, 0xE9, 0xE1, 0xBF, 0xBF } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData00Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000079, 0x00000B66, { { 0x9B, 0x8C, 0x17, 0xFA, 0xD2, 0x4F, 0x4B, 0x0E, 0x3A, 0x43, 0xB1, 0x86, 0x0C, 0xDC, 0x73, 0xAB } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData01Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000063, 0x00000A03, { { 0xBB, 0x31, 0xEA, 0x35, 0xFB, 0x99, 0x4C, 0x3E, 0x72, 0xBD, 0x36, 0x6B, 0x5C, 0x03, 0x19, 0x7F } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData02Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x000007C2, { { 0xF6, 0x83, 0x37, 0x58, 0x3C, 0x59, 0x84, 0x8F, 0x97, 0x80, 0xE2, 0xD8, 0xFD, 0x77, 0xA9, 0x54 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData03Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000042, 0x0000092B, { { 0x47, 0xE4, 0x34, 0xE8, 0x72, 0xCC, 0xA4, 0x4A, 0xA4, 0x8F, 0xBA, 0xBC, 0x0C, 0x04, 0x18, 0xAF } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData04Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000058, 0x0000080B, { { 0x16, 0xDB, 0x77, 0x4C, 0x8E, 0xFD, 0x44, 0xF6, 0x5E, 0x28, 0x0B, 0x74, 0x93, 0x45, 0x8F, 0xD9 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData05Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000004D, 0x00000C72, { { 0x6C, 0x57, 0x56, 0x7E, 0x87, 0x10, 0x9C, 0xE7, 0x69, 0xAC, 0x3B, 0x3F, 0xF6, 0x43, 0x5C, 0xD4 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData06Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x00000264, { { 0x48, 0x49, 0x5D, 0x78, 0xE2, 0xF1, 0x0D, 0x87, 0xEE, 0xEE, 0xD1, 0xA1, 0xC6, 0x64, 0xCA, 0x13 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData07Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000042, 0x00000ABE, { { 0xFE, 0xA9, 0x5D, 0x87, 0xAF, 0x55, 0x04, 0x92, 0x41, 0xD3, 0xAD, 0x1D, 0xFF, 0x03, 0x81, 0x3C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData08Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x000004D8, { { 0x4E, 0xA7, 0xCC, 0x0B, 0x1B, 0x48, 0x22, 0x09, 0x33, 0xF7, 0x23, 0xF1, 0xF5, 0x9F, 0xA5, 0x7B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData09Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000058, 0x000004BE, { { 0xF6, 0xEA, 0xA0, 0x7F, 0x54, 0x61, 0x79, 0x4C, 0x71, 0xD7, 0x9B, 0xA6, 0xC3, 0x45, 0xEE, 0x3E } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData10Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000009A, 0x00000FC4, { { 0xA9, 0xFB, 0x31, 0x55, 0xB8, 0x28, 0x63, 0xC3, 0x4B, 0x9E, 0x7D, 0x41, 0xC7, 0x1F, 0x2F, 0xBD } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData11Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000C6, 0x0000166B, { { 0xCC, 0x16, 0x50, 0xFF, 0xFF, 0xD5, 0xAE, 0x03, 0x40, 0xA3, 0x9A, 0x1F, 0xF8, 0x8E, 0x23, 0x7A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData12Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000FD, 0x00001A69, { { 0x6A, 0x80, 0x89, 0x7E, 0xFC, 0xE4, 0x01, 0xF5, 0xA2, 0x11, 0xE7, 0x26, 0x20, 0x96, 0x62, 0x7B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData13Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000FD, 0x00001886, { { 0xF9, 0x5B, 0x62, 0xDD, 0xAB, 0x14, 0x35, 0x77, 0x53, 0x05, 0xDB, 0xC5, 0xFD, 0x4D, 0x4F, 0x12 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData14Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000108, 0x00001895, { { 0x22, 0xA1, 0x88, 0x69, 0xF9, 0x1C, 0xA2, 0x64, 0x44, 0xCD, 0x00, 0xFA, 0xB1, 0x94, 0xEB, 0x3A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData15Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000D1, 0x000016E5, { { 0xD8, 0xE9, 0xA5, 0xEE, 0x54, 0x1B, 0x3E, 0x32, 0xDA, 0x78, 0x90, 0xC2, 0x54, 0xFC, 0xD5, 0x39 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData16Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000008F, 0x00000C69, { { 0xBC, 0x41, 0xE5, 0xAF, 0x89, 0xE2, 0x54, 0x12, 0x9E, 0xB0, 0x5F, 0x28, 0xFF, 0x92, 0x9D, 0x89 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData17Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000DC, 0x0000170D, { { 0x7A, 0x7B, 0x74, 0xCB, 0x68, 0xC2, 0xFF, 0xC7, 0xBE, 0x47, 0xE9, 0x43, 0xF7, 0x15, 0x8D, 0x59 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData18Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000134, 0x00002651, { { 0x71, 0x26, 0x47, 0x0D, 0x7C, 0x96, 0x45, 0x0B, 0x82, 0xD0, 0x37, 0xB9, 0xD4, 0xD0, 0x84, 0xFC } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData19Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000004D, 0x000009C3, { { 0xDA, 0x96, 0xDF, 0x16, 0xEB, 0x5D, 0x49, 0xA4, 0x3F, 0xD3, 0x31, 0xBE, 0x49, 0x72, 0xF2, 0x71 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEob2FinaleAnimData20Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x000003D8, { { 0xD9, 0xC8, 0x58, 0x4B, 0x7D, 0x79, 0x86, 0xCB, 0xEB, 0x77, 0xC2, 0xD4, 0xB7, 0xB4, 0xE9, 0x6A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2FinaleShapes00Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x0000071C, { { 0xE8, 0x67, 0xCB, 0x76, 0x6D, 0x49, 0xC2, 0x05, 0x0D, 0xAD, 0xB6, 0x83, 0x35, 0xB3, 0xBE, 0xE5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2FinaleShapes03Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x00000571, { { 0x91, 0xEC, 0xAC, 0x12, 0x08, 0x33, 0xDA, 0x7C, 0xBD, 0x51, 0x64, 0xE3, 0xAE, 0x43, 0x75, 0x14 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2FinaleShapes07Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x00000166, { { 0xED, 0x6E, 0x0C, 0x85, 0x13, 0x6F, 0xAC, 0xEB, 0xCA, 0x74, 0x2E, 0x2D, 0x0E, 0xCE, 0x17, 0xD6 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2FinaleShapes09Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000036, 0x00000898, { { 0xEB, 0xB0, 0xD9, 0xC4, 0xB6, 0xBC, 0xE3, 0xAF, 0xB2, 0x5D, 0xE3, 0xCE, 0xF7, 0x26, 0x07, 0xE5 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2FinaleShapes10Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x0000017F, { { 0x0F, 0x37, 0x94, 0xA6, 0xCE, 0x23, 0xE3, 0x2E, 0x5E, 0x2B, 0x78, 0x5B, 0x66, 0xC8, 0xE5, 0x96 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2NpcShapeDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x0000045D, { { 0x69, 0xE0, 0x5E, 0x86, 0xEB, 0x7D, 0x25, 0x95, 0xC2, 0xA0, 0xE9, 0xD5, 0x3A, 0x4A, 0x65, 0xBC } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseClassModifierFlagsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000F, 0x00000059, { { 0x17, 0x2B, 0x23, 0x14, 0x0F, 0x9D, 0x94, 0xD3, 0xBF, 0x94, 0x83, 0x0B, 0x79, 0xDB, 0xC0, 0xA9 } } } }, // EOB 1
	{ UNK_LANG, kPlatformUnknown, { 0x0000000F, 0x00000083, { { 0x54, 0x68, 0xF4, 0x07, 0x3E, 0x2A, 0xD4, 0x06, 0xF3, 0x10, 0x88, 0x6C, 0xE3, 0x34, 0x08, 0x30 } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterStepTable01Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x00000200, { { 0x26, 0x86, 0x10, 0x04, 0xC1, 0x48, 0xDD, 0xC0, 0x9F, 0x92, 0xD6, 0x20, 0x38, 0x36, 0xE2, 0xDD } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterStepTable02Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000008, 0x00000400, { { 0x6E, 0x53, 0x3C, 0x7A, 0x11, 0x46, 0x8B, 0xDC, 0x73, 0x24, 0xF8, 0x13, 0xCB, 0x6C, 0x9B, 0xE6 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterStepTable1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x00000400, { { 0x8B, 0x4C, 0x6B, 0x86, 0x93, 0xDA, 0x82, 0x1B, 0x04, 0x23, 0x92, 0x5B, 0x79, 0xB9, 0xFB, 0x06 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterStepTable2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x00000601, { { 0xE4, 0x36, 0x12, 0x93, 0x44, 0xDE, 0x6E, 0xA0, 0x4B, 0x98, 0x4B, 0x47, 0x87, 0xE3, 0x40, 0xD4 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterStepTable3Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x000007F8, { { 0x00, 0x0C, 0xB0, 0xDA, 0xE1, 0xC8, 0x45, 0x11, 0x57, 0xE4, 0x72, 0xD2, 0x32, 0xC6, 0x16, 0x2B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterCloseAttPosTable1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x00000006, { { 0x4F, 0x9D, 0x50, 0xDA, 0xA1, 0x75, 0xB0, 0xD5, 0x90, 0xCA, 0xFF, 0x3E, 0xB5, 0xE8, 0x0D, 0xAA } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterCloseAttPosTable21Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000008, 0x0000000C, { { 0x6A, 0xED, 0x15, 0xCE, 0x69, 0x54, 0x7D, 0x7B, 0x6D, 0xCE, 0xC7, 0x2A, 0x01, 0xD7, 0xDC, 0xB0 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterCloseAttPosTable22Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x00000018, { { 0x6D, 0xB9, 0x69, 0x4A, 0xE3, 0x72, 0x03, 0x5B, 0x5A, 0xBB, 0x15, 0x4A, 0xDA, 0xFB, 0x99, 0x87 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterCloseAttUnkTableProvider[] = {////
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x000007FE, { { 0xF0, 0xCB, 0x3A, 0x53, 0xDD, 0x59, 0x04, 0x87, 0x6F, 0x1B, 0x5A, 0x13, 0xBA, 0x78, 0x62, 0xEC } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterCloseAttChkTable1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x00000008, { { 0x93, 0x27, 0x19, 0xA7, 0xA7, 0x49, 0x0E, 0xC9, 0xED, 0x5C, 0x8F, 0x9F, 0xC2, 0x34, 0x62, 0x07 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterCloseAttChkTable2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x00000008, { { 0xEB, 0xF0, 0x27, 0x7E, 0xA8, 0x09, 0x3A, 0x95, 0x3B, 0x71, 0x2A, 0x43, 0x2E, 0xCF, 0x22, 0x0B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterCloseAttDstTable1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x00000018, { { 0x1E, 0xC9, 0x6C, 0x5D, 0xDF, 0xD4, 0xC0, 0x87, 0xAD, 0xEE, 0x86, 0x29, 0xD5, 0x3E, 0x8D, 0xB4 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterCloseAttDstTable2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000078, { { 0x4C, 0xA8, 0x2A, 0x53, 0xB3, 0xAA, 0x52, 0x96, 0x1D, 0xE8, 0x37, 0xDB, 0x4A, 0x77, 0xD8, 0x5B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterProximityTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x00000030, { { 0x25, 0xFC, 0xA3, 0xEB, 0x44, 0x93, 0x9B, 0x33, 0xB5, 0x86, 0xC4, 0xCB, 0x17, 0xEF, 0x2D, 0x47 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseFindBlockMonstersTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000040, 0x00000088, { { 0x6F, 0xEE, 0x8B, 0x4C, 0x21, 0xF0, 0xC6, 0x4F, 0x1D, 0x05, 0x95, 0x41, 0xD7, 0xD6, 0x52, 0x66 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterDirChangeTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x0000180C, { { 0xCD, 0xBB, 0xFD, 0xAB, 0xFB, 0x1D, 0x5C, 0x0F, 0xA2, 0xAC, 0x32, 0xA9, 0xA1, 0x93, 0x2D, 0x1C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseMonsterDistAttStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000076, 0x00002965, { { 0x1A, 0x22, 0x50, 0x04, 0x27, 0x05, 0xE9, 0x61, 0xF9, 0x0A, 0xF0, 0x50, 0x01, 0x0E, 0x65, 0xB4 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000008C, 0x00003274, { { 0xE7, 0x24, 0x17, 0x13, 0x4F, 0xB3, 0xF9, 0xB7, 0x90, 0xFA, 0x3D, 0xFF, 0xA7, 0xFB, 0x3F, 0x1F } } } },
	{ EN_ANY, kPlatformUnknown, { 0x00000054, 0x00001D03, { { 0xEB, 0x07, 0xD4, 0x22, 0xFD, 0xA0, 0x77, 0x80, 0x22, 0x04, 0x2E, 0x27, 0x7F, 0x64, 0x99, 0x4E } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000062, 0x000023E0, { { 0xB8, 0x03, 0x5C, 0x31, 0xCC, 0x18, 0xCD, 0x8D, 0x60, 0xD1, 0xFB, 0x5B, 0x66, 0xC2, 0x9A, 0x4C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseEncodeMonsterDefsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000150, 0x00001ACB, { { 0x73, 0x67, 0x5B, 0x64, 0x22, 0xDB, 0x08, 0x3A, 0xCD, 0xEB, 0x30, 0x28, 0xBD, 0xAD, 0xF8, 0x9B } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoBBaseNpcPresetsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000088B, 0x000038D0, { { 0x2E, 0xAE, 0xF0, 0x2A, 0x71, 0x6F, 0x7C, 0x5C, 0xF5, 0xAF, 0xB8, 0xBB, 0x47, 0xE5, 0xB6, 0xC3 } } } }, // EOB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000816, 0x00002C8E, { { 0xAB, 0xEE, 0x18, 0x0E, 0x59, 0xF6, 0xE0, 0x26, 0x93, 0xAB, 0x3B, 0x23, 0x29, 0x7A, 0x2C, 0x97 } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2Npc1StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000B, 0x000003B9, { { 0xB1, 0x67, 0x80, 0x21, 0x92, 0xDD, 0xFA, 0x4C, 0x4D, 0x64, 0x7C, 0x05, 0x08, 0xDC, 0x55, 0xFD } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000000D, 0x0000049E, { { 0x2D, 0x78, 0xF6, 0x20, 0x30, 0xEC, 0x62, 0x6E, 0x58, 0xF7, 0xC7, 0x6D, 0xD7, 0xBD, 0x70, 0x76 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2Npc2StringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000012, 0x0000064C, { { 0xB0, 0x66, 0x0D, 0xDE, 0x16, 0xEB, 0x5E, 0x51, 0xAF, 0x4D, 0x19, 0xD1, 0x1E, 0x0B, 0xCB, 0xD6 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000018, 0x000008FF, { { 0x59, 0x29, 0x01, 0x6F, 0xF0, 0x49, 0xC8, 0x57, 0x3E, 0x70, 0x01, 0x7E, 0x5E, 0xF2, 0xEB, 0x35 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2MonsterDustStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000001F, 0x00000AD7, { { 0x2B, 0x66, 0x27, 0xFD, 0xC6, 0x17, 0x0B, 0x6B, 0xFC, 0x7D, 0x7F, 0xD2, 0xC4, 0x12, 0x8F, 0x33 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x0000001F, 0x00000A91, { { 0x1D, 0x7D, 0xEE, 0xB8, 0x9B, 0x37, 0x2E, 0x64, 0x13, 0xB6, 0x39, 0xED, 0x88, 0xB6, 0x8B, 0xD7 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2DreamStepsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000E, 0x00000114, { { 0x27, 0x32, 0xCB, 0x89, 0x27, 0xC5, 0xDD, 0x91, 0xBE, 0x97, 0x62, 0xF5, 0x76, 0xF7, 0xCD, 0x25 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2KheldranStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000001A, 0x00000887, { { 0xA6, 0xB4, 0x45, 0x1B, 0x33, 0x54, 0x36, 0xAD, 0x1D, 0xB1, 0xDA, 0xC3, 0x12, 0x85, 0x3C, 0x58 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x00000012, 0x00000511, { { 0xEE, 0x21, 0xA8, 0x6E, 0xF7, 0xEC, 0x9A, 0x8D, 0xBA, 0x8D, 0xE3, 0x4A, 0x17, 0x15, 0xCA, 0x8C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2HornStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000009A, 0x00003541, { { 0xA5, 0x4D, 0x88, 0xAC, 0x1C, 0xCD, 0x57, 0xD4, 0x1E, 0x9F, 0xAE, 0x13, 0x46, 0xAD, 0xA0, 0x22 } } } },
	{ DE_DEU, kPlatformUnknown, { 0x000000AB, 0x00003B6C, { { 0x36, 0x34, 0xB3, 0xB1, 0x55, 0x66, 0x7A, 0x90, 0x97, 0x01, 0xDC, 0x4A, 0xAF, 0x17, 0x6B, 0x5A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2HornSoundsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000004, 0x00000106, { { 0x3E, 0x7B, 0x96, 0xFD, 0xCA, 0x4E, 0xA7, 0xA6, 0xB8, 0x82, 0x67, 0xCF, 0x93, 0x86, 0xE4, 0x45 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2WallOfForceDsXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000024, 0x00000D67, { { 0x51, 0xCF, 0xAB, 0x1E, 0xB4, 0xE0, 0xE3, 0x44, 0x29, 0xD1, 0xDC, 0x82, 0xCD, 0x08, 0x50, 0xF5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2WallOfForceDsYProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000003, 0x00000048, { { 0x45, 0xFC, 0xEA, 0x8C, 0x34, 0xD7, 0xBE, 0x74, 0x05, 0x03, 0xE6, 0x94, 0x34, 0xB5, 0x45, 0x4D } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2WallOfForceNumWProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000003, 0x00000006, { { 0x52, 0x89, 0xDF, 0x73, 0x7D, 0xF5, 0x73, 0x26, 0xFC, 0xDD, 0x22, 0x59, 0x7A, 0xFB, 0x1F, 0xAC } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2WallOfForceNumHProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000003, 0x00000011, { { 0x33, 0x86, 0x06, 0xBE, 0x8D, 0xC8, 0x37, 0x2D, 0x0F, 0x61, 0x97, 0xA4, 0x26, 0xA9, 0xBC, 0x60 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kEoB2WallOfForceShpIdProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000003, 0x00000006, { { 0x77, 0xAE, 0x9B, 0x52, 0x9E, 0xF7, 0xEB, 0x48, 0xA8, 0x5E, 0xED, 0xC2, 0x08, 0x53, 0xCE, 0x3C } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLIngamePakFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000088, 0x0000224F, { { 0xDA, 0x24, 0x18, 0xA3, 0xEF, 0x16, 0x70, 0x8F, 0xA8, 0xC2, 0x2E, 0xC2, 0xED, 0x39, 0x03, 0xD1 } } } },
	{ UNK_LANG, kPlatformPC98, { 0x00000084, 0x00002125, { { 0x7A, 0x89, 0xE2, 0x36, 0xEC, 0x6F, 0x52, 0x2B, 0xEF, 0xBA, 0x3D, 0x28, 0x54, 0xDA, 0xFB, 0x72 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCharacterDefsProvider[] = {
	{ RU_RUS, kPlatformPC, { 0x00000492, 0x000052BA, { { 0x52, 0x29, 0x0D, 0x49, 0xFD, 0x17, 0xD7, 0x70, 0x6D, 0xCA, 0xEB, 0xB6, 0x7E, 0xFA, 0xBE, 0x08 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000492, 0x000046B0, { { 0x7A, 0x94, 0x8B, 0xC6, 0xF7, 0xF1, 0x2F, 0xF3, 0xBC, 0x1B, 0x0B, 0x4E, 0x00, 0xC9, 0x44, 0x58 } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000492, 0x000047FD, { { 0x8C, 0x0B, 0x8B, 0xCE, 0xE0, 0xB0, 0x8F, 0xA9, 0x06, 0xC3, 0x98, 0xE6, 0x2E, 0x09, 0xB6, 0x93 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000492, 0x00004ACD, { { 0xDF, 0x87, 0xFE, 0x89, 0x59, 0xCC, 0x01, 0xD7, 0xC7, 0xEB, 0x16, 0xA4, 0x09, 0xAF, 0x5D, 0xC0 } } } }, // CD
	{ DE_DEU, kPlatformPC, { 0x00000492, 0x00004ACD, { { 0xDF, 0x87, 0xFE, 0x89, 0x59, 0xCC, 0x01, 0xD7, 0xC7, 0xEB, 0x16, 0xA4, 0x09, 0xAF, 0x5D, 0xC0 } } } }, // CD
	{ FR_FRA, kPlatformPC, { 0x00000492, 0x00004ACD, { { 0xDF, 0x87, 0xFE, 0x89, 0x59, 0xCC, 0x01, 0xD7, 0xC7, 0xEB, 0x16, 0xA4, 0x09, 0xAF, 0x5D, 0xC0 } } } }, // CD
	{ RU_RUS, kPlatformPC, { 0x00000492, 0x00004ACD, { { 0xDF, 0x87, 0xFE, 0x89, 0x59, 0xCC, 0x01, 0xD7, 0xC7, 0xEB, 0x16, 0xA4, 0x09, 0xAF, 0x5D, 0xC0 } } } }, // CD
	{ IT_ITA, kPlatformPC, { 0x00000492, 0x00004ACD, { { 0xDF, 0x87, 0xFE, 0x89, 0x59, 0xCC, 0x01, 0xD7, 0xC7, 0xEB, 0x16, 0xA4, 0x09, 0xAF, 0x5D, 0xC0 } } } }, // CD
	{ JA_JPN, kPlatformPC98, { 0x00000492, 0x00005893, { { 0x7C, 0x7E, 0xFB, 0x80, 0xD9, 0xB6, 0x16, 0x87, 0x80, 0xB7, 0x46, 0x9B, 0x96, 0x1A, 0x6A, 0xBE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLIngameSfxFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000008F2, 0x0001E5B6, { { 0x63, 0x5E, 0x37, 0xAA, 0x27, 0x80, 0x4C, 0x85, 0xB1, 0x9D, 0x7B, 0x1D, 0x64, 0xA3, 0xEB, 0x97 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x000008F2, 0x0001E5B7, { { 0x9E, 0xC8, 0xE8, 0x19, 0x2F, 0x58, 0x0B, 0xC7, 0x2D, 0x41, 0x72, 0xE7, 0xF4, 0x80, 0x03, 0xCB } } } }, // CD
	{ UNK_LANG, kPlatformPC98, { 0x000008EF, 0x0001E585, { { 0x85, 0x81, 0x5C, 0xA4, 0x34, 0x44, 0xF4, 0x58, 0xF9, 0x82, 0xEE, 0x0F, 0x6A, 0x0D, 0xA2, 0x7F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLIngameSfxIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000003E8, 0x0000E8D2, { { 0x19, 0x39, 0x17, 0xED, 0xAE, 0xDC, 0x7A, 0xAC, 0x45, 0x5F, 0x2D, 0xCD, 0x65, 0x8D, 0xAD, 0xAE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMusicTrackMapProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000000F0, 0x0000210D, { { 0x55, 0x25, 0x3E, 0x35, 0xD2, 0xD8, 0x13, 0xE3, 0x1D, 0xB1, 0xB3, 0x00, 0x2E, 0x17, 0x91, 0x2F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLIngameGMSfxIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000000FA, 0x00006281, { { 0x25, 0x89, 0xB0, 0x3B, 0x12, 0x09, 0x02, 0xF6, 0xFE, 0x76, 0xD5, 0xC9, 0x5B, 0x88, 0xAC, 0xAA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLIngameMT32SfxIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000000FA, 0x00006579, { { 0x16, 0x40, 0x1C, 0x09, 0x69, 0xA9, 0x0D, 0x6D, 0x4B, 0x0C, 0x99, 0xF0, 0x40, 0x5D, 0xBB, 0x6E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLIngamePcSpkSfxIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000000FA, 0x00005EFC, { { 0xA3, 0x5C, 0x69, 0xED, 0x13, 0xEC, 0x08, 0x0E, 0xFA, 0x72, 0x83, 0x0D, 0xD7, 0x8D, 0x9C, 0x70 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLSpellPropertiesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000118, 0x00000B06, { { 0x27, 0x69, 0x53, 0x01, 0xA0, 0xE3, 0x76, 0xAA, 0x33, 0xA4, 0x52, 0x11, 0x52, 0xB1, 0x0E, 0xDA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLGameShapeMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000114, 0x000038D3, { { 0xB2, 0x8A, 0x5D, 0x9A, 0x51, 0x63, 0x4D, 0x65, 0xE4, 0xF5, 0xBA, 0x88, 0x70, 0x6C, 0xA6, 0xF8 } } } }, // floppy + PC98
	{ UNK_LANG, kPlatformPC, { 0x00000114, 0x00003B97, { { 0x29, 0xE5, 0x0F, 0x51, 0xF0, 0x10, 0x35, 0x3E, 0x70, 0x3A, 0xAA, 0xFE, 0xD7, 0xD5, 0xAA, 0x9F } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLSceneItemOffsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000008, 0x00000300, { { 0x69, 0x80, 0x5A, 0x3E, 0x63, 0xC1, 0x04, 0x60, 0x09, 0x2F, 0x49, 0xD7, 0x26, 0x32, 0xAA, 0xE2 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCharInvIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x00000006, { { 0x19, 0x79, 0x4E, 0xFC, 0x05, 0x14, 0x89, 0x23, 0xEB, 0xCA, 0x94, 0x50, 0xE8, 0xD3, 0x81, 0x24 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCharInvDefsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000058, 0x00001D7A, { { 0x25, 0xE4, 0xEB, 0x6D, 0xBE, 0xEA, 0xBD, 0x9A, 0x9F, 0xA5, 0x9E, 0xEB, 0x3D, 0x03, 0x1D, 0x72 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCharDefsManProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x0000003D, { { 0xEB, 0x82, 0x32, 0x9D, 0x76, 0xC8, 0x3D, 0x5E, 0x8C, 0x26, 0x53, 0xDF, 0xC1, 0xFD, 0x0F, 0xC5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCharDefsWomanProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x0000003D, { { 0xEB, 0x82, 0x32, 0x9D, 0x76, 0xC8, 0x3D, 0x5E, 0x8C, 0x26, 0x53, 0xDF, 0xC1, 0xFD, 0x0F, 0xC5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCharDefsKieranProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x000002E3, { { 0xBF, 0xB1, 0x0F, 0x40, 0xBF, 0xA1, 0xD0, 0x2B, 0xC9, 0x80, 0x35, 0x40, 0xA9, 0xA3, 0x01, 0xC8 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCharDefsAkshelProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x000002FB, { { 0x47, 0x3C, 0x07, 0x15, 0x20, 0xE6, 0x90, 0x59, 0x55, 0xF2, 0xA7, 0xC3, 0x27, 0x22, 0xAB, 0xDC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLExpRequirementsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x00000C0A, { { 0x3F, 0x36, 0xFA, 0xE3, 0xB0, 0x76, 0x5E, 0xFF, 0xE9, 0xBA, 0xDF, 0xD0, 0x9D, 0xFF, 0xDD, 0x27 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMonsterModifiersProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x000002C6, { { 0x38, 0x9A, 0x8B, 0x50, 0xD2, 0x9B, 0x95, 0x38, 0x91, 0x02, 0xA9, 0xBE, 0x78, 0xE5, 0x89, 0x65 } } } }, // floppy + PC98
	{ UNK_LANG, kPlatformPC, { 0x00000018, 0x000002EE, { { 0x4E, 0x37, 0x56, 0xE3, 0x42, 0xB3, 0x15, 0x2C, 0x7E, 0x9B, 0x7E, 0x50, 0x32, 0x91, 0x55, 0xBE } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMonsterShiftOffsetsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x00000803, { { 0x30, 0x55, 0x74, 0x0D, 0xC7, 0x3B, 0xD9, 0x5C, 0x26, 0xF0, 0x4E, 0x8F, 0xE4, 0x4D, 0xCB, 0x2A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMonsterDirFlagsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x00000080, { { 0xE5, 0xA1, 0xE3, 0xCE, 0xA0, 0x5F, 0x15, 0xE9, 0x5B, 0x28, 0x90, 0xC0, 0xDF, 0x21, 0xEC, 0x24 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMonsterScaleYProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001E, 0x00000982, { { 0xE2, 0x71, 0x5F, 0x57, 0x4A, 0x8F, 0x50, 0xDB, 0x3E, 0xDA, 0xAB, 0x10, 0xEB, 0xDB, 0x0D, 0x14 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMonsterScaleXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000918, { { 0xF6, 0x14, 0xE6, 0x48, 0x4E, 0x5B, 0x43, 0xCC, 0xCE, 0x4E, 0x98, 0x71, 0x5A, 0xC2, 0x00, 0x1E } } } },
	{ UNK_LANG, kPlatformPC98, { 0x0000001D, 0x000008D2, { { 0x1C, 0x25, 0x38, 0xE2, 0xBB, 0xB2, 0xDB, 0x93, 0x1B, 0x25, 0xB6, 0x89, 0xA9, 0x9B, 0x0A, 0xFE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMonsterScaleWHProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000082, 0x00001D2A, { { 0x85, 0x7E, 0x18, 0xDD, 0x74, 0x1C, 0x62, 0x6F, 0xF4, 0xE5, 0xAF, 0x65, 0xEC, 0x6A, 0x90, 0xAD } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLFlyingObjectShpProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000082, 0x00000252, { { 0xDE, 0x9D, 0x89, 0xAF, 0x0F, 0x50, 0x14, 0x60, 0x68, 0xAF, 0x19, 0xD8, 0x54, 0x8A, 0x36, 0x27 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLInventoryDescProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x0000082D, { { 0x86, 0xB4, 0xB9, 0x50, 0xB6, 0xDA, 0x29, 0xB2, 0xC0, 0x0D, 0x34, 0x3F, 0x8D, 0x88, 0xAA, 0xE4 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLLevelShpListProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000007F, 0x00002090, { { 0x17, 0x31, 0x8A, 0xB5, 0x9B, 0x3A, 0xDA, 0x16, 0x9E, 0xE3, 0xD1, 0x5F, 0xB4, 0x7B, 0xB2, 0x25 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLLevelDatListProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000007F, 0x00001FB8, { { 0xF6, 0xE9, 0x98, 0x79, 0x51, 0xCA, 0xA0, 0x35, 0xE4, 0xD0, 0xA1, 0xCD, 0x23, 0x89, 0x7D, 0x11 } } } }, // floppy + PC98
	{ UNK_LANG, kPlatformPC, { 0x000000FF, 0x000047EC, { { 0x0D, 0xA5, 0xFD, 0x8A, 0x33, 0xDB, 0x93, 0x43, 0xE2, 0x57, 0x35, 0xEC, 0xA6, 0xCF, 0x7A, 0xA1 } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCompassDefsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000100, 0x000030EF, { { 0x6F, 0xF0, 0x46, 0x6E, 0xB3, 0x72, 0xCF, 0xC7, 0xE3, 0xAF, 0xBE, 0x63, 0xA1, 0x1C, 0x33, 0x20 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLItemPricesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000005C, 0x00001251, { { 0x18, 0x62, 0x5E, 0xE2, 0xE4, 0x2A, 0xB0, 0xA0, 0x8B, 0x8D, 0x9D, 0x07, 0x5F, 0x83, 0x53, 0xF7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLStashSetupProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x0000001E, { { 0x1C, 0x93, 0x66, 0x56, 0xDB, 0xD7, 0xA4, 0xB3, 0xE7, 0x2F, 0xEA, 0x88, 0xE2, 0xC8, 0x79, 0xD0 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscWallsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000024, 0x00000A2A, { { 0xAC, 0x4E, 0x73, 0x2C, 0xB0, 0xEE, 0x24, 0x0E, 0x66, 0x8D, 0x48, 0xE5, 0xCA, 0x6B, 0x7F, 0x7F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscShapeIndexProvider[] = {
	// LOL:
	{ UNK_LANG, kPlatformUnknown, { 0x00000024, 0x00000728, { { 0x14, 0xBA, 0x6D, 0x5C, 0x7D, 0x20, 0x0D, 0x35, 0xA7, 0xB0, 0x8D, 0x2F, 0x1D, 0x2A, 0x49, 0xA4 } } } },
	// EOB:
	{ UNK_LANG, kPlatformUnknown, { 0x00000024, 0x00000632, { { 0xBE, 0x3E, 0x84, 0x71, 0x89, 0x04, 0xC9, 0x1D, 0xCF, 0xE4, 0x3B, 0xD8, 0x2A, 0xF4, 0x0F, 0x54 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscOvlMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x0000001F, { { 0x9C, 0xF2, 0xCC, 0x48, 0x42, 0xC6, 0x76, 0x83, 0xD3, 0x1A, 0x43, 0x42, 0x7F, 0xEF, 0x19, 0x0F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscScaleWidthDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00000ABE, { { 0x28, 0x9A, 0xAA, 0x16, 0xC4, 0xFD, 0x52, 0xA9, 0x76, 0x98, 0x72, 0x0C, 0x2D, 0xE4, 0xB0, 0x57 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscScaleHeightDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x000009E8, { { 0x25, 0x35, 0x07, 0xBC, 0xF9, 0x82, 0x8B, 0x5B, 0x67, 0x7C, 0x38, 0xD1, 0xF8, 0x35, 0x81, 0xC7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscXProvider[] = {
	// LOL
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00001468, { { 0x55, 0xC5, 0x30, 0x76, 0x0A, 0xDC, 0xEC, 0xAB, 0x68, 0x9B, 0x61, 0xF0, 0x58, 0x78, 0x56, 0xA6 } } } },
	// EOB
	{ UNK_LANG, kPlatformUnknown, { 0x00000024, 0x00000BFA, { { 0x5F, 0x86, 0x9B, 0xDA, 0x5D, 0x6E, 0xC0, 0xB9, 0x29, 0x82, 0xA5, 0xD7, 0xC9, 0x34, 0x90, 0x63 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscYProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000024, 0x00000282, { { 0x09, 0x98, 0x3A, 0x33, 0x15, 0xA1, 0x4A, 0xFF, 0x76, 0x19, 0x2B, 0xB1, 0x74, 0x89, 0xF4, 0x37 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscTileIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x00000099, { { 0x18, 0xD9, 0x39, 0x27, 0x5B, 0x34, 0xAE, 0x7C, 0xA9, 0xA9, 0xDB, 0x42, 0x49, 0x61, 0x6B, 0x37 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscUnk2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000017, 0x000000D2, { { 0xDE, 0xDA, 0x75, 0x15, 0x2B, 0xDC, 0x90, 0x3F, 0xC9, 0x92, 0x04, 0x01, 0x23, 0x7A, 0xDA, 0x2E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDoorShapeIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000017, 0x0000000A, { { 0x2E, 0xC4, 0xA1, 0x47, 0x7C, 0xAE, 0xAD, 0xD8, 0x8A, 0x72, 0x95, 0x2F, 0x18, 0xC5, 0x08, 0x19 } } } }, // LOL
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x00000021, { { 0xE3, 0x00, 0x85, 0x1C, 0x13, 0xCE, 0x5D, 0xA7, 0xA2, 0x93, 0x9B, 0x56, 0x1A, 0x0C, 0x43, 0x3E } } } }, // EOB 1
	{ UNK_LANG, kPlatformUnknown, { 0x00000035, 0x0000000B, { { 0xC2, 0xBC, 0xCA, 0x95, 0x69, 0xE8, 0x3F, 0x1F, 0xC2, 0x1C, 0x37, 0x90, 0x63, 0x8F, 0xE6, 0x1D } } } }, // EOB 2
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDimData1Provider[] = {
	// LOL
	{ UNK_LANG, kPlatformUnknown, { 0x00000144, 0x0001007D, { { 0x18, 0x3D, 0xA5, 0xF7, 0x1A, 0x5A, 0x90, 0xA7, 0x4E, 0x66, 0x1A, 0x4E, 0x0C, 0x69, 0x58, 0x31 } } } },
	// EOB
	{ UNK_LANG, kPlatformUnknown, { 0x00000144, 0x00010115, { { 0x89, 0x37, 0x1C, 0x85, 0x53, 0xEE, 0xC0, 0xEC, 0x17, 0x26, 0x0B, 0xE5, 0xCC, 0x9C, 0x30, 0x58 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDimData2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000288, 0x00001BC2, { { 0x30, 0xD1, 0xD1, 0x35, 0x74, 0x2C, 0x86, 0x81, 0x23, 0xE2, 0x05, 0xCE, 0x75, 0x60, 0x3C, 0x55 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscBlockMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x00000012, { { 0x01, 0xEE, 0x32, 0xA6, 0x71, 0x15, 0x8D, 0xFB, 0x33, 0xF2, 0xD6, 0x8A, 0x30, 0x00, 0x10, 0x4B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDimMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x00000014, { { 0x4D, 0x53, 0x2E, 0xF2, 0xA3, 0xF9, 0xE2, 0xEC, 0x44, 0xBE, 0x5F, 0x04, 0x91, 0xF8, 0xE1, 0x04 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscOvlIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000028, 0x00000048, { { 0x3E, 0x8E, 0x62, 0xAF, 0xD1, 0x28, 0x39, 0x73, 0x0D, 0xD8, 0x4A, 0xA7, 0xF4, 0xD7, 0x32, 0x25 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscBlockIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00002200, { { 0xF4, 0x8B, 0x32, 0xC3, 0xD3, 0xFB, 0x46, 0xF2, 0xB8, 0x3A, 0x58, 0x39, 0x94, 0x57, 0x97, 0x4B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDoorY2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000080, 0x00000348, { { 0xC6, 0x58, 0x8B, 0xFE, 0x18, 0x72, 0x47, 0xF1, 0xB6, 0x3A, 0x0F, 0xFB, 0x3D, 0x99, 0x74, 0xD0 } } } }, // LOL
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x00000046, { { 0x35, 0x36, 0xBC, 0xD8, 0x63, 0x25, 0x31, 0xA9, 0x61, 0x8E, 0xF6, 0x54, 0x4A, 0x79, 0x17, 0xF8 } } } }, // EOB
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDoorFrameY1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x0000053B, { { 0xF0, 0x9E, 0xC5, 0xB1, 0xEA, 0x5A, 0x58, 0xBD, 0xAC, 0x7B, 0xB2, 0xD4, 0xFE, 0x3F, 0x4F, 0x51 } } } }, // EOB I
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x00000046, { { 0xD4, 0xA4, 0xEC, 0xA2, 0x99, 0xB6, 0x5E, 0x12, 0x98, 0xFF, 0xF2, 0x55, 0xC8, 0xBD, 0xC5, 0x8F } } } }, // EOB II
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDoorFrameY2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x0000053B, { { 0xF0, 0x9E, 0xC5, 0xB1, 0xEA, 0x5A, 0x58, 0xBD, 0xAC, 0x7B, 0xB2, 0xD4, 0xFE, 0x3F, 0x4F, 0x51 } } } }, // EOB I
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x00000150, { { 0x49, 0x7E, 0xF4, 0xDF, 0x8D, 0x04, 0x0A, 0xCE, 0x49, 0xBB, 0xA2, 0x1D, 0x8D, 0xC2, 0x14, 0x9E } } } }, // EOB II
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDoorFrameIndex1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x00000058, { { 0xC9, 0xAF, 0x1F, 0x68, 0xF1, 0xDE, 0x96, 0x9B, 0x3B, 0xCB, 0x56, 0xEC, 0x2E, 0x62, 0x9A, 0x0A } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kRpgCommonDscDoorFrameIndex2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x000000E8, { { 0x8C, 0x10, 0x56, 0xEA, 0x4D, 0x1A, 0x9C, 0xB2, 0x55, 0x54, 0xA5, 0x61, 0x1D, 0x19, 0x4E, 0x50 } } } },
	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscDoorScaleProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x0000024C, { { 0x8D, 0x83, 0x26, 0xEE, 0xDC, 0xF7, 0x13, 0xC0, 0xAA, 0x88, 0xC2, 0xAA, 0x66, 0xA7, 0x59, 0x41 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscDoor4Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000008, 0x00000103, { { 0x29, 0xC0, 0x4B, 0x7F, 0x36, 0x23, 0xBB, 0x38, 0x4C, 0x83, 0xC6, 0x9D, 0xB4, 0x8F, 0x29, 0x2E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscDoorXProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000480, 0x0001654D, { { 0x2A, 0x1F, 0xBF, 0xE3, 0xC4, 0xEF, 0x7E, 0xD1, 0x61, 0x51, 0xFE, 0x88, 0x8D, 0x1F, 0x59, 0x70 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLDscDoorYProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000480, 0x00026666, { { 0x06, 0xBF, 0xA4, 0xD4, 0x6E, 0x29, 0x42, 0xA2, 0xA0, 0x8E, 0x3C, 0x14, 0xF3, 0xD6, 0x3F, 0x87 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLScrollXTopProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x0000004B, { { 0x18, 0x1E, 0x6E, 0xE9, 0x34, 0xF0, 0x02, 0xC6, 0x57, 0x34, 0xDF, 0x55, 0xD9, 0x39, 0xE8, 0x98 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLScrollYTopProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x00000022, { { 0xF3, 0x20, 0x5A, 0xC1, 0xBB, 0x0C, 0x79, 0x52, 0x23, 0xC1, 0x36, 0x81, 0x70, 0x2F, 0x92, 0xFC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLScrollXBottomProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x0000004B, { { 0x18, 0x1E, 0x6E, 0xE9, 0x34, 0xF0, 0x02, 0xC6, 0x57, 0x34, 0xDF, 0x55, 0xD9, 0x39, 0xE8, 0x98 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLScrollYBottomProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x0000003C, { { 0x5B, 0x4F, 0xB7, 0xB5, 0x55, 0xA2, 0x9A, 0x21, 0xEF, 0xB4, 0x98, 0x47, 0x05, 0x57, 0x49, 0x55 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonDefsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000082A, 0x0000CAAE, { { 0xC1, 0x83, 0x0D, 0xA0, 0x66, 0x16, 0x3D, 0x31, 0xCE, 0x30, 0x9F, 0x4E, 0x00, 0x65, 0x5A, 0xC8 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x0000082A, 0x0000C34E, { { 0x7F, 0x9A, 0x0F, 0x28, 0x1A, 0x8F, 0x03, 0x46, 0x48, 0xEB, 0xC9, 0xB9, 0x23, 0x29, 0x5E, 0x50 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x0000082A, 0x0000C47B, { { 0xDF, 0x1A, 0x18, 0x1F, 0x58, 0x05, 0x1F, 0x56, 0xD8, 0x6D, 0xBB, 0x93, 0xEC, 0x35, 0x9D, 0xA5 } } } }, // CD
	{ UNK_LANG, kPlatformPC98, { 0x0000082A, 0x0000AB58, { { 0xDD, 0x2B, 0xA9, 0x54, 0x60, 0x25, 0x2C, 0x74, 0xF8, 0x5D, 0xC6, 0xD2, 0x2C, 0x1A, 0x24, 0x44 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonList1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000050, 0x00000A37, { { 0x0F, 0x73, 0xEC, 0xDD, 0xAB, 0xFF, 0x49, 0x46, 0x5E, 0x8F, 0x0D, 0xC3, 0xE7, 0x1B, 0x89, 0x51 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonList2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001E, 0x00000522, { { 0xEA, 0x41, 0x46, 0xE2, 0xFE, 0xAA, 0x7D, 0x5E, 0x89, 0x7F, 0xBF, 0x9B, 0x30, 0x60, 0x74, 0xF3 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonList3Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x0000023E, { { 0x70, 0xAA, 0xCA, 0xAC, 0x5C, 0x21, 0xCF, 0xA5, 0xBF, 0x7F, 0x5F, 0xBC, 0xF1, 0x24, 0x8A, 0xAF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonList4Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001E, 0x0000054D, { { 0x19, 0x2A, 0xBE, 0x7F, 0x94, 0x10, 0xA0, 0x60, 0x2A, 0x33, 0xD6, 0x11, 0x85, 0xF0, 0xA4, 0xA9 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonList5Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x0000045D, { { 0xE3, 0x7C, 0xC2, 0x36, 0x21, 0x46, 0xDB, 0xF3, 0xDD, 0x38, 0x4B, 0x40, 0xE0, 0x35, 0x09, 0xC3 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonList6Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001C, 0x000004C4, { { 0x21, 0x7C, 0x29, 0x3F, 0x95, 0x6F, 0x91, 0x8C, 0xB2, 0x30, 0x09, 0xA6, 0x7B, 0x48, 0x44, 0x8F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonList7Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000006, 0x0000021D, { { 0xDC, 0xCE, 0x1B, 0xEB, 0x11, 0x6D, 0xDE, 0x37, 0x17, 0xC8, 0x06, 0x51, 0xC3, 0x0C, 0xCB, 0xA6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLButtonList8Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x00000253, { { 0x0C, 0x7B, 0x10, 0x99, 0x93, 0xD0, 0x33, 0xCA, 0xAB, 0x8D, 0x7E, 0x24, 0xE5, 0x7E, 0x6C, 0x91 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLLegendDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000858, { { 0x63, 0x5E, 0x60, 0xC7, 0x62, 0x2C, 0x5D, 0x8F, 0x74, 0x71, 0x98, 0xB7, 0x09, 0xD2, 0x51, 0xC7 } } } },
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000A52, { { 0x81, 0xC5, 0xA4, 0xE7, 0x60, 0xDA, 0xD6, 0x5E, 0x19, 0xAB, 0xF3, 0xC7, 0xDD, 0xDB, 0x92, 0x5E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMapCursorOvlProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000019, 0x000009CD, { { 0xF6, 0xD2, 0xFA, 0x36, 0x62, 0x95, 0x1D, 0x99, 0x7F, 0x11, 0x5F, 0xA8, 0x4D, 0x47, 0x72, 0x40 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLMapStringIdProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000CFB, { { 0xE3, 0xC3, 0x41, 0x06, 0xD1, 0x71, 0x77, 0x78, 0xAD, 0x39, 0xAE, 0x2C, 0x16, 0x21, 0x45, 0xB7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLSpellbookAnimProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000038, 0x000003A1, { { 0x50, 0xA0, 0xF6, 0xA7, 0x53, 0x96, 0x86, 0x49, 0xB0, 0x8D, 0xA8, 0xB2, 0x2D, 0x9A, 0xE2, 0x1F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLSpellbookCoordsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x0000018F, { { 0xA5, 0xF6, 0x8A, 0x58, 0x9A, 0xC7, 0x3C, 0x3A, 0xB5, 0x87, 0x89, 0x87, 0x73, 0x51, 0x9B, 0x1B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLHealShapeFramesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000050, 0x000000F4, { { 0xC9, 0x6E, 0x39, 0xE1, 0xD7, 0xAD, 0x10, 0x4F, 0xE2, 0xFE, 0xDC, 0xAD, 0x00, 0x9D, 0x41, 0xEF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLLightningDefsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000014, 0x00000385, { { 0x68, 0x39, 0x65, 0xCB, 0xA9, 0x80, 0x90, 0xFB, 0xDD, 0x77, 0x0C, 0x76, 0x5A, 0xB5, 0x05, 0x03 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLFireballCoordsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000200, 0x0000FD81, { { 0xB3, 0xE0, 0x6F, 0x89, 0xCD, 0xE5, 0xA9, 0x6A, 0x4B, 0x61, 0x7A, 0x3F, 0x47, 0x26, 0x73, 0x58 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLCreditsProvider[] = {
	{ JA_JPN , kPlatformPC98, { 0x000005E7, 0x0001A1B0, { { 0x2A, 0xD0, 0x38, 0x84, 0x0C, 0x38, 0xCB, 0x52, 0x5D, 0x82, 0xBE, 0x03, 0x76, 0xFA, 0x0A, 0x4A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLoLHistoryProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000001D1, 0x00007F9B, { { 0x25, 0x10, 0x86, 0x40, 0xAC, 0x53, 0xFE, 0x11, 0x4D, 0xE2, 0xD9, 0x35, 0xD6, 0x89, 0xBB, 0x09 } } } },

	EXTRACT_END_ENTRY
};

// Provider tables

struct ExtractEntry {
	int id;
	const ExtractEntrySearchData *providers;
};

const ExtractEntry extractProviders[] = {
	{ k1ForestSeq, k1ForestSeqProvider },
	{ k1KallakWritingSeq, k1KallakWritingSeqProvider },
	{ k1KyrandiaLogoSeq, k1KyrandiaLogoSeqProvider },
	{ k1KallakMalcolmSeq, k1KallakMalcolmSeqProvider },
	{ k1MalcolmTreeSeq, k1MalcolmTreeSeqProvider },
	{ k1WestwoodLogoSeq, k1WestwoodLogoSeqProvider },
	{ k1Demo1Seq, k1Demo1SeqProvider },
	{ k1Demo2Seq, k1Demo2SeqProvider },
	{ k1Demo3Seq, k1Demo3SeqProvider },
	{ k1Demo4Seq, k1Demo4SeqProvider },
	{ k1AmuleteAnimSeq, k1AmuleteAnimSeqProvider },
	{ k1OutroReunionSeq, k1OutroReunionSeqProvider },
	{ k1IntroCPSStrings, k1IntroCPSStringsProvider },
	{ k1IntroCOLStrings, k1IntroCOLStringsProvider },
	{ k1IntroWSAStrings, k1IntroWSAStringsProvider },
	{ k1IntroStrings, k1IntroStringsProvider },
	{ k1OutroHomeString, k1OutroHomeStringProvider },
	{ k1RoomFilenames, k1RoomFilenamesProvider },
	{ k1RoomList, k1RoomListProvider },
	{ k1CharacterImageFilenames, k1CharacterImageFilenamesProvider },
	{ k1AudioTracks, k1AudioTracksProvider },
	{ k1AudioTracks2, k1AudioTracks2Provider },
	{ k1AudioTracksIntro, k1AudioTracksIntroProvider },
	{ k1ItemNames, k1ItemNamesProvider },
	{ k1TakenStrings, k1TakenStringsProvider },
	{ k1PlacedStrings, k1PlacedStringsProvider },
	{ k1DroppedStrings, k1DroppedStringsProvider },
	{ k1NoDropStrings, k1NoDropStringsProvider },
	{ k1PutDownString, k1PutDownStringProvider },
	{ k1WaitAmuletString, k1WaitAmuletStringProvider },
	{ k1BlackJewelString, k1BlackJewelStringProvider },
	{ k1PoisonGoneString, k1PoisonGoneStringProvider },
	{ k1HealingTipString, k1HealingTipStringProvider },
	{ k1WispJewelStrings, k1WispJewelStringsProvider },
	{ k1MagicJewelStrings, k1MagicJewelStringsProvider },
	{ k1ThePoisonStrings, k1ThePoisonStringsProvider },
	{ k1FluteStrings, k1FluteStringsProvider },
	{ k1FlaskFullString, k1FlaskFullStringProvider },
	{ k1FullFlaskString, k1FullFlaskStringProvider },
	{ k1VeryCleverString, k1VeryCleverStringProvider },
	{ k1NewGameString, k1NewGameStringProvider },
	{ k1DefaultShapes, k1DefaultShapesProvider },
	{ k1Healing1Shapes, k1Healing1ShapesProvider },
	{ k1Healing2Shapes, k1Healing2ShapesProvider },
	{ k1PoisonDeathShapes, k1PoisonDeathShapesProvider },
	{ k1FluteShapes, k1FluteShapesProvider },
	{ k1Winter1Shapes, k1Winter1ShapesProvider },
	{ k1Winter2Shapes, k1Winter2ShapesProvider },
	{ k1Winter3Shapes, k1Winter3ShapesProvider },
	{ k1DrinkShapes, k1DrinkShapesProvider },
	{ k1WispShapes, k1WispShapesProvider },
	{ k1MagicAnimShapes, k1MagicAnimShapesProvider },
	{ k1BranStoneShapes, k1BranStoneShapesProvider },
	{ k1SpecialPalette1, k1SpecialPalette1Provider },
	{ k1SpecialPalette2, k1SpecialPalette2Provider },
	{ k1SpecialPalette3, k1SpecialPalette3Provider },
	{ k1SpecialPalette4, k1SpecialPalette4Provider },
	{ k1SpecialPalette5, k1SpecialPalette5Provider },
	{ k1SpecialPalette6, k1SpecialPalette6Provider },
	{ k1SpecialPalette7, k1SpecialPalette7Provider },
	{ k1SpecialPalette8, k1SpecialPalette8Provider },
	{ k1SpecialPalette9, k1SpecialPalette9Provider },
	{ k1SpecialPalette10, k1SpecialPalette10Provider },
	{ k1SpecialPalette11, k1SpecialPalette11Provider },
	{ k1SpecialPalette12, k1SpecialPalette12Provider },
	{ k1SpecialPalette13, k1SpecialPalette13Provider },
	{ k1SpecialPalette14, k1SpecialPalette14Provider },
	{ k1SpecialPalette15, k1SpecialPalette15Provider },
	{ k1SpecialPalette16, k1SpecialPalette16Provider },
	{ k1SpecialPalette17, k1SpecialPalette17Provider },
	{ k1SpecialPalette18, k1SpecialPalette18Provider },
	{ k1SpecialPalette19, k1SpecialPalette19Provider },
	{ k1SpecialPalette20, k1SpecialPalette20Provider },
	{ k1SpecialPalette21, k1SpecialPalette21Provider },
	{ k1SpecialPalette22, k1SpecialPalette22Provider },
	{ k1SpecialPalette23, k1SpecialPalette23Provider },
	{ k1SpecialPalette24, k1SpecialPalette24Provider },
	{ k1SpecialPalette25, k1SpecialPalette25Provider },
	{ k1SpecialPalette26, k1SpecialPalette26Provider },
	{ k1SpecialPalette27, k1SpecialPalette27Provider },
	{ k1SpecialPalette28, k1SpecialPalette28Provider },
	{ k1SpecialPalette29, k1SpecialPalette29Provider },
	{ k1SpecialPalette30, k1SpecialPalette30Provider },
	{ k1SpecialPalette31, k1SpecialPalette31Provider },
	{ k1SpecialPalette32, k1SpecialPalette32Provider },
	{ k1SpecialPalette33, k1SpecialPalette33Provider },
	{ k1GUIStrings, k1GUIStringsProvider },
	{ k1ConfigStrings, k1ConfigStringsProvider },
	{ k1TownsMusicFadeTable, k1TownsMusicFadeTableProvider },
	{ k1TownsSFXwdTable, k1TownsSFXwdTableProvider },
	{ k1TownsSFXbtTable, k1TownsSFXbtTableProvider },
	{ k1TownsCDATable, k1TownsCDATableProvider },
	{ k1PC98StoryStrings, k1PC98StoryStringsProvider },
	{ k1PC98IntroSfx, k1PC98IntroSfxProvider },
	{ k1CreditsStrings, k1CreditsStringsProvider },
	{ k1AmigaIntroSFXTable, k1AmigaIntroSFXTableProvider },
	{ k1AmigaGameSFXTable, k1AmigaGameSFXTableProvider },
	{ k2SeqplayPakFiles, k2SeqplayPakFilesProvider },
	{ k2SeqplayStrings, k2SeqplayStringsProvider },
	{ k2SeqplaySfxFiles, k2SeqplaySfxFilesProvider },
	{ k2SeqplayTlkFiles, k2SeqplayTlkFilesProvider },
	{ k2SeqplaySeqData, k2SeqplaySeqDataProvider },
	{ k2SeqplayCredits, k2SeqplayCreditsProvider },
	{ k2SeqplayCreditsSpecial, k2SeqplayCreditsSpecialProvider },
	{ k2SeqplayIntroTracks, k2SeqplayIntroTracksProvider },
	{ k2SeqplayFinaleTracks, k2SeqplayFinaleTracksProvider },
	{ k2SeqplayIntroCDA, k2SeqplayIntroCDAProvider },
	{ k2SeqplayFinaleCDA, k2SeqplayFinaleCDAProvider },
	{ k2SeqplayShapeAnimData, k2SeqplayShapeAnimDataProvider },
	{ k2IngamePakFiles, k2IngamePakFilesProvider },
	{ k2IngameSfxFiles, k2IngameSfxFilesProvider },
	{ k2IngameSfxIndex, k2IngameSfxIndexProvider },
	{ k2IngameTracks, k2IngameTracksProvider },
	{ k2IngameCDA, k2IngameCDAProvider },
	{ k2IngameTalkObjIndex, k2IngameTalkObjIndexProvider },
	{ k2IngameTimJpStrings, k2IngameTimJpStringsProvider },
	{ k2IngameShapeAnimData, k2IngameShapeAnimDataProvider },
	{ k2IngameTlkDemoStrings, k2IngameTlkDemoStringsProvider },
	{ k3MainMenuStrings, k3MainMenuStringsProvider },
	{ k3MusicFiles, k3MusicFilesProvider },
	{ k3ScoreTable, k3ScoreTableProvider },
	{ k3SfxFiles, k3SfxFilesProvider },
	{ k3SfxMap, k3SfxMapProvider },
	{ k3ItemAnimData, k3ItemAnimDataProvider },
	{ k3ItemMagicTable, k3ItemMagicTableProvider },
	{ k3ItemStringMap, k3ItemStringMapProvider },

	{ kEoBBaseChargenStrings1, kEoBBaseChargenStrings1Provider },
	{ kEoBBaseChargenStrings2, kEoBBaseChargenStrings2Provider },
	{ kEoBBaseChargenStartLevels, kEoBBaseChargenStartLevelsProvider },
	{ kEoBBaseChargenStatStrings, kEoBBaseChargenStatStringsProvider },
	{ kEoBBaseChargenRaceSexStrings, kEoBBaseChargenRaceSexStringsProvider },
	{ kEoBBaseChargenClassStrings, kEoBBaseChargenClassStringsProvider },
	{ kEoBBaseChargenAlignmentStrings, kEoBBaseChargenAlignmentStringsProvider },
	{ kEoBBaseChargenEnterGameStrings, kEoBBaseChargenEnterGameStringsProvider },
	{ kEoBBaseChargenClassMinStats, kEoBBaseChargenClassMinStatsProvider },
	{ kEoBBaseChargenRaceMinStats, kEoBBaseChargenRaceMinStatsProvider },
	{ kEoBBaseChargenRaceMaxStats, kEoBBaseChargenRaceMaxStatsProvider },

	{ kEoBBaseSaveThrowTable1, kEoBBaseSaveThrowTable1Provider },
	{ kEoBBaseSaveThrowTable2, kEoBBaseSaveThrowTable2Provider },
	{ kEoBBaseSaveThrowTable3, kEoBBaseSaveThrowTable3Provider },
	{ kEoBBaseSaveThrowTable4, kEoBBaseSaveThrowTable4Provider },
	{ kEoBBaseSaveThrwLvlIndex, kEoBBaseSaveThrwLvlIndexProvider },
	{ kEoBBaseSaveThrwModDiv, kEoBBaseSaveThrwModDivProvider },
	{ kEoBBaseSaveThrwModExt, kEoBBaseSaveThrwModExtProvider },

	{ kEoBBasePryDoorStrings, kEoBBasePryDoorStringsProvider },
	{ kEoBBaseWarningStrings, kEoBBaseWarningStringsProvider },

	{ kEoBBaseItemSuffixStringsRings, kEoBBaseItemSuffixStringsRingsProvider },
	{ kEoBBaseItemSuffixStringsPotions, kEoBBaseItemSuffixStringsPotionsProvider },
	{ kEoBBaseItemSuffixStringsWands, kEoBBaseItemSuffixStringsWandsProvider },

	{ kEoBBaseRipItemStrings, kEoBBaseRipItemStringsProvider },
	{ kEoBBaseCursedString, kEoBBaseCursedStringProvider },
	{ kEoBBaseEnchantedString, kEoBBaseEnchantedStringProvider },
	{ kEoBBaseMagicObjectStrings, kEoBBaseMagicObjectStringsProvider },
	{ kEoBBaseMagicObject5String, kEoBBaseMagicObject5StringProvider },
	{ kEoBBasePatternSuffix, kEoBBasePatternSuffixProvider },
	{ kEoBBasePatternGrFix1, kEoBBasePatternGrFix1Provider },
	{ kEoBBasePatternGrFix2, kEoBBasePatternGrFix2Provider },
	{ kEoBBaseValidateArmorString, kEoBBaseValidateArmorStringProvider },
	{ kEoBBaseValidateCursedString, kEoBBaseValidateCursedStringProvider },
	{ kEoBBaseValidateNoDropString, kEoBBaseValidateNoDropStringProvider },
	{ kEoBBasePotionStrings, kEoBBasePotionStringsProvider },
	{ kEoBBaseWandString, kEoBBaseWandStringProvider },
	{ kEoBBaseItemMisuseStrings, kEoBBaseItemMisuseStringsProvider },

	{ kEoBBaseTakenStrings, kEoBBaseTakenStringsProvider },
	{ kEoBBasePotionEffectStrings, kEoBBasePotionEffectStringsProvider },

	{ kEoBBaseYesNoStrings, kEoBBaseYesNoStringsProvider },
	{ kRpgCommonMoreStrings, kRpgCommonMoreStringsProvider },
	{ kEoBBaseNpcMaxStrings, kEoBBaseNpcMaxStringsProvider },
	{ kEoBBaseOkStrings, kEoBBaseOkStringsProvider },
	{ kEoBBaseNpcJoinStrings, kEoBBaseNpcJoinStringsProvider },
	{ kEoBBaseCancelStrings, kEoBBaseCancelStringsProvider },
	{ kEoBBaseAbortStrings, kEoBBaseAbortStringsProvider },

	{ kEoBBaseMenuStringsMain, kEoBBaseMenuStringsMainProvider },
	{ kEoBBaseMenuStringsSaveLoad, kEoBBaseMenuStringsSaveLoadProvider },
	{ kEoBBaseMenuStringsOnOff, kEoBBaseMenuStringsOnOffProvider },
	{ kEoBBaseMenuStringsSpells, kEoBBaseMenuStringsSpellsProvider },
	{ kEoBBaseMenuStringsRest, kEoBBaseMenuStringsRestProvider },
	{ kEoBBaseMenuStringsDrop, kEoBBaseMenuStringsDropProvider },
	{ kEoBBaseMenuStringsExit, kEoBBaseMenuStringsExitProvider },
	{ kEoBBaseMenuStringsStarve, kEoBBaseMenuStringsStarveProvider },
	{ kEoBBaseMenuStringsScribe, kEoBBaseMenuStringsScribeProvider },
	{ kEoBBaseMenuStringsDrop2, kEoBBaseMenuStringsDrop2Provider },
	{ kEoBBaseMenuStringsHead, kEoBBaseMenuStringsHeadProvider },
	{ kEoBBaseMenuStringsPoison, kEoBBaseMenuStringsPoisonProvider },
	{ kEoBBaseMenuStringsMgc, kEoBBaseMenuStringsMgcProvider },
	{ kEoBBaseMenuStringsPrefs, kEoBBaseMenuStringsPrefsProvider },
	{ kEoBBaseMenuStringsRest2, kEoBBaseMenuStringsRest2Provider },
	{ kEoBBaseMenuStringsRest3, kEoBBaseMenuStringsRest3Provider },
	{ kEoBBaseMenuStringsRest4, kEoBBaseMenuStringsRest4Provider },
	{ kEoBBaseMenuStringsDefeat, kEoBBaseMenuStringsDefeatProvider },
	{ kEoBBaseMenuStringsTransfer, kEoBBaseMenuStringsTransferProvider },
	{ kEoBBaseMenuStringsSpec, kEoBBaseMenuStringsSpecProvider },
	{ kEoBBaseMenuStringsSpellNo, kEoBBaseMenuStringsSpellNoProvider },
	{ kEoBBaseMenuYesNoStrings, kEoBBaseMenuYesNoStringsProvider },

	{ kEoBBaseSpellLevelsMage, kEoBBaseSpellLevelsMageProvider },
	{ kEoBBaseSpellLevelsCleric, kEoBBaseSpellLevelsClericProvider },
	{ kEoBBaseNumSpellsCleric, kEoBBaseNumSpellsClericProvider },
	{ kEoBBaseNumSpellsWisAdj, kEoBBaseNumSpellsWisAdjProvider },
	{ kEoBBaseNumSpellsPal, kEoBBaseNumSpellsPalProvider },
	{ kEoBBaseNumSpellsMage, kEoBBaseNumSpellsMageProvider },

	{ kEoBBaseCharGuiStringsHp, kEoBBaseCharGuiStringsHpProvider },
	{ kEoBBaseCharGuiStringsWp1, kEoBBaseCharGuiStringsWp1Provider },
	{ kEoBBaseCharGuiStringsWp2, kEoBBaseCharGuiStringsWp2Provider },
	{ kEoBBaseCharGuiStringsWr, kEoBBaseCharGuiStringsWrProvider },
	{ kEoBBaseCharGuiStringsSt1, kEoBBaseCharGuiStringsSt1Provider },
	{ kEoBBaseCharGuiStringsSt2, kEoBBaseCharGuiStringsSt2Provider },
	{ kEoBBaseCharGuiStringsIn, kEoBBaseCharGuiStringsInProvider },

	{ kEoBBaseCharStatusStrings7, kEoBBaseCharStatusStrings7Provider },
	{ kEoBBaseCharStatusStrings81, kEoBBaseCharStatusStrings81Provider },
	{ kEoBBaseCharStatusStrings82, kEoBBaseCharStatusStrings82Provider },
	{ kEoBBaseCharStatusStrings9, kEoBBaseCharStatusStrings9Provider },
	{ kEoBBaseCharStatusStrings12, kEoBBaseCharStatusStrings12Provider },
	{ kEoBBaseCharStatusStrings131, kEoBBaseCharStatusStrings131Provider },
	{ kEoBBaseCharStatusStrings132, kEoBBaseCharStatusStrings132Provider },

	{ kEoBBaseLevelGainStrings, kEoBBaseLevelGainStringsProvider },
	{ kEoBBaseExperienceTable0, kEoBBaseExperienceTable0Provider },
	{ kEoBBaseExperienceTable1, kEoBBaseExperienceTable1Provider },
	{ kEoBBaseExperienceTable2, kEoBBaseExperienceTable2Provider },
	{ kEoBBaseExperienceTable3, kEoBBaseExperienceTable3Provider },
	{ kEoBBaseExperienceTable4, kEoBBaseExperienceTable4Provider },

	{ kEoBBaseWllFlagPreset, kEoBBaseWllFlagPresetProvider },
	{ kEoBBaseDscShapeCoords, kEoBBaseDscShapeCoordsProvider },
	{ kEoBBaseDscDoorScaleOffs, kEoBBaseDscDoorScaleOffsProvider },
	{ kEoBBaseDscDoorScaleMult1, kEoBBaseDscDoorScaleMult1Provider },
	{ kEoBBaseDscDoorScaleMult2, kEoBBaseDscDoorScaleMult2Provider },
	{ kEoBBaseDscDoorScaleMult3, kEoBBaseDscDoorScaleMult3Provider },
	{ kEoBBaseDscDoorScaleMult4, kEoBBaseDscDoorScaleMult4Provider },
	{ kEoBBaseDscDoorScaleMult5, kEoBBaseDscDoorScaleMult5Provider },
	{ kEoBBaseDscDoorScaleMult6, kEoBBaseDscDoorScaleMult6Provider },
	{ kEoBBaseDscDoorType5Offs, kEoBBaseDscDoorType5OffsProvider },
	{ kEoBBaseDscDoorXE, kEoBBaseDscDoorXEProvider },
	{ kEoBBaseDscDoorY1, kEoBBaseDscDoorY1Provider },
	{ kEoBBaseDscDoorY3, kEoBBaseDscDoorY3Provider },
	{ kEoBBaseDscDoorY4, kEoBBaseDscDoorY4Provider },
	{ kEoBBaseDscDoorY5, kEoBBaseDscDoorY5Provider },
	{ kEoBBaseDscDoorY6, kEoBBaseDscDoorY6Provider },
	{ kEoBBaseDscDoorY7, kEoBBaseDscDoorY7Provider },
	{ kEoBBaseDscDoorCoordsExt, kEoBBaseDscDoorCoordsExtProvider },
	{ kEoBBaseDscItemPosIndex, kEoBBaseDscItemPosIndexProvider },
	{ kEoBBaseDscItemShpX, kEoBBaseDscItemShpXProvider },
	{ kEoBBaseDscItemPosUnk, kEoBBaseDscItemPosUnkProvider },
	{ kEoBBaseDscItemTileIndex, kEoBBaseDscItemTileIndexProvider },
	{ kEoBBaseDscItemShapeMap, kEoBBaseDscItemShapeMapProvider },
	{ kEoBBaseDscTelptrShpCoords, kEoBBaseDscTelptrShpCoordsProvider },

	{ kEoBBasePortalSeqData, kEoBBasePortalSeqDataProvider },
	{ kEoBBaseManDef, kEoBBaseManDefProvider },
	{ kEoBBaseManWord, kEoBBaseManWordProvider },
	{ kEoBBaseManPrompt, kEoBBaseManPromptProvider },

	{ kEoBBaseDscMonsterFrmOffsTbl1, kEoBBaseDscMonsterFrmOffsTbl1Provider },
	{ kEoBBaseDscMonsterFrmOffsTbl2, kEoBBaseDscMonsterFrmOffsTbl2Provider },

	{ kEoBBaseInvSlotX, kEoBBaseInvSlotXProvider },
	{ kEoBBaseInvSlotY, kEoBBaseInvSlotYProvider },
	{ kEoBBaseSlotValidationFlags, kEoBBaseSlotValidationFlagsProvider },

	{ kEoBBaseProjectileWeaponTypes, kEoBBaseProjectileWeaponTypesProvider },
	{ kEoBBaseWandTypes, kEoBBaseWandTypesProvider },

	{ kEoBBaseDrawObjPosIndex, kEoBBaseDrawObjPosIndexProvider },
	{ kEoBBaseFlightObjFlipIndex, kEoBBaseFlightObjFlipIndexProvider },
	{ kEoBBaseFlightObjShpMap, kEoBBaseFlightObjShpMapProvider },
	{ kEoBBaseFlightObjSclIndex, kEoBBaseFlightObjSclIndexProvider },

	{ kEoBBaseBookNumbers, kEoBBaseBookNumbersProvider },
	{ kEoBBaseMageSpellsList, kEoBBaseMageSpellsListProvider },
	{ kEoBBaseClericSpellsList, kEoBBaseClericSpellsListProvider },
	{ kEoBBaseSpellNames, kEoBBaseSpellNamesProvider },
	{ kEoBBaseMagicStrings1, kEoBBaseMagicStrings1Provider },
	{ kEoBBaseMagicStrings2, kEoBBaseMagicStrings2Provider },
	{ kEoBBaseMagicStrings3, kEoBBaseMagicStrings3Provider },
	{ kEoBBaseMagicStrings4, kEoBBaseMagicStrings4Provider },
	{ kEoBBaseMagicStrings6, kEoBBaseMagicStrings6Provider },
	{ kEoBBaseMagicStrings7, kEoBBaseMagicStrings7Provider },
	{ kEoBBaseMagicStrings8, kEoBBaseMagicStrings8Provider },

	{ kEoBBaseExpObjectTlMode, kEoBBaseExpObjectTlModeProvider },
	{ kEoBBaseExpObjectTblIndex, kEoBBaseExpObjectTblIndexProvider },
	{ kEoBBaseExpObjectShpStart, kEoBBaseExpObjectShpStartProvider },
	{ kEoBBaseExpObjectTbl1, kEoBBaseExpObjectTbl1Provider },
	{ kEoBBaseExpObjectTbl2, kEoBBaseExpObjectTbl2Provider },
	{ kEoBBaseExpObjectTbl3, kEoBBaseExpObjectTbl3Provider },
	{ kEoBBaseExpObjectY, kEoBBaseExpObjectYProvider },

	{ kEoBBaseSparkDefSteps, kEoBBaseSparkDefStepsProvider },
	{ kEoBBaseSparkDefSubSteps, kEoBBaseSparkDefSubStepsProvider },
	{ kEoBBaseSparkDefShift, kEoBBaseSparkDefShiftProvider },
	{ kEoBBaseSparkDefAdd, kEoBBaseSparkDefAddProvider },
	{ kEoBBaseSparkDefX, kEoBBaseSparkDefXProvider },
	{ kEoBBaseSparkDefY, kEoBBaseSparkDefYProvider },
	{ kEoBBaseSparkOfFlags1, kEoBBaseSparkOfFlags1Provider },
	{ kEoBBaseSparkOfFlags2, kEoBBaseSparkOfFlags2Provider },
	{ kEoBBaseSparkOfShift, kEoBBaseSparkOfShiftProvider },
	{ kEoBBaseSparkOfX, kEoBBaseSparkOfXProvider },
	{ kEoBBaseSparkOfY, kEoBBaseSparkOfYProvider },

	{ kEoBBaseSpellProperties, kEoBBaseSpellPropertiesProvider },
	{ kEoBBaseMagicFlightProps, kEoBBaseMagicFlightPropsProvider },
	{ kEoBBaseTurnUndeadEffect, kEoBBaseTurnUndeadEffectProvider },
	{ kEoBBaseBurningHandsDest, kEoBBaseBurningHandsDestProvider },
	{ kEoBBaseConeOfColdDest1, kEoBBaseConeOfColdDest1Provider },
	{ kEoBBaseConeOfColdDest2, kEoBBaseConeOfColdDest2Provider },
	{ kEoBBaseConeOfColdDest3, kEoBBaseConeOfColdDest3Provider },
	{ kEoBBaseConeOfColdDest4, kEoBBaseConeOfColdDest4Provider },
	{ kEoBBaseConeOfColdGfxTbl, kEoBBaseConeOfColdGfxTblProvider },

	{ kEoB1MainMenuStrings, kEoB1MainMenuStringsProvider },
	{ kEoB1BonusStrings, kEoB1BonusStringsProvider },

	{ kEoB1IntroFilesOpening, kEoB1IntroFilesOpeningProvider },
	{ kEoB1IntroFilesTower, kEoB1IntroFilesTowerProvider },
	{ kEoB1IntroFilesOrb, kEoB1IntroFilesOrbProvider },
	{ kEoB1IntroFilesWdEntry, kEoB1IntroFilesWdEntryProvider },
	{ kEoB1IntroFilesKing, kEoB1IntroFilesKingProvider },
	{ kEoB1IntroFilesHands, kEoB1IntroFilesHandsProvider },
	{ kEoB1IntroFilesWdExit, kEoB1IntroFilesWdExitProvider },
	{ kEoB1IntroFilesTunnel, kEoB1IntroFilesTunnelProvider },
	{ kEoB1IntroOpeningFrmDelay, kEoB1IntroOpeningFrmDelayProvider },
	{ kEoB1IntroWdEncodeX, kEoB1IntroWdEncodeXProvider },
	{ kEoB1IntroWdEncodeY, kEoB1IntroWdEncodeYProvider },
	{ kEoB1IntroWdEncodeWH, kEoB1IntroWdEncodeWHProvider },
	{ kEoB1IntroWdDsX, kEoB1IntroWdDsXProvider },
	{ kEoB1IntroWdDsY, kEoB1IntroWdDsYProvider },
	{ kEoB1IntroTvlX1, kEoB1IntroTvlX1Provider },
	{ kEoB1IntroTvlY1, kEoB1IntroTvlY1Provider },
	{ kEoB1IntroTvlX2, kEoB1IntroTvlX2Provider },
	{ kEoB1IntroTvlY2, kEoB1IntroTvlY2Provider },
	{ kEoB1IntroTvlW, kEoB1IntroTvlWProvider },
	{ kEoB1IntroTvlH, kEoB1IntroTvlHProvider },

	{ kEoB1DoorShapeDefs, kEoB1DoorShapeDefsProvider },
	{ kEoB1DoorSwitchShapeDefs, kEoB1DoorSwitchShapeDefsProvider },
	{ kEoB1DoorSwitchCoords, kEoB1DoorSwitchCoordsProvider },
	{ kEoB1MonsterProperties, kEoB1MonsterPropertiesProvider },

	{ kEoB1EnemyMageSpellList, kEoB1EnemyMageSpellListProvider },
	{ kEoB1EnemyMageSfx, kEoB1EnemyMageSfxProvider },
	{ kEoB1BeholderSpellList, kEoB1BeholderSpellListProvider },
	{ kEoB1BeholderSfx, kEoB1BeholderSfxProvider },
	{ kEoB1TurnUndeadString, kEoB1TurnUndeadStringProvider },

	{ kEoB1CgaMappingDefault, kEoB1CgaMappingDefaultProvider },
	{ kEoB1CgaMappingAlt, kEoB1CgaMappingAltProvider },
	{ kEoB1CgaMappingInv, kEoB1CgaMappingInvProvider },
	{ kEoB1CgaMappingItemsL, kEoB1CgaMappingItemsLProvider },
	{ kEoB1CgaMappingItemsS, kEoB1CgaMappingItemsSProvider },
	{ kEoB1CgaMappingThrown, kEoB1CgaMappingThrownProvider },
	{ kEoB1CgaMappingIcons, kEoB1CgaMappingIconsProvider },
	{ kEoB1CgaMappingDeco, kEoB1CgaMappingDecoProvider },
	{ kEoB1CgaLevelMappingIndex, kEoB1CgaLevelMappingIndexProvider },
	{ kEoB1CgaMappingLevel0, kEoB1CgaMappingLevel0Provider },
	{ kEoB1CgaMappingLevel1, kEoB1CgaMappingLevel1Provider },
	{ kEoB1CgaMappingLevel2, kEoB1CgaMappingLevel2Provider },
	{ kEoB1CgaMappingLevel3, kEoB1CgaMappingLevel3Provider },
	{ kEoB1CgaMappingLevel4, kEoB1CgaMappingLevel4Provider },

	{ kEoB1NpcShpData, kEoB1NpcShpDataProvider },
	{ kEoB1NpcSubShpIndex1, kEoB1NpcSubShpIndex1Provider },
	{ kEoB1NpcSubShpIndex2, kEoB1NpcSubShpIndex2Provider },
	{ kEoB1NpcSubShpY, kEoB1NpcSubShpYProvider },
	{ kEoB1Npc0Strings, kEoB1Npc0StringsProvider },
	{ kEoB1Npc11Strings, kEoB1Npc11StringsProvider },
	{ kEoB1Npc12Strings, kEoB1Npc12StringsProvider },
	{ kEoB1Npc21Strings, kEoB1Npc21StringsProvider },
	{ kEoB1Npc22Strings, kEoB1Npc22StringsProvider },
	{ kEoB1Npc31Strings, kEoB1Npc31StringsProvider },
	{ kEoB1Npc32Strings, kEoB1Npc32StringsProvider },
	{ kEoB1Npc4Strings, kEoB1Npc4StringsProvider },
	{ kEoB1Npc5Strings, kEoB1Npc5StringsProvider },
	{ kEoB1Npc6Strings, kEoB1Npc6StringsProvider },
	{ kEoB1Npc7Strings, kEoB1Npc7StringsProvider },

	{ kEoB2MainMenuStrings, kEoB2MainMenuStringsProvider },

	{ kEoB2TransferPortraitFrames, kEoB2TransferPortraitFramesProvider },
	{ kEoB2TransferConvertTable, kEoB2TransferConvertTableProvider },
	{ kEoB2TransferItemTable, kEoB2TransferItemTableProvider },
	{ kEoB2TransferExpTable, kEoB2TransferExpTableProvider },
	{ kEoB2TransferStrings1, kEoB2TransferStrings1Provider },
	{ kEoB2TransferStrings2, kEoB2TransferStrings2Provider },
	{ kEoB2TransferLabels, kEoB2TransferLabelsProvider },

	{ kEoB2IntroStrings, kEoB2IntroStringsProvider },
	{ kEoB2IntroCPSFiles, kEoB2IntroCPSFilesProvider },
	{ kEob2IntroAnimData00, kEob2IntroAnimData00Provider },
	{ kEob2IntroAnimData01, kEob2IntroAnimData01Provider },
	{ kEob2IntroAnimData02, kEob2IntroAnimData02Provider },
	{ kEob2IntroAnimData03, kEob2IntroAnimData03Provider },
	{ kEob2IntroAnimData04, kEob2IntroAnimData04Provider },
	{ kEob2IntroAnimData05, kEob2IntroAnimData05Provider },
	{ kEob2IntroAnimData06, kEob2IntroAnimData06Provider },
	{ kEob2IntroAnimData07, kEob2IntroAnimData07Provider },
	{ kEob2IntroAnimData08, kEob2IntroAnimData08Provider },
	{ kEob2IntroAnimData09, kEob2IntroAnimData09Provider },
	{ kEob2IntroAnimData10, kEob2IntroAnimData10Provider },
	{ kEob2IntroAnimData11, kEob2IntroAnimData11Provider },
	{ kEob2IntroAnimData12, kEob2IntroAnimData12Provider },
	{ kEob2IntroAnimData13, kEob2IntroAnimData13Provider },
	{ kEob2IntroAnimData14, kEob2IntroAnimData14Provider },
	{ kEob2IntroAnimData15, kEob2IntroAnimData15Provider },
	{ kEob2IntroAnimData16, kEob2IntroAnimData16Provider },
	{ kEob2IntroAnimData17, kEob2IntroAnimData17Provider },
	{ kEob2IntroAnimData18, kEob2IntroAnimData18Provider },
	{ kEob2IntroAnimData19, kEob2IntroAnimData19Provider },
	{ kEob2IntroAnimData20, kEob2IntroAnimData20Provider },
	{ kEob2IntroAnimData21, kEob2IntroAnimData21Provider },
	{ kEob2IntroAnimData22, kEob2IntroAnimData22Provider },
	{ kEob2IntroAnimData23, kEob2IntroAnimData23Provider },
	{ kEob2IntroAnimData24, kEob2IntroAnimData24Provider },
	{ kEob2IntroAnimData25, kEob2IntroAnimData25Provider },
	{ kEob2IntroAnimData26, kEob2IntroAnimData26Provider },
	{ kEob2IntroAnimData27, kEob2IntroAnimData27Provider },
	{ kEob2IntroAnimData28, kEob2IntroAnimData28Provider },
	{ kEob2IntroAnimData29, kEob2IntroAnimData29Provider },
	{ kEob2IntroAnimData30, kEob2IntroAnimData30Provider },
	{ kEob2IntroAnimData31, kEob2IntroAnimData31Provider },
	{ kEob2IntroAnimData32, kEob2IntroAnimData32Provider },
	{ kEob2IntroAnimData33, kEob2IntroAnimData33Provider },
	{ kEob2IntroAnimData34, kEob2IntroAnimData34Provider },
	{ kEob2IntroAnimData35, kEob2IntroAnimData35Provider },
	{ kEob2IntroAnimData36, kEob2IntroAnimData36Provider },
	{ kEob2IntroAnimData37, kEob2IntroAnimData37Provider },
	{ kEob2IntroAnimData38, kEob2IntroAnimData38Provider },
	{ kEob2IntroAnimData39, kEob2IntroAnimData39Provider },
	{ kEob2IntroAnimData40, kEob2IntroAnimData40Provider },
	{ kEob2IntroAnimData41, kEob2IntroAnimData41Provider },
	{ kEob2IntroAnimData42, kEob2IntroAnimData42Provider },
	{ kEob2IntroAnimData43, kEob2IntroAnimData43Provider },
	{ kEoB2IntroShapes00, kEoB2IntroShapes00Provider },
	{ kEoB2IntroShapes01, kEoB2IntroShapes01Provider },
	{ kEoB2IntroShapes04, kEoB2IntroShapes04Provider },
	{ kEoB2IntroShapes07, kEoB2IntroShapes07Provider },

	{ kEoB2FinaleStrings, kEoB2FinaleStringsProvider },
	{ kEoB2CreditsData, kEoB2CreditsDataProvider },
	{ kEoB2FinaleCPSFiles, kEoB2FinaleCPSFilesProvider },
	{ kEob2FinaleAnimData00, kEob2FinaleAnimData00Provider },
	{ kEob2FinaleAnimData01, kEob2FinaleAnimData01Provider },
	{ kEob2FinaleAnimData02, kEob2FinaleAnimData02Provider },
	{ kEob2FinaleAnimData03, kEob2FinaleAnimData03Provider },
	{ kEob2FinaleAnimData04, kEob2FinaleAnimData04Provider },
	{ kEob2FinaleAnimData05, kEob2FinaleAnimData05Provider },
	{ kEob2FinaleAnimData06, kEob2FinaleAnimData06Provider },
	{ kEob2FinaleAnimData07, kEob2FinaleAnimData07Provider },
	{ kEob2FinaleAnimData08, kEob2FinaleAnimData08Provider },
	{ kEob2FinaleAnimData09, kEob2FinaleAnimData09Provider },
	{ kEob2FinaleAnimData10, kEob2FinaleAnimData10Provider },
	{ kEob2FinaleAnimData11, kEob2FinaleAnimData11Provider },
	{ kEob2FinaleAnimData12, kEob2FinaleAnimData12Provider },
	{ kEob2FinaleAnimData13, kEob2FinaleAnimData13Provider },
	{ kEob2FinaleAnimData14, kEob2FinaleAnimData14Provider },
	{ kEob2FinaleAnimData15, kEob2FinaleAnimData15Provider },
	{ kEob2FinaleAnimData16, kEob2FinaleAnimData16Provider },
	{ kEob2FinaleAnimData17, kEob2FinaleAnimData17Provider },
	{ kEob2FinaleAnimData18, kEob2FinaleAnimData18Provider },
	{ kEob2FinaleAnimData19, kEob2FinaleAnimData19Provider },
	{ kEob2FinaleAnimData20, kEob2FinaleAnimData20Provider },
	{ kEoB2FinaleShapes00, kEoB2FinaleShapes00Provider },
	{ kEoB2FinaleShapes03, kEoB2FinaleShapes03Provider },
	{ kEoB2FinaleShapes07, kEoB2FinaleShapes07Provider },
	{ kEoB2FinaleShapes09, kEoB2FinaleShapes09Provider },
	{ kEoB2FinaleShapes10, kEoB2FinaleShapes10Provider },

	{ kEoB2NpcShapeData, kEoB2NpcShapeDataProvider },
	{ kEoBBaseClassModifierFlags, kEoBBaseClassModifierFlagsProvider },

	{ kEoBBaseMonsterStepTable01, kEoBBaseMonsterStepTable01Provider },
	{ kEoBBaseMonsterStepTable02, kEoBBaseMonsterStepTable02Provider },
	{ kEoBBaseMonsterStepTable1, kEoBBaseMonsterStepTable1Provider },
	{ kEoBBaseMonsterStepTable2, kEoBBaseMonsterStepTable2Provider },
	{ kEoBBaseMonsterStepTable3, kEoBBaseMonsterStepTable3Provider },
	{ kEoBBaseMonsterCloseAttPosTable1, kEoBBaseMonsterCloseAttPosTable1Provider },
	{ kEoBBaseMonsterCloseAttPosTable21, kEoBBaseMonsterCloseAttPosTable21Provider },
	{ kEoBBaseMonsterCloseAttPosTable22, kEoBBaseMonsterCloseAttPosTable22Provider },
	{ kEoBBaseMonsterCloseAttUnkTable, kEoBBaseMonsterCloseAttUnkTableProvider },
	{ kEoBBaseMonsterCloseAttChkTable1, kEoBBaseMonsterCloseAttChkTable1Provider },
	{ kEoBBaseMonsterCloseAttChkTable2, kEoBBaseMonsterCloseAttChkTable2Provider },
	{ kEoBBaseMonsterCloseAttDstTable1, kEoBBaseMonsterCloseAttDstTable1Provider },
	{ kEoBBaseMonsterCloseAttDstTable2, kEoBBaseMonsterCloseAttDstTable2Provider },

	{ kEoBBaseMonsterProximityTable, kEoBBaseMonsterProximityTableProvider },
	{ kEoBBaseFindBlockMonstersTable, kEoBBaseFindBlockMonstersTableProvider },
	{ kEoBBaseMonsterDirChangeTable, kEoBBaseMonsterDirChangeTableProvider },
	{ kEoBBaseMonsterDistAttStrings, kEoBBaseMonsterDistAttStringsProvider },
	{ kEoBBaseEncodeMonsterDefs, kEoBBaseEncodeMonsterDefsProvider },
	{ kEoBBaseNpcPresets, kEoBBaseNpcPresetsProvider },
	{ kEoB2Npc1Strings, kEoB2Npc1StringsProvider },
	{ kEoB2Npc2Strings, kEoB2Npc2StringsProvider },
	{ kEoB2MonsterDustStrings, kEoB2MonsterDustStringsProvider },
	{ kEoB2DreamSteps, kEoB2DreamStepsProvider },
	{ kEoB2KheldranStrings, kEoB2KheldranStringsProvider },
	{ kEoB2HornStrings, kEoB2HornStringsProvider },
	{ kEoB2HornSounds, kEoB2HornSoundsProvider },
	{ kEoB2WallOfForceDsX, kEoB2WallOfForceDsXProvider },
	{ kEoB2WallOfForceDsY, kEoB2WallOfForceDsYProvider },
	{ kEoB2WallOfForceNumW, kEoB2WallOfForceNumWProvider },
	{ kEoB2WallOfForceNumH, kEoB2WallOfForceNumHProvider },
	{ kEoB2WallOfForceShpId, kEoB2WallOfForceShpIdProvider },

	{ kLoLIngamePakFiles, kLoLIngamePakFilesProvider },
	{ kLoLCharacterDefs, kLoLCharacterDefsProvider },
	{ kLoLIngameSfxFiles, kLoLIngameSfxFilesProvider },
	{ kLoLIngameSfxIndex, kLoLIngameSfxIndexProvider },
	{ kLoLMusicTrackMap, kLoLMusicTrackMapProvider },
	{ kLoLIngameGMSfxIndex, kLoLIngameGMSfxIndexProvider },
	{ kLoLIngameMT32SfxIndex, kLoLIngameMT32SfxIndexProvider },
	{ kLoLIngamePcSpkSfxIndex, kLoLIngamePcSpkSfxIndexProvider },
	{ kLoLSpellProperties, kLoLSpellPropertiesProvider },
	{ kLoLGameShapeMap, kLoLGameShapeMapProvider },
	{ kLoLSceneItemOffs, kLoLSceneItemOffsProvider },
	{ kLoLCharInvIndex, kLoLCharInvIndexProvider },
	{ kLoLCharInvDefs, kLoLCharInvDefsProvider },
	{ kLoLCharDefsMan, kLoLCharDefsManProvider },
	{ kLoLCharDefsWoman, kLoLCharDefsWomanProvider },
	{ kLoLCharDefsKieran, kLoLCharDefsKieranProvider },
	{ kLoLCharDefsAkshel, kLoLCharDefsAkshelProvider },
	{ kLoLExpRequirements, kLoLExpRequirementsProvider },
	{ kLoLMonsterModifiers, kLoLMonsterModifiersProvider },
	{ kLoLMonsterShiftOffsets, kLoLMonsterShiftOffsetsProvider },
	{ kLoLMonsterDirFlags, kLoLMonsterDirFlagsProvider },
	{ kLoLMonsterScaleY, kLoLMonsterScaleYProvider },
	{ kLoLMonsterScaleX, kLoLMonsterScaleXProvider },
	{ kLoLMonsterScaleWH, kLoLMonsterScaleWHProvider },
	{ kLoLFlyingObjectShp, kLoLFlyingObjectShpProvider },
	{ kLoLInventoryDesc, kLoLInventoryDescProvider },
	{ kLoLLevelShpList, kLoLLevelShpListProvider },
	{ kLoLLevelDatList, kLoLLevelDatListProvider },
	{ kLoLCompassDefs, kLoLCompassDefsProvider },
	{ kLoLItemPrices, kLoLItemPricesProvider },
	{ kLoLStashSetup, kLoLStashSetupProvider },
	{ kLoLDscWalls, kLoLDscWallsProvider },
	{ kRpgCommonDscShapeIndex, kRpgCommonDscShapeIndexProvider },
	{ kLoLDscOvlMap, kLoLDscOvlMapProvider },
	{ kLoLDscScaleWidthData, kLoLDscScaleWidthDataProvider },
	{ kLoLDscScaleHeightData, kLoLDscScaleHeightDataProvider },
	{ kRpgCommonDscX, kRpgCommonDscXProvider },
	{ kLoLDscY, kLoLDscYProvider },
	{ kRpgCommonDscTileIndex, kRpgCommonDscTileIndexProvider },
	{ kRpgCommonDscUnk2, kRpgCommonDscUnk2Provider },
	{ kRpgCommonDscDoorShapeIndex, kRpgCommonDscDoorShapeIndexProvider },
	{ kRpgCommonDscDimData1, kRpgCommonDscDimData1Provider },
	{ kRpgCommonDscDimData2, kRpgCommonDscDimData2Provider },
	{ kRpgCommonDscBlockMap, kRpgCommonDscBlockMapProvider },
	{ kRpgCommonDscDimMap, kRpgCommonDscDimMapProvider },
	{ kLoLDscOvlIndex, kLoLDscOvlIndexProvider },
	{ kRpgCommonDscBlockIndex, kRpgCommonDscBlockIndexProvider },
	{ kRpgCommonDscDoorY2, kRpgCommonDscDoorY2Provider },
	{ kRpgCommonDscDoorFrameY1, kRpgCommonDscDoorFrameY1Provider },
	{ kRpgCommonDscDoorFrameY2, kRpgCommonDscDoorFrameY2Provider },
	{ kRpgCommonDscDoorFrameIndex1, kRpgCommonDscDoorFrameIndex1Provider },
	{ kRpgCommonDscDoorFrameIndex2, kRpgCommonDscDoorFrameIndex2Provider },
	{ kLoLDscDoorScale, kLoLDscDoorScaleProvider },
	{ kLoLDscDoor4, kLoLDscDoor4Provider },
	{ kLoLDscDoorX, kLoLDscDoorXProvider },
	{ kLoLDscDoorY, kLoLDscDoorYProvider },
	{ kLoLScrollXTop, kLoLScrollXTopProvider },
	{ kLoLScrollYTop, kLoLScrollYTopProvider },
	{ kLoLScrollXBottom, kLoLScrollXBottomProvider },
	{ kLoLScrollYBottom, kLoLScrollYBottomProvider },
	{ kLoLButtonDefs, kLoLButtonDefsProvider },
	{ kLoLButtonList1, kLoLButtonList1Provider },
	{ kLoLButtonList2, kLoLButtonList2Provider },
	{ kLoLButtonList3, kLoLButtonList3Provider },
	{ kLoLButtonList4, kLoLButtonList4Provider },
	{ kLoLButtonList5, kLoLButtonList5Provider },
	{ kLoLButtonList6, kLoLButtonList6Provider },
	{ kLoLButtonList7, kLoLButtonList7Provider },
	{ kLoLButtonList8, kLoLButtonList8Provider },
	{ kLoLLegendData, kLoLLegendDataProvider },
	{ kLoLMapCursorOvl, kLoLMapCursorOvlProvider },
	{ kLoLMapStringId, kLoLMapStringIdProvider },
	{ kLoLSpellbookAnim, kLoLSpellbookAnimProvider },
	{ kLoLSpellbookCoords, kLoLSpellbookCoordsProvider },
	{ kLoLHealShapeFrames, kLoLHealShapeFramesProvider },
	{ kLoLLightningDefs, kLoLLightningDefsProvider },
	{ kLoLFireballCoords, kLoLFireballCoordsProvider },
	{ kLoLCredits, kLoLCreditsProvider },
	{ kLoLHistory, kLoLHistoryProvider },
	{ -1, NULL }
};

} // end of anonymous namespace

ExtractEntryList getProvidersForId(int id) {
	ExtractEntryList list;

	for (const ExtractEntry *p = extractProviders; p->id != -1; ++p) {
		if (p->id == id) {
			for (const ExtractEntrySearchData *d = p->providers; d->hint.size != 0; ++d)
				list.push_back(*d);
		}
	}

	return list;
}
