/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "CardArea.h"

#include <Lum/Base/DateTime.h>

#define FADE_STEP        5
#define TIME_FRACTION    20

CardArea::Selection::Selection(size_t cardIndex)
 : cardIndex(cardIndex)
{
  // no code
}

size_t CardArea::Selection::GetCardIndex() const
{
  return cardIndex;
}

bool CardArea::Entry::IsEmpty() const
{
  return state==stateEmpty || state==stateDisappears;
}

bool CardArea::Entry::IsSet() const
{
  return state==stateFilled || state==stateAppears;
}

CardArea::CardArea()
 : backgroundColor(0,0.5,0,Lum::OS::display->GetColor(Lum::OS::Display::backgroundColor)),
   cardWidth(0),cardHeight(0),
   columns(0),rows(0),
   timerAction(new Lum::Model::Action())
{
  images.resize(40);

  SetCanFocus(true);

  AttachModel(timerAction);
}

void CardArea::Unselect()
{
  for (size_t i=0; i<entries.size(); i++) {
    entries[i].selected=false;
  }
}

void CardArea::FadeInitial()
{
  bool fading=false;

  for (size_t i=0; i<entries.size(); i++) {
    if (entries[i].state==Entry::stateAppears || entries[i].state==Entry::stateDisappears) {
      //DrawCard(i);

      fading=true;
    }
  }

  Redraw();

  if (fading) {
    Lum::OS::display->AddTimer(0,1000000/TIME_FRACTION,timerAction);
  }
  else {
    if (fadeFinishedAction.Valid()) {
      fadeFinishedAction->Trigger();
    }
  }
}

void CardArea::Fade()
{
  bool fading=false;

  for (size_t i=0; i<entries.size(); i++) {
    if (entries[i].state==Entry::stateAppears) {
      if (entries[i].visibility<100-FADE_STEP) {
        entries[i].visibility+=FADE_STEP;

        fading=true;
      }
      else {
        entries[i].state=Entry::stateFilled;
        entries[i].visibility=100;
      }

      //DrawCard(i);
    }
    else if (entries[i].state==Entry::stateDisappears) {
      if (entries[i].visibility>FADE_STEP) {
        entries[i].visibility-=FADE_STEP;

        fading=true;
      }
      else {
        entries[i].state=Entry::stateEmpty;
        entries[i].visibility=100;
      }

      //DrawCard(i);
    }
  }

  Redraw();

  if (fading) {
    Lum::OS::display->AddTimer(0,1000000/TIME_FRACTION,timerAction);
  }
  else {
    if (fadeFinishedAction.Valid()) {
      fadeFinishedAction->Trigger();
    }
  }
}

void CardArea::DrawCardInternal(size_t row, size_t column)
{
  Lum::OS::DrawInfo *draw=GetDrawInfo();

  Card card=cards[entries[columns*row+column].index];

  Lum::OS::FontRef font=Lum::OS::display->GetFont();

  if (entries[columns*row+column].visibility!=100) {
    return;
  }

  draw->PushForeground(Lum::OS::Display::whiteColor);
  draw->FillRectangle(this->x+(column+1)*5+column*cardWidth,this->y+(row+1)*5+row*cardHeight,
                      cardWidth,cardHeight);
  draw->PopForeground();

  draw->PushForeground(Lum::OS::Display::blackColor);
  draw->DrawRectangle(this->x+(column+1)*5+column*cardWidth,this->y+(row+1)*5+row*cardHeight,
                      cardWidth,cardHeight);
  draw->PopForeground();

  draw->PushForeground(Lum::OS::Display::blackColor);
  draw->PushFont(font);
  draw->DrawString(this->x+(column+1)*5+column*cardWidth+(cardWidth-font->StringWidth(card.GetColorName()))/2,
                   this->y+(row+1)*5+row*cardHeight+font->ascent,
                   card.GetColorName());
  draw->DrawString(this->x+(column+1)*5+column*cardWidth+(cardWidth-font->StringWidth(card.GetValueName()))/2,
                   this->y+(row+1)*5+row*cardHeight+font->height+font->ascent,
                   card.GetValueName());
  draw->PopFont();
  draw->PopForeground();
}

void CardArea::DrawCard(size_t row, size_t column)
{
  Lum::OS::DrawInfo *draw=GetDrawInfo();

  draw->PushForeground(backgroundColor);
  draw->FillRectangle(this->x+(column+1)*5+column*cardWidth,this->y+(row+1)*5+row*cardHeight,
                      cardWidth,cardHeight);
  draw->PopForeground();

  if (entries[columns*row+column].state==Entry::stateAppears) {
    Card card=cards[entries[columns*row+column].index];

    if (images[card.GetIndex()].Valid()) {
      Lum::Images::ImageRef image;
      size_t                visibility=entries[columns*row+column].visibility;

      image=images[card.GetIndex()]->Clone();
      image->ModifyRelative(0,0,0,-(100-visibility));
      image->Draw(draw,this->x+(column+1)*5+column*cardWidth,this->y+(row+1)*5+row*cardHeight);
    }
    else {
      DrawCardInternal(row,column);
    }
  }
  else if (entries[columns*row+column].state==Entry::stateDisappears) {
    Card card=oldCards[entries[columns*row+column].index];

    if (images[card.GetIndex()].Valid()) {
      Lum::Images::ImageRef image;
      size_t                visibility=entries[columns*row+column].visibility;

      image=images[card.GetIndex()]->Clone();
      image->ModifyRelative(0,0,0,-(100-visibility));
      image->Draw(draw,this->x+(column+1)*5+column*cardWidth,this->y+(row+1)*5+row*cardHeight);
    }
    else {
      DrawCardInternal(row,column);
    }
  }
  else if (entries[columns*row+column].state==Entry::stateFilled) {
    Card card=cards[entries[columns*row+column].index];

    if (entries[columns*row+column].selected) {
      if (images[card.GetIndex()].Valid()) {
        Lum::Images::ImageRef image;

        image=images[card.GetIndex()]->Clone();
        image->ModifyRelative(-30,-30,-30);
        image->Draw(draw,this->x+(column+1)*5+column*cardWidth,this->y+(row+1)*5+row*cardHeight);
      }
      else {
        DrawCardInternal(row,column);
      }
    }
    else {
      if (images[card.GetIndex()].Valid()) {
        Lum::Images::ImageRef image;

        image=images[card.GetIndex()];
        image->Draw(draw,this->x+(column+1)*5+column*cardWidth,this->y+(row+1)*5+row*cardHeight);
      }
      else {
        DrawCardInternal(row,column);
      }
    }
  }
}

void CardArea::DrawCard(size_t index)
{
  DrawCard(index/columns,index%columns);
}

void CardArea::SetSize(size_t columns, size_t rows)
{
  this->columns=columns;
  this->rows=rows;

  entries.resize(columns*rows);

  for (size_t i=0; i<entries.size(); i++) {
    entries[i].index=0;
    entries[i].state=Entry::stateEmpty;
    entries[i].selected=false;
  }
}

void CardArea::SetImages(const std::vector<Lum::Images::ImageRef>& images)
{
  assert(images.size()==40);

  this->images=images;

  Redraw();
}

void CardArea::SetSelectionAction(Lum::Model::Action* action)
{
  selectionAction=action;
}

void CardArea::SetFadeFinishedAction(Lum::Model::Action* action)
{
  fadeFinishedAction=action;
}

void CardArea::SetCards(const std::vector<Card>& cards, bool fade)
{
  bool redraw=false;

  Lum::OS::display->RemoveTimer(timerAction);

  // If we are currently fading...finished that
  for (size_t i=0; i<entries.size(); i++) {
    if (entries[i].state==Entry::stateAppears) {
      entries[i].state=Entry::stateFilled;
      entries[i].visibility=100;
      redraw=true;
    }
    else if (entries[i].state==Entry::stateDisappears) {
      entries[i].state=Entry::stateEmpty;
      entries[i].visibility=100;
      redraw=true;
    }

    if (entries[i].selected) {
      entries[i].selected=false;
      redraw=true;
    };
  }

  if (redraw) {
    Redraw();
  }

  oldCards=this->cards;

  // First remove all cards in the area that are not parts of the new card set
  for (size_t i=0; i<entries.size(); i++) {
    if (entries[i].IsSet()) {
      size_t c=0;

      while (c<cards.size() && this->cards[entries[i].index]!=cards[c]) {
        c++;
      }

      if (c<cards.size()) {
        entries[i].index=c;
      }
      else {
        entries[i].state=Entry::stateDisappears;
        entries[i].visibility=100-FADE_STEP;
      }
    }
  }

  this->cards=cards;

  // New add all cards in the new card set that are not yet in the area
  for (size_t c=0; c<cards.size(); c++) {
    bool found=false;

    for (size_t i=0; i<entries.size(); i++) {
      if (entries[i].IsSet() && this->cards[entries[i].index]==cards[c]) {
        found=true;
        break;
      }
    }

    if (!found) {
      size_t i=0;

      while (entries[i].IsSet()) {
        i++;
      }

      entries[i].index=c;
      entries[i].state=Entry::stateAppears;
      entries[i].visibility=FADE_STEP;
    }
  }

  if (fade) {
    FadeInitial();
  }
  else {
    for (size_t i=0; i<entries.size(); i++) {
      if (entries[i].state==Entry::stateAppears) {
        entries[i].state=Entry::stateFilled;
        entries[i].visibility=100;
      }
      else if (entries[i].state==Entry::stateDisappears) {
        entries[i].state=Entry::stateEmpty;
        entries[i].visibility=100;
      }
    }

    Redraw();
  }
}

void CardArea::CalcSize()
{
  cardWidth=0;
  cardHeight=0;

  for (size_t i=0; i<images.size(); i++) {
    if (images[i].Valid()) {
      cardWidth=std::max(cardWidth,images[i]->GetWidth());
      cardHeight=std::max(cardHeight,images[i]->GetHeight());
    }
  }

  if (cardWidth==0 || cardHeight==0) {
    cardWidth=55;
    cardHeight=101;

    for (size_t i=0; i<images.size(); i++) {
      images[i]=NULL;
    }
  }

  minWidth=columns*cardWidth+(columns+1)*5;
  minHeight=rows*cardHeight+(rows+1)*5;

  width=minWidth;
  height=minHeight;

  Object::CalcSize();
}

void CardArea::Draw(int x, int y, size_t w, size_t h)
{
  Object::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  Lum::OS::DrawInfo *draw=GetDrawInfo();

  //DrawBackground(x,y,w,h);
  draw->PushForeground(backgroundColor);
  draw->FillRectangle(x,y,w,h);
  draw->PopForeground();

  for (size_t j=0; j<rows; j++) {
    for (size_t i=0; i<columns; i++) {
      DrawCard(j,i);
    }
  }
}

bool CardArea::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event) && event.button==Lum::OS::MouseEvent::button1) {

    for (size_t j=0; j<rows; j++) {
      for (size_t i=0; i<columns; i++) {
        if (event.x>=this->x+(int)((i+1)*5+i*cardWidth) && event.x<this->x+(int)((i+1)*5+(i+1)*cardWidth) &&
            event.y>=this->y+(int)((j+1)*5+j*cardHeight) && event.y<this->y+(int)((j+1)*5+(j+1)*cardHeight) &&
            entries[j*columns+i].IsSet()) {

          if (entries[j*columns+i].selected) {
            if (selectionAction.Valid()) {
              Selection selection(entries[j*columns+i].index);

              selectionAction->Trigger(selection);
            }
          }
          else {
            Unselect();

            entries[j*columns+i].selected=true;

            Redraw();
          }

          return true;
        }
      }
    }
  }

  return false;
}

void CardArea::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==timerAction && timerAction->IsFinished()) {
    Fade();
  }

  Object::Resync(model,msg);
}


