#!/usr/bin/python
# -*- coding: utf-8 -*-
# Sayhoo: an audio navigation help for velib' like projects.
# Copyright 2008 Pierre Amadio
# pierre.amadio@libertysurf.fr
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


from sayhooGlobals import _
import sayhooGlobals as G
import math
import threading
import sys 
import os
import tilenames as tn
import gc
try:
    import sqlite3
except:
    from pysqlite2 import dbapi2 as sqlite3



class Node:
    def __init__(self,id_node=False,lat=False,lon=False):
        self.id_node=id_node
        self.lat=float(lat)
        self.lon=float(lon)

    def calculate_nautical_distance(self,another_node):
        lat1=self.deg2rad(self.lat)
        lon1=self.deg2rad(self.lon)
        lat2=self.deg2rad(another_node.lat)
        lon2=self.deg2rad(another_node.lon)
        dlat = lat2 - lat1
        dlon = lon2 - lon1
        slat = math.sin(dlat / 2.0)
        slon = math.sin(dlon / 2.0)
        a = (slat * slat) + (math.cos(lat1) * math.cos(lat2) * slon * slon)
        return ((2.0 * math.atan2(math.sqrt(a), math.sqrt(1.0 - a))) * G.EARTH_RADIUS)

    def deg2rad(self,deg):
        return G.deg2rad(deg)

    def rad2deg(self,rad):
        return G.rad2deg(rad)

    def nauticalmiles2meter(self,d):
        return d*1852.0

    def calculate_metric_distance(self,another_node):
        nautical=self.calculate_nautical_distance(another_node)
        meters=self.nauticalmiles2meter(nautical)
        return meters

    """
     Let's assume the earth is flat to make things easier.
     It's not like Paris is large enough to be curved...
     We just assume lat and lon are regular coordinate.
    """
    def calculate_simple_distance(self,another_node):
        lat1=self.lat
        lon1=self.lon
        lat2=another_node.lat
        lon2=another_node.lon
        dlat=lat1-lat2
        dlon=lon1-lon2
        return(math.sqrt(dlat*dlat + dlon*dlon))

    """
    Calculate the bearing between the current node and 
    another one.
    """
    def calculate_bearing(self,another_node):
        lat1=self.lat
        lon1=self.lon
        lat2=another_node.lat
        lon2=another_node.lon
        dlon = self.deg2rad(lon2 - lon1)
        lat1 = self.deg2rad(lat1)
        lat2 = self.deg2rad(lat2)

        y = math.sin(dlon) * math.cos(lat2)
        x = (math.cos(lat1) * math.sin(lat2)) - \
            (math.sin(lat1) * math.cos(lat2) * math.cos(dlon))
        dlon = self.rad2deg(math.atan2(y, x))
        if dlon < 0.0:
            dlon += 360.0
        return float(dlon)

    def print_summary(self):
        print "Node id=%s (%s,%s)"%(self.id_node,self.lat,self.lon)
        #for k in self.tags:
        #    print k,self.tags[k].encode("utf8")


class Way:
    #routeTypes = ('cycle','car','train','foot','horse')
    routeTypes = ('cycle','car')

    def __init__(self,id_way=False,nodes=[],tags={}):
        self.id_way=id_way
        self.nodes=nodes
        self.tags={}
        self.access={}
        self.is_reversible=True

        if len(tags.keys()):
            self.set_tags(tags)

    def set_tags(self,tags):
        self.tags=tags
        highway=self.equivalent(self.tags.get('highway', ''))
        railway=self.equivalent(self.tags.get('railway', ''))
        oneway=self.tags.get('oneway', '')

        self.is_reversible=not oneway in('yes','true','1')
        self.access['cycle'] = highway in ('primary','secondary','tertiary','unclassified','minor','cycleway','residential', 'track','service')
        self.access['car'] = highway in ('motorway','trunk','primary','secondary','tertiary','unclassified','minor','residential', 'service')
        self.access['train'] = railway in('rail','light_rail','subway')
        self.access['foot'] = self.access['cycle'] or highway in('footway','steps')
        self.access['horse'] = highway in ('track','unclassified','bridleway')
        

    def equivalent(self,tag):
        """Simplifies a bunch of tags to nearly-equivalent ones"""
        equivalent = { \
            "primary_link":"primary",
            "trunk":"primary",
            "trunk_link":"primary",
            "secondary_link":"secondary",
            "tertiary":"secondary",
            "tertiary_link":"secondary",
            "residential":"unclassified",
            "minor":"unclassified",
            "steps":"footway",
            "driveway":"service",
            "pedestrian":"footway",
            "bridleway":"cycleway",
            "track":"cycleway",
            "arcade":"footway",
            "canal":"river",
            "riverbank":"river",
            "lake":"river",
            "light_rail":"railway"
            }  
        try:
            return(equivalent[tag])
        except KeyError:
            return(tag)

    def print_summary(self):
        print "Way Summary:%s"%self.id_way
        ak=self.tags.keys()
        ak.sort()
        #print "is_reversible",self.is_reversible
        #print "access",self.access
        for k in ak:
            print k+":",self.tags[k].encode('utf8')
        
        print "Nodes:",self.nodes
        print "__"

class MapArea:
    """
    This is supposed to be equivalent to the LoadOsm class in pyroute.
    """
    def __init__(self,center_node,max_distance,nodes={},ways={}):
        self.center=center_node
        self.max_distance=max_distance
        self.nodes=nodes
        self.ways=ways

        #store edges's weight
        self.routing={}
        #store node usable per type of roads
        self.routeableNodes={}
        self.velib_stations=[]

    def print_summary(self):
        print "Map Area: ",self.center.lat,self.center.lon,self.max_distance
        #nearest=Node(self.nearest_node_id,self.db)
        print "________Nodes___________"
        for i in self.nodes.keys():
            self.nodes[i].print_summary()

        
        print "_______Ways_____________"
        for i in self.ways.keys():
            print "**"
            self.ways[i].print_summary()
        

        
        print "_______Routing_________"
        for i in self.routing:
            print "__"
            print i
            print self.routing[i]
        

    def find_nearest_node_for(self,lat,lon):
        min=9999999999999999999999
        out=False
        target=Node(lat=lat,lon=lon)
        for nid in self.nodes:
            cur_node_dic=self.nodes[nid]
            cur_node=Node(id_node=nid,lat=cur_node_dic["lat"],lon=cur_node_dic['lon'])
            dist=cur_node.calculate_metric_distance(target)
            if dist<min:
                #print "New dist",dist
                min=dist
                out=nid
        return out
            
            
class MapAreaCreator:
    def __init__(self,db_location=""):
        print "Creating MapAreaCreator with db_location",db_location
        self.db_location=db_location
        self.maparea=False
        
    def prepare_area_for(self,lat,lon,max_distance):
        #print "we got to prepare_area_for ",lat,lon,max_distance
        #print "lat=%s lon=%s"%(lat,lon)
        db=sqlite3.connect(self.db_location)
        cursor=db.cursor()

        real_center=Node(lat=lat,lon=lon)
        self.maparea=MapArea(center_node=real_center,max_distance=max_distance)
        
        """
            Getting the list of nodes in the area.
            We do not store them yet, it s just to get 
            a list of ways in the area (the area will get all the
            nodes of all the ways, even if the node is far away.
        """
        print "getting first list of nodes"
        nodes_id=[]
        snt="select id_node,lat,lon from nodes order by (($LAT-lat)*($LAT-lat)+($LON-lon)*($LON-lon))"
        cursor.execute(snt,(lat,lon))
        for sql_info in cursor.fetchall():
            tmpNode=Node(id_node=sql_info[0],lat=sql_info[1],lon=sql_info[2])

            if max_distance>0:
                if tmpNode.calculate_metric_distance(real_center)<max_distance:
                    nodes_id.append(sql_info[0])
                else:
                    break
            else:
                nodes_id.append(sql_info[0])

        """
             Getting a list of ways based on the nodes in view.
        """
        print "getting list of ways"
        ways_id_list=[]
        for cur_node_id in nodes_id:
            tags={}
            snt="select id_way from ways_nodes where id_node=?"
            cursor.execute(snt,(cur_node_id,))
            for line in cursor.fetchall():
                if line[0] not in ways_id_list:
                    ways_id_list.append(line[0])
                                        
        """
            Now that we have a list of ways in view, lets fill data.
        """
        print "filling ways data"
        for cur_way_id in ways_id_list:
            nodes_in_way=[]
            tags_in_way={}
            snt="select key,value from ways_tags where id_way=? and key in ('highway','name','oneway','foot','bicyle','cycleway','cycycleway')"
            cursor.execute(snt,(cur_way_id,))
            for line in cursor.fetchall():
                key=line[0]
                value=line[1]
                tags_in_way[key]=value
            
            snt="select id_node,indice from ways_nodes where id_way=? order by indice"
            cursor.execute(snt,(cur_way_id,))

            lines=cursor.fetchall()
            for line in lines:
                id_node=line[0]
                nodes_in_way.append(id_node)
                if id_node not in nodes_id:
                    nodes_id.append(id_node)
            
            curWay=Way(id_way=cur_way_id,nodes=nodes_in_way,tags=tags_in_way)
            self.maparea.ways[cur_way_id]=curWay

        print "filling nodes data"
        for id_node in nodes_id:
            snt="select lat,lon from nodes where id_node=?"
            cursor.execute(snt,(id_node,))
            line=cursor.fetchone()
            curNode=Node(id_node=id_node,lat=line[0],lon=line[1])
            tags={}
            """
            snt="select key,value from nodes_tags where id_node=?"
            cursor.execute(snt,(id_node,))
            lines=cursor.fetchall()
            for line in lines:
                tags[line[0]]=line[1]
            """
            
            self.maparea.nodes[id_node]=curNode

        """Computing edges"""
        for routeType in Way.routeTypes:
            self.maparea.routing[routeType]={}
            self.maparea.routeableNodes[routeType]={}

        print "computing edges"
        for cur_way_id in self.maparea.ways:
            w=self.maparea.ways[cur_way_id]
            
            highway=w.equivalent(w.tags.get('highway', ''))
            railway=w.equivalent(w.tags.get('railway', ''))
            last=-1
            for node_id in w.nodes:
                if last != -1:
                    for routeType in Way.routeTypes:
#                        if(w.access[routeType]):
                        if(w.access.get(routeType)):
                            if highway:
                                weight=self.getWeight(routeType,highway)
                            elif railway:
                                weight=self.getWeight(routeType,railway)
                            if not weight:
                                #w.print_summary()
                                print "zut"
                                print w.access
                                print last,node_id,routeType,weight
                                weight=0
                                #sys.exit()
                                
                            self.add_link(last,node_id,routeType,weight)
                            if w.is_reversible or routeType=='foot':
                                self.add_link(node_id,last,routeType,weight)
                last=node_id            


        print "map area is done"
        
    def getWeight(self,transport,wayType):
        #print "getWeight transport=%s wayType=%s"%(transport,wayType)
        Weightings = { \
            'motorway': {'car':10},
            'trunk':    {'car':10, 'cycle':0.05},
            'primary':  {'cycle': 0.3, 'car':2, 'foot':1, 'horse':0.1},
            'secondary': {'cycle': 1, 'car':1.5, 'foot':1, 'horse':0.2},
            'tertiary': {'cycle': 1, 'car':1, 'foot':1, 'horse':0.3},
            'unclassified': {'cycle': 1, 'car':1, 'foot':1, 'horse':1},
            'minor': {'cycle': 1, 'car':1, 'foot':1, 'horse':1},
            'cycleway': {'cycle': 3, 'foot':0.2},
            'residential': {'cycle': 3, 'car':0.7, 'foot':1, 'horse':1},
            'track': {'cycle': 1, 'car':1, 'foot':1, 'horse':1, 'mtb':3},
            'service': {'cycle': 1, 'car':1, 'foot':1, 'horse':1},
            'bridleway': {'cycle': 0.8, 'foot':1, 'horse':10, 'mtb':3},
            'footway': {'cycle': 0.2, 'foot':1},
            'steps': {'foot':1, 'cycle':0.3},
            'rail':{'train':1},
            'light_rail':{'train':1},
            'subway':{'train':1}
            }

        try:
            return(Weightings[wayType][transport])
        except KeyError:
            # Default: if no weighting is defined, then assume it can't be routed
            return(0)

    def add_link(self,fr,to,routeType,weight=1):
        """Add a routeable edge to the scenario"""
        self.maparea.routeableNodes[routeType][fr]=True
        try:
            if to in self.maparea.routing[routeType][fr].keys():
                return
            self.maparea.routing[routeType][fr][to]= weight
        except KeyError:
            self.maparea.routing[routeType][fr]={to:weight}

        return True



class Router:
    def __init__(self,map_area):
        self.map_area=map_area
        self.search_end=False
        self.queue = []

    def do_route(self,start_id,end_id,transport):
        #self.search_end=Node(end_id,self.map_area.db)
        search_end_dic=self.map_area.nodes[end_id]
        search_end=Node(id_node=end_id,lat=search_end_dic['lat'],lon=search_end_dic['lon'])
        self.search_end=search_end
        print "end is found"
        closed=[start_id]
        self.queue = []

        blankQueueItem = {'end':-1,'distance':0,'nodes':str(start_id)}
        try:
            for i,weight in self.map_area.routing[transport][start_id].items():
                self.add_to_queue(start_id,i, blankQueueItem, weight)
        except KeyError:
            return("no such node",[])

        count=0
        while count<10000:
            count+=1
            try:
                #print "queue=",self.queue
                next_item=self.queue.pop(0)
            except IndexError:
                return("no_route",[])
            #print "next_item",next_item
            x = next_item['end']
            if x in closed:
                continue
            if x == end_id:
                route_nodes=[int(i) for i in next_item['nodes'].split(",")]
                return('success',route_nodes)
            closed.append(x)
            try:
                for i,weight in self.map_area.routing[transport][x].items():
                    if not i in closed:
                        self.add_to_queue(x,i,next_item,weight)
            except KeyError:
                pass
        else:
            return('gave_up',[])

    def add_to_queue(self,start_id,end_id,cur_queue,weight=1):
        for test in self.queue:
            if test['end'] == end_id:
                return
        
        if(weight==0):
            return

        start_node_dic=self.map_area.nodes[start_id]
        end_node_dic=self.map_area.nodes[end_id]

        start_node=Node(id_node=start_id,lat=start_node_dic['lat'],lon=start_node_dic['lon'])
        end_node=Node(id_node=end_id,lat=end_node_dic['lat'],lon=end_node_dic['lon'])

        distance=start_node.calculate_simple_distance(end_node)
        distance=distance/weight

        cur_distance=cur_queue['distance']
        queueItem = { \
            'distance': cur_distance + distance,
            'maxdistance': cur_distance + end_node.calculate_simple_distance(self.search_end),
            'nodes': cur_queue['nodes'] + "," + str(end_id),
            'end': end_id}

        count=0
        for test in self.queue:
            if test['maxdistance']> queueItem['maxdistance']:
                self.queue.insert(count,queueItem)
                break
            count+=1
        else:
            self.queue.append(queueItem)

        

"""
a=MapAreaCreator(db_location)
a.prepare_area_for(lat=48.8407481,lon=2.4024637,max_distance=40000)
router=Router(a.maparea)
#router.do_route(243896275,243896275,"cycle")
print "route",router.do_route(159666147,246650874,'cycle')
"""

class sayhooRouter(threading.Thread):

    def __init__(self,qq_to_router,qq_from_router):
        self.qq_from_controller=qq_to_router
        self.qq_to_controller=qq_from_router
        threading.Thread.__init__(self,name="main routeur thread")
        self.map_creator=False
        self.last_map_area=False
        self.destination_node_id=0
        self.destination_lat=0
        self.destination_lon=0
        self.start_point_node_id=0
        self.start_point_lat=0
        self.start_point_lon=0
        self.last_position=False
        self.transport_type="cycle"
        self.route_info=False

        self.gps_signal=False

        self.cur_lat=False
        self.cur_lon=False
        
        self.running=True
        self.start()

    def pull_request(self):
        out=[]
        while self.qq_from_controller.qsize():
            try:
                curEvent=self.qq_from_controller.get(0)
                #print "sayhooRouter curEvent=",curEvent
            except:
                raise
            out.append(curEvent)

        return out

    def new_start_point_set(self,req):
        lat=float(req.args['lat'])
        lon=float(req.args['lon'])
        self.start_point_lat=lat
        self.start_point_lon=lon
        db=sqlite3.connect(G.db_location)
        cursor=db.cursor()
        snt="select id_node,lat,lon from nodes order by (($LAT-lat)*($LAT-lat)+($LON-lon)*($LON-lon)) limit 100"
        cursor.execute(snt,(lat,lon))
        db_nodes=[]
        start_id_flag=False
        start_lat=0
        start_lon=0

        for line in cursor.fetchall():
            cur_id=line[0]

            if cur_id in self.map_creator.maparea.routing[self.transport_type]:
                start_id_flag=True
                self.start_point_node_id=cur_id
                self.start_point_lat=line[1]
                self.start_point_lon=line[2]
                break
        db.close()

        if not start_id_flag:
            print "No routable node near the starting point lat=%s,lon=%s"%(lat,lon)
            return 

        backToGui=Node(id_node=self.start_point_node_id,lat=self.start_point_lat,lon=self.start_point_lon)
        event=G.MapAnswer(args={"type":"new-start-node"},answer=backToGui)
        self.qq_to_controller.put(event)


        if self.destination_node_id:
            #print "In sayhoo router, we got a starting point. destination is known: got to make map"
            self.prepare_map_area()
        else:
            #print "We got a new starting point but there is no destination set: no map making"
            coincoin=0
        
        print "collecting prepare_map_area in new_start_point_set",gc.collect()
            


    def new_destination_set(self,req):
        db=sqlite3.connect(G.db_location)
        cursor=db.cursor()
        lat=req.args['lat']
        lon=req.args['lon']


        snt="select id_node,lat,lon from nodes order by (($LAT-lat)*($LAT-lat)+($LON-lon)*($LON-lon)) limit 100"
        cursor.execute(snt,(lat,lon))
        db_nodes=[]
        dest_id_flag=False
        for line in cursor.fetchall():
            cur_id=line[0]
            if cur_id in self.map_creator.maparea.routing[self.transport_type]:
                dest_id_flag=True
                self.destination_lat=line[1]
                self.destination_lon=line[2]
                self.destination_node_id=cur_id
                break
        db.close()
        if dest_id_flag:
            #print "New destination is node ID#%s"%self.destination_node_id
            #print "select * from ways_tags where id_way in (select id_way from ways_nodes where id_node=%s);"%self.destination_node_id

            backToGui=Node(id_node=self.destination_node_id,lat=self.destination_lat,lon=self.destination_lon)
            event=G.MapAnswer(args={"type":"new-destination-node"},answer=backToGui)
            self.qq_to_controller.put(event)
        else:
            print "No routeable destination for lat=%s,lon=%s"%(lat,lon)

        if self.start_point_node_id:
            #print "In sayhoo router, we got a destination. start point exist: got to make map"
            self.prepare_map_area()
        else:
            print "New destination is set, but no starting point"
        print "collecting prepare_map_area in new_destination_set",gc.collect()


    def calculate_next_segment_lenght(self,updated_info):
        cnt=0
        #print "calculate with ",updated_info
        #print "route_info['changes']=",self.route_info['changes']
        #print "route_info['nodes']=",self.route_info['nodes']

        cur_node_id=updated_info['next_change_node_id']
        #print "cur_node_id",cur_node_id
        cur_node_index=self.route_info['nodes'].index(cur_node_id)
        #print "cur_node_index",cur_node_index
        next_change_node_index_in_changes=self.route_info['changes'].index(cur_node_id)+1
        #print "next_change_node_index_in_changes",next_change_node_index_in_changes
        next_change_node_id=self.route_info['changes'][next_change_node_index_in_changes]

        next_node_index=cur_node_index
        while True:
            next_node_index+=1
            next_node_id=self.route_info["nodes"][next_node_index]
            cur_node_dic=self.last_map_area.nodes[cur_node_id]
            next_node_dic=self.last_map_area.nodes[next_node_id]
            cur_node=Node(id_node=cur_node_id,lat=cur_node_dic['lat'],lon=cur_node_dic['lon'])
            next_node=Node(id_node=next_node_id,lat=next_node_dic['lat'],lon=next_node_dic['lon'])
            cnt+=cur_node.calculate_metric_distance(next_node)
            if next_node_id==next_change_node_id:
                break
            cur_node_id=next_node_id
            cur_node_index=next_node_index
        print " "
        return cnt 
        
    def update_route_info(self,lat,lon):
        z=15

        curpos_node=Node(lat=lat,lon=lon)
        (curpos_x,curpos_y)=tn.latlon2xy(lat,lon,z)

        prev_node_id=self.route_info['changes'][0]
        prev_node=self.last_map_area.nodes[prev_node_id]
        (prev_x,prev_y)=tn.latlon2xy(prev_node['lat'],prev_node['lon'],z)
        min_dist=1000000000000000
        nearest_way=False

        next_change_node_id=False
        for cnt in range(1,len(self.route_info['changes'])+1):
            #print "cnt=",cnt
            if cnt==len(self.route_info['changes']):
                #print "yo"
                cur_node_id=self.route_info["nodes"][-1:][0]
            else:
                cur_node_id=self.route_info["changes"][cnt]
            cur_node=self.last_map_area.nodes[cur_node_id]
            (cur_x,cur_y)=tn.latlon2xy(cur_node['lat'],cur_node['lon'],z)
            dist=G.distancePointToLine(curpos_x,curpos_y,cur_x,cur_y,prev_x,prev_y)

            #print "prev=%s cur=%s dist=%s"%(prev_node_id,cur_node_id,dist)
            
            if dist<min_dist:
                min_dist=dist
                nearest_way=self.route_info['ways'][prev_node_id]
                next_change_node_id=cur_node_id
            
            prev_node_id=cur_node_id
            prev_node=cur_node
            prev_x=cur_x
            prev_y=cur_y

        out={}
        out["way"]=nearest_way
        out["next_change_node_id"]=next_change_node_id
        out["dist"]=min_dist
        return out

    def angle_to_oclock(self,angle,turn_type):
        #print "angle_to_oclock",angle,turn_type
        raw_clock=int(round(12*angle/360))
        #print "raw=",raw_clock
        if turn_type=="left":
            raw_clock=12-raw_clock
        return raw_clock

    def next_changes_info(self):
        """
        We need to know where will be the next change in
        road to follow the route.
        Information needed: distance before change
        ,angle of turn, is it left or right, current road
        next road

        a_node : The first point in route after the change 
        b_node : The last  point in route before the change
        c_node : The point in route where the change occurs.

        """
        updated_info=self.update_route_info(self.cur_lat,self.cur_lon)


        """
        print self.route_info["nodes"]
        for i in self.route_info['changes']:
            w=self.route_info["ways"][i].tags.get("name")
            print i,w
        
        print "next=",updated_info["next_change_node_id"]
        print updated_info["way"].tags.get("name",_("Unkown"))
        if updated_info["dist"]>0.001:
            print "Hmmmmm, where are you ?"        
        """



        #print "len=",len(self.route_info["nodes"])

        c_node_id=updated_info["next_change_node_id"]
        index_c=self.route_info["nodes"].index(c_node_id)
        b_node_id=self.route_info["nodes"][index_c-1]

        last_road_flag=False
        
        if index_c==len(self.route_info["nodes"])-1:
            #print "Last road,no more change needed"
            a_node_id=c_node_id
            last_road_flag=True
        else:
            a_node_id=self.route_info["nodes"][index_c+1]

        b_dic=self.last_map_area.nodes[b_node_id]
        c_dic=self.last_map_area.nodes[c_node_id]
        a_dic=self.last_map_area.nodes[a_node_id]
        b_node=Node(id_node=b_node_id,lat=b_dic["lat"],lon=b_dic["lon"])
        c_node=Node(id_node=c_node_id,lat=c_dic["lat"],lon=c_dic["lon"])
        a_node=Node(id_node=a_node_id,lat=a_dic["lat"],lon=a_dic["lon"])


        cur_node=Node(lat=self.cur_lat,lon=self.cur_lon)
        real_distance_to_change=cur_node.calculate_metric_distance(c_node)

        if not last_road_flag:
            #print "distance to change=",real_distance_to_change
            a_dist=b_node.calculate_metric_distance(c_node)
            b_dist=c_node.calculate_metric_distance(a_node)
            c_dist=a_node.calculate_metric_distance(b_node)
            #print "id=",b_node_id,c_node_id,a_node_id
            turn_angle=G.vertex_angle(a_dist,b_dist,c_dist)
            #before_to_after_angle=b_node.calculate_bearing(a_node)
            before_to_change_angle=b_node.calculate_bearing(c_node)
            #print "bearing to change",before_to_change_angle
            #print "bearing to after",before_to_after_angle
            """
            Now, if we rotate the triangle so that b to c is a vertical line
            (b being the center of the rotation).
            We can say if this is a turn to the left or the right just
            by looking if the after node is before left or right than the
            center of rotation.
            """

            rotation_angle=before_to_change_angle
            #print "rotate angle=",rotation_angle

            z=15
            (b_xy_x,b_xy_y)=tn.latlon2xy(b_node.lat,b_node.lon,z)
            (c_xy_x,c_xy_y)=tn.latlon2xy(c_node.lat,c_node.lon,z)
            (a_xy_a,a_xy_y)=tn.latlon2xy(a_node.lat,a_node.lon,z)

            (rcx,rcy)=G.rotate(b_xy_x,b_xy_y,c_xy_x,c_xy_y,rotation_angle)
            (rax,ray)=G.rotate(b_xy_x,b_xy_y,a_xy_a,a_xy_y,rotation_angle)
            #print "before:",b_xy_x,b_xy_y
            #print "new change",rcx,rcy
            #print "new after",rax,ray
            #print "rax-b_xy_x",rax-b_xy_x
            turn_type="None"
            if rax<b_xy_x:
                turn_type="left"
            else:
                turn_type="right"

            relative_angle=180-turn_angle



        cur_road=updated_info["way"].tags.get("name",_("Unknown"))
        if not last_road_flag:
            next_road=self.route_info["ways"][updated_info["next_change_node_id"]].tags.get("name",_("Unknown"))
            lenght_next_segment=self.calculate_next_segment_lenght(updated_info)
        else:
            next_road=_("None")
            turn_type=False
            relative_angle=0
            lenght_next_segment=0

        oclock_angle=self.angle_to_oclock(relative_angle,turn_type)
        #print "cur",cur_road
        #print "next",next_road

        on_the_route_flag=True
        if updated_info["dist"]>0.001:
            print "Hmmmmm, where are you ?"
            on_the_route_flag=False
        #print "distance to turn:",real_distance_to_change
        #print "turn type",turn_type
        #print "angle",relative_angle
        out={}
        out['current_road']=cur_road
        out['next_road']=next_road
        out['on_the_route_flag']=on_the_route_flag
        out['turn_type']=turn_type
        out['distance_to_change']=int(real_distance_to_change)
        out['clock_angle']=oclock_angle
        out['lenght_next_segment']=int(lenght_next_segment)
        out['last_road_flag']=last_road_flag
        return out
        
    def new_gps_event  (self,req):
        print "sayhooRouter new gps event recevied"
        self.cur_lat=req.args["lat"]
        self.cur_lon=req.args["lon"]
        if not self.route_info:
            return
        route_info=self.next_changes_info()
        #for i in route_info:
        #    print i,route_info[i]
        event=G.TtsEvent(args={'type':'route_info','info':route_info})
        self.qq_to_controller.put(event)        
            
    def prepare_map_area(self):
        self.route_info={}
        event=G.MapRequest(args={"type":"map-computing","computing":True})
        self.qq_to_controller.put(event)
        dest=Node(lat=self.destination_lat,lon=self.destination_lon)
        startp=Node(lat=self.start_point_lat,lon=self.start_point_lon)
        self.last_map_area=self.map_creator.maparea
        known_cur_node_id=self.start_point_node_id
        destination_node_id=self.last_map_area.find_nearest_node_for(dest.lat,dest.lon)
        router=Router(self.last_map_area)
        #c=self.last_map_area.nodes[known_cur_node_id]
        #d=self.last_map_area.nodes[destination_node_id]
        route=router.do_route(known_cur_node_id,destination_node_id,self.transport_type)
        if route[0]=="success":
            ids=route[1]
            out=[]
            for i in ids:
                a=self.last_map_area.nodes[i]
                out.append(a)
            event=G.MapAnswer(args={"type":"route-found"},answer=out)
            self.qq_to_controller.put(event)
        else:
            print "no route found:",route[0]
            G.system_info(_("No route found."))
            event=G.MapAnswer(args={"type":"route-found"},answer=[])
            self.qq_to_controller.put(event)
            return 

        #We now have a list of nodes from starting point to destination.
        #Lets find road names from each node to the next.
        self.route_info["nodes"]=route[1]
        self.route_info["ways"]={}
        self.route_info["changes"]=[]
        db=sqlite3.connect(G.db_location)
        cursor=db.cursor()
        snt="select id_way,indice from ways_nodes where id_node=?"
        cursor.execute(snt,[self.route_info['nodes'][0],])
        next_ways_lines=cursor.fetchall()
        next_ways={}
        for line in next_ways_lines:
            (id_way,indice)=line
            next_ways[id_way]=indice

                       
        for i in range(0,len(self.route_info["nodes"])-1):
            cur_node_id=self.route_info["nodes"][i]
            next_node_id=self.route_info["nodes"][i+1]
            #print self.last_map_area.routing[self.transport_type][cur_node_id]
            cur_ways=next_ways
            cursor.execute(snt,[next_node_id,])
            next_ways_lines=cursor.fetchall()
            next_ways={}
            for line in next_ways_lines:
                (id_way,indice)=line
                next_ways[id_way]=indice

            #print "cur_way=",cur_ways
            #print "next_ways=",next_ways
            #The way we are interested in is present in both cur_ways
            #and next_ways.
            possible_ways=[]
            for candidate in cur_ways.keys():
                if candidate in next_ways:
                    possible_ways.append(self.last_map_area.ways[candidate])

            if len(possible_ways)==0:
                G.system_info(_("No way found from node %s to %s"%(cur_node_id,next_node_id)))
                continue
            if len(possible_ways)>1:
                G.system_info(_("Several ways found from node %s to %s"%(cur_node_id,next_node_id)))
                for i in possible_ways:
                    print "______________"
                    way_name=possible_ways[0].tags.get("name",_("Unknown"))
                    print "-",way_name
                    print "--------------"

            #way_name=possible_ways[0].tags.get("name",_("Unknown"))
            self.route_info["ways"][cur_node_id]=possible_ways[0]

        prev_name=""
        for i in range(0,len(self.route_info["nodes"])-1):
            node_id=self.route_info["nodes"][i]
            way=self.route_info["ways"][node_id]
            cur_name=way.tags.get("name",_("Unknown"))
            if cur_name!=prev_name:
                self.route_info["changes"].append(node_id)
                prev_name=cur_name
        

        self.route_info['changes'].append(self.route_info['nodes'][-1:][0])
        #for nid in self.route_info["changes"]:
        #    w=self.route_info["ways"][nid].tags.get("name",_("Unknown"))
        #    print nid,w
        
        
        db.close()


    def load_city_data_dir(self,data_dir):
        import cPickle
        jar_dir=data_dir+"/pickle-jar"
        center=Node(lat=0,lon=0)
        maparea=MapArea(center_node=center,max_distance=0)

        G.system_info(_("Loading nodes"))
        pkl_file = open(jar_dir+"/nodes.bin", 'rb')
        cur_nodes=cPickle.load(pkl_file)
        pkl_file.close()

        G.system_info(_("Loading ways"))
        pkl_file = open(jar_dir+"/ways.bin", 'rb')
        maparea.ways=cPickle.load(pkl_file)
        pkl_file.close()

        G.system_info(_("Loading routing"))
        pkl_file = open(jar_dir+"/routing.bin", 'rb')
        maparea.routing=cPickle.load(pkl_file)
        pkl_file.close()

        G.system_info(_( "Loading routeablesNodes"))
        pkl_file = open(jar_dir+"/routeableNodes.bin", 'rb')
        maparea.routeableNodes=cPickle.load(pkl_file)
        pkl_file.close()

        maparea.nodes=cur_nodes    
        self.map_creator.maparea=maparea
        print "collecting after_load_city_data_dir",gc.collect()
        
        
    def run(self):
        import time

        print "bb",G.city_data_dir

        while not G.db_location and self.running:
            for req in self.pull_request():
                if isinstance(req,G.StopEvent):
                    self.running=False
                    print "Map says bye bye"
                if isinstance(req,G.MapRequest):
                    if req.args["type"]=="new_city_data_dir":
                        print "ok, lets have this data",req.args


                        lock = threading.Lock()
                        lock.acquire()
                        G.city_data_dir=req.args['data_dir']
                        G.db_location=G.city_data_dir+"/osm.db"
                        G.tiles_location=G.city_data_dir+"/tiles/%s/%s/%s.png"
                        G.pickle_jar=G.city_data_dir+"/pickle-jar"

                        (G.start_view_lat,G.start_view_lon)=G.find_center_point(G.db_location)

                        
                        lock.release()
                        
                        event=G.MapRequest(args={"type":"city_data_dir_ready"})
                        self.qq_to_controller.put(event)
            time.sleep(1)

        if not self.running:
            sys.exit()
            
        self.map_creator=MapAreaCreator(G.db_location)            




        before=time.time()
        if os.path.isdir(G.pickle_jar):
            self.load_city_data_dir(G.city_data_dir)
        else:
            G.system_info(_("pickled data not ready: %s!"%G.pickle_jar))
            sys.exit()

            
        after=time.time()

        spent=after-before
        print "it took %s"%spent

        event=G.MapRequest(args={"type":"map-computing","computing":False})
        self.qq_to_controller.put(event)
        
        event=G.TtsEvent(args={"type":"tts","lang":G.user_lang['application'],"snt":_("City is loaded.")})
        self.qq_to_controller.put(event)

        while self.running:
            for req in self.pull_request():
                if isinstance(req,G.MapRequest):

                    if req.args["type"]=="new_destination":
                        self.new_destination_set(req)
                    
                    if req.args['type']=='new_start_point':
                        self.new_start_point_set(req)

                    if req.args['type']=='clear_route':
                        self.destination_node_id=0
                        self.destination_lat=0
                        self.destination_lon=0
                        self.start_point_node_id=0
                        self.start_point_lat=0
                        self.start_point_lon=0
                        self.route_info=False

                if isinstance(req,G.GpsEvent):
                    self.new_gps_event(req)

                if isinstance(req,G.StopEvent):
                    self.running=False
                    print "Map says bye bye"
                
            time.sleep(.1)





        

