/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qstringlist.h>
#include <qlistbox.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qtimer.h>
#include <qapplication.h>

#include "playlistoverview.h"
#include "playlistmanager.h"
#include "playlistview.h"
#include "configuration.h"
#include "mediadatabase.h"
#include "skin.h"

#include "debug.h"

PlayListOverview::PlayListOverview(PlayListView *playListView, QWidget *parent, const char *name)
	: QHBox(parent, name),
	PlayListViewExtension(),
	playListView_(playListView),
	updatingGenres_(false),
	updatingArtists_(false),
	updatingAlbums_(false),
	selectedGenresFilter(""),
	selectedArtistsFilter(""),
	selectedAlbumsFilter("")
{

	createColumnBox("Genres", genreBox_, genreHeading_, genreListBox_);
	connect(genreListBox_, SIGNAL(selectionChanged()),
			this, SLOT(genreListBoxSelectionChanged()));
	genreListBox_->installEventFilter(this);

	// black line separator between list boxes...
	QWidget *dummywidget = new QWidget(this);
	dummywidget->setFixedWidth(1);
	dummywidget->setBackgroundColor(QColor("#000000"));

	createColumnBox("Artists", artistBox_, artistHeading_, artistListBox_);
	connect(artistListBox_, SIGNAL(selectionChanged()),
			this, SLOT(artistListBoxSelectionChanged()));
	artistListBox_->installEventFilter(this);

	// black line separator between list boxes...
	dummywidget = new QWidget(this);
	dummywidget->setFixedWidth(1);
	dummywidget->setBackgroundColor(QColor("#000000"));

	createColumnBox("Albums", albumBox_, albumHeading_, albumListBox_);
	connect(albumListBox_, SIGNAL(selectionChanged()),
			this, SLOT(albumListBoxSelectionChanged()));
	albumListBox_->installEventFilter(this);

	updateTimer_ = new QTimer();
	connect(updateTimer_, SIGNAL(timeout()), this, SLOT(updateView()));

	this->setFocusPolicy(QWidget::StrongFocus);
	this->setFocusProxy(genreListBox_);
}

PlayListOverview::~PlayListOverview()
{
	delete updateTimer_;
}

void PlayListOverview::createColumnBox(QString title, QVBox *&box, QLabel *&heading, QListBox *&listbox)
{
	box = new QVBox(this);
	heading = new QLabel(box);
	heading->setAlignment(Qt::AlignCenter | Qt::AlignVCenter | Qt::ExpandTabs);
	heading->setText(title);

	listbox = new QListBox(box);
	listbox->setFrameStyle(QFrame::NoFrame);
	listbox->setSelectionMode(QListBox::Multi);
	listbox->setHScrollBarMode(QScrollView::AlwaysOff);
}

void PlayListOverview::setSkinMode(const SkinModeInformation &modeInfo)
{
	if (modeInfo.playListOverviewFont().font.isUsed())
	{
		genreListBox_->setFont(modeInfo.playListOverviewFont().font);
		artistListBox_->setFont(modeInfo.playListOverviewFont().font);
		albumListBox_->setFont(modeInfo.playListOverviewFont().font);
	}
	else
	{
		genreListBox_->unsetFont();
		artistListBox_->unsetFont();
		albumListBox_->unsetFont();
	}

	// set general palette:
	QPalette palette = QApplication::palette();

	if (modeInfo.playListOverviewFont().fontColor.isUsed())
	{
		palette.setColor(QColorGroup::Text, modeInfo.playListOverviewFont().fontColor);
		palette.setColor(QColorGroup::Foreground, modeInfo.playListOverviewFont().fontColor);
	}

	if (modeInfo.playListOverviewBackgroundColorSelected().isUsed())
	{
		palette.setColor(QColorGroup::Highlight, modeInfo.playListOverviewBackgroundColorSelected());
		palette.setColor(QColorGroup::HighlightedText, modeInfo.playListOverviewFontColorSelected());
	}

	// set even and odd colors for items:
	if (modeInfo.playListOverviewBackgroundColor().isUsed())
		palette.setColor(QColorGroup::Base, modeInfo.playListOverviewBackgroundColor());

	genreListBox_->setPalette(palette);
	artistListBox_->setPalette(palette);
	albumListBox_->setPalette(palette);

	if (modeInfo.playListOverviewHeaderFont().font.isUsed())
	{
		genreHeading_->setFont(modeInfo.playListOverviewHeaderFont().font);
		artistHeading_->setFont(modeInfo.playListOverviewHeaderFont().font);
		albumHeading_->setFont(modeInfo.playListOverviewHeaderFont().font);
	}
	else
	{
		genreHeading_->unsetFont();
		artistHeading_->unsetFont();
		albumHeading_->unsetFont();
	}

	if (modeInfo.playListOverviewHeaderColor().isUsed())
	{
		QPalette palette(modeInfo.playListOverviewHeaderColor());

		if (modeInfo.playListOverviewHeaderFont().fontColor.isUsed())
		{
			palette.setColor(QColorGroup::Text, modeInfo.playListOverviewHeaderFont().fontColor);
			palette.setColor(QColorGroup::ButtonText, modeInfo.playListOverviewHeaderFont().fontColor);
			palette.setColor(QColorGroup::Foreground, modeInfo.playListOverviewHeaderFont().fontColor);
		}

		genreHeading_->setPalette(palette);
		artistHeading_->setPalette(palette);
		albumHeading_->setPalette(palette);
	}
	else
	{
		genreHeading_->unsetPalette();
		artistHeading_->unsetPalette();
		albumHeading_->unsetPalette();
	}

	if (modeInfo.playListOverviewScrollBarColor().isUsed())
	{
		genreListBox_->horizontalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		genreListBox_->verticalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		artistListBox_->horizontalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		artistListBox_->verticalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		albumListBox_->horizontalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		albumListBox_->verticalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
	}
	else
	{
		genreListBox_->horizontalScrollBar()->unsetPalette();
		genreListBox_->verticalScrollBar()->unsetPalette();
		artistListBox_->horizontalScrollBar()->unsetPalette();
		artistListBox_->verticalScrollBar()->unsetPalette();
		albumListBox_->horizontalScrollBar()->unsetPalette();
		albumListBox_->verticalScrollBar()->unsetPalette();
	}
}

void PlayListOverview::setSelectionMode(QListBox::SelectionMode mode)
{
	genreListBox_->setSelectionMode(mode);
	artistListBox_->setSelectionMode(mode);
	albumListBox_->setSelectionMode(mode);
}

void PlayListOverview::showEvent(QShowEvent *e)
{
	if (!e->spontaneous() && isDirty_)
		execute();
}

bool PlayListOverview::eventFilter(QObject *object, QEvent *event)
{
	if (event->type() == QEvent::KeyPress)
	{
		QKeyEvent *e = static_cast<QKeyEvent *>(event);
		int key = e->key();

		QWidget *newFocusedWidget = NULL;

		if (object == genreListBox_)
		{
			if (key == Qt::Key_Right)
				newFocusedWidget = artistListBox_;
			else if (key == Qt::Key_Left)
				emit focusOutLeft();
		}
		else if (object == artistListBox_)
		{
			if (key == Qt::Key_Right)
				newFocusedWidget = albumListBox_;
			else if (key == Qt::Key_Left)
				newFocusedWidget = genreListBox_;
		}
		else if (object == albumListBox_)
		{
			if (key == Qt::Key_Right)
				emit focusOutRight();
			else if (key == Qt::Key_Left)
				newFocusedWidget = artistListBox_;
		}

		if (newFocusedWidget)
		{
			newFocusedWidget->setFocus();
			e->accept();
			return true;
		}

		return QHBox::eventFilter(object, event);
	}

	if (event->type() == QEvent::FocusIn)
		setFocusProxy(static_cast<QWidget *>(object));

	return QHBox::eventFilter(object, event);
}

void PlayListOverview::saveExtensionState(PlayListViewState *dst)
{
	QStringList *state = new QStringList();

	state->append("genre:" + buildFilter(genreListBox_, tr("(Undefined genre)")));
	state->append("artist:" + buildFilter(artistListBox_, tr("(Unknown artist)")));
	state->append("album:" + buildFilter(albumListBox_, tr("(Unknown album)")));

	QString keyName = inputViewSource() + "_overview";
	dst->extensions.remove(keyName);
	dst->extensions.insert(keyName, state);
}

void PlayListOverview::loadExtensionState(PlayListViewState *src)
{
	QString keyName = inputViewSource() + "_overview";

	QStringList *state = src->extensions.find(keyName);

	if (state && state->count() == 3)
	{
		selectedGenresFilter = (*state)[0].mid((*state)[0].find(":") + 1);
		selectedArtistsFilter = (*state)[1].mid((*state)[1].find(":") + 1);
		selectedAlbumsFilter = (*state)[2].mid((*state)[2].find(":") + 1);

		isDirty_ = true;
		execute();
	}
}

QString PlayListOverview::buildFilter(QListBox *list, const QString &unknown)
{
	QString filter = "";
	int selectedCount = 0;

	if (list->count() > 1 && !list->isSelected(0))
		for (int i = 1; i < list->count(); ++i)
			if (list->isSelected(i))
			{
				++selectedCount;
				if (list->item(i)->text() == unknown)
					filter += (selectedCount > 1 ? QString(", ") : QString("")) + "NULL, \"\"";
				else
					filter += (selectedCount > 1 ? QString(", ") : QString("")) + "\"" + dbSafeString(list->item(i)->text()) + "\"";
			}

	return filter;
}

int PlayListOverview::selectionCount(QListBox *list)
{
	int selectedCount = 0;

	for (int i = 1; i < list->count(); ++i)
		if (list->isSelected(i))
			++selectedCount;

	return selectedCount;
}

void PlayListOverview::execute()
{
	DENTERMETHOD("PlayListOverview::execute()");

	// if the overview widget is invisible and no genre, artist or album filter is set, we can just set
	// it into dirty state and thus speed up processing. Once the overview widget is shown again, we check for isDirty
	// in the showEvent method and update the lists' content accordingly...
	if (!isVisible() && genreListBox_->isSelected(0) && artistListBox_->isSelected(0) && albumListBox_->isSelected(0))
	{
		isDirty_ = true;
		PlayListViewExtension::execute();
	}
	else
	{
		isDirty_ = false;

		sqlite3_exec(playListView_->mediaDatabase()->db(),
			"DROP TABLE IF EXISTS overview;"
			"CREATE TEMPORARY TABLE overview AS "
				"SELECT DISTINCT artist, album, genre FROM " + inputViewSource() + ";", NULL, NULL, NULL);

		updateGenres(false);
	}

	DEXITMETHOD("PlayListOverview::execute()");
}

void PlayListOverview::createFilteredTable(const QString &selectedFilter, const QString &srctable, const QString &dsttable, const QString &what, const QString &wherecol)
{
	QString query =
		"DROP TABLE IF EXISTS " + dsttable + ";"
		"CREATE TEMPORARY TABLE " + dsttable + " AS "
			"SELECT DISTINCT " + what + " FROM " + srctable +
			(selectedFilter.length() > 0 ? " WHERE " + wherecol + " IN (" + selectedFilter + ");" : QString(";"));

	DPRINTF("query : %s", (const char*)query.utf8());
	sqlite3_exec(playListView_->mediaDatabase()->db(), query.utf8(), NULL, NULL, NULL);
}

void PlayListOverview::beginUpdateOn(QWidget *widget)
{
	widget->setUpdatesEnabled(false);
	widget->blockSignals(true);
}

void PlayListOverview::endUpdateOn(QWidget *widget)
{
	widget->blockSignals(false);
	widget->setUpdatesEnabled(true);
}

void PlayListOverview::doUpdateOnListBox(QListBox *listbox, const QString &selectedFilter, const QString &colname, const QString &srctable, const QString &unknown, const QString &one, const QString &more)
{
	//	NOTE: This query might yield incorrect results if the optimization settings at compile time are too low. Probably a bug in SQLite...
	QString query = "SELECT DISTINCT " + colname + ", " + (selectedFilter.length() > 0 ? colname + " IN (" + selectedFilter + ")" : QString("0")) +
					" AS selected FROM " + srctable + " ORDER BY " + colname + " ASC;";

	DPRINTF("query : %s", (const char*)query.utf8());

	beginUpdateOn(listbox);

	listbox->clearSelection();
	listbox->clear();

	sqlite3_stmt *vm;
	sqlite3_prepare_v2(playListView_->mediaDatabase()->db(), query.utf8(), -1, &vm, 0);

	int selcount = 0;
	while (sqlite3_step(vm) == SQLITE_ROW)
	{
		listbox->insertItem(QString::fromUtf8((const char *)sqlite3_column_text(vm, 0)));

		if (sqlite3_column_int(vm, 1))
		{
			listbox->setSelected(listbox->count() - 1, true);
			++selcount;
		}
		else
			listbox->setSelected(listbox->count() - 1, false);
	}
	sqlite3_finalize(vm);

	if (listbox->count() != 1)
		listbox->insertItem(more.arg(listbox->count()), 0);
	else
		listbox->insertItem(one, 0);

	if (listbox->count() > 1 && (listbox->text(1).isNull() || listbox->text(1).isEmpty()))
		listbox->changeItem(unknown, 1);

	if (selcount == 0)
		listbox->setSelected(0, true);
}

void PlayListOverview::updateGenres(bool interactive)
{
	DENTERMETHOD("PlayListOverview::updateGenres()");

	DTIMERINIT(time);

	if (selectedGenresFilter.isEmpty())
		selectedGenresFilter = buildFilter(genreListBox_, tr("(Undefined genre)"));

	doUpdateOnListBox(genreListBox_, selectedGenresFilter, "genre", "overview", tr("(Undefined genre)"), tr("All (1 genre)"), tr("All (%1 genres)"));

	DTIMERPRINT(time, "updateGenres");

	updateArtists(interactive);

	endUpdateOn(genreListBox_);

	DEXITMETHOD("PlayListOverview::updateGenres()");
}

void PlayListOverview::updateArtists(bool interactive)
{
	DENTERMETHOD("PlayListOverview::updateArtists()");

	DTIMERINIT(time);

	selectedGenresFilter = buildFilter(genreListBox_, tr("(Undefined genre)"));
	createFilteredTable(selectedGenresFilter, "overview", "overview_genre_filtered", "artist, album", "genre");

	if (selectedArtistsFilter.isEmpty())
		selectedArtistsFilter = buildFilter(artistListBox_, tr("(Unknown artist)"));

	doUpdateOnListBox(artistListBox_, selectedArtistsFilter, "artist", "overview_genre_filtered", tr("(Unknown artist)"), tr("All (1 artist)"), tr("All (%1 artists)"));

	DTIMERPRINT(time, "updateArtists");

	updateAlbums(interactive);

	endUpdateOn(artistListBox_);

	DEXITMETHOD("PlayListOverview::updateArtists()");
}

void PlayListOverview::updateAlbums(bool interactive)
{
	DENTERMETHOD("PlayListOverview::updateAlbums()");

	DTIMERINIT(time);

	selectedArtistsFilter = buildFilter(artistListBox_, tr("(Unknown artist)"));
	createFilteredTable(selectedArtistsFilter, "overview_genre_filtered", "overview_artist_filtered", "album", "artist");

	if (selectedAlbumsFilter.isEmpty())
		selectedAlbumsFilter = buildFilter(albumListBox_, tr("(Unknown album)"));

	doUpdateOnListBox(albumListBox_, selectedAlbumsFilter, "album", "overview_artist_filtered", tr("(Unknown album)"), tr("All (1 album)"), tr("All (%1 albums)"));

	DTIMERPRINT(time, "updateAlbums");

	// if we're returning into the event loop because this should be interactive,
	// we need to finalize and re-prepare statements NOW, since the schema changed
	// and something may access those prepared statements in the meantime before updateView is
	// triggered by the timer...
	if (interactive)
		playListView_->mediaDatabase()->ensurePreparedStatementsAreValid();

	// if we're entering updateView directly (i.e. non-interactively), the prepared statements will be taken care of there...
	updateView(interactive);

	endUpdateOn(albumListBox_);

	DEXITMETHOD("PlayListOverview::updateAlbums()");
}

void PlayListOverview::updateView(bool interactive)
{
	if (interactive)
	{
		if (!updateTimer_->isActive())
			updateTimer_->stop();

		updateTimer_->start(qConfig.playlistAutoTriggerFilterInterval, TRUE);
	}
	else
		updateView();
}

void PlayListOverview::updateView()
{
	DENTERMETHOD("PlayListOverview::updateView()");

	selectedAlbumsFilter = buildFilter(albumListBox_, tr("(Unknown album)"));

	if (selectedGenresFilter.length() > 0 || selectedArtistsFilter.length() > 0 || selectedAlbumsFilter.length() > 0)
	{
		QString query =
			"DROP TABLE IF EXISTS playlist_view_overview_filtered;"
			"CREATE TEMPORARY TABLE playlist_view_overview_filtered AS "
				"SELECT * FROM " + inputViewSource() + " WHERE ";

		if (selectedGenresFilter.length() > 0)
		{
			query += "genre IN (" + selectedGenresFilter + ")";
			if (selectedArtistsFilter.length() > 0 || selectedAlbumsFilter.length() > 0)
				query += " AND ";
		}

		if (selectedArtistsFilter.length() > 0)
		{
			query += "artist IN (" + selectedArtistsFilter + ")";
			if (selectedAlbumsFilter.length() > 0)
				query += " AND ";
		}

		if (selectedAlbumsFilter.length() > 0)
			query += "album IN (" + selectedAlbumsFilter + ")";

		query += ";";

		DTIMERINIT(time);
		sqlite3_exec(playListView_->mediaDatabase()->db(), query.utf8(), NULL, NULL, NULL);
		// the schema changed, so we need to finalize and re-prepare statements...
		playListView_->mediaDatabase()->ensurePreparedStatementsAreValid();
		DTIMERPRINT(time, (const char*)query.utf8());
	}
	else // no overview filtering...
	{
		QString query = "DROP TABLE IF EXISTS playlist_view_overview_filtered;";
		DTIMERINIT(time);
		sqlite3_exec(playListView_->mediaDatabase()->db(), query.utf8(), NULL, NULL, NULL);
		// the schema changed, so we need to finalize and re-prepare statements...
		playListView_->mediaDatabase()->ensurePreparedStatementsAreValid();
		DTIMERPRINT(time, (const char*)query.utf8());
	}

	PlayListViewExtension::execute();

	DEXITMETHOD("PlayListOverview::updateView()");
}

QString PlayListOverview::outputViewSource()
{
	if (selectedGenresFilter.length() > 0 || selectedArtistsFilter.length() > 0 || selectedAlbumsFilter.length() > 0)
		return "playlist_view_overview_filtered";
	else
		return inputViewSource();
}

void PlayListOverview::validateSelection(QListBox *list)
{
	if (list->currentItem() == 0 && list->isSelected(0) || selectionCount(list) == 0)
	{
		list->clearSelection();
		list->setSelected(0, TRUE);
		list->setCurrentItem(0);
	}
	else
		list->setSelected(0, FALSE);
}

void PlayListOverview::genreListBoxSelectionChanged()
{
	DENTERMETHOD("PlayListOverview::genreListBoxSelectionChanged()");
	if (updatingGenres_)
	{
		DEXITMETHOD("PlayListOverview::genreListBoxSelectionChanged()");
		return;
	}
	updatingGenres_ = true;
	validateSelection(genreListBox_);
	updateArtists(true);
	updatingGenres_ = false;
	DEXITMETHOD("PlayListOverview::genreListBoxSelectionChanged()");
}

void PlayListOverview::artistListBoxSelectionChanged()
{
	DENTERMETHOD("PlayListOverview::artistListBoxSelectionChanged()");
	if (updatingArtists_)
	{
		DEXITMETHOD("PlayListOverview::artistListBoxSelectionChanged()");
		return;
	}
	updatingArtists_ = true;
	validateSelection(artistListBox_);
	updateAlbums(true);
	updatingArtists_ = false;
	DEXITMETHOD("PlayListOverview::artistListBoxSelectionChanged()");
}

void PlayListOverview::albumListBoxSelectionChanged()
{
	DENTERMETHOD("PlayListOverview::albumListBoxSelectionChanged()");
	if (updatingAlbums_)
	{
		DEXITMETHOD("PlayListOverview::albumListBoxSelectionChanged()");
		return;
	}
	updatingAlbums_ = true;
	validateSelection(albumListBox_);
	updateView(true);
	updatingAlbums_ = false;
	DEXITMETHOD("PlayListOverview::albumListBoxSelectionChanged()");
}

void PlayListOverview::reset()
{
	albumListBox_->selectAll(false);
	artistListBox_->selectAll(false);
	genreListBox_->selectAll(false);
}
