/*
 * Copyright (C) 2007 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <qthread.h>

#ifdef WINDOWS
#include "wincompat.h"
#endif

#include "backgroundtasks.h"
#include "debug.h"


class TaskProcessingThread : public QThread
{
public:
	TaskProcessingThread(TaskProcessingController *owner);
	virtual ~TaskProcessingThread();
	
	virtual void run();
	
	bool sleeping() { return sleeping_; }
	
	void terminate() { terminated_ = true; }
	Task *currentTask() { return currentTask_; }
	
protected:
	TaskProcessingController *owner_;
	Task *currentTask_;
	bool terminated_;
	bool sleeping_;
};


class TaskFinishedEvent : public QEvent
{
public:
	TaskFinishedEvent(Task *task) : QEvent(QEvent::User), task_(task) { }
	Task* task() { return task_; }
private:
	Task *task_;
};


/* Task */

Task::Task()
	:	aborted_(false),
		finished_(false)
{
	
}

Task::~Task()
{
}


/* TaskProcessingThread */

TaskProcessingThread::TaskProcessingThread(TaskProcessingController *owner)
	:	QThread(),
		owner_(owner),
		currentTask_(NULL),
		terminated_(false),
		sleeping_(false)
{
	
}

TaskProcessingThread::~TaskProcessingThread()
{

}

void TaskProcessingThread::run()
{
	while (!terminated_)
	{
		currentTask_ = owner_->getNextTask();
		
		if (currentTask_)
			DPRINTF("thread: %d -> task: %d", (int)currentThread(), (int)currentTask_);
		
		if (currentTask_)
		{
			if (!currentTask_->aborted())
			{
				currentTask_->run();
				
				if (!currentTask_->aborted() && !terminated_)
				{
					currentTask_->finished_ = true;
					postEvent(owner_, new TaskFinishedEvent(currentTask_));
					currentTask_ = NULL;
				}
				else
				{
					currentTask_->finished_ = true;
					currentTask_->notifyTaskFinished();
					delete currentTask_;
					currentTask_ = NULL;
				}
			}
			else
			{
				currentTask_->finished_ = true;
				currentTask_->notifyTaskFinished();
				delete currentTask_;
				currentTask_ = NULL;
			}
			
			usleep(1);
		}
		else
		{
			DPRINTF("thread: %d -> nothing to be done, Zzz...", (int)currentThread());
			sleeping_ = true;
#ifdef WINDOWS			
			usleep(100000);
#endif			
			owner_->conditionWorkAvailable_.wait();
			sleeping_ = false;
			DPRINTF("thread: %d -> nugded, woke up...", (int)currentThread());
		}
	}
	DPRINTF("thread %d died.", (int)currentThread());
}


/* TaskProcessingController */

TaskProcessingController::TaskProcessingController(QObject *parent, const char *name)
	:	QObject(parent, name),
		mutexTaskQueue_(true),
		mutexLowPrioTaskQueue_(true),
		conditionWorkAvailable_(),
		threads_(),
		taskQueue_(),
		lowPrioTaskQueue_()
{
	threads_.setAutoDelete(true);
	taskQueue_.setAutoDelete(true);
	lowPrioTaskQueue_.setAutoDelete(true);
	
	// only two threads for now...
	TaskProcessingThread *thread = new TaskProcessingThread(this); 
	threads_.append(thread);
	thread->start();

#ifndef QTOPIA
	thread = new TaskProcessingThread(this);
	threads_.append(thread);
	thread->start();
#endif
}

TaskProcessingController::~TaskProcessingController()
{
	DENTERMETHOD("TaskProcessingController::~TaskProcessingController()");
	
	TaskProcessingThread *thread;
	
	// Signal termination...
	for (thread = threads_.first(); thread != 0; thread = threads_.next())
		thread->terminate();
	
	// Clear the queues...
	clear();
	
	// Signal sleeping threads to wake up and discover they are marked for termination...
	for (thread = threads_.first(); thread != 0; thread = threads_.next())
		while (thread->running())
		{
			conditionWorkAvailable_.wakeAll();
			usleep(1);
		}
	
	// One last sanity waitfor...
	for (thread = threads_.first(); thread != 0; thread = threads_.next())
		thread->wait();
	
	// Now go die...
	threads_.clear();
	
	DEXITMETHOD("TaskProcessingController::~TaskProcessingController()");
}

void TaskProcessingController::clear()
{
	lockQueues();	
	
	for (Task *task = taskQueue_.last(); task != NULL; task = taskQueue_.prev())
	{
		task->notifyTaskFinished();
		taskQueue_.remove(); // autoDelete is enabled, so also free the task...
	}

	for (Task *task = lowPrioTaskQueue_.last(); task != NULL; task = lowPrioTaskQueue_.prev())
	{
		task->notifyTaskFinished();
		lowPrioTaskQueue_.remove(); // autoDelete is enabled, so also free the task...
	}
	
	unlockQueues();
}

void TaskProcessingController::addTask(Task *task, TaskPriority prio, bool prepend)
{
	DPRINTF("Adding task %d", (int)task);
	
	if (prio == TaskPriorityHigh)
	{
		mutexTaskQueue_.lock();
		if (prepend)
			taskQueue_.insert(0, task);
		else
			taskQueue_.append(task);
		mutexTaskQueue_.unlock();
	}
	else
	{
		mutexLowPrioTaskQueue_.lock();
		if (prepend)
			lowPrioTaskQueue_.insert(0, task);
		else
			lowPrioTaskQueue_.append(task);
		mutexLowPrioTaskQueue_.unlock();
	}
	
	conditionWorkAvailable_.wakeOne();
}

bool TaskProcessingController::removeTask(Task *task)
{
	lockQueues();
	
	DPRINTF("Removing task %d", (int)task);
	
	bool result = taskQueue_.removeRef(task);
	
	if (!result)
		result = lowPrioTaskQueue_.removeRef(task);

	unlockQueues();	
	
	return result;
}

bool TaskProcessingController::taskEnqueued(Task *task)
{
	lockQueues();
	
	bool result = taskQueue_.containsRef(task);

	if (!result)
		result = lowPrioTaskQueue_.containsRef(task);

	unlockQueues();	
	
	return result;
}

bool TaskProcessingController::taskRunning(Task *task)
{
	lockQueues();
	
	bool result = false;
	
	for (TaskProcessingThread *thread = threads_.first(); thread != 0; thread = threads_.next())
		if (thread->currentTask() == task)
		{
			result = true;
			break;
		}
	
	unlockQueues();
	
	return result;
}

bool TaskProcessingController::rescheduleTask(Task *task, TaskPriority prio, bool prepend)
{
	lockQueues();

	bool result = false;

	if (taskEnqueued(task) && !taskRunning(task))
	{
		if (lowPrioTaskQueue_.findRef(task) > -1)
			lowPrioTaskQueue_.take();
		
		if (taskQueue_.findRef(task) > -1)
			taskQueue_.take();

		if (prio == TaskPriorityHigh)
		{
			if (prepend)
				taskQueue_.insert(0, task);
			else
				taskQueue_.append(task);
		}
		else if (prio == TaskPriorityLow)
		{
			if (prepend)
				lowPrioTaskQueue_.insert(0, task);
			else
				lowPrioTaskQueue_.append(task);
		}
		
		result = true;
	}
	
	unlockQueues();
	
	return result;
}

bool TaskProcessingController::idle()
{
	lockQueues();
	bool queuesEmpty = taskQueue_.count() == 0 && lowPrioTaskQueue_.count() == 0;
	bool threadsSleeping = false;
	
	if (queuesEmpty)
	{
		threadsSleeping = true;
		for (TaskProcessingThread *thread = threads_.first(); thread != 0; thread = threads_.next())
			if (thread->currentTask())
			{
				threadsSleeping = false;
				break;
			}
	}

	unlockQueues();
	
	DPRINTF("queuesEmpty: %d, threadsSleeping: %d", queuesEmpty == true ? 1 : 0, threadsSleeping == true ? 1 : 0);

	return queuesEmpty && threadsSleeping;
}

void TaskProcessingController::waitUntilFinished()
{
	while (!idle())
		usleep(10000); // sleep 10 ms
}

void TaskProcessingController::lockQueues()
{
	mutexTaskQueue_.lock();
	mutexLowPrioTaskQueue_.lock();	
}

void TaskProcessingController::unlockQueues()
{
	mutexLowPrioTaskQueue_.unlock();
	mutexTaskQueue_.unlock();
}

Task *TaskProcessingController::getNextTask()
{
	DPRINTF("Getting next task...");
	
	Task *result = NULL;
	
	mutexTaskQueue_.lock();
	
	if (!taskQueue_.isEmpty())
	{
		Task *currentTask = taskQueue_.first();
		
		while (!taskQueue_.isEmpty() && currentTask && currentTask->aborted())
		{
			currentTask->notifyTaskFinished();
			taskQueue_.remove(); // autoDelete is enabled, so also free the task...
		}
		
		if (!taskQueue_.isEmpty())
			result = taskQueue_.take();
	}
	
	mutexTaskQueue_.unlock();
	
	if (!result)
	{
		mutexLowPrioTaskQueue_.lock();
		
		if (!lowPrioTaskQueue_.isEmpty())
		{
			Task *currentTask = lowPrioTaskQueue_.first();
			
			while (!lowPrioTaskQueue_.isEmpty() && currentTask && currentTask->aborted())
			{
				currentTask->notifyTaskFinished();
				lowPrioTaskQueue_.remove(); // autoDelete is enabled, so also free the task...
			}
			
			if (!lowPrioTaskQueue_.isEmpty())
				result = lowPrioTaskQueue_.take();
		}
		
		mutexLowPrioTaskQueue_.unlock();
	}
	
	DPRINTF("Next task: %d", (int)result);
	
	return result;
}

bool TaskProcessingController::event(QEvent *e)
{
	DPRINTF("TaskProcessingController::event(QEvent *e)");
	
	if (e->type() == QEvent::User)
	{
		TaskFinishedEvent *event = static_cast<TaskFinishedEvent *>(e);
		DPRINTF("Task %d finished.", (int)(event->task()));
		event->task()->notifyTaskFinished();
		delete event->task();
		return true;
	}
	
	return false;
}
