#include "scrollbar.h"

/* AbstractScrollBar */

void AbstractScrollBar::resized()
{
	int scrollBarButtonWidth = buttonSize();
	rectButtonLeft_ = QRect(geometry().left(), geometry().top(), scrollBarButtonWidth, geometry().height());
	rectButtonRight_ = QRect(geometry().right() - scrollBarButtonWidth + 1, geometry().top(), scrollBarButtonWidth, geometry().height());
	PositionedLayer::resized();	
}

void AbstractScrollBar::paintToBuffer(QImage &buffer, const QRect &clipRect)
{
	PositionedLayer::paintToBuffer(buffer, clipRect);

	if (itemCount() == 0 || itemsPerPage() == 0) return;
	
	int innerScrollBarWidth = geometry().width() - 2 * buttonSize();

	int maxScrollBarSliderWidth = innerScrollBarWidth / 2;
	int minScrollBarSliderWidth = minSliderWidth();
	
	int scrollBarSliderWidth = 
		QMIN(maxScrollBarSliderWidth, QMAX(minScrollBarSliderWidth, 
			((float)itemsPerPage() / itemCount()) * innerScrollBarWidth));
	
	int scrollBarSliderPosition = (displayPosition() * (innerScrollBarWidth - scrollBarSliderWidth)) / (itemCount() - 1);
	
	rectSlider_ = QRect(rectButtonLeft_.right() + scrollBarSliderPosition + 1, geometry().top(), scrollBarSliderWidth, geometry().height());
	rectSliderAreaLeft_ = QRect(rectButtonLeft_.topRight(), rectSlider_.bottomLeft());
	rectSliderAreaRight_ = QRect(rectSlider_.topRight(), rectButtonRight_.bottomLeft());
}

void AbstractScrollBar::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	PositionedLayer::paintOverlay(p, pixmap);

	p.setPen(QColor(255, 255, 255));
	p.drawRect(geometry());
	
	p.setBrush(QColor(128, 128, 128));
	p.drawRect(rectButtonLeft_);
	p.drawRect(rectButtonRight_);

	if (itemCount() == 0 || itemsPerPage() == 0) return;
	
	p.setBrush(QColor(50, 50, 50));
	p.drawRect(rectSliderAreaLeft_);
	p.drawRect(rectSliderAreaRight_);
	
	p.setBrush(QColor(128, 128, 128));
	p.drawRect(rectSlider_);
}

void AbstractScrollBar::mousePressEvent(QMouseEvent *event)
{
	mouseDown_ = true;
	
	if (rectButtonLeft_.contains(event->pos()))
	{
		scrollTimer_ = new QTimer();
		connect(scrollTimer_, SIGNAL(timeout()), this, SLOT(scrollToPreviousItem()));
		scrollTimer_->start(100);
		scrollToPreviousItem();
	}
	else if (rectButtonRight_.contains(event->pos()))
	{
		scrollTimer_ = new QTimer();
		connect(scrollTimer_, SIGNAL(timeout()), this, SLOT(scrollToNextItem()));
		scrollTimer_->start(100);
		scrollToNextItem();
	}
	else if (rectSlider_.contains(event->pos()))
	{
		sliderPressedOffset_ = event->x() - rectSlider_.left() - 1;
		setCurrentValueFromScrollBarPosition(event->x());
	}
	else if (rectSliderAreaLeft_.contains(event->pos()))
	{
		scrollTimer_ = new QTimer();
		connect(scrollTimer_, SIGNAL(timeout()), this, SLOT(scrollToPreviousPage()));
		scrollTimer_->start(100);
		scrollToPreviousPage();
	}
	else if (rectSliderAreaRight_.contains(event->pos()))
	{
		scrollTimer_ = new QTimer();
		connect(scrollTimer_, SIGNAL(timeout()), this, SLOT(scrollToNextPage()));
		scrollTimer_->start(100);
		scrollToNextPage();
	}
}

void AbstractScrollBar::mouseMoveEvent(QMouseEvent *event)
{
	if (mouseDown_)
	{
		if (sliderPressedOffset_ > -1)
		{
			setCurrentValueFromScrollBarPosition(event->x());
		}
	}
}

void AbstractScrollBar::setCurrentValueFromScrollBarPosition(int x)
{
	int innerScrollBarWidth = geometry().width() - 2 * buttonSize() - rectSlider_.width();
	x = x - sliderPressedOffset_ - geometry().left() - 20;
	int newCurrentValue = ((float)x / innerScrollBarWidth) * itemCount();
	changePosition(newCurrentValue);
}

void AbstractScrollBar::mouseReleaseEvent(QMouseEvent *event)
{
	if (mouseDown_)
	{
		mouseDown_ = false;
		sliderPressedOffset_ = -1;
		
		if (scrollTimer_)
		{
			delete scrollTimer_;
			scrollTimer_ = NULL;
		}
	}
}

void AbstractScrollBar::scrollToNextItem()
{
	if (!changePosition(position() + 1))
		scrollTimer_->stop();
}

void AbstractScrollBar::scrollToPreviousItem()
{
	if (!changePosition(position() - 1))
		scrollTimer_->stop();
}

void AbstractScrollBar::scrollToNextPage()
{
	if (!changePosition(position() + itemsPerPage()))
		scrollTimer_->stop();
}

void AbstractScrollBar::scrollToPreviousPage()
{
	if (!changePosition(position() - itemsPerPage()))
		scrollTimer_->stop();
}


/* ScrollBar */

bool ScrollBar::changePosition(float position)
{
	position = QMIN(itemCount() - 1, QMAX(position, 0));
	if (position != position_)
	{
		float oldPosition = position_;
		emit userChangedPosition(position, position_);
		if (oldPosition == position_)
			position_ = position;
		
		invalidate();
		
		return true;
	}
	else
		return false;
}


/* FlowScrollBar */

FlowScrollBar::FlowScrollBar(CustomFlow *flow, LayeredPaintBox *parent)
	:	AbstractScrollBar(parent), 
		flow_(flow)
{
	connect(flow_, SIGNAL(transitionValueChanged(float)), this, SLOT(flowTransitionValueChanged(float)));
	connect(flow_, SIGNAL(currentValueChanged(float)), this, SLOT(flowCurrentValueChanged(float)));
}

FlowScrollBar::~FlowScrollBar()
{
	disconnect(flow_, SIGNAL(currentValueChanged(float)), this, SLOT(flowCurrentValueChanged(float)));
	disconnect(flow_, SIGNAL(transitionValueChanged(float)), this, SLOT(flowTransitionValueChanged(float)));
}

void FlowScrollBar::flowTransitionValueChanged(float value)
{
	invalidate();
}

void FlowScrollBar::flowCurrentValueChanged(float value)
{
	invalidate();
}
