/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef PAINTBOX_H_
#define PAINTBOX_H_

#include <qwidget.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qregion.h>

class Layer;

class PaintBox : public QWidget
{
	Q_OBJECT;
public:
	PaintBox(QWidget *parent = 0, const char *name = 0, WFlags f = 0);

	QImage &buffer() { return buffer_; }
	QPixmap &displayPixmap() { return displayPixmap_; }

	virtual void invalidate();
	virtual void invalidate(const QRect &rect);

protected:
	virtual void paintEvent(QPaintEvent *event);
	virtual void resizeEvent(QResizeEvent* event);

	virtual void resizeBuffer();

	virtual void bufferResized(int oldWidth, int oldHeight);
	virtual void paintToBuffer(QImage &buffer, const QRect &clipRect);
	virtual void paintOverlay(QPainter &p, const QPixmap &pixmap);

private:
	QImage buffer_;
	bool bufferDirty_;
	QPixmap displayPixmap_;
};

typedef QList<Layer> LayerCollection;

class LayeredPaintBox : public PaintBox
{
	Q_OBJECT
public:
	LayeredPaintBox(QWidget *parent = 0, const char *name = 0, WFlags f = 0);

	LayerCollection &layers() { return layers_; }

signals:
	void rearrangeLayers();

protected:
	virtual void paintToBuffer(QImage &buffer, const QRect &clipRect);
	virtual void paintOverlay(QPainter &p, const QPixmap &pixmap);

	virtual Layer *doHitTest(const QPoint &);

	virtual void mousePressEvent(QMouseEvent *event);
	virtual void mouseMoveEvent(QMouseEvent *event);
	virtual void mouseReleaseEvent(QMouseEvent *event);
	virtual void mouseDoubleClickEvent(QMouseEvent *e);

	virtual void resizeEvent(QResizeEvent *event);

private:
	LayerCollection layers_;
	Layer *mouseListener_;
};

class Layer
{
public:
	Layer(LayeredPaintBox *parent);
	virtual ~Layer();

	virtual void setVisible(bool visible)
	{
		if (visible != visible_)
		{
			if (visible_)
				invalidate(); // we are about to hide, so invalidate at position

			visible_ = visible;

			if (visible_)
				invalidate(); // we are about to show, so invalidate at position
		}
	}

	const bool visible() const { return visible_; }

	virtual void invalidate(const QRect &rect)
	{
		if (parent_ && visible_)
		{
			parent_->invalidate(rect);
			//qDebug("PositionedLayer: %p, rect: %d, %d, %d, %d", this, rect.left(), rect.top(), rect.width(), rect.height());
		}
	}

	virtual void invalidate()
	{
		if (parent_)
			invalidate(parent_->geometry());
	}

	virtual void repaint(const QRect &rect)
	{
		if (parent_ && visible_)
		{
			parent_->invalidate(rect);
			parent_->repaint(false);
		}
	}

	virtual void repaint()
	{
		if (parent_)
			repaint(parent_->geometry());
	}

protected:
	friend class LayeredPaintBox;

	inline LayeredPaintBox *parent() { return parent_; }

	virtual void paintToBuffer(QImage &buffer, const QRect &clipRect) {}
	virtual void paintOverlay(QPainter &p, const QPixmap &pixmap) {}

	virtual void mousePressEvent(QMouseEvent *) {}
	virtual void mouseMoveEvent(QMouseEvent *) {}
	virtual void mouseReleaseEvent(QMouseEvent *) {}
	virtual void mouseDoubleClickEvent(QMouseEvent *) {}

	virtual bool hitTest(const QPoint &) { return false; }
	virtual bool paintTest(QRect &rect) { return true; }

private:
	LayeredPaintBox *parent_;
	bool visible_;
};

class PositionedLayer : public Layer
{
public:
	PositionedLayer(LayeredPaintBox *parent) : Layer(parent) {}
	virtual ~PositionedLayer() {}

	virtual void invalidate() { Layer::invalidate(geometry_); }
	virtual void repaint() { Layer::repaint(geometry_); }

	void setGeometry(const QRect &rect)
	{
		if (geometry_ != rect)
		{
			invalidate(); // invalidate with old position
			geometry_ = rect;
			resized();
			invalidate(); // invalidate with new position
		}
	}

	inline int left() { return geometry_.left(); }
	inline int top() { return geometry_.top(); }
	inline int height() { return geometry_.height(); }
	inline int width() { return geometry_.width(); }

	inline const QRect &geometry() { return geometry_; }

protected:
	virtual void resized() {}
	virtual bool hitTest(const QPoint &pos) { return geometry_.contains(pos); }
	virtual bool paintTest(QRect &rect) { rect = rect.intersect(geometry()); return rect.isValid(); }

private:
	QRect geometry_;
};

class TextLayer : public PositionedLayer
{
public:
	TextLayer(LayeredPaintBox *parent);

	virtual void invalidate();

	void setText(const QString &text) { text_ = text; invalidate(); }
	const QString text() const { return text_; }

	void setFont(const QFont &font) { font_ = font; invalidate(); }
	const QFont font() const { return font_; }

	void setFontMinAutoScale(bool value) { fontMinAutoScale_ = value; invalidate(); }
	const bool fontMinAutoScale() const { return fontMinAutoScale_; }

	void setColor(const QColor &color) { color_ = color; invalidate(); }
	const QColor color() const { return color_; }

	void setAlignment(int alignment) { alignment_ = alignment; invalidate(); }
	int alignment() const { return alignment_; }

	const QSize renderedTextSize() const;

protected:
	virtual void resized();
	virtual void paintOverlay(QPainter &p, const QPixmap &pixmap);

private:
	QString text_;
	QFont font_;
	QFont drawFont_;
	QColor color_;
	int alignment_;
	QSize renderedTextSize_;
	bool fontMinAutoScale_;
	bool invalid_;

	int measuredFontSize_;
	void remeasureFontSize();
};

void imageFillRect(QImage &dst, int x1, int y1, int x2, int y2, const QColor &color);
void imageFillRectS(QImage &dst, int x1, int y1, int x2, int y2, const QColor &color);
void imageFillRectS(QImage &dst, const QRect &rect, QColor color);

#endif /*PAINTBOX_H_*/
