/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qapplication.h>
#include <qtextstream.h>
#include <qlistview.h>
#include <qstringlist.h>
#include <qtextcodec.h>
#include <qwidgetstack.h>
#include <qmessagebox.h>
#include <qregexp.h>
#include <unistd.h>
#include <qtl.h>

#include "simplefiledialog.h"
#include "qinputdialog.h"
#include "qprogressdialog.h"
#include "configuration.h"
#include "playlistmanager.h"
#include "playlistview.h"
#include "helpers.h"
#include "media.h"
#include "mediadatabase.h"
#include "mediaidentifier.h"
#include "sqlite3.h"

#include "debug.h"

PlayListManager::PlayListManager(PlayListView *targetPlayList, QWidgetStack *container, Configuration &config, QWidget *owner)
	:	playlistFile(""),
		owner_(owner),
		playListContainer_(container),
		config_(config),
		playListView_(targetPlayList),
		isDynPlayList_(false),
		lboxDynPlayList_(NULL),
		previouslyVisibleWidget_(NULL),
		activeMode_(PlayListManager::ModeMax),
		playlistParams_(),
		otgPlaylistParams_(),
		stmtSelectTotalFileSizeAndCountForMediaLocation(NULL),
		stmtSelectAllMediaIDsForMediaLocation(NULL)
{
	dynPlayList_.clear();

	playlistParams_.playListSource = "playlist";
	otgPlaylistParams_.playListSource = "onthego_playlist";

	connect(playListView_->mediaDatabase(), SIGNAL(preparingStatements(sqlite3*)),
			this, SLOT(mediaDatabasePreparingStatements(sqlite3*)));

	connect(playListView_->mediaDatabase(), SIGNAL(finalizingStatements(sqlite3*)),
			this, SLOT(mediaDatabaseFinalizingStatements(sqlite3*)));
}

PlayListManager::~PlayListManager()
{
}

QString PlayListManager::getDisplayPlayListFilename()
{
	if (isOnTheGoPlayListActive())
		return "On-The-Go Play List";
	else if (playlistFile.isNull() || playlistFile.isEmpty())
		return "Untitled." + (isDynamicPlaylist() ? QString("dynplaylist") : QString("playlist"));
	else
		return QFileInfo(playlistFile).fileName();
}

void PlayListManager::execDialogAddFileToPlayList()
{
	addDialog(false);
}

void PlayListManager::execDialogAddDirToPlayList()
{
	addDialog(true);
}

void PlayListManager::execDialogAddDirWithSubDirsToPlayList()
{
	addDialog(true, true);
}

void PlayListManager::addDialog(const bool dir, const bool subdir)
{
	if (isDynamicPlaylist())
		setActiveMode(PlayListManager::ModeDynPlaylistEditor);

	if (!dir)
	{
		QString formatFilter = QString::null;

		for (QStringList::ConstIterator it = config_.validExtensions().begin(); it != config_.validExtensions().end(); ++it)
			formatFilter += "*." + (*it) + ";";

		formatFilter = formatFilter.left(formatFilter.length() - 1);

		QString selectedFilename = SimpleFileDialog::getOpenFileName(
			config_.lastDirectory,
			"All registered formats (" + formatFilter + ");;All files (*.*)",
			owner_,
			"add to play list dialog",
			tr("Add file to Playlist")
		);

		if (!selectedFilename.isNull() && !selectedFilename.isEmpty())
		{
			addFile(selectedFilename);

			// Update last directory
			config_.lastDirectory = QFileInfo(selectedFilename).filePath();
		}
	}
	else
	{
		QString selectedDirectory = SimpleFileDialog::getExistingDirectory(
			config_.lastDirectory,
			owner_,
			"add directory to playlist dialog",
			tr("Add directory to Playlist"),
			true,
			true
		);

		if (!selectedDirectory.isNull() && !selectedDirectory.isEmpty())
		{
			addDir(selectedDirectory, subdir);

			// Update last directory
			config_.lastDirectory = selectedDirectory;
		}
	}
}

bool PlayListManager::openASX(const QString &asx)
{
	QFile file(asx);
	if (file.open(IO_ReadOnly))
	{
		// Parse ASX
		QTextStream stream(&file);
		const QStringList list = QStringList::split('\n', stream.read());
		for (QStringList::ConstIterator it = list.begin(); it != list.end(); ++it)
		{
			QString line(*it);
			QString uri;

			if (line.contains("ref href", false))
			{
				// String Indexes
				int uriBegin, uriEnd;

				// Find URI Index
				uriBegin = line.find("href", 0, false);
				uriBegin = line.find('\"', uriBegin);
				++uriBegin;

				uriEnd = line.find('\"', uriBegin);

				// Get URI
				uri = line.mid(uriBegin, uriEnd - uriBegin);

				// Add to PlayListView
				openURL(uri);
			}
		}

		file.close();

		return true;
	}
	else
		return false;
}

void PlayListManager::openURL(const QString &uri)
{
	bool ok;
	QString url;

	if (uri == NULL)
		url = QInputDialog::getText(tr("URL to open..."), tr("URL"), QLineEdit::Normal, QString::null, &ok, owner_);
	else
		url = uri;

	if (ok && !url.isEmpty())
	{
		playListView_->insertItem(url);
	}

	playListView_->updateFilter();
}

void PlayListManager::addFile(const QString &path)
{
	if (isDynamicPlaylist())
	{
		if (isDynamicPlaylistEditorActive())
			lboxDynPlayList_->insertItem(path);
		else
			dynPlayList_.append(path);
	}
	else
		playListView_->insertItem(QFileInfo(path).absFilePath());
}

void PlayListManager::mediaDatabasePreparingStatements(sqlite3 *db)
{
	QString query =
		"SELECT SUM(filesize), COUNT() "
		"FROM media, media_location "
		"WHERE media.location_id = media_location.location_id AND media_location.Location = ?1";

	sqlite3_prepare_v2(db, query.utf8(), -1, &stmtSelectTotalFileSizeAndCountForMediaLocation, 0);

	query =
		"SELECT media_id "
		"FROM media, media_location "
		"WHERE media_location.location = ?1 AND media.location_id = media_location.location_id "
		"ORDER BY filename";

	sqlite3_prepare_v2(db, query.utf8(), -1, &stmtSelectAllMediaIDsForMediaLocation, 0);
}

void PlayListManager::mediaDatabaseFinalizingStatements(sqlite3 *db)
{
	sqlite3_finalize(stmtSelectTotalFileSizeAndCountForMediaLocation);
	stmtSelectTotalFileSizeAndCountForMediaLocation = NULL;

	sqlite3_finalize(stmtSelectAllMediaIDsForMediaLocation);
	stmtSelectAllMediaIDsForMediaLocation = NULL;
}

void PlayListManager::fillFileList(QExtStringList &filelist, const QString &path, const bool subdir)
{
	QDir dir(path);
	if (!dir.exists()) return;

	dir.setFilter(QDir::Files | QDir::Hidden | (subdir ? QDir::Dirs : 0));
	dir.setSorting(QDir::Unsorted);

	const QFileInfoList *list = dir.entryInfoList();
	QExtString fileName;
	QFileInfoListIterator it(*list);
	QFileInfo *info;

	QExtStringList localFiles;
	uint localFilesTotalSize = 0;

	for (; (info = it.current()); ++it)
	{
		if (info->isFile())
		{
			// only add the file, if its extension is known...
			if (config_.isValidExtension(info->extension(false)))
			{
				localFiles.append(info->absFilePath());
				localFilesTotalSize += info->size();
			}
		}
		else
		{
			fileName = info->fileName();
			if (fileName != "." && fileName != "..")
				fillFileList(filelist, info->absFilePath(), subdir);
		}
	}

	if (localFiles.count() > 0)
	{
#ifdef WINDOWS
		// On Windows canonicalPath() has a really weird and slow behavior (possibly due to QT_GETCWD and QT_CHDIR calls)
		// using absPath() instead...
		QString canonicalPath = dir.absPath();
#else
		QString canonicalPath = dir.canonicalPath();
#endif
		if (canonicalPath.right(1) != '/')
			canonicalPath += '/';

		sqlite3_bind_text(stmtSelectTotalFileSizeAndCountForMediaLocation, 1, canonicalPath.utf8(), -1, SQLITE_TRANSIENT);

		uint filesize = 0;
		uint count = 0;

		if (sqlite3_step(stmtSelectTotalFileSizeAndCountForMediaLocation) == SQLITE_ROW)
		{
			filesize = sqlite3_column_int(stmtSelectTotalFileSizeAndCountForMediaLocation, 0);
			count = sqlite3_column_int(stmtSelectTotalFileSizeAndCountForMediaLocation, 1);
		}

		DHPRINTF("canonicalPath: %s", (const char*)canonicalPath.utf8());
		DHPRINTF("filesize: %d, count: %d", filesize, count);

		sqlite3_reset(stmtSelectTotalFileSizeAndCountForMediaLocation);

		if (filesize == localFilesTotalSize && count == localFiles.count())
		{
			DHPRINTF("using fastpath 1");
			sqlite3_bind_text(stmtSelectAllMediaIDsForMediaLocation, 1, canonicalPath.utf8(), -1, SQLITE_TRANSIENT);

			qHeapSort(localFiles);
			QExtStringList::Iterator it = localFiles.begin();

			while (sqlite3_step(stmtSelectAllMediaIDsForMediaLocation) == SQLITE_ROW)
			{
				QExtString &item = *it;
				item.setID(sqlite3_column_int(stmtSelectAllMediaIDsForMediaLocation, 0));
				DHPRINTF("item: %s, id: %d", (const char *)item.utf8(), item.id());
				++it;
			}

			sqlite3_reset(stmtSelectAllMediaIDsForMediaLocation);
		}
		// TODO: Implement and profile a second fast path attempt, that reads
		// partially available information from the database and assigns media IDs
		// based on this information...

		filelist += localFiles;
	}
}

void PlayListManager::addDir(const QString &path, const bool subdir)
{
	DENTERMETHOD("PlayListManager::addDir(\"%s\", %s)", (const char*)path.utf8(), subdir ? "true" : "false");

	QString location = path;

	if (location.right(1) != "/")
		location += "/";

	if (subdir)
		location += "**";

	if (isDynamicPlaylist())
	{
		if (isDynamicPlaylistEditorActive())
			lboxDynPlayList_->insertItem(location);
		else
			dynPlayList_.append(location);
	}
	else
	{
		QExtStringList filelist;
		filelist.append(location);
		addFromPlayList(filelist, true);
	}

	DEXITMETHOD("PlayListManager::addDir(\"%s\", %s)", (const char*)path.utf8(), subdir ? "true" : "false");
}

void PlayListManager::newPlayList()
{
	setActiveMode(PlayListManager::ModeOpenedPlaylist);

	clearPlayList();
	playlistFile = "";
	config_.lastPlaylistFile = "";
	isDynPlayList_ = false;

	emit playlistFileNameChanged();
}

void PlayListManager::newDynamicPlayList()
{
	setActiveMode(PlayListManager::ModeOpenedPlaylist);

	clearPlayList();
	playlistFile = "";
	config_.lastPlaylistFile = "";
	isDynPlayList_ = true;

	checkInputMode();

	emit playlistFileNameChanged();

	//setActiveMode(PlayListManager::ModeDynPlaylistEditor);
}

void PlayListManager::clearPlayList()
{
	if (activeMode_ == ModeOpenedPlaylist)
	{
		dynPlayList_.clear();
	}
	else if (activeMode_ == ModeDynPlaylistEditor)
	{
		lboxDynPlayList_->clear();
		dynPlayList_.clear();
	}

	playListView_->clear();
	emit playlistCleared();

	//switchToDynamicPlayListEditor();

	DMEMSTAT();
}

void PlayListManager::savePlayList()
{
	if (!playlistFile || playlistFile.isEmpty() || activeMode_ == ModeOnTheGoPlaylist) {
		execDialogSavePlayListAs();
		return;
	}

	writePlayList(playlistFile);
}

void PlayListManager::execDialogSavePlayListAs()
{
	QString newFilename = SimpleFileDialog::getSaveFileName(
		config_.lastPlaylistDirectory,
		(isDynPlayList_ && activeMode_ != ModeOnTheGoPlaylist) ?
			"Dynamic playlist (*.dynplaylist)" :
			"Playlist files (*.playlist;*.m3u)",
		owner_,
		"save play list dialog",
		tr("Save Playlist Dialog")
	);

	if (!newFilename.isNull())
		writePlayList(newFilename);
}

void PlayListManager::execDialogOpenPlayList()
{
	QString selectedFilename = SimpleFileDialog::getOpenFileName(
		config_.lastPlaylistDirectory,
		"All supported playlist formats (*.dynplaylist;*.playlist;*.m3u;*.asx);;"\
			"Quasar dynamic playlists (*.dynplaylist);;"\
			"Quasar playlists (*.playlist);;"\
			"M3U playlists (*.m3u);;"\
			"Windows Media Meta Files (*.asx)",
		owner_,
		"open play list dialog",
		tr("Open Playlist Dialog")
	);

	if (!selectedFilename.isNull())
	{
		if (activeMode_ == ModeOnTheGoPlaylist)
			clearPlayList();
		else
			newPlayList();

		readPlayList(selectedFilename);
		playListView_->setActiveItem(0);
		playListView_->setFocus();
	}
}


void PlayListManager::refreshDynamicPlayList()
{
	playListView_->clear();
	readPlayList(playlistFile, dynPlayList_, tr("Refreshing playlist..."));
}

void PlayListManager::writePlayList(const QString &filename)
{
	QString fullFilename = filename;
	QString extension = QFileInfo(fullFilename).extension(false);

	if (isDynamicPlaylist())
	{
		if (extension != "dynplaylist")
			fullFilename += ".dynplaylist";
	}
	else if (extension != "playlist")
		fullFilename += ".playlist";

	QString playListDir = QFileInfo(fullFilename).dirPath(true);

	QFile file(fullFilename);
	if (file.open(IO_WriteOnly))
	{
		QTextStream s(&file);
		s.setEncoding(QTextStream::UnicodeUTF8);

		if (isDynamicPlaylist())
		{
			if (isDynamicPlaylistEditorActive())
				for (int i = 0; i < lboxDynPlayList_->count(); i++)
					s << makeLocationRelative(playListDir, lboxDynPlayList_->text(i)) << "\n";
			else
				for (QExtStringList::Iterator it = dynPlayList_.begin(); it != dynPlayList_.end(); ++it )
					s << makeLocationRelative(playListDir, *it) << "\n";
		}
		else
			for (QListViewItemIterator it(playListView_); it.current(); ++it)
				s << makeLocationRelative(playListDir, static_cast<PlayListViewItem *>(it.current())->media()->location().toString()) << "\n";

		// Update last directory/file if this is the real playlist, ie. it's not the On-The-Go playlist...
		if (activeMode_ != ModeOnTheGoPlaylist)
		{
			playlistFile = fullFilename;
			config_.lastPlaylistFile = playlistFile;
		}

		config_.lastPlaylistDirectory = QFileInfo(fullFilename).dirPath(true);

		// Close File
		file.close();

		emit playlistFileNameChanged();
	}
}

void PlayListManager::addToPlayList(const QExtStringList &filelist, bool interpretPlayList, const QString &playlistDirectory, const QString &dialogTitle)
{
	if (isDynamicPlaylist())
	{
		if (isDynamicPlaylistEditorActive())
		{
			for (QExtStringList::ConstIterator it = filelist.begin(); it != filelist.end(); ++it)
				lboxDynPlayList_->insertItem(*it);
		}
		else
		{
			dynPlayList_ += filelist;
			refreshDynamicPlayList();
		}
	}
	else
		addFromPlayList(filelist, interpretPlayList, playlistDirectory, dialogTitle);
}

void PlayListManager::addFromPlayList(const QExtStringList &filelist, bool interpretPlayList, const QString &playlistDirectory, const QString &dialogTitle)
{
	DTIMERINIT(timer);

	QExtStringList list(filelist);

	// initialize dialog and shot it...
	QProgressDialog progress(tr("Please wait..."), tr("Cancel"), 0, owner_, "progress", true);

	progress.setCaption(tr("Scanning for files..."));

	progress.setMinimumDuration(1000);
	progress.setFixedWidth(QMIN(qApp->desktop()->width() * 0.9, 640));
	progress.setAutoClose(false);
	progress.setAutoReset(false);
	progress.show();
	qApp->processEvents();

	// expand various entries in the list if specifically wished for...
	if (interpretPlayList)
	{
		// expand dynamic playlist to real content...
		QExtStringList expandedList;
		for (QExtStringList::ConstIterator it = list.begin(); it != list.end(); ++it)
		{
			QString location(*it);

			if (location.startsWith("#") || location.startsWith(";"))
				continue;

			// WHY OH WHY is there startsWith but no endsWith in QString in Qt 2.3.x ?!?!
			if (location.right(3) == "/**" || location == "**")
			{
				// recursive scanning of location...
				location = resolveLocation(playlistDirectory, location.left(QMAX(0, location.length() - 3)));
				QExtStringList fileList;
				fillFileList(fileList, location, true);
				qHeapSort(fileList);
				expandedList += fileList;
			}
			else if (location.right(2) == "/*" || location.right(1) == "/")
			{
				// all files of the location...
				location = resolveLocation(playlistDirectory, location.left(location.findRev('/')));
				QExtStringList fileList;
				fillFileList(fileList, location, false);
				qHeapSort(fileList);
				expandedList += fileList;
			}
			else if (location.startsWith("http://") || location.startsWith("mms://"))
				expandedList.append(location);
			else
			{
				location = resolveLocation(playlistDirectory, location);
				QFileInfo fi(location);

				if (fi.exists())
				{
					if (fi.isDir())
					{
						QExtStringList fileList;
						fillFileList(fileList, location, false);
						qHeapSort(fileList);
						expandedList += fileList;
					}
					else if (config_.isValidExtension(fi.extension()))
						// just add the (file) location to the expanded list...
						expandedList.append(location);
				}
			}
		}

		list = expandedList;
	}

	// start reading the playlist entries...
	int numFiles = list.count();
	progress.setTotalSteps(numFiles);

	if (!dialogTitle)
		progress.setCaption(tr("Adding files..."));
	else
		progress.setCaption(dialogTitle);

	qApp->processEvents();

	int i = 0;

	playListView_->beginChangingUpdate();

	QTime time;
	time.start();

	PlayListViewItem *lastItem = NULL;

	for (QExtStringList::ConstIterator it = list.begin(); it != list.end(); ++it, ++i)
	{
		const QExtString &location = *it;

		if (location.id())
		{
			DHPRINTF("ID %d set on %s", location.id(), (const char *)location.utf8());
			lastItem = playListView_->insertItem(location.id(), lastItem);
		}
		else
		{
			// skip comments
			if (location.startsWith("#") || location.startsWith(";"))
				continue;

			DHPRINTF("No ID set on %s", (const char *)location.utf8());

			if (!(isPathAbsolute(location) || location.startsWith("http://") || location.startsWith("mms://")))
				lastItem = playListView_->insertItem(resolveLocation(playlistDirectory, location), lastItem);
			else
				lastItem = playListView_->insertItem(location, lastItem);
		}

		if (time.elapsed() > 1000)
		{
			progress.setProgress(i);

			if (location.length() > 55)
				progress.setLabelText(location.left(20) + "..." + location.right(40));
			else
				progress.setLabelText(location);

			qApp->processEvents();

			time.restart();
		}

		if (progress.wasCancelled())
			break;
	}

	progress.setProgress(numFiles);

	progress.setCaption(tr("Updating playlist..."));
	progress.setLabelText(tr("Please wait..."));
	qApp->processEvents();

	DTIMERPRINT(timer, "playlist added to DB in");
	playListView_->endChangingUpdate();

	playListView_->reloadView();

	progress.hide();
	qApp->processEvents();

	checkInputMode();

	DMEMSTAT();
}

void PlayListManager::readPlayList(const QString &filename, const QExtStringList &filelist, const QString &dialogTitle, bool assumeInDB)
{
	QString playlistDirectory = QFileInfo(filename).dirPath(true);

	// Are we dealing with a dynamic playlist here?
	// If so, we need to expand various entries in that list...
	bool expandPlayList = (filename.lower().right(12) == ".dynplaylist" ||
		(isDynamicPlaylist() && (filename.isNull() || filename.isEmpty())));

	if (activeMode_ != ModeOnTheGoPlaylist)
	{
		if (expandPlayList)
			dynPlayList_ = filelist;

		isDynPlayList_ = expandPlayList;
	}

	if (!assumeInDB)
		addFromPlayList(filelist, expandPlayList, playlistDirectory, dialogTitle);
	else
		playListView_->reloadView();

	emit playlistLoaded();
}

bool PlayListManager::readPlayList(const QString &filename, bool assumeInDB)
{
	DENTERMETHOD("PlayListManager::readPlayList()");
	DTIMERINIT(timer);

	// Load ASX
	QRegExp asxExp("^[^\\.]+.asx$", false);
	if (asxExp.match(filename) != -1)
		return openASX(filename);

	bool result = false;

	// Load PlayListView
	QFile file(filename);
	if (file.open(IO_ReadOnly))
	{
		QString playlistDirectory = QFileInfo(filename).dirPath(true);

		// Update last directory/file if this is the real playlist, ie. it's not the On-The-Go playlist...
		if (activeMode_ != ModeOnTheGoPlaylist)
		{
			playlistFile = filename;
			config_.lastPlaylistFile = playlistFile;
		}

		config_.lastPlaylistDirectory = playlistDirectory;

		// Load PlayListView
		QTextStream stream(&file);

		QExtStringList list;
		QString line;

		while (!(line = stream.readLine()).isNull())
		{
			if (containsUTF8(line))
			{
				DHPRINTF("UTF8 detected: %s", (const char *)line.latin1());
				line = reinterpretUTF8(line);
			}
			else
				DHPRINTF("no UTF8: %s", (const char *)line.latin1());

			if (!line.startsWith("#") && !line.startsWith(";") && !line.isEmpty())
				line = resolveLocation(playlistDirectory, line);

			list.append(line);
		}

		readPlayList(filename, list, NULL, assumeInDB);

		file.close();

		emit playlistFileNameChanged();

		result = true;
	}

	DTIMERPRINT(timer, "playlist read in");
	DEXITMETHOD("PlayListManager::readPlayList()");

	return result;
}

void PlayListManager::removeFileFromPlayList()
{
	if (isDynamicPlaylist())
	{
		if (isDynamicPlaylistEditorActive())
		{
			for (int i = lboxDynPlayList_->count() - 1; i > -1; --i)
				if (lboxDynPlayList_->isSelected(i))
					lboxDynPlayList_->removeItem(i);
		}
		else
		{
			int result = QMessageBox::information(owner_, tr("Dynamic playlist"),
					 tr("This playlist is dynamic, thus you can't delete\nan item from this playlist directly.\nDo you want to switch to the editor\nfor dynamic playlists and delete\nthe item there?"),
					 QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape);

			if (result == QMessageBox::Yes)
				setActiveMode(PlayListManager::ModeDynPlaylistEditor);
		}
	}
	else
	{
		playListView_->removeSelectedItems();
		playListView_->updateFilter();
	}
}

void PlayListManager::cutSelectionFromPlayList()
{
	mediaIDClipboard_ = playListView_->getSelectedMediaIDs();
	removeFileFromPlayList();
	emit clipboardChanged();
}

void PlayListManager::copySelectionFromPlayList()
{
	mediaIDClipboard_ = playListView_->getSelectedMediaIDs();
	emit clipboardChanged();
}

void PlayListManager::pasteIntoPlayList()
{
	if (!isDynamicPlaylist() && mediaIDClipboard_.count() > 0)
	{
		// if the playlist is not sorted or filtered, then we can
		// directly insert the new items at the current position...
		if (!playListView_->isSorted() &&
			!playListView_->isFiltered())
		{
			PlayListViewItem *lastItem = NULL;
			bool moveFirstItem = false;

			// is anything selected?
			if (playListView_->selectedItems().count() > 0)
			{
				// if yes, take the first item as place to insert...
				lastItem = playListView_->selectedItems().at(0);
				DHOP(lastItem->media()->dumpInfo());

				// since we want to place the new items at the position/index of
				// the previously selected insertion point, we need to get the item
				// above it, because we can only tell QListView to insert after
				// a given item...
				if (lastItem->itemAbove())
					lastItem = static_cast<PlayListViewItem *>(lastItem->itemAbove());
				else if (lastItem == playListView_->firstItem())
					// if there is no item above, this means we have selected the first
					// child in the list. Do a sanity check and then mark the item separately
					// so we can move it to the end of the new stuff after our items have
					// been inserted...
					moveFirstItem = true;
			}
			else
				// insert at end of list if nothing is selected or the playlist is empty...
				lastItem = playListView_->lastItem();

			playListView_->selectAll(false);

			for (QValueList<unsigned long>::Iterator listit = mediaIDClipboard_.begin(); listit != mediaIDClipboard_.end(); ++listit)
			{
				lastItem = playListView_->insertItem((*listit), lastItem);
				playListView_->setSelected(lastItem, true);
			}

			// if we have a marked an item that was (and still is) the first child in the list
			// move it to the end of our inserted items...
			if (moveFirstItem)
				playListView_->firstItem()->moveItem(lastItem);
		}
		// the playlist is sorted or filtered, so insert at DB level...
		else
		{
			playListView_->beginChangingUpdate();
			directlyAppendToPlaylistTable(mediaIDClipboard_, playListView_->playListSource());
			playListView_->endChangingUpdate();
			playListView_->reloadView();
		}
	}
}

void PlayListManager::directlyAppendToPlaylistTable(const QValueList<unsigned long> &itemIDs, const QString &tableName)
{
	int newPlaylistID = 0;
	sqlite3_stmt *vm;

	// query playlist index for first new item...
	QString query = "SELECT MAX(idx)+1 FROM " + tableName + ";";
	sqlite3_prepare_v2(playListView_->mediaDatabase()->db(), (const char*)query.utf8(), -1, &vm, 0);
	if (sqlite3_step(vm) == SQLITE_ROW)
		newPlaylistID = sqlite3_column_int(vm, 0);
	else
	{
		// Nothing found? Funny we got here...
		// Something is wrong with our DB! clean up - back out.
		sqlite3_finalize(vm);
		return;
	}
	sqlite3_finalize(vm);

	// append new items to the OTG playlist...
	query = "INSERT INTO " + tableName + " VALUES(?1, NULL, ?2);";
	sqlite3_prepare_v2(playListView_->mediaDatabase()->db(), (const char*)query.utf8(), -1, &vm, 0);
	for (QValueList<unsigned long>::ConstIterator it = itemIDs.begin(); it != itemIDs.end(); ++it )
	{
		sqlite3_bind_int(vm, 1, newPlaylistID);
		sqlite3_bind_int(vm, 2, (*it));
		sqlite3_step(vm);
		sqlite3_reset(vm);
		++newPlaylistID;
	}

	sqlite3_finalize(vm);
}

void PlayListManager::selectAll()
{
	if (isDynamicPlaylistEditorActive())
	{
		lboxDynPlayList_->selectAll(true);
		if (lboxDynPlayList_->currentItem() == -1 && lboxDynPlayList_->count() > 0)
			lboxDynPlayList_->setCurrentItem(0);
	}
	else
		playListView_->selectAll(true);
}

void PlayListManager::deselectAll()
{
	if (isDynamicPlaylistEditorActive())
	{
		lboxDynPlayList_->selectAll(false);
		lboxDynPlayList_->setCurrentItem(-1);
	}
	else
		playListView_->selectAll(false);
}

void PlayListManager::copyToOnTheGoPlayList()
{
	if (playListView_->selectedItems().count() > 0 && activeMode_ != PlayListManager::ModeOnTheGoPlaylist)
	{
		directlyAppendToPlaylistTable(playListView_->getSelectedMediaIDs(), "onthego_playlist");

		if (playListView_->selectedItems().count() > 1)
			emit showMessage(tr("Items copied to On-The-Go Playlist."), 1500);
		else
			emit showMessage(tr("Item copied to On-The-Go Playlist."), 1500);
	}
}

bool PlayListManager::setActiveMode(Mode mode)
{
	bool result = false;

	if (mode == activeMode_)
		result = true;

	// if we are already in a playlist view and the last active playlist was requested,
	// do nothing and return positive.
	if (mode == PlayListManager::ModeLastActivePlaylist &&
		(activeMode_ == PlayListManager::ModeOnTheGoPlaylist || activeMode_ == PlayListManager::ModeOpenedPlaylist))
		result = true;

	emit switchingModeTo(mode);

	if (!result)
	{
		if (mode == PlayListManager::ModeOpenedPlaylist ||
			mode == PlayListManager::ModeOnTheGoPlaylist ||
			mode == PlayListManager::ModeLastActivePlaylist)
		{
			if (isDynamicPlaylist() && activeMode_ == PlayListManager::ModeDynPlaylistEditor)
			{
				QExtStringList newDynPlayList;

				for (int i = 0; i < lboxDynPlayList_->count(); i++)
					newDynPlayList.append(lboxDynPlayList_->text(i));

				// did anything change?
				if (!(newDynPlayList == dynPlayList_))
				{
					// if so, copy over
					dynPlayList_ = newDynPlayList;
					// and refresh the real playlist...
					refreshDynamicPlayList();
				}

				// make previous widget visible...
				playListContainer_->raiseWidget(previouslyVisibleWidget_);
				previouslyVisibleWidget_ = NULL;

				delete lboxDynPlayList_;
				lboxDynPlayList_ = NULL;
			}

			// resolve last active play list and set mode accordingly...
			if (mode == PlayListManager::ModeLastActivePlaylist)
				mode = (playListView_->playListSource() == "playlist" ? PlayListManager::ModeOpenedPlaylist : PlayListManager::ModeOnTheGoPlaylist);

			// save current playlist params...
			if (playListView_->playListSource() == "playlist")
				playListView_->saveState(&playlistParams_);
			else if (playListView_->playListSource() == "onthego_playlist")
				playListView_->saveState(&otgPlaylistParams_);

			if (mode == PlayListManager::ModeOpenedPlaylist)
				playListView_->loadState(&playlistParams_);
			else if (mode == PlayListManager::ModeOnTheGoPlaylist)
				playListView_->loadState(&otgPlaylistParams_);

			checkInputMode();

			result = true;
			activeMode_ = mode;
		}
		else if (mode == PlayListManager::ModeDynPlaylistEditor)
		{
			if (isDynamicPlaylist())
			{
				if (!lboxDynPlayList_)
				{
					lboxDynPlayList_ = new QListBox(playListContainer_);
					connect(lboxDynPlayList_, SIGNAL(selectionChanged()), this, SIGNAL(selectionChanged()));

				    for (QExtStringList::ConstIterator it = dynPlayList_.begin(); it != dynPlayList_.end(); ++it)
				    	lboxDynPlayList_->insertItem(new QListBoxText(*it), -1);

					// remember previously visible widget, so we can reselect it afterwards...
					previouslyVisibleWidget_ = playListContainer_->visibleWidget();
				}

				// show our editor...
				playListContainer_->raiseWidget(lboxDynPlayList_);

				result = true;
				activeMode_ = PlayListManager::ModeDynPlaylistEditor;
			}
			else
				result = false;
		}
	}

	if (result)
		emit switchedModeTo(mode);

	checkInputMode();

	return result;
}

void PlayListManager::checkInputMode()
{
	if (isDynamicPlaylist() && playListView_->inputMode() == PlayListView::Move)
		playListView_->setInputMode(PlayListView::Select);

	if (isDynamicPlaylistEditorActive())
	{
		switch (inputMode())
		{
		case PlayListView::Select:
			lboxDynPlayList_->selectAll(false);
			lboxDynPlayList_->setSelectionMode(QListBox::Single);
			if (lboxDynPlayList_->currentItem())
				lboxDynPlayList_->setSelected(lboxDynPlayList_->currentItem(), true);
			break;
		case PlayListView::MultiSelect:
		case PlayListView::Move:
			lboxDynPlayList_->setSelectionMode(QListBox::Extended);
			break;
		}
	}
}

PlayListView::InputMode PlayListManager::inputMode()
{
	return playListView_->inputMode();
}

void PlayListManager::setInputMode(PlayListView::InputMode inputMode)
{
	playListView_->setInputMode(inputMode);
	checkInputMode();
}
