/*
 * playlist.cpp
 *
 *  Created on: 18.11.2009
 *      Author: darkstar
 */

#include "playlist.h"
#include "helpers.h"
#include "media.h"
#include "mediadatabase.h"
#include "mediaidentifier.h"
#include "sqlite3.h"

#include "configuration.h"
#include "debug.h"

PlayList::PlayList(MediaDatabase *mediaDatabase, const QString dbTable)
	:	QObject(),
	 	mediaDatabase_(mediaDatabase),
	 	dbTable_(dbTable),
		stmtSelectTotalFileSizeAndCountForMediaLocation(NULL),
		stmtSelectAllMediaIDsForMediaLocation(NULL),
		type_(Static),
		playListFileName_(QString::null),
		dynPlayList_()
{
	connect(mediaDatabase_, SIGNAL(preparingStatements(sqlite3*)),
			this, SLOT(mediaDatabasePreparingStatements(sqlite3*)));

	connect(mediaDatabase_, SIGNAL(finalizingStatements(sqlite3*)),
			this, SLOT(mediaDatabaseFinalizingStatements(sqlite3*)));
}

PlayList::~PlayList()
{

}

void PlayList::beginUpdate()
{
	DENTERMETHOD("PlayList::beginUpdate()");

	if (updateCount_ == 0)
	{
		blockSignals(true);
		mediaDatabase_->beginUpdate();
	}
	++updateCount_;

	DEXITMETHOD("PlayList::beginUpdate()");
}

void PlayList::endUpdate()
{
	DENTERMETHOD("PlayList::endUpdate()");

	--updateCount_;
	if (updateCount_ == 0)
	{
		DTIMERINIT(time);
		mediaDatabase_->endUpdate();
		DTIMERPRINT(time, "mediaDatabase_->endUpdate()");

		DTIMERSTART(time);
		blockSignals(false);
		propagateChanges();
		DTIMERPRINT(time, "change propagation");
	}

	DEXITMETHOD("PlayList::endUpdate()");
}

void PlayList::propagateChanges()
{
	if (updateCount_ == 0)
		emit playListChanged();
}

void PlayList::setType(Type type)
{
	if (type_ != type)
	{
		type_ = type;
		reset();
	}
}

void PlayList::requireRepopulation()
{
	if (updateCount_ == 0)
		repopulate();
}

void PlayList::repopulate()
{
	beginUpdate();

	clearTable();
	addFromFileList(dynPlayList_, true);

	endUpdate();
	propagateChanges();
}

void PlayList::reset()
{
	playListFileName_ = QString::null;
	clear();
}

void PlayList::clear()
{
	dynPlayList_.clear();
	clearTable();
}

void PlayList::clearTable()
{
	beginUpdate();

	QString query = "DELETE FROM \"" + dbSafeString(dbTable_) + "\";";
	sqlite3_exec(mediaDatabase_->db(), (const char *)query.utf8(), NULL, NULL, NULL);

	endUpdate();
	propagateChanges();
}

void PlayList::setDynPlayList(const QExtStringList &dynPlayList)
{
	if (isDynamic())
	{
		dynPlayList_ = dynPlayList;
		requireRepopulation();
	}
}

void PlayList::addFile(const QString &fileName)
{
	DENTERMETHOD("PlayList::addFile(\"%s\")", (const char*)fileName.utf8());

	QString location = QFileInfo(fileName).absFilePath();

	if (isDynamic())
	{
		dynPlayList_.append(location);
		requireRepopulation();
	}
	else
	{
		QExtStringList filelist;
		filelist.append(location);
		addFromFileList(filelist, false);
	}

	DEXITMETHOD("PlayList::addFile(\"%s\")", (const char*)fileName.utf8());
}

void PlayList::addDir(const QString &path, const bool subdir)
{
	DENTERMETHOD("PlayList::addDir(\"%s\", %s)", (const char*)path.utf8(), subdir ? "true" : "false");

	QString location = path;

	if (location.right(1) != "/")
		location += "/";

	if (subdir)
		location += "**";

	if (isDynamic())
	{
		dynPlayList_.append(location);
		requireRepopulation();
	}
	else
	{
		QExtStringList filelist;
		filelist.append(location);
		addFromFileList(filelist, true);
	}

	DEXITMETHOD("PlayList::addDir(\"%s\", %s)", (const char*)path.utf8(), subdir ? "true" : "false");
}

void PlayList::addURL(const QString &url)
{
	DENTERMETHOD("PlayList::addURL(\"%s\")", (const char*)url.utf8());

	if (isDynamic())
	{
		dynPlayList_.append(url);
		requireRepopulation();
	}
	else
	{
		QExtStringList filelist;
		filelist.append(url);
		addFromFileList(filelist, false);
	}

	DEXITMETHOD("PlayList::addURL(\"%s\")", (const char*)url.utf8());
}

int PlayList::getNextIndex() const
{
	int result = 0;
	sqlite3_stmt *vm;

	// query playlist index for first new item...
	QString query = "SELECT MAX(idx)+1 FROM \"" + dbSafeString(dbTable_) + "\";";
	sqlite3_prepare_v2(mediaDatabase_->db(), (const char*)query.utf8(), -1, &vm, 0);

	if (sqlite3_step(vm) == SQLITE_ROW)
		result = sqlite3_column_int(vm, 0);

	sqlite3_finalize(vm);

	return result;
}

void PlayList::appendMediaIDs(const MediaIDList &mediaIDs)
{
	int newIndex = getNextIndex();

	sqlite3_stmt *vm;

	QString query = "INSERT INTO \"" + dbSafeString(dbTable_) + "\" VALUES(?1, NULL, ?2);";
	sqlite3_prepare_v2(mediaDatabase_->db(), (const char*)query.utf8(), -1, &vm, 0);
	for (MediaIDList::ConstIterator it = mediaIDs.begin(); it != mediaIDs.end(); ++it )
	{
		sqlite3_bind_int(vm, 1, newIndex);
		sqlite3_bind_int(vm, 2, (*it));
		sqlite3_step(vm);
		sqlite3_reset(vm);
		++newIndex;
	}

	sqlite3_finalize(vm);
}

void PlayList::mediaDatabasePreparingStatements(sqlite3 *db)
{
	QString query =
		"SELECT SUM(filesize), COUNT() "
		"FROM media, media_location "
		"WHERE media.location_id = media_location.location_id AND media_location.Location = ?1";

	sqlite3_prepare_v2(db, query.utf8(), -1, &stmtSelectTotalFileSizeAndCountForMediaLocation, 0);

	query =
		"SELECT media_id "
		"FROM media, media_location "
		"WHERE media_location.location = ?1 AND media.location_id = media_location.location_id "
		"ORDER BY filename";

	sqlite3_prepare_v2(db, query.utf8(), -1, &stmtSelectAllMediaIDsForMediaLocation, 0);

	query =
		"INSERT INTO \"" + dbSafeString(dbTable_) + "\" "
		"VALUES(?1, ?2, ?3);";

	sqlite3_prepare_v2(db, query.utf8(), -1, &stmtInsertMediaIDIntoPlaylist, 0);
}

void PlayList::mediaDatabaseFinalizingStatements(sqlite3 *db)
{
	sqlite3_finalize(stmtSelectTotalFileSizeAndCountForMediaLocation);
	stmtSelectTotalFileSizeAndCountForMediaLocation = NULL;

	sqlite3_finalize(stmtSelectAllMediaIDsForMediaLocation);
	stmtSelectAllMediaIDsForMediaLocation = NULL;

	sqlite3_finalize(stmtInsertMediaIDIntoPlaylist);
	stmtInsertMediaIDIntoPlaylist = NULL;
}

void PlayList::fillFileList(QExtStringList &filelist, const QString &path, const bool subdir)
{
	QDir dir(path);
	if (!dir.exists()) return;

	dir.setFilter(QDir::Files | QDir::Hidden | (subdir ? QDir::Dirs : 0));
	dir.setSorting(QDir::Unsorted);

	const QFileInfoList *list = dir.entryInfoList();
	QExtString fileName;
	QFileInfoListIterator it(*list);
	QFileInfo *info;

	QExtStringList localFiles;
	uint localFilesTotalSize = 0;

	for (; (info = it.current()); ++it)
	{
		if (info->isFile())
		{
			// only add the file, if its extension is known...
			if (qConfig.isValidExtension(info->extension(false)))
			{
				localFiles.append(info->absFilePath());
				localFilesTotalSize += info->size();
			}
		}
		else
		{
			fileName = info->fileName();
			if (fileName != "." && fileName != "..")
				fillFileList(filelist, info->absFilePath(), subdir);
		}
	}

	if (localFiles.count() > 0)
	{
#ifdef WINDOWS
		// On Windows canonicalPath() has a really weird and slow behavior (possibly due to QT_GETCWD and QT_CHDIR calls)
		// using absPath() instead...
		QString canonicalPath = dir.absPath();
#else
		QString canonicalPath = dir.canonicalPath();
#endif
		if (canonicalPath.right(1) != '/')
			canonicalPath += '/';

		sqlite3_bind_text(stmtSelectTotalFileSizeAndCountForMediaLocation, 1, canonicalPath.utf8(), -1, SQLITE_TRANSIENT);

		uint filesize = 0;
		uint count = 0;

		if (sqlite3_step(stmtSelectTotalFileSizeAndCountForMediaLocation) == SQLITE_ROW)
		{
			filesize = sqlite3_column_int(stmtSelectTotalFileSizeAndCountForMediaLocation, 0);
			count = sqlite3_column_int(stmtSelectTotalFileSizeAndCountForMediaLocation, 1);
		}

		DHPRINTF("canonicalPath: %s", (const char*)canonicalPath.utf8());
		DHPRINTF("filesize: %d, count: %d", filesize, count);

		sqlite3_reset(stmtSelectTotalFileSizeAndCountForMediaLocation);

		if (filesize == localFilesTotalSize && count == localFiles.count())
		{
			DHPRINTF("using fastpath 1");
			sqlite3_bind_text(stmtSelectAllMediaIDsForMediaLocation, 1, canonicalPath.utf8(), -1, SQLITE_TRANSIENT);

			qHeapSort(localFiles);
			QExtStringList::Iterator it = localFiles.begin();

			while (sqlite3_step(stmtSelectAllMediaIDsForMediaLocation) == SQLITE_ROW)
			{
				QExtString &item = *it;
				item.setID(sqlite3_column_int(stmtSelectAllMediaIDsForMediaLocation, 0));
				DHPRINTF("item: %s, id: %d", (const char *)item.utf8(), item.id());
				++it;
			}

			sqlite3_reset(stmtSelectAllMediaIDsForMediaLocation);
		}
		// TODO: Implement and profile a second fast path attempt, that reads
		// partially available information from the database and assigns media IDs
		// based on this information...

		filelist += localFiles;
	}
}

void PlayList::addFromFileList(const QExtStringList &filelist, bool interpretPlayList, const QString &playlistDirectory)
{
	DTIMERINIT(timer);

	QExtStringList list(filelist);

	emit startingActivity(tr("Scanning for files..."));

	// expand various entries in the list if specifically wished for...
	if (interpretPlayList)
	{
		// expand dynamic playlist to real content...
		QExtStringList expandedList;
		for (QExtStringList::ConstIterator it = list.begin(); it != list.end(); ++it)
		{
			QString location(*it);

			if (location.startsWith("#") || location.startsWith(";"))
				continue;

			// WHY OH WHY is there startsWith but no endsWith in QString in Qt 2.3.x ?!?!
			if (location.right(3) == "/**" || location == "**")
			{
				// recursive scanning of location...
				location = resolveLocation(playlistDirectory, location.left(QMAX(0, location.length() - 3)));
				QExtStringList fileList;
				fillFileList(fileList, location, true);
				qHeapSort(fileList);
				expandedList += fileList;
			}
			else if (location.right(2) == "/*" || location.right(1) == "/")
			{
				// all files of the location...
				location = resolveLocation(playlistDirectory, location.left(location.findRev('/')));
				QExtStringList fileList;
				fillFileList(fileList, location, false);
				qHeapSort(fileList);
				expandedList += fileList;
			}
			else if (location.startsWith("http://") || location.startsWith("mms://"))
				expandedList.append(location);
			else
			{
				location = resolveLocation(playlistDirectory, location);
				QFileInfo fi(location);

				if (fi.exists())
				{
					if (fi.isDir())
					{
						QExtStringList fileList;
						fillFileList(fileList, location, false);
						qHeapSort(fileList);
						expandedList += fileList;
					}
					else if (qConfig.isValidExtension(fi.extension()))
						// just add the (file) location to the expanded list...
						expandedList.append(location);
				}
			}
		}

		list = expandedList;
	}

	// start reading the playlist entries...
	int numFiles = list.count();
	bool cancel = false;

	QTime time;
	time.start();

	emit updateActivityProgress(tr("Adding files..."), 0, numFiles, cancel);

	beginUpdate();

	int index = getNextIndex();
	int i = 0;

	sqlite3_stmt *vm = stmtInsertMediaIDIntoPlaylist;

	for (QExtStringList::ConstIterator it = list.begin(); it != list.end(); ++it, ++i)
	{
		const QExtString &location = *it;

		unsigned long mediaID = NULL;

		if (location.id())
		{
			DHPRINTF("ID %d set on %s", location.id(), (const char *)location.utf8());
			mediaID = location.id();
		}
		else
		{
			// skip comments
			if (location.startsWith("#") || location.startsWith(";"))
				continue;

			DHPRINTF("No ID set on %s", (const char *)location.utf8());

			QString resolvedLoc;

			if (!(isPathAbsolute(location) || location.startsWith("http://") || location.startsWith("mms://")))
				resolvedLoc = resolveLocation(playlistDirectory, location);
			else
				resolvedLoc = location;

			mediaID = mediaDatabase_->getMediaIDForLocation(MediaLocation(resolvedLoc));
		}

		sqlite3_bind_int(vm, 1, index++);
		sqlite3_bind_null(vm, 2);
		sqlite3_bind_int(vm, 3, mediaID);

		sqlite3_step(vm);

		sqlite3_reset(vm);

		if (time.elapsed() > 1000)
		{
			if (location.length() > 55)
				emit updateActivityProgress(location.left(20) + "..." + location.right(40), i, numFiles, cancel);
			else
				emit updateActivityProgress(location, i, numFiles, cancel);

			time.restart();
		}

		if (cancel)
			break;
	}

	DTIMERPRINT(timer, "playlist added to DB in");

	endUpdate();
	propagateChanges();

	emit finishedActivity(tr("Scanning for files..."), cancel);

	DMEMSTAT();
}

bool PlayList::loadFromFile(const QString &filename)
{
	DENTERMETHOD("PlayList::loadFromFile()");
	DTIMERINIT(timer);

	bool result = false;

	// Load PlayListView
	QFile file(filename);
	if (file.open(IO_ReadOnly))
	{
		QString playlistDirectory = QFileInfo(filename).dirPath(true);

		// Load PlayListView
		QTextStream stream(&file);

		QExtStringList list;
		QString line;

		while (!(line = stream.readLine()).isNull())
		{
			if (containsUTF8(line))
			{
				DHPRINTF("UTF8 detected: %s", (const char *)line.latin1());
				line = reinterpretUTF8(line);
			}
			else
				DHPRINTF("no UTF8: %s", (const char *)line.latin1());

			if (!line.startsWith("#") && !line.startsWith(";") && !line.isEmpty())
				line = resolveLocation(playlistDirectory, line);

			list.append(line);
		}

		// Are we dealing with a dynamic playlist here?
		// If so, we need to expand various entries in that list...
		bool interpretPlayList = filename.lower().right(12) == ".dynplaylist";
		if (interpretPlayList)
		{
			type_ = Dynamic;
			dynPlayList_ = list;
		}
		else
			type_ = Static;

		addFromFileList(list, interpretPlayList, playlistDirectory);

		file.close();

		result = true;
	}

	DTIMERPRINT(timer, "playlist read in");
	DEXITMETHOD("PlayList::loadFromFile()");

	return result;
}

const QString PlayList::saveToFile()
{
	if (playListFileName_)
		return exportToFile(playListFileName_);
	else
		return QString::null;
}

const QString PlayList::saveToFile(const QString &filename)
{
	playListFileName_ = exportToFile(filename);
	return playListFileName_;
}

const QString PlayList::exportToFile(const QString &filename)
{
	QString fullFilename = filename;
	QString extension = QFileInfo(fullFilename).extension(false);

	if (isDynamic())
	{
		if (extension != "dynplaylist")
			fullFilename += ".dynplaylist";
	}
	else if (extension != "playlist")
		fullFilename += ".playlist";

	QString playListDir = QFileInfo(fullFilename).dirPath(true);

	QFile file(fullFilename);
	if (file.open(IO_WriteOnly))
	{
		QTextStream s(&file);
		s.setEncoding(QTextStream::UnicodeUTF8);

		if (isDynamic())
		{
			for (QExtStringList::Iterator it = dynPlayList_.begin(); it != dynPlayList_.end(); ++it )
				s << makeLocationRelative(playListDir, *it) << "\n";
		}
		else
		{
			QString query = "SELECT media_id FROM \"" + dbSafeString(dbTable_) + "\" ORDER BY idx, ASC;";

			sqlite3_stmt *vm;
			sqlite3_prepare_v2(mediaDatabase_->db(), query.utf8(), -1, &vm, 0);

			while (sqlite3_step(vm) == SQLITE_ROW)
			{
				Media *media = mediaDatabase_->media(sqlite3_column_int(vm, 0));
				if (media)
					s << makeLocationRelative(playListDir, media->location().toString()) << "\n";
			}

			sqlite3_finalize(vm);
		}

		// Close File
		file.close();

		return fullFilename;
	}
	else
		return QString::null;
}

const QString PlayList::loadFromDatabase(const QString &tableName)
{
	dbTable_ = tableName;


}

void PlayList::saveToDatabase()
{

}

