/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef MEDIA_H_INCLUDED
#define MEDIA_H_INCLUDED

#include <qfileinfo.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qfile.h>
#include <qdir.h>
#include <qdatetime.h>
#include <qdatastream.h>
#include <qstringlist.h>
#include <qobject.h>
#include <qmap.h>
#include <qvaluelist.h>
#include <qintcache.h>

#include "sqlite3.h"

class MediaDatabase;
class Media;
class MediaIdentifier;

class QExtString : public QString
{
public:
	QExtString(const QString &item, unsigned long id = 0) : QString(item), id_(id) {}
	QExtString(const QExtString &item) : QString(item), id_(item.id_) {}
	QExtString() : QString(), id_(0) {}
	unsigned long id() const { return id_; }
	void setID(unsigned long id) { id_ = id; }

	QExtString &operator=(const QExtString &item)
	{
		QString::operator=(item);
		id_ = item.id_;
		return *this;
	}

private:
	unsigned long id_;
};

typedef QValueList<QExtString> QExtStringList;

class MediaLocation
{
public:
	MediaLocation();
	MediaLocation(const QString& url);

	QString fileName() const;
	QString dirPath() const;
	bool isLocalFile() const;

	QString toString() const { return location_; }
	operator QString() const;

private:
	QString location_;
};

class MediaVideo
{
public:
	MediaVideo(Media *parent);

	QString codec;
	int bitrate;
	short width;
	short height;
	float FPS;
	float aspectRatio;

	bool load();
	bool save();
private:
	Media *owner;
};
typedef QIntCache<MediaVideo> MediaVideoCache;

class MediaAudio
{
public:
	MediaAudio(Media *parent);

	QString codec;
	int bitrate;
	int sampleRate;
	int channels;

	bool load();
	bool save();
private:
	Media *owner;
};
typedef QIntCache<MediaAudio> MediaAudioCache;

typedef QValueList<QString> MetadataEntry;
typedef QMap<QCString, MetadataEntry> MetadataMap;

class MediaMetadataExt
{
public:
	MediaMetadataExt(Media *parent);

	MetadataMap map;

	bool load();
	bool save();
private:
	Media *owner;
};
typedef QIntCache<MediaMetadataExt> MediaMetadataExtCache;

class MediaListener
{
public:
	virtual void mediaDestroyed(Media *) = 0;
};

class Media
{
public:
	enum MediaType {
		MEDIATYPE_UNKNOWN = 0,
		MEDIATYPE_AUDIO = 1,
		MEDIATYPE_VIDEO = 2
	};

	Media(MediaDatabase *parent);
	~Media();

	void registerListener(MediaListener *listener) { listeners_.append(listener); }
	void unregisterListener(MediaListener *listener) { listeners_.remove(listener); }

	void initializeFromLocation(const MediaLocation &location);

	bool loadByID(int mediaid);

	void saveToDB();

	MediaDatabase *owningMediaDatabase() const { return parent_; }

	bool isAudio() const { return mediaType_ == MEDIATYPE_AUDIO; }
	bool isVideo() const { return mediaType_ == MEDIATYPE_VIDEO; }
	bool isSupported() const { return mediaType_ != MEDIATYPE_UNKNOWN; }

	// get method
	unsigned long length() const { return length_; }

	const MediaLocation &location() const { return location_; }
	const QDateTime &lastModified() const { return lastModified_; }
	uint fileSize() const { return fileSize_; }

	MediaAudio *mediaAudio(bool autoNew = true) const;
	MediaVideo *mediaVideo(bool autoNew = true) const;

	short width()	{ return mediaVideo()->width; }
	short height()	{ return mediaVideo()->height; }
	int audioRate() { return mediaAudio()->sampleRate; }

	// Metadata
	QString metadata(const QCString& key, int pos = -1) const;
	QStringList metadataStringList() const;

	inline unsigned long mediaID() const { return mediaid_; };

	inline const QString &trackNo()  const { return trackNo_; };
	inline const QString &title()	  const { return title_; };
	inline const QString &artist()   const { return artist_; };
	inline const QString &album()    const { return album_; };
	inline const QString &playTime() const { return playTime_; };

	bool errorDetected() const { return !lastError_.isEmpty(); }
	const QString &lastError() const { return lastError_; }
	void setLastError(const QString &errorMessage) { lastError_ = errorMessage; }

	void dumpInfo() const;

private:
	friend class PlayListView;
	friend class MediaDatabase;
	friend class MediaAudio;
	friend class MediaVideo;
	friend class MediaMetadataExt;
	friend class MediaIdentifier;

	MediaDatabase *parent_;
	unsigned long mediaid_;

	MediaType mediaType_;		// media type of file
	uint locationHash_;
	MediaLocation location_;
	uint fileSize_;
	QDateTime lastModified_;	// last modified time (local file only)

	QString trackNo_;
	QString title_;
	QString artist_;
	QString album_;
	QString playTime_;
	QString genre_;
	unsigned long length_;
	int year_;
	int compilation_;

	QString lastError_;

	QList<MediaListener> listeners_;

	bool loadFromDBVM(sqlite3_stmt *vm);

	void identify(const MediaLocation &location);

	MediaMetadataExt *mediaMetadataExt(bool autoNew = true) const;

	bool saveMediaFileInfo();
	bool saveMediaMetadata();

	bool loadMedia();

	void updateDisplayInformation(const MediaLocation &location);
	void updatePlaytime();
};

#endif // MEDIA_H_INCLUDED
