#include <math.h>
#include <qapplication.h>
#include <qsplitter.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qpainter.h>
#include <qhbox.h>
#include <qprogressdialog.h>

#ifdef QTOPIA
#include <qpe/qpemenubar.h>
#else
#include <qmenubar.h>
// for Qtopia we are using the QPE variants for toolbar and menubar
// to enable special features like the menu key etc.
// Define aliases to avoid ifdefs...
#define QPEMenuBar QMenuBar
#define QPEToolBar QToolBar
#endif

#include "coverartdownloader.h"
#include "reflectionimageflow.h"
#include "coverartprovider.h"
#include "coverartflow.h"
#include "debug.h"

#define cCaptionHeight 40
#define cSearchGrabberCount 4
#define cDownloadGrabberCount 2


static CoverArtDownloaderAddOn addOn;


/* DownloadedCoverArtImageFlow */

DownloadedCoverArtImageFlow::DownloadedCoverArtImageFlow(LayeredPaintBox *parent)
	:	CustomPerspectiveImageFlow(parent),
		albumListViewItem_(NULL)
{
	images_.setAutoDelete(true);

	setTransitionTime(300);
	setConstantVerticalOffset(-cCaptionHeight);
	setSlideSize(QSize(300, 300));
}

QImage DownloadedCoverArtImageFlow::getImage(int index)
{
	if (index >= images_.count())
		return NULL;

	return *images_.at(index);
}

void DownloadedCoverArtImageFlow::setAlbumListViewItem(AlbumListViewItem *albumListViewItem)
{
	albumListViewItem_ = albumListViewItem;

    setItemCount(albumListViewItem_->coverArtSearchResults.count());

    if (albumListViewItem_->selectedResultIndex == -1)
    	showIndex(0);
    else
    	showIndex(albumListViewItem_->selectedResultIndex);
}

void DownloadedCoverArtImageFlow::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	CustomPerspectiveImageFlow::paintOverlay(p, pixmap);

	p.setPen(QColor(255, 255, 255));
	p.setBrush(NoBrush);

	QRect rect;
	int itemIndex = floor(currentValue());

	if (itemIndex >= 0 && itemIndex < images_.count())
	{
		CoverArtSearchResult *item = albumListViewItem_->coverArtSearchResults.at(itemIndex);

		int boxHeight = cCaptionHeight;
#ifdef QTOPIA
		int fontHeight = 15; // TODO: FIXME!
#else
		int fontHeight = p.fontInfo().pixelSize();
#endif

		QString resolution;
		QImage *image = images_.at(itemIndex);

		if (image->isNull())
			resolution = "Not loaded yet.";
		else
			resolution = QString::number(image->width()) + " x " + QString::number(image->height()) + " Pixel";

		rect = QRect(0, height() - boxHeight, width(), boxHeight);
		p.drawText(rect, Qt::AlignCenter, item->imageURL + "\n" + resolution);
	}

	if (!albumListViewItem_)
	{
		rect = QRect(0, 0, width(), height());
		p.drawText(rect, Qt::AlignCenter, tr("Please select an entry in the list above to review the search results."));
	}
	else if (images_.count() == 0)
	{
		rect = QRect(0, 0, width(), height());
		//p.drawText(rect, Qt::AlignCenter, tr("No albums were found that are missing cover art images."));
		p.drawText(rect, Qt::AlignCenter, tr("No cover art images were found for the selected album."));
	}
}

void DownloadedCoverArtImageFlow::transitionFinished()
{
	DENTERMETHOD("DownloadedCoverArtImageFlow::transitionFinished()");

	CustomPerspectiveImageFlow::transitionFinished();

	int itemIndex = floor(currentValue());
	CoverArtSearchResult *item = albumListViewItem_->coverArtSearchResults.at(itemIndex);
	albumListViewItem_->selectedResultIndex = itemIndex;

	albumListViewItem_->setText(3, QString::number(itemIndex + 1));

	DEXITMETHOD("DownloadedCoverArtImageFlow::transitionFinished()");
}

/* HttpGrabber */

HttpGrabber::HttpGrabber(QObject *parent, const char *name)
	:	QObject(parent, name),
		http_(parent),
		url_(""),
		userData_(NULL),
		timeout_(5000),
		busy_(false),
		downloadRes_(0),
		lastBytesAvailable_(0)
{
	connect(&http_, SIGNAL(requestFinished(int, bool)), this, SLOT(httpRequestFinished(int, bool)));
	connect(&timer_, SIGNAL(timeout()), this, SLOT(timerTimeout()));
}

HttpGrabber::~HttpGrabber()
{
	abort();
}

bool HttpGrabber::download(const QString &url, void *userData)
{
	DENTERMETHOD("Grabber(%p)::download(%s, %p)", &http_, (const char *)url.utf8(), userData);

	if (http_.hasPendingRequests())
		http_.abort();

	timer_.stop();
	lastBytesAvailable_ = 0;
	timer_.start(timeout_, false);

	busy_ = true;
	url_ = url;
	userData_ = userData;

    QString proxyHost = QUrl(url).host();
   	int proxyPort = 80;
   	getHTTPProxySettings(proxyHost, proxyPort);
    http_.setHost(proxyHost, proxyPort);

	downloadRes_ = http_.get(url);

	DPRINTF("downloadRes_: %d", downloadRes_);

	DEXITMETHOD("Grabber(%p)::download(%s, %p)", &http_, (const char *)url.utf8(), userData);
}

void HttpGrabber::abort()
{
	downloadRes_ = 0;
	timer_.stop();
	http_.abort();
	busy_ = false;
}

void HttpGrabber::timerTimeout()
{
	DENTERMETHOD("Grabber(%p)::timerTimeout()", &http_);

	uint bytesAvailable = http_.bytesAvailable();

	if (bytesAvailable == lastBytesAvailable_)
	{
		abort();
		emit finished(this, url_, userData_, true, true);
	}
	else
		lastBytesAvailable_ = bytesAvailable;

	DEXITMETHOD("Grabber(%p)::timerTimeout()", &http_);
}

void HttpGrabber::httpRequestFinished(int res, bool error)
{
	DENTERMETHOD("Grabber(%p)::downloadHttpRequestFinished(%d, %s)", &http_, res, error ? "true" : "false");

	if (res == downloadRes_)
	{
		timer_.stop();
		busy_ = false;

		if (!error)
		{
			QByteArray data = http_.readAll();
			emit dataAvailable(this, url_, userData_, &data);
			emit finished(this, url_, userData_, false, false);
		}
		else
			emit finished(this, url_, userData_, true, false);
	}

    DEXITMETHOD("Grabber(%p)::downloadHttpRequestFinished(%d, %s)", &http_, res, error ? "true" : "false");
}

/* CoverArtDownloader */

CoverArtDownloader::CoverArtDownloader(PlayListView *playListView, SkinManager* skinManager, QWidget *parent, const char *name, bool modal)
	:	QDialog(parent, name, modal,
#ifdef QTOPIA
			0),
#else
			WType_TopLevel | WStyle_Customize | WStyle_NormalBorder | WStyle_Title | WStyle_SysMenu | WStyle_MinMax | WDestructiveClose),
#endif
		playListView_(playListView),
		skinManager_(skinManager),
		searchingOnline_(false),
		downloadingImageFiles_(false),
		currentAlbumIndex_(-1),
		currentResultIndex_(-1),
		coverArtSearchResultsToDownload_(NULL)
{
#ifdef QTOPIA
	setWState(WState_Reserved1);
#endif
	setSizeGripEnabled(true);

	setCaption(tr("Quasar Cover Art Downloader"));

	QVBoxLayout *mainLayout = new QVBoxLayout(this);
	QVBoxLayout *contentLayout = new QVBoxLayout(this, 8);

	splitter_ = new QSplitter(this);
	splitter_->setOrientation(Qt::Vertical);

	listView_ = new QListView(splitter_);

	listView_->addColumn(tr("Artist"));
	listView_->addColumn(tr("Album"));
	listView_->addColumn(tr("Results"));
	listView_->addColumn(tr("Download Image #"));
	listView_->addColumn(tr("Status"));

	connect(listView_, SIGNAL(selectionChanged(QListViewItem *)), this, SLOT(listViewSelectionChanged(QListViewItem *)));

	imageFlowViewBox_ = new LayeredPaintBox(splitter_);
	connect(imageFlowViewBox_, SIGNAL(rearrangeLayers()), this, SLOT(imageFlowViewBoxRearrangeLayers()));

	imageFlowView_ = new DownloadedCoverArtImageFlow(imageFlowViewBox_);
	flowScrollBar_ = new SkinFlowScrollBar(Skin::FlowScrollBar, Skin::FlowScrollBarSlider, skinManager_, imageFlowView_, imageFlowViewBox_);

	listView_->installEventFilter(this); // Install event filter to intercept left and right key press events...

	contentLayout->addWidget(splitter_);

	QValueList<int> sizes;
	sizes.append(50);
	sizes.append(50);
	splitter_->setSizes(sizes);

	QHBox *twoButtonsBox = new QHBox(this);
	twoButtonsBox->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));

	startDownloadButton_ = new QPushButton(tr("Start Download"), twoButtonsBox);
	connect(startDownloadButton_, SIGNAL(clicked()), this, SLOT(accept()));
	startDownloadButton_->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));

	cancelButton_ = new QPushButton(tr("Close"), twoButtonsBox);
	connect(cancelButton_, SIGNAL(clicked()), this, SLOT(reject()));
	cancelButton_->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));

	progressLabel_ = new QLabel(this);
	progressLabel_->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));

	progress_ = new QProgressBar(this);

	QHBoxLayout *buttonsLayout = new QHBoxLayout(this, 8);
	buttonsLayout->setDirection(QBoxLayout::RightToLeft);
	buttonsLayout->setAlignment(AlignRight);
	buttonsLayout->addWidget(twoButtonsBox, 1);
	buttonsLayout->addWidget(progress_, 1);
	buttonsLayout->addWidget(progressLabel_, 1);

    QPopupMenu *tools = new QPopupMenu(this);
    tools->insertItem(tr("&Refresh list..."), this, SLOT(scanForMissingCoverArt()));
    tools->insertItem(tr("&Start Download"), this, SLOT(accept()));
    tools->insertSeparator();
    tools->insertItem(tr("&Close"), this, SLOT(reject()));

    QPEMenuBar *menu = new QPEMenuBar(this);
	menu->setMargin(0);
    menu->insertItem("&Tools", tools);

    mainLayout->addWidget(menu);
	mainLayout->addLayout(contentLayout);
	mainLayout->addLayout(buttonsLayout);

	searchGrabbers_.setAutoDelete(true);
	for (int i = 0; i < cSearchGrabberCount; ++i)
	{
		HttpGrabber *grabber = new HttpGrabber(this);
		grabber->setTimeout(30000);
		connect(grabber, SIGNAL(dataAvailable(HttpGrabber *, const QString &, void *, QByteArray *)), this, SLOT(searchDataAvailable(HttpGrabber *, const QString &, void *, QByteArray *)));
		connect(grabber, SIGNAL(finished(HttpGrabber *, const QString &, void *, bool, bool)), this, SLOT(searchFinished(HttpGrabber *, const QString &, void *, bool, bool)));
		searchGrabbers_.append(grabber);
	}

	downloadGrabbers_.setAutoDelete(true);
	for (int i = 0; i < cDownloadGrabberCount; ++i)
	{
		HttpGrabber *grabber = new HttpGrabber(this);
		connect(grabber, SIGNAL(dataAvailable(HttpGrabber *, const QString &, void *, QByteArray *)), this, SLOT(downloadDataAvailable(HttpGrabber *, const QString &, void *, QByteArray *)));
		connect(grabber, SIGNAL(finished(HttpGrabber *, const QString &, void *, bool, bool)), this, SLOT(downloadFinished(HttpGrabber *, const QString &, void *, bool, bool)));
		downloadGrabbers_.append(grabber);
	}

	resultImageDataCache_.setAutoDelete(true);
	resultImageDataCache_.setMaxCost(8 * 1024 * 1024); // 8 MB downloaded image cache...

	scanForMissingCoverArt();
}

void CoverArtDownloader::imageFlowViewBoxRearrangeLayers()
{
	imageFlowView_->setGeometry(QRect(0, 15, imageFlowViewBox_->width(), imageFlowViewBox_->height() - 15 - 20));
	flowScrollBar_->setGeometry(QRect(30, imageFlowViewBox_->height() - 30, imageFlowViewBox_->width() - 60, 20));
}

void CoverArtDownloader::closeEvent(QCloseEvent *e)
{
	emit closing();
	e->accept();
}

void CoverArtDownloader::done(int r)
{
	emit closing();
	QDialog::done(r);
}

void CoverArtDownloader::showEvent(QShowEvent *)
{
#ifdef QTOPIA
	showMaximized();
#endif;
}

bool CoverArtDownloader::eventFilter(QObject *o, QEvent *e)
{
	if (e->type() == QEvent::KeyPress)
	{
		QKeyEvent *ke = static_cast<QKeyEvent *>(e);

		if (ke->key() == Qt::Key_Left)
		{
			imageFlowView_->showPreviousItem();
			ke->accept();
			return true;
		}
		else if (ke->key() == Qt::Key_Right)
		{
			imageFlowView_->showNextItem();
			ke->accept();
			return true;
		}
		else
			return QDialog::eventFilter(o, e);
	}
	else
		return QDialog::eventFilter(o, e);
}

void CoverArtDownloader::abortOnlineSearching()
{
	searchGrabbers_.abortAll();
    searchingOnline_ = false;
    currentAlbumIndex_ = -1;
}

void CoverArtDownloader::accept()
{
    abortOnlineSearching();

    splitter_->setEnabled(false);
    imageFlowViewBox_->hide();
    startDownloadButton_->setEnabled(false);

	downloadList_.clear();

	AlbumListViewItem *alvi;
    for (alvi = albumListItems_.first(); alvi != 0; alvi = albumListItems_.next())
    	if (alvi->isOn() && alvi->coverArtSearchResults.count() > 0 && alvi->selectedResultIndex != -1)
    		downloadList_.append(alvi->coverArtSearchResults.at(alvi->selectedResultIndex));

    downloadGrabbers_.abortAll();
    currentResultIndex_ = -1;
    downloadingImageFiles_ = true;
	currentAlbumListViewItem_ = NULL;
	coverArtSearchResultsToDownload_ = &downloadList_;

	progress_->setTotalSteps(downloadList_.count());
	progress_->show();

    for (HttpGrabber *g = downloadGrabbers_.first(); g != 0; g = downloadGrabbers_.next())
    	downloadNextResultImageData();
}

void CoverArtDownloader::reject()
{
	DENTERMETHOD("CoverArtDownloader::reject()");
	searchGrabbers_.abortAll();
	downloadGrabbers_.abortAll();
    QDialog::reject();
    DEXITMETHOD("CoverArtDownloader::reject()");
}

void CoverArtDownloader::scanForMissingCoverArt()
{
	DENTERMETHOD("CoverArtDownloader::scanForMissingCoverArt()");
	DTIMERINIT(timer);

	QProgressDialog progress(tr("Please wait..."), tr("Cancel"), 0, NULL, "progress", true);

	progress.setCaption(tr("Scanning for missing cover art image files..."));

	progress.setMinimumDuration(0);
	progress.setFixedWidth(QMIN(qApp->desktop()->width() * 0.9, 640));
	progress.setAutoClose(false);
	progress.setAutoReset(false);
	progress.show();
	qApp->processEvents();

	QString query =
		"SELECT artist, album, media_location.location || media.filename FROM " + playListView_->playListSource() + " AS pl "
		"LEFT JOIN media ON pl.media_id = media.media_id "
		"LEFT JOIN media_location ON media.location_id = media_location.location_id "
		"GROUP BY album ORDER BY artist, track ASC;";

	sqlite3_stmt *vm;
	sqlite3_prepare_v2(playListView_->mediaDatabase()->db(), query.utf8(), -1, &vm, 0);

	albumListItems_.clear();
	listView_->clear();

	QStringList artists;
	QStringList albums;
	QStringList filenames;

	while (sqlite3_step(vm) == SQLITE_ROW)
	{
		artists.append(QString::fromUtf8((const char *)sqlite3_column_text(vm, 0)));
		albums.append(QString::fromUtf8((const char *)sqlite3_column_text(vm, 1)));
		filenames.append(QString::fromUtf8((const char *)sqlite3_column_text(vm, 2)));
	}

	sqlite3_finalize(vm);

	progress.setTotalSteps(albums.count());

	QTime time;
	time.start();

	for (int i = 0; i < albums.count(); ++i)
	{
		QString artist = artists[i];
		QString album = albums[i];
		QStringList imageFilenames = gCoverArtProvider.findCoverArtImageFilenames(artist, album, filenames[i]);

		if (album.length() > 0 && imageFilenames.count() == 0)
		{
			AlbumListViewItem *albumListViewItem = new AlbumListViewItem(listView_);
			albumListViewItem->setText(0, artist);
			albumListViewItem->setText(1, album);
			albumListViewItem->setText(2, "0");
			albumListViewItem->setText(3, "");
			albumListViewItem->setEnabled(false);
			albumListViewItem->setOn(false);
			albumListItems_.append(albumListViewItem);
		}

		if (time.elapsed() > 1000)
		{
			progress.setProgress(i);
			progress.setLabelText(artist + " - " + album);
			qApp->processEvents();
			time.restart();
		}

		if (progress.wasCancelled())
			break;
	}

	DTIMERPRINT(timer, "Getting albums from DB...");

	DPRINTF("Album count: %d", albumListItems_.count());

	currentAlbumIndex_ = -1;
	searchingOnline_ = true;

    for (HttpGrabber *g = searchGrabbers_.first(); g != 0; g = searchGrabbers_.next())
    	requestNextCoverArt();

	DEXITMETHOD("CoverArtDownloader::scanForMissingCoverArt()");
}

void CoverArtDownloader::updateProgressBar()
{
	if (!searchingOnline_ && !downloadingImageFiles_)
	{
		progress_->hide();
		progressLabel_->hide();
	}
	else if (searchingOnline_)
	{
		progressLabel_->setText(tr("Searching online..."));
		progress_->setTotalSteps(albumListItems_.count());
		progress_->setProgress(currentAlbumIndex_);

		if (!progress_->isVisible())
			progress_->show();

		if (!progressLabel_->isVisible())
			progressLabel_->show();
	}
	else if (downloadingImageFiles_)
	{
		progressLabel_->setText(tr("Loading cover images..."));
		progress_->setTotalSteps(coverArtSearchResultsToDownload_->count());
		progress_->setProgress(currentResultIndex_);

		if (!progress_->isVisible())
			progress_->show();

		if (!progressLabel_->isVisible())
			progressLabel_->show();
	}
}

void CoverArtDownloader::requestNextCoverArt()
{
	DENTERMETHOD("CoverArtDownloader::requestNextCoverArt()");

	++currentAlbumIndex_;
	updateProgressBar();

	if (searchingOnline_ && currentAlbumIndex_ < albumListItems_.count())
	{
		AlbumListViewItem *albumListViewItem = albumListItems_.at(currentAlbumIndex_);

		QString artist = CoverArtProvider::simplifyString(albumListViewItem->artist());
		QUrl::encode(artist);

		QString album = CoverArtProvider::simplifyString(albumListViewItem->album());
		QUrl::encode(album);

	    QChar connectorChar = (addOn.coverArtWebServiceURL().contains('?') > 0 ? '&' : '?');
	    QString requestURL = addOn.coverArtWebServiceURL() + connectorChar + "artist=" + artist + "&album=" + album;

	    albumListViewItem->setText(4, "Searching...");

		HttpGrabber *grabber = searchGrabbers_.getIdleGrabber();
	    grabber->download(requestURL, albumListViewItem);
		DPRINTF("Starting grabber %p for %s - %s, %s", grabber, (const char *)albumListViewItem->text(0).utf8(), (const char *)albumListViewItem->text(1).utf8(), (const char *)requestURL.utf8());
	}
	else
	{
		DPRINTF("Nothing to be done...");

		if (searchGrabbers_.busyGrabbersCount() == 0)
		{
			DPRINTF("Last grabber, finishing...");
			// FINISHED!
			currentAlbumIndex_ = -1;
			searchingOnline_ = false;
			updateProgressBar();
		}
	}

	DEXITMETHOD("CoverArtDownloader::requestNextCoverArt()");
}


void CoverArtDownloader::searchFinished(HttpGrabber *grabber, const QString &url, void *userData, bool error, bool timedOut)
{
	DENTERMETHOD("CoverArtDownloader::searchFinished(%p, %s, %p, %d, %d)", grabber, (const char *)url.utf8(), userData, error, timedOut);

	AlbumListViewItem *albumListViewItem = static_cast<AlbumListViewItem *>(userData);
	if (!error)
		albumListViewItem->setText(4, "");
	else
	{
		if (timedOut)
			albumListViewItem->setText(4, "Timed out");
		else
			albumListViewItem->setText(4, "Error");
	}

	requestNextCoverArt();
	DEXITMETHOD("CoverArtDownloader::searchFinished(%p, %s, %p, %d, %d)", grabber, (const char *)url.utf8(), userData, error, timedOut);
}

void CoverArtDownloader::searchDataAvailable(HttpGrabber *grabber, const QString &url, void *userData, QByteArray *data)
{
	DENTERMETHOD("CoverArtDownloader::searchDataAvailable(%p, %s, %p, %p)", grabber, (const char *)url.utf8(), userData, data);

	AlbumListViewItem *albumListViewItem = static_cast<AlbumListViewItem *>(userData);
	albumListViewItem->coverArtSearchResults.clear();

	QString response(*data);
	QStringList responseLines = QStringList::split("\n", response);

	for (QStringList::Iterator line = responseLines.begin(); line != responseLines.end(); ++line)
	{
		QStringList lineItems = QStringList::split(":|:", (*line), true);

		DPRINTF("lineItems.count() = %d", lineItems.count());

		if (lineItems.count() == 4)
		{
			CoverArtSearchResult *item = new CoverArtSearchResult();
			item->title = lineItems[0];
			item->detailsURL = lineItems[1];
			item->thumbnailImageURL = lineItems[2];
			item->imageURL = lineItems[3];

			item->owner = albumListViewItem;

			albumListViewItem->coverArtSearchResults.append(item);
		}

		DPRINTF("%s \n", (*line).latin1());
	}

	if (albumListViewItem->coverArtSearchResults.count() == 0)
	{
		albumListViewItem->setText(2, "0");
		albumListViewItem->setText(3, "");
		albumListViewItem->setEnabled(false);
		albumListViewItem->setOn(false);
	}
	else
	{
		albumListViewItem->setText(2, QString::number(albumListViewItem->coverArtSearchResults.count()));
		albumListViewItem->setText(3, "1");
		albumListViewItem->setEnabled(true);
		albumListViewItem->setOn(true);
	}

	if (responseLines.count() == 0)
		DPRINTF("No results.");

	DEXITMETHOD("CoverArtDownloader::searchDataAvailable(%p, %s, %p, %p)", grabber, (const char *)url.utf8(), userData, data);
}

void CoverArtDownloader::listViewSelectionChanged(QListViewItem *lvi)
{
	loadFlowView(static_cast<AlbumListViewItem *>(lvi));
}

void CoverArtDownloader::loadFlowView(AlbumListViewItem *alvi)
{
	imageFlowView_->images().clear();
	downloadGrabbers_.abortAll();

	CoverArtSearchResult *result;
    for (result = alvi->coverArtSearchResults.first(); result != 0; result = alvi->coverArtSearchResults.next())
    {
    	QImage *image = new QImage();
    	imageFlowView_->images().append(image);
    }

    imageFlowView_->setAlbumListViewItem(alvi);

    currentResultIndex_ = -1;
    downloadingImageFiles_ = true;
	currentAlbumListViewItem_ = alvi;
	coverArtSearchResultsToDownload_ = &alvi->coverArtSearchResults;

    for (HttpGrabber *g = downloadGrabbers_.first(); g != 0; g = downloadGrabbers_.next())
    	downloadNextResultImageData();
}

void CoverArtDownloader::downloadNextResultImageData()
{
	DENTERMETHOD("CoverArtDownloader::downloadNextResultImageData()");

	++currentResultIndex_;
	updateProgressBar();

	if (downloadingImageFiles_ && coverArtSearchResultsToDownload_ && currentResultIndex_ < coverArtSearchResultsToDownload_->count())
	{
		CoverArtSearchResult *coverArtSearchResult = coverArtSearchResultsToDownload_->at(currentResultIndex_);
		QByteArray *cachedImageData = resultImageDataCache_.find((int)coverArtSearchResult, true);

		if (cachedImageData)
		{
			handleCoverArtSearchResultImageData(coverArtSearchResult, cachedImageData);
			downloadNextResultImageData();
		}
		else
		{
			HttpGrabber *grabber = downloadGrabbers_.getIdleGrabber(); // there has to be at least one idle grabber in our case...
		    grabber->download(coverArtSearchResult->imageURL, coverArtSearchResult);

			DPRINTF("Starting grabber %p for download of %s, %s - %s",
				grabber, (const char *)coverArtSearchResult->imageURL.utf8(),
				(const char *)coverArtSearchResult->owner->artist().utf8(), (const char *)coverArtSearchResult->owner->album().utf8());
		}
	}
	else
	{
		DPRINTF("Nothing to be done...");

		if (downloadGrabbers_.busyGrabbersCount() == 0)
		{
			DPRINTF("Last grabber, finishing...");

			currentResultIndex_ = -1;
			downloadingImageFiles_ = false;
			coverArtSearchResultsToDownload_ = NULL;

			updateProgressBar();

			if (!currentAlbumListViewItem_)
				QDialog::accept();
		}
	}

	DEXITMETHOD("CoverArtDownloader::downloadNextResultImageData()");
}

void CoverArtDownloader::handleCoverArtSearchResultImageData(CoverArtSearchResult *coverArtSearchResult, QByteArray *imageData)
{
	if (currentAlbumListViewItem_)
	{
		if (coverArtSearchResult->owner == currentAlbumListViewItem_)
		{
			int index = currentAlbumListViewItem_->coverArtSearchResults.findRef(coverArtSearchResult);

			// prepare image
			QImage *image = imageFlowView_->images().at(index);
			image->loadFromData(*imageData);
			imageFlowView_->invalidate();
		}
	}
	else
	{
		if (!QDir(qConfig.coverArtDirectory).exists())
			QDir().mkdir(qConfig.coverArtDirectory, true);

		QString coverArtDirectory = qConfig.coverArtDirectory + QDir::separator() + "00DownloadedCoverArt";

		if (!QDir(coverArtDirectory).exists())
			QDir().mkdir(coverArtDirectory, true);

		QString filename = coverArtDirectory + QDir::separator() +
			CoverArtProvider::simplifyString(coverArtSearchResult->owner->artist()) + " - " +
			CoverArtProvider::simplifyString(coverArtSearchResult->owner->album()) + ".jpg";

		QFile f(filename);
		f.open(IO_WriteOnly);
		f.writeBlock(*imageData);
		f.close();
	}
}

void CoverArtDownloader::downloadFinished(HttpGrabber *grabber, const QString &url, void *userData, bool error, bool timedOut)
{
	DENTERMETHOD("CoverArtDownloader::downloadFinished(%p, %s, %p, %d, %d)", grabber, (const char *)url.utf8(), userData, error, timedOut);
	CoverArtSearchResult *coverArtSearchResult = static_cast<CoverArtSearchResult *>(userData);
	if (!error)
		coverArtSearchResult->owner->setText(4, "OK");
	else
	{
		if (timedOut)
			coverArtSearchResult->owner->setText(4, "Timed out");
		else
			coverArtSearchResult->owner->setText(4, "Error");
	}
	coverArtSearchResult->owner->listView()->ensureItemVisible(coverArtSearchResult->owner);

	downloadNextResultImageData();
	DEXITMETHOD("CoverArtDownloader::downloadFinished(%p, %s, %p, %d, %d)", grabber, (const char *)url.utf8(), userData, error, timedOut);
}

void CoverArtDownloader::downloadDataAvailable(HttpGrabber *grabber, const QString &url, void *userData, QByteArray *data)
{
	DENTERMETHOD("CoverArtDownloader::downloadDataAvailable(%p, %s, %p, %p)", grabber, (const char *)url.utf8(), userData, data);
	CoverArtSearchResult *coverArtSearchResult = static_cast<CoverArtSearchResult *>(userData);
	QByteArray *result = new QByteArray(*data);
	resultImageDataCache_.insert((int)coverArtSearchResult, result, result->size());
	handleCoverArtSearchResultImageData(coverArtSearchResult, result);
	DEXITMETHOD("CoverArtDownloader::downloadDataAvailable(%p, %s, %p, %p)", grabber, (const char *)url.utf8(), userData, data);
}
