/* -*- mode: c++; tab-width: 4; c-basic-offset: 4 -*- */
/*
 * Copyright (C) 2005 Atmark <atmarkat _AT_ msn _DOT_ com>
 *                    AGAWA Koji <i _AT_ atty _DOT_ jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qpainter.h>
#include "skinnedbutton.h"
#include "skinpanel.h"

/*!
  \class SkinnedButton skinnedbutton.h
  \brief SkinnedButtonクラスはウィジェットの描画にスキンを使うQButtonです。

  \sa Skin
*/

/*!
  スキンパーツの種類を \a type 、初期スキンを \a skin および
  親ウィジェット \a parent と名前 \name を設定してボタンを作成します。
*/
SkinnedButton::SkinnedButton(Skin::PartsType type, const SkinManager *skinmgr, QWidget *parent, const char *name)
	: QButton(parent, name),
	  type_(type)
{
	setFocusPolicy(NoFocus);
	setBackgroundMode(NoBackground);
	setWFlags(Qt::WNorthWestGravity | Qt::WRepaintNoErase | Qt::WResizeNoErase);
	
	setSkin(skinmgr->skin());
	connect(skinmgr, SIGNAL(skinChanged(const Skin *)), this, SLOT(setSkin(const Skin *)));
	
	if (parent && parent->inherits("SkinPanel"))
		connect(((SkinPanel *)parent), SIGNAL(resized()), this, SLOT(skinPanelResized()));
}

void SkinnedButton::setToggleButton(bool value)
{
	QButton::setToggleButton(value);
}

void SkinnedButton::setOn(bool enable)
{
	QButton::setOn(enable);	
}

/*!
  スキンを \a skin に変更します。

  ボタンの位置とサイズ、および画像は新しいスキンの情報をもとに再設定されます。

  \sa OperationPanel::setSkin()

  \todo 正常に再描画されるかテストする。
*/
void SkinnedButton::setSkin(const Skin *skin)
{
	parts_ = skin->lookup(type_);
	
	if (parentWidget() && parentWidget()->inherits("SkinPanel"))
		skinPanelResized();
	else
		setGeometry(parts_->geometry());
	
	repaint();
}

void SkinnedButton::skinPanelResized()
{
	SkinPanel *parent = (SkinPanel *)parentWidget();
	setGeometry(parent->calculateGeometry(parts_));
}

/*!
  スキンを描画するために再実装されたメソッドです。
*/
void SkinnedButton::drawButton(QPainter *p)
{
	const QWidget *parent = parentWidget();

	if (parent && parent->inherits("SkinPanel") && ((SkinPanel *)parent)->backgroundImage())
	{
		const QRect &rect = parts_->geometry();
		p->drawPixmap(0, 0, *((SkinPanel *)parent)->backgroundImage(), rect.x(), rect.y(), rect.width(), rect.height());
	}

	if (parts_->imageCount() > 2 && !isEnabled())
		p->drawPixmap(0, 0, parts_->image(), 2 * width(), 0, width());
	else if (!isDown() && !isOn())
		p->drawPixmap(0, 0, parts_->image(), 0, 0, width());
	else
		p->drawPixmap(0, 0, parts_->image(), width(), 0, width());
}

void SkinnedButton::show()
{
	QButton::show();
	repaint();	
}