#include <qvbox.h>
#include <qlayout.h>
#include <qstylesheet.h>
#include "playlist.h"
#include "imagefx.h"
#include "mediainfodialog.h"
#include "media.h"
#include "helpers.h"
#include "coverartprovider.h"

MediaInfoDialog::MediaInfoDialog(PlayList *playList, const QString &noCoverImageFilename, QWidget *parent, const char *name)
	:	QTabDialog(parent, name, true),
		playList_(playList),
		noCoverImageFilename_(noCoverImageFilename)
{
	setCaption(tr("Media File Information"));

	setupWidgets();

	if (playList_->currentItem())
		setMedia(static_cast<PlayListItem *>(playList_->currentItem())->media());
}

void MediaInfoDialog::showEvent(QShowEvent *)
{
#ifdef QTOPIA
	showMaximized();
#endif;
}

void MediaInfoDialog::setupWidgets()
{
	QVBox *overview = new QVBox(this);

	mimeSourceFactory_ = new QMimeSourceFactory();

	overviewText_ = new QTextView(overview);
	overviewText_->setMimeSourceFactory(mimeSourceFactory_);

	QStyleSheetItem *item = overviewText_->styleSheet()->item("table");
	item->setFontSize(10);

	item = overviewText_->styleSheet()->item("h3");
	item->setFontSize(13);

	addTab(overview, tr("Overview"));
}

void MediaInfoDialog::setMedia(Media *media)
{
	QImage image;

	if (gCoverArtProvider.loadCoverDirectly(image, media->artist(), media->album(), media->location().toString()) ||
		image.load(noCoverImageFilename_))
	{
		QSize outputSize = ImageFX::optimalFitSize(image.size(), QSize(80, 80));
		QImage scaledImage = image.smoothScale(outputSize.height(), outputSize.width());
		mimeSourceFactory_->setImage("coverart", scaledImage);
	}

	QString genericInfo =
		"<table>"
		"<tr>"
		"  <td><img src=\"coverart\"/></td>"
		"  <td><br/>"
		"  " + media->title() + " (" + formatPlayTime(media->length()) + ")<br/>"
		"  " + media->artist() + "<br/>"
		"  " + media->album() +
		"  </td>"
		"</tr>"
		"</table>";

	QString audioInfo = "";

	if (media->isAudio() || media->isVideo())
	{
		audioInfo =
			"<tr>"
			"<td colspan=\"4\">"
			"<h3>Audio:</h3>"
			"</td>"
			"</tr>"
			"<tr>"
			"  <td align=\"right\">"
			"    <strong>Codec:</strong>"
			"  </td>"
			"  <td>"
			"  " + (media->mediaAudio()->codec.isEmpty() ? QString("Unknown") : media->mediaAudio()->codec) +
			"  </td>"
			"  <td align=\"right\">"
			"    <strong>Bitrate:</strong>"
			"  </td>"
			"  <td>"
			"  " + formatBitrate(media->mediaAudio()->bitrate) +
			"  </td>"
			"</tr>"
			"<tr>"
			"  <td align=\"right\">"
			"    <strong>Samplerate:</strong>"
			"  </td>"
			"  <td>"
			"  " + QString().number(media->mediaAudio()->sampleRate) + " Hz"
			"  </td>"
			"  <td align=\"right\">"
			"    <strong>Channels:</strong>"
			"  </td>"
			"  <td>"
			"  " + QString().number(media->mediaAudio()->channels) +
			"  </td>"
			"</tr>";
	}

	QString videoInfo = "";

	if (media->isVideo())
	{
		videoInfo =
			"<tr>"
			"<td colspan=\"4\">"
			"<h3>Video:</h3>"
			"</td>"
			"</tr>"
			"<tr>"
			"  <td align=\"right\">"
			"    <strong>Codec:</strong>"
			"  </td>"
			"  <td>"
			"  " + (media->mediaVideo()->codec.isEmpty() ? QString("Unknown") : media->mediaVideo()->codec) +
			"  </td>"
			"  <td align=\"right\">"
			"    <strong>Bitrate:</strong>"
			"  </td>"
			"  <td>"
			"  " + formatBitrate(media->mediaVideo()->bitrate) +
			"  </td>"
			"</tr>"
			"<tr>"
			"  <td align=\"right\">"
			"    <strong>FPS:</strong>"
			"  </td>"
			"  <td>"
			"  " + QString().number(media->mediaVideo()->FPS) +
			"  </td>"
			"  <td align=\"right\">"
			"    <strong>Resolution:</strong>"
			"  </td>"
			"  <td>"
			"  " + QString().sprintf("%d x %d", media->mediaVideo()->width, media->mediaVideo()->height) +
			"  </td>"
			"</tr>";
	}

	QString extendedInfo = "<table>" + audioInfo + videoInfo + "</table>";

	QString fileInfo =
		"<table>"
		"<tr>"
		"<td colspan=\"4\">"
		"<h3>File Info:</h3>"
		"</td>"
		"</tr>"
		"<tr>"
		"  <td align=\"right\">"
		"    <strong>Size:</strong>"
		"  </td>"
		"  <td>"
		"  " + formatFileSize(media->fileSize()) +
		"  </td>"
		"</tr>"
		"<tr>"
		"  <td align=\"right\">"
		"    <strong>Location:</strong>"
		"  </td>"
		"  <td>"
		"  " + media->location().toString() +
		"  </td>"
		"</tr>"
		"</table>";

	overviewText_->setText(
		genericInfo +
		extendedInfo +
		fileInfo
	);
}
