/* -*- mode: c++; tab-width: 4; c-basic-offset: 4 -*- */
/*
 * Copyright (C) 2006-2008	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * Copyright (C) 2005 Atmark <atmarkat _AT_ msn _DOT_ com>
 *                    AGAWA Koji <i _AT_ atty _DOT_ jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#ifdef OSX
#include <Carbon/Carbon.h>
// Get rid of DPRINTF defined in Carbon.h...
#undef DPRINTF
#endif

#include <qapplication.h>

#ifdef QTOPIA
#include <qpe/qpemenubar.h>
#include <qpe/qpetoolbar.h>
#include <qpe/qcopenvelope_qws.h>
#include <qpe/resource.h>
#include <qpe/config.h>
#include <qpe/global.h>
#else
#include <qmenubar.h>
#include <qtoolbar.h>
#include <qdragobject.h>
#include <resource.h>
#include <config.h>
// for Qtopia we are using the QPE variants for toolbar and menubar
// to enable special features like the menu key etc.
// Define aliases to avoid ifdefs...
#define QPEMenuBar QMenuBar
#define QPEToolBar QToolBar
#endif

#include <qmime.h>
#include <qaccel.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qlayout.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <qlistview.h>
#include <qheader.h>
#include <qlineedit.h>
#include <qcursor.h>
#include <qwidgetstack.h>
#include <qaction.h>
#include <qvbox.h>
#include <qsignalmapper.h>
#include <qtextcodec.h>
#include <qstringlist.h>
#include <qregexp.h>

#ifdef WINDOWS
#include "wincompat.h"
#endif

#include "addonmanager.h"
#include "platform/platformservices.h"
#include "textviewdialog.h"
#include "quasar.h"
#include "mplayer.h"
#include "operationpanel.h"
#include "toolpanel.h"
#include "videoplaybackwindow.h"
#include "playinfo.h"
#include "configuration.h"
#include "configurationdialog.h"
#include "media.h"
#include "qinputdialog.h"
#include "qprogressdialog.h"
#include "qsplitter.h"
#include "aboutdialog.h"
#include "action.h"
#include "system_volume.h"
#include "externalservices.h"
#ifdef QTOPIA
#include "batteryplus.h"
#endif
#include "playlist.h"
#include "playlistoverview.h"
#include "paintbox.h"
#include "coverartflow.h"
#include "scrollbar.h"
#include "coverartprovider.h"
#include "mediainfodialog.h"
#include "playlisttoolpanel.h"
#include "skin.h"
#include "keynames.h"
#include "helpers.h"
#include "debug.h"

#define MinSliderLength 22

#ifdef QTOPIA
// MOC_SKIP_BEGIN
class QuasarStyle: public QPEStyle
{
public:
	QuasarStyle(): QPEStyle() {}
	~QuasarStyle() {}
	int sliderLength () const { return MinSliderLength; };
	void scrollBarMetrics(const QScrollBar *scrollbar, int &sliderMin, int &sliderMax, int &sliderLength, int &buttonDim)
	{
		QPEStyle::scrollBarMetrics(scrollbar, sliderMin, sliderMax, sliderLength, buttonDim);
		if (sliderLength < MinSliderLength)
		{
			sliderMax = sliderMax - (MinSliderLength - sliderLength);
			sliderLength = MinSliderLength;
		}
	}
	/*
    int pixelMetric(PixelMetric metric, const QWidget *widget) const
    {
        if (metric == PM_SplitterWidth)
            return 6;
        else
            return QWindowsStyle::pixelMetric(metric, widget);
    }
    */
};
// MOC_SKIP_END

#else

#include <qwindowsstyle.h>

class QuasarStyle: public QWindowsStyle
{
public:
	QuasarStyle(): QWindowsStyle() {}
	~QuasarStyle() {}

    int pixelMetric(PixelMetric metric, const QWidget *widget) const
    {
        if (metric == PM_SplitterWidth)
            return 14;
        else if (metric == PM_ScrollBarSliderMin)
        	return 25;
        else
            return QWindowsStyle::pixelMetric(metric, widget);
    }
};
#endif

#ifdef OSX
OSErr appleEventHandler(const AppleEvent *ae, AppleEvent *, SRefCon handlerRefcon)
{
	AEDescList docs;
	if (AEGetParamDesc(ae, keyDirectObject, typeAEList, &docs) == noErr)
	{
		long cnt = 0;
		AECountItems(&docs, &cnt);
		UInt8 strBuffer[256];
		QStringList filenameList;

		for (int i = 0; i < cnt; i++)
		{
			FSRef ref;
			if (AEGetNthPtr(&docs, i + 1, typeFSRef, 0, 0, &ref, sizeof(ref), 0) != noErr)
				continue;

			if (FSRefMakePath(&ref, strBuffer, 256) == noErr)
				filenameList.append(QString::fromUtf8(reinterpret_cast<char *>(strBuffer)));
		}

		if (filenameList.count() > 0)
		{
			Quasar *quasar = (Quasar *)handlerRefcon;
			quasar->handleExternalDocuments(filenameList);
		}
	}

	return noErr;
}

#endif

Quasar::Quasar()
	:	keyEventCount_(0)
{
	DMEMSTAT();
	setCaption(APPNAME);

	// configuration
	DPRINTF("Initialize Configuration class...");
	bool isValidConfig = qConfig.isValidConfig();
	if (isValidConfig)
		qConfig.read();
	DPRINTF("Done");

#ifdef QTOPIA
	qPlatformServices.setRemoteCon();
#endif

	receivedExternalDocuments_ = false;

#ifdef OSX
    AEInstallEventHandler(kCoreEventClass, kAEOpenDocuments, appleEventHandler, (long)this, false);
#endif

	DMEMSTAT();

	// mplayer
	DPRINTF("Initialize MPlayer class...");
	mplayer_ = new MPlayer();
	connect(mplayer_, SIGNAL(eof()), this, SLOT(next()));
	connect(mplayer_, SIGNAL(quit()), this, SLOT(stop()));
	connect(mplayer_, SIGNAL(currentPlayTime(int)), this, SLOT(mplayerCurrentPlayTime(int)));
	connect(mplayer_, SIGNAL(errorMessage(QString)), this, SLOT(mplayerErrorMessage(QString)));
	DPRINTF("Done");

	osdTimer_ = new QTimer(this);
	connect(osdTimer_, SIGNAL(timeout()), this, SLOT(osdTimerTimedUpdate()));

	DMEMSTAT();

	// skin
	DPRINTF("Initialize SkinManager class...");
	skinManager_ = new SkinManager(qGlobalConfig.skinPath(), this);
	if (!skinManager_->load(qConfig.skin1)) {
		QMessageBox::warning(this, tr("Warning"), tr("Couldn't load skin."));
		if (!skinManager_->load("default")) {
			QMessageBox::warning(this, tr("Fatal Error"), tr("Couldn't load default skin."));
			exit(1);
		}
	}

	// TODO: get rid of hardcoding...
#ifdef QTOPIA
	if (qApp->desktop()->width() == 640 || qApp->desktop()->width() == 320) {
		skinManager_->setStyle(SkinManager::Landscape);
		DPRINTF("  Initial style is Landscape");
	} else {
		skinManager_->setStyle(SkinManager::Portrait);
		DPRINTF("  Initial style is Portrait");
	}
#else
	skinManager_->setStyle(SkinManager::Landscape);
#endif
	DPRINTF("Done");

	DMEMSTAT();

	centralWidget_ = new QVBox(this);

	// Toggle Widget(Playlist/PlayInfo)
	DPRINTF("Initialize widget stack class...");

	widgetStack_ = new QWidgetStack(centralWidget_);
	widgetStack_->setFrameStyle(QFrame::NoFrame);
	widgetStack_->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize playlist container...");

	playListContainer_ = new QVBox(widgetStack_);
	playListContainer_->setFrameStyle(QFrame::NoFrame);
	widgetStack_->addWidget(playListContainer_, PlayListWidget);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize Playlist tool panel class...");
	playListToolPanel_ = new PlayListToolPanel(skinManager_, playListContainer_);
	// signals are connected further down...

	DPRINTF("Done");
	DMEMSTAT();

	playlistStack_ = new QWidgetStack(playListContainer_);
	playlistStack_->setFrameStyle(QFrame::NoFrame);
	playlistStack_->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));

	DPRINTF("Initialize splitter container...");

	horizontalSplitterContainer_ = new QSplitter(playlistStack_);
	horizontalSplitterContainer_->setFrameStyle(QFrame::NoFrame);
	horizontalSplitterContainer_->setOpaqueResize(true);
	horizontalSplitterContainer_->setOrientation(QSplitter::Horizontal);

	verticalSplitterContainer_ = new QSplitter(horizontalSplitterContainer_);
	verticalSplitterContainer_->setFrameStyle(QFrame::NoFrame);
	verticalSplitterContainer_->setOpaqueResize(true);
	verticalSplitterContainer_->setOrientation(QSplitter::Vertical);

	playlistStack_->addWidget(horizontalSplitterContainer_, 0);
	playlistStack_->raiseWidget(horizontalSplitterContainer_);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Checking application version...");

	// TODO: The following lines are temporary until a better solution is found.
	// Check for pre-TagLib condition and remove database to
	// force rescanning of all files...
	if (qConfig.readAppVersion() == -1)
	{
		qDebug("Removing old database...");
		QFile::remove(qGlobalConfig.databaseFilename());
	}

	DPRINTF("Initialize Playlist class...");

	playList_ = new PlayList(qGlobalConfig.databaseFilename());
	connect(playList_, SIGNAL(playbackIntent()), this, SLOT(stop_play()));
	connect(playList_, SIGNAL(selectionChanged()), this, SLOT(scheduleUpdateButtonAndMenuState()));

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize PlaylistManager class...");

	playListManager_ = new PlayListManager(playList_, playlistStack_, qConfig, this);
	connect(playListManager_, SIGNAL(switchingModeTo(PlayListManager::Mode)), this, SLOT(playlistManagerSwitchingModeTo(PlayListManager::Mode)));
	connect(playListManager_, SIGNAL(switchedModeTo(PlayListManager::Mode)), this, SLOT(playlistManagerSwitchedModeTo(PlayListManager::Mode)));
	connect(playListManager_, SIGNAL(playlistFileNameChanged()), this, SLOT(playlistManagerPlayListFileNameChanged()));
	connect(playListManager_, SIGNAL(playlistLoaded()), this, SLOT(playlistManagerPlayListFileNameChanged()));
	connect(playListManager_, SIGNAL(clipboardChanged()), this, SLOT(updateButtonAndMenuState()));
	connect(playListManager_, SIGNAL(selectionChanged()), this, SLOT(scheduleUpdateButtonAndMenuState()));

	connect(playListToolPanel_, SIGNAL(modeChanged(PlayListManager::Mode)), this, SLOT(playlistToolPanelModeChanged(PlayListManager::Mode)));
	connect(playListToolPanel_, SIGNAL(inputModeChanged(PlayList::InputMode)), this, SLOT(setInputMode(PlayList::InputMode)));
	connect(playListToolPanel_, SIGNAL(copyClicked()), playListManager_, SLOT(copySelectionFromPlayList()));
	connect(playListToolPanel_, SIGNAL(cutClicked()), playListManager_, SLOT(cutSelectionFromPlayList()));
	connect(playListToolPanel_, SIGNAL(pasteClicked()), playListManager_, SLOT(pasteIntoPlayList()));
	connect(playListToolPanel_, SIGNAL(deleteClicked()), playListManager_, SLOT(removeFileFromPlayList()));
	connect(playListToolPanel_, SIGNAL(resetFilterAndSortingClicked()), this, SLOT(resetFilterAndSorting()));
	connect(playListToolPanel_, SIGNAL(copyToOTGClicked()), playListManager_, SLOT(copyToOnTheGoPlayList()));
	connect(playListToolPanel_, SIGNAL(openPlaylistClicked()), playListManager_, SLOT(execDialogOpenPlayList()));
	connect(playListToolPanel_, SIGNAL(savePlaylistClicked()), playListManager_, SLOT(savePlayList()));

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize Playlist overview class...");
	playListOverview_ = new PlayListOverview(playList_);
	connect(playListOverview_, SIGNAL(focusOutLeft()), this, SLOT(playlistOverviewFocusOutLeft()));
	connect(playListOverview_, SIGNAL(focusOutRight()), this, SLOT(playlistOverviewFocusOutRight()));

	playList_->registerFilterExtension(playListOverview_);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize CoverArtFlow class...");
	coverArtProviderCoverArtFlow_ = new CoverArtImageProvider(this);

	coverArtProviderCoverArtFlow_->setImageCacheDirectory(qConfig.coverArtFlowCacheDirectory);
	coverArtProviderCoverArtFlow_->setPreloadImageFilename(skinManager_->activeSkinDirectory() + "/nocover.png");

	coverArtFlowArea_ = new LayeredPaintBox();
	connect(coverArtFlowArea_, SIGNAL(rearrangeLayers()), this, SLOT(coverArtFlowAreaRearrangeLayers()));

	coverArtFlow_ = new CoverArtFlow(coverArtProviderCoverArtFlow_, coverArtFlowArea_);
	coverArtFlow_->setActive(false);
	playList_->registerSortExtension(coverArtFlow_);

	flowScrollBar_ = new SkinFlowScrollBar(Skin::FlowScrollBar, Skin::FlowScrollBarSlider, skinManager_, coverArtFlow_, coverArtFlowArea_);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Setting up playlist layout...");
	playListOverview_->reparent(verticalSplitterContainer_, QPoint(0, 0), true);
	coverArtFlowArea_->reparent(verticalSplitterContainer_, QPoint(0, 0), true);
	playList_->reparent(verticalSplitterContainer_, QPoint(0, 0), true);

	// hide playlist overview initially, so setOverviewVisibility works correctly...
	coverArtFlowArea_->hide();
	playListOverview_->hide();
	playList_->setFocus();

	if (skinManager_->style() == SkinManager::Landscape)
		playList_->loadColumnOrder(qGlobalConfig.configFilename(), "PlayListLandscape");
	else
		playList_->loadColumnOrder(qGlobalConfig.configFilename(), "PlayListPortrait");

	changeOrientation(skinManager_->style() == SkinManager::Landscape, false, false);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize PlayInfo class...");
	// PlayInfo
	coverArtProviderPlayInfo_ = new CoverArtImageProvider(this);

	coverArtProviderPlayInfo_->setImageCacheDirectory(qConfig.coverArtFlowCacheDirectory);
	coverArtProviderPlayInfo_->setPreloadImageFilename(skinManager_->activeSkinDirectory() + "/nocover.png");

	playInfo_ = new PlayInfo(coverArtProviderPlayInfo_, skinManager_, widgetStack_);
	widgetStack_->addWidget(playInfo_, InfoWidget);

	DPRINTF("Done");
	DMEMSTAT();

	// operation panel
	DPRINTF("Initialize OperationPanel class...");
	operationPanel_ = new OperationPanel(skinManager_, centralWidget_);
	connect(operationPanel_, SIGNAL(playClicked()), this, SLOT(play_pause()));
	connect(operationPanel_, SIGNAL(stopClicked()), this, SLOT(stop()));
	connect(operationPanel_, SIGNAL(nextClicked()), this, SLOT(next()));
	connect(operationPanel_, SIGNAL(previousClicked()), this, SLOT(previous()));
	connect(this, SIGNAL(playbackStarted(Media*)), operationPanel_, SLOT(showPlaybackWidgets()));
	connect(this, SIGNAL(playbackStopped()), operationPanel_, SLOT(hidePlaybackWidgets()));
	connect(operationPanel_, SIGNAL(seek(int)), this, SLOT(setAbsoluteSeconds(int)));

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize ToolPanel class...");
	toolPanel_ = new ToolPanel(skinManager_, centralWidget_);
	connect(playList_, SIGNAL(viewUpdated()), this, SLOT(playlistViewUpdated())); // required here! Will access toolPanel_
	connect(toolPanel_, SIGNAL(playOrderChanged(int)), this, SLOT(setPlayOrder(int)));
	connect(toolPanel_, SIGNAL(overviewVisibilityChanged(bool)), this, SLOT(setOverviewVisibility(bool)));
	connect(toolPanel_, SIGNAL(coverArtFlowVisibilityChanged(bool)), this, SLOT(setCoverArtFlowVisibility(bool)));
	connect(toolPanel_, SIGNAL(viewPlayListActivated()), this, SLOT(switchToLastActivePlayList()));
	connect(toolPanel_, SIGNAL(viewPlayInfoActivated()), this, SLOT(switchToInfoWidget()));

	connect(playListManager_, SIGNAL(showMessage(const QString &, int)), toolPanel_, SLOT(setStatusText(const QString &, int)));

	DPRINTF("Done");
	DMEMSTAT();

	// menu bar
	DPRINTF("Initialize Menus...");
	toolBar_ = new QPEToolBar(this);
	toolBar_->setHorizontalStretchable(true);
	setToolBarsMovable(false);

	subBar_ = new QPEToolBar(this);

	menuBar_ = new QPEMenuBar(toolBar_);
	menuBar_->setMargin(0);
	menuBar_->setFrameStyle(QFrame::NoFrame);

	setupMenu();

	connect(playList_, SIGNAL(viewSorted()), this, SLOT(playlistViewSorted()));

	DPRINTF("Done");
	DMEMSTAT();

	// player window
	DPRINTF("Initialize PlayerWindow class...");
	videoPlaybackWindow_ = new VideoPlaybackWindow(this);
	videoPlaybackWindow_->hide();
	connect(videoPlaybackWindow_, SIGNAL(seekAbsolutePercentage(int)), mplayer_, SLOT(seekAbsolutePercentage(int)));
	connect(videoPlaybackWindow_, SIGNAL(seekSliderMoved(int)), mplayer_, SLOT(osdShowProgressBar(int)));
	connect(videoPlaybackWindow_, SIGNAL(seekStart()), mplayer_, SLOT(enableTemporaryOSD()));
	connect(videoPlaybackWindow_, SIGNAL(seekEnd()), mplayer_, SLOT(disableTemporaryOSD()));
	DPRINTF("Done");

	DMEMSTAT();

	setupActions();

	DMEMSTAT();

	setStyleRecursively(this, new QuasarStyle());

	// set current display info mode...
	DPRINTF("Setting Playlist font...");
	setPlayListFont(qConfig.infoMode);

	DPRINTF("Setting Playlist input mode...");
	setInputMode(qConfig.inputMode);

	isFullScreen_ = false;
	wasMaximized_ = false;

	// Set Repeat Mode
	DPRINTF("Setting play order...");
	setPlayOrder(qConfig.playOrder);

	DPRINTF("Setting central widget...");
	setCentralWidget(centralWidget_);

	// raise play list widget...
	DPRINTF("Raising playlist widget...");
	widgetStack_->raiseWidget(PlayListWidget);

	DPRINTF("Setting view on tool panel...");
	toolPanel_->setView(ToolPanel::PlayList);

	// BatteryPlus
	DPRINTF("Initializing BatteryPlus singleton...");
#ifdef QTOPIA
	BatteryPlus::singleton();
#endif

	DPRINTF("Checking for valid config file...");
	if (!isValidConfig)
	{
		QMessageBox::warning(centralWidget_, APPNAME,
							 tr("Invalid or non existing settings file.\nUsing the default settings."),
							 QMessageBox::Ok, QMessageBox::NoButton);
	}

	DPRINTF("Installing event filter...");
	qApp->installEventFilter(this);

	playList_->installEventFilter(this);
	playList_->setAcceptDrops(true);

	// allows us to catch cancel/escape to avoid quitting the player by accident
	DPRINTF("Reparenting and setting WFlags...");
#ifdef QTOPIA
	reparent(0, WStyle_Customize | WStyle_NoBorder, QPoint(0, 0));
#endif

	DPRINTF("Initialize %s finished.", APPNAME);
	DMEMSTAT();

	connect(this, SIGNAL(playbackStarted(Media*)), &qExternalServices, SLOT(playbackStarted(Media*)));
	connect(this, SIGNAL(playbackStopped()), &qExternalServices, SLOT(playbackStopped()));
	connect(this, SIGNAL(playbackSeekedToAbsoluteSeconds(int)), &qExternalServices, SLOT(playbackSeekedToAbsoluteSeconds(int)));
	connect(this, SIGNAL(playbackPaused()), &qExternalServices, SLOT(playbackPaused()));
	connect(this, SIGNAL(playbackUnpaused()), &qExternalServices, SLOT(playbackUnpaused()));

	loadWidgetPosition(this, qGlobalConfig.configFilename(), "QuasarMainWindow");

	qAddOnManager.hookMainWindow(this);

	// we can't use polish, because we need to show the window and
	// some additional dialogs...
	QTimer::singleShot(1, this, SLOT(init()));
}

Quasar::~Quasar()
{
	delete mplayer_;

#ifdef OSX
	AERemoveEventHandler(kCoreEventClass, kAEOpenDocuments, appleEventHandler, false);
#endif
}

void Quasar::init()
{

	// set overview visibility here since we need to be sure
	// the splitter container is correctly sized for the
	// setting of the splitter position to work correctly.
	setOverviewVisibility(qConfig.isOverviewVisible);
	setCoverArtFlowVisibility(qConfig.isCoverArtFlowVisible);

	readFromConfig();

	DPRINTF("Done");
	DMEMSTAT();

	this->setAcceptDrops(true);

	setActiveView(ViewOpenedPlaylist);

	if (!receivedExternalDocuments_)
	{
		DPRINTF("Loading last playlist...");
		if (!qConfig.lastPlaylistFile.isEmpty())
		{
			playListManager_->clearPlayList();
			if (!playListManager_->readPlayList(qConfig.lastPlaylistFile, false))
				playList_->reloadView();
		}
		else
		{
			playListManager_->newPlayList();
		}
	}

	DPRINTF("Done");
	DMEMSTAT();
}

void Quasar::coverArtFlowAreaRearrangeLayers()
{
	coverArtFlow_->setGeometry(QRect(0, 0, coverArtFlowArea_->width(), coverArtFlowArea_->height()));
	flowScrollBar_->setGeometry(QRect(30, coverArtFlowArea_->height() - 30, coverArtFlowArea_->width() - 60, 20));
}

QAction *Quasar::createAction(const QCString& actionName, const QObject *receiver, const char *member)
{
	const Action &act = gActionManager.actionByName(actionName);
	connect(&act, SIGNAL(executed()), receiver, member);

	QAction *a = new QAction(this);
	a->setAccel(act.accelID());
	connect(a, SIGNAL(activated()), &act, SLOT(execute()));
	return a;
}

/*! メニューなどに無いアクションの設定をします。 */
void Quasar::setupActions()
{
	struct SeekAction {
		const char *action;
		int seconds;
	};

	static SeekAction actions[] = {
		{ "Playback/Seek Ahead By 1 Second", 1 },
		{ "Playback/Seek Ahead By 5 Seconds", 5 },
		{ "Playback/Seek Ahead By 10 Seconds", 10 },
		{ "Playback/Seek Ahead By 20 Seconds", 20 },
		{ "Playback/Seek Ahead By 30 Seconds", 30 },
		{ "Playback/Seek Ahead By 1 Minute", 60 * 1 },
		{ "Playback/Seek Ahead By 2 Minutes", 60 * 2 },
		{ "Playback/Seek Ahead By 5 Minutes", 60 * 5 },
		{ "Playback/Seek Ahead By 10 Minutes", 60 * 10 },
		{ "Playback/Seek Back By 1 Second", -1 },
		{ "Playback/Seek Back By 5 Seconds", -5 },
		{ "Playback/Seek Back By 10 Seconds", -10 },
		{ "Playback/Seek Back By 20 Seconds", -20 },
		{ "Playback/Seek Back By 30 Seconds", -30 },
		{ "Playback/Seek Back By 1 Minute", -60 * 1 },
		{ "Playback/Seek Back By 2 Minutes", -60 * 2 },
		{ "Playback/Seek Back By 5 Minutes", -60 * 5 },
		{ "Playback/Seek Back By 10 Minutes", -60 * 10 },
		{ 0, 0 }
	};

	QSignalMapper *sm = new QSignalMapper(this);
	connect(sm, SIGNAL(mapped(int)), this, SLOT(setRelativeSeconds(int)));
	for (SeekAction *p = actions; p->action != NULL; ++p)
	{
		createAction(p->action, sm, SLOT(map()));
		sm->setMapping(&gActionManager.actionByName(p->action), p->seconds);
	}

	// volume
	sm = new QSignalMapper(this);
	connect(sm, SIGNAL(mapped(int)), &qPlatformServices, SLOT(setVolumeRelative(int)));
	createAction("Playback/Volume Up", sm, SLOT(map()));
	sm->setMapping(&gActionManager.actionByName("Playback/Volume Up"), 5);
	createAction("Playback/Volume Down", sm, SLOT(map()));
	sm->setMapping(&gActionManager.actionByName("Playback/Volume Down"), -5);

	// misc
#ifdef QTOPIA
	createAction("Playback/Turn On Display", &qPlatformServices, SLOT(turnOnDisplay()));
#endif
	createAction("Playback/Toggle OSD", mplayer_, SLOT(toggleOSD()));
	createAction("Playlist/Focus Filterbox", filterEditBox_, SLOT(setFocus()));
	createAction("Playlist/Reset Filter And Focus Filterbox", this, SLOT(resetFilterAndFocusFilterEditBox()));
	createAction("Playlist/Reset Filter", this, SLOT(resetFilter()));
	createAction("Playlist/Play Selected Song", this, SLOT(stop_play()));
	createAction("Playback/Toggle Play Order", this, SLOT(togglePlayOrder()));
	createAction("Playback/Toggle Random Play Order", this, SLOT(toggleRandomPlayOrder()));
	createAction("Playback/Toggle PlayInfo", this, SLOT(toggleInfoWidget()));
	createAction("Playlist/Focus Playlist", playList_, SLOT(setFocus()));
	createAction("Overview/Focus Overview", this, SLOT(setFocusPlayListOverview()));

#ifdef QTOPIA
	// brightness
	sm = new QSignalMapper(this);
	connect(sm, SIGNAL(mapped(int)), mplayer_, SLOT(setBrightness(int)));
	createAction("Playback/Decrease Brightness", sm, SLOT(map()));
	sm->setMapping(&gActionManager.actionByName("Playback/Decrease Brightness"), -10);
	createAction("Playback/Increase Brightness", sm, SLOT(map()));
	sm->setMapping(&gActionManager.actionByName("Playback/Increase Brightness"), 10);
#endif
}

//! アプリケーション全体のイベントを監視するイベントフィルタです。
/*!
  ホットキーを処理するために使われています。
*/
bool Quasar::eventFilter(QObject *object, QEvent *event)
{
#ifndef QTOPIA
	if (event->type() == QEvent::DragEnter)
	{
		this->dragEnterEvent((QDragEnterEvent *)event);
		return true;
	}

	if (event->type() == QEvent::DragMove)
	{
		this->dragMoveEvent((QDragMoveEvent *)event);
		return true;
	}

	if (event->type() == QEvent::Drop)
	{
		this->dropEvent((QDropEvent *)event);
		return true;
	}
#endif

	if (object == filterEditBox_ &&
		((event->type() == QEvent::FocusIn) || (event->type() == QEvent::FocusOut)))
	{
		const SkinModeInformation *modeInfo = skinManager_->getInfo(qConfig.infoMode);

		QPalette palette = QApplication::palette();

		if (event->type() == QEvent::FocusIn)
		{
			setActiveView(Quasar::ViewLastActivePlaylist);

			if (modeInfo->filterBoxBackgroundColorFocused().isUsed())
			{
				palette.setColor(QColorGroup::Base, modeInfo->filterBoxBackgroundColorFocused());
			}

			if (modeInfo->filterBoxFontColorFocused().isUsed())
			{
				palette.setColor(QColorGroup::Text, modeInfo->filterBoxFontColorFocused());
				palette.setColor(QColorGroup::Foreground, modeInfo->filterBoxFontColorFocused());
			}
		}
		else if (event->type() == QEvent::FocusOut)
		{
			if (modeInfo->filterBoxBackgroundColor().isUsed())
			{
				palette.setColor(QColorGroup::Base, modeInfo->filterBoxBackgroundColor());
			}

			if (modeInfo->filterBoxFont().fontColor.isUsed())
			{
				palette.setColor(QColorGroup::Text, modeInfo->filterBoxFont().fontColor);
				palette.setColor(QColorGroup::Foreground, modeInfo->filterBoxFont().fontColor);
			}
		}

		filterEditBox_->setPalette(palette);
	}

	if (event->type() == QEvent::KeyPress)
	{
		if (!qApp->activeWindow()->isA("Quasar") &&
			!qApp->activeWindow()->isA("VideoPlaybackWindow"))
			return QMainWindow::eventFilter(object, event);

		QKeyEvent *e = static_cast<QKeyEvent *>(event);
		int key = e->key();

		DPRINTF("QEvent::KeyPress key=%d", key);

		// override special keys when the playlist overview is focused.
		if (playListOverview_->hasFocus() && (key == Qt::Key_Right || key == Qt::Key_Left || key == Qt::Key_Space))
			return QMainWindow::eventFilter(object, event);

		if (qApp->focusWidget() == filterEditBox_)
		{
			if (key == Qt::Key_Down || key == Qt::Key_Up || key == Qt::Key_PageDown || key == Qt::Key_PageUp)
			{
				if (playListManager_->isDynamicPlaylistEditorActive())
				{
					playlistStack_->visibleWidget()->setFocus();
					QMainWindow::eventFilter(playlistStack_->visibleWidget(), event);
				}
				else
				{
					if (playListOverview_->isVisible())
					{
						playListOverview_->setFocus();
						QMainWindow::eventFilter(playListOverview_, event);
					}
					else
					{
						playList_->setFocus();
						QMainWindow::eventFilter(playList_, event);
					}
				}
			}
			else
				return QMainWindow::eventFilter(object, event);
		}

		// Hotkeys are handled in a very non-standard fashion because we support different contexts.
		// Instead of relying on the accelerator key feature provided by Qt, we lookup the current key in our
		// context key -> accelID mapping and then send out a new QEvent::Accel carrying our translated accelID
		// to the event loop.
		// QActions and menuitems that have the accelID set in their accel property are triggered.
		// Thus, the accelID is not really a valid accel key per Qt definition but rather our internal ID that
		// we use to mark and call the associated action...

		if (e->state() & ShiftButton)
			key |= SHIFT;
		if (e->state() & ControlButton)
			key |= CTRL;
		if (e->state() & AltButton)
			key |= ALT;

		Configuration::HotKeyContext context = Configuration::HOTKEY_CONTEXT_GLOBAL;

		Configuration::HotKeyMap::Iterator it(qConfig.hotKeyMap[context].find(key));

		// if no key was found in the global context, look
		// in the other context key -> accelID mappings...
		if (it == qConfig.hotKeyMap[context].end())
		{
			if (videoPlaybackWindow_->isVisible())
				context = Configuration::HOTKEY_CONTEXT_VIDEO;
			else if (activeView() == ViewPlayInfo)
				context = Configuration::HOTKEY_CONTEXT_AUDIO;
			else if ((activeView() >= ViewDynPlaylistEditor && activeView() < ViewPlayInfo) || playList_->hasFocus() || playListOverview_->hasFocus())
				context = Configuration::HOTKEY_CONTEXT_PLAYLIST;

			it = qConfig.hotKeyMap[context].find(key);
		}

		// did we find the key in the current context?
		if (it != qConfig.hotKeyMap[context].end())
		{
			int accelID = it.data();
			DPRINTF("%d (%s) -> %d", key, (const char *)(KeyNames::getName(key).utf8()), accelID);

			// create a new Accel QEvent...
			QKeyEvent a(QEvent::Accel, accelID, 0, 0);
			// we ignore it...
			a.ignore();
			// and delegate it...
			QApplication::sendEvent(topLevelWidget(), &a);

			// Finally accept the key event, because we've processed it.
			// That is, we created a new QEvent::Accel...
			e->accept();
			return true;
		}
	}

	if (event->type() == QEvent::Accel)
	{
		QKeyEvent *e = static_cast<QKeyEvent *>(event);

		int key = e->key();

		DPRINTF("QEvent::Accel key=%d", key);

		if (key == 4096) // ESC / Cancel
		{
			e->ignore();
			return true;
		}
	}

	return QMainWindow::eventFilter(object, event);
}

void Quasar::play_pause()
{
	play_pause(false);
}

void Quasar::play_pause(bool forcePlayNewFile)
{
	// Toggle play/pause
	if (mplayer_->isPlaying() && !forcePlayNewFile)
	{
		if (mplayer_->isPaused())
		{
			mplayer_->enableTemporaryOSD();

			mplayer_->pause();
			playList_->setActiveItemPaused(false);
			operationPanel_->setPlayPause(false);

			emit playbackUnpaused();

			osdTimer_->start(3000, true);
		}
		else
		{
			mplayer_->enableTemporaryOSD();

			mplayer_->pause();
			playList_->setActiveItemPaused(true);
			operationPanel_->setPlayPause(true);

			emit playbackPaused();
		}
	}
	// Start mplayer
	else if (playList_->childCount() > 0 && playList_->currentItem() != NULL)
	{
		playList_->generatePlayOrderTable();	// class PlayList内部に持って行く
		startPlaying();
		Media *media = playList_->currentMedia();
		if (media->isAudio() && qConfig.automaticallySwitchToPlayInfoEnabled) {
			// Show info widget
			setActiveView(Quasar::ViewPlayInfo);
		}

#ifdef QTOPIA
		// BatteryPlus関係の処理
		if (qConfig.isBatteryPlusEnabled) {
			if (media->isVideo()) {
				g_BatteryPlus.setMode(qConfig.batteryPlusMode4Video);
			} else {
				g_BatteryPlus.setMode(qConfig.batteryPlusMode4Audio);
			}
		}

		qPlatformServices.disableSuspend();
#endif
	}
}

void Quasar::stop_play()
{
	DENTERMETHOD("Quasar::stop_play()");

	PlayListItem *item = static_cast<PlayListItem *>(playList_->currentItem());

	if (!item)
		return;

	if (item->media()->errorDetected())
	{
		TextViewDialog errDialog(tr("MPlayer Error Message"),
			 tr("MPlayer could not be started for the following reason:"),
			 item->media()->lastError(),
			 tr("Perhaps your format configuration is wrong.") + "\n" +
			 tr("Do you want to open the configuration dialog?"),
			 tr("Yes"),
			 tr("No"),
			 this
		);

		if (errDialog.exec())
			configuration(true);
	}
	else
	{
		stopPlaying(true, false);
		playList_->setActiveItem(item);
		playList_->setActiveItemPaused(false);
		qApp->processEvents();
		play_pause(true);
	}

	DEXITMETHOD("Quasar::stop_play()");
}

void Quasar::stop()
{
	setActiveView(Quasar::ViewLastActivePlaylist);
	stopPlaying();

#ifdef QTOPIA
	qPlatformServices.enableSuspend();

	// BatteryPlus関係の処理
	if (qConfig.isBatteryPlusEnabled)
		g_BatteryPlus.setMode(g_BatteryPlus.initialMode());
#endif
}

/*!
	@brief	MPlayerを起動し再生を開始する
*/
void Quasar::startPlaying()
{
	Media *media = playList_->currentMedia();

	operationPanel_->setCurrentPlayTime(0);

	DPRINTF("Current Media: %s", (const char *)(media->location().toString().utf8()));

	nextMediaPrefetched_ = false;
	currentMediaLength_ = media->length();

	// Start mplayer
	if (mplayer_->play(media) == MPlayer::InitializationFailed)
	{
		int result = QMessageBox::critical(this, tr("mplayer"),
				 tr("mplayer could not be executed or\nwas not found on your system.\nWould you like to specify the location\nof the mplayer binary on your system?"),
				 QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape);

		if (result == QMessageBox::Yes)
		{
			configuration(true);
		}

		return;
	}

	// Set Operation Panel
	operationPanel_->setPlayPause(false);
	operationPanel_->setTotalPlayTime(currentMediaLength_);

	playList_->setActiveItemPaused(false);

	if (media->isVideo())
	{
#ifndef QTOPIA
		if (qConfig.isOverlayEnabled)
		{
#endif
			videoPlaybackWindow_->hide();
			videoPlaybackWindow_->reparent(0, QPoint(0, 0));
			videoPlaybackWindow_->resize(qApp->desktop()->size());
			videoPlaybackWindow_->setCursor(QCursor(blankCursor));
			videoPlaybackWindow_->showFullScreen();
#ifndef QTOPIA
		}
#endif
	}
	else if (media->isAudio())
	{
		playInfo_->setMediaInfo(*playList_->currentMedia());

		// Get current number
		int n = playList_->itemIndex(playList_->activeItem()) + 1;
		playInfo_->setTrackNumbers(n, playList_->childCount());
	}

	emit playbackStarted(media);
}

/*!
	@brief	MPlayerを終了する
*/
void Quasar::stopPlaying(bool temporarily, bool stopBackend)
{
	// Set Panels
	if (!temporarily)
	{
		operationPanel_->setCurrentPlayTime(0);
		operationPanel_->setPlayPause(true);

		playInfo_->clear();
	}

	// Stop mplayer
	emit playbackStopped();

	if (stopBackend)
		mplayer_->stop();

	emit playbackStopped();

	if (!temporarily)
	{
		playList_->setActiveItem(NULL);
		playList_->setActiveItemPaused(false);
	}

	if (videoPlaybackWindow_->isVisible())
	{
		videoPlaybackWindow_->hide();
		videoPlaybackWindow_->reparent(this, QPoint(0, 0));
		videoPlaybackWindow_->resize(QSize(0, 0));
		videoPlaybackWindow_->unsetCursor();
	}
}

/*!
	@brief	現在選択されている次のアイテムを再生
	@see	previous()
*/
void Quasar::next()
{
	DPRINTF("Quasar::next()");
	if (playList_->nextMedia() != NULL)
	{
		++keyEventCount_;
		qApp->processEvents();
		--keyEventCount_;
		if (keyEventCount_ <= 0)
		{
			stopPlaying(true, false);
			startPlaying();
			keyEventCount_ = 0;
		}
	}
	else
	{
		stop();
	}
}

/*!
	@brief	現在選択されている前のアイテムを再生
	@see	next()
*/
void Quasar::previous()
{
	if (playList_->previousMedia() != NULL)
	{
		++keyEventCount_;
		qApp->processEvents();
		--keyEventCount_;
		if (keyEventCount_ <= 0)
		{
			stopPlaying(true, false);
			startPlaying();
			keyEventCount_ = 0;
		}
	}
}

/*!
	@brief	関連付からの起動時に呼ばれるスロット
*/
void Quasar::setDocument(const QString &path)
{
	stop();
	playListManager_->newPlayList();

	// Load ASX
	QRegExp asxExp("^[^\\.]+.asx$", false);
	if (asxExp.match(path) != -1)
	{
		playListManager_->openASX(path);
	}
	else
	{
		qConfig.lastDirectory = QFileInfo(path).dirPath(true);
		playList_->insertItem(path);
	}

	playList_->updateFilter();
	playList_->setActiveItem(0);
	play_pause();
}

void Quasar::handleExternalDocuments(const QStringList &filenames)
{
	if (filenames.count() == 1)
	{
		QString filename = filenames[0];
		if (filename.right(11).lower() == "dynplaylist" ||
			filename.right(8).lower() == "playlist" ||
			filename.right(3).lower() == "m3u" ||
			filename.right(3).lower() == "asx")
		{
			playListManager_->clearPlayList();
			if (!playListManager_->readPlayList(filename, false))
				playList_->reloadView();

			receivedExternalDocuments_ = true;
			return;
		}
	}

	QExtStringList playList;

	for (QStringList::ConstIterator it = filenames.begin(); it != filenames.end(); ++it)
    {
        QExtString filename(*it);

#ifdef WINDOWS
        filename = filename.replace('\\', '/');
#endif

		if (QFileInfo(filename).isDir())
		{
			if (filename.right(1) != "/")
				filename += "/**";
			else
				filename += "**";
		}

		playList.append(filename);
    }

	if (playList.count() > 0)
		playListManager_->addToPlayList(playList, true);

	receivedExternalDocuments_ = true;
}

#ifndef QTOPIA
void Quasar::dragEnterEvent(QDragEnterEvent *e)
{
	if (!QUriDrag::canDecode(e))
	{
		e->ignore();
		return;
	}

	e->accept();
}


void Quasar::dragMoveEvent(QDragMoveEvent *e)
{
	if (!QUriDrag::canDecode(e))
	{
		e->ignore();
		return;
	}

	e->accept();
}

void Quasar::dropEvent(QDropEvent *e)
{
	if (!QUriDrag::canDecode(e))
	{
		e->ignore();
		return;
	}

	QStrList lst;
	QUriDrag::decode(e, lst);

	e->acceptAction();

	QStringList dropList;

	for (uint i = 0; i < lst.count(); ++i)
		dropList.append(QUriDrag::uriToLocalFile(lst.at(i)));

	handleExternalDocuments(dropList);

	e->accept();
}
#endif

void Quasar::closeEvent(QCloseEvent *e)
{
	if (mplayer_->isPlaying())
	{
		stop();
		e->ignore();
		return;
	}
	else
	{
		// 終了確認
		if (qConfig.isAskQuitingEnabled)
		{
			int ret;

			ret = QMessageBox::warning(this, tr("Confirmation"),
										tr("Do you really want to quit?"),
										tr("Yes"), tr("No"), QString::null,
										1, 1);

			/*
				MessageBoxを出した後はe->ignore()が無効になっている？
				e->ignore();
				return;
				としても終了してしまう
			*/
			if (ret == 1)
			{
				DPRINTF("close cancelled.");
				e->ignore();
				DPRINTF("event ignored.");
				return;
			}
		}

#ifdef QTOPIA
		qPlatformServices.setRemoteCon(true);
#endif

		if (skinManager_->style() == SkinManager::Landscape)
			playList_->saveColumnWidth(qGlobalConfig.configFilename(), "PlayListLandscape");
		else
			playList_->saveColumnWidth(qGlobalConfig.configFilename(), "PlayListPortrait");

		playList_->saveColumnOrder(qGlobalConfig.configFilename(), "PlayListLandscape");
		playList_->saveColumnOrder(qGlobalConfig.configFilename(), "PlayListPortrait");

		saveWidgetPosition(this, qGlobalConfig.configFilename(), "QuasarMainWindow");

		// save current splitter position if playlist overview is visible...
		if (playListOverview_->isVisible())
			qConfig.overviewSplitterPosition = verticalSplitterContainer_->sizes()[0];

		if (coverArtFlowArea_->isVisible())
			qConfig.coverArtFlowSplitterPosition = verticalSplitterContainer_->sizes()[1];

		qConfig.write();

		//delete playList_;

		DPRINTF("close ok.");

		e->accept();
		return;
	}
}

void Quasar::resizeEvent(QResizeEvent *e)
{
	DPRINTF("Quasar::resizeEvent : oldSize(%dx%d), newSize(%dx%d)",
 		   e->oldSize().width(), e->oldSize().height(),
 		   e->size().width(), e->size().height());
 	DPRINTF("  desktop(%dx%d)", qApp->desktop()->width(), qApp->desktop()->height());

	// are we switching back from full screen mode?
	if (e->oldSize().width() == qApp->desktop()->width() &&
		e->oldSize().height() == qApp->desktop()->height())
	{
		// necessary to switch back to normal mode in a clean fashion
		// otherwise the fullscreen mode won't work again.
		showNormal();
#ifdef QTOPIA
		 // on Qtopia force maximized view immediately after switching back to normal mode...
		showMaximized();
#endif
		isFullScreen_ = false;
	}

	checkOrientation();
}

void Quasar::setPlayOrder(int itemID)
{
	// 再生順序に関するアイテムが選択された場合のみ
	if (itemID >= 0 && itemID < PlayList::ORDER_MAX) {
		for (int i = 0; i < PlayList::ORDER_MAX; ++i) {
			controlsMenu_->setItemChecked(i, i == itemID ? true : false);
		}

		PlayList::PlayOrder playOrder = static_cast<PlayList::PlayOrder>(itemID);
		playList_->setPlayOrder(playOrder);
		toolPanel_->setPlayOrder(playOrder);
		/*
		filteredPlayList_->setPlayOrder(playOrder);
		*/
		qConfig.playOrder = playOrder;

		QString statusMsg;
		switch (qConfig.playOrder) {
		case PlayList::ORDER_REPEAT_ALL:
			statusMsg = tr("Switched to play order Repeat All.");
			break;
		case PlayList::ORDER_REPEAT_ONE:
			statusMsg = tr("Switched to play order Repeat One.");
			break;
		case PlayList::ORDER_RANDOM:
			statusMsg = tr("Switched to play order Random.");
			break;
		case PlayList::ORDER_REPEAT_RANDOM:
			statusMsg = tr("Switched to play order Repeat Random.");
			break;
		default:
			statusMsg = tr("Switched to play order Normal.");
			break;
		}

		toolPanel_->setStatusText(statusMsg);
	}
}

void Quasar::togglePlayOrder()
{
	qConfig.playOrder = static_cast<PlayList::PlayOrder>(qConfig.playOrder + 1);
	if (qConfig.playOrder >= PlayList::ORDER_MAX)
		qConfig.playOrder = PlayList::ORDER_NORMAL;

	setPlayOrder(qConfig.playOrder);
}

void Quasar::toggleRandomPlayOrder()
{
	if (qConfig.playOrder >= PlayList::ORDER_NORMAL &&
		qConfig.playOrder <= PlayList::ORDER_REPEAT_ALL)
		qConfig.playOrder = static_cast<PlayList::PlayOrder>(PlayList::ORDER_RANDOM + qConfig.playOrder - PlayList::ORDER_NORMAL);
	else if (qConfig.playOrder >= PlayList::ORDER_RANDOM &&
			 qConfig.playOrder <= PlayList::ORDER_REPEAT_RANDOM)
		qConfig.playOrder = static_cast<PlayList::PlayOrder>(PlayList::ORDER_NORMAL + qConfig.playOrder - PlayList::ORDER_RANDOM);

	setPlayOrder(qConfig.playOrder);
}

void Quasar::setFullScreen(int id)
{
	if (id != fullScreenMenuItemID_)
		return;

	qConfig.isFullScreenEnabled = qConfig.isFullScreenEnabled ? false : true;
	controlsMenu_->setItemChecked(fullScreenMenuItemID_, qConfig.isFullScreenEnabled);
}

void Quasar::about()
{
	AboutDialog *dialog = new AboutDialog();
	dialog->show();
}

void Quasar::rescanForCoverArt()
{
	gCoverArtProvider.rescan();
	coverArtFlow_->reset();
	coverArtFlow_->execute();

	coverArtProviderPlayInfo_->resetNullImageNameList();
	coverArtProviderPlayInfo_->flushMemoryCache();
	if (playInfo_->isVisible())
		playInfo_->repaint();
}

void Quasar::showMediaInfo()
{
	MediaInfoDialog dialog(playList_, skinManager_->activeSkinDirectory() + "/nocover.png", this);
	loadWidgetPosition(&dialog, qGlobalConfig.configFilename(), "MediaInfoDialog");
	dialog.exec();
	saveWidgetPosition(&dialog, qGlobalConfig.configFilename(), "MediaInfoDialog");
}

int Quasar::createMenuItem(QPopupMenu *menu, const QCString &menuText, QAction *action)
{
	return menu->insertItem(tr(menuText), action, SIGNAL(activated()));
}

void Quasar::setupMenu()
{
	updateButtonAndMenuTimer_ = new QTimer(this);
	connect(updateButtonAndMenuTimer_, SIGNAL(timeout()), this, SLOT(updateButtonAndMenuState()));

	// Playlist
	fileMenu_ = new QPopupMenu(menuBar_);

	idFileMenuNewPlaylist_ = createMenuItem(fileMenu_, "New Playlist", createAction("Playlist/New", playListManager_, SLOT(newPlayList())));
	idFileMenuNewDynPlaylist_ = fileMenu_->insertItem(tr("New Dynamic Playlist..."), playListManager_, SLOT(newDynamicPlayList()));
	idFileMenuOpenPlaylist_ = createMenuItem(fileMenu_, "Open Playlist...", createAction("Playlist/Open...", playListManager_, SLOT(execDialogOpenPlayList())));
	fileMenu_->insertSeparator();
	idFileMenuSavePlaylist_ = createMenuItem(fileMenu_, "Save Playlist", createAction("Playlist/Save", playListManager_, SLOT(savePlayList())));
	idFileMenuSaveAsPlaylist_ = createMenuItem(fileMenu_, "Save Playlist As...", createAction("Playlist/Save As...", playListManager_, SLOT(execDialogSavePlayListAs())));
	fileMenu_->insertSeparator();
	fileMenu_->insertItem(tr("Quit"), this, SLOT(close()));

	connect(fileMenu_, SIGNAL(aboutToShow()), this, SLOT(fileMenuAboutToShow()));
	menuBar_->insertItem(tr("File"), fileMenu_);

	editMenu_ = new QPopupMenu(menuBar_);

	createMenuItem(editMenu_, "Add From URL...", createAction("Playlist/Add Location...", playListManager_, SLOT(execDialogAddURLToPlayList())));
	createMenuItem(editMenu_, "Add From File...", createAction("Playlist/Add Files...", playListManager_, SLOT(execDialogAddFileToPlayList())));
	createMenuItem(editMenu_, "Add From Directory...", createAction("Playlist/Add Directory...", playListManager_, SLOT(execDialogAddDirToPlayList())));
	editMenu_->insertItem(tr("Add From Directory Including Subdirectories..."), playListManager_, SLOT(execDialogAddDirWithSubDirsToPlayList()));
	editMenu_->insertSeparator();
	idEditMenuCutItem_ = createMenuItem(editMenu_, "Cut", createAction("Playlist/Cut", playListManager_, SLOT(cutSelectionFromPlayList())));
	idEditMenuCopyItem_ = createMenuItem(editMenu_, "Copy", createAction("Playlist/Copy", playListManager_, SLOT(copySelectionFromPlayList())));
	idEditMenuPasteItem_ = createMenuItem(editMenu_, "Paste", createAction("Playlist/Paste", playListManager_, SLOT(pasteIntoPlayList())));
	idEditMenuDeleteItem_ = createMenuItem(editMenu_, "Delete", createAction("Playlist/Delete", playListManager_, SLOT(removeFileFromPlayList())));
	editMenu_->insertSeparator();
	createMenuItem(editMenu_, "Clear", createAction("Playlist/Clear", playListManager_, SLOT(clearPlayList())));
	editMenu_->insertSeparator();
	createMenuItem(editMenu_, "Select All", createAction("Playlist/Select All", playListManager_, SLOT(selectAll())));
	createMenuItem(editMenu_, "Deselect All", createAction("Playlist/Deselect All", playListManager_, SLOT(deselectAll())));
	editMenu_->insertSeparator();
	idEditMenuSelectModeItem_ = createMenuItem(editMenu_, "Select Mode", createAction("Playlist/Select Mode", this, SLOT(toggleSelectMode())));
	idEditMenuMultiSelectModeItem_ = createMenuItem(editMenu_, "Multiselect Mode", createAction("Playlist/Multiselect Mode", this, SLOT(toggleMultiSelectMode())));
	idEditMenuMoveItemModeItem_ = createMenuItem(editMenu_, "Move Item Mode", createAction("Playlist/Move Item Mode", this, SLOT(toggleMoveItemMode())));
	editMenu_->insertSeparator();
	idEditMenuCopyToOTGItem_ = createMenuItem(editMenu_, "Copy to On-The-Go play list", createAction("Playlist/Copy To On-The-Go Playlist", playListManager_, SLOT(copyToOnTheGoPlayList())));
	idEditMenuShowMediaInfoItem_ = createMenuItem(editMenu_, "Show Media Info...", createAction("Playlist/Show Media Info...", this, SLOT(showMediaInfo())));

	// connect handler for updating the above entries everytime the popup menu is shown...
	connect(editMenu_, SIGNAL(aboutToShow()), this, SLOT(editMenuAboutToShow()));
	menuBar_->insertItem(tr("Edit"), editMenu_);

	playList_->setPopupMenu(editMenu_);

	// View
	viewMenu_ = new QPopupMenu(menuBar_);

#ifdef QTOPIA
	createMenuItem(viewMenu_, "Turn Off Display", createAction("Playback/Turn Off Display", &qPlatformServices, SLOT(turnOffDisplay())));
#endif

	idViewMenuFullscreenItem_ = createMenuItem(viewMenu_, "Fullscreen", createAction("Display/Toggle Fullscreen", this, SLOT(toggleFullScreen())));
	viewMenu_->insertSeparator();
	idViewMenuShowOverviewItem_ = createMenuItem(viewMenu_, "Show Overview", createAction("Display/Toggle Overview Visibility", this, SLOT(toggleOverviewVisibility())));
	idViewMenuShowCoverArtFlowItem_ = createMenuItem(viewMenu_, "Show Cover Art Flow", createAction("Display/Toggle Cover Art Flow Visibility", this, SLOT(toggleCoverArtFlowVisibility())));
	viewMenu_->insertSeparator();

	// Sort By Submenu
	sortByMenu_ = new QPopupMenu(menuBar_);
	sortByMenu_->insertItem(tr("List order"), PlayList::COLUMN_NONE);
	sortByMenu_->insertSeparator();
	sortByMenu_->insertItem(tr("Track number"), PlayList::COLUMN_TRACKNO);
	sortByMenu_->insertItem(tr("Title"), PlayList::COLUMN_TITLE);
	sortByMenu_->insertItem(tr("Artist"), PlayList::COLUMN_ARTIST);
	sortByMenu_->insertItem(tr("Album"), PlayList::COLUMN_ALBUM);
	sortByMenu_->insertItem(tr("Length"), PlayList::COLUMN_LENGTH);
	sortByMenu_->setItemChecked(playList_->sortedColumn(), true);
	connect(sortByMenu_, SIGNAL(activated(int)), playList_, SLOT(setSortedColumn(int)));

	viewMenu_->insertItem(tr("Sort by"), sortByMenu_);
	viewMenu_->insertSeparator();
	idViewMenuDynamicPlayListEditorItem_ = viewMenu_->insertItem(tr("Dynamic Playlist Editor"), this, SLOT(switchPlaylistEditorView()));
	idViewMenuCurrentPlayListItem_ = createMenuItem(viewMenu_, "Current Playlist", createAction("View/Current Playlist", this, SLOT(switchToCurrentPlayList())));
	idViewMenuOnTheGoPlayListItem_ = createMenuItem(viewMenu_, "On-The-Go Playlist", createAction("View/On-The-Go Playlist", this, SLOT(switchToOnTheGoPlayList())));
	idViewMenuPlayInfoItem_ = createMenuItem(viewMenu_, "PlayInfo", createAction("View/PlayInfo", this, SLOT(switchToInfoWidget())));
	viewMenu_->insertSeparator();
	idViewMenuIncDisplaySize_ = createMenuItem(viewMenu_, "Increase Size", createAction("Display/Increase Size", this, SLOT(incDisplaySize())));
	idViewMenuDecDisplaySize_ = createMenuItem(viewMenu_, "Decrease Size", createAction("Display/Decrease Size", this, SLOT(decDisplaySize())));
	viewMenu_->insertSeparator();
	idViewMenuQuickswitchSkin1_ = createMenuItem(viewMenu_, "Quickswitch Skin 1", createAction("Display/Quickswitch Skin 1", this, SLOT(loadPrimarySkin())));
	idViewMenuQuickswitchSkin2_ = createMenuItem(viewMenu_, "Quickswitch Skin 2", createAction("Display/Quickswitch Skin 2", this, SLOT(loadSecondarySkin())));

	connect(viewMenu_, SIGNAL(aboutToShow()), this, SLOT(viewMenuAboutToShow()));
	menuBar_->insertItem(tr("View"), viewMenu_);

	// Controls
	controlsMenu_ = new QPopupMenu(menuBar_);
	controlsMenu_->setCheckable(true);
	fullScreenMenuItemID_ = controlsMenu_->insertItem(tr("Play Videos In Fullscreen Mode"));
	controlsMenu_->setItemChecked(fullScreenMenuItemID_, qConfig.isFullScreenEnabled);
	connect(controlsMenu_, SIGNAL(activated(int)), this, SLOT(setFullScreen(int)));
	controlsMenu_->insertSeparator();
	createMenuItem(controlsMenu_, "Play Or Pause", createAction("Playback/Play Or Pause", this, SLOT(play_pause())));
	createMenuItem(controlsMenu_, "Stop", createAction("Playback/Stop", this, SLOT(stop())));
	createMenuItem(controlsMenu_, "Previous", createAction("Playback/Previous", this, SLOT(previous())));
	createMenuItem(controlsMenu_, "Next", createAction("Playback/Next", this, SLOT(next())));
	controlsMenu_->insertSeparator();
	controlsMenu_->insertItem(tr("Normal"), PlayList::ORDER_NORMAL);
	controlsMenu_->insertItem(tr("Repeat All"), PlayList::ORDER_REPEAT_ALL);
	controlsMenu_->insertItem(tr("Repeat One"), PlayList::ORDER_REPEAT_ONE);
	controlsMenu_->insertItem(tr("Random"), PlayList::ORDER_RANDOM);
	controlsMenu_->insertItem(tr("Repeat Random"), PlayList::ORDER_REPEAT_RANDOM);
	controlsMenu_->setItemChecked(qConfig.playOrder, true);
	connect(controlsMenu_, SIGNAL(activated(int)), this, SLOT(setPlayOrder(int)));

	menuBar_->insertItem(tr("Controls"), controlsMenu_);

	// Tools
	toolsMenu_ = new QPopupMenu(menuBar_);
	idToolsMenuRescanDynamicPlaylist_ = toolsMenu_->insertItem(tr("Rescan Dynamic Playlist..."), playListManager_, SLOT(refreshDynamicPlayList()));
	toolsMenu_->insertSeparator();
	idToolsMenuRescanForCoverArt_ = toolsMenu_->insertItem(tr("Rescan For Cover Art Images..."), this, SLOT(rescanForCoverArt()));

#ifdef OSX
	// Move menu items to hidden submenu to workaround a bug in Qt's special handling of menu items on OS X
	QPopupMenu *dummyMenuOSX = new QPopupMenu(menuBar_);
	dummyMenuOSX->insertItem(tr("Configuration..."), this, SLOT(configuration()));
	dummyMenuOSX->insertItem(tr("About..."), this, SLOT(about()));
	menuBar_->insertItem("OSX", dummyMenuOSX);

	idToolsMenuConfiguration_ = -1;
	idToolsMenuAbout_ = -1;
#else
	toolsMenu_->insertSeparator();
	idToolsMenuConfiguration_ = toolsMenu_->insertItem(tr("Configuration..."), this, SLOT(configuration()));
	toolsMenu_->insertSeparator();
	idToolsMenuAbout_ = toolsMenu_->insertItem(tr("About..."), this, SLOT(about()));
#endif

	connect(toolsMenu_, SIGNAL(aboutToShow()), this, SLOT(toolsMenuAboutToShow()));
	menuBar_->insertItem(tr("Tools"), toolsMenu_);

	// Filter Edit Box
	filterEditBox_ = new QLineEdit(subBar_);
	filterEditTimer_ = new QTimer(filterEditBox_);
	connect(filterEditTimer_, SIGNAL(timeout()), this, SLOT(filterEditBoxTimedUpdate()));
	connect(filterEditBox_, SIGNAL(returnPressed()), this, SLOT(filterEditBoxReturnPressed()));
	connect(filterEditBox_, SIGNAL(textChanged(const QString&)), this, SLOT(filterEditBoxTextChanged(const QString&)));
	connect(playListManager_, SIGNAL(playlistFilterChanged(const QString &)), this, SLOT(playlistManagerPlayListFilterChanged(const QString &)));
	subBar_->setStretchableWidget(filterEditBox_);
}

void Quasar::playlistManagerPlayListFilterChanged(const QString &filter)
{
	if (filterEditBox_->text() != filter)
	{
		filterEditBox_->blockSignals(true);
		filterEditBox_->setText(filter);
		filterEditBox_->blockSignals(false);
	}
}

void Quasar::fileMenuAboutToShow()
{
	Quasar::View view = activeView();
	bool viewIsOpenedPlaylist = (view == Quasar::ViewOpenedPlaylist || view == Quasar::ViewDynPlaylistEditor);

	// New Playlist Menu Item
	fileMenu_->changeItem(idFileMenuNewPlaylist_, viewIsOpenedPlaylist ? tr("New Playlist") : tr("New Playlist (N/A)"));
	fileMenu_->setItemEnabled(idFileMenuNewPlaylist_, viewIsOpenedPlaylist);

	// New Dynamic Playlist Menu Item
	fileMenu_->changeItem(idFileMenuNewDynPlaylist_, viewIsOpenedPlaylist ? tr("New Dynamic Playlist") : tr("New Dynamic Playlist (N/A)"));
	fileMenu_->setItemEnabled(idFileMenuNewDynPlaylist_, viewIsOpenedPlaylist);

	// Open Menu Item
	fileMenu_->changeItem(idFileMenuOpenPlaylist_, viewIsOpenedPlaylist ? tr("Open Playlist...") : tr("Load From Playlist File..."));

	// Save Menu Item
	fileMenu_->changeItem(idFileMenuSavePlaylist_, viewIsOpenedPlaylist ? tr("Save Playlist") : tr("Save Playlist (N/A)"));
	fileMenu_->setItemEnabled(idFileMenuSavePlaylist_, viewIsOpenedPlaylist);

	// Save As Menu Item
	fileMenu_->changeItem(idFileMenuSaveAsPlaylist_, viewIsOpenedPlaylist ? tr("Save As Playlist...") : tr("Save To Playlist File..."));
}

void Quasar::updateButtonAndMenuState()
{
	if (updateButtonAndMenuTimer_->isActive())
		updateButtonAndMenuTimer_->stop();

	editMenuAboutToShow();
}

void Quasar::scheduleUpdateButtonAndMenuState()
{
	if (updateButtonAndMenuTimer_->isActive())
		updateButtonAndMenuTimer_->stop();

	updateButtonAndMenuTimer_->start(500, TRUE);
}

void Quasar::editMenuAboutToShow()
{
	bool isDynamicPlaylist = playListManager_->isDynamicPlaylist();
	bool isDynamicPlaylistEditorActive = playListManager_->isDynamicPlaylistEditorActive();
	bool anythingSelected = playListManager_->isAnythingSelected();

	editMenu_->changeItem(idEditMenuCutItem_, isDynamicPlaylist ? tr("Cut (N/A for Dynamic Playlists)") : tr("Cut"));
	bool cutEnabled = !isDynamicPlaylist && !isDynamicPlaylistEditorActive && anythingSelected;
	editMenu_->setItemEnabled(idEditMenuCutItem_, cutEnabled);
	playListToolPanel_->setCutEnabled(cutEnabled);

	bool copyEnabled = !isDynamicPlaylistEditorActive && anythingSelected;
	editMenu_->setItemEnabled(idEditMenuCopyItem_, copyEnabled);
	playListToolPanel_->setCopyEnabled(copyEnabled);

	editMenu_->changeItem(idEditMenuPasteItem_, isDynamicPlaylist ? tr("Paste (N/A for Dynamic Playlists)") : tr("Paste"));
	bool pasteEnabled =
		!isDynamicPlaylist &&
		!isDynamicPlaylistEditorActive &&
		playListManager_->mediaClipboardCount() > 0;

	editMenu_->setItemEnabled(idEditMenuPasteItem_, pasteEnabled);
	playListToolPanel_->setPasteEnabled(pasteEnabled);

	editMenu_->changeItem(idEditMenuDeleteItem_, isDynamicPlaylist && !isDynamicPlaylistEditorActive ? tr("Delete (N/A for Dynamic Playlists)") : tr("Delete"));
	bool deleteEnabled = (!isDynamicPlaylist || isDynamicPlaylistEditorActive) && anythingSelected;
	editMenu_->setItemEnabled(idEditMenuDeleteItem_, deleteEnabled);
	playListToolPanel_->setDeleteEnabled(deleteEnabled);

	bool otgEnabled = playListManager_->activeMode() == PlayListManager::ModeOpenedPlaylist && anythingSelected;
	playListToolPanel_->setCopyToOTGEnabled(otgEnabled);
	editMenu_->setItemEnabled(idEditMenuCopyToOTGItem_, otgEnabled);

	editMenu_->setItemEnabled(idEditMenuShowMediaInfoItem_, anythingSelected);

	editMenu_->setItemChecked(idEditMenuSelectModeItem_, playList_->inputMode() == PlayList::Select);
	editMenu_->setItemChecked(idEditMenuMultiSelectModeItem_, playList_->inputMode() == PlayList::MultiSelect);
	editMenu_->setItemChecked(idEditMenuMoveItemModeItem_, playList_->inputMode() == PlayList::Move);
	editMenu_->setItemEnabled(idEditMenuMoveItemModeItem_, !isDynamicPlaylist && anythingSelected);
}

void Quasar::viewMenuAboutToShow()
{
	Quasar::View view = activeView();

	viewMenu_->setItemChecked(idViewMenuFullscreenItem_, isFullScreen_);

	viewMenu_->setItemChecked(idViewMenuShowOverviewItem_, playListOverview_->isVisible());
	viewMenu_->setItemChecked(idViewMenuShowCoverArtFlowItem_, coverArtFlowArea_->isVisible());

	// current playlist menu item
	viewMenu_->setItemChecked(idViewMenuCurrentPlayListItem_, view == Quasar::ViewOpenedPlaylist);

	// on the go playlist menu item
	viewMenu_->setItemChecked(idViewMenuOnTheGoPlayListItem_, view == Quasar::ViewOnTheGoPlaylist);

	// Dynamic playlist editor menu item
	viewMenu_->changeItem(idViewMenuDynamicPlayListEditorItem_, playListManager_->isDynamicPlaylist() ? tr("Dynamic Playlist Editor") : tr("Dynamic Playlist Editor (N/A)"));
	viewMenu_->setItemEnabled(idViewMenuDynamicPlayListEditorItem_, playListManager_->isDynamicPlaylist());
	viewMenu_->setItemChecked(idViewMenuDynamicPlayListEditorItem_, view == Quasar::ViewDynPlaylistEditor);

	// Play info menu item
	viewMenu_->setItemChecked(idViewMenuPlayInfoItem_, view == Quasar::ViewPlayInfo);

	// Update Increase / Decrease size menu items
	viewMenu_->setItemEnabled(idViewMenuIncDisplaySize_, qConfig.infoMode != SkinManager::BigMode);
	viewMenu_->setItemEnabled(idViewMenuDecDisplaySize_, qConfig.infoMode != SkinManager::SmallMode);
}

void Quasar::toolsMenuAboutToShow()
{
	toolsMenu_->setItemEnabled(idToolsMenuRescanDynamicPlaylist_, playListManager_->isDynamicPlaylist());
}

void Quasar::setAbsoluteSeconds(int secs)
{
	emit playbackSeekedToAbsoluteSeconds(secs);
	mplayer_->seekAbsoluteSeconds(secs);
}

void Quasar::setRelativeSeconds(int secs)
{
	mplayer_->enableTemporaryOSD();
	mplayer_->seekRelativeSeconds(secs);

	osdTimer_->start(3000, true);

	// TODO: Implement handling of relative seeking for external services...
	//qExternalServices.playbackSeekedToAbsoluteSeconds(mplayer_->currentPlayTime());
}

void Quasar::osdTimerTimedUpdate()
{
	if (!mplayer_->isPaused())
		mplayer_->disableTemporaryOSD();
}

void Quasar::mplayerErrorMessage(QString msg)
{
	DENTERMETHOD("Quasar::mplayerErrorMessage(%s)", (const char *)msg.utf8());

	playList_->currentMedia()->setLastError(msg);
	playList_->activeItem()->repaint();

	DEXITMETHOD("Quasar::mplayerErrorMessage(%s)", (const char *)msg.utf8());
}

void Quasar::mplayerCurrentPlayTime(int seconds)
{
	operationPanel_->setCurrentPlayTime(seconds);
	playInfo_->setCurrentPlayTime(seconds);

	if (!nextMediaPrefetched_ && currentMediaLength_ - seconds < 21)
	{
		// prefetch next track...
		nextMediaPrefetched_ = true;

		PlayListItem *item = playList_->nextPlayItem();
		Media *media = NULL;

		if (item)
			media = item->media();

		if (media)
		{
			QFile mediaFile(media->location());

			// make sure all necessary data is read from database...
			if (media->isAudio())
				MediaAudio *audio = media->mediaAudio();
			else if (media->isVideo())
				MediaVideo *video = media->mediaVideo();

			if (qConfig.isPrefetchFileEnabled && mediaFile.open(IO_Raw | IO_ReadOnly))
			{
				char buffer[65536];

				int maxFetchBytes = 1024 * 1024;
				if (qConfig.isCacheEnabled && (qConfig.cacheSize + 128) * 1024 > maxFetchBytes)
					maxFetchBytes = (qConfig.cacheSize + 128) * 1024;

				QTime time;
				time.start();

		        while (!mediaFile.atEnd() && mediaFile.at() < maxFetchBytes)
		        {
		        	// only read a block any x milliseconds
		        	// in order not to overload the I/O and induce skips
		        	while (time.elapsed() < 25)
		        		qApp->processOneEvent();

		        	mediaFile.readBlock(&buffer[0], 65536);

		        	time.restart();
		        }

		        mediaFile.close();
			}

			playInfo_->precacheCoverArt(*media);
		}
	}
}

/*! 設定ダイアログを実行し、設定の変更を反映します。

 設定ファイルにも書き戻します。
*/
void Quasar::configuration()
{
	configuration(false);
}

void Quasar::configuration(bool showMPlayerConfig)
{
	DPRINTF("[ConfigurationDialog] Initializing...");
	ConfigurationDialog configDialog(qConfig, this, "Configuration Dialog");

	connect(&configDialog, SIGNAL(nukeMediaCacheButtonClicked()), playList_, SLOT(cleanUpMediaCache()));
	DPRINTF("[ConfigurationDialog] Done");

	if (showMPlayerConfig)
		configDialog.showMPlayerConfig();

	DPRINTF("[ConfigurationDialog] Executing...");
	if (configDialog.exec() == QDialog::Accepted)
	{
		DPRINTF("[ConfigurationDialog] Done");

		qConfig = configDialog.result();
		qConfig.write();

		readFromConfig();
	}
	DPRINTF("[ConfigurationDialog] Destroy");
}

void Quasar::readFromConfig()
{
#ifdef QTOPIA
	qPlatformServices.setRemoteCon();
#endif

	loadSkin(qConfig.skin1);

	DPRINTF("Scanning cover art directory...");
	gCoverArtProvider.setCoverArtDirectory(qConfig.coverArtDirectory);

	if (qConfig.isTouchModeEnabled)
		playListOverview_->setSelectionMode(QListBox::Multi);
	else
		playListOverview_->setSelectionMode(QListBox::Extended);

	playList_->setKineticScrollingEnabled(qConfig.isKineticScrollingEnabled);
}

void Quasar::filterEditBoxReturnPressed()
{
	playList_->setFocus();
}

void Quasar::filterEditBoxTextChanged(const QString &text)
{
	newFilterText_ = text;

	setActiveView(Quasar::ViewLastActivePlaylist);

	if (filterEditTimer_->isActive())
		filterEditTimer_->stop();

	filterEditTimer_->start(qConfig.playlistAutoTriggerFilterInterval, TRUE);
}

void Quasar::filterEditBoxTimedUpdate()
{
	DPRINTF("searchEditTextChanged %s", newFilterText_.latin1());
	setFilter(newFilterText_);
}

void Quasar::setFilter(const QString &text)
{
	DTIMERINIT(filteroverall);

	setActiveView(Quasar::ViewLastActivePlaylist);

	DTIMERINIT(setfilter);
	playList_->setFilter(text);
	DTIMERPRINT(setfilter, "playList_->setFilter(text);");

	DTIMERPRINT(filteroverall, "Quasar::setFilter");
}

void Quasar::resetFilter()
{
	filterEditBox_->clear(); // will trigger setFilter
}

void Quasar::resetFilterAndFocusFilterEditBox()
{
	resetFilter();
	filterEditBox_->setFocus();
}

void Quasar::setPlayListFont(SkinManager::InfoMode mode)
{
	const SkinModeInformation *modeInfo = skinManager_->getInfo(mode);

	playList_->setSkinMode(*modeInfo);
	playListOverview_->setSkinMode(*modeInfo);

	if (modeInfo->playListOverviewSeparatorColor().isUsed())
	{
		QPalette palette(modeInfo->playListOverviewSeparatorColor());
		verticalSplitterContainer_->setPalette(palette);
		horizontalSplitterContainer_->setPalette(palette);
	}
	else
	{
		verticalSplitterContainer_->unsetPalette();
		horizontalSplitterContainer_->unsetPalette();
	}

	if (modeInfo->toolBarColor().isUsed())
	{
		QPalette palette(modeInfo->toolBarColor());

		if (modeInfo->toolBarFont().fontColor.isUsed())
		{
			palette.setColor(QColorGroup::Text, modeInfo->toolBarFont().fontColor);
			palette.setColor(QColorGroup::ButtonText, modeInfo->toolBarFont().fontColor);
			palette.setColor(QColorGroup::Foreground, modeInfo->toolBarFont().fontColor);
		}

		toolBar_->setPalette(palette);
		subBar_->setPalette(palette);
	}
	else
	{
		toolBar_->unsetPalette();
		subBar_->unsetPalette();
	}

	if (modeInfo->toolBarFont().font.isUsed())
	{
		toolBar_->setFont(modeInfo->toolBarFont().font);
		subBar_->setFont(modeInfo->toolBarFont().font);
	}
	else
	{
		toolBar_->unsetFont();
		subBar_->unsetFont();
	}

	if (modeInfo->filterBoxBackgroundColor().isUsed())
	{
		QPalette palette(modeInfo->filterBoxBackgroundColor());

		if (modeInfo->filterBoxFont().fontColor.isUsed())
		{
			palette.setColor(QColorGroup::Text, modeInfo->filterBoxFont().fontColor);
			palette.setColor(QColorGroup::Foreground, modeInfo->filterBoxFont().fontColor);
		}

		filterEditBox_->setPalette(palette);
	}
	else
	{
		filterEditBox_->unsetPalette();
	}

	if (modeInfo->filterBoxFont().font.isUsed())
		filterEditBox_->setFont(modeInfo->filterBoxFont().font);
	else
		filterEditBox_->unsetFont();

#ifndef QTOPIA
	// Workaround for realignment issues in Qt3
	toolBar_->hide();
	toolBar_->show();
#endif
}

void Quasar::incDisplaySize()
{
	if (qConfig.infoMode == SkinManager::BigMode) return;
	if (qConfig.infoMode == SkinManager::SmallMode)
		qConfig.infoMode = SkinManager::NormalMode;
	else
		qConfig.infoMode = SkinManager::BigMode;
	setPlayListFont(qConfig.infoMode);
}

void Quasar::decDisplaySize()
{
	if (qConfig.infoMode == SkinManager::SmallMode) return;
	if (qConfig.infoMode == SkinManager::BigMode)
		qConfig.infoMode = SkinManager::NormalMode;
	else
		qConfig.infoMode = SkinManager::SmallMode;
	setPlayListFont(qConfig.infoMode);
}

void Quasar::changeOrientation(bool landscape, bool setSkinStyle, bool saveCurrentColumnWidths)
{
	if (landscape)
	{
		if (setSkinStyle)
			skinManager_->setStyle(SkinManager::Landscape);

		if (saveCurrentColumnWidths)
			playList_->saveColumnWidth(qGlobalConfig.configFilename(), "PlayListPortrait");

		// TODO: Remove hardcoded default column widths to some other place...
		const int defaultColumnWidth[6] = {20, 42, 182, 154, 162, 54};
		for (int i = 0; i < playList_->columns(); ++i)
			playList_->setColumnWidth(playList_->header()->mapToSection(i), defaultColumnWidth[i]);

		playList_->loadColumnWidth(qGlobalConfig.configFilename(), "PlayListLandscape");
	}
	else
	{
		if (setSkinStyle)
			skinManager_->setStyle(SkinManager::Portrait);

		if (saveCurrentColumnWidths)
			playList_->saveColumnWidth(qGlobalConfig.configFilename(), "PlayListLandscape");

		// TODO: Remove hardcoded default column widths to some other place...
		const int defaultColumnWidth[6] = {24, 40, 107, 105, 119, 48};
		for (int i = 0; i < playList_->columns(); ++i)
			playList_->setColumnWidth(playList_->header()->mapToSection(i), defaultColumnWidth[i]);

		playList_->loadColumnWidth(qGlobalConfig.configFilename(), "PlayListPortrait");
	}

	if (centralWidget())
	{
		centralWidget()->updateGeometry();
		if (centralWidget()->layout())
			centralWidget()->layout()->activate();
	}

	currentWidth_ = qApp->desktop()->width();
}

void Quasar::checkOrientation()
{
	if (qApp->desktop()->width() != currentWidth_)
	{
		changeOrientation(qApp->desktop()->width() > qApp->desktop()->height());
	}
}

void Quasar::toggleFullScreen()
{
	if (!isFullScreen_)
	{
		showFullScreen();
		isFullScreen_ = true;
		wasMaximized_ = isMaximized();
	}
	else
	{
		// necessary to switch back to normal mode in a clean fashion
		// otherwise the fullscreen mode won't work again.
		showNormal();
#ifdef QTOPIA
		 // on Qtopia force maximized view immediately after switching back to normal mode...
		showMaximized();
#endif

		if (wasMaximized_)
			showMaximized();

		isFullScreen_ = false;
	}
}

void Quasar::toggleOverviewVisibility()
{
	setOverviewVisibility(!playListOverview_->isVisible());
}

void Quasar::updateSplitterSectionHeights()
{
	// Dirty hack to get actual total size...
	int totalSize = 0;

	for (int i = 0; i < verticalSplitterContainer_->sizes().count(); ++i)
		totalSize += verticalSplitterContainer_->sizes()[i];

	// create list of heights for each of the two widgets in the splitter container
	// that is: PlayListOverview and PlayList
	QValueList<int> sizes;
	sizes.append(playListOverview_->isVisible() ? qConfig.overviewSplitterPosition : 0);
	sizes.append(coverArtFlowArea_->isVisible() ? qConfig.coverArtFlowSplitterPosition : 0);
	sizes.append(totalSize - sizes[0] - sizes[1]);

	verticalSplitterContainer_->setSizes(sizes);
}

void Quasar::setOverviewVisibility(bool visible)
{
	setActiveView(Quasar::ViewLastActivePlaylist);

	if (visible)
	{
		playListOverview_->show();
		updateSplitterSectionHeights();
		playListOverview_->setFocus();
	}
	else
	{
		// only set current splitter position if playlist overview is visible...
		if (playListOverview_->isVisible())
			qConfig.overviewSplitterPosition = verticalSplitterContainer_->sizes()[0];

		if (coverArtFlowArea_->isVisible())
			qConfig.coverArtFlowSplitterPosition = verticalSplitterContainer_->sizes()[1];

		playListOverview_->hide();
		updateSplitterSectionHeights();
		playList_->setFocus();
	}

	qConfig.isOverviewVisible = visible;
	toolPanel_->setOverviewVisibility(visible);
}

void Quasar::toggleCoverArtFlowVisibility()
{
	setCoverArtFlowVisibility(!coverArtFlowArea_->isVisible());
}

void Quasar::setCoverArtFlowVisibility(bool visible)
{
	setActiveView(Quasar::ViewLastActivePlaylist);

	if (visible)
	{
		coverArtFlowArea_->show();
		coverArtFlow_->setActive(true);
		updateSplitterSectionHeights();
		coverArtFlowArea_->setFocus();
	}
	else
	{
		// only set current splitter position if playlist overview is visible...
		if (playListOverview_->isVisible())
			qConfig.overviewSplitterPosition = verticalSplitterContainer_->sizes()[0];

		if (coverArtFlowArea_->isVisible())
			qConfig.coverArtFlowSplitterPosition = verticalSplitterContainer_->sizes()[1];

		coverArtFlowArea_->hide();
		coverArtFlow_->setActive(false);
		updateSplitterSectionHeights();
		playList_->setFocus();
	}

	qConfig.isCoverArtFlowVisible = visible;
	toolPanel_->setCoverArtFlowVisibility(visible);
}

void Quasar::loadSkin(const QString &name, bool force)
{
	if (skinManager_->activeSkin() != name || force)
	{
		operationPanel_->hide();
		skinManager_->load(name);

		if (qApp->desktop()->width() > qApp->desktop()->height())
			skinManager_->setStyle(SkinManager::Landscape);
		else
			skinManager_->setStyle(SkinManager::Portrait);

		setPlayListFont(qConfig.infoMode);
		coverArtProviderCoverArtFlow_->setPreloadImageFilename(skinManager_->activeSkinDirectory() + "/nocover.png");
		coverArtProviderPlayInfo_->setPreloadImageFilename(skinManager_->activeSkinDirectory() + "/nocover.png");

		operationPanel_->show();
		operationPanel_->updateGeometry();
		repaint();
	}
}

void Quasar::loadPrimarySkin()
{
	loadSkin(qConfig.skin1, true);
}

void Quasar::loadSecondarySkin()
{
	loadSkin(qConfig.skin2, true);
}

void Quasar::playlistViewUpdated()
{
	if (playList_->itemCount() == 0)
		toolPanel_->setInfoText("No titles found.");
	else
	{
		QString filesizeDisplay = formatFileSize(playList_->totalFileSize());
		QString timeDisplay = formatPlayTime(playList_->totalPlaytime());

		if (playList_->itemCount() == 1)
			toolPanel_->setInfoText(QString().number(playList_->itemCount()) + " title, " + timeDisplay + ", " + filesizeDisplay);
		else
			toolPanel_->setInfoText(QString().number(playList_->itemCount()) + " titles, " + timeDisplay + ", " + filesizeDisplay);
	}

	updateButtonAndMenuState();
}

void Quasar::playlistViewSorted()
{
	for (int i = 0; i < PlayList::COLUMN_MAX; ++i)
		sortByMenu_->setItemChecked(i, i == playList_->sortedColumn() ? true : false);

	updateButtonAndMenuState();
}

void Quasar::playlistManagerPlayListFileNameChanged()
{
	setCaption(QString(APPNAME) + " - " + playListManager_->getDisplayPlayListFilename());
}

void Quasar::playlistManagerPlayListCleared()
{
	resetFilter();
	updateButtonAndMenuState();
}

void Quasar::playlistOverviewFocusOutLeft()
{
	filterEditBox_->setFocus();
}

void Quasar::playlistOverviewFocusOutRight()
{
	playList_->setFocus();
}

void Quasar::setFocusPlayListOverview()
{
	if (!playListOverview_->isVisible())
		setOverviewVisibility(true);

	playListOverview_->setFocus();
}

Quasar::View Quasar::activeView()
{
	if (widgetStack_->visibleWidget() == playInfo_)
		return Quasar::ViewPlayInfo;
	else
	{
		switch (playListManager_->activeMode())
		{
			case PlayListManager::ModeDynPlaylistEditor:
				return Quasar::ViewDynPlaylistEditor;
			case PlayListManager::ModeOpenedPlaylist:
				return Quasar::ViewOpenedPlaylist;
			case PlayListManager::ModeOnTheGoPlaylist:
				return Quasar::ViewOnTheGoPlaylist;
			default:
				return Quasar::ViewLastActivePlaylist;
		}
	}
}

bool Quasar::setActiveView(View view)
{
	bool result = false;

	if (view == activeView())
		return true;

	if (view == Quasar::ViewPlayInfo)
	{
		toolBar_->hide();
		toolPanel_->hide();
		operationPanel_->hide();
		subBar_->hide();

		playListManager_->setActiveMode(PlayListManager::ModeLastActivePlaylist);
		widgetStack_->raiseWidget(InfoWidget);
		toolPanel_->setView(ToolPanel::PlayInfo);
	}
	else
	{
		toolBar_->show();
		toolPanel_->show();
		operationPanel_->show();
		subBar_->show();

		PlayListManager::Mode mode = PlayListManager::ModeLastActivePlaylist;

		switch (view)
		{
			case Quasar::ViewDynPlaylistEditor:
				mode = PlayListManager::ModeDynPlaylistEditor;
				break;
			case Quasar::ViewOpenedPlaylist:
				mode = PlayListManager::ModeOpenedPlaylist;
				break;
			case Quasar::ViewOnTheGoPlaylist:
				mode = PlayListManager::ModeOnTheGoPlaylist;
				break;
			default:
				mode = PlayListManager::ModeLastActivePlaylist;
		}

		// this will fire playlistManagerSwitchingModeTo below...
		// play info widget is taken care of there...
		playListManager_->setActiveMode(mode);
	}

	updateButtonAndMenuState();

	return result;
}

void Quasar::playlistManagerSwitchingModeTo(PlayListManager::Mode mode)
{
	// make sure play info widget is lower than playlist...
	if (widgetStack_->visibleWidget() == playInfo_)
	{
		widgetStack_->raiseWidget(PlayListWidget);
		toolPanel_->setView(ToolPanel::PlayList);

		// if we are switching because the filter edit box was activated,
		// keep the focus in there, otherwise shift focus to playlist widget...
		if (qApp->focusWidget() != filterEditBox_)
			playList_->setFocus();
	}
}

void Quasar::playlistManagerSwitchedModeTo(PlayListManager::Mode mode)
{
	playlistManagerPlayListFileNameChanged();
	playListToolPanel_->setMode(mode);
	updateButtonAndMenuState();
}

void Quasar::playlistToolPanelModeChanged(PlayListManager::Mode mode)
{
	playListManager_->setActiveMode(mode);
}

void Quasar::toggleInfoWidget()
{
	if (activeView() != Quasar::ViewPlayInfo)
		setActiveView(Quasar::ViewPlayInfo);
	else
		setActiveView(Quasar::ViewLastActivePlaylist);
}

void Quasar::switchPlaylistEditorView()
{
	setActiveView(Quasar::ViewDynPlaylistEditor);
}

void Quasar::switchToLastActivePlayList()
{
	setActiveView(Quasar::ViewLastActivePlaylist);
}

void Quasar::switchToCurrentPlayList()
{
	setActiveView(Quasar::ViewOpenedPlaylist);
}

void Quasar::switchToOnTheGoPlayList()
{
	setActiveView(Quasar::ViewOnTheGoPlaylist);
}

void Quasar::switchToInfoWidget()
{
	setActiveView(Quasar::ViewPlayInfo);
}

void Quasar::setInputMode(PlayList::InputMode inputMode)
{
	qConfig.inputMode = inputMode;
	playListManager_->setInputMode(inputMode);
	playListToolPanel_->setInputMode(inputMode);
}

void Quasar::resetFilterAndSorting()
{
	playList_->resetSorting();
	playListOverview_->reset();
	resetFilter();
	updateButtonAndMenuState();
}
