/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "customscaledcachedimageprovider.h"
#include "debug.h"

#include <qfile.h>
#include <qbuffer.h>
#include <qdatetime.h>

const int sizeScalesCount = 7;
const float sizeScales[sizeScalesCount] = { 1, 2.0/3, 1.0/2, 1.0/3, 1.0/4, 1.0/5, 1.0/6};


CustomScaledCachedImageProvider::CustomScaledCachedImageProvider(QObject *owner, const char *name)
	:	CustomCachedImageProvider(owner, name),
		scale_(1),
		scaleSizeBlock_(0)
{
	
}

void CustomScaledCachedImageProvider::setScale(float scale)
{
	scale_ = scale;
	
	//qDebug("scale_: %f", scale_);
	
	int newScaleSizeBlock = scaleSizeBlock_;
	
	for (int i = sizeScalesCount - 1; i >= 0; --i)
	{
		//qDebug("sizeScales[%d]: %f", i, sizeScales[i]);
		if (sizeScales[i] > scale)
		{
			newScaleSizeBlock = i;
			break;
		}
	}

	//qDebug("scaleSizeBlock_: %d, newScaleSizeBlock: %d", scaleSizeBlock_, newScaleSizeBlock);	
	
	if (newScaleSizeBlock != scaleSizeBlock_)
	{
		scaleSizeBlock_ = newScaleSizeBlock;
		
		flushMemoryCache();
		emit scaleChangedAndFlushed();
	}
}

void CustomScaledCachedImageProvider::threadedSaveImageToCacheDirectory(CachedImage *cachedImage)
{
	DPRINTF("threadedSaveImageToCacheDirectory");

	//DTIMERINIT(savetimer);
	
	if (cachedImage)
	{
		if (cachedImage->image())
		{
			DPRINTF("Saving image '%s' to '%s'...", (const char *)(cachedImage->name().utf8()), (const char*)(cachedImage->cacheFileName().utf8()));
			//cachedImage->image()->save(cachedImage->cacheFileName(), "JPEG", 70);
			
			QFile file(cachedImage->cacheFileName());
			file.open(IO_WriteOnly);
			
			QImageIO iio(&file, "JPEG");
		
			int cx = cachedImage->image()->width();
			int cy = cachedImage->image()->height();
			
			int offsets[sizeScalesCount];
		
			for (int i = 0; i < sizeScalesCount; ++i)
			{
				iio.setImage(
					i == 0 ? 
					*cachedImage->image() : 
					cachedImage->image()->smoothScale((int)((float)cx * sizeScales[i]), (int)((float)cy * sizeScales[i]))
				);
				
				// increase JPEG quality for smaller images...
				int quality = 80 + 3 * i;
				
#ifdef QTOPIA
				iio.setParameters(QString::number(quality).latin1());
#else
				iio.setQuality(quality);
#endif
				
				iio.write();
				offsets[i] = file.at();
			}

			// Write offsets as suffix...
			file.writeBlock((const char *)offsets, sizeScalesCount * sizeof(int));
			file.writeBlock((const char *)&cx, sizeof(int));
			file.writeBlock((const char *)&cy, sizeof(int));
			file.writeBlock("CBCI", 4);
			file.close();
		}
		else
			DPRINTF("Can't save null image '%s'...", (const char *)(cachedImage->name().utf8()));
	}
	else
		DPRINTF("UH OH, cachedImage isn't assigned!!!");
	
	//DPRINTF("Timer threadedSaveImageToCacheDirectory...");
	//DTIMERPRINT(savetimer, "threadedSaveImageToCacheDirectory");
}


QImage* CustomScaledCachedImageProvider::loadImageFromCacheDirectory(const QString &cacheFileName)
{
    DTIMERINIT(timer);
	
    QFile file(cacheFileName);
    file.open(IO_ReadOnly);

    char signature[4];
    file.at(file.size() - 4);
    file.readBlock(signature, 4);
    
    if (memcmp(signature, "CBCI", 4) != 0)
    {
    	qWarning("No valid signature found in cache file %s. Deleting file.", (const char*)cacheFileName.latin1());
	    QFile::remove(cacheFileName);
    	return new QImage();
    }
    
    // Read offsets from the end of the file...
    int offsets[sizeScalesCount];
    file.at(file.size() - sizeScalesCount * sizeof(int) - 2 * sizeof(int) - 4);
    file.readBlock((char *)offsets, sizeScalesCount * sizeof(int));

    // Read size of the biggest image...
    int cx, cy;
    file.readBlock((char *)&cx, sizeof(int));
    file.readBlock((char *)&cy, sizeof(int));
       
    // Determine the block to use...
    int block = scaleSizeBlock_;
    
    // Determine the start and end offsets of the source block...
    int startOffset = 0;
    int endOffset;
    
    if (block > 0)
    {
    	startOffset = offsets[block - 1];
    	endOffset = offsets[block];
    }
    else
    	endOffset = offsets[0];
    
    // Read content of block into our buffer...
    int blockLength = endOffset - startOffset;
    QByteArray content(blockLength);
    
    file.at(startOffset);    
    file.readBlock(content.data(), blockLength);
    file.close();
    
    // Read image from buffer...    
    QBuffer buffer;
    buffer.open(IO_ReadWrite);
    buffer.writeBlock(content, content.size());
    buffer.at(0);
    
    QImageIO iio(&buffer, "JPEG");
    
	DPRINTF("Fastloading Image '%s'...", (const char *)(cacheFileName.utf8()));
    
#ifdef QTOPIA
   	//iio.setParameters(QString("Fast Shrink( %1 )").arg(scaleDenominator_).latin1());
	iio.setParameters("Fast");
#else
    //iio.setParameters(QString("Scale( %1, %2, ScaleMax )").arg(dstWidth).arg(dstHeight).latin1());
#endif
    
   	QImage *dstImage = new QImage();
   	
    bool result = iio.read();
    if (result)
    	*dstImage = iio.image();
    
    DTIMERPRINT(timer, "fastLoadJPEGImage");
    
    return dstImage;
}
