/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef CACHEDIMAGEPROVIDER_H_INCLUDED
#define CACHEDIMAGEPROVIDER_H_INCLUDED

#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qwidget.h>
#include <qimage.h>
#include <qdict.h>
#include <qcache.h>
#include <qthread.h>
#include <qtimer.h>

#include "backgroundtasks.h"

//#define DEBUG_ACTIVE_CACHEDIMAGES

class QImage;
class CustomCachedImageProvider;
class Task;
class ImageLoadTask;
class ImageSaveTask;

class CachedImage
{
public:
	CachedImage(CustomCachedImageProvider *owner, QString name, QImage *image, void *data);
	virtual ~CachedImage();

	CustomCachedImageProvider *owner() { return owner_; }
	QString name() { return name_; }
	QImage *image() { return image_; }
	void *data() { return data_; }
	bool finished() { return finished_; }
	bool readFromCache() { return readFromCache_; }
	TaskProcessingController::TaskPriority priority() { return priority_; }
	Task *currentTask() { return currentTask_; }
	QString cacheFileName() { return cacheFileName_; }

	void setName(const QString& name) { name_ = name; }
	void setImage(QImage *image) { image_ = image; }
	void setData(void *data) { data_ = data; }
	void setFinished(bool isFinished) { finished_ = isFinished; }
	void setReadFromCache(bool value) { readFromCache_ = value; }
	void setPriority(TaskProcessingController::TaskPriority priority) { priority_ = priority; }
	void setCurrentTask(Task* task) { currentTask_ = task; }
	void setCacheFileName(const QString& name) { cacheFileName_ = name; }
private:
	CustomCachedImageProvider *owner_;
	QString name_;
	QImage *image_;
	void *data_;
	bool finished_;
	bool readFromCache_;
	TaskProcessingController::TaskPriority priority_;
	Task *currentTask_;
	QString cacheFileName_;
};

class CacheImageCache : public QCache<CachedImage>
{
public:
	virtual ~CacheImageCache();
protected:
	virtual void deleteItem(Item d);
};

class CustomCachedImageProvider : public QObject
{
	Q_OBJECT;
public:
	friend class CacheImageCache;
	friend class CachedImage;
	friend class ImageLoadTask;
	friend class ImageSaveTask;

	CustomCachedImageProvider(QObject *owner, const char *name = 0);
	virtual ~CustomCachedImageProvider();

	void setImageCacheDirectory(const QString &directory);
	QString imageCacheDirectory() { return imageCacheDirectory_; }

	void setMemoryCacheSize(int entries)
	{
		if (entries != imageCache_.maxCost())
			imageCache_.setMaxCost(entries);
	}
	int memoryCacheSize() { return imageCache_.maxCost(); }

	bool imageExists(const QString &name, bool searchNullImageList = true);
	bool containsData(void *data);
	QImage *getImage(const QString &name, void *data = NULL);
	void queue(const QString &name, void *data = NULL);

	void prefetchHint(const QStringList &names);

	bool isWorking() { return !backgroundTasks_->idle(); }

	void lock();
	void unlock();

	void resetNullImageNameList() { nullImageNameList_.clear(); }

	void flushMemoryCache();
	void flushCache();

	QObject *owner() { return owner_; }

signals:
	void imageCached(CachedImage *cachedImage);
	void imageIsNull(CachedImage *cachedImage);
	void imageSaved(CachedImage *cachedImage);

protected:
	virtual QImage* threadedCreateImage(CachedImage *cachedImage) = 0;
	virtual void threadedSaveImageToCacheDirectory(CachedImage *cachedImage) = 0;
	virtual QImage* preloadImage(const QString &name, void *data = NULL) = 0;
	virtual QImage* nullImage(const QString &name, void *data = NULL) { return preloadImage(name, data); }
	virtual QImage* loadImageFromCacheDirectory(const QString &cacheFileName);
	virtual void initializeCachedImage(CachedImage *cachedImage);
	virtual void finalizeCachedImage(CachedImage *cachedImage);

private slots:
	void eventPollTimerTimedOut();
	void prefetchTimerTimeOut();

#ifdef DEBUG_ACTIVE_CACHEDIMAGES

private:
	QList<CachedImage> activeCachedImages_;
	QMutex mutexActiveCachedImages_;
	void debugPrintActiveCachedImages();

#endif

private:
	QObject *owner_;
	QString imageCacheDirectory_;
	CacheImageCache imageCache_;
	QMutex mutexCache_;

	QDict<void> nullImageNameList_;

	QDict<QTime> prefetchedNamesTimestampList_;
	QStringList prefetchNamesList_;
	bool prefetching_;
	QTimer prefetchTimer_;

	QDict<CachedImage> imagesWaitList_;
	QMutex mutexImagesWaitList_;
	QMutex mutexCacheImageSharedDataAccess_;

#ifdef QTOPIA
	QTimer eventPollTimer_;
#endif

	TaskProcessingController *backgroundTasks_;

	QString buildCacheFilename(const QString &name);

	QImage *internalGetImage(const QString &name, void *data, bool returnImage, TaskProcessingController::TaskPriority prio);
	void freeCachedImage(CachedImage *cachedImage);

	void loadTaskFinished(ImageLoadTask *task);
	void saveTaskFinished(ImageSaveTask *task);

	void cachedImageRemovedFromCache(CachedImage *cachedImage);
};

#endif /*CACHEDIMAGEPROVIDER_H_INCLUDED*/
