/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include "paintbox.h"
#include <qpainter.h>
#include <qdatetime.h>

#ifdef QTOPIA
#include <qdirectpainter_qws.h>
#define QTOPIA_OPTIMIZED_BLITTER
#endif

//#define DEBUG_TIME_PAINT
//#define DEBUG_DRAW_RANDOM_COLOR

#define SDL_memset4(dst, val, len)		\
do {						\
	unsigned _count = (len);		\
	unsigned _n = (_count + 3) / 4;		\
	uint *_p = (uint *)(dst);		\
	uint _val = (val);			\
        switch (_count % 4) {			\
        case 0: do {    *_p++ = _val;		\
        case 3:         *_p++ = _val;		\
        case 2:         *_p++ = _val;		\
        case 1:         *_p++ = _val;		\
		} while ( --_n );		\
	}					\
} while(0)

void imageFillRect(QImage &dst, int x1, int y1, int x2, int y2, const QColor &color)
{
	if (dst.bits())
	{
		int stride = dst.bytesPerLine() / 4;
		int length = (x2 - x1);
		uint value = color.rgb();
		uint *bits = (uint *)dst.bits();

		//qDebug("coords: %d, %d, %d, %d", x1, y1, x2, y2);

		for (int y = y1; y < y2; ++y)
		{
			SDL_memset4(&bits[y * stride + x1], value, length);
		}
	}
}

void imageFillRectS(QImage &dst, int x1, int y1, int x2, int y2, const QColor &color)
{
	if ((x2 > x1) && (y2 > y1) &&
		(x1 < dst.width()) && (y1 < dst.height()) &&
		(x2 > 0) && (y2 > 0))
	{
		if (x1 < 0)
			x1 = 0;

		if (y1 < 0)
			y1 = 0;

		if (x2 > dst.width())
			x2 = dst.width();

		if (y2 > dst.height())
			y2 = dst.height();

		//qDebug("coords: %d, %d, %d, %d", x1, y1, x2, y2);

		imageFillRect(dst, x1, y1, x2, y2, color);
	}
}

void imageFillRectS(QImage &dst, const QRect &rect, QColor color)
{
	imageFillRectS(dst, rect.left(), rect.top(), rect.right() + 1, rect.bottom() + 1, color);
}

// Special optimized blit for RGBx888 --> RGB565
// The unrotated blit is loosely based on the SDL blit function...
#define HI 1
#define LO 0

#define RGB888_RGB565(dst, src) { \
	*(ushort *)(dst) = \
		(ushort)((((*src) & 0x00F80000) >> 8) | \
				(((*src) & 0x0000FC00) >> 5) | \
				(((*src) & 0x000000F8) >> 3)); \
}
#define RGB888_RGB565_TWO(dst, src) { \
	*(uint *)(dst) = \
		(((((src[HI]) & 0x00F80000) >> 8) | \
		(((src[HI]) & 0x0000FC00) >> 5) | \
		(((src[HI]) & 0x000000F8) >> 3)) << 16) | \
		(((src[LO]) & 0x00F80000) >> 8) | \
		(((src[LO]) & 0x0000FC00) >> 5) | \
		(((src[LO]) & 0x000000F8) >> 3); \
}

#define RGB888_RGB565_TWO_2Param(dst, src1, src2) { \
	dst = \
		(((((src1) & 0x00F80000) >> 8) | \
		(((src1) & 0x0000FC00) >> 5) | \
		(((src1) & 0x000000F8) >> 3)) << 16) | \
		(((src2) & 0x00F80000) >> 8) | \
		(((src2) & 0x0000FC00) >> 5) | \
		(((src2) & 0x000000F8) >> 3); \
}

// High Perfomance RGB888 -> RGB565 converter
static void Blit_RGB888_RGB565(uint *srcBits, int srcWidth, int srcHeight, int srcStride, ushort *dstBits, int dstStride)
{
	uint *src = srcBits;
	ushort *dst = dstBits;

	int srcskip = srcStride - srcWidth;
	int dstskip = dstStride - srcWidth;

	int x;

	// Is dstBits pointer word-aligned?
	// If not we need to take special care...
	if (uint(dstBits) & 3)
	{
		if (srcWidth == 0)
			return;

		--srcWidth;

		while (srcHeight--)
		{
			// Copy one pixel to align on word-boundary...
			RGB888_RGB565(dst, src);
			++src;
			++dst;

			// Copy 4 pixels at a time...
			for (x = srcWidth / 4; x; --x)
			{
				RGB888_RGB565_TWO(dst, src);
				src += 2;
				dst += 2;
				RGB888_RGB565_TWO(dst, src);
				src += 2;
				dst += 2;
			}

			switch (srcWidth & 3)
			{
			case 3:
				RGB888_RGB565(dst, src);
				++src;
				++dst;
			case 2:
				RGB888_RGB565_TWO(dst, src);
				src += 2;
				dst += 2;
				break;
			case 1:
				RGB888_RGB565(dst, src);
				++src;
				++dst;
				break;
			}
			src += srcskip;
			dst += dstskip;
		}
	}
	else
	{
		while (srcHeight--)
		{
			// Copy 4 pixels at a time...
			for (x = srcWidth / 4; x; --x)
			{
				RGB888_RGB565_TWO(dst, src);
				src += 2;
				dst += 2;
				RGB888_RGB565_TWO(dst, src);
				src += 2;
				dst += 2;
			}

			switch (srcWidth & 3)
			{
			case 3:
				RGB888_RGB565(dst, src);
				++src;
				++dst;
			case 2:
				RGB888_RGB565_TWO(dst, src);
				src += 2;
				dst += 2;
				break;
			case 1:
				RGB888_RGB565(dst, src);
				++src;
				++dst;
				break;
			}
			src += srcskip;
			dst += dstskip;
		}
	}
}

// High Performance RGB888 -> RGB565 converter + Rotation by 270 degree
//*
static void Blit_RGB888_RGB565_Rot270(uint *srcBits, int srcWidth, int srcHeight, int srcStride, ushort *dstBits, int dstStride)
{
	uint *src = srcBits;
	ushort *dst = dstBits;

	register int neighborPixelAbove;
	register int currentPixel;

	int heightm = srcHeight - 1;
	int widthm = srcWidth - 1;

	int y = heightm;
	int x;

	// Check if the dstBits pointer is word-aligned (4 bytes)
	// If not process a single pixel / line,
	// so we can process two pixels at once in the while loop...
	if (uint(dstBits) & 3)
	{
		src = srcBits + srcStride * y;
		dst = dstBits + heightm - y;
		--y;

		x = srcWidth;
		while (x--)
		{
			RGB888_RGB565(dst, src);
			++src;
			dst += dstStride;
		}
	}

	// Can we process two source pixels at a time?
	while (y > 0)
	{
		src = srcBits + srcStride * y;
		dst = dstBits + heightm - y;
		y -= 2;

		x = srcWidth;
		while (x--)
		{
			neighborPixelAbove = *(src - srcStride);
			currentPixel = *src;
			++src;
			RGB888_RGB565_TWO_2Param(*(uint *)dst, neighborPixelAbove, currentPixel);
			dst += dstStride;
		}
	}

	// Process remaining line...
	if (y == 0)
	{
		src = srcBits;
		dst = dstBits + heightm;

		x = srcWidth;
		while (x--)
		{
			RGB888_RGB565(dst, src);
			++src;
			dst += dstStride;
		}
	}
}
//*/

// High Performance RGB888 -> RGB565 converter + Rotation by 270 degree
/*
static void Blit_RGB888_RGB565_Rot270(uint *srcBits, int srcWidth, int srcHeight, int srcStride, ushort *dstBits, int dstStride)
{
	uint *src = srcBits;
	ushort *dst = dstBits;

	register int neighborPixelAbove;
	register int currentPixel;

	int y = srcHeight;
	int x;

	// Can we process two source pixels at a time?
	while (y > 0)
	{
		dst = dstBits + srcHeight - y;
		--y;
		src = srcBits + srcStride * y;

		x = srcWidth;
		while (x--)
		{
			RGB888_RGB565(dst, src);
			++src;
			dst += dstStride;
		}
	}
}
//*/

PaintBox::PaintBox(QWidget *parent, const char *name, WFlags f)
	:	QWidget(parent, name, f),
		buffer_(),
		bufferDirty_(true),
		displayPixmap_()
{
	setBackgroundMode(NoBackground);
#ifdef QTOPIA
	setWFlags(getWFlags() | Qt::WNorthWestGravity | Qt::WRepaintNoErase | Qt::WResizeNoErase);
#else
	setWFlags(getWFlags() | Qt::WStaticContents | Qt::WNoAutoErase);
#endif
}

void PaintBox::invalidate(const QRect &rect)
{
	bufferDirty_ = true;
	update(rect);
}

void PaintBox::invalidate()
{
	bufferDirty_ = true;
	update();
}

#ifdef DEBUG_TIME_PAINT
static int lastTime;
#endif

void PaintBox::paintEvent(QPaintEvent *event)
{
	if (bufferDirty_)
	{
		QArray<QRect> rects = event->region().rects();

		//qDebug("rects.count(): %d", rects.count());

		for (QArray<QRect>::ConstIterator it = rects.begin(); it != rects.end(); ++it)
		{
			//qDebug("rect: %d, %d, %d, %d", (*it).left(), (*it).top(), (*it).width(), (*it).height());
#ifdef DEBUG_DRAW_RANDOM_COLOR
			imageFillRectS(buffer_, *it, QColor(QRgb(rand())));
#else
			imageFillRectS(buffer_, *it, QColor(0, 0, 0));
#endif

			paintToBuffer(buffer_, *it);

#ifdef QTOPIA_OPTIMIZED_BLITTER
			QDirectPainter dp(this);
			QRect r = *it;

			int srcStride = buffer_.bytesPerLine() / 4;
			uint *srcBits = (uint *)buffer_.bits() + r.top() * srcStride + r.left();

			int dstStride = displayPixmap_.bytesPerLine() / 2;

			if (dp.transformOrientation() == 3)
			{
				ushort *dstBits = (ushort *)displayPixmap_.scanLine(0) + r.left() * dstStride + buffer_.height() - r.bottom() - 1;
				Blit_RGB888_RGB565_Rot270(srcBits, r.width(), r.height(), srcStride, dstBits, dstStride);
			}
			else if (dp.transformOrientation() == 0)
			{
				ushort *dstBits = (ushort *)displayPixmap_.scanLine(0) + r.top() * dstStride + r.left();
				Blit_RGB888_RGB565(srcBits, r.width(), r.height(), srcStride, dstBits, dstStride);
			}
#endif
		}

#ifdef DEBUG_DRAW_RANDOM_COLOR
		for (QArray<QRect>::ConstIterator it = rects.begin(); it != rects.end(); ++it)
		{
			QPainter pi(&displayPixmap_);
			pi.setPen(QPen(QColor(QRgb(rand())), 1, Qt::DashDotLine));
			pi.drawRect(*it);
		}
#endif

		QPainter pi(&displayPixmap_);
		pi.setClipRegion(event->region());
#ifndef QTOPIA_OPTIMIZED_BLITTER
		pi.drawImage(QPoint(0, 0), buffer_);
#endif
		paintOverlay(pi, displayPixmap_);

		bufferDirty_ = false;
	}

	QPainter p(this);
	p.setClipRegion(event->region());
	p.drawPixmap(QPoint(0, 0), displayPixmap_);

#ifdef DEBUG_TIME_PAINT
	QTime t = QTime::currentTime();
	int time = t.minute() * 60 * 1000 + t.second() * 1000 + t.msec();
	qDebug("PaintBox::paintEvent: %d (Delta %d)", time, time - lastTime);
	lastTime = time;
#endif
}

void PaintBox::resizeBuffer()
{
	int newWidth = width();
	int newHeight = height();

	const int bufferOversize = 64;

	int bufferWidth = buffer_.width();
	int bufferHeight = buffer_.height();

	if (newWidth > bufferWidth)
		bufferWidth = newWidth + bufferOversize;
	else if (newWidth < bufferWidth - bufferOversize)
		bufferWidth = newWidth;

	if (bufferWidth < 1)
		bufferWidth = 1;

	if (newHeight > bufferHeight)
		bufferHeight = newHeight + bufferOversize;
	else if (newHeight < bufferHeight - bufferOversize)
		bufferHeight = newHeight;

	if (bufferHeight < 1)
		bufferHeight = 1;

	if ((bufferWidth != buffer_.width()) ||
		(bufferHeight != buffer_.height()))
	{
		int oldWidth = buffer_.width();
		int oldHeight = buffer_.height();

		qDebug("Old buffer size: %d x %d, New buffer size: %d x %d", oldWidth, oldHeight, bufferWidth, bufferHeight);

		displayPixmap_.resize(bufferWidth, bufferHeight);
		buffer_.create(bufferWidth, bufferHeight, 32);

		bufferResized(oldWidth, oldHeight);
	}
}

void PaintBox::bufferResized(int oldWidth, int oldHeight)
{
	// Descendants implement buffer painting...
}

void PaintBox::paintToBuffer(QImage &buffer, const QRect &clipRect)
{
	// Descendants implement buffer painting...
}

void PaintBox::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	// Descendants implement overlay painting...
}

void PaintBox::resizeEvent(QResizeEvent* event)
{
	resizeBuffer();
	QWidget::resizeEvent(event);
	repaint();
}


/* Layer */

Layer::Layer(LayeredPaintBox *parent)
	:	visible_(true)
{
	parent_ = parent;
	if (parent_)
	{
		int index = parent_->layers().findRef(this);
		if (index == -1)
			parent_->layers().append(this);
	}
}

Layer::~Layer()
{
	if (parent_)
		parent_->layers().removeRef(this);

	invalidate();
}


/* LayeredPaintBox */

LayeredPaintBox::LayeredPaintBox(QWidget *parent, const char *name, WFlags f)
	:	PaintBox(parent, name, f),
		mouseListener_(NULL)
{
	setFocusPolicy(QWidget::StrongFocus);
	layers_.setAutoDelete(true);
}

void LayeredPaintBox::paintToBuffer(QImage &buffer, const QRect &clipRect)
{
	for (Layer *layer = layers_.first(); layer != NULL; layer = layers_.next())
	{
		if (layer->visible())
		{
			QRect intersectedClipRect = clipRect;
			if (layer->paintTest(intersectedClipRect))
				layer->paintToBuffer(buffer, intersectedClipRect);
		}
	}
}

void LayeredPaintBox::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	for (Layer *layer = layers_.first(); layer != NULL; layer = layers_.next())
		if (layer->visible())
			layer->paintOverlay(p, pixmap);
}

Layer *LayeredPaintBox::doHitTest(const QPoint &pos)
{
	Layer *result = NULL;

	for (Layer *layer = layers_.last(); layer != NULL; layer = layers_.prev())
	{
		if (layer->visible() && layer->hitTest(pos))
		{
			result = layer;
			break;
		}
	}

	return result;
}

void LayeredPaintBox::mousePressEvent(QMouseEvent *event)
{
	Layer *layer = doHitTest(event->pos());
	if (layer)
	{
		mouseListener_ = layer;
		layer->mousePressEvent(event);
	}
	else
		PaintBox::mousePressEvent(event);
}

void LayeredPaintBox::mouseMoveEvent(QMouseEvent *event)
{
	if (mouseListener_)
		mouseListener_->mouseMoveEvent(event);
	else
		PaintBox::mouseMoveEvent(event);
}

void LayeredPaintBox::mouseReleaseEvent(QMouseEvent *event)
{
	if (mouseListener_)
		mouseListener_->mouseReleaseEvent(event);
	else
		PaintBox::mouseReleaseEvent(event);

	mouseListener_ = NULL;
}

void LayeredPaintBox::mouseDoubleClickEvent(QMouseEvent *event)
{
	Layer *layer = doHitTest(event->pos());
	if (layer)
	{
		mouseListener_ = layer;
		layer->mouseDoubleClickEvent(event);
	}
	else
		PaintBox::mouseDoubleClickEvent(event);
}

void LayeredPaintBox::resizeEvent(QResizeEvent *event)
{
	invalidate();
	emit rearrangeLayers();
	PaintBox::resizeEvent(event);
}

/* TextLayer */

TextLayer::TextLayer(LayeredPaintBox *parent)
	:	PositionedLayer(parent),
		text_(""),
		font_(),
		color_(255, 255, 255),
		alignment_(Qt::AlignCenter),
		fontMinAutoScale_(false),
		invalid_(true),
		measuredFontSize_(0)
{
}

void TextLayer::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	if (invalid_)
		remeasureFontSize();

	p.setFont(drawFont_);
	p.setPen(color_);
	p.drawText(geometry(), alignment_, text_);
}

void TextLayer::resized()
{
	if (fontMinAutoScale_)
		remeasureFontSize();
}

void TextLayer::remeasureFontSize()
{
	QPainter p(&parent()->displayPixmap());

	if (fontMinAutoScale_)
	{
		drawFont_ = font_;
		measuredFontSize_ = drawFont_.pointSize();
		p.setFont(drawFont_);

		while ((renderedTextSize_ = p.boundingRect(parent()->displayPixmap().rect(), alignment_, text_).size()).width() > width() && measuredFontSize_ > 1)
		{
			measuredFontSize_ -= 1;
			drawFont_.setPointSize(measuredFontSize_);
			p.setFont(drawFont_);
		}
	}
	else
	{
		drawFont_ = font_;
		measuredFontSize_ = drawFont_.pointSize();
		p.setFont(drawFont_);

		renderedTextSize_ = p.boundingRect(parent()->displayPixmap().rect(), alignment_, text_).size();
	}

	invalid_ = false;
}

void TextLayer::invalidate()
{
	invalid_ = true;
	PositionedLayer::invalidate();
}

const QSize TextLayer::renderedTextSize() const
{
	if (invalid_)
	{
		TextLayer *self = const_cast<TextLayer *>(this);
		self->remeasureFontSize();
	}

	return renderedTextSize_;
}
