/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include <qpainter.h>
#include <qapplication.h>
#include <stdlib.h>

#include "debug.h"
#include "paintbox.h"
#include "customperspectiveimageflow.h"
#include "imagefx.h"

//#define DPRINTF qDebug
//#define DEBUG_PAINT_OVERLAY

CustomPerspectiveImageFlow::CustomPerspectiveImageFlow(LayeredPaintBox *parent)
	:	CustomFlow(parent),
	 	offset_(0),
	 	constantOffset_(0),
	 	antialias_(true),
	 	antialiasMode_(AntialiasStaticSceneOnly)
{
	mouseDown_ = false;
	lastPos_ = 0;
	lastDiff_ = -1;
	startPos_ = 0;
	mouseMoved_ = false;
	isScrolling_ = false;
	pixelPerSec_ = 0;
	pixelsLeft_ = 0;

	setTransitionTime(1000);

	setMinDistanceValue(0.2);
	setDistance(20);
	setDistanceSpan(5);

	setMinFlowValue(0.4);
	setFlowSpan(3);
}

void CustomPerspectiveImageFlow::setVerticalOffset(int offset)
{
	if (offset != offset_)
	{
		offset_ = offset;
		invalidate();
	}
}

void CustomPerspectiveImageFlow::setConstantVerticalOffset(int offset)
{
	if (offset != constantOffset_)
	{
		constantOffset_ = offset;
		invalidate();
	}
}

void CustomPerspectiveImageFlow::setAntialiasMode(AntialiasMode mode)
{
	if (mode != antialiasMode_)
	{
		antialiasMode_ = mode;

		if ((antialiasMode_ == AntialiasStaticSceneOnly && !transitionIsRunning()) ||
			antialiasMode_ == AntialiasAlways)
			antialias_ = true;
		else
			antialias_ = false;

		invalidate();
	}
}

void addLUT(uint *lut, int add)
{
	int value = 0;

	for (int i = 0; i < 256; ++i)
	{
		value = i + add;

		if (value < 0)
			value = 0;
		else if (value > 255)
			value = 255;

		lut[i] = value;
	}
}

void CustomPerspectiveImageFlow::paintToBuffer(QImage &buffer, const QRect &clipRect)
{
	if (itemCount() == 0)
		return;

	QRect dstClip = clipRect.intersect(geometry());

	if (dstClip.isEmpty())
		return;

	FloatPoint p1, p2, p3, p4;
	int x, y;
	int mid;
	int absIndex;

	int scaledSlideHeight = (int)(slideSize().height() * scale());
	int scaledSlideWidth = (int)(slideSize().width() * scale());
	float scaledDepth = 0.2 * scale();

	int verticalOffset = (int)(offset_ * scale()) + (constantOffset_ > 0 ? constantOffset_ : 0);

	QImage image;
	QPoint midPoint(width() / 2, (height() - abs(constantOffset_)) / 2);

	uint lut[256];

	for (int i = leftBound(); i < midIndex(); ++i)
	{
		mid = i - midIndex();
		absIndex = mid + (int)floor(transitionValue());

		image = getImage(absIndex);

		x = midPoint.x() + ((int)(offsetValues()[i]));
		y = (midPoint.y() - scaledSlideHeight / 2) + verticalOffset;

		ImageFX::calculatePerspectivePoints(
			x, y, scaledSlideWidth, scaledSlideHeight,
			1 - widthScaleValues()[i], scaledDepth, 0, false,
			p1, p2, p3, p4
		);

		addLUT(lut, (int)(128 * (widthScaleValues()[i] - 1)));

		ImageFX::projectiveTransformationLUT(
			image, buffer, dstClip,
			p1, p2, p3, p4,
			antialias_, lut
		);
	}

	for (int i = rightBound(); i > midIndex(); --i)
	{
		mid = i - midIndex();
		absIndex = mid + (int)floor(transitionValue());

		image = getImage(absIndex);

		x = midPoint.x() + ((int)(offsetValues()[i + 1])) - ((int)(distanceValues()[i]));
		y = (midPoint.y() - scaledSlideHeight / 2) + verticalOffset;

		ImageFX::calculatePerspectivePoints(
			x, y, scaledSlideWidth, scaledSlideHeight,
			widthScaleValues()[i] - 1, scaledDepth, 1, false,
			p1, p2, p3, p4
		);

		addLUT(lut, (int)(128 * (widthScaleValues()[i] - 1)));

		ImageFX::projectiveTransformationLUT(
			image, buffer, dstClip,
			p1, p2, p3, p4,
			antialias_, lut
		);
	}

	int i = midIndex();
	absIndex = (int)floor(transitionValue());

	image = getImage(absIndex);

	x = midPoint.x() + ((int)(offsetValues()[i]));
	y = (midPoint.y() - scaledSlideHeight / 2) + verticalOffset;

	ImageFX::calculatePerspectivePoints(
		x, y, scaledSlideWidth, scaledSlideHeight,
		1 - widthScaleValues()[i], scaledDepth, 0, false,
		p1, p2, p3, p4
	);

	addLUT(lut, (int)(128 * (widthScaleValues()[i] - 1)));

	ImageFX::projectiveTransformationLUT(
		image, buffer, dstClip,
		p1, p2, p3, p4,
		antialias_, lut
	);
}

void CustomPerspectiveImageFlow::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
#ifdef DEBUG_PAINT_OVERLAY
	p.setPen(QColor(255, 255, 255));
	p.setBrush(NoBrush);

	QPoint midPoint(width() / 2, height() / 2);

	// Draw selection frame
	p.drawRect(
		midPoint.x() - (slideSize().width() * scale()) / 2,
		midPoint.y() - (slideSize().height() * scale()) / 2,
		slideSize().width() * scale(),
		slideSize().height() * scale()
	);

	// Draw midpoint
	p.drawPoint(midPoint);

	// Draw current and transition values
	p.drawText(10, 20, QString("%1").arg(transitionValue()));
	p.drawText(10, 35, QString("%1").arg(currentValue()));
#endif
}

QImage CustomPerspectiveImageFlow::getImage(int index)
{
	return QImage();
}

void CustomPerspectiveImageFlow::transitionStep(float start, float stop, float value, float step)
{
	DPRINTF("ImageFlow::transitionStep");

	if (antialiasMode_ != AntialiasAlways)
		antialias_ = false;

	isScrolling_ = true;

	CustomFlow::transitionStep(start, stop, value, step);
}

void CustomPerspectiveImageFlow::transitionFinished()
{
	DPRINTF("ImageFlow::transitionFinished");

	if (antialiasMode_ == AntialiasStaticSceneOnly || antialiasMode_ == AntialiasAlways)
		antialias_ = true;

	invalidate();

	CustomFlow::transitionFinished();
}

void CustomPerspectiveImageFlow::transitionStopped()
{
	DPRINTF("ImageFlow::transitionStopped");
	isScrolling_ = false;

	CustomFlow::transitionStopped();
}

void CustomPerspectiveImageFlow::resized()
{
	setScale(((float)height() - abs(constantOffset_)) / slideSize().height());
	CustomFlow::updateValues();
	PositionedLayer::resized();
}

float CustomPerspectiveImageFlow::hitTestIndex(int xPos, float transitionValue, const FloatArray &offsetValues, const FloatArray &distanceValues, bool omitDistanceSpace)
{
	float halfMidWidth = fabs(offsetValues[midIndex()]);
	int xOffsetRel = xPos - (width() / 2 - (int)halfMidWidth);
	float clickedIndex = -1;

	if (!omitDistanceSpace)
	{
		if (xOffsetRel >= 0)
			for (int i = midIndex() + 1; i < elementCount(); ++i)
			{
				if (xOffsetRel < offsetValues[i] + halfMidWidth)
				{
					clickedIndex = (i - 1) + (xOffsetRel - offsetValues[i - 1] - halfMidWidth) / (offsetValues[i] - offsetValues[i - 1]);
					break;
				}
			}
		else if (xOffsetRel < 0)
			for (int i = midIndex() - 1; i >= 0; --i)
			{
				if (xOffsetRel > offsetValues[i] + halfMidWidth)
				{
					clickedIndex = i + (xOffsetRel - offsetValues[i] - halfMidWidth) / (offsetValues[i + 1] - offsetValues[i]);
					break;
				}
			}
	}
	else
	{
		if (xOffsetRel >= 0)
			for (int i = midIndex() + 1; i < elementCount(); ++i)
			{
				if (xOffsetRel < offsetValues[i] + halfMidWidth - distanceValues[i - 1] && xOffsetRel > offsetValues[i - 1] + halfMidWidth)
				{
					clickedIndex = (i - 1) + (xOffsetRel - offsetValues[i - 1] - halfMidWidth) / (offsetValues[i] - offsetValues[i - 1] - distanceValues[i - 1]);
					break;
				}
			}
		else if (xOffsetRel < 0)
			for (int i = midIndex() - 1; i >= 0; --i)
			{
				if (xOffsetRel > offsetValues[i] + halfMidWidth && xOffsetRel < offsetValues[i + 1] + halfMidWidth - distanceValues[i])
				{
					clickedIndex = i + (xOffsetRel - offsetValues[i] - halfMidWidth) / (offsetValues[i + 1] - offsetValues[i] - distanceValues[i]);
					break;
				}
			}
	}

	float absClickedIndex = floor(transitionValue) + clickedIndex - midIndex();

	if (clickedIndex > -1)
		return absClickedIndex;
	else
		return -1;
}

void CustomPerspectiveImageFlow::mousePressEvent(QMouseEvent *event)
{
	// Copy current offset and distance arrays for the hit test in mouseMoveEvent...
	offsetValues_ = offsetValues().copy();
	distanceValues_ = distanceValues().copy();

	// Save the mouse down position, so we can use it for the movement tolerance below...
	mouseDownPosition_ = event->pos();

	int xPos = event->x();

	// Get the index value of the slide that is currently mid-screen and
	// the index value of the slide we pressed the mouse button on...
	mouseDownStartIndexValue_ = transitionValue();
	mouseDownIndexValue_ = hitTestIndex(xPos, mouseDownStartIndexValue_, offsetValues_, distanceValues_, false);

	//if (!isScrolling_)
	//	mouseMoved_ = false;

	stopTransition();

	// Initialize helper members used for kinetic scrolling...
	mouseDown_ = true;
	lastDiff_ = -1;
	lastPos_ = xPos;
	startPos_ = xPos;
	startTime_ = QTime::currentTime();
	lastUpdateTime_ = QTime::currentTime();

	PositionedLayer::mousePressEvent(event);
}

void CustomPerspectiveImageFlow::mouseMoveEvent(QMouseEvent *event)
{
	if (!mouseMoved_)
		mouseMoved_ = event->x() < mouseDownPosition_.x() - 10 || event->x() > mouseDownPosition_.x() + 10;

	if (mouseDown_ && mouseMoved_)
	{
		int xPos = event->x();

		int diff = lastPos_ - xPos;
		if (diff == 0)
			return;

		if ((lastDiff_ != -1) && ((diff < 0) != (lastDiff_ < 0)))
		{
			lastPos_ = xPos;
			startPos_ = xPos;
			startTime_ = QTime::currentTime();
		}

		lastDiff_ = diff;

		// Set new index value for the current position...

		if (antialiasMode_ != AntialiasAlways)
			antialias_ = false;

		float mouseOverIndex = hitTestIndex(xPos, mouseDownStartIndexValue_, offsetValues_, distanceValues_, false);
		setCurrentValue(mouseDownStartIndexValue_ + (mouseDownIndexValue_ - mouseOverIndex));
		updateValues();
		repaint();

		lastPos_ = xPos;
		lastUpdateTime_ = QTime::currentTime();
	}

	PositionedLayer::mouseMoveEvent(event);
}

void CustomPerspectiveImageFlow::mouseReleaseEvent(QMouseEvent *event)
{
	if (mouseDown_)
	{
		mouseDown_ = false;
		int endPos = event->x();

		// Did the user simply click on a slide?
		// If so, use the index value we got via hit test in the mousePressEvent
		// and start the transition...
		if (!mouseMoved_ && !isScrolling_)
		{
			// Clicked
			DPRINTF("mouseReleaseEvent::clicked mouseDownIndexValue: %f", mouseDownIndexValue_);
			showIndex(mouseDownIndexValue_);
			return;
		}

		mouseMoved_ = false;
		isScrolling_ = false;

		QTime endTime = QTime::currentTime();

		// Do not scroll if the finger was paused, instead
		// start transition to the nearest index...
		if (lastUpdateTime_.msecsTo(endTime) > 80)
		{
			DPRINTF("ImageFlow::mouseReleaseEvent FingerPause");
			DPRINTF("showIndex: %f (%d)", currentValue(), currentValue());
			showIndex(floor(0.5 + currentValue()));
			return;
		}

		// The user is flicking through the slides, so
		// calculate the end index and start transition...
		int posDiff = endPos - startPos_;
		int timeDiff = startTime_.msecsTo(endTime);

		if (timeDiff != 0)
			pixelPerSec_ = posDiff * 1000 / timeDiff;
		else
			pixelPerSec_ = 0;

		DPRINTF("ImageFlow::mouseReleaseEvent posDiff: %d  timeDiff: %d  pixel/sec: %d", posDiff, timeDiff, pixelPerSec_);

		showIndex(floor(0.5 + currentValue() - pixelPerSec_ / (slideSize().width() * scale())));
	}

	PositionedLayer::mouseReleaseEvent(event);
}
