/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef CUSTOMFLOW_H_
#define CUSTOMFLOW_H_

#include <qwidget.h>
#include <qimage.h>
#include <qarray.h>
#include <qsize.h>

#include "paintbox.h"
#include "transitiontimer.h"

typedef QArray<float> FloatArray;

class CustomFlow : public QObject, public PositionedLayer
{
	Q_OBJECT
public:
	CustomFlow(LayeredPaintBox *parent = 0);
	virtual ~CustomFlow();
	
	virtual void setSlideSize(QSize size);
	QSize slideSize() const { return slideSize_; }

	virtual void setScale(float scale);
	float scale() const { return scale_; }
	
	virtual void setItemCount(int count);
	int itemCount() const { return itemCount_; }

	inline int elementCount() { return elementCount_; }	
	inline float transitionValue() { return transitionValue_; }
	inline float currentValue() { return currentValue_; }
	
	void setTransitionTime(uint interval) { timer_.setTransitionTime(interval); }
	uint transitionTime() const { return timer_.transitionTime(); }
	
	virtual void setDistance(float distance);
	float distance() const { return distance_; }
	
	virtual void setMinFlowValue(float minVal);
	float minFlowValue() const { return minFlowValue_; }
	
	virtual void setFlowSpan(float span);
	float flowSpan() const { return flowSpan_; }
	
	virtual void setMinDistanceValue(float minVal);
	float minDistanceValue() const { return minDistanceValue_; }

	virtual void setDistanceSpan(float span);
	float distanceSpan() const { return distanceSpan_; }
	
	virtual void showIndex(int index);
	
public slots:	
	virtual void showPreviousItem();
	virtual void showNextItem();
	
	virtual void showPreviousPage();
	virtual void showNextPage();

	virtual void jumpTo(float value);
	virtual void transitionTo(float value);
	
signals:
	void transitionValueChanged(float value);
	void currentValueChanged(float value);
	
protected:
	typedef float (CustomFlow::*ValueFuncPtr)(float value, float minValue, float width);
	
	virtual float flowValue(float value, float minValue, float width);
	virtual float distanceValue(float value, float minValue, float width);
	
	virtual void calcScaleValues(FloatArray &dst, int count, int midIndex, float shift, float minValue, float flowSpan, CustomFlow::ValueFuncPtr func);
	virtual void calcWidthValues(FloatArray &dst, const FloatArray &scales, float maxWidth);
	
	virtual void calcOffsetValues(FloatArray &dst, const FloatArray &widths, const FloatArray &distances);

	virtual void updateElementCountAndMidIndex();
	virtual void updateValues();
	virtual void invalidate();
	
	virtual void setCurrentValue(float value);
	
	void stopTransition();
	
	inline bool transitionIsRunning() { return timer_.isRunning(); }
	
	inline FloatArray &widthScaleValues() { return widthScaleValues_; }
	inline FloatArray &widthValues() { return widthValues_; }
	
	inline FloatArray &distanceScaleValues() { return distanceScaleValues_; }
	inline FloatArray &distanceValues() { return distanceValues_; }
	
	inline FloatArray &offsetValues() { return offsetValues_; }
	
	inline int leftBound() { return leftBound_; }
	inline int rightBound() { return rightBound_; }
	inline int midIndex() { return midIndex_; }

protected slots:
	virtual void transitionStep(float start, float stop, float value, float step);
	virtual void transitionFinished();
	virtual void transitionStopped();
	
private:
	FloatArray widthScaleValues_;
	FloatArray widthValues_;
	
	FloatArray distanceScaleValues_;
	FloatArray distanceValues_;
	
	FloatArray offsetValues_;
	
	TransitionTimer timer_;
	float transitionValue_;
	float currentValue_;
	int itemCount_;
	
	float distance_;
	
	float minFlowValue_;
	float minDistanceValue_;
	
	float flowSpan_;
	float distanceSpan_;
	
	QSize slideSize_;
	float scale_;
	
	int elementCount_;
	int leftBound_;
	int rightBound_;
	int midIndex_;
};

#endif /*CUSTOMFLOW_H_*/
