/* -*- mode: c++; tab-width: 4; c-basic-offset: 4 -*- */
/*
 * Copyright (C) 2005 Atmark <atmarkat _AT_ msn _DOT_ com>
 *                    AGAWA Koji <i _AT_ atty _DOT_ jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef CONFIGURATION_H_INCLUDED
#define CONFIGURATION_H_INCLUDED

#include <qstring.h>
#include <qmap.h>
#include <qarray.h>
#include <qdir.h>

#include "quasar.h"
#include "playlist.h"
#include "skinmanager.h"

#include "quasar_version.h"

#define APPNAME "Quasar"
#define CONFIGPATHNAME ".quasar"
#define CONFIGNAME "quasar.conf"
#define MEDIADBNAME "quasar.sqlite3"
#define CONFIGGROUPNAME "Quasar"
#define CONFIGVERSION 2
#define DBVERSION 3
#define BINNAME "quasar"

class GlobalConfiguration
{
public:
	static GlobalConfiguration& singleton();
#define qGlobalConfig GlobalConfiguration::singleton()

	void read();

	const QString appPath() const { return appPath_; }
	const QString appVolumePath() const { return appVolumePath_; }
	const QString appBinaryFilename() const { return appBinaryFilename_; }

	bool isRunningSelfcontained() const { return selfcontainedMode_; }

	QString configPath() const {	return configPath_; }
	QString configFilename() const { return configFilename_; }
	QString databaseFilename() const { return databaseFilename_; }
	QString skinPath() const { return skinPath_; }

	QString getRelativeFilePath(const QString &input) const;
	QString resolveRelativeFilePath(const QString &input) const;

private:
	QString appPath_;
	QString appVolumePath_;
	QString appBinaryFilename_;
	QString configPath_;
	QString configFilename_;
	QString databaseFilename_;
	QString skinPath_;
	bool selfcontainedMode_;
};

/*! \class Configuration
  \brief ZPlayerの設定を管理するクラスです。

 * 新たに設定項目を追加するには？

 a. configuration.h
  1. Configuration クラスに変数を追加します。
 b. configuration.cpp
  1. Configuration::Configuration()に a-1 の変数をデフォルト値で初期化する文を追加します。
  2. Configuration::write(), Configuration::read()に設定保存のための処理を追加します。
*/
class Configuration
{
public:
	static Configuration& singleton();
#define qConfig Configuration::singleton()

	//! Configurationクラスをデフォルト設定で構築します。
	Configuration();
	~Configuration();

	float readAppVersion() const;
	int readDatabaseVersion() const;

	//! 設定ファイルのバージョンが正常かチェックします。
	bool isValidConfig() const;
	//! Qtopiaの設定ファイルに設定を書き出します。
	void write() const;
	//! Qtopiaの設定ファイルから設定を読み出します。
	void read();

public:
	// ---------------- ZPlayerに関連する設定
	QString lastDirectory;		//!< 最後に開いたディレクトリ
	QString lastPlaylistDirectory;
	QString lastPlaylistFile;	//!< 最後に開いたプレイリストファイル
	PlayList::PlayOrder playOrder; //!< プレイリスト内の再生順序
	SkinManager::InfoMode infoMode;

	PlayList::InputMode inputMode;

	bool isBackendLoggingEnabled; //!< バックエンドのコンソール出力を記録するか
	QString backendLogFileName;	//!< ログファイルのファイル名

	bool changePlayInfoLayoutInPortaitMode;
	int playlistAutoTriggerFilterInterval;
	QString skin1;
	QString skin2;

	enum HotKeyContext {
		HOTKEY_CONTEXT_GLOBAL = 0,
		HOTKEY_CONTEXT_PLAYLIST = 1,
		HOTKEY_CONTEXT_VIDEO = 2,
		HOTKEY_CONTEXT_AUDIO = 3,
		HOTKEY_CONTEXT_HOTAREAS = 4,
		HOTKEY_CONTEXT_MAX = 5
	};
	typedef QMap<int, int> HotKeyMap;
	HotKeyMap hotKeyMap[HOTKEY_CONTEXT_MAX]; //!< ホットキー定義

	bool automaticallySwitchToPlayInfoEnabled;
	bool isAskQuitingEnabled;
	bool registerAsRemoteConMaster;
	bool qcopBroadcastingEnabled;
	bool createTemporaryPlaybackStatusFile;

	bool isOverviewVisible;
	int overviewSplitterPosition;

	bool isCoverArtFlowVisible;
	int coverArtFlowSplitterPosition;

	bool isTouchModeEnabled;
	bool isKineticScrollingEnabled;

#ifdef QTOPIA
	// ---------------- BatteryPlusに関連する設定
	bool isBatteryPlusEnabled;	//!< BatteryPlus関係の機能を使うか
	QString batteryPlusMode4Video; //!< ビデオ再生時に設定するモード
	QString batteryPlusMode4Audio; //!< オーディオ再生時に設定するモード
#endif

	// ---------------- MPlayerに関連する設定
	QString mplayerPath;		//!< MPlayer実行ファイルのパス
	QString mplayerBinLocation();

	bool isOverlayEnabled;		//!< オーバレイを使うか
	bool isFullScreenEnabled;	//!< ビデオをフルスクリーン再生するか
	bool isDoubleBufferingEnabled; //!< ダブルバッファリングを行なうか

	enum FramedropMethod {
		FRAMEDROP_NONE = 0,		//!< フレームドロップ無し
		FRAMEDROP_NORMAL = 1,	//!< 通常のフレームドロップ
		FRAMEDROP_HARD = 2		//!< 強力なフレームドロップ
	};
	FramedropMethod framedropMethod; //!< フレームドロップの方法

	bool isCacheEnabled;
	int cacheSize;
	bool isPrefetchFileEnabled;

	bool isAdditionalOptionsEnabled; //!< コマンドラインオプションを追加するか
	QString additionalOptions;	//!< 追加のコマンドラインオプション

	QString coverArtDirectory;
	QString coverArtFlowCacheDirectory;

	QStringList formatFileExtensions;
	QStringList formatMPlayerArguments;

	// ----------------
	//! ZPlayerを動作させている機種の分類
	enum MachineCategory {
		MACHINE_SLC700,			//!< SL-C700系の機種
		MACHINE_SLC3000,		//!< SL-C3000系の機種
		MACHINE_GENERIC			//!< その他の機種
	};
	//! ZPlayerを動作させている機種の分類を返します。
	MachineCategory machineCategory() const { return machineCategory_; }
	//! ZPlayerを動作させている機種の名前を返します。
	QString machineName() const { return machineName_; }

	void updateValidExtensionsList();

	QStringList &validExtensions() { return validExtensionsList_; }
	inline bool isValidExtension(const QString &extension) { return validExtensionsList_.contains(extension.lower()); }

	QStringList &ambiguousFormatExtensions() { return ambiguousFormatExtensions_; }
	inline bool isAmbiguousFormatExtension(const QString &extension) { return ambiguousFormatExtensions_.contains(extension.lower()); }

	const QString mplayerArguments(QString fileExtension);

private:
	void detectMachine();

	QStringList readListEntry(Config &cf, const QString &key, const QChar &sep, bool allowEmptyEntries = false);
	void setDefaultFormats();

	MachineCategory machineCategory_;
	QString machineName_;
	QStringList validExtensionsList_;
	QStringList ambiguousFormatExtensions_;
};

#endif // CONFIGURATION_H_INCLUDED
