/***************************************************************************
 *   Copyright (C) 2004 by Steven Scott                                    *
 *   progoth@progoth.com                                                   *
 *                                                                         *
 *   Copyright (C) 2007-2008 Andre Beckedorf                               *
 *   evilJazz _AT_ katastrophos _DOT_ net                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef AUDIOSCROBBLER_H
#define AUDIOSCROBBLER_H

#include <qstring.h>
#include <qdatetime.h>
#include <qtimer.h>
#include <qbuffer.h>

#include "kascachelist.h"

class QUrl;
class QHttp;

/**
@author Steven Scott
*/
class AudioScrobbler : public QObject
{
	Q_OBJECT
public:
    AudioScrobbler( QObject *parent, const QString &clientName = "qsc", const QString &clientVersion = "0.1" );
	//AudioScrobbler( QObject *parent, const QString &clientName = "tst", const QString &clientVersion = "1.0" );

    ~AudioScrobbler();

    QString handshakeAddress() const
    { return QString("http://post.audioscrobbler.com?hs=true&p=1.1&c=%1&v=%2&u=").arg(m_clientName).arg(m_clientVersion); }
//    { return QString("http://darkstar.obsidian.ath.cx/audioscrobbler/?hs=true&p=1.1&c=%1&v=%2&u=").arg(m_clientName).arg(m_clientVersion); }
//	  { return QString("http://324dfgasd23/audioscrobbler/?hs=true&p=1.1&c=%1&v=%2&u=").arg(m_clientName).arg(m_clientVersion); }

    const KASCacheList & cache() const { return m_subcache; }
    void clearCache();

signals:
    void statusMessage( const QString& );
    void songAddedForSubmission(const QString&);
    void songAdded(const QString &artist, const QString &songtitle, const QString &album, const QString &musicbrainzid, unsigned int length);
    void songRemoved();
    void cacheSaved();

public slots:
    int play( const QString &artist, const QString &songtitle, const QString &album,
               const QString &musicbrainzid, unsigned int length );
    void pause();
    void unpause();
    void seek();
    void stop();

    bool isActive() { return songPlayTimer->isActive(); }

    void setUsername( const QString &username );
    void setMD5Password( const QString &passwordMD5 );
    void setPassword( const QString &password );
    void setInterval( unsigned int interval );
    void setCacheSaveInterval( unsigned int interval );
    void setCacheFileName(const QString &fileName);
    void setSubmissionPaused(bool pause);

    const ScrobSongData &currentSong() { return m_currentsong; }
    unsigned int currentRemainingTime() { return m_remainingtime - m_playtime.secsTo(QDateTime::currentDateTime()); }

    unsigned int requiredTime() { return m_requiredtime; }

    QString lastStatus() { return m_laststatus; }

    QString username() { return m_username; }
    QString md5Password() { return m_password; }
    unsigned int interval() { return m_interval / 1000; }
    unsigned int cacheSaveInterval() { return m_cachesaveinterval / 1000; }
    QString cacheFileName() { return m_cacheFileName; }
    bool submissionPaused() { return m_submissionPaused; }

    void reinitialize();

protected slots:
    void saveCacheToDisk();
    void saveLockedCacheToDisk();
    void loadCacheFromDisk();

    void submit();
    void newSong();

    void doHandshake();

    void httpRequestFinished(int, bool);

protected:
    void doRequest( const QUrl &address, const QString &postdata = QString::null );
    void parseResponse( const QString &response );
    QString md5Response() const;

    void sendStatus(const QString &status);

    QHttp *m_http;
    QBuffer *m_buf;

    KASCacheList m_subcache;
    bool m_subcachechanged;
    int m_itemsFetchedWithPostData;
    QTimer *cacheTimer;

    QString m_clientName;
    QString m_clientVersion;

    QString m_postaddress;
    QString m_challenge;
    bool m_handshake_done;

    QString m_username;
    QString m_password;
    QString m_cacheFileName;

    bool m_submissionPaused;

    QTimer *hsTimer;
    QTimer *intervalTimer;
    QTimer *songPlayTimer;

    unsigned int m_interval;
    unsigned int m_cachesaveinterval;

    ScrobSongData m_currentsong;
    QString m_laststatus;
    QDateTime m_playtime;
    unsigned int m_remainingtime;
    unsigned int m_requiredtime;
    bool m_songplaying;

    static unsigned int MIN_SONG_LENGTH;
    static unsigned int MAX_SONG_LENGTH;
    static unsigned int MIN_PLAY_TIME;
};

#endif
