#ifndef WINDOWMANAGER_H_
#define WINDOWMANAGER_H_

#include <QObject>
#include <QTime>
#include "Room.h"
#include "Meeting.h"
#include "PasswordDialog.h"
#include "DeviceManager.h"

class QTimer;
class RoomStatusIndicatorWidget;
class WeeklyViewWidget;
class Engine;
class MeetingInfoDialog;
class SettingsView;

//! UserInterface class. Behaves as a proxy between the user interface and application's business logic.
/*!
 * UserInterface class. Controls the whole user interface, starting with displaying the appropriate
 * views. It behaves as a proxy between the user interface and application's business logic, it connects
 * the specified components together and forwards the data to the correct place. It also manages the correct
 * appearance of current views on the screen.
 */
class WindowManager : public QObject
{
	Q_OBJECT

public:
	//! Constructor.
	/*!
	 * Constructor of WindowManager.
	 */
	WindowManager();
	//! Destructor.
	virtual ~WindowManager();

signals:
	//! Request current status of the room.
	/*!
	 * Signal is emitted when there is need to check current status of room aRoom.
	 * \param aRoom Meetingroom which status is requested.
	 */
	void roomStatusInfoNeeded( Room *aRoom );

private slots:
	//! Closes the application.
	void closeApplication();
	//! Updates the time.
	/*!
	 * Forwards the signal of changed time to current view.
	 * \param aCurrentDateTime Current date and time.
	 */
	void distributeDateTimeInfo( QDateTime aCurrentDateTime );
	//! Updates the rooms status.
	/*!
	 * Forwards the signal of changed status to current view.
	 * \param aRoom Room which status is changed.
	 * \param aStatus Current status of room.
	 * \param aTime Time when status is changed.
	 */
	void roomStatusChanged( Room *aRoom, Room::Status aStatus, QTime aTime );
	//! Displays the weekly view
	void showWeeklyView();
	//! Displays the screensaver (room status view)
	void showRoomStatus();
	//! Displays the settings view
	void showSettingsView();
	/*!
	 * Displays the meeting info dialog
	 * \param aMeeting Meeting to be displayd
	 */
	void showMeetingInfo( Meeting *aMeeting );
	/*!
	 * Displays an error message
	 * \param aErrorMessage Message to be displayd
	 */
	void error( const QString &aErrorMessage );
	//! Restarts the timer to launch the screensaver.
	void observedEventDetected();
	//! Slot for fetching meetings.
	/*!
	 * Slot. Fetches meetings for room aRoom for currently visible week
	 * \param aRoom
	 */
	void fetchMeetings( Room *aRoom );
	//! Slot for fetching meetings.
	/*!
	 * Slot. Fetches meetings for current room from date aFrom to week ahead.
	 * \param aFrom Date where to begin fetching
	 */
	void fetchMeetings( QDate aFrom );
	//! Slot for popping up the confirmation dialog to change the current operation mode
	/*!
	 * Slot. Asks PopUpMessageBox to pop up a confirmation dialog.
	 * \param aMode The operation mode to be changed to
	 */
	void changeModeOrdered( DeviceManager::OperationMode aMode );
	//! Slot for receiving the status of the entered password
	/*!
	 * Slot. Receives the status of the entered password and makes the DeviceManager to change the
	 * operation mode if the password is correct.
	 * \param aPasswordStatus The status of the password.
	 */
	void passwordEntered( PasswordDialog::PasswordStatus aPasswordStatus );

private:
	//! Name of the application.
	QString iApplicationName;
	//! Pointer to the weekly view.
	WeeklyViewWidget *iWeeklyView;
	//! Pointer to the screensaver (room status view).
	RoomStatusIndicatorWidget *iRoomStatusView;
	//! Pointer to the meeting info dialog
	MeetingInfoDialog *iMeetingInfo;
	//! Pointer to the engine.
	Engine *iEngine;
	//! Timer to launch the screensaver widget
	QTimer *iIdleTimeCounter;
	//! Pointer to the settings view
	SettingsView *iSettingsView;

};

#endif /*WINDOWMANAGER_H_*/
