#include "WindowManager.h"

#include <QApplication>
#include <QTimer>
#include "Configuration.h"
#include "DisplaySettings.h"
#include "Meeting.h"
#include "Room.h"
#include "Engine.h"
#include "Clock.h"
#include "WeeklyViewWidget.h"
#include "RoomStatusIndicatorWidget.h"
#include "MeetingInfoDialog.h"
#include "PopUpMessageBox.h"
#include "DeviceManager.h"
#include "SettingsView.h"

#include <QtDebug>

const int IDLE_TIME_MULTIPLIER = 60000; // Multiplies milliseconds to minutes

WindowManager::WindowManager() :
		QObject(),
		iApplicationName( tr( "Qt Meetings" ) ),
		iWeeklyView( 0 ),
		iRoomStatusView( 0 ),
		iMeetingInfo( 0 )
{
	iEngine = new Engine;
	connect( iEngine, SIGNAL( initializationFailed() ), this, SLOT( closeApplication() ) );
	connect( this, SIGNAL( roomStatusInfoNeeded( Room * ) ), iEngine, SLOT( roomStatusInfoNeeded( Room * ) ) );
	connect( iEngine, SIGNAL( roomStatusChanged( Room *, Room::Status, QTime ) ), this, SLOT( roomStatusChanged( Room *, Room::Status, QTime ) ) );
	connect( iEngine->clock(), SIGNAL( tick( QDateTime ) ), this, SLOT( distributeDateTimeInfo( QDateTime ) ) );
	connect( iEngine, SIGNAL( error( QString ) ), this, SLOT( error( QString ) ) );
	connect( iEngine->deviceManager(), SIGNAL( changeModeOrdered( DeviceManager::OperationMode ) ), this, SLOT( changeModeOrdered( DeviceManager::OperationMode ) ) );

	iWeeklyView = new WeeklyViewWidget( QDateTime::currentDateTime(), iEngine->configuration() );
	iWeeklyView->setWindowTitle( iApplicationName );
	connect( iEngine, SIGNAL( meetingAdded( Meeting * ) ), iWeeklyView, SLOT( insertMeeting( Meeting * ) ) );
	connect( iEngine, SIGNAL( meetingDeleted( Meeting * ) ), iWeeklyView, SLOT( deleteMeeting( Meeting * ) ) );
	connect( iWeeklyView, SIGNAL( observedEventDetected() ), this, SLOT( observedEventDetected() ) );
	connect( iWeeklyView, SIGNAL( meetingActivated( Meeting * ) ), this, SLOT( showMeetingInfo( Meeting * ) ) );
	connect( iWeeklyView, SIGNAL( currentRoomChanged( Room * ) ), iEngine, SLOT( currentRoomChanged( Room * ) ) );
	connect( iWeeklyView, SIGNAL( currentRoomChanged( Room * ) ), this, SLOT( fetchMeetings( Room * ) ) );
	// TODO: fetch meetings for specific week
	connect( iWeeklyView, SIGNAL( shownWeekChanged( QDate ) ), this, SLOT( fetchMeetings( QDate ) ) );

	iIdleTimeCounter = new QTimer();
	iIdleTimeCounter->setSingleShot( true );
	iIdleTimeCounter->setInterval( IDLE_TIME_MULTIPLIER * iEngine->configuration()->displaySettings()->screensaver() );
	iIdleTimeCounter->start();
	connect( iIdleTimeCounter, SIGNAL( timeout() ), this, SLOT( showRoomStatus() ) );

	if ( iEngine->deviceManager()->currentOperationMode() == DeviceManager::KioskMode )
	{
		iWeeklyView->setWindowState( Qt::WindowFullScreen );
	}
	else
	{
		iWeeklyView->setWindowState( Qt::WindowMaximized );
	}
	showWeeklyView();

	//QTimer::singleShot( 0, this, SLOT( closeApplication() ) );
}

WindowManager::~WindowManager()
{
	if ( iWeeklyView != 0 )
	{
		delete iWeeklyView;
		iWeeklyView = 0;
	}

	if ( iRoomStatusView != 0 )
	{
		delete iRoomStatusView;
		iRoomStatusView = 0;
	}

	if ( iMeetingInfo != 0 )
	{
		delete iMeetingInfo;
		iMeetingInfo = 0;
	}

	if ( iIdleTimeCounter )
	{
		iIdleTimeCounter->stop();
		delete iIdleTimeCounter;
		iIdleTimeCounter = 0;
	}
}

void WindowManager::closeApplication()
{
	qDebug() << "WindowManager::closeApplication\tclose application";
	// closes application after 1 second
	QTimer::singleShot( 1000, QApplication::instance(), SLOT( quit() ) );
}

void WindowManager::distributeDateTimeInfo( QDateTime aCurrentDateTime )
{
	if ( iRoomStatusView != 0 && iRoomStatusView->isActiveWindow() )
	{
		iRoomStatusView->setCurrentTime( aCurrentDateTime.time() );
	}

	if ( iWeeklyView != 0 && iWeeklyView->isActiveWindow() )
	{
		iWeeklyView->setCurrentDateTime( aCurrentDateTime );
	}
}

void WindowManager::roomStatusChanged( Room *aRoom, Room::Status aStatus, QTime aTime )
{
	// currently works only for default room
	if ( aRoom->equals( iEngine->defaultRoom() ) )
	{
		if ( iRoomStatusView == 0 )
		{
			iRoomStatusView = new RoomStatusIndicatorWidget( aRoom, aStatus, aTime, iEngine->configuration()->displaySettings()->timeFormat() );
			iRoomStatusView->setWindowTitle( iApplicationName );
			connect( iRoomStatusView, SIGNAL( observedEventDetected() ), this, SLOT( observedEventDetected() ) );
			if ( iEngine->deviceManager()->currentOperationMode() == DeviceManager::KioskMode )
			{
				iRoomStatusView->setWindowState( Qt::WindowFullScreen );
			}
			else
			{
				iRoomStatusView->setWindowState( Qt::WindowMaximized );
			}
		}
		else
		{
			iRoomStatusView->statusChanged( aStatus, aTime );
		}

		if ( !iWeeklyView->isVisible() && !iRoomStatusView->isVisible() )
		{
			showRoomStatus();
		}
	}
}

void WindowManager::showRoomStatus()
{
	qDebug() << "WindowManager::showRoomStatus";

	if ( iRoomStatusView == 0 )
	{
		iEngine->roomStatusInfoNeeded( iWeeklyView->currentRoom() );
	}
	else
	{
		iRoomStatusView->show();
		if ( iWeeklyView->isVisible() )
		{
			iWeeklyView->hide();
		}
	}

	// closing/deleting meeting info dialog
	if ( iMeetingInfo != 0 )
	{
		iMeetingInfo->reject();
	}
}

void WindowManager::showWeeklyView()
{
	qDebug() << "WindowManager::showWeeklyView";
	if ( iRoomStatusView != 0 && iRoomStatusView->isVisible() )
	{
		iRoomStatusView->hide();
	}

	iWeeklyView->show();
}

void WindowManager::showMeetingInfo( Meeting *aMeeting )
{
	iMeetingInfo = new MeetingInfoDialog( aMeeting );
	// Display modal dialog
	iMeetingInfo->exec();

	delete iMeetingInfo;
	iMeetingInfo = 0;
}

void WindowManager::showSettingsView()
{
	// TODO : give the Torspo for the person who was responsible to write this method
}

void WindowManager::error( const QString &aErrorMessage )
{
	qDebug() << "WindowManager::showErrorPopup";

	PopUpMessageBox::error( 0, aErrorMessage );
}

void WindowManager::observedEventDetected()
{
	// if event was detected on room status view
	if ( iRoomStatusView != 0 && iRoomStatusView->isVisible() )
	{
		// show weekly view
		showWeeklyView();
	}
	// otherwise
	else
	{
		// prepare to restart idle counter
		if ( iIdleTimeCounter->isActive() )
		{
			iIdleTimeCounter->stop();
		}
	}
	// (re)start idle counter
	iIdleTimeCounter->start();
}

void WindowManager::fetchMeetings( Room * aRoom )
{
	QDateTime from( iWeeklyView->beginnigOfShownWeek() );
	QDateTime to( from.addDays( 8 ) );
	qDebug() << "WindowManager::fetchMeetings from " << from.toString( "d.m. h:mm" )
	<< " to " << to.toString( "d.m. h:mm" );
	iEngine->fetchMeetings( from, to, aRoom );
}

void WindowManager::fetchMeetings( QDate aFrom )
{
	QDateTime from( aFrom );
	QDateTime to( aFrom.addDays( 7 ), QTime( 23, 59 ) );
	qDebug() << "WindowManager::fetchMeetings from " << from.toString( "d.m. h:mm" )
	<< " to " << to.toString( "d.m. h:mm" );
	iEngine->fetchMeetings( from, to, iWeeklyView->currentRoom() );
}

void WindowManager::changeModeOrdered( DeviceManager::OperationMode aMode )
{
	QString message = tr( "You are about to change operation mode to %1." )
				.arg( iEngine->deviceManager()->operationModeToString( aMode ) );
	PasswordDialog *dlg = PasswordDialog::query( 0, iEngine->configuration()->adminPassword(), message );
	qDebug() << "WindowManager::changeModeOrdered/tpassword: " << iEngine->configuration()->adminPassword();
	//TODO make this modal!!!
	connect( dlg, SIGNAL( passwordEntered( PasswordDialog::PasswordStatus ) ), this, SLOT( passwordEntered( PasswordDialog::PasswordStatus ) ) );
}

void WindowManager::passwordEntered( PasswordDialog::PasswordStatus aPasswordStatus )
{
	switch ( aPasswordStatus )
	{
		case PasswordDialog::Correct :
		{
			PopUpMessageBox::information( 0, tr("Device is rebooting now.") );
			iEngine->deviceManager()->changeMode();
			break;
		}
		case PasswordDialog::Incorrect :
		{
			error( tr("Incorrect password.") );
			break;
		}
		default: // PasswordDialog::Canceled:
		{
		}
	}
}
