#ifndef WEEKLYVIEWWIDGET_H_
#define WEEKLYVIEWWIDGET_H_

#include "ObservedWidget.h"
#include <QDateTime>

class QLabel;
class QPushButton;
class MeetingRoomCombo;
class TimeDisplayWidget;
class ScheduleWidget;
class Configuration;
class Meeting;
class Room;

//! Userinterface class. Shows a weekly calendar for the selected room, provides date and time information.
/*!
 * UserInterface class. Shows the current date and time and selected week's calendar for
 * selected meeting room. User can select meeting room, browse weeks back and forth, and can navigate
 * back to the current week.
 */
class WeeklyViewWidget : public ObservedWidget
{
	Q_OBJECT

public:
	//! Constructor.
	/*!
	 * Constructor to initialize an WeeklyViewWidget instance.
	 * \param aCurrentDateTime
	 * \param aConfiguration Pointer to the configuration object. Not owned.
	 * \param aParent Pointer to the parent widget.
	 */
	WeeklyViewWidget( QDateTime aCurrentDateTime, Configuration *aConfiguration, QWidget *aParent = 0 );
	//! Destructor.
	virtual ~WeeklyViewWidget();

	//! Current meeting
	/*!
	 * Returns the current meeting if any
	 * \return Current meeting if any, otherwise null pointer
	 */
	Meeting* currentMeeting();
	//! Current room
	/*!
	 * Returns the currently selected meeting room.
	 * \return Selected meeting room.
	 */
	Room* currentRoom();

	//! First day of week currently displayd
	/*!
	 * Returns the first day of week currently displayd.
	 * \return First day of week.
	 */
	QDate beginnigOfShownWeek();

signals:
	//! Signals change of the meeting room.
	/*!
	 * Signal is emited when meeting room is changed.
	 * \param aNewRoom Selected meeting room.
	 */
	void currentRoomChanged( Room *aNewRoom );
	//! Signals change of the current meeting.
	/*!
	 * Signal is emited when new meeting is started.
	 * \param aNewMeeting Meeting that has been started.
	 */
	void currentMeetingChanged( Meeting *aNewMeeting );
	//! Meeting activated.
	/*!
	 * Signal is emitted when a meeting is clicked by the user.
	 * \param aMeeting actived meeting.
	 */
	void meetingActivated( Meeting *aMeeting );
	//! Signals creation of new meeting.
	/*!
	 * Signal is emited when new meeting is created.
	 * \param aMeeting Meeting that has been created.
	 * \param aUsername User who created the meeting.
	 * \param aPassword Password of the "aUsername"
	 */
	void meetingToCreate( Meeting *aMeeting, const QString &aUsername, const QString &aPassword );
	//! Signals deletion of a meeting.
	/*!
	 * Signal is emited when meeting is deleted.
	 * \param aMeeting Deleted meeting.
	 * \param aUsername User who deleted the meeting.
	 * \param aPassword Password of the "aUsername"
	 */
	void meetingToDelete( Meeting *aMeeting, const QString &aUsername, const QString &aPassword );
	//! Signals modifications of a meeting.
	/*!
	 * Signal is emited when meeting is modified.
	 * \param aMeeting Modified meeting.
	 * \param aUsername User who modified the meeting.
	 * \param aPassword Password of the "aUsername"
	 */
	void meetingToUpdate( Meeting *aMeeting, const QString &aUsername, const QString &aPassword );

	//! Signals change of shown week.
	/*!
	 * Signal is emited when week is changed.
	 * \param aFirstDayOfWeek Date of the first day of week.
	 */
	void shownWeekChanged( QDate aFirstDayOfWeek );

	//! Signals
	/*!
	 * Signal is emited when settings button is clicked.
	 */
	void showSettingsView();

public slots:
	//! Sets the date and time
	/*!
	 * Sets the current date and time
	 * \param aCurrentDateTime Date and time to be displayd.
	 */
	void setCurrentDateTime( QDateTime aCurrentDateTime );
	//! Insert meeting
	/*!
	 * Inserts new meeting into the calendar
	 * \param aMeeting Meeting to be inserted.
	 */
	void insertMeeting( Meeting *aMeeting );
	//! Delete meeting
	/*!
	 * Removes meeting from the calendar
	 * \param aMeeting Meeting to be deleted.
	 */
	void deleteMeeting( Meeting *aMeeting );
	//! Update meeting
	/*!
	 * Updates the display of given meeting in calendar.
	 * \param aMeeting Meeting to be updated.
	 */
	void updateMeeting( Meeting *aMeeting );

private slots:
	//! Displays the selected week.
	/*!
	 *  Displays the week by aWeekNumber
	 * \param aWeekNumber Weeknumber to be shown.
	 */
	void shownWeekChanged( int aWeekNumber );

private:
	//! Displays the selectable meeting rooms.
	MeetingRoomCombo *iRoomsCombo;
	//! Displays the time.
	TimeDisplayWidget *iTimeDisplay;
	//! Displays the calendar.
	ScheduleWidget *iSchedule;
	//! Displays the date.
	QLabel *iCurrentDayLabel;
	//! Button used to display previous week's calendar.
	QPushButton *iPreviousWeekButton;
	//! Button used to display current week's calendar.
	QPushButton *iCurrentWeekButton;
	//! Button used to display next week's calendar.
	QPushButton *iNextWeekButton;
	//! Settings button. TODO : Correct widget is needed!!
	QPushButton *iSettingsButton;
	//! About button. TODO : Correct widget is needed!!
	QPushButton *iAboutButton;
	/* !
	 * Pointer to configuration object.
	 * Contains configurable data and IS NOT OWNED by the widget.
	 */
	Configuration *iConfiguration;
};

#endif /*WEEKLYVIEWWIDGET_H_*/
